<?php
/**
 * Flying Book Creator - Version Progressive
 * /document/creator/index.php
 */
 

require_once __DIR__ . '/config/auth_check.php';
checkAuth();

$user = getUser();

// Récupérer le nom du projet depuis l'URL ou créer un nouveau
$project_name = $_GET['name'] ?? 'nouveau_projet_' . time();
$project_id = $_GET['id'] ?? null;

// Si reprise d'un projet existant
$existing_project = null;
if ($project_id) {
    $pdo = DatabaseConfig::getConnection();
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$project_id, $user['id']]);
    $existing_project = $stmt->fetch();
    
    if ($existing_project) {
        $project_name = $existing_project['project_name'];
    }
}

// Gestion AJAX pour sauvegarde progressive
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        $pdo = DatabaseConfig::getConnection();
        $response = ['success' => false, 'message' => ''];
        
        switch ($_POST['action']) {
            case 'save_project':
                // Sauvegarde des métadonnées du projet
                $data = json_decode($_POST['data'], true);
                
                if (!$project_id) {
                    // Nouveau projet
                    $slug = generateSlug($data['project_name']);
                    $folder = 'project_' . uniqid();
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO flying_book_projects 
                        (user_id, project_name, project_slug, title, subtitle, description, 
                         style_theme, transition_type, navigation_type, project_folder, status)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
                    ");
                    $stmt->execute([
                        $user['id'], $data['project_name'], $slug, $data['title'],
                        $data['subtitle'] ?? null, $data['description'] ?? null,
                        $data['style_theme'] ?? 'moderne', $data['transition_type'] ?? '3d',
                        $data['navigation_type'] ?? 'all', $folder
                    ]);
                    
                    $project_id = $pdo->lastInsertId();
                    createProjectFolder($folder);
                    
                } else {
                    // Mise à jour projet existant
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET title = ?, subtitle = ?, description = ?, 
                            style_theme = ?, transition_type = ?, navigation_type = ?,
                            updated_at = NOW()
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([
                        $data['title'], $data['subtitle'] ?? null, $data['description'] ?? null,
                        $data['style_theme'] ?? 'moderne', $data['transition_type'] ?? '3d',
                        $data['navigation_type'] ?? 'all', $project_id, $user['id']
                    ]);
                }
                
                $response = ['success' => true, 'project_id' => $project_id, 'message' => 'Projet sauvegardé'];
                break;
                
            case 'save_slide':
                // Sauvegarde d'une slide individuelle
                $slide_data = json_decode($_POST['data'], true);
                
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_slides 
                    (project_id, slide_number, slide_title, media_type, media_file, 
                     background_audio, auto_duration, text_overlay, display_order)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    slide_title = VALUES(slide_title),
                    media_type = VALUES(media_type),
                    media_file = VALUES(media_file),
                    background_audio = VALUES(background_audio),
                    auto_duration = VALUES(auto_duration),
                    text_overlay = VALUES(text_overlay),
                    display_order = VALUES(display_order),
                    updated_at = NOW()
                ");
                
                $stmt->execute([
                    $project_id, $slide_data['slide_number'], $slide_data['title'] ?? null,
                    $slide_data['media_type'], $slide_data['media_file'],
                    $slide_data['background_audio'] ?? null, $slide_data['auto_duration'] ?? null,
                    $slide_data['text_overlay'] ?? null, $slide_data['display_order'] ?? 0
                ]);
                
                $response = ['success' => true, 'message' => 'Slide sauvegardée'];
                break;
                
            case 'autosave':
                // Auto-sauvegarde de l'état complet
                $state = $_POST['state'];
                
                // Sauvegarder l'état en session pour récupération rapide
                $_SESSION['flying_book_autosave'] = [
                    'project_id' => $project_id,
                    'timestamp' => time(),
                    'state' => $state
                ];
                
                $response = ['success' => true, 'message' => 'Auto-sauvegarde OK'];
                break;
        }
        
        echo json_encode($response);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
}

// Fonctions utilitaires
function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}

function createProjectFolder($folder_name) {
    $base_path = __DIR__ . '/../projects/' . $folder_name;
    if (!file_exists($base_path)) {
        mkdir($base_path, 0755, true);
        mkdir($base_path . '/images', 0755, true);
        mkdir($base_path . '/videos', 0755, true);
        mkdir($base_path . '/audio', 0755, true);
        mkdir($base_path . '/thumbnails', 0755, true);
    }
    return $base_path;
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - <?= htmlspecialchars($project_name) ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            min-height: 100vh;
            color: #333;
        }
        
        /* Header avec auto-save status */
        .creator-header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 70px;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 30px;
            z-index: 1000;
            box-shadow: 0 4px 30px rgba(0, 0, 0, 0.1);
        }
        
        .creator-title {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .creator-title h1 {
            font-size: 1.5rem;
            color: #1a4b73;
        }
        
        .autosave-status {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9rem;
            transition: all 0.3s ease;
        }
        
        .autosave-status.saving {
            background: rgba(255, 193, 7, 0.2);
            color: #856404;
        }
        
        .autosave-status.saved {
            background: rgba(40, 167, 69, 0.2);
            color: #155724;
        }
        
        .autosave-status.error {
            background: rgba(220, 53, 69, 0.2);
            color: #721c24;
        }
        
        /* Container principal avec steps */
        .creator-container {
            padding-top: 90px;
            max-width: 1400px;
            margin: 0 auto;
            padding-left: 20px;
            padding-right: 20px;
        }
        
        /* Progress bar des étapes */
        .progress-steps {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
        }
        
        .steps-list {
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
        }
        
        .steps-list::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 0;
            right: 0;
            height: 2px;
            background: #e9ecef;
            z-index: 1;
        }
        
        .step {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: relative;
            z-index: 2;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .step-circle {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: #e9ecef;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            color: #6c757d;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        
        .step.active .step-circle {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            transform: scale(1.1);
        }
        
        .step.completed .step-circle {
            background: #28a745;
            color: white;
        }
        
        .step-label {
            font-size: 0.9rem;
            font-weight: 500;
            text-align: center;
            white-space: nowrap;
        }
        
        /* Contenu des étapes */
        .step-content {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            min-height: 500px;
        }
        
        .step-panel {
            display: none;
        }
        
        .step-panel.active {
            display: block;
            animation: fadeInUp 0.5s ease;
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Formulaires */
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #495057;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.8);
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-select {
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='M6 8l4 4 4-4'/%3e%3c/svg%3e");
            background-position: right 12px center;
            background-repeat: no-repeat;
            background-size: 16px;
            padding-right: 40px;
            appearance: none;
        }
        
        /* Slide builder */
        .slides-container {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        
        .slide-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .slide-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        }
        
        .slide-card.active {
            border-color: #667eea;
        }
        
        .slide-number {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-bottom: 15px;
        }
        
        /* Boutons navigation */
        .step-navigation {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 40px;
            padding-top: 30px;
            border-top: 1px solid #e9ecef;
        }
        
        .btn {
            padding: 12px 30px;
            border: none;
            border-radius: 25px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-outline {
            background: transparent;
            border: 2px solid #667eea;
            color: #667eea;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .creator-header {
                padding: 0 15px;
                height: 60px;
            }
            
            .creator-title h1 {
                font-size: 1.2rem;
            }
            
            .steps-list {
                flex-wrap: wrap;
                gap: 20px;
            }
            
            .step-circle {
                width: 40px;
                height: 40px;
            }
            
            .slides-container {
                grid-template-columns: 1fr;
            }
        }
        
        /* Spinner de chargement */
        .spinner {
            width: 20px;
            height: 20px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <!-- Header avec status -->
    <div class="creator-header">
        <div class="creator-title">
            <span style="font-size: 2rem;">🎪</span>
            <h1>Flying Book Creator</h1>
            <span style="color: #666;">- <?= htmlspecialchars($project_name) ?></span>
        </div>
        
        <div class="autosave-status saved" id="autosaveStatus">
            <span id="autosaveIcon">✅</span>
            <span id="autosaveText">Sauvegardé</span>
        </div>
    </div>

    <!-- Container principal -->
    <div class="creator-container">
        <!-- Progress steps -->
        <div class="progress-steps">
            <div class="steps-list">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div class="step-label">Informations</div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div class="step-label">Configuration</div>
                </div>
                <div class="step" data-step="3">
                    <div class="step-circle">3</div>
                    <div class="step-label">Slides</div>
                </div>
                <div class="step" data-step="4">
                    <div class="step-circle">4</div>
                    <div class="step-label">Aperçu</div>
                </div>
                <div class="step" data-step="5">
                    <div class="step-circle">5</div>
                    <div class="step-label">Publication</div>
                </div>
            </div>
        </div>

        <!-- Contenu des étapes -->
        <div class="step-content">
            <!-- Étape 1: Informations du projet -->
            <div class="step-panel active" id="step-1">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">📋 Informations du projet</h2>
                
                <form id="projectInfoForm">
                    <div class="form-group">
                        <label class="form-label" for="project_name">Nom du projet *</label>
                        <input type="text" class="form-control" id="project_name" name="project_name" 
                               value="<?= htmlspecialchars($project_name) ?>" required>
                        <small style="color: #6c757d;">Ce nom sera utilisé pour l'URL du Flying Book</small>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="title">Titre principal *</label>
                        <input type="text" class="form-control" id="title" name="title" 
                               value="<?= htmlspecialchars($existing_project['title'] ?? '') ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="subtitle">Sous-titre</label>
                        <input type="text" class="form-control" id="subtitle" name="subtitle"
                               value="<?= htmlspecialchars($existing_project['subtitle'] ?? '') ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="4"><?= htmlspecialchars($existing_project['description'] ?? '') ?></textarea>
                    </div>
                </form>
                
                <div class="step-navigation">
                    <div></div>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Suivant <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 2: Configuration -->
            <div class="step-panel" id="step-2">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">⚙️ Configuration du Flying Book</h2>
                
                <form id="configForm">
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 30px;">
                        <div class="form-group">
                            <label class="form-label" for="style_theme">Style visuel</label>
                            <select class="form-control form-select" id="style_theme" name="style_theme">
                                <option value="moderne">Moderne</option>
                                <option value="classique">Classique</option>
                                <option value="corporate">Corporate</option>
                                <option value="minimal">Minimal</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="transition_type">Type de transition</label>
                            <select class="form-control form-select" id="transition_type" name="transition_type">
                                <option value="3d">3D (Recommandé)</option>
                                <option value="fade">Fondu</option>
                                <option value="slide">Glissement</option>
                                <option value="flip">Retournement</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="navigation_type">Navigation</label>
                            <select class="form-control form-select" id="navigation_type" name="navigation_type">
                                <option value="all">Complète (Boutons + Miniatures + Clavier)</option>
                                <option value="buttons">Boutons uniquement</option>
                                <option value="thumbnails">Miniatures uniquement</option>
                                <option value="keyboard">Clavier uniquement</option>
                            </select>
                        </div>
                    </div>
                </form>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Suivant <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 3: Création des slides -->
            <div class="step-panel" id="step-3">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">🎬 Création des slides</h2>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                    <div>
                        <label class="form-label">Nombre de slides</label>
                        <input type="number" class="form-control" id="total_slides" min="1" max="50" value="5" 
                               style="width: 100px; display: inline-block;" onchange="generateSlideCards()">
                    </div>
                    
                    <button class="btn btn-outline" onclick="addSlide()">
                        + Ajouter une slide
                    </button>
                </div>
                
                <div class="slides-container" id="slidesContainer">
                    <!-- Les cartes de slides seront générées ici -->
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Aperçu <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 4: Aperçu -->
            <div class="step-panel" id="step-4">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">👁️ Aperçu du Flying Book</h2>
                
                <div style="text-align: center; padding: 60px 20px;">
                    <div style="font-size: 4rem; margin-bottom: 20px;">🎪</div>
                    <h3>Aperçu en cours de développement</h3>
                    <p style="color: #666; margin: 20px 0;">
                        L'aperçu temps réel sera disponible dans la prochaine version.<br>
                        Vous pouvez continuer vers la publication pour générer votre Flying Book.
                    </p>
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Publier <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 5: Publication -->
            <div class="step-panel" id="step-5">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">🚀 Publication du Flying Book</h2>
                
                <div style="text-align: center; padding: 60px 20px;">
                    <div style="font-size: 4rem; margin-bottom: 20px;">🎉</div>
                    <h3>Projet sauvegardé avec succès !</h3>
                    <p style="color: #666; margin: 20px 0;">
                        Votre Flying Book est prêt.<br>
                        La génération HTML sera implémentée progressivement.
                    </p>
                    
                    <div style="margin-top: 40px;">
                        <button class="btn btn-primary" onclick="window.location.href='/document/flying_book_slides.php'">
                            🎪 Retour au Flying Book original
                        </button>
                    </div>
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="startOver()">
                        Nouveau projet
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentStep = 1;
        let maxStep = 5;
        let projectId = <?= $project_id ? $project_id : 'null' ?>;
        let autoSaveTimer;
        let slidesData = {};

        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            initAutoSave();
            generateSlideCards();
            loadExistingData();
            
            // Détection fermeture navigateur
            window.addEventListener('beforeunload', function(e) {
                autoSave();
                return 'Votre travail non sauvegardé sera perdu.';
            });
        });

        // Navigation entre étapes
        function nextStep() {
            if (validateCurrentStep()) {
                autoSave();
                
                if (currentStep < maxStep) {
                    currentStep++;
                    updateStepDisplay();
                }
            }
        }

        function prevStep() {
            if (currentStep > 1) {
                currentStep--;
                updateStepDisplay();
            }
        }

        function goToStep(step) {
            if (step >= 1 && step <= maxStep) {
                currentStep = step;
                updateStepDisplay();
            }
        }

        function updateStepDisplay() {
            // Mettre à jour les indicateurs d'étapes
            document.querySelectorAll('.step').forEach((step, index) => {
                const stepNum = index + 1;
                step.classList.remove('active', 'completed');
                
                if (stepNum === currentStep) {
                    step.classList.add('active');
                } else if (stepNum < currentStep) {
                    step.classList.add('completed');
                }
            });

            // Afficher le bon panel
            document.querySelectorAll('.step-panel').forEach((panel, index) => {
                panel.classList.remove('active');
                if (index + 1 === currentStep) {
                    panel.classList.add('active');
                }
            });
        }

        // Validation des étapes
        function validateCurrentStep() {
            switch (currentStep) {
                case 1:
                    const projectName = document.getElementById('project_name').value.trim();
                    const title = document.getElementById('title').value.trim();
                    
                    if (!projectName || !title) {
                        alert('Veuillez remplir les champs obligatoires.');
                        return false;
                    }
                    return true;
                    
                default:
                    return true;
            }
        }

        // Génération des cartes de slides
        function generateSlideCards() {
            const totalSlides = parseInt(document.getElementById('total_slides').value);
            const container = document.getElementById('slidesContainer');
            
            container.innerHTML = '';
            
            for (let i = 1; i <= totalSlides; i++) {
                const slideCard = document.createElement('div');
                slideCard.className = 'slide-card';
                slideCard.innerHTML = `
                    <div class="slide-number">${i}</div>
                    <div class="form-group">
                        <label class="form-label">Type de média</label>
                        <select class="form-control form-select" name="media_type_${i}" onchange="updateSlideType(${i}, this.value)">
                            <option value="image">Image</option>
                            <option value="video">Vidéo</option>
                            <option value="audio_image">Audio + Image</option>
                            <option value="text_image">Texte + Image</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Titre (optionnel)</label>
                        <input type="text" class="form-control" name="slide_title_${i}" placeholder="Titre de la slide ${i}">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Fichier principal</label>
                        <input type="file" class="form-control" name="media_file_${i}" accept="image/*,video/*" onchange="handleFileUpload(${i}, this)">
                    </div>
                    <div id="slide_options_${i}"></div>
                `;
                
                container.appendChild(slideCard);
            }
        }

        function updateSlideType(slideNumber, type) {
            const optionsContainer = document.getElementById(`slide_options_${slideNumber}`);
            let additionalOptions = '';
            
            switch (type) {
                case 'video':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" name="autoplay_${slideNumber}"> Lecture automatique
                            </label>
                        </div>
                    `;
                    break;
                case 'audio_image':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">Fichier audio</label>
                            <input type="file" class="form-control" name="audio_file_${slideNumber}" accept="audio/*">
                        </div>
                    `;
                    break;
                case 'text_image':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">Texte superposé</label>
                            <textarea class="form-control" name="text_overlay_${slideNumber}" rows="3"></textarea>
                        </div>
                    `;
                    break;
            }
            
            optionsContainer.innerHTML = additionalOptions;
        }

        function addSlide() {
            const totalSlidesInput = document.getElementById('total_slides');
            const currentTotal = parseInt(totalSlidesInput.value);
            totalSlidesInput.value = currentTotal + 1;
            generateSlideCards();
        }

        // Upload de fichiers
        function handleFileUpload(slideNumber, input) {
            const file = input.files[0];
            if (!file) return;
            
            // Ici, on implementera l'upload progressif vers le serveur
            updateAutosaveStatus('saving', 'Upload en cours...');
            
            // Simulation de l'upload pour l'instant
            setTimeout(() => {
                slidesData[slideNumber] = {
                    ...slidesData[slideNumber],
                    media_file: file.name,
                    media_type: document.querySelector(`[name="media_type_${slideNumber}"]`).value
                };
                
                updateAutosaveStatus('saved', 'Fichier uploadé');
                autoSave();
            }, 1000);
        }

        // Système d'auto-sauvegarde
        function initAutoSave() {
            // Auto-save toutes les 30 secondes
            autoSaveTimer = setInterval(autoSave, 30000);
            
            // Sauvegarde sur changement de champ
            document.addEventListener('input', function(e) {
                if (e.target.matches('input, textarea, select')) {
                    clearTimeout(autoSaveTimer);
                    autoSaveTimer = setTimeout(autoSave, 2000);
                }
            });
        }

        function autoSave() {
            const formData = collectFormData();
            
            updateAutosaveStatus('saving', 'Sauvegarde...');
            
            fetch(window.location.href, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=save_project&data=${encodeURIComponent(JSON.stringify(formData))}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (data.project_id && !projectId) {
                        projectId = data.project_id;
                    }
                    updateAutosaveStatus('saved', 'Sauvegardé automatiquement');
                } else {
                    updateAutosaveStatus('error', 'Erreur de sauvegarde');
                }
            })
            .catch(error => {
                updateAutosaveStatus('error', 'Erreur réseau');
                console.error('Erreur auto-save:', error);
            });
        }

        function collectFormData() {
            const formElements = document.querySelectorAll('input, textarea, select');
            const data = {};
            
            formElements.forEach(element => {
                if (element.name) {
                    data[element.name] = element.value;
                }
            });
            
            data.slides = slidesData;
            return data;
        }

        function updateAutosaveStatus(status, message) {
            const statusElement = document.getElementById('autosaveStatus');
            const iconElement = document.getElementById('autosaveIcon');
            const textElement = document.getElementById('autosaveText');
            
            statusElement.className = `autosave-status ${status}`;
            
            switch (status) {
                case 'saving':
                    iconElement.innerHTML = '<div class="spinner"></div>';
                    break;
                case 'saved':
                    iconElement.textContent = '✅';
                    break;
                case 'error':
                    iconElement.textContent = '❌';
                    break;
            }
            
            textElement.textContent = message;
        }

        function loadExistingData() {
            // Charger les données existantes si disponibles
            <?php if ($existing_project): ?>
            document.getElementById('style_theme').value = '<?= $existing_project['style_theme'] ?? 'moderne' ?>';
            document.getElementById('transition_type').value = '<?= $existing_project['transition_type'] ?? '3d' ?>';
            document.getElementById('navigation_type').value = '<?= $existing_project['navigation_type'] ?? 'all' ?>';
            <?php endif; ?>
        }

        function startOver() {
            if (confirm('Êtes-vous sûr de vouloir créer un nouveau projet ?')) {
                window.location.href = window.location.pathname;
            }
        }

        // Click handlers pour les étapes
        document.querySelectorAll('.step').forEach(step => {
            step.addEventListener('click', function() {
                const stepNum = parseInt(this.dataset.step);
                goToStep(stepNum);
            });
        });
    </script>
</body>
</html>