<?php
/**
 * Flying Book Creator - WIZARD BUSINESS READY
 * /document/creator/index.php
 */

require_once __DIR__ . '/../../config/auth_check.php';
checkAuth();

$user = getUser();
$pdo = DatabaseConfig::getConnection();

// Récupérer le plan utilisateur
$stmt = $pdo->prepare("SELECT * FROM flying_book_subscriptions WHERE user_id = ? AND status = 'active' ORDER BY id DESC LIMIT 1");
$stmt->execute([$user['id']]);
$userPlan = $stmt->fetch() ?: ['plan_type' => 'free', 'limits' => json_encode(['max_projects' => 3, 'max_slides' => 10, 'max_storage_mb' => 100])];
$limits = json_decode($userPlan['limits'], true);

// Vérifier les projets existants
$stmt = $pdo->prepare("SELECT COUNT(*) as project_count FROM flying_book_projects WHERE user_id = ? AND status != 'archived'");
$stmt->execute([$user['id']]);
$projectCount = $stmt->fetchColumn();

// Récupérer le nom du projet depuis l'URL
$project_name = $_GET['name'] ?? 'nouveau_projet_' . time();
$project_id = $_GET['id'] ?? null;

// Si reprise d'un projet existant
$existing_project = null;
if ($project_id) {
    $stmt = $pdo->prepare("SELECT * FROM flying_book_projects WHERE id = ? AND user_id = ?");
    $stmt->execute([$project_id, $user['id']]);
    $existing_project = $stmt->fetch();
    
    if ($existing_project) {
        $project_name = $existing_project['project_name'];
    }
}

// Gestion AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        $response = ['success' => false, 'message' => ''];
        
        switch ($_POST['action']) {
            case 'check_limits':
                // Vérifier les limites du plan
                $canCreate = true;
                $message = '';
                
                if ($limits['max_projects'] != -1 && $projectCount >= $limits['max_projects']) {
                    $canCreate = false;
                    $message = "Limite de projets atteinte ({$limits['max_projects']}). Upgradez votre plan !";
                }
                
                $response = ['success' => $canCreate, 'message' => $message, 'current_count' => $projectCount, 'limit' => $limits['max_projects']];
                break;
                
            case 'save_project':
                // Sauvegarde des métadonnées du projet
                $data = json_decode($_POST['data'], true);
                
                // Validation business
                if ($limits['max_projects'] != -1 && $projectCount >= $limits['max_projects'] && !$project_id) {
                    throw new Exception("Limite de projets atteinte. Upgradez votre plan !");
                }
                
                if (!$project_id) {
                    // Nouveau projet
                    $slug = generateSlug($data['project_name']);
                    $folder = 'project_' . uniqid();
                    
                    // Vérifier unicité du slug
                    $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE project_slug = ?");
                    $stmt->execute([$slug]);
                    if ($stmt->fetch()) {
                        $slug .= '_' . time();
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO flying_book_projects 
                        (user_id, project_name, project_slug, title, subtitle, description, 
                         style_theme, transition_type, navigation_type, project_folder, plan_type, status)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
                    ");
                    $stmt->execute([
                        $user['id'], $data['project_name'], $slug, $data['title'],
                        $data['subtitle'] ?? null, $data['description'] ?? null,
                        $data['style_theme'] ?? 'moderne', $data['transition_type'] ?? '3d',
                        $data['navigation_type'] ?? 'all', $folder, $userPlan['plan_type']
                    ]);
                    
                    $project_id = $pdo->lastInsertId();
                    createProjectFolder($folder);
                    
                } else {
                    // Mise à jour projet existant
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET title = ?, subtitle = ?, description = ?, 
                            style_theme = ?, transition_type = ?, navigation_type = ?,
                            updated_at = NOW()
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([
                        $data['title'], $data['subtitle'] ?? null, $data['description'] ?? null,
                        $data['style_theme'] ?? 'moderne', $data['transition_type'] ?? '3d',
                        $data['navigation_type'] ?? 'all', $project_id, $user['id']
                    ]);
                }
                
                $response = ['success' => true, 'project_id' => $project_id, 'message' => 'Projet sauvegardé'];
                break;
                
            case 'save_slide':
                // Sauvegarde d'une slide individuelle
                $slide_data = json_decode($_POST['data'], true);
                
                // Vérification limites slides
                if ($limits['max_slides'] != -1 && $slide_data['slide_number'] > $limits['max_slides']) {
                    throw new Exception("Limite de slides atteinte ({$limits['max_slides']}). Upgradez votre plan !");
                }
                
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_slides 
                    (project_id, slide_number, slide_title, media_type, media_file, 
                     background_audio, auto_duration, text_overlay, display_order)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    slide_title = VALUES(slide_title),
                    media_type = VALUES(media_type),
                    media_file = VALUES(media_file),
                    background_audio = VALUES(background_audio),
                    auto_duration = VALUES(auto_duration),
                    text_overlay = VALUES(text_overlay),
                    display_order = VALUES(display_order),
                    updated_at = NOW()
                ");
                
                $stmt->execute([
                    $project_id, $slide_data['slide_number'], $slide_data['title'] ?? null,
                    $slide_data['media_type'], $slide_data['media_file'],
                    $slide_data['background_audio'] ?? null, $slide_data['auto_duration'] ?? null,
                    $slide_data['text_overlay'] ?? null, $slide_data['display_order'] ?? 0
                ]);
                
                $response = ['success' => true, 'message' => 'Slide sauvegardée'];
                break;
                
            case 'upload_file':
                // Upload de fichier
                if (!isset($_FILES['file'])) {
                    throw new Exception('Aucun fichier uploadé');
                }
                
                $file = $_FILES['file'];
                $projectFolder = $_POST['project_folder'];
                
                // Validation taille fichier selon plan
                $maxSize = $limits['max_storage_mb'] * 1024 * 1024; // Convertir en bytes
                if ($file['size'] > $maxSize) {
                    throw new Exception("Fichier trop volumineux. Limite: {$limits['max_storage_mb']}MB");
                }
                
                $uploadResult = handleFileUpload($file, $projectFolder, $project_id, $user['id']);
                $response = ['success' => true, 'file_data' => $uploadResult];
                break;
                
            case 'load_templates':
                // Charger les templates disponibles
                $category = $_POST['category'] ?? 'all';
                $isPremium = $userPlan['plan_type'] !== 'free';
                
                $sql = "SELECT * FROM flying_book_templates WHERE is_public = 1";
                $params = [];
                
                if ($category !== 'all') {
                    $sql .= " AND category = ?";
                    $params[] = $category;
                }
                
                if (!$isPremium) {
                    $sql .= " AND is_premium = 0";
                }
                
                $sql .= " ORDER BY is_featured DESC, rating DESC LIMIT 20";
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $templates = $stmt->fetchAll();
                
                $response = ['success' => true, 'templates' => $templates];
                break;
                
            case 'autosave':
                // Auto-sauvegarde de l'état complet
                $state = $_POST['state'];
                
                $_SESSION['flying_book_autosave'] = [
                    'project_id' => $project_id,
                    'timestamp' => time(),
                    'state' => $state
                ];
                
                $response = ['success' => true, 'message' => 'Auto-sauvegarde OK'];
                break;
        }
        
        echo json_encode($response);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
}

// Fonctions utilitaires
function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}

function createProjectFolder($folder_name) {
    $base_path = __DIR__ . '/../projects/' . $folder_name;
    if (!file_exists($base_path)) {
        mkdir($base_path, 0755, true);
        mkdir($base_path . '/images', 0755, true);
        mkdir($base_path . '/videos', 0755, true);
        mkdir($base_path . '/audio', 0755, true);
        mkdir($base_path . '/thumbnails', 0755, true);
    }
    return $base_path;
}

function handleFileUpload($file, $projectFolder, $projectId, $userId) {
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'video/mp4', 'video/webm', 'audio/mpeg', 'audio/wav'];
    
    if (!in_array($file['type'], $allowedTypes)) {
        throw new Exception('Type de fichier non autorisé');
    }
    
    $uploadDir = __DIR__ . '/../projects/' . $projectFolder . '/';
    
    // Déterminer le sous-dossier selon le type
    if (strpos($file['type'], 'image/') === 0) {
        $uploadDir .= 'images/';
    } elseif (strpos($file['type'], 'video/') === 0) {
        $uploadDir .= 'videos/';
    } else {
        $uploadDir .= 'audio/';
    }
    
    $filename = uniqid() . '_' . basename($file['name']);
    $filepath = $uploadDir . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Erreur lors de l\'upload');
    }
    
    // Enregistrer en BDD
    $pdo = DatabaseConfig::getConnection();
    $stmt = $pdo->prepare("
        INSERT INTO flying_book_media 
        (project_id, user_id, original_filename, stored_filename, file_path, 
         file_type, mime_type, file_size, upload_status)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'ready')
    ");
    
    $fileType = strpos($file['type'], 'image/') === 0 ? 'image' : 
                (strpos($file['type'], 'video/') === 0 ? 'video' : 'audio');
    
    $stmt->execute([
        $projectId, $userId, $file['name'], $filename, $filepath,
        $fileType, $file['type'], $file['size']
    ]);
    
    return [
        'filename' => $filename,
        'original_name' => $file['name'],
        'file_type' => $fileType,
        'file_size' => $file['size'],
        'url' => '/document/projects/' . $projectFolder . '/' . $fileType . 's/' . $filename
    ];
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - <?= htmlspecialchars($project_name) ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            min-height: 100vh;
            color: #333;
            overflow-x: hidden;
        }
        
        /* Header avec plan utilisateur */
        .creator-header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 70px;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 30px;
            z-index: 1000;
            box-shadow: 0 4px 30px rgba(0, 0, 0, 0.1);
        }
        
        .creator-title {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .creator-title h1 {
            font-size: 1.5rem;
            color: #1a4b73;
        }
        
        .project-name {
            color: #666;
            font-weight: 500;
        }
        
        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .plan-badge {
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .plan-free { background: rgba(108, 117, 125, 0.2); color: #6c757d; }
        .plan-pro { background: rgba(102, 126, 234, 0.2); color: #667eea; }
        .plan-business { background: rgba(255, 107, 107, 0.2); color: #ff6b6b; }
        
        .autosave-status {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9rem;
            transition: all 0.3s ease;
        }
        
        .autosave-status.saving { background: rgba(255, 193, 7, 0.2); color: #856404; }
        .autosave-status.saved { background: rgba(40, 167, 69, 0.2); color: #155724; }
        .autosave-status.error { background: rgba(220, 53, 69, 0.2); color: #721c24; }
        
        /* Upgrade modal */
        .upgrade-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 2000;
        }
        
        .upgrade-modal.active { display: flex; }
        
        .upgrade-content {
            background: white;
            padding: 40px;
            border-radius: 20px;
            max-width: 500px;
            text-align: center;
            position: relative;
        }
        
        .upgrade-close {
            position: absolute;
            top: 15px;
            right: 20px;
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #999;
        }
        
        /* Container principal */
        .creator-container {
            padding-top: 90px;
            max-width: 1400px;
            margin: 0 auto;
            padding-left: 20px;
            padding-right: 20px;
        }
        
        /* Progress bar des étapes */
        .progress-steps {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
        }
        
        .steps-list {
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
        }
        
        .steps-list::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 0;
            right: 0;
            height: 2px;
            background: #e9ecef;
            z-index: 1;
        }
        
        .step {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: relative;
            z-index: 2;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .step-circle {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: #e9ecef;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            color: #6c757d;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        
        .step.active .step-circle {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            transform: scale(1.1);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }
        
        .step.completed .step-circle {
            background: #28a745;
            color: white;
        }
        
        .step-label {
            font-size: 0.9rem;
            font-weight: 500;
            text-align: center;
            white-space: nowrap;
        }
        
        /* Contenu des étapes */
        .step-content {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            min-height: 600px;
        }
        
        .step-panel {
            display: none;
        }
        
        .step-panel.active {
            display: block;
            animation: fadeInUp 0.5s ease;
        }
        
        @keyframes fadeInUp {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Formulaires */
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #495057;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.8);
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }
        
        .form-select {
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='M6 8l4 4 4-4'/%3e%3c/svg%3e");
            background-position: right 12px center;
            background-repeat: no-repeat;
            background-size: 16px;
            padding-right: 40px;
            appearance: none;
        }
        
        /* Templates grid */
        .templates-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 25px;
            margin-top: 30px;
        }
        
        .template-card {
            background: white;
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            cursor: pointer;
            border: 2px solid transparent;
        }
        
        .template-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.15);
        }
        
        .template-card.selected {
            border-color: #667eea;
            transform: translateY(-5px);
        }
        
        .template-preview {
            height: 150px;
            background: linear-gradient(45deg, #f8f9fa, #e9ecef);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 3rem;
        }
        
        .template-info {
            padding: 20px;
        }
        
        .template-name {
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }
        
        .template-description {
            font-size: 0.9rem;
            color: #666;
            margin-bottom: 15px;
        }
        
        .template-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .template-price {
            font-weight: 600;
            color: #667eea;
        }
        
        .template-price.free {
            color: #28a745;
        }
        
        .template-rating {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 0.9rem;
            color: #666;
        }
        
        /* Slide builder */
        .slides-container {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 25px;
            margin-top: 30px;
        }
        
        .slide-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
        }
        
        .slide-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        }
        
        .slide-card.active {
            border-color: #667eea;
        }
        
        .slide-number {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            width: 35px;
            height: 35px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-bottom: 20px;
            position: absolute;
            top: -10px;
            right: -10px;
        }
        
        .slide-preview {
            height: 120px;
            background: #f8f9fa;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 20px;
            border: 2px dashed #dee2e6;
            transition: all 0.3s ease;
        }
        
        .slide-preview.has-media {
            border: none;
            background: none;
        }
        
        .slide-preview img, .slide-preview video {
            max-width: 100%;
            max-height: 100%;
            border-radius: 8px;
        }
        
        /* Upload area */
        .upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 30px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            background: rgba(248, 249, 250, 0.5);
        }
        
        .upload-area:hover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.05);
        }
        
        .upload-area.dragover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.1);
            transform: scale(1.02);
        }
        
        /* Boutons */
        .btn {
            padding: 12px 30px;
            border: none;
            border-radius: 25px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            font-size: 0.95rem;
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-outline {
            background: transparent;
            border: 2px solid #667eea;
            color: #667eea;
        }
        
        .btn-success {
            background: linear-gradient(45deg, #28a745, #20c997);
            color: white;
        }
        
        .btn-upgrade {
            background: linear-gradient(45deg, #ff6b6b, #feca57);
            color: white;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
        
        /* Navigation */
        .step-navigation {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 40px;
            padding-top: 30px;
            border-top: 1px solid #e9ecef;
        }
        
        /* Limits warning */
        .limits-warning {
            background: rgba(255, 193, 7, 0.1);
            border: 1px solid rgba(255, 193, 7, 0.3);
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .limits-error {
            background: rgba(220, 53, 69, 0.1);
            border: 1px solid rgba(220, 53, 69, 0.3);
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .creator-header {
                padding: 0 15px;
                height: 60px;
                flex-wrap: wrap;
            }
            
            .creator-title h1 {
                font-size: 1.2rem;
            }
            
            .header-right {
                gap: 10px;
            }
            
            .steps-list {
                flex-wrap: wrap;
                gap: 15px;
            }
            
            .step-circle {
                width: 40px;
                height: 40px;
            }
            
            .templates-grid,
            .slides-container {
                grid-template-columns: 1fr;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
        }
        
        /* Spinner et loading */
        .spinner {
            width: 20px;
            height: 20px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.9);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 1500;
        }
        
        .loading-overlay.active {
            display: flex;
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="creator-header">
        <div class="creator-title">
            <span style="font-size: 2rem;">🎪</span>
            <h1>Flying Book Creator</h1>
            <span class="project-name">- <?= htmlspecialchars($project_name) ?></span>
        </div>
        
        <div class="header-right">
            <div class="plan-badge plan-<?= $userPlan['plan_type'] ?>">
                <?= strtoupper($userPlan['plan_type']) ?>
            </div>
            
            <div class="autosave-status saved" id="autosaveStatus">
                <span id="autosaveIcon">✅</span>
                <span id="autosaveText">Sauvegardé</span>
            </div>
        </div>
    </div>

    <!-- Loading overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div style="text-align: center;">
            <div class="spinner" style="width: 40px; height: 40px; margin-bottom: 20px;"></div>
            <p>Traitement en cours...</p>
        </div>
    </div>

    <!-- Upgrade modal -->
    <div class="upgrade-modal" id="upgradeModal">
        <div class="upgrade-content">
            <button class="upgrade-close" onclick="closeUpgradeModal()">×</button>
            <div style="font-size: 3rem; margin-bottom: 20px;">🚀</div>
            <h2 style="margin-bottom: 15px; color: #1a4b73;">Limite atteinte !</h2>
            <p style="margin-bottom: 25px; color: #666;">
                Votre plan <strong><?= strtoupper($userPlan['plan_type']) ?></strong> est limité.<br>
                Upgradez pour débloquer plus de fonctionnalités !
            </p>
            
            <div style="display: flex; gap: 15px; justify-content: center; margin-top: 30px;">
                <button class="btn btn-outline" onclick="closeUpgradeModal()">Annuler</button>
                <button class="btn btn-upgrade">💎 Upgrader maintenant</button>
            </div>
        </div>
    </div>

    <!-- Container principal -->
    <div class="creator-container">
        <!-- Progress steps -->
        <div class="progress-steps">
            <div class="steps-list">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div class="step-label">Projet</div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div class="step-label">Template</div>
                </div>
                <div class="step" data-step="3">
                    <div class="step-circle">3</div>
                    <div class="step-label">Configuration</div>
                </div>
                <div class="step" data-step="4">
                    <div class="step-circle">4</div>
                    <div class="step-label">Slides</div>
                </div>
                <div class="step" data-step="5">
                    <div class="step-circle">5</div>
                    <div class="step-label">Publication</div>
                </div>
            </div>
        </div>

        <!-- Contenu des étapes -->
        <div class="step-content">
            <!-- Étape 1: Informations du projet -->
            <div class="step-panel active" id="step-1">
                <h2 style="margin-bottom: 30px; color: #1a4b73; display: flex; align-items: center; gap: 15px;">
                    📋 Informations du projet
                    <span style="font-size: 1rem; color: #666; font-weight: normal;">
                        (<?= $projectCount ?>/<?= $limits['max_projects'] == -1 ? '∞' : $limits['max_projects'] ?> projets)
                    </span>
                </h2>
                
                <?php if ($limits['max_projects'] != -1 && $projectCount >= $limits['max_projects'] && !$project_id): ?>
                <div class="limits-error">
                    <span style="font-size: 1.5rem;">⚠️</span>
                    <div>
                        <strong>Limite de projets atteinte !</strong><br>
                        Vous avez atteint la limite de <?= $limits['max_projects'] ?> projets pour votre plan <?= strtoupper($userPlan['plan_type']) ?>.
                        <button class="btn btn-upgrade" style="margin-left: 15px;" onclick="showUpgradeModal()">
                            💎 Upgrader
                        </button>
                    </div>
                </div>
                <?php endif; ?>
                
                <form id="projectInfoForm">
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label" for="project_name">Nom du projet *</label>
                            <input type="text" class="form-control" id="project_name" name="project_name" 
                                   value="<?= htmlspecialchars($project_name) ?>" required>
                            <small style="color: #6c757d;">Ce nom sera utilisé pour l'URL du Flying Book</small>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="title">Titre principal *</label>
                            <input type="text" class="form-control" id="title" name="title" 
                                   value="<?= htmlspecialchars($existing_project['title'] ?? '') ?>" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="subtitle">Sous-titre</label>
                        <input type="text" class="form-control" id="subtitle" name="subtitle"
                               value="<?= htmlspecialchars($existing_project['subtitle'] ?? '') ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="description">Description du projet</label>
                        <textarea class="form-control" id="description" name="description" rows="4" 
                                  placeholder="Décrivez brièvement votre Flying Book..."><?= htmlspecialchars($existing_project['description'] ?? '') ?></textarea>
                    </div>
                </form>
                
                <div class="step-navigation">
                    <div></div>
                    <button class="btn btn-primary" onclick="nextStep()" id="step1NextBtn">
                        Choisir un template <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 2: Choix du template -->
            <div class="step-panel" id="step-2">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">🎨 Choisir un template</h2>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
                    <div style="display: flex; gap: 15px;">
                        <select class="form-control form-select" id="templateCategory" style="width: 200px;" onchange="loadTemplates()">
                            <option value="all">Toutes catégories</option>
                            <option value="business">Business</option>
                            <option value="education">Éducation</option>
                            <option value="portfolio">Portfolio</option>
                            <option value="marketing">Marketing</option>
                            <option value="sales">Ventes</option>
                        </select>
                        
                        <?php if ($userPlan['plan_type'] === 'free'): ?>
                        <div style="display: flex; align-items: center; gap: 10px; color: #666;">
                            <span>💎 Templates premium disponibles avec</span>
                            <button class="btn btn-upgrade" style="padding: 6px 15px; font-size: 0.8rem;" onclick="showUpgradeModal()">
                                Upgrade
                            </button>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="templates-grid" id="templatesGrid">
                    <!-- Templates chargés dynamiquement -->
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()" id="step2NextBtn">
                        Configurer <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 3: Configuration -->
            <div class="step-panel" id="step-3">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">⚙️ Configuration du Flying Book</h2>
                
                <form id="configForm">
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label" for="style_theme">Style visuel</label>
                            <select class="form-control form-select" id="style_theme" name="style_theme">
                                <option value="moderne">🎨 Moderne</option>
                                <option value="classique">📚 Classique</option>
                                <option value="corporate">🏢 Corporate</option>
                                <option value="minimal">⚪ Minimal</option>
                                <?php if ($userPlan['plan_type'] !== 'free'): ?>
                                <option value="premium">💎 Premium</option>
                                <?php endif; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="transition_type">Type de transition</label>
                            <select class="form-control form-select" id="transition_type" name="transition_type">
                                <option value="3d">🎪 3D (Recommandé)</option>
                                <option value="fade">🌅 Fondu</option>
                                <option value="slide">➡️ Glissement</option>
                                <option value="flip">🔄 Retournement</option>
                                <?php if ($userPlan['plan_type'] !== 'free'): ?>
                                <option value="premium_3d">💎 3D Premium</option>
                                <?php endif; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="navigation_type">Navigation</label>
                            <select class="form-control form-select" id="navigation_type" name="navigation_type">
                                <option value="all">🎯 Complète (Recommandé)</option>
                                <option value="buttons">🔘 Boutons uniquement</option>
                                <option value="thumbnails">🖼️ Miniatures uniquement</option>
                                <option value="keyboard">⌨️ Clavier uniquement</option>
                            </select>
                        </div>
                    </div>
                    
                    <?php if ($userPlan['plan_type'] !== 'free'): ?>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label" for="watermark_enabled">
                                <input type="checkbox" id="watermark_enabled" name="watermark_enabled"> 
                                Afficher un watermark
                            </label>
                            <input type="text" class="form-control" id="watermark_text" name="watermark_text" 
                                   placeholder="Texte du watermark" style="margin-top: 10px; display: none;">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="analytics_enabled">
                                <input type="checkbox" id="analytics_enabled" name="analytics_enabled" checked> 
                                Activer les analytics
                            </label>
                        </div>
                    </div>
                    <?php endif; ?>
                </form>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Créer les slides <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 4: Création des slides -->
            <div class="step-panel" id="step-4">
                <h2 style="margin-bottom: 30px; color: #1a4b73; display: flex; align-items: center; gap: 15px;">
                    🎬 Création des slides
                    <span style="font-size: 1rem; color: #666; font-weight: normal;">
                        (Max: <?= $limits['max_slides'] == -1 ? '∞' : $limits['max_slides'] ?> slides)
                    </span>
                </h2>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; flex-wrap: wrap; gap: 15px;">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <label class="form-label" style="margin: 0;">Nombre de slides</label>
                        <input type="number" class="form-control" id="total_slides" min="1" max="<?= $limits['max_slides'] == -1 ? 50 : $limits['max_slides'] ?>" value="5" 
                               style="width: 100px;" onchange="generateSlideCards()">
                    </div>
                    
                    <button class="btn btn-outline" onclick="addSlide()" id="addSlideBtn">
                        + Ajouter une slide
                    </button>
                </div>
                
                <div class="slides-container" id="slidesContainer">
                    <!-- Les cartes de slides seront générées ici -->
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <button class="btn btn-primary" onclick="nextStep()">
                        Publier <span>→</span>
                    </button>
                </div>
            </div>

            <!-- Étape 5: Publication -->
            <div class="step-panel" id="step-5">
                <h2 style="margin-bottom: 30px; color: #1a4b73;">🚀 Publication du Flying Book</h2>
                
                <div style="text-align: center; padding: 60px 20px;">
                    <div style="font-size: 4rem; margin-bottom: 20px;">🎉</div>
                    <h3 style="margin-bottom: 15px;">Félicitations !</h3>
                    <p style="color: #666; margin: 20px 0; font-size: 1.1rem;">
                        Votre Flying Book <strong><?= htmlspecialchars($project_name) ?></strong> est prêt !<br>
                        La génération HTML sera disponible dans la prochaine version.
                    </p>
                    
                    <div style="background: rgba(102, 126, 234, 0.1); padding: 30px; border-radius: 15px; margin: 30px 0; max-width: 500px; margin-left: auto; margin-right: auto;">
                        <h4 style="color: #667eea; margin-bottom: 15px;">🎯 Prochaines fonctionnalités</h4>
                        <ul style="text-align: left; color: #666; line-height: 1.8;">
                            <li>✨ Génération HTML automatique</li>
                            <li>📊 Analytics en temps réel</li>
                            <li>🔗 Partage sécurisé avec liens</li>
                            <li>📱 Export mobile optimisé</li>
                            <li>🎨 Éditeur visuel avancé</li>
                        </ul>
                    </div>
                    
                    <div style="display: flex; gap: 20px; justify-content: center; margin-top: 40px; flex-wrap: wrap;">
                        <button class="btn btn-primary" onclick="window.location.href='/document/flying_book_slides.php'">
                            🎪 Retour au Flying Book
                        </button>
                        <button class="btn btn-success" onclick="startOver()">
                            ✨ Nouveau projet
                        </button>
                    </div>
                </div>
                
                <div class="step-navigation">
                    <button class="btn btn-secondary" onclick="prevStep()">
                        <span>←</span> Précédent
                    </button>
                    <div></div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Variables globales
        let currentStep = 1;
        let maxStep = 5;
        let projectId = <?= $project_id ? $project_id : 'null' ?>;
        let userPlan = <?= json_encode($userPlan) ?>;
        let limits = <?= json_encode($limits) ?>;
        let autoSaveTimer;
        let slidesData = {};
        let selectedTemplate = null;

        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            initAutoSave();
            loadTemplates();
            generateSlideCards();
            loadExistingData();
            setupEventListeners();
            
            // Vérification des limites au démarrage
            checkLimits();
        });

        // Event listeners
        function setupEventListeners() {
            // Watermark toggle
            document.getElementById('watermark_enabled')?.addEventListener('change', function() {
                document.getElementById('watermark_text').style.display = this.checked ? 'block' : 'none';
            });
            
            // Détection fermeture navigateur
            window.addEventListener('beforeunload', function(e) {
                autoSave();
                return 'Votre travail non sauvegardé sera perdu.';
            });
            
            // Click handlers pour les étapes
            document.querySelectorAll('.step').forEach(step => {
                step.addEventListener('click', function() {
                    const stepNum = parseInt(this.dataset.step);
                    if (!this.classList.contains('completed') && stepNum !== currentStep + 1) return;
                    goToStep(stepNum);
                });
            });
        }

        // Navigation entre étapes
        function nextStep() {
            if (validateCurrentStep()) {
                autoSave();
                
                if (currentStep < maxStep) {
                    currentStep++;
                    updateStepDisplay();
                    
                    // Actions spécifiques par étape
                    if (currentStep === 2) {
                        loadTemplates();
                    } else if (currentStep === 4) {
                        generateSlideCards();
                    }
                }
            }
        }

        function prevStep() {
            if (currentStep > 1) {
                currentStep--;
                updateStepDisplay();
            }
        }

        function goToStep(step) {
            if (step >= 1 && step <= maxStep) {
                currentStep = step;
                updateStepDisplay();
            }
        }

        function updateStepDisplay() {
            // Mettre à jour les indicateurs d'étapes
            document.querySelectorAll('.step').forEach((step, index) => {
                const stepNum = index + 1;
                step.classList.remove('active', 'completed');
                
                if (stepNum === currentStep) {
                    step.classList.add('active');
                } else if (stepNum < currentStep) {
                    step.classList.add('completed');
                }
            });

            // Afficher le bon panel
            document.querySelectorAll('.step-panel').forEach((panel, index) => {
                panel.classList.remove('active');
                if (index + 1 === currentStep) {
                    panel.classList.add('active');
                }
            });
        }

        // Validation des étapes
        function validateCurrentStep() {
            switch (currentStep) {
                case 1:
                    // Vérifier les limites
                    if (!checkLimits()) {
                        showUpgradeModal();
                        return false;
                    }
                    
                    const projectName = document.getElementById('project_name').value.trim();
                    const title = document.getElementById('title').value.trim();
                    
                    if (!projectName || !title) {
                        alert('Veuillez remplir les champs obligatoires.');
                        return false;
                    }
                    return true;
                    
                case 2:
                    // Template sélectionné (optionnel)
                    return true;
                    
                default:
                    return true;
            }
        }

        // Vérification des limites
        function checkLimits() {
            const projectCount = <?= $projectCount ?>;
            
            if (limits.max_projects !== -1 && projectCount >= limits.max_projects && !projectId) {
                return false;
            }
            
            return true;
        }

        // Chargement des templates
        function loadTemplates() {
            const category = document.getElementById('templateCategory')?.value || 'all';
            
            showLoading(true);
            
            fetch(window.location.href, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=load_templates&category=${encodeURIComponent(category)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    displayTemplates(data.templates);
                } else {
                    console.error('Erreur chargement templates:', data.message);
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
            })
            .finally(() => {
                showLoading(false);
            });
        }

        function displayTemplates(templates) {
            const grid = document.getElementById('templatesGrid');
            if (!grid) return;
            
            grid.innerHTML = '';
            
            // Template vide par défaut
            const emptyTemplate = document.createElement('div');
            emptyTemplate.className = 'template-card';
            emptyTemplate.onclick = () => selectTemplate(null);
            emptyTemplate.innerHTML = `
                <div class="template-preview">📄</div>
                <div class="template-info">
                    <div class="template-name">Template vide</div>
                    <div class="template-description">Commencer avec un projet vierge</div>
                    <div class="template-meta">
                        <div class="template-price free">Gratuit</div>
                        <div class="template-rating">⭐ —</div>
                    </div>
                </div>
            `;
            grid.appendChild(emptyTemplate);
            
            // Templates de la BDD
            templates.forEach(template => {
                const card = document.createElement('div');
                card.className = 'template-card';
                card.onclick = () => selectTemplate(template);
                
                const isPremium = template.is_premium == 1;
                const canUse = userPlan.plan_type !== 'free' || !isPremium;
                
                card.innerHTML = `
                    <div class="template-preview">${template.template_name.substring(0, 2)}</div>
                    <div class="template-info">
                        <div class="template-name">
                            ${template.template_name}
                            ${isPremium ? ' 💎' : ''}
                        </div>
                        <div class="template-description">${template.template_description}</div>
                        <div class="template-meta">
                            <div class="template-price ${isPremium ? '' : 'free'}">
                                ${isPremium ? template.price + '€' : 'Gratuit'}
                            </div>
                            <div class="template-rating">
                                ⭐ ${parseFloat(template.rating).toFixed(1)}
                            </div>
                        </div>
                    </div>
                    ${!canUse ? '<div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); display: flex; align-items: center; justify-content: center; color: white; font-weight: 600;">PREMIUM</div>' : ''}
                `;
                
                if (!canUse) {
                    card.style.opacity = '0.6';
                    card.onclick = () => showUpgradeModal();
                }
                
                grid.appendChild(card);
            });
        }

        function selectTemplate(template) {
            // Déselectionner tous
            document.querySelectorAll('.template-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            // Sélectionner le cliqué
            event.currentTarget.classList.add('selected');
            selectedTemplate = template;
            
            // Appliquer les paramètres du template si disponible
            if (template && template.template_config) {
                try {
                    const config = JSON.parse(template.template_config);
                    
                    if (config.style) {
                        document.getElementById('style_theme').value = config.style;
                    }
                    if (config.transitions) {
                        document.getElementById('transition_type').value = config.transitions;
                    }
                } catch (e) {
                    console.error('Erreur parsing template config:', e);
                }
            }
        }

        // Génération des cartes de slides
        function generateSlideCards() {
            const totalSlides = parseInt(document.getElementById('total_slides').value);
            const container = document.getElementById('slidesContainer');
            
            if (!container) return;
            
            // Vérifier les limites
            if (limits.max_slides !== -1 && totalSlides > limits.max_slides) {
                document.getElementById('total_slides').value = limits.max_slides;
                showUpgradeModal();
                return;
            }
            
            container.innerHTML = '';
            
            for (let i = 1; i <= totalSlides; i++) {
                const slideCard = document.createElement('div');
                slideCard.className = 'slide-card';
                slideCard.innerHTML = `
                    <div class="slide-number">${i}</div>
                    
                    <div class="slide-preview" id="preview_${i}">
                        <div style="color: #999; text-align: center;">
                            <div style="font-size: 2rem; margin-bottom: 10px;">📁</div>
                            <div>Cliquez pour ajouter</div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Type de média</label>
                        <select class="form-control form-select" name="media_type_${i}" onchange="updateSlideType(${i}, this.value)">
                            <option value="image">🖼️ Image</option>
                            <option value="video">🎬 Vidéo</option>
                            <option value="audio_image">🎵 Audio + Image</option>
                            <option value="text_image">📝 Texte + Image</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Titre (optionnel)</label>
                        <input type="text" class="form-control" name="slide_title_${i}" placeholder="Titre de la slide ${i}">
                    </div>
                    
                    <div class="upload-area" onclick="triggerFileUpload(${i})">
                        <div style="font-size: 2rem; margin-bottom: 10px;">📤</div>
                        <div>Cliquer ou glisser un fichier</div>
                        <input type="file" id="file_${i}" style="display: none;" accept="image/*,video/*,audio/*" onchange="handleFileUpload(${i}, this)">
                    </div>
                    
                    <div id="slide_options_${i}"></div>
                `;
                
                container.appendChild(slideCard);
                
                // Setup drag & drop
                setupDragDrop(slideCard.querySelector('.upload-area'), i);
            }
            
            // Mettre à jour le bouton d'ajout
            const addBtn = document.getElementById('addSlideBtn');
            if (addBtn) {
                if (limits.max_slides !== -1 && totalSlides >= limits.max_slides) {
                    addBtn.style.display = 'none';
                } else {
                    addBtn.style.display = 'inline-flex';
                }
            }
        }

        function updateSlideType(slideNumber, type) {
            const optionsContainer = document.getElementById(`slide_options_${slideNumber}`);
            let additionalOptions = '';
            
            switch (type) {
                case 'video':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" name="autoplay_${slideNumber}"> 🔄 Lecture automatique
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" name="muted_${slideNumber}" checked> 🔇 Son coupé (recommandé)
                            </label>
                        </div>
                    `;
                    break;
                case 'audio_image':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">Fichier audio</label>
                            <input type="file" class="form-control" name="audio_file_${slideNumber}" accept="audio/*">
                        </div>
                    `;
                    break;
                case 'text_image':
                    additionalOptions = `
                        <div class="form-group">
                            <label class="form-label">Texte superposé</label>
                            <textarea class="form-control" name="text_overlay_${slideNumber}" rows="3" placeholder="Votre texte ici..."></textarea>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Position du texte</label>
                            <select class="form-control form-select" name="text_position_${slideNumber}">
                                <option value="center">🎯 Centre</option>
                                <option value="top">⬆️ Haut</option>
                                <option value="bottom">⬇️ Bas</option>
                                <option value="left">⬅️ Gauche</option>
                                <option value="right">➡️ Droite</option>
                            </select>
                        </div>
                    `;
                    break;
            }
            
            optionsContainer.innerHTML = additionalOptions;
        }

        function addSlide() {
            const totalSlidesInput = document.getElementById('total_slides');
            const currentTotal = parseInt(totalSlidesInput.value);
            
            if (limits.max_slides !== -1 && currentTotal >= limits.max_slides) {
                showUpgradeModal();
                return;
            }
            
            totalSlidesInput.value = currentTotal + 1;
            generateSlideCards();
        }

        function triggerFileUpload(slideNumber) {
            document.getElementById(`file_${slideNumber}`).click();
        }

        // Setup drag & drop
        function setupDragDrop(uploadArea, slideNumber) {
            uploadArea.addEventListener('dragover', function(e) {
                e.preventDefault();
                this.classList.add('dragover');
            });
            
            uploadArea.addEventListener('dragleave', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
            });
            
            uploadArea.addEventListener('drop', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
                
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    const fileInput = document.getElementById(`file_${slideNumber}`);
                    fileInput.files = files;
                    handleFileUpload(slideNumber, fileInput);
                }
            });
        }

        // Upload de fichiers
        function handleFileUpload(slideNumber, input) {
            const file = input.files[0];
            if (!file) return;
            
            // Vérifier la taille du fichier
            const maxSize = limits.max_storage_mb * 1024 * 1024;
            if (file.size > maxSize) {
                alert(`Fichier trop volumineux. Limite: ${limits.max_storage_mb}MB`);
                return;
            }
            
            updateAutosaveStatus('saving', 'Upload en cours...');
            
            const formData = new FormData();
            formData.append('action', 'upload_file');
            formData.append('file', file);
            formData.append('project_folder', 'temp_' + Date.now()); // Sera remplacé par le vrai dossier
            formData.append('slide_number', slideNumber);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    displayUploadedFile(slideNumber, file, data.file_data);
                    slidesData[slideNumber] = {
                        ...slidesData[slideNumber],
                        media_file: data.file_data.filename,
                        media_type: document.querySelector(`[name="media_type_${slideNumber}"]`).value
                    };
                    updateAutosaveStatus('saved', 'Fichier uploadé');
                    autoSave();
                } else {
                    updateAutosaveStatus('error', data.message);
                    alert('Erreur upload: ' + data.message);
                }
            })
            .catch(error => {
                updateAutosaveStatus('error', 'Erreur upload');
                console.error('Erreur upload:', error);
            });
        }

        function displayUploadedFile(slideNumber, file, fileData) {
            const preview = document.getElementById(`preview_${slideNumber}`);
            const uploadArea = preview.parentElement.querySelector('.upload-area');
            
            let content = '';
            
            if (fileData.file_type === 'image') {
                content = `<img src="${fileData.url}" alt="${file.name}" style="max-width: 100%; max-height: 100%; border-radius: 8px;">`;
            } else if (fileData.file_type === 'video') {
                content = `
                    <video controls style="max-width: 100%; max-height: 100%; border-radius: 8px;">
                        <source src="${fileData.url}" type="${file.type}">
                        Votre navigateur ne supporte pas la vidéo.
                    </video>
                `;
            } else if (fileData.file_type === 'audio') {
                content = `
                    <div style="text-align: center; padding: 20px;">
                        <div style="font-size: 2rem; margin-bottom: 10px;">🎵</div>
                        <div style="font-weight: 600; margin-bottom: 10px;">${file.name}</div>
                        <audio controls style="width: 100%;">
                            <source src="${fileData.url}" type="${file.type}">
                        </audio>
                    </div>
                `;
            }
            
            preview.innerHTML = content;
            preview.classList.add('has-media');
            
            // Mettre à jour l'upload area
            uploadArea.innerHTML = `
                <div style="color: #28a745;">
                    <div style="font-size: 1.5rem; margin-bottom: 5px;">✅</div>
                    <div>Fichier uploadé</div>
                    <small>${file.name}</small>
                </div>
            `;
            uploadArea.style.borderColor = '#28a745';
            uploadArea.style.background = 'rgba(40, 167, 69, 0.1)';
        }

        // Système d'auto-sauvegarde
        function initAutoSave() {
            // Auto-save toutes les 30 secondes
            autoSaveTimer = setInterval(autoSave, 30000);
            
            // Sauvegarde sur changement de champ
            document.addEventListener('input', function(e) {
                if (e.target.matches('input, textarea, select')) {
                    clearTimeout(autoSaveTimer);
                    autoSaveTimer = setTimeout(autoSave, 2000);
                }
            });
        }

        function autoSave() {
            const formData = collectFormData();
            
            updateAutosaveStatus('saving', 'Sauvegarde...');
            
            fetch(window.location.href, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=save_project&data=${encodeURIComponent(JSON.stringify(formData))}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (data.project_id && !projectId) {
                        projectId = data.project_id;
                        // Mettre à jour l'URL sans recharger
                        window.history.replaceState({}, '', window.location.pathname + '?id=' + projectId);
                    }
                    updateAutosaveStatus('saved', 'Sauvegardé automatiquement');
                } else {
                    updateAutosaveStatus('error', 'Erreur: ' + data.message);
                }
            })
            .catch(error => {
                updateAutosaveStatus('error', 'Erreur réseau');
                console.error('Erreur auto-save:', error);
            });
        }

        function collectFormData() {
            const formElements = document.querySelectorAll('input, textarea, select');
            const data = {};
            
            formElements.forEach(element => {
                if (element.name && element.name !== '') {
                    if (element.type === 'checkbox') {
                        data[element.name] = element.checked;
                    } else {
                        data[element.name] = element.value;
                    }
                }
            });
            
            data.slides = slidesData;
            data.selected_template = selectedTemplate;
            return data;
        }

        function updateAutosaveStatus(status, message) {
            const statusElement = document.getElementById('autosaveStatus');
            const iconElement = document.getElementById('autosaveIcon');
            const textElement = document.getElementById('autosaveText');
            
            if (!statusElement) return;
            
            statusElement.className = `autosave-status ${status}`;
            
            switch (status) {
                case 'saving':
                    iconElement.innerHTML = '<div class="spinner"></div>';
                    break;
                case 'saved':
                    iconElement.textContent = '✅';
                    break;
                case 'error':
                    iconElement.textContent = '❌';
                    break;
            }
            
            textElement.textContent = message;
        }

        // Modal upgrade
        function showUpgradeModal() {
            document.getElementById('upgradeModal').classList.add('active');
        }

        function closeUpgradeModal() {
            document.getElementById('upgradeModal').classList.remove('active');
        }

        // Loading overlay
        function showLoading(show) {
            const overlay = document.getElementById('loadingOverlay');
            if (overlay) {
                overlay.classList.toggle('active', show);
            }
        }

        function loadExistingData() {
            // Charger les données existantes si disponibles
            <?php if ($existing_project): ?>
            document.getElementById('style_theme').value = '<?= $existing_project['style_theme'] ?? 'moderne' ?>';
            document.getElementById('transition_type').value = '<?= $existing_project['transition_type'] ?? '3d' ?>';
            document.getElementById('navigation_type').value = '<?= $existing_project['navigation_type'] ?? 'all' ?>';
            
            <?php if ($existing_project['watermark_enabled']): ?>
            document.getElementById('watermark_enabled').checked = true;
            document.getElementById('watermark_text').style.display = 'block';
            document.getElementById('watermark_text').value = '<?= $existing_project['watermark_text'] ?? '' ?>';
            <?php endif; ?>
            
            <?php if (!$existing_project['analytics_enabled']): ?>
            document.getElementById('analytics_enabled').checked = false;
            <?php endif; ?>
            <?php endif; ?>
        }

        function startOver() {
            if (confirm('Êtes-vous sûr de vouloir créer un nouveau projet ?')) {
                window.location.href = window.location.pathname;
            }
        }

        // Gestion clavier
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey || e.metaKey) {
                switch (e.key) {
                    case 's':
                        e.preventDefault();
                        autoSave();
                        break;
                    case 'ArrowRight':
                        e.preventDefault();
                        nextStep();
                        break;
                    case 'ArrowLeft':
                        e.preventDefault();
                        prevStep();
                        break;
                }
            }
            
            if (e.key === 'Escape') {
                closeUpgradeModal();
            }
        });

        // Analytics et tracking
        function trackEvent(eventName, data = {}) {
            // Ici on pourrait ajouter Google Analytics, Mixpanel, etc.
            console.log('Event:', eventName, data);
        }

        // Track les actions importantes
        document.addEventListener('DOMContentLoaded', function() {
            trackEvent('wizard_started', {
                user_plan: userPlan.plan_type,
                project_id: projectId
            });
        });

        // Track changement d'étape
        function trackStepChange(from, to) {
            trackEvent('step_changed', {
                from_step: from,
                to_step: to,
                user_plan: userPlan.plan_type
            });
        }

        // Override nextStep pour tracking
        const originalNextStep = nextStep;
        nextStep = function() {
            const oldStep = currentStep;
            originalNextStep();
            if (currentStep !== oldStep) {
                trackStepChange(oldStep, currentStep);
            }
        };

        // Messages d'aide contextuelle
        const helpMessages = {
            1: "💡 Astuce: Choisissez un nom de projet explicite, il apparaîtra dans l'URL finale !",
            2: "🎨 Info: Les templates premium offrent des animations et styles avancés.",
            3: "⚙️ Conseil: Le mode 3D offre la meilleure expérience utilisateur !",
            4: "🎬 Important: Les vidéos sont automatiquement redimensionnées pour s'adapter parfaitement.",
            5: "🚀 Félicitations: Votre Flying Book est maintenant sauvegardé et prêt !"
        };

        // Afficher l'aide contextuelle
        function showStepHelp(step) {
            if (helpMessages[step]) {
                // On pourrait afficher une tooltip ou notification
                console.log(helpMessages[step]);
            }
        }

        // Validation avancée des fichiers
        function validateFile(file, slideNumber) {
            const allowedTypes = {
                image: ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
                video: ['video/mp4', 'video/webm', 'video/mov'],
                audio: ['audio/mp3', 'audio/wav', 'audio/ogg']
            };
            
            const maxSizes = {
                image: limits.max_storage_mb * 0.3 * 1024 * 1024,  // 30% pour images
                video: limits.max_storage_mb * 0.6 * 1024 * 1024,  // 60% pour vidéos
                audio: limits.max_storage_mb * 0.1 * 1024 * 1024   // 10% pour audio
            };
            
            const fileType = file.type.split('/')[0];
            
            if (!allowedTypes[fileType] || !allowedTypes[fileType].includes(file.type)) {
                throw new Error(`Type de fichier non supporté: ${file.type}`);
            }
            
            if (file.size > maxSizes[fileType]) {
                throw new Error(`Fichier trop volumineux pour le type ${fileType}. Max: ${Math.round(maxSizes[fileType] / 1024 / 1024)}MB`);
            }
            
            return true;
        }

        // Système de backup local
        function backupToLocalStorage() {
            const backupData = {
                timestamp: Date.now(),
                projectId: projectId,
                formData: collectFormData(),
                currentStep: currentStep
            };
            
            try {
                localStorage.setItem('flying_book_backup', JSON.stringify(backupData));
            } catch (e) {
                console.warn('Impossible de sauvegarder localement:', e);
            }
        }

        function restoreFromLocalStorage() {
            try {
                const backup = localStorage.getItem('flying_book_backup');
                if (backup) {
                    const data = JSON.parse(backup);
                    
                    // Vérifier si le backup est récent (moins de 24h)
                    if (Date.now() - data.timestamp < 24 * 60 * 60 * 1000) {
                        if (confirm('Un brouillon récent a été trouvé. Voulez-vous le restaurer ?')) {
                            // Restaurer les données
                            Object.keys(data.formData).forEach(key => {
                                const element = document.querySelector(`[name="${key}"]`);
                                if (element) {
                                    if (element.type === 'checkbox') {
                                        element.checked = data.formData[key];
                                    } else {
                                        element.value = data.formData[key];
                                    }
                                }
                            });
                            
                            goToStep(data.currentStep);
                        }
                    }
                }
            } catch (e) {
                console.warn('Erreur restauration backup:', e);
            }
        }

        // Appeler la restauration au chargement
        document.addEventListener('DOMContentLoaded', function() {
            if (!projectId) {  // Seulement pour nouveaux projets
                restoreFromLocalStorage();
            }
        });

        // Backup automatique toutes les 2 minutes
        setInterval(backupToLocalStorage, 2 * 60 * 1000);

        // Nettoyage backup à la publication
        function cleanupBackup() {
            localStorage.removeItem('flying_book_backup');
        }

        // Preview temps réel des paramètres
        function updatePreview() {
            const theme = document.getElementById('style_theme')?.value;
            const transition = document.getElementById('transition_type')?.value;
            
            // Ici on pourrait mettre à jour un aperçu visuel
            // Pour l'instant, on log juste les changements
            console.log('Preview updated:', { theme, transition });
        }

        // Event listeners pour preview
        document.getElementById('style_theme')?.addEventListener('change', updatePreview);
        document.getElementById('transition_type')?.addEventListener('change', updatePreview);

        console.log('🎪 Flying Book Creator WIZARD initialisé !');
        console.log('Plan utilisateur:', userPlan.plan_type);
        console.log('Limites:', limits);
    </script>
</body>
</html>