<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book - Tokens & IA</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            min-height: 100vh;
            overflow-x: hidden;
            color: #333;
        }

        /* Header fixe avec lien arabe */
        .header {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 70px;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 30px;
            z-index: 1000;
            box-shadow: 0 4px 30px rgba(0, 0, 0, 0.1);
        }

        .logo {
            font-size: 1.5rem;
            font-weight: 700;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .arabic-link {
            background: linear-gradient(45deg, #ff6b6b, #feca57);
            color: white;
            padding: 12px 24px;
            border-radius: 50px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.3);
        }

        .arabic-link:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(255, 107, 107, 0.4);
        }

        /* Container principal */
        .container {
            padding-top: 90px;
            padding-bottom: 120px;
            max-width: 1400px;
            margin: 0 auto;
            padding-left: 20px;
            padding-right: 20px;
        }

        /* Flying Book Container */
        .book-container {
            perspective: 1500px;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 70vh;
            position: relative;
        }

        .book {
            width: 900px;
            height: 600px;
            position: relative;
            transform-style: preserve-3d;
            transition: transform 0.6s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .book:hover {
            transform: rotateY(5deg) rotateX(2deg);
        }

        /* Page du livre */
        .page {
            position: absolute;
            width: 100%;
            height: 100%;
            background: linear-gradient(145deg, #ffffff 0%, #f8f9fa 100%);
            border-radius: 20px;
            box-shadow: 
                0 25px 50px rgba(0, 0, 0, 0.15),
                0 10px 20px rgba(0, 0, 0, 0.1),
                inset 0 1px 0 rgba(255, 255, 255, 0.6);
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            opacity: 0;
            visibility: hidden;
            transform: rotateY(-20deg) translateZ(-50px);
            transition: all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        .page.active {
            opacity: 1;
            visibility: visible;
            transform: rotateY(0deg) translateZ(0px);
            z-index: 10;
        }

        .page.prev {
            opacity: 0;
            visibility: hidden;
            transform: rotateY(15deg) translateZ(-30px) translateX(-50px);
            z-index: 5;
        }

        .page.next {
            opacity: 0;
            visibility: hidden;
            transform: rotateY(-15deg) translateZ(-30px) translateX(50px);
            z-index: 5;
        }

        /* Image de la slide */
        .slide-image {
            max-width: 95%;
            max-height: 95%;
            object-fit: contain;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            transition: transform 0.3s ease;
        }

        .slide-image:hover {
            transform: scale(1.02);
        }

        /* Loading state */
        .slide-loading {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: #666;
        }

        .spinner {
            width: 50px;
            height: 50px;
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-bottom: 20px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Navigation en bas */
        .navigation {
            position: fixed;
            bottom: 30px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            padding: 20px 30px;
            border-radius: 60px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 20px;
            z-index: 1000;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        /* Boutons navigation */
        .nav-btn {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            border: none;
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .nav-btn:hover {
            transform: translateY(-3px) scale(1.1);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .nav-btn:active {
            transform: translateY(-1px) scale(1.05);
        }

        .nav-btn:disabled {
            opacity: 0.4;
            cursor: not-allowed;
            transform: none !important;
        }

        /* Indicateur de progression */
        .progress-container {
            display: flex;
            align-items: center;
            gap: 15px;
            margin: 0 20px;
        }

        .progress-bar {
            width: 200px;
            height: 6px;
            background: rgba(0, 0, 0, 0.1);
            border-radius: 3px;
            overflow: hidden;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(45deg, #667eea, #764ba2);
            border-radius: 3px;
            transition: width 0.6s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .slide-counter {
            font-weight: 600;
            color: #333;
            min-width: 80px;
            text-align: center;
            font-size: 14px;
        }

        /* Aperçu miniatures */
        .thumbnails {
            position: fixed;
            right: 30px;
            top: 50%;
            transform: translateY(-50%);
            width: 80px;
            max-height: 70vh;
            overflow-y: auto;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 15px 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
            z-index: 900;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .thumbnails::-webkit-scrollbar {
            display: none;
        }

        .thumbnail {
            width: 60px;
            height: 40px;
            margin-bottom: 10px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            overflow: hidden;
        }

        .thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .thumbnail.active {
            border-color: #667eea;
            transform: scale(1.1);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .thumbnail:hover {
            transform: scale(1.05);
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
        }

        /* Zoom overlay amélioré pour mobile */
        .zoom-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.95);
            z-index: 2000;
            display: none;
            align-items: center;
            justify-content: center;
            backdrop-filter: blur(10px);
            padding: 20px;
        }

        .zoom-overlay.active {
            display: flex;
        }

        .zoom-content {
            max-width: 95vw;
            max-height: 95vh;
            position: relative;
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .zoom-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.5);
        }

        .zoom-close {
            position: absolute;
            top: -60px;
            right: 0;
            background: rgba(255, 255, 255, 0.9);
            border: none;
            color: #333;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 24px;
            font-weight: bold;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }

        .zoom-close:hover {
            background: rgba(255, 255, 255, 1);
            transform: scale(1.1);
        }

        .zoom-info {
            margin-top: 15px;
            color: white;
            text-align: center;
            font-size: 14px;
            opacity: 0.8;
        }

        /* Indicateur de zoom sur les slides */
        .slide-image {
            max-width: 95%;
            max-height: 95%;
            object-fit: contain;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            transition: transform 0.3s ease;
            cursor: zoom-in;
            position: relative;
        }

        .slide-image:hover {
            transform: scale(1.02);
        }

        /* Icône zoom pour indiquer que c'est cliquable */
        .slide-image::after {
            content: "🔍";
            position: absolute;
            bottom: 10px;
            right: 10px;
            background: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 8px;
            border-radius: 50%;
            font-size: 16px;
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .slide-image:hover::after {
            opacity: 1;
        }

        /* Mobile optimizations */
        @media (max-width: 768px) {
            .zoom-overlay {
                padding: 10px;
                background: rgba(0, 0, 0, 0.98);
            }
            
            .zoom-close {
                top: -50px;
                width: 45px;
                height: 45px;
                font-size: 20px;
            }
            
            .zoom-info {
                font-size: 12px;
                margin-top: 10px;
            }
            
            .slide-image::after {
                bottom: 5px;
                right: 5px;
                padding: 6px;
                font-size: 14px;
            }
            
            /* Double tap zoom sur mobile */
            .slide-image {
                touch-action: manipulation;
            }
        }

        /* Responsive Design */
        @media (max-width: 1024px) {
            .book {
                width: 700px;
                height: 500px;
            }
            
            .thumbnails {
                display: none;
            }
            
            .navigation {
                bottom: 20px;
                padding: 15px 20px;
            }
            
            .progress-bar {
                width: 150px;
            }
        }

        @media (max-width: 768px) {
            .book {
                width: 90vw;
                height: 60vw;
                max-height: 500px;
            }
            
            .header {
                padding: 0 20px;
                height: 60px;
            }
            
            .logo {
                font-size: 1.2rem;
            }
            
            .arabic-link {
                padding: 10px 20px;
                font-size: 14px;
            }
            
            .navigation {
                padding: 12px 20px;
                gap: 15px;
            }
            
            .nav-btn {
                width: 45px;
                height: 45px;
                font-size: 18px;
            }
            
            .progress-bar {
                width: 120px;
            }
            
            .slide-counter {
                font-size: 12px;
                min-width: 60px;
            }
        }

        /* Animations d'entrée */
        @keyframes slideInFromBottom {
            from {
                opacity: 0;
                transform: translateY(100px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        @keyframes bookEntrance {
            from {
                opacity: 0;
                transform: perspective(1500px) rotateY(-30deg) rotateX(15deg) translateZ(-200px);
            }
            to {
                opacity: 1;
                transform: perspective(1500px) rotateY(0deg) rotateX(0deg) translateZ(0px);
            }
        }

        .book {
            animation: bookEntrance 1.2s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }

        .navigation {
            animation: slideInFromBottom 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94) 0.5s both;
        }

        .thumbnails {
            animation: slideInFromBottom 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94) 0.7s both;
        }
    </style>
</head>
<body>
    <!-- Header fixe -->
    <div class="header">
        <div class="logo">📚 Flying Book</div>
        <a href="/var/www/laloee-fr/document/tokens.htm" class="arabic-link">
            🌙 Consulter la traduction en arabe
        </a>
    </div>

    <!-- Container principal -->
    <div class="container">
        <div class="book-container">
            <div class="book" id="book">
                <!-- Les pages seront générées dynamiquement -->
            </div>
        </div>
    </div>

    <!-- Navigation en bas -->
    <div class="navigation">
        <button class="nav-btn" id="prevBtn" onclick="previousSlide()">‹</button>
        
        <div class="progress-container">
            <div class="progress-bar">
                <div class="progress-fill" id="progressFill"></div>
            </div>
            <div class="slide-counter" id="slideCounter">1 / 39</div>
        </div>
        
        <button class="nav-btn" id="nextBtn" onclick="nextSlide()">›</button>
    </div>

    <!-- Aperçu miniatures -->
    <div class="thumbnails" id="thumbnails">
        <!-- Les miniatures seront générées dynamiquement -->
    </div>

    <!-- Overlay pour zoom -->
    <div class="zoom-overlay" id="zoomOverlay">
        <div class="zoom-content">
            <button class="zoom-close" onclick="closeZoom()">×</button>
            <img class="zoom-image" id="zoomImage" src="" alt="Slide agrandie">
        </div>
    </div>

    <script>
        let currentSlide = 1;
        const totalSlides = 39;
        const basePath = '/var/www/laloee-fr/document/';

        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            generateSlides();
            generateThumbnails();
            updateDisplay();
            setupKeyboardNavigation();
            setupTouchNavigation();
        });

        // Génération des slides
        function generateSlides() {
            const book = document.getElementById('book');
            
            for (let i = 1; i <= totalSlides; i++) {
                const page = document.createElement('div');
                page.className = 'page';
                page.id = `slide-${i}`;
                
                // État de chargement initial
                page.innerHTML = `
                    <div class="slide-loading">
                        <div class="spinner"></div>
                        <p>Chargement de la slide ${i}...</p>
                    </div>
                `;
                
                book.appendChild(page);
                
                // Chargement de l'image
                const img = new Image();
                img.onload = function() {
                    page.innerHTML = `<img class="slide-image" src="${this.src}" alt="Diapositive ${i}" onclick="openZoom('${this.src}')">`;
                };
                img.onerror = function() {
                    page.innerHTML = `
                        <div style="text-align: center; color: #666;">
                            <h3>❌ Erreur de chargement</h3>
                            <p>Impossible de charger la slide ${i}</p>
                        </div>
                    `;
                };
                img.src = `${basePath}Diapositive${i}.PNG`;
            }
        }

        // Génération des miniatures
        function generateThumbnails() {
            const thumbnails = document.getElementById('thumbnails');
            
            for (let i = 1; i <= totalSlides; i++) {
                const thumb = document.createElement('div');
                thumb.className = 'thumbnail';
                thumb.id = `thumb-${i}`;
                thumb.onclick = () => goToSlide(i);
                
                const img = document.createElement('img');
                img.src = `${basePath}Diapositive${i}.PNG`;
                img.alt = `Slide ${i}`;
                img.onerror = function() {
                    this.src = 'data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 60"><rect width="100" height="60" fill="%23f0f0f0"/><text x="50" y="35" text-anchor="middle" fill="%23999">' + i + '</text></svg>';
                };
                
                thumb.appendChild(img);
                thumbnails.appendChild(thumb);
            }
        }

        // Navigation
        function nextSlide() {
            if (currentSlide < totalSlides) {
                currentSlide++;
                updateDisplay();
            }
        }

        function previousSlide() {
            if (currentSlide > 1) {
                currentSlide--;
                updateDisplay();
            }
        }

        function goToSlide(slideNumber) {
            if (slideNumber >= 1 && slideNumber <= totalSlides) {
                currentSlide = slideNumber;
                updateDisplay();
            }
        }

        // Mise à jour de l'affichage
        function updateDisplay() {
            // Mise à jour des pages
            const pages = document.querySelectorAll('.page');
            pages.forEach((page, index) => {
                const slideNumber = index + 1;
                page.className = 'page';
                
                if (slideNumber === currentSlide) {
                    page.classList.add('active');
                } else if (slideNumber === currentSlide - 1) {
                    page.classList.add('prev');
                } else if (slideNumber === currentSlide + 1) {
                    page.classList.add('next');
                }
            });

            // Mise à jour des miniatures
            const thumbnails = document.querySelectorAll('.thumbnail');
            thumbnails.forEach((thumb, index) => {
                thumb.classList.toggle('active', index + 1 === currentSlide);
            });

            // Mise à jour de la progression
            const progressFill = document.getElementById('progressFill');
            const progress = (currentSlide / totalSlides) * 100;
            progressFill.style.width = `${progress}%`;

            // Mise à jour du compteur
            const slideCounter = document.getElementById('slideCounter');
            slideCounter.textContent = `${currentSlide} / ${totalSlides}`;

            // Mise à jour des boutons
            const prevBtn = document.getElementById('prevBtn');
            const nextBtn = document.getElementById('nextBtn');
            prevBtn.disabled = currentSlide === 1;
            nextBtn.disabled = currentSlide === totalSlides;

            // Scroll auto des miniatures
            scrollThumbnailsToActive();
        }

        // Scroll automatique des miniatures
        function scrollThumbnailsToActive() {
            const thumbnailsContainer = document.getElementById('thumbnails');
            const activeThumb = document.getElementById(`thumb-${currentSlide}`);
            
            if (activeThumb && thumbnailsContainer) {
                const containerHeight = thumbnailsContainer.clientHeight;
                const thumbHeight = activeThumb.offsetHeight;
                const thumbTop = activeThumb.offsetTop;
                const scrollTop = thumbTop - (containerHeight / 2) + (thumbHeight / 2);
                
                thumbnailsContainer.scrollTo({
                    top: scrollTop,
                    behavior: 'smooth'
                });
            }
        }

        // Navigation clavier
        function setupKeyboardNavigation() {
            document.addEventListener('keydown', function(e) {
                switch(e.key) {
                    case 'ArrowLeft':
                    case 'ArrowUp':
                        e.preventDefault();
                        previousSlide();
                        break;
                    case 'ArrowRight':
                    case 'ArrowDown':
                    case ' ':
                        e.preventDefault();
                        nextSlide();
                        break;
                    case 'Home':
                        e.preventDefault();
                        goToSlide(1);
                        break;
                    case 'End':
                        e.preventDefault();
                        goToSlide(totalSlides);
                        break;
                    case 'Escape':
                        closeZoom();
                        break;
                }
            });
        }

        // Navigation tactile
        function setupTouchNavigation() {
            let startX = 0;
            let startY = 0;
            const book = document.getElementById('book');

            book.addEventListener('touchstart', function(e) {
                startX = e.touches[0].clientX;
                startY = e.touches[0].clientY;
            });

            book.addEventListener('touchend', function(e) {
                if (!startX || !startY) return;

                const endX = e.changedTouches[0].clientX;
                const endY = e.changedTouches[0].clientY;
                const diffX = startX - endX;
                const diffY = startY - endY;

                // Détection swipe horizontal
                if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > 50) {
                    if (diffX > 0) {
                        nextSlide(); // Swipe left = next
                    } else {
                        previousSlide(); // Swipe right = previous
                    }
                }

                startX = 0;
                startY = 0;
            });
        }

        // Fonctions zoom
        function openZoom(imageSrc) {
            const zoomOverlay = document.getElementById('zoomOverlay');
            const zoomImage = document.getElementById('zoomImage');
            
            zoomImage.src = imageSrc;
            zoomOverlay.classList.add('active');
            document.body.style.overflow = 'hidden';
        }

        function closeZoom() {
            const zoomOverlay = document.getElementById('zoomOverlay');
            zoomOverlay.classList.remove('active');
            document.body.style.overflow = '';
        }

        // Fermeture zoom en cliquant sur l'overlay
        document.getElementById('zoomOverlay').addEventListener('click', function(e) {
            if (e.target === this) {
                closeZoom();
            }
        });

        // Préchargement intelligent des images voisines
        function preloadNearbyImages() {
            const preloadSlides = [];
            
            // Précharger slide précédente et suivante
            if (currentSlide > 1) preloadSlides.push(currentSlide - 1);
            if (currentSlide < totalSlides) preloadSlides.push(currentSlide + 1);
            
            preloadSlides.forEach(slideNumber => {
                const img = new Image();
                img.src = `${basePath}Diapositive${slideNumber}.PNG`;
            });
        }

        // Préchargement après chaque changement de slide
        setInterval(() => {
            if (document.hasFocus()) {
                preloadNearbyImages();
            }
        }, 1000);
    </script>
</body>
</html>