<?php
/**
 * ================================================================
 * 🔄 WORKFLOW MANAGER SMG STUDIO - BY PAPY 77
 * ================================================================
 * 
 * 🎯 GESTION WORKFLOW COMPLET :
 * ✅ Vue d'ensemble workflow par projet
 * ✅ Transfert entre étapes
 * ✅ Gestion permissions utilisateurs
 * ✅ Génération formats réseaux sociaux
 * ✅ Suivi temps réel des exports
 * 
 * 💡 SIMPLE ET EFFICACE - COMME AU TEMPS DU GW-BASIC !
 * ================================================================
 */

require_once __DIR__ . '/auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// ================================================================
// 🔧 FONCTIONS WORKFLOW SIMPLES
// ================================================================

function getWorkflowPermissions($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT stage_name, can_view, can_process, can_approve, can_delete, is_manager
            FROM user_workflow_permissions 
            WHERE user_id = ?
        ");
        $stmt->execute([$user_id]);
        
        $permissions = [];
        while ($row = $stmt->fetch()) {
            $permissions[$row['stage_name']] = [
                'can_view' => (bool)$row['can_view'],
                'can_process' => (bool)$row['can_process'],
                'can_approve' => (bool)$row['can_approve'],
                'can_delete' => (bool)$row['can_delete'],
                'is_manager' => (bool)$row['is_manager']
            ];
        }
        
        return $permissions;
    } catch (Exception $e) {
        return [];
    }
}

function getAllWorkflowExports($pdo, $user_id, $stage = null) {
    try {
        $sql = "
            SELECT we.*, p.name as project_name, ws.display_name as stage_display,
                   u.name as processed_by_name
            FROM workflow_exports we
            JOIN clipmaker_projects p ON we.project_id = p.id
            JOIN workflow_stages ws ON we.stage = ws.stage_name
            LEFT JOIN users u ON we.processed_by_user_id = u.id
            WHERE we.user_id = ?
        ";
        
        $params = [$user_id];
        
        if ($stage) {
            $sql .= " AND we.stage = ?";
            $params[] = $stage;
        }
        
        $sql .= " ORDER BY we.created_at DESC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        return [];
    }
}

function getWorkflowStages($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM workflow_stages WHERE is_active = 1 ORDER BY stage_order");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function transferToNextStage($pdo, $user_id, $export_id, $next_stage) {
    try {
        // Vérifier permissions
        $permStmt = $pdo->prepare("
            SELECT can_process FROM user_workflow_permissions 
            WHERE user_id = ? AND stage_name = ?
        ");
        $permStmt->execute([$user_id, $next_stage]);
        $permission = $permStmt->fetch();
        
        if (!$permission || !$permission['can_process']) {
            throw new Exception("Permission insuffisante pour: $next_stage");
        }
        
        // Récupérer export source
        $stmt = $pdo->prepare("SELECT * FROM workflow_exports WHERE id = ?");
        $stmt->execute([$export_id]);
        $sourceExport = $stmt->fetch();
        
        if (!$sourceExport) {
            throw new Exception("Export source non trouvé");
        }
        
        // Créer nouvelle entrée au stage suivant
        $stmt = $pdo->prepare("
            INSERT INTO workflow_exports 
            (user_id, project_id, stage, export_name, original_filename, file_path, 
             file_size, format, codec_video, codec_audio, resolution_width, resolution_height, 
             fps, bitrate_video, bitrate_audio, duration_seconds, status, quality_preset, 
             version_number, parent_export_id, processed_by_user_id, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'processing', ?, 1, ?, ?, NOW())
        ");
        
        $newExportName = str_replace(
            ['CLIP_', 'AUDIO_', 'FINAL_'], 
            [
                'clipmaker' => 'AUDIO_',
                'audio_postprod' => 'FINAL_', 
                'emission_final' => 'SOCIAL_'
            ][$next_stage] ?? 'NEXT_',
            $sourceExport['export_name']
        );
        
        $stmt->execute([
            $user_id,
            $sourceExport['project_id'],
            $next_stage,
            $newExportName,
            $sourceExport['original_filename'],
            $sourceExport['file_path'], // TODO: Copier vers nouveau dossier
            $sourceExport['file_size'],
            $sourceExport['format'],
            $sourceExport['codec_video'],
            $sourceExport['codec_audio'],
            $sourceExport['resolution_width'],
            $sourceExport['resolution_height'],
            $sourceExport['fps'],
            $sourceExport['bitrate_video'],
            $sourceExport['bitrate_audio'],
            $sourceExport['duration_seconds'],
            $sourceExport['quality_preset'],
            $export_id,
            $user_id
        ]);
        
        $new_export_id = $pdo->lastInsertId();
        
        // Marquer comme terminé
        $pdo->prepare("UPDATE workflow_exports SET status = 'completed', completed_at = NOW() WHERE id = ?")
           ->execute([$new_export_id]);
        
        return $new_export_id;
        
    } catch (Exception $e) {
        throw $e;
    }
}

function generateSocialFormats($pdo, $user_id, $source_export_id) {
    try {
        // Récupérer formats disponibles
        $stmt = $pdo->prepare("SELECT * FROM social_export_formats WHERE is_active = 1");
        $stmt->execute();
        $formats = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Récupérer export source
        $stmt = $pdo->prepare("SELECT * FROM workflow_exports WHERE id = ?");
        $stmt->execute([$source_export_id]);
        $sourceExport = $stmt->fetch();
        
        if (!$sourceExport) {
            throw new Exception("Export source non trouvé");
        }
        
        $generated = [];
        
        // Grouper par réseau
        $networks = [];
        foreach ($formats as $format) {
            $networks[$format['network_name']][] = $format;
        }
        
        // Générer pour chaque réseau
        foreach ($networks as $network => $networkFormats) {
            foreach ($networkFormats as $format) {
                $socialExportName = strtoupper($network) . '_' . 
                                  str_replace(' ', '_', $format['format_name']) . '_' . 
                                  time();
                
                // Créer entrée workflow
                $stmt = $pdo->prepare("
                    INSERT INTO workflow_exports 
                    (user_id, project_id, stage, export_name, original_filename, file_path, 
                     file_size, format, codec_video, codec_audio, resolution_width, resolution_height, 
                     fps, bitrate_video, bitrate_audio, duration_seconds, status, quality_preset, 
                     version_number, parent_export_id, processed_by_user_id, created_at) 
                    VALUES (?, ?, 'social_ready', ?, ?, ?, ?, 'mp4', 'h264', 'aac', ?, ?, ?, ?, '128k', ?, 'completed', 'web', 1, ?, ?, NOW())
                ");
                
                $stmt->execute([
                    $user_id,
                    $sourceExport['project_id'],
                    $socialExportName,
                    $sourceExport['original_filename'],
                    $sourceExport['file_path'], // TODO: Générer vraiment le fichier
                    $sourceExport['file_size'],
                    $format['resolution_width'],
                    $format['resolution_height'],
                    $format['recommended_fps'],
                    $format['recommended_bitrate'],
                    $sourceExport['duration_seconds'],
                    $source_export_id,
                    $user_id
                ]);
                
                $generated[] = [
                    'export_id' => $pdo->lastInsertId(),
                    'network' => $network,
                    'format_name' => $format['format_name']
                ];
            }
        }
        
        return $generated;
        
    } catch (Exception $e) {
        throw $e;
    }
}

// ================================================================
// 🔧 TRAITEMENT ACTIONS
// ================================================================

$message = '';
$messageType = 'info';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        switch ($action) {
            case 'transfer_stage':
                $export_id = $_POST['export_id'] ?? null;
                $next_stage = $_POST['next_stage'] ?? null;
                
                if (!$export_id || !$next_stage) {
                    throw new Exception("Paramètres manquants");
                }
                
                $new_export_id = transferToNextStage($pdo, $user['id'], $export_id, $next_stage);
                $message = "✅ Export transféré vers: $next_stage (ID: $new_export_id)";
                $messageType = 'success';
                break;
                
            case 'generate_social':
                $export_id = $_POST['export_id'] ?? null;
                
                if (!$export_id) {
                    throw new Exception("Export ID manquant");
                }
                
                $generated = generateSocialFormats($pdo, $user['id'], $export_id);
                $message = "✅ " . count($generated) . " formats réseaux sociaux générés";
                $messageType = 'success';
                break;
                
            case 'approve_export':
                $export_id = $_POST['export_id'] ?? null;
                
                if (!$export_id) {
                    throw new Exception("Export ID manquant");
                }
                
                $stmt = $pdo->prepare("
                    UPDATE workflow_exports 
                    SET status = 'approved', approved_by_user_id = ?, approved_at = NOW() 
                    WHERE id = ?
                ");
                $stmt->execute([$user['id'], $export_id]);
                
                $message = "✅ Export approuvé";
                $messageType = 'success';
                break;
                
            default:
                throw new Exception("Action inconnue: $action");
        }
    } catch (Exception $e) {
        $message = "❌ Erreur: " . $e->getMessage();
        $messageType = 'error';
    }
    
    // Redirection POST
    header("Location: workflow.php?msg=" . urlencode($message) . "&type=" . $messageType);
    exit;
}

// Affichage message depuis redirect
if (isset($_GET['msg'])) {
    $message = $_GET['msg'];
    $messageType = $_GET['type'] ?? 'info';
}

// ================================================================
// 🔧 RÉCUPÉRATION DONNÉES
// ================================================================

$stage_filter = $_GET['stage'] ?? null;

// Permissions utilisateur
$permissions = getWorkflowPermissions($pdo, $user['id']);

// Stages disponibles
$stages = getWorkflowStages($pdo);

// Exports workflow
$exports = getAllWorkflowExports($pdo, $user['id'], $stage_filter);

// Statistiques rapides
$stats = [];
foreach ($stages as $stage) {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count, 
               COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed,
               COUNT(CASE WHEN status = 'processing' THEN 1 END) as processing
        FROM workflow_exports 
        WHERE user_id = ? AND stage = ?
    ");
    $stmt->execute([$user['id'], $stage['stage_name']]);
    $stats[$stage['stage_name']] = $stmt->fetch(PDO::FETCH_ASSOC);
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔄 Workflow Manager - SMG Studio</title>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px;
        }

        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            text-align: center;
        }

        .header h1 {
            color: #333;
            font-size: 32px;
            margin-bottom: 10px;
        }

        .header .subtitle {
            color: #666;
            font-size: 16px;
        }

        .back-link {
            position: absolute;
            top: 20px;
            left: 20px;
            background: rgba(255, 255, 255, 0.9);
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            color: #333;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .back-link:hover {
            background: white;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .alert {
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 25px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: linear-gradient(135deg, rgba(72, 187, 120, 0.2), rgba(56, 161, 105, 0.2));
            color: #2f855a;
            border-left: 4px solid #48bb78;
        }

        .alert-error {
            background: linear-gradient(135deg, rgba(245, 101, 101, 0.2), rgba(229, 62, 62, 0.2));
            color: #c53030;
            border-left: 4px solid #f56565;
        }

        .alert-info {
            background: linear-gradient(135deg, rgba(66, 153, 225, 0.2), rgba(49, 130, 206, 0.2));
            color: #2b6cb0;
            border-left: 4px solid #4299e1;
        }

        .stage-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 30px;
            flex-wrap: wrap;
        }

        .stage-tab {
            background: rgba(255, 255, 255, 0.9);
            padding: 15px 25px;
            border-radius: 12px;
            text-decoration: none;
            color: #555;
            font-weight: 600;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            position: relative;
        }

        .stage-tab:hover,
        .stage-tab.active {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .stage-tab .badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: #ff6b6b;
            color: white;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            font-size: 11px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }

        .stage-tab.active .badge {
            background: rgba(255, 255, 255, 0.3);
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0, 0, 0, 0.15);
        }

        .stat-card .stat-icon {
            font-size: 28px;
            margin-bottom: 10px;
        }

        .stat-card .stat-number {
            font-size: 24px;
            font-weight: 700;
            color: #333;
            margin-bottom: 5px;
        }

        .stat-card .stat-label {
            color: #666;
            font-size: 12px;
            font-weight: 500;
        }

        .exports-section {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }

        .exports-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }

        .exports-header h2 {
            color: #333;
            font-size: 24px;
        }

        .export-item {
            background: rgba(255, 255, 255, 0.5);
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 20px;
            border-left: 4px solid #667eea;
            transition: all 0.3s ease;
        }

        .export-item:hover {
            transform: translateY(-3px);
            box-shadow: 0 15px 40px rgba(0, 0, 0, 0.1);
        }

        .export-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }

        .export-name {
            font-size: 18px;
            font-weight: 600;
            color: #333;
            margin-bottom: 5px;
        }

        .export-details {
            color: #666;
            font-size: 14px;
            margin-bottom: 15px;
            line-height: 1.5;
        }

        .export-status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }

        .export-status.completed {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .export-status.processing {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
            animation: pulse 2s ease-in-out infinite;
        }

        .export-status.failed {
            background: linear-gradient(135deg, #ff6b6b, #ff5252);
            color: white;
        }

        .export-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .btn {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            font-size: 12px;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .btn-success {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
        }

        .btn-warning {
            background: linear-gradient(135deg, #f093fb, #f5576c);
        }

        .btn-danger {
            background: linear-gradient(135deg, #ff6b6b, #ff5252);
        }

        .btn-secondary {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
            border: 1px solid rgba(102, 126, 234, 0.2);
        }

        .btn-secondary:hover {
            background: rgba(102, 126, 234, 0.2);
        }

        .empty-state {
            text-align: center;
            padding: 60px;
            color: #666;
        }

        .empty-state .icon {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }

        .empty-state h3 {
            margin-bottom: 15px;
            color: #555;
        }

        .empty-state p {
            margin-bottom: 30px;
        }

        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }

        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }
            
            .stage-tabs {
                flex-direction: column;
            }
            
            .export-header {
                flex-direction: column;
                align-items: start;
                gap: 10px;
            }
            
            .export-actions {
                width: 100%;
                justify-content: stretch;
            }
            
            .btn {
                flex: 1;
                justify-content: center;
            }
        }
    </style>
</head>

<body>
    <a href="dashboard.php" class="back-link">← Retour Dashboard</a>
    
    <div class="container">
        <div class="header">
            <h1>🔄 Workflow Manager</h1>
            <div class="subtitle">Gestion pipeline de production SMG Studio - <?= htmlspecialchars($user['name']) ?></div>
        </div>

        <?php if ($message): ?>
            <div class="alert alert-<?= $messageType ?>">
                <div style="font-size: 1.2em;">
                    <?php if ($messageType === 'success'): ?>🎉<?php elseif ($messageType === 'error'): ?>❌<?php else: ?>ℹ️<?php endif; ?>
                </div>
                <?= htmlspecialchars($message) ?>
            </div>
        <?php endif; ?>

        <!-- ONGLETS STAGES -->
        <div class="stage-tabs">
            <a href="workflow.php" class="stage-tab <?= !$stage_filter ? 'active' : '' ?>">
                🔄 Tous les Stages
                <?php if (array_sum(array_column($stats, 'processing'))): ?>
                    <span class="badge"><?= array_sum(array_column($stats, 'processing')) ?></span>
                <?php endif; ?>
            </a>
            
            <?php foreach ($stages as $stage): ?>
                <a href="workflow.php?stage=<?= $stage['stage_name'] ?>" 
                   class="stage-tab <?= $stage_filter === $stage['stage_name'] ? 'active' : '' ?>">
                    <?= $stage['display_name'] ?>
                    <?php if ($stats[$stage['stage_name']]['processing'] > 0): ?>
                        <span class="badge"><?= $stats[$stage['stage_name']]['processing'] ?></span>
                    <?php endif; ?>
                </a>
            <?php endforeach; ?>
        </div>

        <!-- STATS WORKFLOW -->
        <div class="stats-grid">
            <?php foreach ($stages as $stage): ?>
                <div class="stat-card">
                    <div class="stat-icon">
                        <?php 
                        $icons = [
                            'clipmaker' => '🎬',
                            'audio_postprod' => '🎵', 
                            'emission_final' => '📺',
                            'social_ready' => '📱'
                        ];
                        echo $icons[$stage['stage_name']] ?? '🔄';
                        ?>
                    </div>
                    <div class="stat-number"><?= $stats[$stage['stage_name']]['count'] ?></div>
                    <div class="stat-label"><?= str_replace(['🎬 ', '🎵 ', '📺 ', '📱 '], '', $stage['display_name']) ?></div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- EXPORTS WORKFLOW -->
        <div class="exports-section">
            <div class="exports-header">
                <h2>📤 Exports Workflow <?= $stage_filter ? "($stage_filter)" : '' ?></h2>
                <a href="modules/clipmaker-video-engine.php" class="btn">
                    ➕ Nouveau Export
                </a>
            </div>

            <?php if (empty($exports)): ?>
                <div class="empty-state">
                    <div class="icon">🔄</div>
                    <h3>Aucun export dans le workflow</h3>
                    <p>Commencez par créer un export depuis Clipmaker pour voir le workflow en action.</p>
                    <a href="modules/clipmaker-video-engine.php" class="btn">
                        🎬 Aller à Clipmaker
                    </a>
                </div>
            <?php else: ?>
                <?php foreach ($exports as $export): ?>
                    <div class="export-item">
                        <div class="export-header">
                            <div>
                                <div class="export-name"><?= htmlspecialchars($export['export_name']) ?></div>
                                <div style="color: #667eea; font-size: 14px; font-weight: 600;">
                                    📁 Projet: <?= htmlspecialchars($export['project_name']) ?>
                                </div>
                            </div>
                            
                            <span class="export-status <?= $export['status'] ?>">
                                <?= $export['status'] ?>
                            </span>
                        </div>
                        
                        <div class="export-details">
                            <div><strong>Stage:</strong> <?= htmlspecialchars($export['stage_display']) ?></div>
                            <div><strong>Format:</strong> <?= strtoupper($export['format']) ?> <?= $export['codec_video'] ?></div>
                            <div><strong>Résolution:</strong> <?= $export['resolution_width'] ?>x<?= $export['resolution_height'] ?> @ <?= $export['fps'] ?>fps</div>
                            <div><strong>Taille:</strong> <?= number_format($export['file_size'] / 1024 / 1024, 1) ?>MB</div>
                            <div><strong>Durée:</strong> <?= gmdate('i:s', (int)$export['duration_seconds']) ?></div>
                            <div><strong>Créé:</strong> <?= date('d/m/Y H:i', strtotime($export['created_at'])) ?></div>
                            <?php if ($export['completed_at']): ?>
                                <div><strong>Terminé:</strong> <?= date('d/m/Y H:i', strtotime($export['completed_at'])) ?></div>
                            <?php endif; ?>
                            <?php if ($export['processed_by_name']): ?>
                                <div><strong>Traité par:</strong> <?= htmlspecialchars($export['processed_by_name']) ?></div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="export-actions">
                            <!-- Actions selon stage et permissions -->
                            <?php if ($export['status'] === 'completed'): ?>
                                
                                <?php if ($export['stage'] === 'clipmaker' && ($permissions['audio_postprod']['can_process'] ?? false)): ?>
                                    <form method="post" style="display: inline;">
                                        <input type="hidden" name="action" value="transfer_stage">
                                        <input type="hidden" name="export_id" value="<?= $export['id'] ?>">
                                        <input type="hidden" name="next_stage" value="audio_postprod">
                                        <button type="submit" class="btn btn-success">
                                            🎵 → Audio Post-Prod
                                        </button>
                                    </form>
                                <?php endif; ?>
                                
                                <?php if ($export['stage'] === 'audio_postprod' && ($permissions['emission_final']['can_process'] ?? false)): ?>
                                    <form method="post" style="display: inline;">
                                        <input type="hidden" name="action" value="transfer_stage">
                                        <input type="hidden" name="export_id" value="<?= $export['id'] ?>">
                                        <input type="hidden" name="next_stage" value="emission_final">
                                        <button type="submit" class="btn btn-success">
                                            📺 → Émission Finale
                                        </button>
                                    </form>
                                <?php endif; ?>
                                
                                <?php if ($export['stage'] === 'emission_final' && ($permissions['social_ready']['can_process'] ?? false)): ?>
                                    <form method="post" style="display: inline;">
                                        <input type="hidden" name="action" value="generate_social">
                                        <input type="hidden" name="export_id" value="<?= $export['id'] ?>">
                                        <button type="submit" class="btn btn-warning">
                                            📱 Générer Formats Réseaux
                                        </button>
                                    </form>
                                <?php endif; ?>
                                
                                <?php if (($permissions[$export['stage']]['can_approve'] ?? false) && $export['status'] !== 'approved'): ?>
                                    <form method="post" style="display: inline;">
                                        <input type="hidden" name="action" value="approve_export">
                                        <input type="hidden" name="export_id" value="<?= $export['id'] ?>">
                                        <button type="submit" class="btn">
                                            ✅ Approuver
                                        </button>
                                    </form>
                                <?php endif; ?>
                                
                            <?php endif; ?>
                            
                            <a href="<?= htmlspecialchars($export['file_path']) ?>" 
                               target="_blank" class="btn btn-secondary">
                                📥 Télécharger
                            </a>
                            
                            <?php if ($export['parent_export_id']): ?>
                                <button class="btn btn-secondary" onclick="showHistory(<?= $export['id'] ?>)">
                                    📜 Historique
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <script>
        function showHistory(exportId) {
            // TODO: Popup historique workflow
            alert('Historique export ID: ' + exportId + '\nFonctionnalité à développer');
        }

        // Confirmation actions importantes
        document.querySelectorAll('form[method="post"]').forEach(form => {
            form.addEventListener('submit', function(e) {
                const action = this.querySelector('input[name="action"]').value;
                
                if (action === 'transfer_stage') {
                    const stage = this.querySelector('input[name="next_stage"]').value;
                    if (!confirm(`Transférer vers le stage: ${stage} ?`)) {
                        e.preventDefault();
                    }
                } else if (action === 'generate_social') {
                    if (!confirm('Générer tous les formats réseaux sociaux ?\nCela peut prendre plusieurs minutes.')) {
                        e.preventDefault();
                    }
                }
            });
        });

        // Auto-refresh toutes les 30 secondes si exports en cours
        <?php if (array_sum(array_column($stats, 'processing')) > 0): ?>
        setInterval(function() {
            const url = new URL(window.location);
            url.searchParams.set('autorefresh', '1');
            
            fetch(url)
                .then(response => response.text())
                .then(html => {
                    // TODO: Mettre à jour seulement la section exports
                    console.log('🔄 Auto-refresh workflow...');
                })
                .catch(error => console.error('Erreur auto-refresh:', error));
        }, 30000);
        <?php endif; ?>

        console.log('🔄 Workflow Manager Ready - by Papy 77');
        console.log('✅ Permissions utilisateur:', <?= json_encode($permissions) ?>);
        console.log('📊 Stats stages:', <?= json_encode($stats) ?>);
    </script>
</body>
</html>