<?php
/**
 * ================================================================
 * ⚙️ ADMINISTRATION SYSTÈME SMG STUDIO v2.0 - PANNEAU SUPER ADMIN
 * ================================================================
 * 
 * 🎯 Configuration globale + maintenance + monitoring
 * ✅ Logs système + sécurité + performances
 * 🔧 Paramètres IA + tokens + API
 * 🚨 Alertes + notifications + backups
 * 
 * 💡 BY PAPY 77 - CONTRÔLE TOTAL SYSTÈME !
 * ================================================================
 */

session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session + droits super admin
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur courant
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $current_user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$current_user || $current_user['role'] !== 'admin') {
        header('Location: dashboard.php?error=insufficient_privileges');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur vérification admin: " . $e->getMessage());
    header('Location: dashboard.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS SYSTÈME
// ================================================================

$action = $_GET['action'] ?? 'overview';
$message = '';
$message_type = '';

// ⚙️ MISE À JOUR CONFIGURATION
if ($_POST['action'] ?? '' === 'update_config') {
    try {
        $config_updates = [];
        
        // Configuration IA
        if (isset($_POST['default_ai_model'])) {
            $config_updates['default_ai_model'] = $_POST['default_ai_model'];
        }
        if (isset($_POST['max_tokens_per_request'])) {
            $config_updates['max_tokens_per_request'] = intval($_POST['max_tokens_per_request']);
        }
        if (isset($_POST['ai_temperature'])) {
            $config_updates['ai_temperature'] = floatval($_POST['ai_temperature']);
        }
        
        // Configuration tokens
        if (isset($_POST['default_token_package'])) {
            $config_updates['default_token_package'] = $_POST['default_token_package'];
        }
        if (isset($_POST['token_expiry_days'])) {
            $config_updates['token_expiry_days'] = intval($_POST['token_expiry_days']);
        }
        
        // Configuration système
        if (isset($_POST['maintenance_mode'])) {
            $config_updates['maintenance_mode'] = $_POST['maintenance_mode'] === '1' ? '1' : '0';
        }
        if (isset($_POST['max_file_size'])) {
            $config_updates['max_file_size'] = intval($_POST['max_file_size']);
        }
        
        // Enregistrer en base
        foreach ($config_updates as $key => $value) {
            $stmt = $pdo->prepare("
                INSERT INTO configuration (config_key, config_value) 
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)
            ");
            $stmt->execute([$key, $value]);
        }
        
        logSystemAction($current_user['id'], 'update_config', $config_updates);
        
        $message = "✅ Configuration mise à jour avec succès";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur mise à jour config : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 🗑️ NETTOYAGE SYSTÈME
if ($_POST['action'] ?? '' === 'system_cleanup') {
    try {
        $cleanup_results = [];
        
        // Nettoyer logs anciens
        if (isset($_POST['cleanup_logs'])) {
            $days = intval($_POST['logs_days'] ?? 30);
            $stmt = $pdo->prepare("DELETE FROM activity_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)");
            $stmt->execute([$days]);
            $cleanup_results['logs'] = $stmt->rowCount();
        }
        
        // Nettoyer sessions expirées
        if (isset($_POST['cleanup_sessions'])) {
            $stmt = $pdo->prepare("DELETE FROM user_sessions WHERE expires_at < NOW()");
            $stmt->execute();
            $cleanup_results['sessions'] = $stmt->rowCount();
        }
        
        // Nettoyer tokens expirés
        if (isset($_POST['cleanup_tokens'])) {
            $stmt = $pdo->prepare("UPDATE user_token_packages SET status = 'expired' WHERE expires_at < NOW() AND status = 'active'");
            $stmt->execute();
            $cleanup_results['tokens'] = $stmt->rowCount();
        }
        
        // Nettoyer médias orphelins
        if (isset($_POST['cleanup_media'])) {
            $stmt = $pdo->prepare("
                SELECT cm.file_path 
                FROM clipmaker_media cm
                LEFT JOIN clipmaker_projects cp ON cm.project_id = cp.id
                WHERE cp.id IS NULL AND cm.created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)
            ");
            $stmt->execute();
            $orphan_files = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            foreach ($orphan_files as $file_path) {
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
            }
            
            $stmt = $pdo->prepare("
                DELETE cm FROM clipmaker_media cm
                LEFT JOIN clipmaker_projects cp ON cm.project_id = cp.id
                WHERE cp.id IS NULL AND cm.created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)
            ");
            $stmt->execute();
            $cleanup_results['media'] = $stmt->rowCount() + count($orphan_files);
        }
        
        logSystemAction($current_user['id'], 'system_cleanup', $cleanup_results);
        
        $message = "✅ Nettoyage système terminé : " . json_encode($cleanup_results);
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur nettoyage : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 🔄 BACKUP BASE DE DONNÉES
if ($_POST['action'] ?? '' === 'create_backup') {
    try {
        $backup_filename = 'smg_backup_' . date('Y-m-d_H-i-s') . '.sql';
        $backup_path = 'backups/' . $backup_filename;
        
        // Créer le dossier s'il n'existe pas
        if (!is_dir('backups')) {
            mkdir('backups', 0755, true);
        }
        
        // Commande mysqldump (à adapter selon votre config)
        $host = DB_HOST;
        $username = DB_USER;
        $password = DB_PASS;
        $database = DB_NAME;
        
        $command = "mysqldump --host=$host --user=$username --password=$password $database > $backup_path";
        $output = shell_exec($command);
        
        if (file_exists($backup_path) && filesize($backup_path) > 0) {
            logSystemAction($current_user['id'], 'create_backup', [
                'backup_file' => $backup_filename,
                'backup_size' => filesize($backup_path)
            ]);
            
            $message = "✅ Backup créé avec succès : $backup_filename";
            $message_type = 'success';
        } else {
            throw new Exception("Échec création backup");
        }
        
    } catch (Exception $e) {
        $message = "❌ Erreur backup : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 📊 OPTIMISATION BASE
if ($_POST['action'] ?? '' === 'optimize_database') {
    try {
        $optimization_results = [];
        
        // Récupérer toutes les tables
        $stmt = $pdo->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($tables as $table) {
            // Optimiser chaque table
            $stmt = $pdo->query("OPTIMIZE TABLE `$table`");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $optimization_results[$table] = $result['Msg_text'] ?? 'OK';
        }
        
        logSystemAction($current_user['id'], 'optimize_database', $optimization_results);
        
        $message = "✅ Optimisation base terminée : " . count($tables) . " tables traitées";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur optimisation : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DONNÉES SYSTÈME
// ================================================================

// Informations système
$system_info = getSystemInfo();

// Configuration actuelle
$current_config = getCurrentConfig();

// Stats performance
$performance_stats = getPerformanceStats();

// Logs système récents
$system_logs = getSystemLogs();

// Alertes système
$system_alerts = getSystemAlerts();

// Backups disponibles
$available_backups = getAvailableBackups();

// ================================================================
// 🔧 FONCTIONS MÉTIER SYSTÈME
// ================================================================

function getSystemInfo() {
    global $pdo;
    
    try {
        // Infos PHP et serveur
        $php_version = phpversion();
        $server_software = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
        $memory_limit = ini_get('memory_limit');
        $max_execution_time = ini_get('max_execution_time');
        $upload_max_filesize = ini_get('upload_max_filesize');
        
        // Infos base de données
        $stmt = $pdo->query("SELECT VERSION() as mysql_version");
        $mysql_version = $stmt->fetchColumn();
        
        // Espace disque
        $disk_free = disk_free_space('.') ?? 0;
        $disk_total = disk_total_space('.') ?? 1;
        $disk_used_percent = round((($disk_total - $disk_free) / $disk_total) * 100, 2);
        
        // Stats base
        $stmt = $pdo->query("
            SELECT 
                COUNT(*) as total_tables,
                SUM(data_length + index_length) as total_size
            FROM information_schema.tables 
            WHERE table_schema = DATABASE()
        ");
        $db_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Temps de réponse système
        $start_time = microtime(true);
        $stmt = $pdo->query("SELECT 1");
        $stmt->fetch();
        $db_response_time = round((microtime(true) - $start_time) * 1000, 2);
        
        return [
            'php_version' => $php_version,
            'server_software' => $server_software,
            'mysql_version' => $mysql_version,
            'memory_limit' => $memory_limit,
            'max_execution_time' => $max_execution_time,
            'upload_max_filesize' => $upload_max_filesize,
            'disk_free' => $disk_free,
            'disk_total' => $disk_total,
            'disk_used_percent' => $disk_used_percent,
            'total_tables' => $db_stats['total_tables'],
            'db_size' => $db_stats['total_size'],
            'db_response_time' => $db_response_time
        ];
        
    } catch (Exception $e) {
        error_log("Erreur getSystemInfo: " . $e->getMessage());
        return [];
    }
}

function getCurrentConfig() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT config_key, config_value FROM configuration");
        $config = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $config[$row['config_key']] = $row['config_value'];
        }
        
        // Valeurs par défaut si non définies
        $defaults = [
            'default_ai_model' => 'gpt-3.5-turbo',
            'max_tokens_per_request' => 1000,
            'ai_temperature' => 0.7,
            'default_token_package' => 'starter',
            'token_expiry_days' => 365,
            'maintenance_mode' => '0',
            'max_file_size' => 100
        ];
        
        return array_merge($defaults, $config);
        
    } catch (Exception $e) {
        error_log("Erreur getCurrentConfig: " . $e->getMessage());
        return [];
    }
}

function getPerformanceStats() {
    global $pdo;
    
    try {
        $stats = [];
        
        // Requêtes IA dernières 24h
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as ai_requests,
                AVG(processing_time_ms) as avg_processing_time,
                COUNT(CASE WHEN success = 0 THEN 1 END) as failed_requests
            FROM ai_usage_logs 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute();
        $ai_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Connexions utilisateurs dernières 24h
        $stmt = $pdo->prepare("
            SELECT COUNT(DISTINCT user_id) as active_users
            FROM log_connexions 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute();
        $users_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Uploads dernières 24h
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as uploads_count,
                SUM(file_size) as total_size
            FROM clipmaker_media 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute();
        $uploads_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Erreurs système dernières 24h
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as error_count
            FROM activity_logs 
            WHERE action = 'error' AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute();
        $errors_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return array_merge($ai_stats, $users_stats, $uploads_stats, $errors_stats);
        
    } catch (Exception $e) {
        error_log("Erreur getPerformanceStats: " . $e->getMessage());
        return [
            'ai_requests' => 0,
            'avg_processing_time' => 0,
            'failed_requests' => 0,
            'active_users' => 0,
            'uploads_count' => 0,
            'total_size' => 0,
            'error_count' => 0
        ];
    }
}

function getSystemLogs() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                al.*,
                u.name as user_name
            FROM activity_logs al
            LEFT JOIN users u ON al.user_id = u.id
            WHERE al.action IN ('system_cleanup', 'create_backup', 'optimize_database', 'update_config', 'error')
            ORDER BY al.created_at DESC
            LIMIT 50
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Erreur getSystemLogs: " . $e->getMessage());
        return [];
    }
}

function getSystemAlerts() {
    global $pdo;
    
    $alerts = [];
    
    try {
        // Alerte espace disque
        $disk_free = disk_free_space('.');
        $disk_total = disk_total_space('.');
        if ($disk_free && $disk_total) {
            $disk_percent = (($disk_total - $disk_free) / $disk_total) * 100;
            if ($disk_percent > 85) {
                $alerts[] = [
                    'type' => 'warning',
                    'title' => 'Espace disque faible',
                    'message' => "Utilisation disque : " . round($disk_percent, 1) . "%",
                    'action' => 'cleanup_media'
                ];
            }
        }
        
        // Alerte tokens en fin de stock
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as low_balance_users
            FROM user_token_balance 
            WHERE current_balance < 100 AND current_balance > 0
        ");
        $stmt->execute();
        $low_balance = $stmt->fetchColumn();
        
        if ($low_balance > 5) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'Utilisateurs avec peu de tokens',
                'message' => "$low_balance utilisateurs ont moins de 100 tokens",
                'action' => 'view_tokens'
            ];
        }
        
        // Alerte quotas IA dépassés
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as exceeded_quotas
            FROM user_ai_quotas 
            WHERE tokens_used >= tokens_limit AND is_active = 1
        ");
        $stmt->execute();
        $exceeded_quotas = $stmt->fetchColumn();
        
        if ($exceeded_quotas > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Quotas IA dépassés',
                'message' => "$exceeded_quotas quotas ont atteint leur limite",
                'action' => 'view_quotas'
            ];
        }
        
        // Alerte erreurs IA fréquentes
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as recent_errors
            FROM ai_usage_logs 
            WHERE success = 0 AND created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        $stmt->execute();
        $recent_errors = $stmt->fetchColumn();
        
        if ($recent_errors > 10) {
            $alerts[] = [
                'type' => 'danger',
                'title' => 'Erreurs IA fréquentes',
                'message' => "$recent_errors erreurs IA dans la dernière heure",
                'action' => 'check_ai_logs'
            ];
        }
        
        // Alerte backups anciens
        $backups = getAvailableBackups();
        if (empty($backups)) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Aucun backup récent',
                'message' => "Aucun backup trouvé - Créez un backup",
                'action' => 'create_backup'
            ];
        }
        
        return $alerts;
        
    } catch (Exception $e) {
        error_log("Erreur getSystemAlerts: " . $e->getMessage());
        return [];
    }
}

function getAvailableBackups() {
    $backups = [];
    $backup_dir = 'backups/';
    
    if (is_dir($backup_dir)) {
        $files = scandir($backup_dir);
        foreach ($files as $file) {
            if (strpos($file, 'smg_backup_') === 0 && substr($file, -4) === '.sql') {
                $full_path = $backup_dir . $file;
                $backups[] = [
                    'filename' => $file,
                    'size' => filesize($full_path),
                    'created_at' => filemtime($full_path)
                ];
            }
        }
        
        // Trier par date de création (plus récent en premier)
        usort($backups, function($a, $b) {
            return $b['created_at'] - $a['created_at'];
        });
    }
    
    return $backups;
}

function logSystemAction($admin_id, $action, $details) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, entity_type, description, details, ip_address, created_at)
            VALUES (?, ?, 'system', ?, ?, ?, NOW())
        ");
        
        $description = "Action système admin: $action";
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        $stmt->execute([
            $admin_id, 
            $action, 
            $description, 
            json_encode($details), 
            $ip
        ]);
    } catch (Exception $e) {
        error_log("Erreur logSystemAction: " . $e->getMessage());
    }
}

function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

function formatDate($timestamp) {
    return date('d/m/Y H:i:s', $timestamp);
}

function getAlertBadge($type) {
    $badges = [
        'info' => '<span class="badge badge-info">ℹ️ Info</span>',
        'warning' => '<span class="badge badge-warning">⚠️ Alerte</span>',
        'danger' => '<span class="badge badge-danger">🚨 Critique</span>',
        'success' => '<span class="badge badge-success">✅ OK</span>'
    ];
    return $badges[$type] ?? $type;
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>⚙️ Administration Système - SMG Studio v2.0</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #9b59b6, #8e44ad);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .nav-ia-principale {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-principale a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-principale a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .admin-nav {
            display: flex;
            gap: 8px;
            background: rgba(155, 89, 182, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(155, 89, 182, 0.2);
        }

        .admin-nav a {
            padding: 8px 16px;
            background: rgba(155, 89, 182, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .admin-nav a:hover, .admin-nav a.active {
            background: #9b59b6;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(155, 89, 182, 0.4);
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #9b59b6;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .stat-details {
            margin-top: 10px;
            font-size: 11px;
            color: #999;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #9b59b6, #8e44ad);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #27ae60, #229954);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .alert-warning {
            background: #fff3cd;
            border-color: #ffeaa7;
            color: #856404;
        }

        .alert-info {
            background: #d1ecf1;
            border-color: #bee5eb;
            color: #0c5460;
        }

        .system-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .system-table th,
        .system-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #dee2e6;
        }

        .system-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
        }

        .system-table tr:hover {
            background: #f8f9fa;
        }

        .badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }

        .badge-danger {
            background: #f8d7da;
            color: #721c24;
        }

        .badge-info {
            background: #d1ecf1;
            color: #0c5460;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }

        .form-control {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: #9b59b6;
            box-shadow: 0 0 0 3px rgba(155, 89, 182, 0.1);
        }

        .form-select {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            background: white;
        }

        .tabs {
            display: flex;
            gap: 5px;
            margin-bottom: 20px;
            border-bottom: 2px solid #f8f9fa;
        }

        .tab {
            padding: 10px 20px;
            background: none;
            border: none;
            border-bottom: 3px solid transparent;
            cursor: pointer;
            font-weight: 600;
            color: #666;
            transition: all 0.3s ease;
        }

        .tab.active {
            color: #9b59b6;
            border-bottom-color: #9b59b6;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        .config-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }

        .config-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            border-left: 4px solid #9b59b6;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e9ecef;
            border-radius: 4px;
            overflow: hidden;
            margin: 5px 0;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #27ae60, #f39c12, #e74c3c);
            transition: width 0.3s ease;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .nav-ia-principale, .admin-nav {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .config-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">⚙️</div>
                <div>
                    <h1>Administration Système</h1>
                    <p>Contrôle Total SMG Studio v2.0</p>
                </div>
            </div>
            
            <div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                <!-- Navigation IA -->
                <div class="nav-ia-principale">
                    <a href="ai_content_generator.php">🎨 IA Générateur</a>
                    <a href="ai_dashboard.php">📊 IA Dashboard</a>
                    <a href="ai_profiles.php">⚙️ IA Profils</a>
                </div>
                
                <!-- Navigation Admin -->
                <div class="admin-nav">
                    <a href="users_management.php">👥 Utilisateurs</a>
                    <a href="tokens_management.php">💰 Tokens</a>
                    <a href="system_admin.php" class="active">⚙️ Système</a>
                </div>
                
                <!-- Retour Dashboard -->
                <a href="dashboard.php" class="btn btn-primary">
                    📊 Dashboard Principal
                </a>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Alertes Système -->
        <?php if (!empty($system_alerts)): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">🚨 Alertes Système</h2>
                </div>
                
                <?php foreach ($system_alerts as $alert): ?>
                    <div class="alert alert-<?= $alert['type'] ?>">
                        <?= getAlertBadge($alert['type']) ?>
                        <div>
                            <strong><?= htmlspecialchars($alert['title']) ?></strong><br>
                            <?= htmlspecialchars($alert['message']) ?>
                        </div>
                        <button onclick="handleAlert('<?= $alert['action'] ?>')" class="btn btn-sm btn-primary">
                            🔧 Action
                        </button>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <!-- Stats Système -->
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-icon">💻</span>
                <div class="stat-value">PHP <?= $system_info['php_version'] ?? 'N/A' ?></div>
                <div class="stat-label">Version PHP</div>
                <div class="stat-details">MySQL <?= $system_info['mysql_version'] ?? 'N/A' ?></div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">💾</span>
                <div class="stat-value"><?= $system_info['disk_used_percent'] ?? 0 ?>%</div>
                <div class="stat-label">Utilisation Disque</div>
                <div class="stat-details">
                    <?= formatBytes($system_info['disk_free'] ?? 0) ?> libre
                </div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🗄️</span>
                <div class="stat-value"><?= formatBytes($system_info['db_size'] ?? 0) ?></div>
                <div class="stat-label">Taille Base</div>
                <div class="stat-details"><?= $system_info['total_tables'] ?? 0 ?> tables</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">⚡</span>
                <div class="stat-value"><?= $system_info['db_response_time'] ?? 0 ?>ms</div>
                <div class="stat-label">Temps Réponse DB</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🤖</span>
                <div class="stat-value"><?= number_format($performance_stats['ai_requests'] ?? 0) ?></div>
                <div class="stat-label">Requêtes IA (24h)</div>
                <div class="stat-details"><?= $performance_stats['failed_requests'] ?? 0 ?> échecs</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">👥</span>
                <div class="stat-value"><?= $performance_stats['active_users'] ?? 0 ?></div>
                <div class="stat-label">Utilisateurs Actifs (24h)</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📁</span>
                <div class="stat-value"><?= number_format($performance_stats['uploads_count'] ?? 0) ?></div>
                <div class="stat-label">Uploads (24h)</div>
                <div class="stat-details"><?= formatBytes($performance_stats['total_size'] ?? 0) ?></div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">❌</span>
                <div class="stat-value"><?= $performance_stats['error_count'] ?? 0 ?></div>
                <div class="stat-label">Erreurs (24h)</div>
            </div>
        </div>

        <!-- Onglets Administration -->
        <div class="content-card">
            <div class="tabs">
                <button class="tab active" onclick="showTab('configuration')">⚙️ Configuration</button>
                <button class="tab" onclick="showTab('maintenance')">🔧 Maintenance</button>
                <button class="tab" onclick="showTab('backups')">💾 Backups</button>
                <button class="tab" onclick="showTab('logs')">📋 Logs Système</button>
                <button class="tab" onclick="showTab('monitoring')">📊 Monitoring</button>
            </div>

            <!-- Onglet Configuration -->
            <div id="configuration" class="tab-content active">
                <div class="card-header">
                    <h3 class="card-title">⚙️ Configuration Système</h3>
                </div>
                
                <form method="POST">
                    <input type="hidden" name="action" value="update_config">
                    
                    <div class="config-grid">
                        <!-- Configuration IA -->
                        <div class="config-section">
                            <h4>🤖 Configuration IA</h4>
                            
                            <div class="form-group">
                                <label class="form-label">Modèle IA par défaut</label>
                                <select name="default_ai_model" class="form-select">
                                    <option value="gpt-3.5-turbo" <?= ($current_config['default_ai_model'] ?? '') === 'gpt-3.5-turbo' ? 'selected' : '' ?>>GPT-3.5 Turbo</option>
                                    <option value="gpt-4" <?= ($current_config['default_ai_model'] ?? '') === 'gpt-4' ? 'selected' : '' ?>>GPT-4</option>
                                    <option value="claude-3" <?= ($current_config['default_ai_model'] ?? '') === 'claude-3' ? 'selected' : '' ?>>Claude 3</option>
                                    <option value="gemini-pro" <?= ($current_config['default_ai_model'] ?? '') === 'gemini-pro' ? 'selected' : '' ?>>Gemini Pro</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Max tokens par requête</label>
                                <input type="number" name="max_tokens_per_request" class="form-control" 
                                       value="<?= $current_config['max_tokens_per_request'] ?? 1000 ?>" min="100" max="4000">
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Température IA (créativité)</label>
                                <input type="number" name="ai_temperature" class="form-control" 
                                       value="<?= $current_config['ai_temperature'] ?? 0.7 ?>" step="0.1" min="0" max="2">
                            </div>
                        </div>
                        
                        <!-- Configuration Tokens -->
                        <div class="config-section">
                            <h4>💰 Configuration Tokens</h4>
                            
                            <div class="form-group">
                                <label class="form-label">Package par défaut</label>
                                <select name="default_token_package" class="form-select">
                                    <option value="starter" <?= ($current_config['default_token_package'] ?? '') === 'starter' ? 'selected' : '' ?>>Starter (1000 tokens)</option>
                                    <option value="business" <?= ($current_config['default_token_package'] ?? '') === 'business' ? 'selected' : '' ?>>Business (5000 tokens)</option>
                                    <option value="pro" <?= ($current_config['default_token_package'] ?? '') === 'pro' ? 'selected' : '' ?>>Pro (10000 tokens)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Expiration tokens (jours)</label>
                                <input type="number" name="token_expiry_days" class="form-control" 
                                       value="<?= $current_config['token_expiry_days'] ?? 365 ?>" min="30" max="3650">
                            </div>
                        </div>
                        
                        <!-- Configuration Système -->
                        <div class="config-section">
                            <h4>🖥️ Configuration Système</h4>
                            
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="maintenance_mode" value="1" 
                                           <?= ($current_config['maintenance_mode'] ?? '0') === '1' ? 'checked' : '' ?>>
                                    Mode maintenance
                                </label>
                                <small>Bloque l'accès aux utilisateurs non-admin</small>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Taille max fichiers (MB)</label>
                                <input type="number" name="max_file_size" class="form-control" 
                                       value="<?= $current_config['max_file_size'] ?? 100 ?>" min="1" max="1000">
                            </div>
                        </div>
                    </div>
                    
                    <div class="action-buttons">
                        <button type="submit" class="btn btn-success">
                            ✅ Sauvegarder Configuration
                        </button>
                        <button type="button" onclick="resetConfig()" class="btn btn-warning">
                            🔄 Réinitialiser
                        </button>
                    </div>
                </form>
            </div>

            <!-- Onglet Maintenance -->
            <div id="maintenance" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">🔧 Outils de Maintenance</h3>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                    <!-- Nettoyage Système -->
                    <div class="config-section">
                        <h4>🗑️ Nettoyage Système</h4>
                        <p>Supprime les données obsolètes pour optimiser les performances</p>
                        
                        <form method="POST">
                            <input type="hidden" name="action" value="system_cleanup">
                            
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="cleanup_logs" checked>
                                    Nettoyer logs anciens
                                </label>
                                <input type="number" name="logs_days" value="30" min="7" max="365" style="width: 60px; margin-left: 10px;">
                                jours
                            </div>
                            
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="cleanup_sessions" checked>
                                    Nettoyer sessions expirées
                                </label>
                            </div>
                            
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="cleanup_tokens">
                                    Marquer tokens expirés
                                </label>
                            </div>
                            
                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="cleanup_media">
                                    Supprimer médias orphelins
                                </label>
                            </div>
                            
                            <button type="submit" class="btn btn-warning" onclick="return confirm('⚠️ Confirmer le nettoyage ?')">
                                🗑️ Lancer Nettoyage
                            </button>
                        </form>
                    </div>
                    
                    <!-- Optimisation Base -->
                    <div class="config-section">
                        <h4>📊 Optimisation Base</h4>
                        <p>Optimise les tables de la base de données</p>
                        
                        <form method="POST">
                            <input type="hidden" name="action" value="optimize_database">
                            
                            <p><strong>Actions :</strong></p>
                            <ul>
                                <li>✅ Optimiser toutes les tables</li>
                                <li>✅ Reconstruire les index</li>
                                <li>✅ Mettre à jour statistiques</li>
                                <li>✅ Libérer espace fragmenté</li>
                            </ul>
                            
                            <button type="submit" class="btn btn-primary" onclick="return confirm('⚠️ L\'optimisation peut prendre du temps. Continuer ?')">
                                ⚡ Optimiser Base
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Onglet Backups -->
            <div id="backups" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">💾 Gestion des Backups</h3>
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="action" value="create_backup">
                        <button type="submit" class="btn btn-success">
                            ➕ Créer Backup
                        </button>
                    </form>
                </div>
                
                <table class="system-table">
                    <thead>
                        <tr>
                            <th>📁 Fichier</th>
                            <th>📏 Taille</th>
                            <th>📅 Date création</th>
                            <th>⚡ Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($available_backups)): ?>
                            <tr>
                                <td colspan="4" style="text-align: center; color: #999;">
                                    🚫 Aucun backup disponible
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($available_backups as $backup): ?>
                                <tr>
                                    <td>
                                        <strong><?= htmlspecialchars($backup['filename']) ?></strong>
                                    </td>
                                    <td><?= formatBytes($backup['size']) ?></td>
                                    <td><?= formatDate($backup['created_at']) ?></td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="backups/<?= htmlspecialchars($backup['filename']) ?>" 
                                               class="btn btn-primary btn-sm" download>
                                                📥 Télécharger
                                            </a>
                                            <button onclick="deleteBackup('<?= htmlspecialchars($backup['filename']) ?>')" 
                                                    class="btn btn-danger btn-sm">
                                                🗑️ Supprimer
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Logs -->
            <div id="logs" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">📋 Logs Système</h3>
                    <button onclick="refreshLogs()" class="btn btn-primary">
                        🔄 Actualiser
                    </button>
                </div>
                
                <table class="system-table">
                    <thead>
                        <tr>
                            <th>📅 Date</th>
                            <th>👤 Utilisateur</th>
                            <th>⚡ Action</th>
                            <th>📋 Description</th>
                            <th>🔍 Détails</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($system_logs as $log): ?>
                            <tr>
                                <td><?= formatDate(strtotime($log['created_at'])) ?></td>
                                <td>
                                    <?php if ($log['user_name']): ?>
                                        <strong><?= htmlspecialchars($log['user_name']) ?></strong>
                                    <?php else: ?>
                                        <em>Système</em>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="badge badge-info"><?= htmlspecialchars($log['action']) ?></span>
                                </td>
                                <td><?= htmlspecialchars($log['description']) ?></td>
                                <td>
                                    <?php if ($log['details']): ?>
                                        <button onclick="showDetails(<?= htmlspecialchars($log['details']) ?>)" 
                                                class="btn btn-sm btn-primary">
                                            🔍 Voir
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Monitoring -->
            <div id="monitoring" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">📊 Monitoring Temps Réel</h3>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                    <!-- Monitoring Serveur -->
                    <div class="config-section">
                        <h4>🖥️ Serveur</h4>
                        <div class="form-group">
                            <label>💾 Utilisation Disque</label>
                            <div class="progress-bar">
                                <div class="progress-fill" style="width: <?= $system_info['disk_used_percent'] ?? 0 ?>%;"></div>
                            </div>
                            <small><?= $system_info['disk_used_percent'] ?? 0 ?>% utilisé</small>
                        </div>
                        
                        <div class="form-group">
                            <label>🧠 Limite Mémoire PHP</label>
                            <strong><?= $system_info['memory_limit'] ?? 'N/A' ?></strong>
                        </div>
                        
                        <div class="form-group">
                            <label>⏱️ Temps Exécution Max</label>
                            <strong><?= $system_info['max_execution_time'] ?? 'N/A' ?>s</strong>
                        </div>
                    </div>
                    
                    <!-- Monitoring IA -->
                    <div class="config-section">
                        <h4>🤖 Performance IA</h4>
                        <div class="form-group">
                            <label>📊 Requêtes 24h</label>
                            <strong><?= number_format($performance_stats['ai_requests'] ?? 0) ?></strong>
                        </div>
                        
                        <div class="form-group">
                            <label>⏱️ Temps Moyen Traitement</label>
                            <strong><?= number_format($performance_stats['avg_processing_time'] ?? 0) ?>ms</strong>
                        </div>
                        
                        <div class="form-group">
                            <label>❌ Taux d'Échec</label>
                            <?php 
                            $total = $performance_stats['ai_requests'] ?? 1;
                            $failed = $performance_stats['failed_requests'] ?? 0;
                            $failure_rate = round(($failed / $total) * 100, 2);
                            ?>
                            <strong style="color: <?= $failure_rate > 5 ? '#e74c3c' : '#27ae60' ?>;">
                                <?= $failure_rate ?>%
                            </strong>
                        </div>
                    </div>
                    
                    <!-- Monitoring Utilisateurs -->
                    <div class="config-section">
                        <h4>👥 Activité Utilisateurs</h4>
                        <div class="form-group">
                            <label>🟢 Utilisateurs Actifs 24h</label>
                            <strong><?= $performance_stats['active_users'] ?? 0 ?></strong>
                        </div>
                        
                        <div class="form-group">
                            <label>📁 Uploads 24h</label>
                            <strong><?= number_format($performance_stats['uploads_count'] ?? 0) ?></strong>
                            <br><small><?= formatBytes($performance_stats['total_size'] ?? 0) ?></small>
                        </div>
                        
                        <div class="form-group">
                            <label>❌ Erreurs 24h</label>
                            <strong style="color: <?= ($performance_stats['error_count'] ?? 0) > 10 ? '#e74c3c' : '#27ae60' ?>;">
                                <?= $performance_stats['error_count'] ?? 0 ?>
                            </strong>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Gestion des onglets
        function showTab(tabName) {
            // Masquer tous les contenus
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // Retirer active de tous les onglets
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Afficher le contenu sélectionné
            document.getElementById(tabName).classList.add('active');
            
            // Activer l'onglet
            event.target.classList.add('active');
        }

        // Actions alertes
        function handleAlert(action) {
            switch(action) {
                case 'cleanup_media':
                    showTab('maintenance');
                    break;
                case 'view_tokens':
                    window.open('tokens_management.php', '_blank');
                    break;
                case 'view_quotas':
                    window.open('tokens_management.php?tab=ai_quotas', '_blank');
                    break;
                case 'check_ai_logs':
                    window.open('ai_dashboard.php', '_blank');
                    break;
                case 'create_backup':
                    showTab('backups');
                    break;
                default:
                    alert('🚧 Action: ' + action);
            }
        }

        // Actions configuration
        function resetConfig() {
            if (confirm('⚠️ Réinitialiser toute la configuration aux valeurs par défaut ?')) {
                // Implémenter reset config
                alert('🚧 Fonctionnalité en développement : Reset configuration');
            }
        }

        // Actions backups
        function deleteBackup(filename) {
            if (confirm(`⚠️ Supprimer définitivement le backup "${filename}" ?`)) {
                // Implémenter suppression backup
                alert('🚧 Fonctionnalité en développement : Suppression backup');
            }
        }

        // Actions logs
        function refreshLogs() {
            window.location.reload();
        }

        function showDetails(details) {
            try {
                const data = JSON.parse(details);
                alert('📋 Détails :\n\n' + JSON.stringify(data, null, 2));
            } catch (e) {
                alert('📋 Détails :\n\n' + details);
            }
        }

        // Auto-refresh monitoring (toutes les 30 secondes sur onglet monitoring)
        let monitoringInterval;
        
        function startMonitoring() {
            if (document.getElementById('monitoring').classList.contains('active')) {
                monitoringInterval = setInterval(() => {
                    // Actualiser uniquement les données de monitoring
                    fetch('system_admin.php?ajax=monitoring')
                        .then(response => response.json())
                        .then(data => {
                            // Mettre à jour les valeurs en temps réel
                            console.log('Monitoring data updated', data);
                        })
                        .catch(error => console.error('Erreur monitoring:', error));
                }, 30000);
            }
        }
        
        function stopMonitoring() {
            if (monitoringInterval) {
                clearInterval(monitoringInterval);
            }
        }

        // Event listeners
        document.querySelectorAll('.tab').forEach(tab => {
            tab.addEventListener('click', () => {
                stopMonitoring();
                if (tab.textContent.includes('Monitoring')) {
                    setTimeout(startMonitoring, 100);
                }
            });
        });

        console.log('⚙️ SMG Studio v2.0 - Administration Système Ready!');
        console.log('✅ Contrôle total système par Papy 77');
        console.log('🔧 Config + maintenance + monitoring + backups');
    </script>
</body>
</html>