<?php
/**
 * ================================================================
 * 🔄 SMG WORKFLOW HUB - ORCHESTRATEUR COMPLET
 * ================================================================
 * 
 * 🎯 WORKFLOW SMG STUDIO COMPLET :
 * ✅ Navigation entre étapes 1-4
 * ✅ Détection état utilisateur
 * ✅ Gestion fichiers workflow
 * ✅ Intégration base données
 * 
 * 💡 BY PAPY 77 - WORKFLOW ORCHESTRATOR
 * ================================================================
 */

require_once '../../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// ================================================================
// 🔧 DÉTECTION ÉTAPE WORKFLOW
// ================================================================

function detectCurrentStep($pdo, $user_id) {
    try {
        // Vérifier les exports workflow
        if (tableExists($pdo, 'workflow_exports')) {
            $stmt = $pdo->prepare("
                SELECT stage, status, COUNT(*) as count
                FROM workflow_exports 
                WHERE user_id = ? 
                GROUP BY stage, status
                ORDER BY 
                    CASE stage 
                        WHEN 'clipmaker' THEN 1
                        WHEN 'audio_postprod' THEN 2  
                        WHEN 'emission_final' THEN 3
                        WHEN 'social_ready' THEN 4
                    END DESC
            ");
            $stmt->execute([$user_id]);
            $exports = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Logique détection étape
            $completed_stages = [];
            foreach ($exports as $export) {
                if ($export['status'] === 'completed') {
                    $completed_stages[] = $export['stage'];
                }
            }
            
            if (in_array('social_ready', $completed_stages)) return 4;
            if (in_array('emission_final', $completed_stages)) return 4;
            if (in_array('audio_postprod', $completed_stages)) return 3;
            if (in_array('clipmaker', $completed_stages)) return 2;
        }
        
        // Fallback - vérifier clipmaker_projects
        if (tableExists($pdo, 'clipmaker_projects')) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM clipmaker_projects WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $project_count = $stmt->fetchColumn();
            
            if ($project_count > 0) return 1; // Retour audio post-prod
        }
        
        return 1; // Démarrer workflow
        
    } catch (Exception $e) {
        error_log("Erreur détection workflow: " . $e->getMessage());
        return 1;
    }
}

// ================================================================
// 🔧 GESTION ÉTAPES WORKFLOW
// ================================================================

function getWorkflowProgress($pdo, $user_id) {
    $progress = [
        'clipmaker' => false,
        'audio_postprod' => false,
        'emission_final' => false,
        'social_ready' => false
    ];
    
    try {
        // Check clipmaker projects
        if (tableExists($pdo, 'clipmaker_projects')) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM clipmaker_projects WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $progress['clipmaker'] = $stmt->fetchColumn() > 0;
        }
        
        // Check workflow exports
        if (tableExists($pdo, 'workflow_exports')) {
            $stmt = $pdo->prepare("
                SELECT stage 
                FROM workflow_exports 
                WHERE user_id = ? AND status = 'completed'
                GROUP BY stage
            ");
            $stmt->execute([$user_id]);
            $completed = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            foreach ($completed as $stage) {
                $progress[$stage] = true;
            }
        }
        
    } catch (Exception $e) {
        error_log("Erreur progress workflow: " . $e->getMessage());
    }
    
    return $progress;
}

// ================================================================
// 🔧 ÉTAPE DEMANDÉE
// ================================================================

$requested_step = intval($_GET['step'] ?? 0);
$auto_detect = $_GET['auto'] ?? 'true';

// Auto-détection ou étape forcée
if ($requested_step === 0 || $auto_detect === 'true') {
    $current_step = detectCurrentStep($pdo, $user['id']);
} else {
    $current_step = $requested_step;
}

// Limiter étapes valides
$current_step = max(1, min(4, $current_step));

// Récupérer progression
$workflow_progress = getWorkflowProgress($pdo, $user['id']);

// ================================================================
// 🔧 INFORMATIONS ÉTAPE
// ================================================================

$steps_info = [
    1 => [
        'title' => 'Audio Post-Production',
        'icon' => '🎙️',
        'description' => 'Enregistrement commentaire via KB11',
        'file' => 'smg_audio_postprod.html',
        'color' => '#ff6b6b'
    ],
    2 => [
        'title' => 'Mix Final Audio',
        'icon' => '🎵', 
        'description' => 'Commentaire + Musique avec KB11',
        'file' => 'smg_mix_final.html',
        'color' => '#667eea'
    ],
    3 => [
        'title' => 'Émission Finale',
        'icon' => '🎬',
        'description' => 'Post → Clip → Habillage → Export',
        'file' => 'smg_emission_finale.html',
        'color' => '#4a5568'
    ],
    4 => [
        'title' => 'Social Ready',
        'icon' => '📱',
        'description' => 'Multi-plateformes optimisé',
        'file' => 'smg_social_ready.html', 
        'color' => '#667eea'
    ]
];

// ================================================================
// 🔧 SÉLECTEUR D'ÉTAPE OU AFFICHAGE DIRECT
// ================================================================

$show_selector = $_GET['selector'] ?? 'false';

if ($show_selector === 'true') {
    // Afficher sélecteur d'étapes
    ?>
    <!DOCTYPE html>
    <html lang="fr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>🔄 SMG Workflow Hub</title>
        <style>
            * { margin: 0; padding: 0; box-sizing: border-box; }
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: #ffffff;
                min-height: 100vh;
                padding: 20px;
            }
            .container { max-width: 1200px; margin: 0 auto; }
            .header {
                background: rgba(255, 255, 255, 0.1);
                backdrop-filter: blur(20px);
                border-radius: 15px;
                padding: 30px;
                text-align: center;
                margin-bottom: 30px;
            }
            .steps-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
                gap: 25px;
            }
            .step-card {
                background: rgba(255, 255, 255, 0.05);
                backdrop-filter: blur(20px);
                border-radius: 15px;
                padding: 30px;
                text-align: center;
                transition: all 0.3s ease;
                cursor: pointer;
                border: 2px solid transparent;
                text-decoration: none;
                color: inherit;
            }
            .step-card:hover {
                transform: translateY(-10px);
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
            }
            .step-card.completed {
                border-color: #48bb78;
                background: rgba(72, 187, 120, 0.1);
            }
            .step-card.current {
                border-color: #FFD700;
                background: rgba(255, 215, 0, 0.1);
                box-shadow: 0 0 30px rgba(255, 215, 0, 0.3);
            }
            .step-icon {
                font-size: 48px;
                margin-bottom: 20px;
                display: block;
            }
            .step-title {
                font-size: 20px;
                font-weight: bold;
                margin-bottom: 10px;
            }
            .step-description {
                color: #a0aec0;
                font-size: 14px;
                margin-bottom: 20px;
            }
            .step-status {
                padding: 8px 15px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: bold;
                text-transform: uppercase;
            }
            .status-completed {
                background: rgba(72, 187, 120, 0.2);
                color: #48bb78;
            }
            .status-current {
                background: rgba(255, 215, 0, 0.2);
                color: #FFD700;
            }
            .status-pending {
                background: rgba(255, 255, 255, 0.1);
                color: #a0aec0;
            }
            .back-btn {
                background: rgba(255, 255, 255, 0.1);
                color: white;
                border: none;
                padding: 15px 25px;
                border-radius: 12px;
                cursor: pointer;
                font-weight: bold;
                margin-bottom: 30px;
                text-decoration: none;
                display: inline-block;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <a href="../../dashboard.php" class="back-btn">⬅️ Retour Dashboard</a>
            
            <div class="header">
                <h1>🔄 SMG Workflow Hub</h1>
                <p>Choisissez l'étape de workflow - User: <?php echo htmlspecialchars($user['name']); ?></p>
            </div>

            <div class="steps-grid">
                <?php foreach ($steps_info as $step_num => $info): ?>
                    <?php
                    $is_completed = false;
                    $is_current = ($step_num === $current_step);
                    
                    // Logique statut
                    if ($step_num === 1) {
                        $is_completed = $workflow_progress['clipmaker'];
                    } elseif ($step_num === 2) {
                        $is_completed = $workflow_progress['audio_postprod'];
                    } elseif ($step_num === 3) {
                        $is_completed = $workflow_progress['emission_final'];
                    } elseif ($step_num === 4) {
                        $is_completed = $workflow_progress['social_ready'];
                    }
                    
                    $card_class = '';
                    $status_class = 'status-pending';
                    $status_text = 'En attente';
                    
                    if ($is_completed) {
                        $card_class = 'completed';
                        $status_class = 'status-completed';
                        $status_text = 'Terminé';
                    } elseif ($is_current) {
                        $card_class = 'current';
                        $status_class = 'status-current';
                        $status_text = 'Actuel';
                    }
                    ?>
                    
                    <a href="workflow_hub.php?step=<?php echo $step_num; ?>&auto=false" class="step-card <?php echo $card_class; ?>">
                        <span class="step-icon"><?php echo $info['icon']; ?></span>
                        <div class="step-title">ÉTAPE <?php echo $step_num; ?></div>
                        <div class="step-title"><?php echo $info['title']; ?></div>
                        <div class="step-description"><?php echo $info['description']; ?></div>
                        <div class="step-status <?php echo $status_class; ?>">
                            <?php echo $status_text; ?>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>

            <div style="text-align: center; margin-top: 30px;">
                <a href="workflow_hub.php?auto=true" style="background: linear-gradient(135deg, #48bb78, #38a169); color: white; padding: 15px 30px; border-radius: 12px; text-decoration: none; font-weight: bold;">
                    🎯 Auto-Détection Étape
                </a>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// ================================================================
// 🔧 AFFICHAGE ÉTAPE WORKFLOW
// ================================================================

$step_info = $steps_info[$current_step];
$step_file = __DIR__ . '/' . $step_info['file'];

// Vérifier si fichier existe
if (!file_exists($step_file)) {
    die("❌ Erreur: Fichier workflow introuvable - " . $step_info['file']);
}

// ================================================================
// 🔧 INJECTION NAVIGATION DANS HTML
// ================================================================

$html_content = file_get_contents($step_file);

// Variables à injecter
$navigation_html = '
<div style="position: fixed; top: 20px; left: 20px; z-index: 2000;">
    <a href="../../dashboard.php" style="background: rgba(0,0,0,0.8); color: white; padding: 10px 15px; border-radius: 8px; text-decoration: none; font-size: 12px; font-weight: bold;">
        ⬅️ Dashboard
    </a>
</div>

<div style="position: fixed; top: 20px; right: 20px; z-index: 2000; display: flex; gap: 10px;">
    <a href="workflow_hub.php?selector=true" style="background: rgba(102, 126, 234, 0.9); color: white; padding: 10px 15px; border-radius: 8px; text-decoration: none; font-size: 12px; font-weight: bold;">
        🔄 Hub
    </a>
    ' . ($current_step > 1 ? '<a href="workflow_hub.php?step=' . ($current_step - 1) . '&auto=false" style="background: rgba(255, 107, 107, 0.9); color: white; padding: 10px 15px; border-radius: 8px; text-decoration: none; font-size: 12px; font-weight: bold;">⬅️ Précédent</a>' : '') . '
    ' . ($current_step < 4 ? '<a href="workflow_hub.php?step=' . ($current_step + 1) . '&auto=false" style="background: rgba(72, 187, 120, 0.9); color: white; padding: 10px 15px; border-radius: 8px; text-decoration: none; font-size: 12px; font-weight: bold;">Suivant ➡️</a>' : '') . '
</div>

<script>
// Variables workflow injectées
window.SMG_WORKFLOW = {
    currentStep: ' . $current_step . ',
    userId: ' . $user['id'] . ',
    userName: "' . addslashes($user['name']) . '",
    progress: ' . json_encode($workflow_progress) . ',
    hubUrl: "workflow_hub.php"
};

// Fonctions navigation
function nextStep() {
    if (window.SMG_WORKFLOW.currentStep < 4) {
        window.location.href = "workflow_hub.php?step=" + (window.SMG_WORKFLOW.currentStep + 1) + "&auto=false";
    }
}

function prevStep() {
    if (window.SMG_WORKFLOW.currentStep > 1) {
        window.location.href = "workflow_hub.php?step=" + (window.SMG_WORKFLOW.currentStep - 1) + "&auto=false";
    }
}

function goToHub() {
    window.location.href = "workflow_hub.php?selector=true";
}

console.log("🔄 SMG Workflow Hub - Étape " + window.SMG_WORKFLOW.currentStep);
console.log("👤 User: " + window.SMG_WORKFLOW.userName);
console.log("📊 Progress:", window.SMG_WORKFLOW.progress);
</script>
';

// Injecter navigation avant </body>
$html_content = str_replace('</body>', $navigation_html . '</body>', $html_content);

// Injecter variables PHP dans titre si possible
$workflow_title = "SMG Workflow - Étape {$current_step} : {$step_info['title']}";
$html_content = str_replace('<title>', '<title>' . $workflow_title . ' - ', $html_content);

// ================================================================
// 🔧 LOGGING WORKFLOW
// ================================================================

try {
    if (tableExists($pdo, 'activity_logs')) {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (user_id, action, entity_type, entity_id, description) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user['id'],
            'workflow_access',
            'workflow_step',
            $current_step,
            "Accès étape {$current_step}: {$step_info['title']}"
        ]);
    }
} catch (Exception $e) {
    error_log("Erreur log workflow: " . $e->getMessage());
}

// ================================================================
// 🔧 AFFICHAGE FINAL
// ================================================================

header('Content-Type: text/html; charset=UTF-8');
echo $html_content;

// Log console
error_log("🔄 SMG Workflow Hub - User {$user['id']} - Étape {$current_step} - " . $step_info['title']);
?>