<?php
/**
 * SCRIPT DE RENDU VIDÉO EN ARRIÈRE-PLAN
 * Traitement des exports Clipmaker en tâche de fond
 * Usage: php render_video.php [export_id]
 */

// Configuration
require_once __DIR__ . '/../auth_functions.php';

// Récupération de l'ID export depuis les arguments
$export_id = $argv[1] ?? null;
if (!$export_id) {
    echo "Usage: php render_video.php [export_id]\n";
    exit(1);
}

// Configuration DB
$config = [
    'db' => [
        'host' => getenv('DB_HOST') ?: 'localhost',
        'name' => getenv('DB_NAME') ?: 'laloedata', 
        'user' => getenv('DB_USER') ?: 'laloedata',
        'pass' => getenv('DB_PASS') ?: 'PiL256Mx78!',
        'charset' => 'utf8mb4'
    ]
];

/**
 * Classe de rendu vidéo
 */
class VideoRenderer {
    private $db;
    private $export_id;
    private $export_data;
    
    public function __construct($config, $export_id) {
        $this->export_id = $export_id;
        $this->connectDB($config);
        $this->loadExportData();
    }
    
    private function connectDB($config) {
        try {
            $dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['name']};charset={$config['db']['charset']}";
            $this->db = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } catch (PDOException $e) {
            $this->log("Erreur DB: " . $e->getMessage());
            exit(1);
        }
    }
    
    private function loadExportData() {
        $stmt = $this->db->prepare("
            SELECT we.*, cp.timeline_data, cp.name as project_name
            FROM workflow_exports we
            JOIN clipmaker_projects cp ON we.project_id = cp.id
            WHERE we.id = ?
        ");
        $stmt->execute([$this->export_id]);
        $this->export_data = $stmt->fetch();
        
        if (!$this->export_data) {
            $this->log("Export ID $this->export_id non trouvé");
            exit(1);
        }
    }
    
    public function render() {
        try {
            $this->log("Début du rendu pour export ID: {$this->export_id}");
            $this->updateStatus('processing', 0);
            
            // Décoder les données de timeline
            $timeline = json_decode($this->export_data['timeline_data'], true);
            
            if (!$timeline || empty($timeline['clips'])) {
                throw new Exception("Timeline vide ou invalide");
            }
            
            $this->log("Timeline chargée: " . count($timeline['clips']) . " clips");
            
            // Préparation des fichiers de séquences
            $sequences = $this->prepareSequences($timeline['clips']);
            $this->updateProgress(25);
            
            // Génération du fichier de concaténation FFmpeg
            $concat_file = $this->generateConcatFile($sequences);
            $this->updateProgress(50);
            
            // Rendu final avec FFmpeg
            $output_path = $this->renderFinalVideo($concat_file);
            $this->updateProgress(90);
            
            // Finalisation
            $this->finalizeExport($output_path);
            $this->updateProgress(100);
            $this->updateStatus('completed');
            
            $this->log("Rendu terminé avec succès: $output_path");
            
        } catch (Exception $e) {
            $this->log("Erreur de rendu: " . $e->getMessage());
            $this->updateStatus('failed', null, $e->getMessage());
        }
    }
    
    private function prepareSequences($clips) {
        $sequences = [];
        $temp_dir = '/tmp/clipmaker_' . $this->export_id . '/';
        
        if (!is_dir($temp_dir)) {
            mkdir($temp_dir, 0755, true);
        }
        
        foreach ($clips as $index => $clip) {
            $this->log("Traitement clip $index: {$clip['media']['name']}");
            
            // Récupération du média source
            $media_stmt = $this->db->prepare("SELECT * FROM clipmaker_media WHERE id = ?");
            $media_stmt->execute([$clip['media']['id']]);
            $media = $media_stmt->fetch();
            
            if (!$media) {
                $this->log("Média ID {$clip['media']['id']} non trouvé, skip");
                continue;
            }
            
            $source_path = $_SERVER['DOCUMENT_ROOT'] . $media['file_path'];
            
            // Découpage du clip selon in/out points
            if (isset($clip['in_point']) && isset($clip['out_point'])) {
                $sequence_path = $this->extractSequence(
                    $source_path, 
                    $clip['in_point'], 
                    $clip['out_point'], 
                    $temp_dir . "sequence_$index.mp4"
                );
            } else {
                // Copie directe si pas de découpage
                $sequence_path = $temp_dir . "sequence_$index.mp4";
                copy($source_path, $sequence_path);
            }
            
            $sequences[] = $sequence_path;
        }
        
        return $sequences;
    }
    
    private function extractSequence($source_path, $in_point, $out_point, $output_path) {
        $duration = $out_point - $in_point;
        
        $cmd = "ffmpeg -ss " . escapeshellarg($in_point) . 
               " -i " . escapeshellarg($source_path) . 
               " -t " . escapeshellarg($duration) . 
               " -c copy -avoid_negative_ts make_zero" .
               " -y " . escapeshellarg($output_path) . " 2>&1";
        
        $result = shell_exec($cmd);
        $this->log("Extraction: $cmd");
        
        if (!file_exists($output_path)) {
            throw new Exception("Échec extraction séquence: $result");
        }
        
        return $output_path;
    }
    
    private function generateConcatFile($sequences) {
        $concat_file = '/tmp/clipmaker_' . $this->export_id . '/concat.txt';
        $content = '';
        
        foreach ($sequences as $sequence) {
            $content .= "file '" . $sequence . "'\n";
        }
        
        file_put_contents($concat_file, $content);
        $this->log("Fichier concat généré: $concat_file");
        
        return $concat_file;
    }
    
    private function renderFinalVideo($concat_file) {
        $output_dir = $_SERVER['DOCUMENT_ROOT'] . '/uploads/clipmaker/exports/';
        if (!is_dir($output_dir)) {
            mkdir($output_dir, 0755, true);
        }
        
        $output_filename = $this->export_data['export_name'] . '_' . date('Y-m-d_H-i-s') . '.mp4';
        $output_path = $output_dir . $output_filename;
        
        // Commande FFmpeg selon le preset de qualité
        $preset = $this->getFFmpegPreset($this->export_data['quality_preset']);
        $resolution = $this->export_data['resolution_width'] . 'x' . $this->export_data['resolution_height'];
        $fps = $this->export_data['fps'];
        
        $cmd = "ffmpeg -f concat -safe 0 -i " . escapeshellarg($concat_file) . 
               " -vf \"scale=$resolution:force_original_aspect_ratio=decrease,pad=$resolution:(ow-iw)/2:(oh-ih)/2\"" .
               " -r $fps $preset" .
               " -c:a aac -b:a 192k -movflags +faststart" .
               " -y " . escapeshellarg($output_path) . " 2>&1";
        
        $this->log("Rendu final: $cmd");
        $result = shell_exec($cmd);
        
        if (!file_exists($output_path)) {
            throw new Exception("Échec rendu final: $result");
        }
        
        // Mise à jour du chemin fichier en DB
        $stmt = $this->db->prepare("
            UPDATE workflow_exports 
            SET file_path = ?, file_size = ? 
            WHERE id = ?
        ");
        $stmt->execute([
            '/uploads/clipmaker/exports/' . $output_filename,
            filesize($output_path),
            $this->export_id
        ]);
        
        return $output_path;
    }
    
    private function getFFmpegPreset($quality_preset) {
        $presets = [
            'draft' => '-crf 28 -preset ultrafast',
            'preview' => '-crf 23 -preset fast',
            'broadcast' => '-crf 18 -preset medium',
            'web' => '-crf 23 -preset fast',
            'mobile' => '-crf 26 -preset fast'
        ];
        
        return $presets[$quality_preset] ?? $presets['broadcast'];
    }
    
    private function finalizeExport($output_path) {
        // Nettoyage des fichiers temporaires
        $temp_dir = '/tmp/clipmaker_' . $this->export_id . '/';
        if (is_dir($temp_dir)) {
            shell_exec("rm -rf " . escapeshellarg($temp_dir));
        }
        
        // Génération de métadonnées
        $metadata = $this->analyzeOutput($output_path);
        
        $stmt = $this->db->prepare("
            UPDATE workflow_exports 
            SET duration_seconds = ?, completed_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $metadata['duration'] ?? 0,
            $this->export_id
        ]);
    }
    
    private function analyzeOutput($file_path) {
        $cmd = "ffprobe -v quiet -print_format json -show_format " . escapeshellarg($file_path);
        $output = shell_exec($cmd);
        
        if ($output) {
            $data = json_decode($output, true);
            return [
                'duration' => (float)($data['format']['duration'] ?? 0),
                'size' => (int)($data['format']['size'] ?? 0),
                'bitrate' => (int)($data['format']['bit_rate'] ?? 0)
            ];
        }
        
        return [];
    }
    
    private function updateStatus($status, $progress = null, $error = null) {
        $sql = "UPDATE workflow_exports SET status = ?";
        $params = [$status];
        
        if ($progress !== null) {
            $sql .= ", progress = ?";
            $params[] = $progress;
        }
        
        if ($error) {
            $sql .= ", error_message = ?";
            $params[] = $error;
        }
        
        $sql .= " WHERE id = ?";
        $params[] = $this->export_id;
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
    }
    
    private function updateProgress($progress) {
        $this->updateStatus($this->export_data['status'], $progress);
        $this->log("Progression: $progress%");
    }
    
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[$timestamp] Export {$this->export_id}: $message\n";
        
        // Log dans fichier
        $log_file = '/tmp/clipmaker_render.log';
        file_put_contents($log_file, $log_message, FILE_APPEND | LOCK_EX);
        
        // Log en DB aussi
        if ($this->db) {
            $stmt = $this->db->prepare("
                UPDATE workflow_exports 
                SET processing_log = CONCAT(IFNULL(processing_log, ''), ?) 
                WHERE id = ?
            ");
            $stmt->execute([$log_message, $this->export_id]);
        }
        
        echo $log_message;
    }
}

// Exécution du rendu
try {
    $renderer = new VideoRenderer($config, $export_id);
    $renderer->render();
    exit(0);
} catch (Exception $e) {
    echo "Erreur fatale: " . $e->getMessage() . "\n";
    exit(1);
}
?>