<?php
/**
 * ================================================================
 * 🎨 CLIPMAKER TEMPLATES - MODULE MODÈLES & TEMPLATES
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Bibliothèque templates prêts à l'emploi
 * - Catégories (intro, outro, transition, promo, etc.)
 * - Preview et utilisation directe
 * - Création templates personnalisés
 * - Partage communauté
 * - Import/Export templates
 * 
 * 🔥 BY PAPY 77 - CREATIVE POWER !
 * ================================================================
 */

require_once '../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

$action = $_GET['action'] ?? 'library';
$category = $_GET['category'] ?? 'all';
$template_id = $_GET['template'] ?? null;
$search = $_GET['search'] ?? '';

// Catégories disponibles
$categories = [
    'all' => ['name' => 'Tous', 'icon' => '📚', 'description' => 'Tous les templates'],
    'intro' => ['name' => 'Intros', 'icon' => '🎬', 'description' => 'Introductions accrocheuses'],
    'outro' => ['name' => 'Outros', 'icon' => '🎭', 'description' => 'Conclusions marquantes'],
    'transition' => ['name' => 'Transitions', 'icon' => '🔄', 'description' => 'Effets de transition'],
    'promo' => ['name' => 'Promo', 'icon' => '📢', 'description' => 'Promotions et publicités'],
    'social' => ['name' => 'Réseaux Sociaux', 'icon' => '📱', 'description' => 'Optimisés réseaux sociaux'],
    'corporate' => ['name' => 'Corporate', 'icon' => '🏢', 'description' => 'Présentation d\'entreprise'],
    'tutorial' => ['name' => 'Tutoriels', 'icon' => '🎓', 'description' => 'Contenus éducatifs'],
    'event' => ['name' => 'Événements', 'icon' => '🎉', 'description' => 'Célébrations et événements']
];

// Templates par défaut (seeded)
$default_templates = [
    [
        'name' => 'Intro Logo Animé',
        'description' => 'Animation logo avec texte personnalisable',
        'category' => 'intro',
        'duration' => 5.0,
        'resolution_width' => 1920,
        'resolution_height' => 1080,
        'is_public' => 1,
        'rating' => 4.8,
        'usage_count' => 156,
        'thumbnail_url' => '/uploads/templates/intro_logo_thumb.jpg',
        'template_data' => json_encode([
            'scenes' => [
                [
                    'type' => 'logo_animation',
                    'duration' => 5.0,
                    'elements' => [
                        ['type' => 'logo', 'position' => 'center', 'animation' => 'zoom_in'],
                        ['type' => 'text', 'content' => 'Votre Entreprise', 'position' => 'bottom']
                    ]
                ]
            ]
        ])
    ],
    [
        'name' => 'Transition Glitch',
        'description' => 'Effet glitch moderne entre séquences',
        'category' => 'transition',
        'duration' => 1.5,
        'resolution_width' => 1920,
        'resolution_height' => 1080,
        'is_public' => 1,
        'rating' => 4.6,
        'usage_count' => 89,
        'thumbnail_url' => '/uploads/templates/glitch_thumb.jpg',
        'template_data' => json_encode([
            'effects' => [
                ['type' => 'glitch', 'intensity' => 0.7, 'duration' => 1.5]
            ]
        ])
    ],
    [
        'name' => 'Promo Instagram Stories',
        'description' => 'Template vertical pour Instagram Stories',
        'category' => 'social',
        'duration' => 15.0,
        'resolution_width' => 1080,
        'resolution_height' => 1920,
        'is_public' => 1,
        'rating' => 4.9,
        'usage_count' => 234,
        'thumbnail_url' => '/uploads/templates/insta_stories_thumb.jpg',
        'template_data' => json_encode([
            'format' => 'vertical_9_16',
            'scenes' => [
                [
                    'type' => 'cover',
                    'duration' => 3.0,
                    'elements' => [
                        ['type' => 'background_video', 'source' => 'stock'],
                        ['type' => 'title', 'content' => 'VOTRE TITRE']
                    ]
                ]
            ]
        ])
    ]
];

// Traitement création template
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'create_template') {
    header('Content-Type: application/json');
    
    try {
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $category = $_POST['category'];
        $is_public = isset($_POST['is_public']) ? 1 : 0;
        $project_id = $_POST['project_id'] ?? null;
        
        if (empty($name) || empty($category)) {
            throw new Exception("Nom et catégorie obligatoires");
        }
        
        // Récupération données projet source
        $template_data = '{}';
        $duration = 0;
        $resolution_width = 1920;
        $resolution_height = 1080;
        
        if ($project_id) {
            $stmt = $pdo->prepare("SELECT * FROM clipmaker_projects WHERE id = ? AND user_id = ?");
            $stmt->execute([$project_id, $user['id']]);
            $project = $stmt->fetch();
            
            if ($project) {
                $template_data = $project['timeline_data'];
                $duration = $project['duration'] ?: 0;
                $resolution_width = $project['resolution_width'];
                $resolution_height = $project['resolution_height'];
            }
        }
        
        // Insertion template
        $stmt = $pdo->prepare("
            INSERT INTO clipmaker_templates 
            (user_id, name, description, category, template_data, duration, 
             resolution_width, resolution_height, is_public, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $user['id'],
            $name,
            $description,
            $category,
            $template_data,
            $duration,
            $resolution_width,
            $resolution_height,
            $is_public
        ]);
        
        $template_id = $pdo->lastInsertId();
        
        echo json_encode([
            'success' => true,
            'message' => 'Template créé avec succès',
            'template_id' => $template_id
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Utilisation template (création projet)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'use_template') {
    header('Content-Type: application/json');
    
    try {
        $template_id = $_POST['template_id'];
        $project_name = trim($_POST['project_name']);
        
        if (!$template_id || !$project_name) {
            throw new Exception("Template et nom projet obligatoires");
        }
        
        // Récupération template
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_templates WHERE id = ?");
        $stmt->execute([$template_id]);
        $template = $stmt->fetch();
        
        if (!$template) {
            throw new Exception("Template introuvable");
        }
        
        // Création nouveau projet basé sur template
        $stmt = $pdo->prepare("
            INSERT INTO clipmaker_projects 
            (user_id, name, description, timeline_data, duration, 
             resolution_width, resolution_height, fps, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 30, 'draft', NOW())
        ");
        
        $stmt->execute([
            $user['id'],
            $project_name,
            'Projet créé depuis template: ' . $template['name'],
            $template['template_data'],
            $template['duration'],
            $template['resolution_width'],
            $template['resolution_height']
        ]);
        
        $new_project_id = $pdo->lastInsertId();
        
        // Mise à jour compteur usage
        $stmt = $pdo->prepare("UPDATE clipmaker_templates SET usage_count = usage_count + 1 WHERE id = ?");
        $stmt->execute([$template_id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Projet créé depuis template',
            'project_id' => $new_project_id
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Rating template
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'rate_template') {
    header('Content-Type: application/json');
    
    try {
        $template_id = $_POST['template_id'];
        $rating = max(1, min(5, intval($_POST['rating'])));
        
        // Vérifier si déjà noté par cet utilisateur
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count 
            FROM template_ratings 
            WHERE template_id = ? AND user_id = ?
        ");
        $stmt->execute([$template_id, $user['id']]);
        $existing = $stmt->fetch()['count'];
        
        if ($existing) {
            // Mise à jour note existante
            $stmt = $pdo->prepare("
                UPDATE template_ratings 
                SET rating = ?, created_at = NOW() 
                WHERE template_id = ? AND user_id = ?
            ");
            $stmt->execute([$rating, $template_id, $user['id']]);
        } else {
            // Nouvelle note
            $stmt = $pdo->prepare("
                INSERT INTO template_ratings (template_id, user_id, rating, created_at) 
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([$template_id, $user['id'], $rating]);
        }
        
        // Recalcul moyenne
        $stmt = $pdo->prepare("
            SELECT AVG(rating) as avg_rating, COUNT(*) as count 
            FROM template_ratings 
            WHERE template_id = ?
        ");
        $stmt->execute([$template_id]);
        $stats = $stmt->fetch();
        
        $new_rating = round($stats['avg_rating'], 1);
        $rating_count = $stats['count'];
        
        // Mise à jour template
        $stmt = $pdo->prepare("
            UPDATE clipmaker_templates 
            SET rating = ?, rating_count = ? 
            WHERE id = ?
        ");
        $stmt->execute([$new_rating, $rating_count, $template_id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Note enregistrée',
            'new_rating' => $new_rating,
            'rating_count' => $rating_count
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Récupération templates avec filtres
try {
    $where_conditions = ["1=1"];
    $params = [];
    
    // Filtre catégorie
    if ($category !== 'all') {
        $where_conditions[] = "category = ?";
        $params[] = $category;
    }
    
    // Filtre recherche
    if (!empty($search)) {
        $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }
    
    // Visibilité (publics + mes templates privés)
    $where_conditions[] = "(is_public = 1 OR user_id = ?)";
    $params[] = $user['id'];
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT t.*, u.name as author_name,
               CASE WHEN t.user_id = ? THEN 1 ELSE 0 END as is_mine
        FROM clipmaker_templates t
        LEFT JOIN users u ON t.user_id = u.id
        WHERE $where_clause
        ORDER BY t.usage_count DESC, t.rating DESC, t.created_at DESC
        LIMIT 50
    ");
    
    $stmt->execute(array_merge([$user['id']], $params));
    $templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Mes projets pour création template
    $stmt = $pdo->prepare("SELECT id, name FROM clipmaker_projects WHERE user_id = ? ORDER BY name");
    $stmt->execute([$user['id']]);
    $my_projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Stats templates
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_templates,
            SUM(CASE WHEN user_id = ? THEN 1 ELSE 0 END) as my_templates,
            AVG(rating) as avg_rating
        FROM clipmaker_templates 
        WHERE is_public = 1 OR user_id = ?
    ");
    $stmt->execute([$user['id'], $user['id']]);
    $template_stats = $stmt->fetch();
    
} catch (PDOException $e) {
    error_log("Erreur récupération templates: " . $e->getMessage());
    $templates = [];
    $my_projects = [];
    $template_stats = ['total_templates' => 0, 'my_templates' => 0, 'avg_rating' => 0];
}

// Seeding templates par défaut si table vide
try {
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM clipmaker_templates");
    $template_count = $stmt->fetch()['count'];
    
    if ($template_count == 0) {
        foreach ($default_templates as $template) {
            $stmt = $pdo->prepare("
                INSERT INTO clipmaker_templates 
                (user_id, name, description, category, template_data, duration, 
                 resolution_width, resolution_height, thumbnail_url, is_public, 
                 rating, rating_count, usage_count, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $user['id'], // Admin user
                $template['name'],
                $template['description'],
                $template['category'],
                $template['template_data'],
                $template['duration'],
                $template['resolution_width'],
                $template['resolution_height'],
                $template['thumbnail_url'],
                $template['is_public'],
                $template['rating'],
                25, // Rating count simulé
                $template['usage_count']
            ]);
        }
        
        // Recharger templates après seeding
        header("Location: " . $_SERVER['REQUEST_URI']);
        exit;
    }
} catch (PDOException $e) {
    error_log("Erreur seeding templates: " . $e->getMessage());
}

?>
<div class="templates-module">
    <!-- ================================================================
     * 🎨 HEADER MODULE TEMPLATES
     * ================================================================ -->
    <div class="module-header">
        <div class="header-content">
            <h2>🎨 Bibliothèque Templates</h2>
            <p>Modèles prêts à l'emploi pour création rapide</p>
        </div>
        <div class="header-actions">
            <button class="btn btn-secondary" onclick="showCreateModal()">
                <span>➕</span>
                <span>Créer Template</span>
            </button>
            <button class="btn btn-primary" onclick="showImportModal()">
                <span>📥</span>
                <span>Importer</span>
            </button>
        </div>
    </div>

    <!-- ================================================================
     * 📊 STATISTIQUES TEMPLATES
     * ================================================================ -->
    <div class="templates-stats">
        <div class="stat-card">
            <div class="stat-icon">📚</div>
            <div class="stat-number"><?php echo $template_stats['total_templates']; ?></div>
            <div class="stat-label">Templates disponibles</div>
        </div>
        
        <div class="stat-card mine">
            <div class="stat-icon">👤</div>
            <div class="stat-number"><?php echo $template_stats['my_templates']; ?></div>
            <div class="stat-label">Mes templates</div>
        </div>
        
        <div class="stat-card rating">
            <div class="stat-icon">⭐</div>
            <div class="stat-number"><?php echo round($template_stats['avg_rating'], 1); ?></div>
            <div class="stat-label">Note moyenne</div>
        </div>
        
        <div class="stat-card popular">
            <div class="stat-icon">🔥</div>
            <div class="stat-number"><?php echo array_sum(array_column($templates, 'usage_count')); ?></div>
            <div class="stat-label">Utilisations totales</div>
        </div>
    </div>

    <!-- ================================================================
     * 🗂️ NAVIGATION CATÉGORIES
     * ================================================================ -->
    <div class="categories-nav">
        <?php foreach ($categories as $cat_key => $cat_info): ?>
            <button class="category-btn <?php echo $category === $cat_key ? 'active' : ''; ?>" 
                    onclick="filterByCategory('<?php echo $cat_key; ?>')"
                    title="<?php echo $cat_info['description']; ?>">
                <span class="category-icon"><?php echo $cat_info['icon']; ?></span>
                <span class="category-name"><?php echo $cat_info['name']; ?></span>
                <span class="category-count">
                    <?php 
                    $count = $cat_key === 'all' ? count($templates) : 
                             count(array_filter($templates, fn($t) => $t['category'] === $cat_key));
                    echo $count;
                    ?>
                </span>
            </button>
        <?php endforeach; ?>
    </div>

    <!-- ================================================================
     * 🔍 RECHERCHE ET FILTRES
     * ================================================================ -->
    <div class="search-filters">
        <div class="search-box">
            <input type="text" id="searchInput" placeholder="🔍 Rechercher un template..." 
                   value="<?php echo htmlspecialchars($search); ?>" onkeyup="debounceSearch()">
        </div>
        
        <div class="filter-options">
            <select id="sortSelect" onchange="applySorting()">
                <option value="popular">Plus populaires</option>
                <option value="rating">Mieux notés</option>
                <option value="recent">Plus récents</option>
                <option value="name">Nom A-Z</option>
            </select>
            
            <div class="view-toggle">
                <button class="btn-view active" data-view="grid" onclick="switchView('grid')">⊞</button>
                <button class="btn-view" data-view="list" onclick="switchView('list')">☰</button>
            </div>
        </div>
    </div>

    <!-- ================================================================
     * 🎨 GRILLE DES TEMPLATES
     * ================================================================ -->
    <div class="templates-container">
        <?php if (empty($templates)): ?>
            <div class="empty-state">
                <div class="empty-icon">🎨</div>
                <h3>Aucun template trouvé</h3>
                <p>Créez votre premier template ou ajustez vos filtres</p>
                <button class="btn btn-primary" onclick="showCreateModal()">
                    ➕ Créer un template
                </button>
            </div>
        <?php else: ?>
            <div class="templates-grid" id="templatesGrid">
                <?php foreach ($templates as $template): ?>
                    <div class="template-card" data-category="<?php echo $template['category']; ?>">
                        <div class="template-thumbnail">
                            <?php if ($template['thumbnail_url']): ?>
                                <img src="<?php echo htmlspecialchars($template['thumbnail_url']); ?>" 
                                     alt="<?php echo htmlspecialchars($template['name']); ?>">
                            <?php else: ?>
                                <div class="placeholder-thumbnail">
                                    <?php echo $categories[$template['category']]['icon']; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="template-overlay">
                                <button class="btn-overlay primary" onclick="previewTemplate(<?php echo $template['id']; ?>)" title="Aperçu">
                                    👁️
                                </button>
                                <button class="btn-overlay success" onclick="useTemplate(<?php echo $template['id']; ?>)" title="Utiliser">
                                    🚀
                                </button>
                                <?php if ($template['is_mine']): ?>
                                    <button class="btn-overlay secondary" onclick="editTemplate(<?php echo $template['id']; ?>)" title="Modifier">
                                        ✏️
                                    </button>
                                <?php endif; ?>
                            </div>
                            
                            <div class="template-badges">
                                <?php if ($template['is_mine']): ?>
                                    <span class="badge mine">Mien</span>
                                <?php endif; ?>
                                <?php if ($template['is_public']): ?>
                                    <span class="badge public">Public</span>
                                <?php else: ?>
                                    <span class="badge private">Privé</span>
                                <?php endif; ?>
                            </div>
                            
                            <div class="template-duration">
                                <?php echo $template['duration'] ? gmdate("i:s", $template['duration']) : 'N/A'; ?>
                            </div>
                        </div>
                        
                        <div class="template-info">
                            <div class="template-header">
                                <h4 class="template-name"><?php echo htmlspecialchars($template['name']); ?></h4>
                                <div class="template-category">
                                    <?php echo $categories[$template['category']]['name']; ?>
                                </div>
                            </div>
                            
                            <p class="template-description">
                                <?php echo htmlspecialchars(substr($template['description'], 0, 80) . 
                                    (strlen($template['description']) > 80 ? '...' : '')); ?>
                            </p>
                            
                            <div class="template-meta">
                                <div class="meta-row">
                                    <div class="rating">
                                        <?php 
                                        $rating = $template['rating'] ?: 0;
                                        for ($i = 1; $i <= 5; $i++): 
                                        ?>
                                            <span class="star <?php echo $i <= $rating ? 'filled' : ''; ?>">⭐</span>
                                        <?php endfor; ?>
                                        <span class="rating-value"><?php echo $rating; ?></span>
                                    </div>
                                    
                                    <div class="usage-count">
                                        <span>👥 <?php echo $template['usage_count']; ?></span>
                                    </div>
                                </div>
                                
                                <div class="meta-row">
                                    <div class="resolution">
                                        <?php echo $template['resolution_width']; ?>x<?php echo $template['resolution_height']; ?>
                                    </div>
                                    
                                    <div class="author">
                                        Par <?php echo htmlspecialchars($template['author_name']); ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="template-actions">
                                <button class="btn btn-sm btn-primary" onclick="useTemplate(<?php echo $template['id']; ?>)">
                                    🚀 Utiliser
                                </button>
                                <button class="btn btn-sm btn-secondary" onclick="previewTemplate(<?php echo $template['id']; ?>)">
                                    👁️ Aperçu
                                </button>
                                <?php if (!$template['is_mine']): ?>
                                    <button class="btn btn-sm btn-outline" onclick="showRatingModal(<?php echo $template['id']; ?>)">
                                        ⭐ Noter
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- ================================================================
 * 🆕 MODAL CRÉATION TEMPLATE
 * ================================================================ -->
<div id="createTemplateModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>🆕 Créer un Template</h3>
            <button class="btn-close" onclick="closeCreateModal()">&times;</button>
        </div>
        
        <form id="createTemplateForm" onsubmit="createTemplate(event)">
            <div class="modal-body">
                <div class="form-group">
                    <label for="templateName">Nom du template *</label>
                    <input type="text" id="templateName" name="name" required 
                           placeholder="Mon super template">
                </div>
                
                <div class="form-group">
                    <label for="templateDescription">Description</label>
                    <textarea id="templateDescription" name="description" rows="3"
                              placeholder="Description du template..."></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="templateCategory">Catégorie *</label>
                        <select id="templateCategory" name="category" required>
                            <?php foreach ($categories as $cat_key => $cat_info): ?>
                                <?php if ($cat_key !== 'all'): ?>
                                    <option value="<?php echo $cat_key; ?>">
                                        <?php echo $cat_info['icon']; ?> <?php echo $cat_info['name']; ?>
                                    </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="sourceProject">Projet source</label>
                        <select id="sourceProject" name="project_id">
                            <option value="">Template vide</option>
                            <?php foreach ($my_projects as $project): ?>
                                <option value="<?php echo $project['id']; ?>">
                                    <?php echo htmlspecialchars($project['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" id="isPublic" name="is_public">
                        <label for="isPublic">Rendre public (partager avec la communauté)</label>
                    </div>
                </div>
                
                <div class="template-preview" id="templatePreview" style="display: none;">
                    <h4>Aperçu du projet source</h4>
                    <div id="previewContent"></div>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeCreateModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    🎨 Créer Template
                </button>
            </div>
        </form>
    </div>
</div>

<!-- ================================================================
 * 🚀 MODAL UTILISATION TEMPLATE
 * ================================================================ -->
<div id="useTemplateModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="useTemplateTitle">🚀 Utiliser Template</h3>
            <button class="btn-close" onclick="closeUseModal()">&times;</button>
        </div>
        
        <form id="useTemplateForm" onsubmit="confirmUseTemplate(event)">
            <input type="hidden" id="useTemplateId" name="template_id">
            
            <div class="modal-body">
                <div id="templateDetails" class="template-details"></div>
                
                <div class="form-group">
                    <label for="projectName">Nom du nouveau projet *</label>
                    <input type="text" id="projectName" name="project_name" required 
                           placeholder="Mon nouveau projet">
                </div>
                
                <div class="usage-info">
                    <div class="info-icon">ℹ️</div>
                    <div class="info-text">
                        Un nouveau projet sera créé avec la structure de ce template.
                        Vous pourrez ensuite le personnaliser dans l'éditeur.
                    </div>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeUseModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    🚀 Créer Projet
                </button>
            </div>
        </form>
    </div>
</div>

<!-- ================================================================
 * ⭐ MODAL NOTATION TEMPLATE
 * ================================================================ -->
<div id="ratingModal" class="modal">
    <div class="modal-content modal-small">
        <div class="modal-header">
            <h3>⭐ Noter ce Template</h3>
            <button class="btn-close" onclick="closeRatingModal()">&times;</button>
        </div>
        
        <form id="ratingForm" onsubmit="submitRating(event)">
            <input type="hidden" id="ratingTemplateId" name="template_id">
            
            <div class="modal-body">
                <div class="rating-section">
                    <p>Quelle note donnez-vous à ce template ?</p>
                    
                    <div class="star-rating">
                        <?php for ($i = 1; $i <= 5; $i++): ?>
                            <span class="star-input" data-rating="<?php echo $i; ?>" onclick="selectRating(<?php echo $i; ?>)">⭐</span>
                        <?php endfor; ?>
                    </div>
                    
                    <input type="hidden" id="selectedRating" name="rating" value="5">
                    
                    <div class="rating-labels">
                        <span>Décevant</span>
                        <span>Excellent</span>
                    </div>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeRatingModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    ⭐ Noter
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* ================================================================
 * 🎨 STYLES MODULE TEMPLATES
 * ================================================================ */

.templates-module {
    padding: 0;
}

.module-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 2px solid rgba(255, 107, 107, 0.1);
}

.header-content h2 {
    color: #333;
    margin-bottom: 5px;
    font-size: 24px;
}

.header-content p {
    color: #666;
    font-size: 14px;
}

.header-actions {
    display: flex;
    gap: 10px;
}

.header-actions .btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-weight: 600;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 107, 0.3);
}

.btn-secondary {
    background: #f8f9fa;
    color: #666;
    border: 1px solid #ddd;
}

.btn-secondary:hover {
    background: #e9ecef;
}

.btn-sm {
    padding: 6px 12px;
    font-size: 12px;
}

.btn-outline {
    background: transparent;
    border: 1px solid #ddd;
    color: #666;
}

.templates-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 16px;
    padding: 20px;
    text-align: center;
    transition: all 0.3s ease;
    border: 2px solid transparent;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 40px rgba(0, 0, 0, 0.15);
    border-color: rgba(255, 107, 107, 0.2);
}

.stat-card .stat-icon {
    font-size: 32px;
    margin-bottom: 10px;
}

.stat-card .stat-number {
    font-size: 24px;
    font-weight: 700;
    color: #ff6b6b;
    margin-bottom: 5px;
}

.stat-card .stat-label {
    color: #666;
    font-size: 12px;
    font-weight: 500;
}

.stat-card.mine .stat-number { color: #4ecdc4; }
.stat-card.rating .stat-number { color: #ffd700; }
.stat-card.popular .stat-number { color: #ff6b6b; }

.categories-nav {
    display: flex;
    gap: 10px;
    margin-bottom: 30px;
    background: rgba(255, 255, 255, 0.9);
    padding: 20px;
    border-radius: 12px;
    overflow-x: auto;
    scrollbar-width: thin;
}

.category-btn {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 5px;
    padding: 15px 20px;
    background: transparent;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 10px;
    cursor: pointer;
    transition: all 0.3s ease;
    white-space: nowrap;
    min-width: 100px;
}

.category-btn:hover {
    background: rgba(255, 107, 107, 0.1);
    border-color: rgba(255, 107, 107, 0.4);
    transform: translateY(-2px);
}

.category-btn.active {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
    border-color: #ff6b6b;
}

.category-icon {
    font-size: 24px;
}

.category-name {
    font-size: 12px;
    font-weight: 600;
}

.category-count {
    font-size: 10px;
    background: rgba(255, 255, 255, 0.3);
    padding: 2px 6px;
    border-radius: 10px;
    font-weight: 600;
}

.category-btn.active .category-count {
    background: rgba(255, 255, 255, 0.3);
}

.search-filters {
    display: flex;
    gap: 20px;
    align-items: center;
    margin-bottom: 30px;
    background: rgba(255, 255, 255, 0.9);
    padding: 20px;
    border-radius: 12px;
}

.search-box {
    flex: 1;
}

.search-box input {
    width: 100%;
    padding: 12px 16px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.search-box input:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.filter-options {
    display: flex;
    gap: 15px;
    align-items: center;
}

.filter-options select {
    padding: 10px 15px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    background: white;
    cursor: pointer;
}

.view-toggle {
    display: flex;
    gap: 5px;
}

.btn-view {
    background: #f8f9fa;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 6px;
    padding: 8px 12px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-view.active {
    background: #ff6b6b;
    color: white;
}

.templates-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
    gap: 25px;
}

.template-card {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    overflow: hidden;
    transition: all 0.3s ease;
    border: 2px solid transparent;
    cursor: pointer;
}

.template-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
    border-color: rgba(255, 107, 107, 0.3);
}

.template-thumbnail {
    position: relative;
    height: 200px;
    background: linear-gradient(135deg, rgba(255, 107, 107, 0.1), rgba(238, 90, 36, 0.1));
    overflow: hidden;
}

.template-thumbnail img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.placeholder-thumbnail {
    display: flex;
    align-items: center;
    justify-content: center;
    height: 100%;
    font-size: 64px;
    color: rgba(255, 107, 107, 0.5);
}

.template-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 15px;
    opacity: 0;
    transition: all 0.3s ease;
}

.template-card:hover .template-overlay {
    opacity: 1;
}

.btn-overlay {
    background: rgba(255, 255, 255, 0.9);
    border: none;
    border-radius: 50%;
    width: 50px;
    height: 50px;
    cursor: pointer;
    font-size: 18px;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-overlay:hover {
    background: white;
    transform: scale(1.1);
}

.btn-overlay.primary:hover {
    background: #ff6b6b;
    color: white;
}

.btn-overlay.success:hover {
    background: #28a745;
    color: white;
}

.template-badges {
    position: absolute;
    top: 10px;
    left: 10px;
    display: flex;
    gap: 5px;
}

.badge {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.badge.mine {
    background: #4ecdc4;
    color: white;
}

.badge.public {
    background: #28a745;
    color: white;
}

.badge.private {
    background: #6c757d;
    color: white;
}

.template-duration {
    position: absolute;
    bottom: 10px;
    right: 10px;
    background: rgba(0, 0, 0, 0.8);
    color: white;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 11px;
    font-weight: 600;
}

.template-info {
    padding: 20px;
}

.template-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 10px;
}

.template-name {
    font-size: 16px;
    font-weight: 600;
    color: #333;
    margin: 0;
    flex: 1;
}

.template-category {
    font-size: 11px;
    background: rgba(255, 107, 107, 0.1);
    color: #ff6b6b;
    padding: 4px 8px;
    border-radius: 12px;
    font-weight: 600;
}

.template-description {
    color: #666;
    font-size: 13px;
    line-height: 1.4;
    margin-bottom: 15px;
}

.template-meta {
    margin-bottom: 15px;
}

.meta-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
    font-size: 12px;
}

.rating {
    display: flex;
    align-items: center;
    gap: 2px;
}

.star {
    font-size: 14px;
    opacity: 0.3;
}

.star.filled {
    opacity: 1;
}

.rating-value {
    margin-left: 5px;
    color: #666;
    font-weight: 600;
}

.usage-count {
    color: #666;
}

.resolution {
    color: #999;
    font-size: 11px;
}

.author {
    color: #999;
    font-size: 11px;
}

.template-actions {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
}

.empty-state {
    text-align: center;
    padding: 80px 20px;
    color: #666;
}

.empty-icon {
    font-size: 80px;
    margin-bottom: 20px;
    opacity: 0.5;
}

.empty-state h3 {
    margin-bottom: 10px;
    color: #333;
}

.empty-state p {
    margin-bottom: 30px;
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    z-index: 1000;
    backdrop-filter: blur(5px);
}

.modal-content {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    border-radius: 16px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-small {
    max-width: 400px;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    border-bottom: 1px solid #eee;
}

.modal-header h3 {
    margin: 0;
    color: #333;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    padding: 5px;
}

.modal-body {
    padding: 30px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
    font-size: 14px;
}

.form-group input, .form-group textarea, .form-group select {
    width: 100%;
    padding: 10px 12px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-group input:focus, .form-group textarea:focus, .form-group select:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
}

.checkbox-group {
    display: flex;
    align-items: center;
    gap: 10px;
}

.checkbox-group input[type="checkbox"] {
    width: auto;
}

.template-details {
    background: rgba(255, 107, 107, 0.05);
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.usage-info {
    display: flex;
    gap: 15px;
    background: rgba(54, 162, 235, 0.1);
    border: 1px solid rgba(54, 162, 235, 0.3);
    border-radius: 8px;
    padding: 15px;
    margin-top: 20px;
}

.info-icon {
    font-size: 24px;
    color: #007bff;
}

.info-text {
    color: #004085;
    font-size: 14px;
}

.star-rating {
    display: flex;
    justify-content: center;
    gap: 10px;
    margin: 20px 0;
}

.star-input {
    font-size: 32px;
    cursor: pointer;
    opacity: 0.3;
    transition: all 0.3s ease;
}

.star-input:hover,
.star-input.selected {
    opacity: 1;
    transform: scale(1.1);
}

.rating-labels {
    display: flex;
    justify-content: space-between;
    font-size: 12px;
    color: #666;
    margin-top: 10px;
}

.rating-section {
    text-align: center;
}

.modal-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
    padding: 20px 30px;
    border-top: 1px solid #eee;
}

/* Responsive */
@media (max-width: 768px) {
    .templates-stats {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .templates-grid {
        grid-template-columns: 1fr;
    }
    
    .categories-nav {
        flex-direction: column;
        gap: 10px;
    }
    
    .search-filters {
        flex-direction: column;
        gap: 15px;
    }
    
    .filter-options {
        width: 100%;
        justify-content: space-between;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .template-actions {
        justify-content: center;
    }
}
</style>

<script>
/**
 * ================================================================
 * 🎯 JAVASCRIPT MODULE TEMPLATES
 * ================================================================ */

let currentCategory = '<?php echo $category; ?>';
let searchTimeout;
let selectedTemplateId = null;

// Navigation catégories
function filterByCategory(category) {
    currentCategory = category;
    
    // Mise à jour boutons
    document.querySelectorAll('.category-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Mise à jour URL et rechargement
    const url = new URL(window.location);
    url.searchParams.set('category', category);
    window.location.href = url.toString();
}

// Recherche avec debounce
function debounceSearch() {
    clearTimeout(searchTimeout);
    searchTimeout = setTimeout(() => {
        const search = document.getElementById('searchInput').value;
        const url = new URL(window.location);
        url.searchParams.set('search', search);
        if (search === '') {
            url.searchParams.delete('search');
        }
        window.location.href = url.toString();
    }, 500);
}

// Tri
function applySorting() {
    const sort = document.getElementById('sortSelect').value;
    // TODO: Implémenter tri côté client ou serveur
    console.log('Tri par:', sort);
}

// Vue grille/liste
function switchView(view) {
    document.querySelectorAll('.btn-view').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-view="${view}"]`).classList.add('active');
    
    const container = document.getElementById('templatesGrid');
    if (view === 'list') {
        container.style.gridTemplateColumns = '1fr';
    } else {
        container.style.gridTemplateColumns = 'repeat(auto-fill, minmax(320px, 1fr))';
    }
}

// Création template
function showCreateModal() {
    document.getElementById('createTemplateModal').style.display = 'block';
}

function closeCreateModal() {
    document.getElementById('createTemplateModal').style.display = 'none';
    document.getElementById('createTemplateForm').reset();
}

function createTemplate(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '⏳ Création...';
    submitBtn.disabled = true;
    
    fetch('?action=create_template', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('✅ ' + data.message, 'success');
            closeCreateModal();
            setTimeout(() => window.location.reload(), 1500);
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur création: ' + error.message, 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Aperçu template
function previewTemplate(templateId) {
    // TODO: Implémenter modal preview avec détails template
    console.log('Preview template:', templateId);
    showNotification('🔜 Fonctionnalité preview en développement', 'info');
}

// Utilisation template
function useTemplate(templateId) {
    selectedTemplateId = templateId;
    
    // Récupération infos template
    fetch(`../api/clipmaker.php?action=get_template&id=${templateId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const template = data.template;
                document.getElementById('useTemplateTitle').textContent = `🚀 Utiliser: ${template.name}`;
                document.getElementById('useTemplateId').value = templateId;
                document.getElementById('projectName').value = `Projet ${template.name}`;
                
                // Affichage détails template
                document.getElementById('templateDetails').innerHTML = `
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                        <div><strong>Catégorie:</strong> ${template.category}</div>
                        <div><strong>Durée:</strong> ${template.duration}s</div>
                        <div><strong>Résolution:</strong> ${template.resolution_width}x${template.resolution_height}</div>
                        <div><strong>Note:</strong> ⭐ ${template.rating}/5</div>
                    </div>
                    <div><strong>Description:</strong></div>
                    <div style="color: #666; font-size: 14px; margin-top: 5px;">${template.description}</div>
                `;
                
                document.getElementById('useTemplateModal').style.display = 'block';
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur chargement template: ' + error.message, 'error');
        });
}

function closeUseModal() {
    document.getElementById('useTemplateModal').style.display = 'none';
    selectedTemplateId = null;
}

function confirmUseTemplate(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '⏳ Création...';
    submitBtn.disabled = true;
    
    fetch('?action=use_template', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('✅ ' + data.message, 'success');
            closeUseModal();
            
            // Redirection vers éditeur
            setTimeout(() => {
                window.parent.location.href = `../clipmaker.php?module=timeline&project=${data.project_id}`;
            }, 1500);
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur utilisation: ' + error.message, 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Notation template
function showRatingModal(templateId) {
    selectedTemplateId = templateId;
    document.getElementById('ratingTemplateId').value = templateId;
    document.getElementById('ratingModal').style.display = 'block';
    
    // Reset rating
    selectRating(5);
}

function closeRatingModal() {
    document.getElementById('ratingModal').style.display = 'none';
    selectedTemplateId = null;
}

function selectRating(rating) {
    document.getElementById('selectedRating').value = rating;
    
    // Mise à jour visuelle étoiles
    document.querySelectorAll('.star-input').forEach((star, index) => {
        if (index < rating) {
            star.classList.add('selected');
        } else {
            star.classList.remove('selected');
        }
    });
}

function submitRating(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '⏳ Notation...';
    submitBtn.disabled = true;
    
    fetch('?action=rate_template', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('✅ ' + data.message, 'success');
            closeRatingModal();
            
            // Mise à jour note dans interface
            setTimeout(() => window.location.reload(), 1000);
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur notation: ' + error.message, 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Édition template (mes templates uniquement)
function editTemplate(templateId) {
    // TODO: Modal édition template
    console.log('Édition template:', templateId);
    showNotification('🔜 Fonctionnalité édition en développement', 'info');
}

// Import templates
function showImportModal() {
    showNotification('🔜 Fonctionnalité import en développement', 'info');
}

// Utilitaires
function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#28a745' : type === 'info' ? '#17a2b8' : '#dc3545'};
        color: white;
        padding: 15px 20px;
        border-radius: 8px;
        z-index: 10001;
        font-weight: 600;
        box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        animation: slideIn 0.3s ease;
        max-width: 400px;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 4000);
}

// Initialisation
document.addEventListener('DOMContentLoaded', function() {
    // Preview projet source dans création template
    const sourceProjectSelect = document.getElementById('sourceProject');
    if (sourceProjectSelect) {
        sourceProjectSelect.addEventListener('change', function() {
            const projectId = this.value;
            const previewDiv = document.getElementById('templatePreview');
            
            if (projectId) {
                // Récupération info projet pour preview
                fetch(`../api/clipmaker.php?action=get_project&id=${projectId}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            const project = data.project;
                            document.getElementById('previewContent').innerHTML = `
                                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                                    <div><strong>Nom:</strong> ${project.name}</div>
                                    <div><strong>Durée:</strong> ${project.duration || 0}s</div>
                                    <div><strong>Résolution:</strong> ${project.resolution_width}x${project.resolution_height}</div>
                                    <div><strong>Status:</strong> ${project.status}</div>
                                </div>
                            `;
                            previewDiv.style.display = 'block';
                        }
                    })
                    .catch(error => {
                        console.error('Erreur preview projet:', error);
                    });
            } else {
                previewDiv.style.display = 'none';
            }
        });
    }
    
    // Gestion raccourcis clavier
    document.addEventListener('keydown', function(e) {
        // Escape = Fermer modals
        if (e.key === 'Escape') {
            closeCreateModal();
            closeUseModal();
            closeRatingModal();
        }
        
        // Ctrl+N = Nouveau template
        if ((e.ctrlKey || e.metaKey) && e.key === 'n') {
            e.preventDefault();
            showCreateModal();
        }
        
        // Chiffres 1-9 = Navigation catégories
        if (e.key >= '1' && e.key <= '9' && !e.ctrlKey && !e.metaKey) {
            const categories = Object.keys(<?php echo json_encode($categories); ?>);
            const categoryIndex = parseInt(e.key) - 1;
            if (categories[categoryIndex]) {
                filterByCategory(categories[categoryIndex]);
            }
        }
    });
    
    // Animation d'entrée cards
    const cards = document.querySelectorAll('.template-card');
    cards.forEach((card, index) => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            card.style.transition = 'all 0.5s ease';
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
        }, index * 50);
    });
    
    // Lazy loading images
    const images = document.querySelectorAll('.template-thumbnail img');
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src || img.src;
                img.classList.remove('lazy');
                observer.unobserve(img);
            }
        });
    });
    
    images.forEach(img => imageObserver.observe(img));
    
    console.log('🎨 Module Templates Ready!');
    console.log('📁 Catégorie courante:', currentCategory);
    console.log('🎯 Templates chargés:', document.querySelectorAll('.template-card').length);
});

// Auto-refresh si nouvelle activité
setInterval(() => {
    if (!document.hidden) {
        // Vérification nouvelles templates communauté
        fetch('?action=check_new_templates&ajax=1')
            .then(response => response.json())
            .then(data => {
                if (data.new_count > 0) {
                    showNotification(`🆕 ${data.new_count} nouveau(x) template(s) disponible(s) !`, 'info');
                }
            })
            .catch(error => {
                console.log('Erreur vérification templates:', error);
            });
    }
}, 300000); // Toutes les 5 minutes

// Filtrage temps réel côté client
function filterTemplatesClientSide() {
    const searchTerm = document.getElementById('searchInput').value.toLowerCase();
    const cards = document.querySelectorAll('.template-card');
    
    cards.forEach(card => {
        const name = card.querySelector('.template-name').textContent.toLowerCase();
        const description = card.querySelector('.template-description').textContent.toLowerCase();
        const category = card.dataset.category;
        
        const matchesSearch = !searchTerm || name.includes(searchTerm) || description.includes(searchTerm);
        const matchesCategory = currentCategory === 'all' || category === currentCategory;
        
        if (matchesSearch && matchesCategory) {
            card.style.display = 'block';
            card.style.animation = 'fadeIn 0.3s ease';
        } else {
            card.style.display = 'none';
        }
    });
    
    // Mise à jour compteur catégorie
    updateCategoryCounts();
}

function updateCategoryCounts() {
    const visibleCards = document.querySelectorAll('.template-card[style*="block"], .template-card:not([style*="none"])');
    const categories = {};
    
    visibleCards.forEach(card => {
        const category = card.dataset.category;
        categories[category] = (categories[category] || 0) + 1;
    });
    
    document.querySelectorAll('.category-btn').forEach(btn => {
        const category = btn.onclick.toString().match(/'([^']+)'/)?.[1];
        if (category) {
            const count = category === 'all' ? visibleCards.length : (categories[category] || 0);
            const countSpan = btn.querySelector('.category-count');
            if (countSpan) {
                countSpan.textContent = count;
            }
        }
    });
}

// Gestion mode offline/online
window.addEventListener('online', () => {
    showNotification('🌐 Connexion rétablie', 'success');
});

window.addEventListener('offline', () => {
    showNotification('📴 Mode hors ligne - Fonctionnalités limitées', 'info');
});

// Styles animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    @keyframes fadeIn {
        from { opacity: 0; transform: scale(0.9); }
        to { opacity: 1; transform: scale(1); }
    }
    
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }
    
    .template-card:hover .template-thumbnail {
        animation: pulse 0.6s ease;
    }
    
    .btn-overlay:hover {
        animation: pulse 0.3s ease;
    }
    
    .star-input:hover {
        animation: pulse 0.3s ease;
    }
    
    .category-btn:focus {
        outline: 2px solid rgba(255, 107, 107, 0.5);
        outline-offset: 2px;
    }
    
    .template-card:focus-within {
        border-color: rgba(255, 107, 107, 0.5);
        box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.2);
    }
`;
document.head.appendChild(style);
</script>