<?php
/**
 * ================================================================
 * 🎥 CLIPMAKER EXPORT - MODULE RENDU FFMPEG
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Exports multi-formats (MP4, WebM, MOV, GIF)
 * - Presets qualité (High/Medium/Low)
 * - Progress tracking temps réel
 * - Queue de rendu asynchrone
 * - Preview avant export
 * - Historique exports
 * 
 * 🔥 BY PAPY 77 - FFMPEG MONSTER !
 * ================================================================
 */

require_once '../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

$action = $_GET['action'] ?? 'dashboard';
$project_id = $_GET['project'] ?? null;
$export_id = $_GET['export'] ?? null;

// Configuration FFmpeg
$ffmpeg_path = 'ffmpeg'; // Chemin vers FFmpeg sur le VPS
$output_path = '/social-media-generator/uploads/clipmaker/exports/';
$temp_path = '/tmp/clipmaker/';

// Presets qualité
$quality_presets = [
    'high' => [
        'name' => 'Haute Qualité',
        'description' => 'Meilleure qualité, fichier plus lourd',
        'video_codec' => 'libx264',
        'audio_codec' => 'aac',
        'crf' => 18,
        'preset' => 'medium',
        'audio_bitrate' => '320k'
    ],
    'medium' => [
        'name' => 'Qualité Standard',
        'description' => 'Bon compromis qualité/taille',
        'video_codec' => 'libx264',
        'audio_codec' => 'aac',
        'crf' => 23,
        'preset' => 'fast',
        'audio_bitrate' => '192k'
    ],
    'low' => [
        'name' => 'Qualité Web',
        'description' => 'Optimisé pour le web, fichier léger',
        'video_codec' => 'libx264',
        'audio_codec' => 'aac',
        'crf' => 28,
        'preset' => 'faster',
        'audio_bitrate' => '128k'
    ]
];

// Formats de sortie
$output_formats = [
    'mp4' => [
        'name' => 'MP4 (H.264)',
        'extension' => 'mp4',
        'container' => 'mp4',
        'description' => 'Format universel, compatible tous appareils'
    ],
    'webm' => [
        'name' => 'WebM (VP9)',
        'extension' => 'webm',
        'container' => 'webm',
        'description' => 'Format web optimisé, excellent ratio qualité/taille'
    ],
    'mov' => [
        'name' => 'MOV (ProRes)',
        'extension' => 'mov',
        'container' => 'mov',
        'description' => 'Format professionnel, haute qualité'
    ],
    'gif' => [
        'name' => 'GIF Animé',
        'extension' => 'gif',
        'container' => 'gif',
        'description' => 'Animation courte pour réseaux sociaux'
    ]
];

// Traitement lancement export
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'start_export') {
    header('Content-Type: application/json');
    
    try {
        $project_id = $_POST['project_id'] ?? null;
        $export_name = trim($_POST['export_name']);
        $format = $_POST['format'];
        $quality = $_POST['quality'];
        $resolution = $_POST['resolution'] ?? null;
        $fps = $_POST['fps'] ?? null;
        $start_time = $_POST['start_time'] ?? 0;
        $end_time = $_POST['end_time'] ?? null;
        
        if (!$project_id || !$export_name || !$format || !$quality) {
            throw new Exception("Paramètres manquants pour l'export");
        }
        
        // Récupération projet
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_projects WHERE id = ? AND user_id = ?");
        $stmt->execute([$project_id, $user['id']]);
        $project = $stmt->fetch();
        
        if (!$project) {
            throw new Exception("Projet introuvable");
        }
        
        // Validation format/qualité
        if (!isset($output_formats[$format]) || !isset($quality_presets[$quality])) {
            throw new Exception("Format ou qualité invalide");
        }
        
        // Préparation paramètres export
        $export_settings = [
            'format' => $format,
            'quality' => $quality,
            'resolution' => $resolution ?: $project['resolution_width'] . 'x' . $project['resolution_height'],
            'fps' => $fps ?: $project['fps'],
            'start_time' => $start_time,
            'end_time' => $end_time,
            'preset_config' => $quality_presets[$quality],
            'format_config' => $output_formats[$format]
        ];
        
        // Insertion export en base
        $stmt = $pdo->prepare("
            INSERT INTO clipmaker_exports 
            (project_id, user_id, export_name, format, quality, resolution, export_settings, status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $project_id,
            $user['id'],
            $export_name,
            $format,
            $quality,
            $export_settings['resolution'],
            json_encode($export_settings)
        ]);
        
        $export_id = $pdo->lastInsertId();
        
        // Lancement export asynchrone
        $result = startAsyncExport($export_id, $project, $export_settings);
        
        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => 'Export démarré avec succès',
                'export_id' => $export_id
            ]);
        } else {
            throw new Exception($result['error']);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// API Progress tracking
if ($action === 'get_progress' && $export_id) {
    header('Content-Type: application/json');
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_exports WHERE id = ? AND user_id = ?");
        $stmt->execute([$export_id, $user['id']]);
        $export = $stmt->fetch();
        
        if (!$export) {
            throw new Exception("Export introuvable");
        }
        
        echo json_encode([
            'success' => true,
            'export' => $export,
            'progress' => (int)$export['progress'],
            'status' => $export['status'],
            'estimated_time' => $export['estimated_time'],
            'file_size' => $export['file_size'],
            'error_message' => $export['error_message']
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Téléchargement fichier exporté
if ($action === 'download' && $export_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_exports WHERE id = ? AND user_id = ? AND status = 'completed'");
        $stmt->execute([$export_id, $user['id']]);
        $export = $stmt->fetch();
        
        if (!$export || !$export['file_path']) {
            throw new Exception("Fichier d'export introuvable");
        }
        
        $file_path = $_SERVER['DOCUMENT_ROOT'] . $export['file_path'];
        if (!file_exists($file_path)) {
            throw new Exception("Fichier physique introuvable");
        }
        
        // Mise à jour compteur téléchargements
        $stmt = $pdo->prepare("UPDATE clipmaker_exports SET download_count = download_count + 1 WHERE id = ?");
        $stmt->execute([$export_id]);
        
        // Headers téléchargement
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $export['export_name'] . '.' . $export['format'] . '"');
        header('Content-Length: ' . filesize($file_path));
        header('Cache-Control: no-cache, must-revalidate');
        
        readfile($file_path);
        exit;
        
    } catch (Exception $e) {
        http_response_code(404);
        echo "Erreur: " . $e->getMessage();
        exit;
    }
}

// Récupération exports utilisateur
try {
    $stmt = $pdo->prepare("
        SELECT e.*, p.name as project_name 
        FROM clipmaker_exports e
        LEFT JOIN clipmaker_projects p ON e.project_id = p.id
        WHERE e.user_id = ? 
        ORDER BY e.created_at DESC 
        LIMIT 50
    ");
    $stmt->execute([$user['id']]);
    $exports = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Projets pour sélection
    $stmt = $pdo->prepare("SELECT id, name FROM clipmaker_projects WHERE user_id = ? ORDER BY name");
    $stmt->execute([$user['id']]);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Projet spécifique si fourni
    $current_project = null;
    if ($project_id) {
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_projects WHERE id = ? AND user_id = ?");
        $stmt->execute([$project_id, $user['id']]);
        $current_project = $stmt->fetch();
    }
    
    // Stats exports
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_exports,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_exports,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed_exports,
            SUM(CASE WHEN status IN ('pending', 'processing') THEN 1 ELSE 0 END) as active_exports,
            SUM(COALESCE(file_size, 0)) as total_size
        FROM clipmaker_exports 
        WHERE user_id = ?
    ");
    $stmt->execute([$user['id']]);
    $export_stats = $stmt->fetch();
    
} catch (PDOException $e) {
    error_log("Erreur récupération exports: " . $e->getMessage());
    $exports = [];
    $projects = [];
    $current_project = null;
    $export_stats = [
        'total_exports' => 0,
        'completed_exports' => 0,
        'failed_exports' => 0,
        'active_exports' => 0,
        'total_size' => 0
    ];
}

// Fonction export asynchrone
function startAsyncExport($export_id, $project, $settings) {
    global $pdo, $user, $ffmpeg_path, $output_path, $temp_path;
    
    try {
        // Mise à jour statut
        $stmt = $pdo->prepare("UPDATE clipmaker_exports SET status = 'processing', started_at = NOW() WHERE id = ?");
        $stmt->execute([$export_id]);
        
        // Préparation dossiers
        $output_dir = $_SERVER['DOCUMENT_ROOT'] . $output_path;
        $temp_dir = $temp_path . 'export_' . $export_id . '/';
        
        if (!is_dir($output_dir)) mkdir($output_dir, 0755, true);
        if (!is_dir($temp_dir)) mkdir($temp_dir, 0755, true);
        
        // Nom fichier de sortie
        $output_filename = 'export_' . $export_id . '_' . time() . '.' . $settings['format_config']['extension'];
        $output_file = $output_dir . $output_filename;
        
        // Construction commande FFmpeg
        $ffmpeg_command = buildFFmpegCommand($project, $settings, $temp_dir, $output_file);
        
        // Sauvegarde commande en base
        $stmt = $pdo->prepare("UPDATE clipmaker_exports SET ffmpeg_command = ? WHERE id = ?");
        $stmt->execute([$ffmpeg_command, $export_id]);
        
        // Exécution asynchrone avec monitoring
        $result = executeFFmpegAsync($export_id, $ffmpeg_command, $output_file);
        
        return $result;
        
    } catch (Exception $e) {
        // Marquer export comme échoué
        $stmt = $pdo->prepare("
            UPDATE clipmaker_exports 
            SET status = 'failed', error_message = ? 
            WHERE id = ?
        ");
        $stmt->execute([$e->getMessage(), $export_id]);
        
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function buildFFmpegCommand($project, $settings, $temp_dir, $output_file) {
    global $ffmpeg_path;
    
    $timeline_data = json_decode($project['timeline_data'], true) ?: [];
    $preset = $settings['preset_config'];
    $format = $settings['format_config'];
    
    // Base command
    $command = $ffmpeg_path . ' -y';
    
    // Inputs (simulation - à adapter selon structure timeline)
    $command .= ' -f lavfi -i color=black:size=' . $settings['resolution'] . ':duration=10';
    
    // Filtres vidéo
    $video_filters = [];
    
    // Scaling si nécessaire
    if ($settings['resolution'] !== $project['resolution_width'] . 'x' . $project['resolution_height']) {
        $video_filters[] = 'scale=' . str_replace('x', ':', $settings['resolution']);
    }
    
    // FPS
    if ($settings['fps'] != $project['fps']) {
        $video_filters[] = 'fps=' . $settings['fps'];
    }
    
    // Application filtres
    if (!empty($video_filters)) {
        $command .= ' -vf "' . implode(',', $video_filters) . '"';
    }
    
    // Codecs et qualité selon format
    switch ($format['extension']) {
        case 'mp4':
            $command .= ' -c:v ' . $preset['video_codec'];
            $command .= ' -crf ' . $preset['crf'];
            $command .= ' -preset ' . $preset['preset'];
            $command .= ' -c:a ' . $preset['audio_codec'];
            $command .= ' -b:a ' . $preset['audio_bitrate'];
            break;
            
        case 'webm':
            $command .= ' -c:v libvpx-vp9';
            $command .= ' -crf ' . $preset['crf'];
            $command .= ' -b:v 0';
            $command .= ' -c:a libopus';
            $command .= ' -b:a ' . $preset['audio_bitrate'];
            break;
            
        case 'mov':
            $command .= ' -c:v prores_ks';
            $command .= ' -profile:v 2';
            $command .= ' -c:a pcm_s16le';
            break;
            
        case 'gif':
            $command .= ' -vf "fps=15,scale=480:-1:flags=lanczos,palettegen=reserve_transparent=0" -y ' . $temp_dir . 'palette.png';
            $command .= ' && ' . $ffmpeg_path . ' -y -i ' . $temp_dir . 'palette.png';
            $command .= ' -lavfi -i color=black:size=' . $settings['resolution'] . ':duration=10';
            $command .= ' -filter_complex "fps=15,scale=480:-1:flags=lanczos[x];[x][0:v]paletteuse"';
            break;
    }
    
    // Durée si spécifiée
    if ($settings['start_time'] > 0) {
        $command .= ' -ss ' . $settings['start_time'];
    }
    
    if ($settings['end_time']) {
        $duration = $settings['end_time'] - $settings['start_time'];
        $command .= ' -t ' . $duration;
    }
    
    // Fichier de sortie
    $command .= ' "' . $output_file . '"';
    
    // Progress monitoring
    $command .= ' 2>&1';
    
    return $command;
}

function executeFFmpegAsync($export_id, $command, $output_file) {
    global $pdo;
    
    // Lancement process en arrière-plan avec monitoring
    $descriptorspec = [
        0 => ["pipe", "r"],
        1 => ["pipe", "w"],
        2 => ["pipe", "w"]
    ];
    
    $process = proc_open($command, $descriptorspec, $pipes);
    
    if (!is_resource($process)) {
        return [
            'success' => false,
            'error' => 'Impossible de lancer FFmpeg'
        ];
    }
    
    // Fermer stdin
    fclose($pipes[0]);
    
    // Lecture output pour monitoring progress
    stream_set_blocking($pipes[1], false);
    stream_set_blocking($pipes[2], false);
    
    $output = '';
    $error_output = '';
    $start_time = time();
    
    while (true) {
        $stdout = fread($pipes[1], 8192);
        $stderr = fread($pipes[2], 8192);
        
        if ($stdout !== false && $stdout !== '') {
            $output .= $stdout;
        }
        
        if ($stderr !== false && $stderr !== '') {
            $error_output .= $stderr;
            
            // Parse progress FFmpeg
            if (preg_match('/time=(\d+):(\d+):(\d+\.\d+)/', $stderr, $matches)) {
                $current_time = ($matches[1] * 3600) + ($matches[2] * 60) + floatval($matches[3]);
                $estimated_total = 10; // Durée estimée projet (à calculer)
                
                $progress = min(100, ($current_time / $estimated_total) * 100);
                $estimated_remaining = ($estimated_total - $current_time);
                
                // Mise à jour progress en base
                $stmt = $pdo->prepare("
                    UPDATE clipmaker_exports 
                    SET progress = ?, estimated_time = ? 
                    WHERE id = ?
                ");
                $stmt->execute([
                    (int)$progress,
                    (int)$estimated_remaining,
                    $export_id
                ]);
            }
        }
        
        // Vérifier si process terminé
        $status = proc_get_status($process);
        if (!$status['running']) {
            break;
        }
        
        // Timeout sécurité (1 heure max)
        if (time() - $start_time > 3600) {
            proc_terminate($process);
            break;
        }
        
        usleep(100000); // 0.1 seconde
    }
    
    fclose($pipes[1]);
    fclose($pipes[2]);
    $return_code = proc_close($process);
    
    // Vérification résultat
    if ($return_code === 0 && file_exists($output_file)) {
        // Export réussi
        $file_size = filesize($output_file);
        $relative_path = str_replace($_SERVER['DOCUMENT_ROOT'], '', $output_file);
        
        $stmt = $pdo->prepare("
            UPDATE clipmaker_exports 
            SET status = 'completed', 
                progress = 100, 
                file_path = ?, 
                file_size = ?,
                completed_at = NOW(),
                processing_log = ?
            WHERE id = ?
        ");
        
        $stmt->execute([
            $relative_path,
            $file_size,
            substr($output . $error_output, -10000), // Derniers 10k chars
            $export_id
        ]);
        
        return ['success' => true];
        
    } else {
        // Export échoué
        $stmt = $pdo->prepare("
            UPDATE clipmaker_exports 
            SET status = 'failed', 
                error_message = ?,
                processing_log = ?
            WHERE id = ?
        ");
        
        $stmt->execute([
            'FFmpeg error (code: ' . $return_code . ')',
            substr($output . $error_output, -10000),
            $export_id
        ]);
        
        return [
            'success' => false,
            'error' => 'Export échoué'
        ];
    }
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' B';
}

function formatDuration($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $secs = floor($seconds % 60);
    
    if ($hours > 0) {
        return sprintf('%d:%02d:%02d', $hours, $minutes, $secs);
    } else {
        return sprintf('%d:%02d', $minutes, $secs);
    }
}

?>
<div class="export-module">
    <!-- ================================================================
     * 🎥 HEADER MODULE EXPORT
     * ================================================================ -->
    <div class="module-header">
        <div class="header-content">
            <h2>🎥 Export & Rendu</h2>
            <p>Exportez vos projets en haute qualité avec FFmpeg</p>
        </div>
        <div class="header-actions">
            <button class="btn btn-primary" onclick="showExportModal()">
                <span>🚀</span>
                <span>Nouvel Export</span>
            </button>
        </div>
    </div>

    <!-- ================================================================
     * 📊 STATISTIQUES EXPORTS
     * ================================================================ -->
    <div class="export-stats">
        <div class="stat-card">
            <div class="stat-icon">📊</div>
            <div class="stat-number"><?php echo $export_stats['total_exports']; ?></div>
            <div class="stat-label">Total exports</div>
        </div>
        
        <div class="stat-card success">
            <div class="stat-icon">✅</div>
            <div class="stat-number"><?php echo $export_stats['completed_exports']; ?></div>
            <div class="stat-label">Terminés</div>
        </div>
        
        <div class="stat-card processing">
            <div class="stat-icon">⏳</div>
            <div class="stat-number"><?php echo $export_stats['active_exports']; ?></div>
            <div class="stat-label">En cours</div>
        </div>
        
        <div class="stat-card error">
            <div class="stat-icon">❌</div>
            <div class="stat-number"><?php echo $export_stats['failed_exports']; ?></div>
            <div class="stat-label">Échoués</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">💾</div>
            <div class="stat-number"><?php echo formatFileSize($export_stats['total_size']); ?></div>
            <div class="stat-label">Stockage total</div>
        </div>
    </div>

    <!-- ================================================================
     * 🔄 QUEUE DE RENDU EN TEMPS RÉEL
     * ================================================================ -->
    <?php 
    $active_exports = array_filter($exports, fn($e) => in_array($e['status'], ['pending', 'processing']));
    if (!empty($active_exports)): 
    ?>
    <div class="render-queue">
        <h3>🔄 Queue de Rendu</h3>
        
        <?php foreach ($active_exports as $export): ?>
            <div class="queue-item" data-export-id="<?php echo $export['id']; ?>">
                <div class="queue-info">
                    <div class="queue-name"><?php echo htmlspecialchars($export['export_name']); ?></div>
                    <div class="queue-project"><?php echo htmlspecialchars($export['project_name']); ?></div>
                    <div class="queue-format"><?php echo strtoupper($export['format']); ?> • <?php echo ucfirst($export['quality']); ?></div>
                </div>
                
                <div class="queue-progress">
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo $export['progress']; ?>%"></div>
                    </div>
                    <div class="progress-text">
                        <span class="progress-percent"><?php echo $export['progress']; ?>%</span>
                        <span class="progress-status"><?php echo ucfirst($export['status']); ?></span>
                        <?php if ($export['estimated_time']): ?>
                            <span class="progress-eta">Reste: <?php echo formatDuration($export['estimated_time']); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="queue-actions">
                    <?php if ($export['status'] === 'processing'): ?>
                        <button class="btn btn-sm btn-secondary" onclick="cancelExport(<?php echo $export['id']; ?>)">
                            ⏹️ Annuler
                        </button>
                    <?php endif; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>

    <!-- ================================================================
     * 📋 HISTORIQUE DES EXPORTS
     * ================================================================ -->
    <div class="exports-history">
        <h3>📋 Historique des Exports</h3>
        
        <?php if (empty($exports)): ?>
            <div class="empty-state">
                <div class="empty-icon">🎥</div>
                <h4>Aucun export trouvé</h4>
                <p>Lancez votre premier export pour commencer !</p>
                <button class="btn btn-primary" onclick="showExportModal()">
                    🚀 Premier Export
                </button>
            </div>
        <?php else: ?>
            <div class="exports-grid">
                <?php foreach ($exports as $export): ?>
                    <div class="export-card status-<?php echo $export['status']; ?>">
                        <div class="export-header">
                            <div class="export-info">
                                <h4 class="export-name"><?php echo htmlspecialchars($export['export_name']); ?></h4>
                                <div class="export-project"><?php echo htmlspecialchars($export['project_name']); ?></div>
                            </div>
                            
                            <div class="export-status">
                                <div class="status-badge status-<?php echo $export['status']; ?>">
                                    <?php
                                    echo match($export['status']) {
                                        'pending' => '⏳ En attente',
                                        'processing' => '⚙️ En cours',
                                        'completed' => '✅ Terminé',
                                        'failed' => '❌ Échoué',
                                        'cancelled' => '⏹️ Annulé',
                                        default => ucfirst($export['status'])
                                    };
                                    ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="export-meta">
                            <div class="meta-row">
                                <span class="meta-label">Format:</span>
                                <span class="meta-value"><?php echo strtoupper($export['format']); ?></span>
                            </div>
                            <div class="meta-row">
                                <span class="meta-label">Qualité:</span>
                                <span class="meta-value"><?php echo ucfirst($export['quality']); ?></span>
                            </div>
                            <div class="meta-row">
                                <span class="meta-label">Résolution:</span>
                                <span class="meta-value"><?php echo $export['resolution']; ?></span>
                            </div>
                            <?php if ($export['file_size']): ?>
                                <div class="meta-row">
                                    <span class="meta-label">Taille:</span>
                                    <span class="meta-value"><?php echo formatFileSize($export['file_size']); ?></span>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($export['status'] === 'processing'): ?>
                            <div class="export-progress">
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $export['progress']; ?>%"></div>
                                </div>
                                <div class="progress-text"><?php echo $export['progress']; ?>%</div>
                            </div>
                        <?php endif; ?>
                        
                        <div class="export-actions">
                            <?php if ($export['status'] === 'completed'): ?>
                                <button class="btn btn-sm btn-primary" onclick="downloadExport(<?php echo $export['id']; ?>)">
                                    ⬇️ Télécharger
                                </button>
                                <button class="btn btn-sm btn-secondary" onclick="previewExport(<?php echo $export['id']; ?>)">
                                    👁️ Aperçu
                                </button>
                            <?php elseif ($export['status'] === 'failed'): ?>
                                <button class="btn btn-sm btn-warning" onclick="showErrorDetails(<?php echo $export['id']; ?>)">
                                    🔍 Voir erreur
                                </button>
                                <button class="btn btn-sm btn-secondary" onclick="retryExport(<?php echo $export['id']; ?>)">
                                    🔄 Réessayer
                                </button>
                            <?php elseif ($export['status'] === 'processing'): ?>
                                <button class="btn btn-sm btn-secondary" onclick="cancelExport(<?php echo $export['id']; ?>)">
                                    ⏹️ Annuler
                                </button>
                            <?php endif; ?>
                            
                            <button class="btn btn-sm btn-outline" onclick="showExportDetails(<?php echo $export['id']; ?>)">
                                ℹ️ Détails
                            </button>
                        </div>
                        
                        <div class="export-dates">
                            <small>Créé: <?php echo date('d/m/Y H:i', strtotime($export['created_at'])); ?></small>
                            <?php if ($export['completed_at']): ?>
                                <small>Terminé: <?php echo date('d/m/Y H:i', strtotime($export['completed_at'])); ?></small>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- ================================================================
 * 🚀 MODAL NOUVEL EXPORT
 * ================================================================ -->
<div id="exportModal" class="modal">
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h3>🚀 Nouvel Export</h3>
            <button class="btn-close" onclick="closeExportModal()">&times;</button>
        </div>
        
        <form id="exportForm" onsubmit="startExport(event)">
            <div class="modal-body">
                <div class="form-grid">
                    <div class="form-section">
                        <h4>📁 Projet</h4>
                        
                        <div class="form-group">
                            <label for="projectSelect">Projet à exporter *</label>
                            <select id="projectSelect" name="project_id" required onchange="updateProjectInfo()">
                                <option value="">Sélectionnez un projet</option>
                                <?php foreach ($projects as $project): ?>
                                    <option value="<?php echo $project['id']; ?>" 
                                            <?php echo $current_project && $current_project['id'] == $project['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($project['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div id="projectInfo" class="project-info" style="display: none;">
                            <div class="info-grid">
                                <div class="info-item">
                                    <span class="info-label">Résolution:</span>
                                    <span id="projectResolution">-</span>
                                </div>
                                <div class="info-item">
                                    <span class="info-label">FPS:</span>
                                    <span id="projectFps">-</span>
                                </div>
                                <div class="info-item">
                                    <span class="info-label">Durée:</span>
                                    <span id="projectDuration">-</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-section">
                        <h4>⚙️ Paramètres Export</h4>
                        
                        <div class="form-group">
                            <label for="exportName">Nom de l'export *</label>
                            <input type="text" id="exportName" name="export_name" required 
                                   placeholder="Mon_super_export">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="formatSelect">Format *</label>
                                <select id="formatSelect" name="format" required onchange="updateFormatInfo()">
                                    <?php foreach ($output_formats as $key => $format): ?>
                                        <option value="<?php echo $key; ?>"><?php echo $format['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="qualitySelect">Qualité *</label>
                                <select id="qualitySelect" name="quality" required onchange="updateQualityInfo()">
                                    <?php foreach ($quality_presets as $key => $preset): ?>
                                        <option value="<?php echo $key; ?>"><?php echo $preset['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        
                        <div id="formatInfo" class="format-info">
                            <p id="formatDescription"></p>
                        </div>
                        
                        <div id="qualityInfo" class="quality-info">
                            <p id="qualityDescription"></p>
                        </div>
                    </div>
                </div>
                
                <div class="form-section">
                    <h4>🎯 Options Avancées</h4>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="resolutionCustom">Résolution personnalisée</label>
                            <select id="resolutionCustom" name="resolution">
                                <option value="">Résolution du projet</option>
                                <option value="1920x1080">1920x1080 (Full HD)</option>
                                <option value="1280x720">1280x720 (HD)</option>
                                <option value="1080x1080">1080x1080 (Carré)</option>
                                <option value="1080x1920">1080x1920 (Portrait)</option>
                                <option value="854x480">854x480 (SD)</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="fpsCustom">FPS personnalisé</label>
                            <select id="fpsCustom" name="fps">
                                <option value="">FPS du projet</option>
                                <option value="24">24 FPS</option>
                                <option value="25">25 FPS</option>
                                <option value="30">30 FPS</option>
                                <option value="60">60 FPS</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="startTime">Début (secondes)</label>
                            <input type="number" id="startTime" name="start_time" min="0" step="0.1" placeholder="0">
                        </div>
                        
                        <div class="form-group">
                            <label for="endTime">Fin (secondes)</label>
                            <input type="number" id="endTime" name="end_time" min="0" step="0.1" placeholder="Fin du projet">
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeExportModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    🚀 Lancer l'Export
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* ================================================================
 * 🎨 STYLES MODULE EXPORT
 * ================================================================ */

.export-module {
    padding: 0;
}

.module-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 2px solid rgba(255, 107, 107, 0.1);
}

.header-content h2 {
    color: #333;
    margin-bottom: 5px;
    font-size: 24px;
}

.header-content p {
    color: #666;
    font-size: 14px;
}

.header-actions .btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: 10px;
    cursor: pointer;
    font-weight: 600;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 107, 0.3);
}

.btn-secondary {
    background: #f8f9fa;
    color: #666;
    border: 1px solid #ddd;
}

.btn-secondary:hover {
    background: #e9ecef;
}

.btn-sm {
    padding: 6px 12px;
    font-size: 12px;
}

.btn-outline {
    background: transparent;
    border: 1px solid #ddd;
    color: #666;
}

.btn-warning {
    background: #ffc107;
    color: #212529;
}

.export-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border-radius: 16px;
    padding: 20px;
    text-align: center;
    transition: all 0.3s ease;
    border: 2px solid transparent;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 40px rgba(0, 0, 0, 0.15);
    border-color: rgba(255, 107, 107, 0.2);
}

.stat-card .stat-icon {
    font-size: 32px;
    margin-bottom: 10px;
}

.stat-card .stat-number {
    font-size: 20px;
    font-weight: 700;
    color: #ff6b6b;
    margin-bottom: 5px;
}

.stat-card .stat-label {
    color: #666;
    font-size: 12px;
    font-weight: 500;
}

.stat-card.success .stat-number { color: #28a745; }
.stat-card.processing .stat-number { color: #ffc107; }
.stat-card.error .stat-number { color: #dc3545; }

.render-queue {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    padding: 25px;
    margin-bottom: 30px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

.render-queue h3 {
    color: #333;
    margin-bottom: 20px;
    font-size: 18px;
}

.queue-item {
    background: rgba(255, 107, 107, 0.05);
    border: 1px solid rgba(255, 107, 107, 0.1);
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 15px;
    display: grid;
    grid-template-columns: 1fr 2fr auto;
    gap: 20px;
    align-items: center;
}

.queue-info .queue-name {
    font-weight: 600;
    color: #333;
    margin-bottom: 5px;
}

.queue-info .queue-project {
    color: #666;
    font-size: 14px;
    margin-bottom: 5px;
}

.queue-info .queue-format {
    color: #999;
    font-size: 12px;
}

.queue-progress {
    flex: 1;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: rgba(255, 107, 107, 0.2);
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 8px;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #ff6b6b, #ee5a24);
    transition: width 0.3s ease;
}

.progress-text {
    display: flex;
    justify-content: space-between;
    font-size: 12px;
    color: #666;
}

.exports-history {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

.exports-history h3 {
    color: #333;
    margin-bottom: 20px;
    font-size: 18px;
}

.exports-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 20px;
}

.export-card {
    background: rgba(255, 255, 255, 0.9);
    border: 2px solid transparent;
    border-radius: 12px;
    padding: 20px;
    transition: all 0.3s ease;
}

.export-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
}

.export-card.status-completed {
    border-color: rgba(40, 167, 69, 0.3);
}

.export-card.status-processing {
    border-color: rgba(255, 193, 7, 0.3);
}

.export-card.status-failed {
    border-color: rgba(220, 53, 69, 0.3);
}

.export-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 15px;
}

.export-name {
    font-size: 16px;
    font-weight: 600;
    color: #333;
    margin: 0 0 5px 0;
}

.export-project {
    color: #666;
    font-size: 14px;
}

.status-badge {
    padding: 4px 8px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-pending {
    background: rgba(108, 117, 125, 0.2);
    color: #6c757d;
}

.status-processing {
    background: rgba(255, 193, 7, 0.2);
    color: #ff8f00;
}

.status-completed {
    background: rgba(40, 167, 69, 0.2);
    color: #28a745;
}

.status-failed {
    background: rgba(220, 53, 69, 0.2);
    color: #dc3545;
}

.status-cancelled {
    background: rgba(108, 117, 125, 0.2);
    color: #6c757d;
}

.export-meta {
    margin-bottom: 15px;
}

.meta-row {
    display: flex;
    justify-content: space-between;
    margin-bottom: 5px;
    font-size: 12px;
}

.meta-label {
    color: #666;
}

.meta-value {
    color: #333;
    font-weight: 500;
}

.export-progress {
    margin-bottom: 15px;
}

.export-actions {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
    margin-bottom: 15px;
}

.export-dates {
    padding-top: 15px;
    border-top: 1px solid #eee;
    font-size: 11px;
    color: #999;
    display: flex;
    justify-content: space-between;
}

.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: #666;
}

.empty-icon {
    font-size: 64px;
    margin-bottom: 20px;
    opacity: 0.5;
}

.empty-state h4 {
    margin-bottom: 10px;
    color: #333;
}

.empty-state p {
    margin-bottom: 30px;
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    z-index: 1000;
    backdrop-filter: blur(5px);
}

.modal-content {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    border-radius: 16px;
    width: 90%;
    max-width: 800px;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    border-bottom: 1px solid #eee;
}

.modal-header h3 {
    margin: 0;
    color: #333;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    padding: 5px;
}

.modal-body {
    padding: 30px;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 30px;
    margin-bottom: 30px;
}

.form-section {
    background: rgba(255, 107, 107, 0.05);
    padding: 20px;
    border-radius: 12px;
}

.form-section h4 {
    color: #333;
    margin-bottom: 15px;
    font-size: 16px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
    font-size: 14px;
}

.form-group input, .form-group select {
    width: 100%;
    padding: 10px 12px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-group input:focus, .form-group select:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
}

.project-info {
    background: rgba(255, 255, 255, 0.8);
    padding: 15px;
    border-radius: 8px;
    margin-top: 10px;
}

.info-grid {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr;
    gap: 15px;
}

.info-item {
    text-align: center;
}

.info-label {
    display: block;
    font-size: 12px;
    color: #666;
    margin-bottom: 5px;
}

.format-info, .quality-info {
    background: rgba(255, 255, 255, 0.8);
    padding: 10px;
    border-radius: 6px;
    margin-top: 10px;
}

.format-info p, .quality-info p {
    margin: 0;
    font-size: 13px;
    color: #666;
}

.modal-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
    padding: 20px 30px;
    border-top: 1px solid #eee;
}

/* Responsive */
@media (max-width: 768px) {
    .export-stats {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .exports-grid {
        grid-template-columns: 1fr;
    }
    
    .queue-item {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .info-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
/**
 * ================================================================
 * 🎯 JAVASCRIPT MODULE EXPORT
 * ================================================================ */

// Configuration globale
const formatConfigs = <?php echo json_encode($output_formats); ?>;
const qualityPresets = <?php echo json_encode($quality_presets); ?>;
const projects = <?php echo json_encode($projects); ?>;

let activeExports = new Set();
let progressInterval;

// Fonctions principales
function showExportModal() {
    document.getElementById('exportModal').style.display = 'block';
    updateFormatInfo();
    updateQualityInfo();
    
    // Auto-sélection projet si fourni
    const urlParams = new URLSearchParams(window.location.search);
    const projectId = urlParams.get('project');
    if (projectId) {
        document.getElementById('projectSelect').value = projectId;
        updateProjectInfo();
    }
}

function closeExportModal() {
    document.getElementById('exportModal').style.display = 'none';
    document.getElementById('exportForm').reset();
    document.getElementById('projectInfo').style.display = 'none';
}

function updateProjectInfo() {
    const projectSelect = document.getElementById('projectSelect');
    const projectInfo = document.getElementById('projectInfo');
    
    if (!projectSelect.value) {
        projectInfo.style.display = 'none';
        return;
    }
    
    // Récupération info projet (simulation)
    fetch(`../api/clipmaker.php?action=get_project&id=${projectSelect.value}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const project = data.project;
                document.getElementById('projectResolution').textContent = 
                    `${project.resolution_width}x${project.resolution_height}`;
                document.getElementById('projectFps').textContent = `${project.fps} FPS`;
                document.getElementById('projectDuration').textContent = 
                    project.duration ? formatDuration(project.duration) : 'N/A';
                
                projectInfo.style.display = 'block';
            }
        })
        .catch(error => {
            console.error('Erreur récupération projet:', error);
        });
}

function updateFormatInfo() {
    const formatSelect = document.getElementById('formatSelect');
    const formatInfo = document.getElementById('formatInfo');
    const formatDescription = document.getElementById('formatDescription');
    
    const selectedFormat = formatConfigs[formatSelect.value];
    if (selectedFormat) {
        formatDescription.textContent = selectedFormat.description;
    }
}

function updateQualityInfo() {
    const qualitySelect = document.getElementById('qualitySelect');
    const qualityInfo = document.getElementById('qualityInfo');
    const qualityDescription = document.getElementById('qualityDescription');
    
    const selectedQuality = qualityPresets[qualitySelect.value];
    if (selectedQuality) {
        qualityDescription.textContent = selectedQuality.description;
    }
}

function startExport(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    
    // Validation
    if (!formData.get('project_id')) {
        showNotification('❌ Veuillez sélectionner un projet', 'error');
        return;
    }
    
    if (!formData.get('export_name')) {
        showNotification('❌ Veuillez saisir un nom d\'export', 'error');
        return;
    }
    
    // Affichage loading
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '⏳ Démarrage...';
    submitBtn.disabled = true;
    
    fetch('?action=start_export', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('✅ ' + data.message, 'success');
            closeExportModal();
            
            // Ajout à la surveillance
            activeExports.add(data.export_id);
            startProgressMonitoring();
            
            // Rechargement page après délai
            setTimeout(() => {
                window.location.reload();
            }, 2000);
            
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur export: ' + error.message, 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

function startProgressMonitoring() {
    // Surveillance des exports actifs
    if (progressInterval) {
        clearInterval(progressInterval);
    }
    
    progressInterval = setInterval(() => {
        updateExportsProgress();
    }, 2000); // Toutes les 2 secondes
}

function updateExportsProgress() {
    const queueItems = document.querySelectorAll('.queue-item[data-export-id]');
    
    queueItems.forEach(item => {
        const exportId = item.dataset.exportId;
        
        fetch(`?action=get_progress&export=${exportId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    updateQueueItem(item, data.export);
                    
                    // Si terminé, retirer de la surveillance
                    if (data.export.status === 'completed' || data.export.status === 'failed') {
                        activeExports.delete(parseInt(exportId));
                        
                        // Recharger après délai pour voir le résultat
                        setTimeout(() => {
                            window.location.reload();
                        }, 3000);
                    }
                }
            })
            .catch(error => {
                console.error('Erreur progress:', error);
            });
    });
    
    // Arrêter surveillance si plus d'exports actifs
    if (activeExports.size === 0) {
        clearInterval(progressInterval);
    }
}

function updateQueueItem(item, exportData) {
    const progressFill = item.querySelector('.progress-fill');
    const progressPercent = item.querySelector('.progress-percent');
    const progressStatus = item.querySelector('.progress-status');
    const progressEta = item.querySelector('.progress-eta');
    
    if (progressFill) {
        progressFill.style.width = exportData.progress + '%';
    }
    
    if (progressPercent) {
        progressPercent.textContent = exportData.progress + '%';
    }
    
    if (progressStatus) {
        progressStatus.textContent = exportData.status.charAt(0).toUpperCase() + exportData.status.slice(1);
    }
    
    if (progressEta && exportData.estimated_time) {
        progressEta.textContent = 'Reste: ' + formatDuration(exportData.estimated_time);
    }
}

function downloadExport(exportId) {
    window.open(`?action=download&export=${exportId}`, '_blank');
}

function previewExport(exportId) {
    // TODO: Modal aperçu
    console.log('Preview export:', exportId);
}

function cancelExport(exportId) {
    if (confirm('Annuler cet export ?')) {
        fetch(`../api/clipmaker.php?action=cancel_export&id=${exportId}`, {
            method: 'POST'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✅ Export annulé', 'success');
                activeExports.delete(exportId);
                setTimeout(() => window.location.reload(), 1000);
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur annulation: ' + error.message, 'error');
        });
    }
}

function retryExport(exportId) {
    if (confirm('Relancer cet export ?')) {
        fetch(`../api/clipmaker.php?action=retry_export&id=${exportId}`, {
            method: 'POST'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✅ Export relancé', 'success');
                activeExports.add(exportId);
                startProgressMonitoring();
                setTimeout(() => window.location.reload(), 1000);
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur relance: ' + error.message, 'error');
        });
    }
}

function showErrorDetails(exportId) {
    fetch(`../api/clipmaker.php?action=get_export_details&id=${exportId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const export_data = data.export;
                const errorModal = createErrorModal(export_data);
                document.body.appendChild(errorModal);
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur détails: ' + error.message, 'error');
        });
}

function createErrorModal(exportData) {
    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.display = 'block';
    
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3>❌ Détails de l'erreur</h3>
                <button class="btn-close" onclick="this.closest('.modal').remove()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="error-details">
                    <div class="detail-row">
                        <strong>Export:</strong> ${exportData.export_name}
                    </div>
                    <div class="detail-row">
                        <strong>Statut:</strong> ${exportData.status}
                    </div>
                    <div class="detail-row">
                        <strong>Erreur:</strong> ${exportData.error_message || 'Erreur inconnue'}
                    </div>
                    ${exportData.ffmpeg_command ? `
                        <div class="detail-row">
                            <strong>Commande FFmpeg:</strong>
                            <pre style="background: #f8f9fa; padding: 10px; border-radius: 4px; font-size: 12px; overflow-x: auto;">${exportData.ffmpeg_command}</pre>
                        </div>
                    ` : ''}
                    ${exportData.processing_log ? `
                        <div class="detail-row">
                            <strong>Log de traitement:</strong>
                            <pre style="background: #f8f9fa; padding: 10px; border-radius: 4px; font-size: 11px; max-height: 200px; overflow-y: auto;">${exportData.processing_log}</pre>
                        </div>
                    ` : ''}
                </div>
            </div>
            <div class="modal-actions" style="padding: 20px; border-top: 1px solid #eee; text-align: right;">
                <button class="btn btn-secondary" onclick="this.closest('.modal').remove()">Fermer</button>
                <button class="btn btn-primary" onclick="retryExport(${exportData.id}); this.closest('.modal').remove();">🔄 Réessayer</button>
            </div>
        </div>
    `;
    
    return modal;
}

function showExportDetails(exportId) {
    fetch(`../api/clipmaker.php?action=get_export_details&id=${exportId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const export_data = data.export;
                const detailsModal = createDetailsModal(export_data);
                document.body.appendChild(detailsModal);
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur détails: ' + error.message, 'error');
        });
}

function createDetailsModal(exportData) {
    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.display = 'block';
    
    const settings = JSON.parse(exportData.export_settings || '{}');
    
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3>ℹ️ Détails de l'export</h3>
                <button class="btn-close" onclick="this.closest('.modal').remove()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="details-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                    <div class="details-section">
                        <h4>📊 Informations générales</h4>
                        <div class="detail-row"><strong>Nom:</strong> ${exportData.export_name}</div>
                        <div class="detail-row"><strong>Format:</strong> ${exportData.format.toUpperCase()}</div>
                        <div class="detail-row"><strong>Qualité:</strong> ${exportData.quality}</div>
                        <div class="detail-row"><strong>Résolution:</strong> ${exportData.resolution}</div>
                        <div class="detail-row"><strong>Statut:</strong> ${exportData.status}</div>
                        ${exportData.file_size ? `<div class="detail-row"><strong>Taille:</strong> ${formatFileSize(exportData.file_size)}</div>` : ''}
                    </div>
                    
                    <div class="details-section">
                        <h4>⏱️ Temps de traitement</h4>
                        <div class="detail-row"><strong>Créé:</strong> ${new Date(exportData.created_at).toLocaleString('fr-FR')}</div>
                        ${exportData.started_at ? `<div class="detail-row"><strong>Démarré:</strong> ${new Date(exportData.started_at).toLocaleString('fr-FR')}</div>` : ''}
                        ${exportData.completed_at ? `<div class="detail-row"><strong>Terminé:</strong> ${new Date(exportData.completed_at).toLocaleString('fr-FR')}</div>` : ''}
                        ${exportData.download_count ? `<div class="detail-row"><strong>Téléchargements:</strong> ${exportData.download_count}</div>` : ''}
                    </div>
                </div>
                
                ${settings.preset_config ? `
                    <div class="details-section" style="margin-top: 20px;">
                        <h4>⚙️ Paramètres techniques</h4>
                        <div class="tech-details" style="background: #f8f9fa; padding: 15px; border-radius: 8px;">
                            <div class="detail-row"><strong>Codec vidéo:</strong> ${settings.preset_config.video_codec || 'N/A'}</div>
                            <div class="detail-row"><strong>Codec audio:</strong> ${settings.preset_config.audio_codec || 'N/A'}</div>
                            <div class="detail-row"><strong>CRF:</strong> ${settings.preset_config.crf || 'N/A'}</div>
                            <div class="detail-row"><strong>Preset:</strong> ${settings.preset_config.preset || 'N/A'}</div>
                            <div class="detail-row"><strong>Bitrate audio:</strong> ${settings.preset_config.audio_bitrate || 'N/A'}</div>
                        </div>
                    </div>
                ` : ''}
            </div>
            <div class="modal-actions" style="padding: 20px; border-top: 1px solid #eee; text-align: right;">
                <button class="btn btn-secondary" onclick="this.closest('.modal').remove()">Fermer</button>
                ${exportData.status === 'completed' ? `
                    <button class="btn btn-primary" onclick="downloadExport(${exportData.id}); this.closest('.modal').remove();">⬇️ Télécharger</button>
                ` : ''}
            </div>
        </div>
    `;
    
    return modal;
}

// Utilitaires
function formatFileSize(bytes) {
    if (bytes >= 1073741824) {
        return (bytes / 1073741824).toFixed(2) + ' GB';
    } else if (bytes >= 1048576) {
        return (bytes / 1048576).toFixed(2) + ' MB';
    } else if (bytes >= 1024) {
        return (bytes / 1024).toFixed(2) + ' KB';
    }
    return bytes + ' B';
}

function formatDuration(seconds) {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
}

function showNotification(message, type) {
    // Création notification toast
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#28a745' : type === 'warning' ? '#ffc107' : '#dc3545'};
        color: ${type === 'warning' ? '#212529' : 'white'};
        padding: 15px 20px;
        border-radius: 8px;
        z-index: 10000;
        font-weight: 600;
        box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        animation: slideIn 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 4000);
}

// Initialisation
document.addEventListener('DOMContentLoaded', function() {
    // Surveillance des exports actifs au chargement
    const queueItems = document.querySelectorAll('.queue-item[data-export-id]');
    queueItems.forEach(item => {
        activeExports.add(parseInt(item.dataset.exportId));
    });
    
    if (activeExports.size > 0) {
        startProgressMonitoring();
    }
    
    // Fermer modals avec Escape
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeExportModal();
            document.querySelectorAll('.modal').forEach(modal => {
                if (modal.id !== 'exportModal') {
                    modal.remove();
                }
            });
        }
    });
    
    // Auto-refresh page si exports actifs (toutes les 30 secondes)
    if (activeExports.size > 0) {
        setInterval(() => {
            if (document.hidden) return; // Pas de refresh si onglet inactif
            
            // Vérifier s'il y a encore des exports actifs
            fetch('?action=check_active_exports&ajax=1')
                .then(response => response.json())
                .then(data => {
                    if (data.has_active !== (activeExports.size > 0)) {
                        window.location.reload();
                    }
                })
                .catch(error => console.log('Erreur check actifs:', error));
        }, 30000);
    }
    
    console.log('🎥 Module Export Ready!');
    console.log('👁️ Surveillance exports actifs:', activeExports.size);
});

// Styles animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    .detail-row {
        margin-bottom: 10px;
        padding: 5px 0;
        border-bottom: 1px solid rgba(0,0,0,0.05);
    }
    
    .detail-row:last-child {
        border-bottom: none;
    }
    
    .details-section h4 {
        color: #333;
        margin-bottom: 15px;
        font-size: 16px;
        border-bottom: 2px solid rgba(255, 107, 107, 0.2);
        padding-bottom: 8px;
    }
`;
document.head.appendChild(style);
</script>