<?php
/**
 * ================================================================
 * 📸 GESTIONNAIRE MÉDIAS SMG - FFMPEG POWER
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Upload photos toutes sources (PC/mobile/tablette)
 * - Conversion FFmpeg automatique par réseau
 * - Gestion corbeille avec undo
 * - Organisation par user workflow
 * 
 * 💡 BY PAPY 77 - NO JSON INSIDE !
 * ================================================================
 */

session_start();
require_once 'config/db.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur user images: " . $e->getMessage());
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🔧 CONFIGURATION WORKFLOW
// ================================================================

function getUserMediaPath($user_id) {
    return "SMG_STUDIO_WORKFLOW/user_" . str_pad($user_id, 3, '0', STR_PAD_LEFT) . "/photos/";
}

function createUserDirectories($user_id) {
    $base_path = getUserMediaPath($user_id);
    $directories = [
        $base_path . 'incoming/',
        $base_path . 'originals/',
        $base_path . 'converted/instagram/',
        $base_path . 'converted/facebook/',
        $base_path . 'converted/stories/',
        $base_path . 'converted/twitter/',
        $base_path . 'trash/' . date('Y-m-d') . '/'
    ];
    
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }
    
    return $base_path;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS
// ================================================================

$action = $_GET['action'] ?? 'list';
$message = '';
$message_type = '';
$selected_for_post = $_GET['select_mode'] ?? false;

// Créer dossiers user si nécessaire
$user_media_path = createUserDirectories($user['id']);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    if (isset($_POST['upload_photos'])) {
        // 📤 UPLOAD ET CONVERSION PHOTOS
        try {
            $upload_results = [];
            $total_files = count($_FILES['photos']['name']);
            
            for ($i = 0; $i < $total_files; $i++) {
                if ($_FILES['photos']['error'][$i] === UPLOAD_ERR_OK) {
                    $result = processPhotoUpload($_FILES['photos'], $i, $user['id']);
                    $upload_results[] = $result;
                }
            }
            
            $success_count = count(array_filter($upload_results, fn($r) => $r['success']));
            $error_count = $total_files - $success_count;
            
            if ($success_count > 0) {
                $message = "✅ {$success_count} photo(s) uploadée(s) et converties !";
                if ($error_count > 0) {
                    $message .= " ({$error_count} erreur(s))";
                }
                $message_type = 'success';
            } else {
                $message = "❌ Aucune photo n'a pu être traitée";
                $message_type = 'error';
            }
            
        } catch (Exception $e) {
            $message = "❌ Erreur upload : " . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['select_medias'])) {
        // ✅ SÉLECTION MÉDIAS POUR POST
        $selected_ids = $_POST['media_ids'] ?? [];
        if (!empty($selected_ids)) {
            $ids_string = implode(',', $selected_ids);
            header("Location: posts.php?action=" . ($_POST['post_action'] ?? 'new') . "&media_ids={$ids_string}" . ($_POST['post_id'] ? "&id=" . $_POST['post_id'] : ''));
            exit;
        } else {
            $message = "⚠️ Sélectionnez au moins une photo";
            $message_type = 'error';
        }
    }
}

// Actions via GET
if ($action === 'delete' && isset($_GET['id'])) {
    // 🗑️ SUPPRESSION AVEC CORBEILLE
    try {
        $result = moveToTrash($_GET['id'], $user['id']);
        $message = $result['message'];
        $message_type = $result['success'] ? 'success' : 'error';
        header("Location: images.php?msg=" . urlencode($message) . "&type={$message_type}");
        exit;
    } catch (Exception $e) {
        $message = "❌ Erreur suppression : " . $e->getMessage();
        $message_type = 'error';
    }
}

if ($action === 'restore' && isset($_GET['id'])) {
    // 🔄 RESTAURATION DEPUIS CORBEILLE
    try {
        $result = restoreFromTrash($_GET['id'], $user['id']);
        $message = $result['message'];
        $message_type = $result['success'] ? 'success' : 'error';
        header("Location: images.php?msg=" . urlencode($message) . "&type={$message_type}");
        exit;
    } catch (Exception $e) {
        $message = "❌ Erreur restauration : " . $e->getMessage();
        $message_type = 'error';
    }
}

// Messages depuis URL
if (isset($_GET['msg'])) {
    $message = $_GET['msg'];
    $message_type = $_GET['type'] ?? 'info';
}

// ================================================================
// 📊 CHARGEMENT DES DONNÉES
// ================================================================

// Charger médias utilisateur
$user_medias = getUserMedias($user['id']);
$trash_medias = getTrashMedias($user['id']);

// Stats utilisateur
$stats = [
    'total_medias' => count($user_medias),
    'instagram_ready' => count(array_filter($user_medias, fn($m) => !empty($m['instagram_path']))),
    'facebook_ready' => count(array_filter($user_medias, fn($m) => !empty($m['facebook_path']))),
    'stories_ready' => count(array_filter($user_medias, fn($m) => !empty($m['stories_path']))),
    'total_size' => array_sum(array_map(fn($m) => $m['file_size'], $user_medias))
];

// ================================================================
// 🔧 FONCTIONS PRINCIPALES
// ================================================================

function processPhotoUpload($files, $index, $user_id) {
    $file = [
        'name' => $files['name'][$index],
        'type' => $files['type'][$index],
        'tmp_name' => $files['tmp_name'][$index],
        'size' => $files['size'][$index]
    ];
    
    try {
        // Validation
        $validation = validatePhoto($file);
        if (!$validation['success']) {
            return $validation;
        }
        
        // Déplacement vers incoming
        $user_path = getUserMediaPath($user_id);
        $safe_name = sanitizeFilename($file['name']);
        $incoming_path = $user_path . 'incoming/' . time() . '_' . $safe_name;
        
        if (!move_uploaded_file($file['tmp_name'], $incoming_path)) {
            return ['success' => false, 'message' => 'Erreur déplacement fichier'];
        }
        
        // Analyse avec FFprobe
        $media_info = analyzeMedia($incoming_path);
        if (!$media_info['success']) {
            unlink($incoming_path);
            return $media_info;
        }
        
        // Déplacement vers originals
        $original_name = 'original_' . time() . '_' . uniqid() . '.' . pathinfo($safe_name, PATHINFO_EXTENSION);
        $original_path = $user_path . 'originals/' . $original_name;
        rename($incoming_path, $original_path);
        
        // Conversion pour tous les réseaux
        $conversions = convertForAllNetworks($original_path, $user_path, $original_name);
        
        // Sauvegarde en base
        $media_id = saveMediaToDatabase($user_id, $original_path, $conversions, $media_info['info']);
        
        return [
            'success' => true,
            'message' => 'Photo convertie pour ' . count(array_filter($conversions)) . ' réseau(x)',
            'media_id' => $media_id,
            'conversions' => $conversions
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erreur : ' . $e->getMessage()];
    }
}

function validatePhoto($file) {
    // Taille max 50MB
    if ($file['size'] > 50 * 1024 * 1024) {
        return ['success' => false, 'message' => '⚠️ Fichier trop volumineux (' . formatFileSize($file['size']) . ') - Max 50MB'];
    }
    
    // Types autorisés
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/heic', 'image/heif'];
    if (!in_array(strtolower($file['type']), $allowed_types)) {
        return ['success' => false, 'message' => '❌ Format non supporté - Utilisez JPG, PNG, WebP ou HEIC'];
    }
    
    return ['success' => true];
}

function analyzeMedia($file_path) {
    $cmd = "ffprobe -v quiet -print_format csv -show_format -show_streams " . escapeshellarg($file_path) . " 2>&1";
    $output = shell_exec($cmd);
    
    if (empty($output) || strpos($output, 'error') !== false) {
        return ['success' => false, 'message' => '❌ Fichier corrompu ou format invalide'];
    }
    
    // Parser les infos basiques
    $lines = explode("\n", trim($output));
    $format_line = '';
    $stream_line = '';
    
    foreach ($lines as $line) {
        if (strpos($line, 'format,') === 0) {
            $format_line = $line;
        }
        if (strpos($line, 'stream,') === 0 && empty($stream_line)) {
            $stream_line = $line;
        }
    }
    
    $format_data = str_getcsv($format_line);
    $stream_data = str_getcsv($stream_line);
    
    return [
        'success' => true,
        'info' => [
            'width' => intval($stream_data[10] ?? 0),
            'height' => intval($stream_data[11] ?? 0),
            'duration' => floatval($format_data[5] ?? 0),
            'size' => intval($format_data[6] ?? 0),
            'format' => $format_data[2] ?? 'unknown'
        ]
    ];
}

function convertForAllNetworks($original_path, $user_path, $original_name) {
    $base_name = pathinfo($original_name, PATHINFO_FILENAME);
    $conversions = [];
    
    // Configurations par réseau
    $networks = [
        'instagram' => ['width' => 1080, 'height' => 1080, 'description' => 'carré'],
        'facebook' => ['width' => 1200, 'height' => 630, 'description' => 'paysage'],
        'stories' => ['width' => 1080, 'height' => 1920, 'description' => 'vertical'],
        'twitter' => ['width' => 1024, 'height' => 512, 'description' => 'bannière']
    ];
    
    foreach ($networks as $network => $config) {
        $output_path = $user_path . "converted/{$network}/{$base_name}_{$network}.jpg";
        
        $cmd = "ffmpeg -i " . escapeshellarg($original_path) . 
               " -vf \"scale={$config['width']}:{$config['height']}:force_original_aspect_ratio=decrease,pad={$config['width']}:{$config['height']}:(ow-iw)/2:(oh-ih)/2:black\"" .
               " -q:v 85 -y " . escapeshellarg($output_path) . " 2>&1";
        
        $result = shell_exec($cmd);
        
        if (file_exists($output_path) && filesize($output_path) > 0) {
            $conversions[$network . '_path'] = $output_path;
            $conversions[$network . '_size'] = filesize($output_path);
        } else {
            error_log("Erreur conversion {$network}: " . $result);
        }
    }
    
    return $conversions;
}

function saveMediaToDatabase($user_id, $original_path, $conversions, $media_info) {
    global $pdo;
    
    try {
        // Créer table si nécessaire
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS user_medias (
                id INT PRIMARY KEY AUTO_INCREMENT,
                user_id INT NOT NULL,
                original_name VARCHAR(255) NOT NULL,
                original_path VARCHAR(500) NOT NULL,
                instagram_path VARCHAR(500) NULL,
                facebook_path VARCHAR(500) NULL,
                stories_path VARCHAR(500) NULL,
                twitter_path VARCHAR(500) NULL,
                file_size INT DEFAULT 0,
                width INT DEFAULT 0,
                height INT DEFAULT 0,
                status ENUM('active', 'trash') DEFAULT 'active',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_user_status (user_id, status)
            )
        ");
        
        $stmt = $pdo->prepare("
            INSERT INTO user_medias 
            (user_id, original_name, original_path, instagram_path, facebook_path, stories_path, twitter_path, file_size, width, height) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $user_id,
            basename($original_path),
            $original_path,
            $conversions['instagram_path'] ?? null,
            $conversions['facebook_path'] ?? null,
            $conversions['stories_path'] ?? null,
            $conversions['twitter_path'] ?? null,
            $media_info['size'],
            $media_info['width'],
            $media_info['height']
        ]);
        
        return $pdo->lastInsertId();
        
    } catch (Exception $e) {
        error_log("Erreur sauvegarde média: " . $e->getMessage());
        throw $e;
    }
}

function getUserMedias($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM user_medias 
            WHERE user_id = ? AND status = 'active' 
            ORDER BY created_at DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur chargement médias: " . $e->getMessage());
        return [];
    }
}

function getTrashMedias($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM user_medias 
            WHERE user_id = ? AND status = 'trash' 
            ORDER BY created_at DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function moveToTrash($media_id, $user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE user_medias 
            SET status = 'trash' 
            WHERE id = ? AND user_id = ?
        ");
        $result = $stmt->execute([$media_id, $user_id]);
        
        if ($result) {
            return ['success' => true, 'message' => '🗑️ Photo déplacée vers la corbeille'];
        } else {
            return ['success' => false, 'message' => '❌ Photo introuvable'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erreur suppression'];
    }
}

function restoreFromTrash($media_id, $user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE user_medias 
            SET status = 'active' 
            WHERE id = ? AND user_id = ? AND status = 'trash'
        ");
        $result = $stmt->execute([$media_id, $user_id]);
        
        if ($result) {
            return ['success' => true, 'message' => '🔄 Photo restaurée avec succès'];
        } else {
            return ['success' => false, 'message' => '❌ Photo introuvable dans la corbeille'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erreur restauration'];
    }
}

// ================================================================
// 🔧 FONCTIONS UTILITAIRES
// ================================================================

function sanitizeFilename($filename) {
    $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
    return substr($filename, 0, 100);
}

function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, 1) . ' ' . $units[$pow];
}

function formatDate($date) {
    return date('d/m/Y H:i', strtotime($date));
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📸 Gestionnaire Photos - SMG Studio</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        /* ================================================================
         * 🎨 LAYOUT PRINCIPAL
         * ================================================================ */
        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #ff6b6b, #ff8e53);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .header-title h1 {
            font-size: 28px;
            color: #333;
            margin-bottom: 5px;
        }

        .header-title p {
            color: #666;
            font-size: 14px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .user-badge {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
            color: white;
            padding: 10px 20px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 14px;
        }

        .quick-nav {
            display: flex;
            gap: 10px;
        }

        .quick-nav a {
            padding: 8px 16px;
            background: rgba(255, 255, 255, 0.9);
            color: #ff6b6b;
            text-decoration: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
        }

        .quick-nav a:hover {
            background: white;
            transform: translateY(-1px);
        }

        /* ================================================================
         * 📊 STATS ET CARTES
         * ================================================================ */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #ff6b6b;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 22px;
            font-weight: 600;
            color: #333;
        }

        /* ================================================================
         * 🎨 BOUTONS ET FORMULAIRES
         * ================================================================ */
        .btn {
            padding: 12px 20px;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 14px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
        }

        /* ================================================================
         * 📤 ZONE UPLOAD
         * ================================================================ */
        .upload-zone {
            border: 3px dashed #e9ecef;
            border-radius: 15px;
            padding: 50px 30px;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
            margin-bottom: 30px;
            background: rgba(255, 255, 255, 0.5);
        }

        .upload-zone:hover {
            border-color: #ff6b6b;
            background: rgba(255, 107, 107, 0.05);
            transform: translateY(-2px);
        }

        .upload-zone.dragover {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.1);
            transform: scale(1.02);
        }

        .upload-icon {
            font-size: 64px;
            margin-bottom: 20px;
            color: #ff6b6b;
        }

        .upload-text {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 10px;
            color: #333;
        }

        .upload-hint {
            color: #666;
            font-size: 14px;
            margin-bottom: 20px;
        }

        .upload-formats {
            font-size: 12px;
            color: #999;
            background: rgba(255, 255, 255, 0.8);
            padding: 10px;
            border-radius: 8px;
            display: inline-block;
        }

        /* ================================================================
         * 🖼️ GRILLE PHOTOS
         * ================================================================ */
        .photos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .photo-card {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }

        .photo-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .photo-card.selectable {
            cursor: pointer;
            border: 3px solid transparent;
        }

        .photo-card.selected {
            border-color: #4ecdc4;
            transform: translateY(-5px) scale(1.02);
        }

        .photo-preview {
            width: 100%;
            height: 150px;
            background: #f8f9fa;
            position: relative;
            overflow: hidden;
        }

        .photo-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .photo-formats {
            position: absolute;
            top: 8px;
            right: 8px;
            display: flex;
            gap: 4px;
        }

        .format-badge {
            background: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 10px;
            font-weight: 600;
        }

        .photo-info {
            padding: 15px;
        }

        .photo-name {
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
            font-size: 14px;
        }

        .photo-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 12px;
            color: #666;
            margin-bottom: 10px;
        }

        .photo-actions {
            display: flex;
            gap: 8px;
        }

        .photo-actions .btn {
            padding: 6px 12px;
            font-size: 12px;
        }

        /* ================================================================
         * 💬 MESSAGES ET ALERTES
         * ================================================================ */
        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        /* ================================================================
         * 📱 RESPONSIVE
         * ================================================================ */
        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .photos-grid {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            }
            
            .upload-zone {
                padding: 30px 20px;
            }
        }

        /* ================================================================
         * 🎯 MODE SÉLECTION
         * ================================================================ */
        .selection-bar {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
            padding: 15px 25px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .selection-info {
            font-weight: 600;
        }

        .selection-actions {
            display: flex;
            gap: 10px;
        }

        .hidden {
            display: none;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header principal -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">📸</div>
                <div class="header-title">
                    <h1>Gestionnaire Photos</h1>
                    <p>Upload et conversion automatique avec FFmpeg</p>
                </div>
            </div>
            <div class="header-right">
                <div class="user-badge">
                    <?= htmlspecialchars($user['name']) ?> - <?= ucfirst($user['role']) ?>
                </div>
                <div class="quick-nav">
                    <a href="dashboard.php">🏠 Dashboard</a>
                    <a href="posts.php">📝 Posts</a>
                    <?php if ($selected_for_post): ?>
                        <a href="posts.php">← Retour Posts</a>
                    <?php endif; ?>
                </div>
            </div>
        </header>

        <!-- Statistiques -->
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-icon">📱</span>
                <div class="stat-value"><?= $stats['total_medias'] ?></div>
                <div class="stat-label">Photos Total</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📸</span>
                <div class="stat-value"><?= $stats['instagram_ready'] ?></div>
                <div class="stat-label">Instagram Ready</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📘</span>
                <div class="stat-value"><?= $stats['facebook_ready'] ?></div>
                <div class="stat-label">Facebook Ready</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📺</span>
                <div class="stat-value"><?= $stats['stories_ready'] ?></div>
                <div class="stat-label">Stories Ready</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">💾</span>
                <div class="stat-value"><?= formatFileSize($stats['total_size']) ?></div>
                <div class="stat-label">Espace Utilisé</div>
            </div>
        </div>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Mode sélection pour posts -->
        <?php if ($selected_for_post): ?>
            <div class="selection-bar">
                <div class="selection-info">
                    🎯 Mode sélection : Choisissez vos photos pour le post
                </div>
                <div class="selection-actions">
                    <button onclick="selectAllPhotos()" class="btn btn-secondary" style="padding: 8px 16px;">
                        ✅ Tout sélectionner
                    </button>
                    <button onclick="clearSelection()" class="btn btn-secondary" style="padding: 8px 16px;">
                        ❌ Tout désélectionner
                    </button>
                </div>
            </div>
            
            <form method="POST" id="selectionForm">
                <input type="hidden" name="post_action" value="<?= $_GET['post_action'] ?? 'new' ?>">
                <input type="hidden" name="post_id" value="<?= $_GET['post_id'] ?? '' ?>">
                <div id="selectedMedias"></div>
                <div style="margin-bottom: 20px;">
                    <button type="submit" name="select_medias" class="btn btn-success" id="confirmSelection" disabled>
                        ✅ Utiliser les photos sélectionnées
                    </button>
                </div>
            </form>
        <?php endif; ?>

        <!-- Upload de nouvelles photos -->
        <div class="content-card">
            <div class="card-header">
                <h2 class="card-title">📤 Upload Photos</h2>
                <div>
                    <span style="color: #666; font-size: 14px;">FFmpeg conversion automatique</span>
                </div>
            </div>
            
            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                <div class="upload-zone" id="uploadZone">
                    <input type="file" name="photos[]" multiple accept="image/*" style="display: none;" id="photosInput">
                    <div class="upload-icon">📷</div>
                    <div class="upload-text">Glissez vos photos ici</div>
                    <div class="upload-hint">ou cliquez pour parcourir vos fichiers</div>
                    <div class="upload-formats">
                        📱 JPG, PNG, WebP, HEIC • 📊 Max 50MB par photo • 🎯 Conversion auto tous réseaux
                    </div>
                </div>
                
                <div style="text-align: center;">
                    <button type="submit" name="upload_photos" class="btn btn-primary" id="uploadBtn" disabled>
                        🚀 Uploader et Convertir
                    </button>
                </div>
            </form>
        </div>

        <!-- Liste des photos -->
        <div class="content-card">
            <div class="card-header">
                <h2 class="card-title">🖼️ Vos Photos (<?= count($user_medias) ?>)</h2>
                <?php if (!empty($trash_medias)): ?>
                    <a href="?action=trash" class="btn btn-secondary">
                        🗑️ Corbeille (<?= count($trash_medias) ?>)
                    </a>
                <?php endif; ?>
            </div>
            
            <?php if (empty($user_medias)): ?>
                <div style="text-align: center; padding: 60px; color: #666;">
                    <div style="font-size: 64px; margin-bottom: 20px;">📸</div>
                    <h3>Aucune photo uploadée</h3>
                    <p style="margin: 15px 0;">Commencez par uploader vos premières photos !</p>
                </div>
            <?php else: ?>
                <div class="photos-grid">
                    <?php foreach ($user_medias as $media): ?>
                        <div class="photo-card <?= $selected_for_post ? 'selectable' : '' ?>" 
                             <?= $selected_for_post ? 'onclick="togglePhotoSelection(' . $media['id'] . ', this)"' : '' ?>>
                            
                            <div class="photo-preview">
                                <?php if ($media['instagram_path'] && file_exists($media['instagram_path'])): ?>
                                    <img src="<?= $media['instagram_path'] ?>" alt="<?= htmlspecialchars($media['original_name']) ?>">
                                <?php else: ?>
                                    <div style="display: flex; align-items: center; justify-content: center; height: 100%; background: #f8f9fa; color: #999;">
                                        📷 Conversion en cours...
                                    </div>
                                <?php endif; ?>
                                
                                <div class="photo-formats">
                                    <?php if ($media['instagram_path']): ?><span class="format-badge">📸</span><?php endif; ?>
                                    <?php if ($media['facebook_path']): ?><span class="format-badge">📘</span><?php endif; ?>
                                    <?php if ($media['stories_path']): ?><span class="format-badge">📺</span><?php endif; ?>
                                    <?php if ($media['twitter_path']): ?><span class="format-badge">🐦</span><?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="photo-info">
                                <div class="photo-name"><?= htmlspecialchars($media['original_name']) ?></div>
                                <div class="photo-meta">
                                    <span><?= $media['width'] ?>x<?= $media['height'] ?></span>
                                    <span><?= formatFileSize($media['file_size']) ?></span>
                                </div>
                                <div class="photo-meta">
                                    <span><?= formatDate($media['created_at']) ?></span>
                                </div>
                                
                                <?php if (!$selected_for_post): ?>
                                    <div class="photo-actions">
                                        <a href="?action=delete&id=<?= $media['id'] ?>" 
                                           onclick="return confirm('Déplacer vers la corbeille ?')" 
                                           class="btn btn-danger">
                                            🗑️ Supprimer
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Corbeille -->
        <?php if ($action === 'trash' && !empty($trash_medias)): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">🗑️ Corbeille (<?= count($trash_medias) ?>)</h2>
                    <a href="images.php" class="btn btn-secondary">← Retour</a>
                </div>
                
                <div class="photos-grid">
                    <?php foreach ($trash_medias as $media): ?>
                        <div class="photo-card" style="opacity: 0.7;">
                            <div class="photo-preview">
                                <?php if ($media['instagram_path'] && file_exists($media['instagram_path'])): ?>
                                    <img src="<?= $media['instagram_path'] ?>" alt="<?= htmlspecialchars($media['original_name']) ?>">
                                <?php endif; ?>
                            </div>
                            
                            <div class="photo-info">
                                <div class="photo-name"><?= htmlspecialchars($media['original_name']) ?></div>
                                <div class="photo-meta">
                                    <span><?= formatDate($media['created_at']) ?></span>
                                </div>
                                
                                <div class="photo-actions">
                                    <a href="?action=restore&id=<?= $media['id'] ?>" class="btn btn-success">
                                        🔄 Restaurer
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        /**
         * ================================================================
         * 🎯 JAVASCRIPT GESTIONNAIRE PHOTOS - NO JSON INSIDE !
         * ================================================================
         */

        let selectedPhotos = [];

        // ================================================================
        // 📤 GESTION UPLOAD
        // ================================================================
        document.addEventListener('DOMContentLoaded', function() {
            const uploadZone = document.getElementById('uploadZone');
            const fileInput = document.getElementById('photosInput');
            const uploadBtn = document.getElementById('uploadBtn');
            
            // Clic sur la zone d'upload
            uploadZone.addEventListener('click', () => {
                fileInput.click();
            });

            // Changement de fichiers
            fileInput.addEventListener('change', function() {
                updateUploadStatus();
            });

            // Drag & Drop
            ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                uploadZone.addEventListener(eventName, preventDefaults, false);
            });

            ['dragenter', 'dragover'].forEach(eventName => {
                uploadZone.addEventListener(eventName, () => {
                    uploadZone.classList.add('dragover');
                }, false);
            });

            ['dragleave', 'drop'].forEach(eventName => {
                uploadZone.addEventListener(eventName, () => {
                    uploadZone.classList.remove('dragover');
                }, false);
            });

            uploadZone.addEventListener('drop', function(e) {
                const files = e.dataTransfer.files;
                fileInput.files = files;
                updateUploadStatus();
            });
        });

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        function updateUploadStatus() {
            const fileInput = document.getElementById('photosInput');
            const uploadZone = document.getElementById('uploadZone');
            const uploadBtn = document.getElementById('uploadBtn');
            
            if (fileInput.files.length > 0) {
                const count = fileInput.files.length;
                uploadZone.innerHTML = `
                    <div class="upload-icon">✅</div>
                    <div class="upload-text">${count} photo${count > 1 ? 's' : ''} sélectionnée${count > 1 ? 's' : ''}</div>
                    <div class="upload-hint">Cliquez pour en ajouter d'autres ou validez l'upload</div>
                    <div class="upload-formats">
                        🚀 Conversion automatique Instagram + Facebook + Stories + Twitter
                    </div>
                `;
                uploadBtn.disabled = false;
            } else {
                uploadBtn.disabled = true;
            }
        }

        // ================================================================
        // 🎯 SÉLECTION PHOTOS POUR POSTS
        // ================================================================
        function togglePhotoSelection(mediaId, cardElement) {
            const index = selectedPhotos.indexOf(mediaId);
            
            if (index > -1) {
                // Désélectionner
                selectedPhotos.splice(index, 1);
                cardElement.classList.remove('selected');
            } else {
                // Sélectionner
                selectedPhotos.push(mediaId);
                cardElement.classList.add('selected');
            }
            
            updateSelectionForm();
        }

        function selectAllPhotos() {
            selectedPhotos = [];
            document.querySelectorAll('.photo-card.selectable').forEach(card => {
                card.classList.add('selected');
                // Extraire l'ID depuis l'onclick
                const onclick = card.getAttribute('onclick');
                const match = onclick.match(/togglePhotoSelection\((\d+),/);
                if (match) {
                    selectedPhotos.push(parseInt(match[1]));
                }
            });
            updateSelectionForm();
        }

        function clearSelection() {
            selectedPhotos = [];
            document.querySelectorAll('.photo-card.selected').forEach(card => {
                card.classList.remove('selected');
            });
            updateSelectionForm();
        }

        function updateSelectionForm() {
            const selectedMediasDiv = document.getElementById('selectedMedias');
            const confirmBtn = document.getElementById('confirmSelection');
            
            // Vider les anciens inputs
            selectedMediasDiv.innerHTML = '';
            
            // Ajouter les nouveaux
            selectedPhotos.forEach(mediaId => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'media_ids[]';
                input.value = mediaId;
                selectedMediasDiv.appendChild(input);
            });
            
            // Activer/désactiver le bouton
            confirmBtn.disabled = selectedPhotos.length === 0;
            
            // Mettre à jour le texte du bouton
            if (selectedPhotos.length > 0) {
                confirmBtn.textContent = `✅ Utiliser ${selectedPhotos.length} photo${selectedPhotos.length > 1 ? 's' : ''} sélectionnée${selectedPhotos.length > 1 ? 's' : ''}`;
            } else {
                confirmBtn.textContent = '✅ Utiliser les photos sélectionnées';
            }
        }

        console.log('📸 Gestionnaire Photos Ready - FFmpeg Power by Papy 77 !');
        console.log('🚀 Upload multi-format supporté');
        console.log('🎯 Conversion automatique tous réseaux');
        console.log('🗑️ Système corbeille avec undo');
        console.log('📱 Optimisé mobile et tablette');
    </script>
</body>
</html>