import React, { useState, useEffect } from 'react';
import { useAuth } from '../../context/AuthContext';
import ApiService from '../../services/api';

const Dashboard = () => {
  const { user } = useAuth();
  const [stats, setStats] = useState({
    totalPosts: 0,
    platforms: 0,
    thisMonth: 0,
    lastActivity: null
  });
  const [recentPosts, setRecentPosts] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadDashboardData();
  }, []);

  const loadDashboardData = async () => {
    try {
      setLoading(true);
      
      // Simuler des données pour l'instant
      // Plus tard on connectera à l'API
      const mockStats = {
        totalPosts: 24,
        platforms: 3,
        thisMonth: 8,
        lastActivity: new Date().toLocaleDateString()
      };
      
      const mockPosts = [
        { id: 1, platform: 'instagram', content: 'Mon premier post Instagram', date: '2025-07-11' },
        { id: 2, platform: 'facebook', content: 'Post Facebook de test', date: '2025-07-10' },
        { id: 3, platform: 'linkedin', content: 'Article LinkedIn professionnel', date: '2025-07-09' }
      ];

      setStats(mockStats);
      setRecentPosts(mockPosts);
    } catch (error) {
      console.error('Erreur chargement dashboard:', error);
    } finally {
      setLoading(false);
    }
  };

  const getPlatformEmoji = (platform) => {
    const emojis = {
      instagram: '📸',
      facebook: '👥',
      linkedin: '💼'
    };
    return emojis[platform] || '📱';
  };

  const getPlatformColor = (platform) => {
    const colors = {
      instagram: '#E4405F',
      facebook: '#1877F2',
      linkedin: '#0A66C2'
    };
    return colors[platform] || '#6b7280';
  };

  if (loading) {
    return (
      <div style={{ 
        display: 'flex', 
        alignItems: 'center', 
        justifyContent: 'center', 
        height: '400px' 
      }}>
        <div style={{ fontSize: '2rem' }}>⏳ Chargement...</div>
      </div>
    );
  }

  return (
    <div style={{ padding: '2rem', backgroundColor: '#f9fafb', minHeight: '100vh' }}>
      {/* Header */}
      <div style={{
        backgroundColor: 'white',
        borderRadius: '10px',
        padding: '2rem',
        marginBottom: '2rem',
        boxShadow: '0 1px 3px rgba(0,0,0,0.1)'
      }}>
        <h1 style={{ fontSize: '2rem', fontWeight: 'bold', color: '#111827', marginBottom: '0.5rem' }}>
          Bonjour {user?.first_name || user?.username} ! 👋
        </h1>
        <p style={{ color: '#6b7280' }}>
          Voici un résumé de votre activité sur les réseaux sociaux
        </p>
      </div>

      {/* Stats Cards */}
      <div style={{
        display: 'grid',
        gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))',
        gap: '1.5rem',
        marginBottom: '2rem'
      }}>
        <div style={{
          backgroundColor: 'white',
          borderRadius: '10px',
          padding: '1.5rem',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          border: '2px solid #dbeafe'
        }}>
          <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
            <div>
              <p style={{ fontSize: '0.875rem', color: '#6b7280', marginBottom: '0.5rem' }}>
                Total Posts
              </p>
              <p style={{ fontSize: '2rem', fontWeight: 'bold', color: '#111827' }}>
                {stats.totalPosts}
              </p>
            </div>
            <div style={{ fontSize: '2rem' }}>📊</div>
          </div>
        </div>

        <div style={{
          backgroundColor: 'white',
          borderRadius: '10px',
          padding: '1.5rem',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          border: '2px solid #dcfce7'
        }}>
          <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
            <div>
              <p style={{ fontSize: '0.875rem', color: '#6b7280', marginBottom: '0.5rem' }}>
                Plateformes
              </p>
              <p style={{ fontSize: '2rem', fontWeight: 'bold', color: '#111827' }}>
                {stats.platforms}
              </p>
            </div>
            <div style={{ fontSize: '2rem' }}>🌐</div>
          </div>
        </div>

        <div style={{
          backgroundColor: 'white',
          borderRadius: '10px',
          padding: '1.5rem',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          border: '2px solid #fef3c7'
        }}>
          <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
            <div>
              <p style={{ fontSize: '0.875rem', color: '#6b7280', marginBottom: '0.5rem' }}>
                Ce mois
              </p>
              <p style={{ fontSize: '2rem', fontWeight: 'bold', color: '#111827' }}>
                {stats.thisMonth}
              </p>
            </div>
            <div style={{ fontSize: '2rem' }}>📈</div>
          </div>
        </div>

        <div style={{
          backgroundColor: 'white',
          borderRadius: '10px',
          padding: '1.5rem',
          boxShadow: '0 1px 3px rgba(0,0,0,0.1)',
          border: '2px solid #fce7f3'
        }}>
          <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
            <div>
              <p style={{ fontSize: '0.875rem', color: '#6b7280', marginBottom: '0.5rem' }}>
                Dernière activité
              </p>
              <p style={{ fontSize: '1.2rem', fontWeight: 'bold', color: '#111827' }}>
                {stats.lastActivity}
              </p>
            </div>
            <div style={{ fontSize: '2rem' }}>📅</div>
          </div>
        </div>
      </div>

      {/* Posts récents */}
      <div style={{
        backgroundColor: 'white',
        borderRadius: '10px',
        padding: '2rem',
        boxShadow: '0 1px 3px rgba(0,0,0,0.1)'
      }}>
        <h3 style={{ fontSize: '1.25rem', fontWeight: 'bold', color: '#111827', marginBottom: '1.5rem' }}>
          📝 Posts récents
        </h3>
        
        <div style={{ display: 'flex', flexDirection: 'column', gap: '1rem' }}>
          {recentPosts.map((post) => (
            <div
              key={post.id}
              style={{
                display: 'flex',
                alignItems: 'center',
                padding: '1rem',
                border: '1px solid #e5e7eb',
                borderRadius: '8px',
                transition: 'all 0.2s'
              }}
            >
              <div
                style={{
                  width: '40px',
                  height: '40px',
                  borderRadius: '50%',
                  backgroundColor: getPlatformColor(post.platform),
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                  marginRight: '1rem',
                  fontSize: '1.2rem'
                }}
              >
                {getPlatformEmoji(post.platform)}
              </div>
              
              <div style={{ flex: 1 }}>
                <p style={{ fontWeight: '500', color: '#111827', marginBottom: '0.25rem' }}>
                  {post.platform.charAt(0).toUpperCase() + post.platform.slice(1)}
                </p>
                <p style={{ color: '#6b7280', fontSize: '0.875rem' }}>
                  {post.content}
                </p>
              </div>
              
              <div style={{ color: '#9ca3af', fontSize: '0.875rem' }}>
                {post.date}
              </div>
            </div>
          ))}
          
          {recentPosts.length === 0 && (
            <div style={{ textAlign: 'center', padding: '2rem', color: '#6b7280' }}>
              <div style={{ fontSize: '3rem', marginBottom: '1rem' }}>📝</div>
              <p>Aucun post récent. Commencez par créer votre premier contenu !</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Dashboard;