<?php
/**
 * ================================================================
 * 🔧 SCRIPT FIX TABLES SMG - RÉPARATION RAPIDE
 * ================================================================
 * 
 * 🎯 Répare les problèmes détectés par Papy 77
 * 💪 Ajoute les colonnes manquantes
 * 🚀 Fix les fonctions qui plantent
 * ================================================================
 */

require_once 'config/db.php';

echo "<h1>🔧 Fix Tables SMG Studio v2.0</h1>";

// ================================================================
// 🔧 ÉTAPE 1 : FIX FONCTION table_exists QUI PLANTE
// ================================================================

echo "<h2>🔧 Étape 1 : Fix fonction table_exists</h2>";

function table_exists_fixed($pdo, $table_name) {
    try {
        // Version simple sans paramètres préparés
        $result = $pdo->query("SHOW TABLES LIKE '{$table_name}'");
        return $result->rowCount() > 0;
    } catch (Exception $e) {
        // Fallback : essayer directement
        try {
            $pdo->query("SELECT 1 FROM `{$table_name}` LIMIT 1");
            return true;
        } catch (Exception $e2) {
            return false;
        }
    }
}

// Test de la fonction fixée
$test_tables = ['users', 'posts', 'social_platforms'];
echo "<div style='background: #e7f3ff; padding: 15px; border-radius: 8px;'>";
echo "<h3>🧪 Test fonction fixée :</h3>";
foreach ($test_tables as $table) {
    $exists = table_exists_fixed($pdo, $table);
    $status = $exists ? "✅ TRUE" : "❌ FALSE";
    echo "<p><strong>$table :</strong> $status</p>";
}
echo "</div>";

// ================================================================
// 🔧 ÉTAPE 2 : AJOUTER COLONNES MANQUANTES
// ================================================================

echo "<h2>🔧 Étape 2 : Ajout colonnes manquantes</h2>";

$fixes_applied = [];
$errors = [];

try {
    // Fix 1 : Ajouter colonne 'type' à la table posts
    echo "<p>🔄 Vérification colonne 'type' dans table posts...</p>";
    
    $stmt = $pdo->query("SHOW COLUMNS FROM posts LIKE 'type'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE posts ADD COLUMN `type` enum('text','image','video','carousel','story') DEFAULT 'text' AFTER `statut`");
        $fixes_applied[] = "✅ Colonne 'type' ajoutée à table posts";
    } else {
        $fixes_applied[] = "✅ Colonne 'type' déjà présente";
    }
    
    // Fix 2 : Vérifier structure complète de la table posts
    echo "<p>🔄 Vérification structure complète table posts...</p>";
    
    // D'abord voir toutes les colonnes existantes
    $stmt = $pdo->query("SHOW COLUMNS FROM posts");
    $existing_columns = $stmt->fetchAll();
    
    $column_names = [];
    foreach ($existing_columns as $col) {
        $column_names[] = $col['Field'];
    }
    
    echo "<p>📋 Colonnes actuelles : " . implode(', ', $column_names) . "</p>";
    
    // Fix colonnes manquantes selon le vrai schéma
    $missing_columns = [
        'priority' => "ADD COLUMN `priority` int DEFAULT '1'",
        'published_at' => "ADD COLUMN `published_at` timestamp NULL DEFAULT NULL",
        'multi_network' => "ADD COLUMN `multi_network` tinyint(1) DEFAULT '0'",
        'primary_network' => "ADD COLUMN `primary_network` varchar(50) DEFAULT NULL",
        'max_photos' => "ADD COLUMN `max_photos` int DEFAULT '5'",
        'max_videos' => "ADD COLUMN `max_videos` int DEFAULT '1'",
        'link_url' => "ADD COLUMN `link_url` varchar(500) DEFAULT NULL"
    ];
    
    // Fix client_id si elle existe mais sans défaut
    if (in_array('client_id', $column_names)) {
        try {
            $pdo->exec("ALTER TABLE posts MODIFY COLUMN `client_id` int DEFAULT NULL");
            $fixes_applied[] = "✅ Colonne 'client_id' rendue nullable";
        } catch (Exception $e) {
            // Ignorer si déjà OK
        }
    }
    
    foreach ($missing_columns as $column => $alter_sql) {
        $stmt = $pdo->query("SHOW COLUMNS FROM posts LIKE '{$column}'");
        if ($stmt->rowCount() == 0) {
            $pdo->exec("ALTER TABLE posts {$alter_sql}");
            $fixes_applied[] = "✅ Colonne '{$column}' ajoutée";
        }
    }
    
    // Fix 3 : Vérifier table social_platforms existe et a des données
    if (table_exists_fixed($pdo, 'social_platforms')) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM social_platforms");
        $count = $stmt->fetchColumn();
        
        if ($count == 0) {
            $pdo->exec("
                INSERT IGNORE INTO social_platforms (name, display_name, icon, max_chars, is_active) VALUES
                ('instagram', 'Instagram', '📸', 2200, 1),
                ('facebook', 'Facebook', '📘', 63206, 1),
                ('twitter', 'Twitter/X', '🐦', 280, 1),
                ('linkedin', 'LinkedIn', '💼', 3000, 1),
                ('tiktok', 'TikTok', '🎵', 150, 1)
            ");
            $fixes_applied[] = "✅ Données social_platforms ajoutées";
        }
    }
    
    // Fix 4 : Corriger les foreign keys si nécessaire
    try {
        $pdo->exec("
            ALTER TABLE post_networks 
            ADD CONSTRAINT fk_post_networks_platform 
            FOREIGN KEY (network) REFERENCES social_platforms(name) 
            ON DELETE CASCADE
        ");
        $fixes_applied[] = "✅ Foreign key post_networks ajoutée";
    } catch (Exception $e) {
        // Déjà présente
    }
    
} catch (Exception $e) {
    $errors[] = "❌ Erreur fix : " . $e->getMessage();
}

// ================================================================
// 🔧 ÉTAPE 3 : CRÉATION DONNÉES DE TEST
// ================================================================

echo "<h2>🔧 Étape 3 : Création données de test</h2>";

try {
    // Vérifier si user test existe
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
    $stmt->execute(['test@smg.fr']);
    $test_user = $stmt->fetch();
    
    if (!$test_user) {
        // Créer user test
        $stmt = $pdo->prepare("
            INSERT INTO users (name, email, password, role, status, email_verified) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            'Papy 77 Test', 
            'test@smg.fr', 
            password_hash('test123', PASSWORD_DEFAULT),
            'admin',
            'active',
            1
        ]);
        $test_user_id = $pdo->lastInsertId();
        $fixes_applied[] = "✅ User test créé : test@smg.fr / test123";
    } else {
        $test_user_id = $test_user['id'];
        $fixes_applied[] = "✅ User test existe déjà";
    }
    
    // Créer post de test AVEC client_id si nécessaire
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM posts WHERE user_id = ?");
    $stmt->execute([$test_user_id]);
    $post_count = $stmt->fetchColumn();
    
    if ($post_count == 0) {
        // Vérifier si client_id existe dans la structure
        $stmt = $pdo->query("SHOW COLUMNS FROM posts");
        $columns = $stmt->fetchAll();
        $has_client_id = false;
        
        foreach ($columns as $col) {
            if ($col['Field'] === 'client_id') {
                $has_client_id = true;
                break;
            }
        }
        
        if ($has_client_id) {
            // Version avec client_id (mettre NULL ou créer un client par défaut)
            $stmt = $pdo->prepare("
                INSERT INTO posts (user_id, client_id, titre, contenu, hashtags, statut, type) 
                VALUES (?, NULL, ?, ?, ?, ?, ?)
            ");
        } else {
            // Version sans client_id
            $stmt = $pdo->prepare("
                INSERT INTO posts (user_id, titre, contenu, hashtags, statut, type) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
        }
        
        $post_data = [
            $test_user_id,
            'Post test SMG Studio FIXED',
            'Ceci est un post de test CORRIGÉ pour vérifier que le dashboard fonctionne.',
            '#test #smg #fixed',
            'draft',
            'text'
        ];
        
        if (!$has_client_id) {
            array_splice($post_data, 1, 1); // Retirer client_id si pas besoin
        }
        
        $stmt->execute($post_data);
        $test_post_id = $pdo->lastInsertId();
        
        // Associer aux réseaux
        $networks = ['instagram', 'facebook', 'linkedin'];
        foreach ($networks as $network) {
            $stmt = $pdo->prepare("
                INSERT IGNORE INTO post_networks (post_id, network, is_active) 
                VALUES (?, ?, 1)
            ");
            $stmt->execute([$test_post_id, $network]);
        }
        
        $fixes_applied[] = "✅ Post de test créé avec 3 réseaux";
    } else {
        $fixes_applied[] = "✅ Posts existants trouvés : $post_count";
    }
    
} catch (Exception $e) {
    $errors[] = "❌ Erreur création test : " . $e->getMessage();
}

// ================================================================
// 📊 RAPPORT FINAL
// ================================================================

echo "<div style='background: #d4edda; padding: 20px; border-radius: 10px; margin: 20px 0;'>";
echo "<h3>✅ Corrections appliquées :</h3>";
foreach ($fixes_applied as $fix) {
    echo "<p>$fix</p>";
}
echo "</div>";

if (!empty($errors)) {
    echo "<div style='background: #f8d7da; padding: 20px; border-radius: 10px; margin: 20px 0;'>";
    echo "<h3>❌ Erreurs rencontrées :</h3>";
    foreach ($errors as $error) {
        echo "<p>$error</p>";
    }
    echo "</div>";
}

// ================================================================
// 🧪 TEST FINAL
// ================================================================

echo "<h2>🧪 Test Final - Simulation Dashboard</h2>";

try {
    // Test 1 : Compter users actifs
    $stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'");
    $active_users = $stmt->fetchColumn();
    
    // Test 2 : Compter posts
    $stmt = $pdo->query("SELECT COUNT(*) FROM posts");
    $total_posts = $stmt->fetchColumn();
    
    // Test 3 : Compter réseaux
    $stmt = $pdo->query("SELECT COUNT(*) FROM social_platforms WHERE is_active = 1");
    $active_networks = $stmt->fetchColumn();
    
    // Test 4 : Compter associations post-réseau
    $stmt = $pdo->query("SELECT COUNT(*) FROM post_networks");
    $post_networks = $stmt->fetchColumn();
    
    echo "<div style='background: #e7f3ff; padding: 20px; border-radius: 10px;'>";
    echo "<h3>📊 État final de la base :</h3>";
    echo "<p>👥 Users actifs : <strong>$active_users</strong></p>";
    echo "<p>📝 Posts total : <strong>$total_posts</strong></p>";
    echo "<p>🌐 Réseaux actifs : <strong>$active_networks</strong></p>";
    echo "<p>🔗 Associations post-réseau : <strong>$post_networks</strong></p>";
    echo "</div>";
    
    if ($total_posts > 0 && $active_networks > 0) {
        echo "<div style='background: #d4edda; padding: 15px; border-radius: 8px; margin: 10px 0; text-align: center;'>";
        echo "<h3>🎉 SUCCESS ! Base réparée</h3>";
        echo "<a href='dashboard.php' style='display: inline-block; padding: 15px 30px; background: #28a745; color: white; text-decoration: none; border-radius: 8px; font-weight: bold;'>📊 TESTER LE DASHBOARD</a>";
        echo "</div>";
    }
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 8px;'>";
    echo "❌ Erreur test final : " . $e->getMessage();
    echo "</div>";
}

echo "<p style='text-align: center; margin-top: 30px; color: #666;'>";
echo "🔧 Fix SMG Studio v2.0 - " . date('d/m/Y H:i:s') . " - by Papy 77";
echo "</p>";

?>