<?php
/**
 * ================================================================
 * 🔧 SMG STUDIO v2.0 - DIAGNOSTIC & FIX BASE DE DONNÉES IA
 * ================================================================
 * 
 * 🎯 DIAGNOSTIC ET CORRECTION AUTOMATIQUE :
 * ✅ Vérification structure tables IA
 * ✅ Correction liaisons clés API ↔ utilisateurs
 * ✅ Création profils manquants
 * ✅ Validation clés OpenAI
 * ✅ Réparation quotas
 * ✅ Nettoyage données corrompues
 * 
 * 💡 BY PAPY 77 - RÉPARATION AUTOMATIQUE SMG STUDIO TERMINÉE
 * ================================================================
 */

// Configuration base de données
$host = 'localhost';
$dbname = 'laloedata';
$username = 'laloedata';
$password = 'PiL256Mx78!';
$charset = 'utf8mb4';

$dsn = "mysql:host=$host;dbname=$dbname;charset=$charset";
$options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
];

try {
    $pdo = new PDO($dsn, $username, $password, $options);
    echo "✅ Connexion base de données OK\n";
} catch (PDOException $e) {
    die("❌ Erreur de connexion : " . $e->getMessage() . "\n");
}

echo "\n🔧 DÉBUT DIAGNOSTIC SMG STUDIO IA COMPLET\n";
echo "==========================================\n\n";

// ================================================================
// 🔍 PHASE 1: DIAGNOSTIC COMPLET
// ================================================================

echo "📋 PHASE 1: DIAGNOSTIC COMPLET\n";
echo "-------------------------------\n";

// 1. Vérification utilisateurs actifs
echo "1️⃣ Vérification utilisateurs actifs...\n";
$stmt = $pdo->query("SELECT id, name, email, status FROM users WHERE status = 'active' ORDER BY id");
$active_users = $stmt->fetchAll();

echo "   Utilisateurs actifs trouvés: " . count($active_users) . "\n";
foreach ($active_users as $user) {
    echo "   - User {$user['id']}: {$user['name']} ({$user['email']})\n";
}

// 2. Vérification structure tables IA
echo "\n2️⃣ Vérification structure tables IA...\n";
$required_tables = [
    'user_ai_profiles' => 'Profils IA utilisateurs',
    'api_keys' => 'Clés API',
    'ai_usage_logs' => 'Logs utilisation IA',
    'user_ai_quotas' => 'Quotas IA',
    'ai_providers_catalog' => 'Catalogue providers',
    'ai_content_templates' => 'Templates de contenu'
];

$missing_tables = [];
foreach ($required_tables as $table => $description) {
    $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
    if ($stmt->rowCount() == 0) {
        $missing_tables[] = $table;
        echo "   ❌ Table manquante: $table ($description)\n";
    } else {
        echo "   ✅ Table OK: $table\n";
    }
}

// 3. Vérification profils IA existants
echo "\n3️⃣ Vérification profils IA existants...\n";
$stmt = $pdo->query("
    SELECT uap.id, uap.user_id, uap.ai_provider, uap.display_name, uap.is_active, uap.is_default, uap.api_key_id,
           u.name as user_name, ak.encrypted_key
    FROM user_ai_profiles uap
    LEFT JOIN users u ON uap.user_id = u.id
    LEFT JOIN api_keys ak ON uap.api_key_id = ak.id
    ORDER BY uap.user_id, uap.ai_provider
");
$existing_profiles = $stmt->fetchAll();

echo "   Profils IA trouvés: " . count($existing_profiles) . "\n";
$profiles_by_user = [];
foreach ($existing_profiles as $profile) {
    $profiles_by_user[$profile['user_id']][] = $profile;
    $key_status = $profile['encrypted_key'] ? '🔑 Clé OK' : '❌ Pas de clé';
    $active_status = $profile['is_active'] ? '✅ Actif' : '❌ Inactif';
    echo "   - Profil {$profile['id']}: User {$profile['user_id']} ({$profile['user_name']}) | {$profile['ai_provider']} | {$profile['display_name']} | $key_status | $active_status\n";
}

// 4. Vérification clés API
echo "\n4️⃣ Vérification clés API...\n";
$stmt = $pdo->query("
    SELECT ak.id, ak.user_id, ak.provider, ak.key_name, ak.encrypted_key, ak.is_active,
           u.name as user_name
    FROM api_keys ak
    LEFT JOIN users u ON ak.user_id = u.id
    ORDER BY ak.user_id, ak.provider
");
$existing_keys = $stmt->fetchAll();

echo "   Clés API trouvées: " . count($existing_keys) . "\n";
$keys_by_user = [];
foreach ($existing_keys as $key) {
    $keys_by_user[$key['user_id']][] = $key;
    $key_valid = (strlen($key['encrypted_key']) > 20) ? '✅ Valide' : '❌ Invalide';
    $active_status = $key['is_active'] ? '✅ Active' : '❌ Inactive';
    echo "   - Clé {$key['id']}: User {$key['user_id']} ({$key['user_name']}) | {$key['provider']} | {$key['key_name']} | $key_valid | $active_status\n";
}

// 5. Vérification quotas IA
echo "\n5️⃣ Vérification quotas IA...\n";
$stmt = $pdo->query("
    SELECT uaq.*, u.name as user_name
    FROM user_ai_quotas uaq
    LEFT JOIN users u ON uaq.user_id = u.id
    WHERE uaq.is_active = 1
    ORDER BY uaq.user_id, uaq.quota_period
");
$existing_quotas = $stmt->fetchAll();

echo "   Quotas actifs trouvés: " . count($existing_quotas) . "\n";
foreach ($existing_quotas as $quota) {
    $usage_percent = $quota['tokens_limit'] > 0 ? round(($quota['tokens_used'] / $quota['tokens_limit']) * 100, 1) : 0;
    echo "   - Quota {$quota['id']}: User {$quota['user_id']} ({$quota['user_name']}) | {$quota['quota_period']} | {$usage_percent}% utilisé\n";
}

// 6. Identification des problèmes
echo "\n6️⃣ Identification des problèmes...\n";
$problems = [];

foreach ($active_users as $user) {
    $user_id = $user['id'];
    $user_profiles = $profiles_by_user[$user_id] ?? [];
    $user_keys = $keys_by_user[$user_id] ?? [];
    
    echo "   Analyse User {$user_id} ({$user['name']}):\n";
    
    // Problème: Aucun profil IA
    if (empty($user_profiles)) {
        $problems[] = [
            'type' => 'no_profiles',
            'user_id' => $user_id,
            'description' => "Aucun profil IA pour l'utilisateur {$user['name']}"
        ];
        echo "     ❌ Aucun profil IA configuré\n";
    } else {
        echo "     ✅ " . count($user_profiles) . " profil(s) IA trouvé(s)\n";
        
        // Vérifier qu'il y a au moins un profil par défaut
        $has_default = false;
        foreach ($user_profiles as $profile) {
            if ($profile['is_default']) {
                $has_default = true;
                break;
            }
        }
        
        if (!$has_default) {
            $problems[] = [
                'type' => 'no_default_profile',
                'user_id' => $user_id,
                'description' => "Aucun profil IA par défaut pour {$user['name']}"
            ];
            echo "     ⚠️ Aucun profil par défaut défini\n";
        }
        
        // Vérifier chaque profil
        foreach ($user_profiles as $profile) {
            if (!$profile['is_active']) {
                $problems[] = [
                    'type' => 'inactive_profile',
                    'profile_id' => $profile['id'],
                    'user_id' => $user_id,
                    'description' => "Profil {$profile['display_name']} inactif"
                ];
                echo "     ⚠️ Profil {$profile['display_name']} est inactif\n";
            }
            
            if ($profile['ai_provider'] === 'openai' && !$profile['encrypted_key']) {
                $problems[] = [
                    'type' => 'missing_api_key',
                    'profile_id' => $profile['id'],
                    'user_id' => $user_id,
                    'provider' => 'openai',
                    'description' => "Clé API OpenAI manquante pour {$profile['display_name']}"
                ];
                echo "     ❌ Clé API OpenAI manquante pour {$profile['display_name']}\n";
            }
        }
    }
    
    // Problème: Clés API corrompues
    foreach ($user_keys as $key) {
        if ($key['encrypted_key'] === 'TA_CLE_API_BASE64_ICI' || 
            strlen($key['encrypted_key']) < 20) {
            $problems[] = [
                'type' => 'fake_api_key',
                'key_id' => $key['id'],
                'user_id' => $user_id,
                'provider' => $key['provider'],
                'description' => "Clé API factice/invalide pour {$key['provider']}"
            ];
            echo "     ❌ Clé API factice/invalide détectée pour {$key['provider']}\n";
        }
    }
    
    // Vérifier quotas manquants
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count FROM user_ai_quotas 
        WHERE user_id = ? AND is_active = 1 
        AND CURDATE() BETWEEN quota_start_date AND quota_end_date
    ");
    $stmt->execute([$user_id]);
    $active_quotas = $stmt->fetch()['count'];
    
    if ($active_quotas == 0) {
        $problems[] = [
            'type' => 'no_quota',
            'user_id' => $user_id,
            'description' => "Aucun quota IA actif pour {$user['name']}"
        ];
        echo "     ⚠️ Aucun quota IA actif\n";
    }
}

// Vérifier orphelins
echo "\n   Vérification données orphelines...\n";

// Profils sans utilisateur
$stmt = $pdo->query("
    SELECT uap.id, uap.display_name FROM user_ai_profiles uap
    LEFT JOIN users u ON uap.user_id = u.id
    WHERE u.id IS NULL
");
$orphan_profiles = $stmt->fetchAll();
foreach ($orphan_profiles as $orphan) {
    $problems[] = [
        'type' => 'orphan_profile',
        'profile_id' => $orphan['id'],
        'description' => "Profil orphelin: {$orphan['display_name']}"
    ];
    echo "     ❌ Profil orphelin détecté: {$orphan['display_name']}\n";
}

// Clés sans utilisateur
$stmt = $pdo->query("
    SELECT ak.id, ak.key_name FROM api_keys ak
    LEFT JOIN users u ON ak.user_id = u.id
    WHERE u.id IS NULL
");
$orphan_keys = $stmt->fetchAll();
foreach ($orphan_keys as $orphan) {
    $problems[] = [
        'type' => 'orphan_key',
        'key_id' => $orphan['id'],
        'description' => "Clé API orpheline: {$orphan['key_name']}"
    ];
    echo "     ❌ Clé API orpheline détectée: {$orphan['key_name']}\n";
}

echo "\n📊 RÉSUMÉ DIAGNOSTIC:\n";
echo "   Problèmes détectés: " . count($problems) . "\n";
if (empty($problems)) {
    echo "   🎉 Aucun problème détecté - Système OK !\n";
    exit;
}

foreach ($problems as $problem) {
    echo "   - {$problem['description']}\n";
}

// ================================================================
// 🔧 PHASE 2: CORRECTION AUTOMATIQUE
// ================================================================

echo "\n\n🔧 PHASE 2: CORRECTION AUTOMATIQUE\n";
echo "-----------------------------------\n";

$fixes_applied = 0;

foreach ($problems as $problem) {
    echo "Correction: {$problem['description']}\n";
    
    switch ($problem['type']) {
        case 'no_profiles':
            // Créer un profil SMG Free par défaut
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO user_ai_profiles 
                    (user_id, ai_provider, display_name, model_preference, max_tokens_per_request, temperature, is_active, is_default)
                    VALUES (?, 'smg_free', 'SMG Studio Gratuit', 'smg-free', 1000, 0.7, 1, 1)
                ");
                $stmt->execute([$problem['user_id']]);
                echo "  ✅ Profil SMG Free créé pour user {$problem['user_id']}\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur création profil: {$e->getMessage()}\n";
            }
            break;
            
        case 'no_default_profile':
            // Définir le premier profil actif comme défaut
            try {
                $stmt = $pdo->prepare("
                    SELECT id FROM user_ai_profiles 
                    WHERE user_id = ? AND is_active = 1 
                    ORDER BY ai_provider = 'smg_free' DESC, id ASC 
                    LIMIT 1
                ");
                $stmt->execute([$problem['user_id']]);
                $profile = $stmt->fetch();
                
                if ($profile) {
                    // Réinitialiser tous les défauts
                    $stmt = $pdo->prepare("UPDATE user_ai_profiles SET is_default = 0 WHERE user_id = ?");
                    $stmt->execute([$problem['user_id']]);
                    
                    // Définir le nouveau défaut
                    $stmt = $pdo->prepare("UPDATE user_ai_profiles SET is_default = 1 WHERE id = ?");
                    $stmt->execute([$profile['id']]);
                    
                    echo "  ✅ Profil {$profile['id']} défini comme défaut pour user {$problem['user_id']}\n";
                    $fixes_applied++;
                }
            } catch (Exception $e) {
                echo "  ❌ Erreur définition profil défaut: {$e->getMessage()}\n";
            }
            break;
            
        case 'inactive_profile':
            // Réactiver le profil
            try {
                $stmt = $pdo->prepare("UPDATE user_ai_profiles SET is_active = 1 WHERE id = ?");
                $stmt->execute([$problem['profile_id']]);
                echo "  ✅ Profil {$problem['profile_id']} réactivé\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur réactivation: {$e->getMessage()}\n";
            }
            break;
            
        case 'fake_api_key':
            // Supprimer la clé factice et déconnecter le profil
            try {
                // Déconnecter les profils de cette clé
                $stmt = $pdo->prepare("UPDATE user_ai_profiles SET api_key_id = NULL WHERE api_key_id = ?");
                $stmt->execute([$problem['key_id']]);
                
                // Supprimer la clé factice
                $stmt = $pdo->prepare("DELETE FROM api_keys WHERE id = ?");
                $stmt->execute([$problem['key_id']]);
                
                echo "  ✅ Clé API factice supprimée (ID: {$problem['key_id']})\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur suppression clé: {$e->getMessage()}\n";
            }
            break;
            
        case 'missing_api_key':
            // Créer un profil SMG Free de remplacement si OpenAI sans clé
            try {
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) as count FROM user_ai_profiles 
                    WHERE user_id = ? AND ai_provider = 'smg_free'
                ");
                $stmt->execute([$problem['user_id']]);
                $has_smg_free = $stmt->fetch()['count'] > 0;
                
                if (!$has_smg_free) {
                    $stmt = $pdo->prepare("
                        INSERT INTO user_ai_profiles 
                        (user_id, ai_provider, display_name, model_preference, max_tokens_per_request, temperature, is_active, is_default)
                        VALUES (?, 'smg_free', 'SMG Studio Gratuit (Backup)', 'smg-free', 1000, 0.7, 1, 0)
                    ");
                    $stmt->execute([$problem['user_id']]);
                    echo "  ✅ Profil SMG Free de backup créé pour user {$problem['user_id']}\n";
                } else {
                    echo "  ℹ️ Profil SMG Free déjà existant pour user {$problem['user_id']}\n";
                }
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur création backup: {$e->getMessage()}\n";
            }
            break;
            
        case 'no_quota':
            // Créer un quota mensuel par défaut
            try {
                $start_date = date('Y-m-01');
                $end_date = date('Y-m-t');
                
                $stmt = $pdo->prepare("
                    INSERT INTO user_ai_quotas 
                    (user_id, quota_period, quota_start_date, quota_end_date, 
                     tokens_limit, cost_limit, requests_limit, auto_renew, is_active)
                    VALUES (?, 'monthly', ?, ?, 10000, 10.00, 100, 1, 1)
                ");
                $stmt->execute([$problem['user_id'], $start_date, $end_date]);
                echo "  ✅ Quota mensuel par défaut créé pour user {$problem['user_id']}\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur création quota: {$e->getMessage()}\n";
            }
            break;
            
        case 'orphan_profile':
            // Supprimer profil orphelin
            try {
                $stmt = $pdo->prepare("DELETE FROM user_ai_profiles WHERE id = ?");
                $stmt->execute([$problem['profile_id']]);
                echo "  ✅ Profil orphelin supprimé (ID: {$problem['profile_id']})\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur suppression profil orphelin: {$e->getMessage()}\n";
            }
            break;
            
        case 'orphan_key':
            // Supprimer clé orpheline
            try {
                $stmt = $pdo->prepare("DELETE FROM api_keys WHERE id = ?");
                $stmt->execute([$problem['key_id']]);
                echo "  ✅ Clé API orpheline supprimée (ID: {$problem['key_id']})\n";
                $fixes_applied++;
            } catch (Exception $e) {
                echo "  ❌ Erreur suppression clé orpheline: {$e->getMessage()}\n";
            }
            break;
    }
}

// ================================================================
// 🔧 PHASE 3: OPTIMISATIONS SUPPLÉMENTAIRES
// ================================================================

echo "\n\n🚀 PHASE 3: OPTIMISATIONS SUPPLÉMENTAIRES\n";
echo "------------------------------------------\n";

// 1. Nettoyage logs anciens (plus de 30 jours)
echo "1️⃣ Nettoyage logs anciens...\n";
try {
    $stmt = $pdo->prepare("DELETE FROM ai_usage_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
    $stmt->execute();
    $deleted_logs = $stmt->rowCount();
    echo "  ✅ $deleted_logs logs anciens supprimés\n";
} catch (Exception $e) {
    echo "  ❌ Erreur nettoyage logs: {$e->getMessage()}\n";
}

// 2. Mise à jour des quotas expirés
echo "\n2️⃣ Gestion quotas expirés...\n";
try {
    // Désactiver quotas expirés
    $stmt = $pdo->prepare("
        UPDATE user_ai_quotas 
        SET is_active = 0 
        WHERE quota_end_date < CURDATE() AND is_active = 1
    ");
    $stmt->execute();
    $expired_quotas = $stmt->rowCount();
    echo "  ✅ $expired_quotas quotas expirés désactivés\n";
    
    // Créer nouveaux quotas pour utilisateurs avec auto_renew
    $stmt = $pdo->query("
        SELECT DISTINCT user_id FROM user_ai_quotas 
        WHERE quota_end_date < CURDATE() AND auto_renew = 1
        AND user_id NOT IN (
            SELECT user_id FROM user_ai_quotas 
            WHERE is_active = 1 AND CURDATE() BETWEEN quota_start_date AND quota_end_date
        )
    ");
    $users_to_renew = $stmt->fetchAll();
    
    foreach ($users_to_renew as $user_renew) {
        $start_date = date('Y-m-01');
        $end_date = date('Y-m-t');
        
        $stmt = $pdo->prepare("
            INSERT INTO user_ai_quotas 
            (user_id, quota_period, quota_start_date, quota_end_date, 
             tokens_limit, cost_limit, requests_limit, auto_renew, is_active)
            VALUES (?, 'monthly', ?, ?, 10000, 10.00, 100, 1, 1)
        ");
        $stmt->execute([$user_renew['user_id'], $start_date, $end_date]);
        echo "  ✅ Quota auto-renouvelé pour user {$user_renew['user_id']}\n";
    }
} catch (Exception $e) {
    echo "  ❌ Erreur gestion quotas: {$e->getMessage()}\n";
}

// 3. Validation structure index
echo "\n3️⃣ Optimisation index base de données...\n";
$indexes_to_check = [
    'user_ai_profiles' => [
        'idx_user_active' => 'user_id, is_active',
        'idx_user_default' => 'user_id, is_default'
    ],
    'ai_usage_logs' => [
        'idx_user_date' => 'user_id, created_at',
        'idx_profile_usage' => 'ai_profile_id, created_at'
    ],
    'user_ai_quotas' => [
        'idx_user_active' => 'user_id, is_active',
        'idx_period_dates' => 'quota_period, quota_start_date, quota_end_date'
    ]
];

foreach ($indexes_to_check as $table => $indexes) {
    foreach ($indexes as $index_name => $columns) {
        try {
            $stmt = $pdo->query("SHOW INDEX FROM $table WHERE Key_name = '$index_name'");
            if ($stmt->rowCount() == 0) {
                $pdo->query("ALTER TABLE $table ADD INDEX $index_name ($columns)");
                echo "  ✅ Index $index_name créé sur $table\n";
            } else {
                echo "  ℹ️ Index $index_name déjà présent sur $table\n";
            }
        } catch (Exception $e) {
            echo "  ⚠️ Erreur index $index_name sur $table: {$e->getMessage()}\n";
        }
    }
}

// ================================================================
// 📊 PHASE 4: RAPPORT FINAL
// ================================================================

echo "\n\n📊 RAPPORT FINAL DE DIAGNOSTIC\n";
echo "==============================\n";

echo "🔧 Corrections appliquées: $fixes_applied\n";

// Stats finales
$stmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE status = 'active'");
$total_users = $stmt->fetch()['count'];

$stmt = $pdo->query("SELECT COUNT(*) as count FROM user_ai_profiles WHERE is_active = 1");
$total_profiles = $stmt->fetch()['count'];

$stmt = $pdo->query("SELECT COUNT(*) as count FROM api_keys WHERE is_active = 1");
$total_keys = $stmt->fetch()['count'];

$stmt = $pdo->query("
    SELECT COUNT(*) as count FROM user_ai_quotas 
    WHERE is_active = 1 AND CURDATE() BETWEEN quota_start_date AND quota_end_date
");
$active_quotas = $stmt->fetch()['count'];

echo "\n📈 STATISTIQUES SYSTÈME:\n";
echo "   👥 Utilisateurs actifs: $total_users\n";
echo "   🤖 Profils IA actifs: $total_profiles\n";
echo "   🔑 Clés API valides: $total_keys\n";
echo "   📊 Quotas actifs: $active_quotas\n";

// Validation finale
$all_users_have_profiles = true;
foreach ($active_users as $user) {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count FROM user_ai_profiles 
        WHERE user_id = ? AND is_active = 1
    ");
    $stmt->execute([$user['id']]);
    $user_profiles = $stmt->fetch()['count'];
    
    if ($user_profiles == 0) {
        $all_users_have_profiles = false;
        echo "   ❌ User {$user['id']} ({$user['name']}) n'a aucun profil IA actif\n";
    }
}

if ($all_users_have_profiles) {
    echo "\n🎉 SYSTÈME COMPLÈTEMENT OPÉRATIONNEL !\n";
    echo "   ✅ Tous les utilisateurs ont des profils IA configurés\n";
    echo "   ✅ Base de données optimisée\n";
    echo "   ✅ Navigation IA fonctionnelle\n";
} else {
    echo "\n⚠️ PROBLÈMES PERSISTANTS DÉTECTÉS\n";
    echo "   Relancez le diagnostic pour correction supplémentaire\n";
}

echo "\n💡 SMG STUDIO v2.0 - DIAGNOSTIC TERMINÉ BY PAPY 77\n";
echo "🚀 Votre système IA est maintenant optimisé !\n";

?>