<?php
/*
 * SMG Studio v2.0 - Configuration Workflow Destructif
 * Paramétrage comportements et règles du workflow
 * Path: /var/www/laloee-cours.fr/social-media-generator/config/workflow_config.php
 */

class SMGWorkflowConfig {
    
    // ==========================================================================
    // CONFIGURATION GÉNÉRALE
    // ==========================================================================
    
    const VERSION = '2.0';
    const WORKFLOW_TYPE = 'DESTRUCTIVE';
    const MAX_CLIPS_PER_USER = 1;
    
    // Durées et limites
    const AUTO_CLEANUP_DAYS = 30;        // Nettoyer anciens projets après X jours
    const MAX_ARCHIVE_SIZE = 5368709120; // 5GB max par utilisateur
    const SESSION_TIMEOUT = 3600;        // 1 heure
    
    // ==========================================================================
    // STRUCTURE DES ÉTAPES
    // ==========================================================================
    
    public static function getStagesConfig() {
        return [
            1 => [
                'name' => 'clipmaker',
                'display_name' => '🎬 Clipmaker',
                'description' => 'Upload rush → Découpe → Assemblage → Export clip',
                'folder' => 'active_clip/01_clipmaker/',
                'allowed_formats' => ['mp4', 'mov', 'avi', 'webm', 'mkv'],
                'max_file_size' => 2147483648, // 2GB
                'required_files' => ['video'],
                'auto_process' => false,
                'can_rollback_to' => null, // Pas de retour possible depuis clipmaker
                'next_stage_prep' => 'copy_video_to_audio',
                'tools_url' => 'modules/clipmaker_workflow_interface.html'
            ],
            
            2 => [
                'name' => 'audio_postprod',
                'display_name' => '🎵 Audio PostProd',
                'description' => 'Mixage → Mastering → Sync → Export audio final',
                'folder' => 'active_clip/02_audio_postprod/',
                'allowed_formats' => ['mp4', 'mov', 'wav', 'mp3', 'aac'],
                'max_file_size' => 1073741824, // 1GB
                'required_files' => ['video', 'audio'],
                'auto_process' => false,
                'can_rollback_to' => 1,
                'rollback_warning' => 'Le retour détruira définitivement le mixage audio !',
                'next_stage_prep' => 'copy_audio_to_emission',
                'tools_url' => 'modules/audio_postprod.html'
            ],
            
            3 => [
                'name' => 'emission_final',
                'display_name' => '📺 Émission Finale',
                'description' => 'Habillage → Encodage → Conformité → Export master',
                'folder' => 'active_clip/03_emission_final/',
                'allowed_formats' => ['mp4', 'mov', 'mxf', 'prores'],
                'max_file_size' => 5368709120, // 5GB
                'required_files' => ['master'],
                'auto_process' => false,
                'can_rollback_to' => 2,
                'rollback_warning' => 'Le retour détruira l\'habillage et l\'encodage final !',
                'next_stage_prep' => 'copy_master_to_social',
                'tools_url' => 'modules/emission_finale.html'
            ],
            
            4 => [
                'name' => 'social_ready',
                'display_name' => '📱 Social Ready',
                'description' => 'Formats multi-plateformes → Métadonnées → Export social',
                'folder' => 'active_clip/04_social_ready/',
                'allowed_formats' => ['mp4', 'mov', 'webm', 'gif'],
                'max_file_size' => 1073741824, // 1GB
                'required_files' => ['social_formats'],
                'auto_process' => false,
                'can_rollback_to' => 3,
                'rollback_warning' => 'Le retour détruira tous les formats sociaux !',
                'next_stage_prep' => 'prepare_archive',
                'tools_url' => 'modules/social_ready.html'
            ]
        ];
    }
    
    // ==========================================================================
    // RÈGLES DESTRUCTIVES
    // ==========================================================================
    
    public static function getDestructiveRules() {
        return [
            'rollback_confirmation_required' => true,
            'rollback_double_confirm' => true,
            'rollback_confirmation_phrase' => 'DETRUIRE',
            
            'new_clip_destroys_current' => true,
            'new_clip_warning_required' => true,
            
            'archive_cleans_workflow' => true,
            'archive_redirects_to_posts' => true,
            
            'no_intermediate_saves' => true,
            'no_project_versions' => true,
            'one_clip_per_user_only' => true,
            
            // Messages d'avertissement
            'warnings' => [
                'rollback' => '⚠️ RETOUR DESTRUCTIF ⚠️\n\nCeci supprimera DÉFINITIVEMENT toutes les modifications des étapes suivantes !\n\n💀 PERTE IRRÉVERSIBLE DE DONNÉES 💀',
                'new_clip' => '⚠️ ATTENTION ! Un clip est en cours.\n\nLe démarrer détruira définitivement le clip actuel !',
                'workflow_type' => '⚠️ WORKFLOW DESTRUCTIF : Tout retour en arrière supprime définitivement les modifications des étapes suivantes !'
            ]
        ];
    }
    
    // ==========================================================================
    // FORMATS ET CODECS PAR ÉTAPE
    // ==========================================================================
    
    public static function getFormatRules() {
        return [
            'stage_1_output' => [
                'format' => 'mp4',
                'codec_video' => 'h264',
                'codec_audio' => 'aac',
                'resolution' => '1920x1080',
                'fps' => 30,
                'bitrate_video' => '5000k',
                'bitrate_audio' => '192k'
            ],
            
            'stage_2_output' => [
                'format' => 'mp4',
                'codec_video' => 'h264',
                'codec_audio' => 'aac',
                'resolution' => '1920x1080',
                'fps' => 30,
                'bitrate_video' => '8000k',
                'bitrate_audio' => '320k',
                'audio_channels' => 'stereo',
                'audio_sample_rate' => 48000
            ],
            
            'stage_3_output' => [
                'format' => 'mp4',
                'codec_video' => 'h264',
                'codec_audio' => 'aac',
                'resolution' => '1920x1080',
                'fps' => 30,
                'bitrate_video' => '10000k',
                'bitrate_audio' => '320k',
                'profile' => 'high',
                'level' => '4.1'
            ],
            
            'stage_4_outputs' => [
                'youtube' => [
                    'format' => 'mp4',
                    'resolution' => '1920x1080',
                    'fps' => 30,
                    'bitrate_video' => '8000k'
                ],
                'instagram' => [
                    'format' => 'mp4', 
                    'resolution' => '1080x1080',
                    'fps' => 30,
                    'bitrate_video' => '3500k'
                ],
                'tiktok' => [
                    'format' => 'mp4',
                    'resolution' => '1080x1920', 
                    'fps' => 30,
                    'bitrate_video' => '4000k'
                ],
                'facebook' => [
                    'format' => 'mp4',
                    'resolution' => '1920x1080',
                    'fps' => 30,
                    'bitrate_video' => '6000k'
                ]
            ]
        ];
    }
    
    // ==========================================================================
    // AUTO-DÉTECTION DE FICHIERS
    // ==========================================================================
    
    public static function getAutoDetectionRules() {
        return [
            'scan_directories' => [
                'uploads/clipmaker/exports/',
                'uploads/audio/',
                'uploads/final/',
                'uploads/social/',
                'user_workspaces/{user}/temp/',
                'user_workspaces/{user}/imports/'
            ],
            
            'file_patterns' => [
                'stage_1' => [
                    'keywords' => ['clip', 'raw', 'rush', 'source', 'original'],
                    'extensions' => ['mp4', 'mov', 'avi'],
                    'min_size' => 1048576 // 1MB minimum
                ],
                'stage_2' => [
                    'keywords' => ['audio', 'mix', 'master', 'sound'],
                    'extensions' => ['mp4', 'wav', 'mp3'],
                    'min_size' => 1048576
                ],
                'stage_3' => [
                    'keywords' => ['final', 'master', 'emission', 'broadcast'],
                    'extensions' => ['mp4', 'mov', 'mxf'],
                    'min_size' => 10485760 // 10MB minimum
                ],
                'stage_4' => [
                    'keywords' => ['social', 'export', 'youtube', 'instagram', 'tiktok'],
                    'extensions' => ['mp4', 'mov', 'webm'],
                    'min_size' => 1048576
                ]
            ],
            
            'auto_assign' => false, // Demander confirmation avant assignation
            'max_files_per_scan' => 50,
            'ignore_hidden_files' => true,
            'ignore_temp_files' => true
        ];
    }
    
    // ==========================================================================
    // ARCHIVAGE ET NETTOYAGE
    // ==========================================================================
    
    public static function getArchiveRules() {
        return [
            'archive_structure' => [
                'metadata.json',        // Infos projet
                'stage_1/',            // Dossier clipmaker
                'stage_2/',            // Dossier audio
                'stage_3/',            // Dossier émission
                'stage_4/',            // Dossier social
                'logs/',               // Logs du workflow
                'exports/'             // Exports finaux
            ],
            
            'metadata_fields' => [
                'clip_name', 'user_id', 'archive_date', 'workflow_version',
                'stages_completed', 'total_duration', 'total_size', 
                'creation_date', 'completion_date', 'file_counts', 'formats_generated'
            ],
            
            'cleanup_after_archive' => true,
            'verify_archive_integrity' => true,
            'create_archive_index' => true,
            
            'auto_cleanup' => [
                'enabled' => true,
                'max_archives_per_user' => 100,
                'cleanup_older_than_days' => 365,
                'preserve_recent_count' => 10
            ]
        ];
    }
    
    // ==========================================================================
    // INTÉGRATIONS EXTERNES
    // ==========================================================================
    
    public static function getIntegrations() {
        return [
            'clipmaker' => [
                'enabled' => true,
                'url' => 'modules/clipmaker_workflow_interface.html',
                'params' => ['user', 'project', 'stage'],
                'return_url' => 'workflow_manager.php'
            ],
            
            'posts_manager' => [
                'enabled' => true,
                'url' => 'posts.php',
                'auto_redirect_after_archive' => true,
                'pass_latest_video' => true
            ],
            
            'ffmpeg' => [
                'enabled' => true,
                'path' => '/usr/bin/ffmpeg',
                'temp_dir' => '/tmp/smg_workflow/',
                'max_processes' => 4,
                'quality_presets' => [
                    'draft' => ['-crf', '28', '-preset', 'ultrafast'],
                    'preview' => ['-crf', '23', '-preset', 'fast'],
                    'production' => ['-crf', '20', '-preset', 'medium'],
                    'archive' => ['-crf', '18', '-preset', 'slow']
                ]
            ],
            
            'notifications' => [
                'enabled' => true,
                'email_on_completion' => false,
                'discord_webhook' => null,
                'slack_webhook' => null
            ]
        ];
    }
    
    // ==========================================================================
    // SÉCURITÉ ET PERMISSIONS
    // ==========================================================================
    
    public static function getSecurityRules() {
        return [
            'require_user_authentication' => true,
            'validate_user_permissions' => true,
            'log_all_actions' => true,
            'log_destructive_actions' => true,
            
            'rate_limiting' => [
                'enabled' => true,
                'max_requests_per_minute' => 60,
                'max_file_operations_per_hour' => 100,
                'cooldown_after_rollback_seconds' => 30
            ],
            
            'file_security' => [
                'validate_file_types' => true,
                'scan_for_malware' => false, // Nécessite ClamAV
                'quarantine_suspicious_files' => true,
                'max_filename_length' => 255,
                'forbidden_extensions' => ['exe', 'bat', 'sh', 'php', 'js']
            ],
            
            'workspace_isolation' => [
                'strict_user_separation' => true,
                'prevent_path_traversal' => true,
                'validate_workspace_access' => true,
                'auto_create_missing_directories' => true
            ]
        ];
    }
    
    // ==========================================================================
    // MÉTHODES UTILITAIRES
    // ==========================================================================
    
    public static function getStageConfig($stageNumber) {
        $stages = self::getStagesConfig();
        return $stages[$stageNumber] ?? null;
    }
    
    public static function isValidStage($stageNumber) {
        return $stageNumber >= 1 && $stageNumber <= 4;
    }
    
    public static function canRollbackFrom($fromStage, $toStage) {
        $stageConfig = self::getStageConfig($fromStage);
        return $stageConfig && $stageConfig['can_rollback_to'] === $toStage;
    }
    
    public static function getRollbackWarning($fromStage) {
        $stageConfig = self::getStageConfig($fromStage);
        return $stageConfig['rollback_warning'] ?? 'Cette action est irréversible !';
    }
    
    public static function getMaxFileSize($stage) {
        $stageConfig = self::getStageConfig($stage);
        return $stageConfig['max_file_size'] ?? 1073741824; // 1GB par défaut
    }
    
    public static function getAllowedFormats($stage) {
        $stageConfig = self::getStageConfig($stage);
        return $stageConfig['allowed_formats'] ?? ['mp4'];
    }
    
    public static function getStageUrl($stage, $userId, $projectName = null) {
        $stageConfig = self::getStageConfig($stage);
        if (!$stageConfig || !isset($stageConfig['tools_url'])) {
            return null;
        }
        
        $url = $stageConfig['tools_url'];
        $params = ['user' => $userId, 'stage' => $stage];
        
        if ($projectName) {
            $params['project'] = $projectName;
        }
        
        return $url . '?' . http_build_query($params);
    }
    
    public static function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= (1 << (10 * $pow));
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
    
    public static function validateFileName($filename) {
        $security = self::getSecurityRules();
        
        // Longueur
        if (strlen($filename) > $security['file_security']['max_filename_length']) {
            return false;
        }
        
        // Extensions interdites
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if (in_array($extension, $security['file_security']['forbidden_extensions'])) {
            return false;
        }
        
        // Caractères dangereux
        if (preg_match('/[<>:"|?*]/', $filename)) {
            return false;
        }
        
        return true;
    }
    
    public static function getWorkflowStats() {
        return [
            'version' => self::VERSION,
            'type' => self::WORKFLOW_TYPE,
            'stages_count' => count(self::getStagesConfig()),
            'max_clips_per_user' => self::MAX_CLIPS_PER_USER,
            'destructive_enabled' => true,
            'rollback_enabled' => true,
            'auto_detection_enabled' => true,
            'archive_enabled' => true
        ];
    }
}

// ==========================================================================
// CONFIGURATION ENVIRONNEMENT
// ==========================================================================

// Charger configuration selon environnement
$environment = $_SERVER['SMG_ENV'] ?? 'production';

switch ($environment) {
    case 'development':
        // Config développement - Plus permissive
        define('SMG_DEBUG', true);
        define('SMG_LOG_LEVEL', 'DEBUG');
        define('SMG_CLEANUP_DISABLED', true);
        break;
        
    case 'staging':
        // Config pré-production
        define('SMG_DEBUG', false);
        define('SMG_LOG_LEVEL', 'INFO');
        define('SMG_CLEANUP_DISABLED', false);
        break;
        
    case 'production':
    default:
        // Config production - Sécurisé
        define('SMG_DEBUG', false);
        define('SMG_LOG_LEVEL', 'WARNING');
        define('SMG_CLEANUP_DISABLED', false);
        define('SMG_STRICT_SECURITY', true);
        break;
}

// Constantes globales
define('SMG_WORKFLOW_VERSION', SMGWorkflowConfig::VERSION);
define('SMG_MAX_CLIPS_PER_USER', SMGWorkflowConfig::MAX_CLIPS_PER_USER);
define('SMG_AUTO_CLEANUP_DAYS', SMGWorkflowConfig::AUTO_CLEANUP_DAYS);
define('SMG_SESSION_TIMEOUT', SMGWorkflowConfig::SESSION_TIMEOUT);
?>