<?php
/*
 * SMG Studio v2.0 - Page Principale: Gestion des Utilisateurs CORRIGÉE
 * Path: /var/www/laloee-cours.fr/social-media-generator/clients.php
 */

session_start();

// 🔥 CORRECTION 1: Inclusion des bons fichiers
require_once 'config/db.php'; // ✅ Utilise le bon nom de fichier
require_once 'includes/phpmailer.php'; // ✅ Inclure PHPMailer

$message = '';
$error = '';

// 🔥 CORRECTION 2: Vérification connexion simplifiée
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// 🔥 CORRECTION 3: Récupérer l'utilisateur actuel de façon sécurisée
function getCurrentUser() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("SELECT id, name, email, role, status FROM users WHERE id = ? AND status = 'active'");
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            // Utilisateur supprimé ou désactivé
            session_destroy();
            header('Location: login.php?expired=1');
            exit;
        }
        
        return $user;
    } catch (Exception $e) {
        error_log("Erreur getCurrentUser: " . $e->getMessage());
        return null;
    }
}

$current_user = getCurrentUser();
if (!$current_user) {
    header('Location: login.php?error=1');
    exit;
}

$is_admin = ($current_user['role'] === 'admin');

// Actions CRUD (réservées aux admins)
$action = $_POST['action'] ?? $_GET['action'] ?? '';

if ($is_admin && $action) {
    switch ($action) {
        case 'create_user':
            $result = createUser($_POST);
            if (strpos($result, '✅') === 0) {
                $message = $result;
            } else {
                $error = $result;
            }
            break;
        case 'update_user':
            $result = updateUser($_POST);
            if (strpos($result, '✅') === 0) {
                $message = $result;
            } else {
                $error = $result;
            }
            break;
        case 'delete_user':
            $result = deleteUser($_POST['user_id']);
            if (strpos($result, '✅') === 0) {
                $message = $result;
            } else {
                $error = $result;
            }
            break;
        case 'reset_password':
            $result = resetUserPassword($_POST['user_id']);
            if (strpos($result, '✅') === 0) {
                $message = $result;
            } else {
                $error = $result;
            }
            break;
    }
}

// FONCTIONS CRUD CORRIGÉES

function createUser($data) {
    global $pdo;
    
    try {
        $email = trim($data['email']);
        $name = trim($data['name']);
        $role = $data['role'];
        
        // Validation des données
        if (empty($email) || empty($name) || empty($role)) {
            return "❌ Tous les champs sont requis";
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return "❌ Format d'email invalide";
        }
        
        $password = generateRandomPassword();
        $hashed_password = password_hash($password, PASSWORD_ARGON2ID);
        
        // Vérifier si email existe déjà
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            return "❌ Cet email existe déjà";
        }
        
        // Créer utilisateur
        $stmt = $pdo->prepare("
            INSERT INTO users (name, email, password, role, status, created_at) 
            VALUES (?, ?, ?, ?, 'active', NOW())
        ");
        $stmt->execute([$name, $email, $hashed_password, $role]);
        
        // Envoyer mot de passe par email
        $user_id = $pdo->lastInsertId();
        $emailResult = sendPasswordToUser($user_id, $password);
        
        if ($emailResult === true) {
            return "✅ Utilisateur créé et mot de passe envoyé par email";
        } else {
            return "✅ Utilisateur créé mais erreur envoi email: " . $emailResult;
        }
        
    } catch (Exception $e) {
        error_log("Erreur createUser: " . $e->getMessage());
        return "❌ Erreur lors de la création: " . $e->getMessage();
    }
}

function updateUser($data) {
    global $pdo;
    
    try {
        // Validation des données
        if (empty($data['name']) || empty($data['email']) || empty($data['user_id'])) {
            return "❌ Données manquantes";
        }
        
        if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            return "❌ Format d'email invalide";
        }
        
        // Vérifier que l'email n'existe pas déjà pour un autre utilisateur
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
        $stmt->execute([$data['email'], $data['user_id']]);
        if ($stmt->fetch()) {
            return "❌ Cet email est déjà utilisé par un autre utilisateur";
        }
        
        $stmt = $pdo->prepare("
            UPDATE users 
            SET name = ?, email = ?, role = ?, status = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $result = $stmt->execute([
            $data['name'], 
            $data['email'], 
            $data['role'], 
            $data['status'], 
            $data['user_id']
        ]);
        
        if ($result) {
            return "✅ Utilisateur mis à jour avec succès";
        } else {
            return "❌ Erreur lors de la mise à jour";
        }
        
    } catch (Exception $e) {
        error_log("Erreur updateUser: " . $e->getMessage());
        return "❌ Erreur: " . $e->getMessage();
    }
}

function deleteUser($user_id) {
    global $pdo, $current_user;
    
    try {
        // Empêcher auto-suppression
        if ($user_id == $current_user['id']) {
            return "❌ Vous ne pouvez pas supprimer votre propre compte";
        }
        
        // Vérifier que l'utilisateur existe
        $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        if (!$stmt->fetch()) {
            return "❌ Utilisateur introuvable";
        }
        
        $stmt = $pdo->prepare("UPDATE users SET status = 'deleted', updated_at = NOW() WHERE id = ?");
        $result = $stmt->execute([$user_id]);
        
        if ($result) {
            return "✅ Utilisateur supprimé avec succès";
        } else {
            return "❌ Erreur lors de la suppression";
        }
        
    } catch (Exception $e) {
        error_log("Erreur deleteUser: " . $e->getMessage());
        return "❌ Erreur: " . $e->getMessage();
    }
}

function resetUserPassword($user_id) {
    global $pdo;
    
    try {
        // Vérifier que l'utilisateur existe
        $stmt = $pdo->prepare("SELECT id, name, email FROM users WHERE id = ? AND status = 'active'");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return "❌ Utilisateur introuvable";
        }
        
        $new_password = generateRandomPassword();
        $hashed_password = password_hash($new_password, PASSWORD_ARGON2ID);
        
        $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?");
        $result = $stmt->execute([$hashed_password, $user_id]);
        
        if ($result) {
            $emailResult = sendPasswordToUser($user_id, $new_password);
            
            if ($emailResult === true) {
                return "✅ Nouveau mot de passe généré et envoyé par email";
            } else {
                return "✅ Mot de passe réinitialisé mais erreur envoi email: " . $emailResult;
            }
        } else {
            return "❌ Erreur lors de la réinitialisation";
        }
        
    } catch (Exception $e) {
        error_log("Erreur resetUserPassword: " . $e->getMessage());
        return "❌ Erreur: " . $e->getMessage();
    }
}

function sendPasswordToUser($user_id, $password) {
    global $pdo;
    
    try {
        // Récupérer info utilisateur
        $stmt = $pdo->prepare("SELECT name, email FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return "Utilisateur introuvable";
        }
        
        // Préparer l'email
        $subject = "🔐 Vos identifiants SMG Studio";
        $body = "
        <html>
        <head><meta charset='UTF-8'></head>
        <body style='font-family: Arial, sans-serif; color: #333;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='background: #667eea; color: white; padding: 20px; border-radius: 8px 8px 0 0; text-align: center;'>
                    <h2>🎬 SMG Studio v2.0</h2>
                </div>
                <div style='background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px;'>
                    <p>Bonjour <strong>" . htmlspecialchars($user['name']) . "</strong>,</p>
                    <p>Voici vos identifiants de connexion pour SMG Studio :</p>
                    <div style='background: white; padding: 20px; border-radius: 8px; margin: 20px 0; border-left: 4px solid #667eea;'>
                        <p><strong>📧 Email :</strong> " . htmlspecialchars($user['email']) . "</p>
                        <p><strong>🔐 Mot de passe :</strong> <code style='background:#e9ecef;padding:8px 12px;border-radius:4px;font-size:14px;'>" . htmlspecialchars($password) . "</code></p>
                    </div>
                    <div style='text-align: center; margin: 30px 0;'>
                        <a href='https://laloee-cours.fr/social-media-generator/login.php' 
                           style='background:#667eea;color:white;padding:15px 30px;text-decoration:none;border-radius:8px;display:inline-block;'>
                            🚀 Se connecter maintenant
                        </a>
                    </div>
                    <p><strong>⚠️ Important :</strong> Pour votre sécurité, changez votre mot de passe après votre première connexion.</p>
                    <p>Si vous avez des questions, contactez l'administrateur.</p>
                </div>
                <div style='text-align: center; color: #666; font-size: 12px; margin-top: 20px;'>
                    <p>SMG Studio v2.0 - Social Media Generator by Audiovizu</p>
                </div>
            </div>
        </body>
        </html>
        ";
        
        // Utiliser la fonction sendMail de phpmailer.php
        return sendMail($user['email'], $subject, $body);
        
    } catch (Exception $e) {
        error_log("Erreur sendPasswordToUser: " . $e->getMessage());
        return "Erreur envoi email: " . $e->getMessage();
    }
}

function generateRandomPassword($length = 12) {
    $lowercase = 'abcdefghijklmnopqrstuvwxyz';
    $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $numbers = '0123456789';
    $symbols = '!@#$%';
    
    // Au moins un caractère de chaque type
    $password = '';
    $password .= $lowercase[random_int(0, strlen($lowercase) - 1)];
    $password .= $uppercase[random_int(0, strlen($uppercase) - 1)];
    $password .= $numbers[random_int(0, strlen($numbers) - 1)];
    $password .= $symbols[random_int(0, strlen($symbols) - 1)];
    
    // Compléter avec des caractères aléatoires
    $all_chars = $lowercase . $uppercase . $numbers . $symbols;
    for ($i = 4; $i < $length; $i++) {
        $password .= $all_chars[random_int(0, strlen($all_chars) - 1)];
    }
    
    // Mélanger le mot de passe
    return str_shuffle($password);
}

function getAllUsers() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, name, email, role, status, created_at, updated_at,
                   (SELECT MAX(attempted_at) FROM login_attempts WHERE email = users.email AND success = 1) as last_login
            FROM users 
            WHERE status != 'deleted'
            ORDER BY created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getAllUsers: " . $e->getMessage());
        return [];
    }
}

// Récupérer liste des utilisateurs
$users = getAllUsers();

// 🔥 CORRECTION 4: Statistiques sécurisées
$stats = [
    'total' => count($users),
    'active' => count(array_filter($users, fn($u) => $u['status'] === 'active')),
    'admins' => count(array_filter($users, fn($u) => $u['role'] === 'admin')),
    'recent' => count(array_filter($users, fn($u) => $u['last_login'] && $u['last_login'] >= date('Y-m-d', strtotime('-7 days'))))
];

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>👥 SMG Studio - Gestion Utilisateurs</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8f9fa;
            color: #333;
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 12px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .header h1 {
            font-size: 28px;
            margin-bottom: 8px;
        }

        .header p {
            opacity: 0.9;
            font-size: 16px;
        }

        .user-info {
            float: right;
            text-align: right;
            font-size: 14px;
        }

        .card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            border: 1px solid #e9ecef;
        }

        .card h2 {
            color: #495057;
            margin-bottom: 20px;
            font-size: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
        }

        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: #212529; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
        }

        .form-label {
            font-weight: 600;
            margin-bottom: 5px;
            color: #495057;
            font-size: 14px;
        }

        .form-input, .form-select {
            padding: 10px 12px;
            border: 2px solid #e9ecef;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }

        .form-input:focus, .form-select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .table-container {
            overflow-x: auto;
        }

        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }

        .users-table th {
            background: #f8f9fa;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #495057;
            border-bottom: 2px solid #dee2e6;
            font-size: 14px;
        }

        .users-table td {
            padding: 12px;
            border-bottom: 1px solid #dee2e6;
            font-size: 14px;
        }

        .users-table tr:hover {
            background: #f8f9fa;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-active { background: #d4edda; color: #155724; }
        .status-inactive { background: #f8d7da; color: #721c24; }
        .status-pending { background: #fff3cd; color: #856404; }

        .role-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }

        .role-admin { background: #d1ecf1; color: #0c5460; }
        .role-manager { background: #d4edda; color: #155724; }
        .role-user { background: #e2e3e5; color: #383d41; }

        .message {
            padding: 12px 16px;
            border-radius: 6px;
            margin-bottom: 20px;
            font-weight: 500;
        }

        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .actions {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }

        .admin-only {
            display: <?= $is_admin ? 'block' : 'none' ?>;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            border: 1px solid #e9ecef;
        }

        .stat-number {
            font-size: 32px;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #6c757d;
            font-size: 14px;
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
        }

        .modal-content {
            background: white;
            margin: 5% auto;
            padding: 30px;
            border-radius: 12px;
            max-width: 500px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .close {
            font-size: 24px;
            cursor: pointer;
            color: #aaa;
        }

        .close:hover {
            color: #000;
        }

        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .header {
                padding: 20px;
            }
            
            .user-info {
                float: none;
                text-align: left;
                margin-top: 15px;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <header class="header">
            <div class="user-info">
                👤 <?= htmlspecialchars($current_user['name']) ?><br>
                🔑 <?= ucfirst($current_user['role']) ?>
                <br><br>
                <a href="dashboard.php" class="btn btn-info">🏠 Dashboard</a>
                <a href="logout.php" class="btn btn-warning">🚪 Déconnexion</a>
            </div>
            <h1>👥 Gestion des Utilisateurs</h1>
            <p>SMG Studio v2.0 - Administration des comptes utilisateurs</p>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="message success"><?= htmlspecialchars($message) ?></div>
        <?php endif; ?>
        <?php if ($error): ?>
            <div class="message error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>

        <!-- Statistiques -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?= $stats['total'] ?></div>
                <div class="stat-label">👥 Utilisateurs Total</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $stats['active'] ?></div>
                <div class="stat-label">✅ Actifs</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $stats['admins'] ?></div>
                <div class="stat-label">🔑 Administrateurs</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $stats['recent'] ?></div>
                <div class="stat-label">🕐 Actifs 7j</div>
            </div>
        </div>

        <!-- Création utilisateur (Admin seulement) -->
        <div class="card admin-only">
            <h2>➕ Créer un Nouvel Utilisateur</h2>
            <form method="POST">
                <input type="hidden" name="action" value="create_user">
                <div class="form-grid">
                    <div class="form-group">
                        <label class="form-label">👤 Nom Complet</label>
                        <input type="text" name="name" class="form-input" required maxlength="100">
                    </div>
                    <div class="form-group">
                        <label class="form-label">📧 Email</label>
                        <input type="email" name="email" class="form-input" required maxlength="255">
                    </div>
                    <div class="form-group">
                        <label class="form-label">🔑 Rôle</label>
                        <select name="role" class="form-select" required>
                            <option value="user">👤 Utilisateur</option>
                            <option value="manager">👔 Manager</option>
                            <option value="admin">🔑 Administrateur</option>
                        </select>
                    </div>
                </div>
                <button type="submit" class="btn btn-success">➕ Créer Utilisateur</button>
                <small style="color: #6c757d; margin-left: 15px;">
                    📧 Un mot de passe sera généré et envoyé par email automatiquement
                </small>
            </form>
        </div>

        <!-- Liste des utilisateurs -->
        <div class="card">
            <h2>👥 Liste des Utilisateurs</h2>
            <div class="table-container">
                <table class="users-table">
                    <thead>
                        <tr>
                            <th>👤 Utilisateur</th>
                            <th>📧 Email</th>
                            <th>🔑 Rôle</th>
                            <th>📊 Statut</th>
                            <th>📅 Créé le</th>
                            <th>🕐 Dernière connexion</th>
                            <?php if ($is_admin): ?>
                            <th>⚙️ Actions</th>
                            <?php endif; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr>
                            <td>
                                <strong><?= htmlspecialchars($user['name']) ?></strong>
                                <?php if ($user['id'] == $current_user['id']): ?>
                                    <span style="color: #667eea; font-size: 12px;">(Vous)</span>
                                <?php endif; ?>
                            </td>
                            <td><?= htmlspecialchars($user['email']) ?></td>
                            <td>
                                <span class="role-badge role-<?= $user['role'] ?>">
                                    <?= ucfirst($user['role']) ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge status-<?= $user['status'] ?>">
                                    <?= ucfirst($user['status']) ?>
                                </span>
                            </td>
                            <td><?= date('d/m/Y', strtotime($user['created_at'])) ?></td>
                            <td>
                                <?= $user['last_login'] ? date('d/m/Y H:i', strtotime($user['last_login'])) : 'Jamais' ?>
                            </td>
                            <?php if ($is_admin): ?>
                            <td>
                                <div class="actions">
                                    <button onclick="editUser(<?= $user['id'] ?>, '<?= addslashes($user['name']) ?>', '<?= addslashes($user['email']) ?>', '<?= $user['role'] ?>', '<?= $user['status'] ?>')" 
                                            class="btn btn-primary">✏️ Modifier</button>
                                    
                                    <form method="POST" style="display: inline;" 
                                          onsubmit="return confirm('Générer un nouveau mot de passe pour <?= addslashes($user['name']) ?> ?')">
                                        <input type="hidden" name="action" value="reset_password">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <button type="submit" class="btn btn-warning">🔄 Reset MDP</button>
                                    </form>
                                    
                                    <?php if ($user['id'] != $current_user['id']): ?>
                                    <form method="POST" style="display: inline;"
                                          onsubmit="return confirm('Supprimer l\'utilisateur <?= addslashes($user['name']) ?> ?')">
                                        <input type="hidden" name="action" value="delete_user">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <button type="submit" class="btn btn-danger">🗑️ Supprimer</button>
                                    </form>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <?php endif; ?>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal Modification Utilisateur -->
    <div id="editModal" class="modal admin-only">
        <div class="modal-content">
            <div class="modal-header">
                <h3>✏️ Modifier Utilisateur</h3>
                <span class="close" onclick="closeModal()">&times;</span>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="update_user">
                <input type="hidden" name="user_id" id="editUserId">
                
                <div class="form-group">
                    <label class="form-label">👤 Nom Complet</label>
                    <input type="text" name="name" id="editUserName" class="form-input" required maxlength="100">
                </div>
                
                <div class="form-group">
                    <label class="form-label">📧 Email</label>
                    <input type="email" name="email" id="editUserEmail" class="form-input" required maxlength="255">
                </div>
                
                <div class="form-group">
                    <label class="form-label">🔑 Rôle</label>
                    <select name="role" id="editUserRole" class="form-select" required>
                        <option value="user">👤 Utilisateur</option>
                        <option value="manager">👔 Manager</option>
                        <option value="admin">🔑 Administrateur</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📊 Statut</label>
                    <select name="status" id="editUserStatus" class="form-select" required>
                        <option value="active">✅ Actif</option>
                        <option value="inactive">❌ Inactif</option>
                        <option value="pending">⏳ En attente</option>
                    </select>
                </div>
                
                <div style="margin-top: 20px;">
                    <button type="submit" class="btn btn-success">💾 Sauvegarder</button>
                    <button type="button" onclick="closeModal()" class="btn btn-danger">❌ Annuler</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function editUser(id, name, email, role, status) {
            document.getElementById('editUserId').value = id;
            document.getElementById('editUserName').value = name;
            document.getElementById('editUserEmail').value = email;
            document.getElementById('editUserRole').value = role;
            document.getElementById('editUserStatus').value = status;
            document.getElementById('editModal').style.display = 'block';
        }

        function closeModal() {
            document.getElementById('editModal').style.display = 'none';
        }

        // Fermer modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modal = document.getElementById('editModal');
            if (event.target === modal) {
                closeModal();
            }
        }

        console.log('👥 SMG Studio - Gestion Utilisateurs Ready! CORRIGÉ');
    </script>
</body>
</html>