<?php
/**
 * ================================================================
 * 🎬 CLIPMAKER TIMELINE API - ENDPOINTS COMPLETS
 * ================================================================
 * 
 * 🎯 ENDPOINTS TIMELINE :
 * - POST /save-project : Sauvegarde projet timeline
 * - POST /load-project : Chargement projet timeline
 * - POST /export-project : Export vidéo FFmpeg
 * - POST /apply-effect : Application effet temps réel
 * - POST /split-clip : Découpe clip
 * - GET /project-media : Médias projet
 * - POST /save-template : Sauvegarde template
 * 
 * 💡 BY PAPY 77 - SMG STUDIO
 * ================================================================
 */

require_once '../auth.php';
requireLogin();

header('Content-Type: application/json');

$user = getCurrentUser();
$action = $_POST['action'] ?? $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'save-project':
            echo json_encode(saveTimelineProject($user, $_POST));
            break;
            
        case 'load-project':
            echo json_encode(loadTimelineProject($user, $_POST));
            break;
            
        case 'export-project':
            echo json_encode(exportTimelineProject($user, $_POST));
            break;
            
        case 'apply-effect':
            echo json_encode(applyTimelineEffect($user, $_POST));
            break;
            
        case 'split-clip':
            echo json_encode(splitTimelineClip($user, $_POST));
            break;
            
        case 'get-project-media':
            echo json_encode(getProjectMedia($user, $_GET));
            break;
            
        case 'save-template':
            echo json_encode(saveTimelineTemplate($user, $_POST));
            break;
            
        case 'get-transitions':
            echo json_encode(getAvailableTransitions());
            break;
            
        default:
            throw new Exception('Action non supportée');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

/**
 * ================================================================
 * 💾 SAUVEGARDE PROJET TIMELINE
 * ================================================================
 */
function saveTimelineProject($user, $data) {
    global $pdo;
    
    $projectId = $data['project_id'] ?? null;
    $timelineData = $data['timeline_data'] ?? [];
    $clips = $data['clips'] ?? [];
    $effects = $data['effects'] ?? [];
    $settings = $data['settings'] ?? [];
    
    // Validation
    if (empty($timelineData)) {
        throw new Exception('Données timeline requises');
    }
    
    $pdo->beginTransaction();
    
    try {
        if ($projectId) {
            // Mise à jour projet existant
            $stmt = $pdo->prepare("
                UPDATE clipmaker_projects 
                SET timeline_data = ?, 
                    settings = ?, 
                    duration = ?,
                    updated_at = NOW()
                WHERE id = ? AND user_id = ?
            ");
            
            $totalDuration = calculateTotalDuration($clips);
            
            $stmt->execute([
                json_encode($timelineData),
                json_encode($settings),
                $totalDuration,
                $projectId,
                $user['id']
            ]);
        } else {
            // Nouveau projet
            $stmt = $pdo->prepare("
                INSERT INTO clipmaker_projects 
                (user_id, name, timeline_data, settings, duration, status)
                VALUES (?, ?, ?, ?, ?, 'draft')
            ");
            
            $projectName = 'Projet ' . date('Y-m-d H:i:s');
            $totalDuration = calculateTotalDuration($clips);
            
            $stmt->execute([
                $user['id'],
                $projectName,
                json_encode($timelineData),
                json_encode($settings),
                $totalDuration
            ]);
            
            $projectId = $pdo->lastInsertId();
        }
        
        // Sauvegarder les clips individuels si nécessaire
        if (!empty($clips)) {
            foreach ($clips as $clip) {
                saveClipData($pdo, $projectId, $clip);
            }
        }
        
        // Log de l'action
        logTimelineAction($pdo, $user['id'], $projectId, 'save', [
            'clips_count' => count($clips),
            'effects_count' => count($effects)
        ]);
        
        $pdo->commit();
        
        return [
            'success' => true,
            'message' => 'Projet sauvegardé avec succès',
            'project_id' => $projectId,
            'saved_at' => date('Y-m-d H:i:s')
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

/**
 * ================================================================
 * 📂 CHARGEMENT PROJET TIMELINE
 * ================================================================
 */
function loadTimelineProject($user, $data) {
    global $pdo;
    
    $projectId = $data['project_id'] ?? null;
    
    if (!$projectId) {
        throw new Exception('ID projet requis');
    }
    
    // Récupération projet
    $stmt = $pdo->prepare("
        SELECT * FROM clipmaker_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$projectId, $user['id']]);
    $project = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$project) {
        throw new Exception('Projet non trouvé');
    }
    
    // Récupération médias associés
    $mediaStmt = $pdo->prepare("
        SELECT * FROM clipmaker_media 
        WHERE project_id = ? OR user_id = ?
        ORDER BY created_at DESC
    ");
    $mediaStmt->execute([$projectId, $user['id']]);
    $medias = $mediaStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Récupération historique timeline
    $historyStmt = $pdo->prepare("
        SELECT * FROM clipmaker_ai_logs 
        WHERE project_id = ? 
        ORDER BY created_at DESC 
        LIMIT 50
    ");
    $historyStmt->execute([$projectId]);
    $history = $historyStmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'success' => true,
        'project' => $project,
        'timeline_data' => json_decode($project['timeline_data'], true),
        'medias' => $medias,
        'history' => $history
    ];
}

/**
 * ================================================================
 * 🎬 EXPORT PROJET TIMELINE
 * ================================================================
 */
function exportTimelineProject($user, $data) {
    global $pdo;
    
    $projectId = $data['project_id'] ?? null;
    $exportSettings = $data['settings'] ?? [];
    $clips = $data['clips'] ?? [];
    
    if (!$projectId || empty($clips)) {
        throw new Exception('Données export insuffisantes');
    }
    
    // Validation projet
    $stmt = $pdo->prepare("
        SELECT * FROM clipmaker_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$projectId, $user['id']]);
    $project = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$project) {
        throw new Exception('Projet non trouvé');
    }
    
    // Préparer données export
    $exportData = [
        'project_id' => $projectId,
        'user_id' => $user['id'],
        'export_name' => 'Export_' . date('Y-m-d_H-i-s'),
        'format' => $exportSettings['format'] ?? 'mp4',
        'quality' => $exportSettings['quality'] ?? 'high',
        'resolution' => $exportSettings['resolution'] ?? '1920x1080',
        'clips' => $clips,
        'export_settings' => json_encode($exportSettings),
        'status' => 'pending'
    ];
    
    // Créer entrée export
    $exportStmt = $pdo->prepare("
        INSERT INTO clipmaker_exports 
        (project_id, user_id, export_name, format, quality, resolution, export_settings, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')
    ");
    
    $exportStmt->execute([
        $projectId,
        $user['id'],
        $exportData['export_name'],
        $exportData['format'],
        $exportData['quality'],
        $exportData['resolution'],
        $exportData['export_settings']
    ]);
    
    $exportId = $pdo->lastInsertId();
    
    // Log export
    logTimelineAction($pdo, $user['id'], $projectId, 'export', [
        'export_id' => $exportId,
        'format' => $exportData['format'],
        'quality' => $exportData['quality']
    ]);
    
    // TODO: Déclencher traitement FFmpeg asynchrone
    // queueFFmpegExport($exportId, $exportData);
    
    return [
        'success' => true,
        'message' => 'Export en cours de traitement',
        'export_id' => $exportId,
        'estimated_time' => estimateExportTime($clips)
    ];
}

/**
 * ================================================================
 * 🎨 APPLICATION EFFET TEMPS RÉEL
 * ================================================================
 */
function applyTimelineEffect($user, $data) {
    global $pdo;
    
    $clipId = $data['clip_id'] ?? null;
    $effectType = $data['effect_type'] ?? null;
    $effectParams = $data['effect_params'] ?? [];
    
    if (!$clipId || !$effectType) {
        throw new Exception('Données effet incomplètes');
    }
    
    // Log application effet
    $stmt = $pdo->prepare("
        INSERT INTO clipmaker_ai_logs 
        (user_id, ai_type, input_data, output_data, status)
        VALUES (?, 'effect_application', ?, ?, 'success')
    ");
    
    $inputData = json_encode([
        'clip_id' => $clipId,
        'effect_type' => $effectType,
        'params' => $effectParams
    ]);
    
    $outputData = json_encode([
        'applied' => true,
        'timestamp' => time()
    ]);
    
    $stmt->execute([$user['id'], $inputData, $outputData]);
    
    return [
        'success' => true,
        'message' => "Effet {$effectType} appliqué",
        'effect_id' => $pdo->lastInsertId()
    ];
}

/**
 * ================================================================
 * ✂️ DÉCOUPE CLIP
 * ================================================================
 */
function splitTimelineClip($user, $data) {
    global $pdo;
    
    $clipId = $data['clip_id'] ?? null;
    $splitTime = $data['split_time'] ?? null;
    $projectId = $data['project_id'] ?? null;
    
    if (!$clipId || !$splitTime || !$projectId) {
        throw new Exception('Données split incomplètes');
    }
    
    // Log split
    $stmt = $pdo->prepare("
        INSERT INTO clipmaker_ai_logs 
        (user_id, project_id, ai_type, input_data, output_data, status)
        VALUES (?, ?, 'clip_split', ?, ?, 'success')
    ");
    
    $inputData = json_encode([
        'clip_id' => $clipId,
        'split_time' => $splitTime
    ]);
    
    $outputData = json_encode([
        'original_clip' => $clipId,
        'new_clip_id' => $clipId . '_split_' . time(),
        'split_time' => $splitTime
    ]);
    
    $stmt->execute([$user['id'], $projectId, $inputData, $outputData]);
    
    return [
        'success' => true,
        'message' => 'Clip découpé avec succès',
        'split_result' => json_decode($outputData, true)
    ];
}

/**
 * ================================================================
 * 📁 RÉCUPÉRATION MÉDIAS PROJET
 * ================================================================
 */
function getProjectMedia($user, $data) {
    global $pdo;
    
    $projectId = $data['project_id'] ?? null;
    $fileType = $data['file_type'] ?? null;
    $limit = $data['limit'] ?? 50;
    
    $sql = "SELECT * FROM clipmaker_media WHERE user_id = ?";
    $params = [$user['id']];
    
    if ($projectId) {
        $sql .= " AND (project_id = ? OR project_id IS NULL)";
        $params[] = $projectId;
    }
    
    if ($fileType) {
        $sql .= " AND file_type = ?";
        $params[] = $fileType;
    }
    
    $sql .= " ORDER BY created_at DESC LIMIT ?";
    $params[] = $limit;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $medias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'success' => true,
        'medias' => $medias,
        'count' => count($medias)
    ];
}

/**
 * ================================================================
 * 📐 SAUVEGARDE TEMPLATE
 * ================================================================
 */
function saveTimelineTemplate($user, $data) {
    global $pdo;
    
    $templateName = $data['template_name'] ?? null;
    $templateData = $data['template_data'] ?? [];
    $category = $data['category'] ?? 'custom';
    
    if (!$templateName || empty($templateData)) {
        throw new Exception('Données template incomplètes');
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO clipmaker_templates 
        (user_id, name, template_data, category, duration)
        VALUES (?, ?, ?, ?, ?)
    ");
    
    $duration = calculateTemplateDuration($templateData);
    
    $stmt->execute([
        $user['id'],
        $templateName,
        json_encode($templateData),
        $category,
        $duration
    ]);
    
    $templateId = $pdo->lastInsertId();
    
    return [
        'success' => true,
        'message' => 'Template sauvegardé',
        'template_id' => $templateId
    ];
}

/**
 * ================================================================
 * 🎭 RÉCUPÉRATION TRANSITIONS
 * ================================================================
 */
function getAvailableTransitions() {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT * FROM transitions 
        WHERE actif = 1 
        ORDER BY nom
    ");
    $stmt->execute();
    $transitions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'success' => true,
        'transitions' => $transitions
    ];
}

/**
 * ================================================================
 * 🛠️ FONCTIONS UTILITAIRES
 * ================================================================
 */

function calculateTotalDuration($clips) {
    $maxEndTime = 0;
    
    foreach ($clips as $clip) {
        $endTime = ($clip['startTime'] ?? 0) + ($clip['duration'] ?? 0);
        $maxEndTime = max($maxEndTime, $endTime);
    }
    
    return $maxEndTime;
}

function calculateTemplateDuration($templateData) {
    // Calculer durée template basée sur les clips
    return $templateData['duration'] ?? 5;
}

function estimateExportTime($clips) {
    // Estimation basée sur nombre de clips et effets
    $baseTime = count($clips) * 2; // 2 secondes par clip
    return max($baseTime, 10); // Minimum 10 secondes
}

function saveClipData($pdo, $projectId, $clip) {
    // Sauvegarder données spécifiques du clip si nécessaire
    // Pour l'instant, tout est dans timeline_data
    return true;
}

function logTimelineAction($pdo, $userId, $projectId, $action, $details) {
    $stmt = $pdo->prepare("
        INSERT INTO clipmaker_ai_logs 
        (user_id, project_id, ai_type, input_data, output_data, status)
        VALUES (?, ?, 'timeline_action', ?, ?, 'success')
    ");
    
    $inputData = json_encode([
        'action' => $action,
        'timestamp' => time()
    ]);
    
    $outputData = json_encode($details);
    
    $stmt->execute([$userId, $projectId, $inputData, $outputData]);
}

/**
 * ================================================================
 * 🎯 UTILISATION API :
 * ================================================================
 * 
 * // Sauvegarde projet
 * POST /api/clipmaker-timeline-api.php
 * {
 *   "action": "save-project",
 *   "project_id": 123,
 *   "timeline_data": {...},
 *   "clips": [...],
 *   "settings": {...}
 * }
 * 
 * // Export projet
 * POST /api/clipmaker-timeline-api.php
 * {
 *   "action": "export-project",
 *   "project_id": 123,
 *   "clips": [...],
 *   "settings": {
 *     "format": "mp4",
 *     "quality": "high",
 *     "resolution": "1920x1080"
 *   }
 * }
 * 
 * // Chargement projet
 * POST /api/clipmaker-timeline-api.php
 * {
 *   "action": "load-project",
 *   "project_id": 123
 * }
 * 
 * ================================================================
 */
?>