<?php
/**
 * ================================================================
 * 📊 SMG STUDIO v2.0 - DASHBOARD TOKENS + DÉBIT AUTOMATIQUE
 * ================================================================
 * 
 * 🎯 Suivi tokens multi-provider en temps réel
 * 💰 Gestion coûts et alertes automatiques + DÉBIT TOKENS
 * 📈 Analytics détaillées utilisation IA
 * ⚡ Interface temps réel avec auto-refresh
 * 🔄 Navigation retour dashboard principal
 * 
 * 🎓 NOUVEAUTÉ : GUIDES TOKENS INTÉGRÉS + DÉBIT AUTO !
 * ✅ Explication tokens pour clients neu-neu
 * ✅ Calculateur coûts en temps réel
 * ✅ Démonstration Question + Réponse = Total
 * ✅ DÉBIT AUTOMATIQUE TOKENS UTILISATEUR
 * 
 * 💡 BY PAPY 77 - PHASE 3 TOKEN MANAGEMENT + ÉDUCATION CLIENT + DÉBIT AUTO !
 * ================================================================
 */
header('Content-Type: text/html; charset=UTF-8');
session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// API endpoint pour données temps réel
if (isset($_GET['api']) && $_GET['api'] === 'realtime') {
    header('Content-Type: application/json');
    echo json_encode(getRealtimeTokenData($pdo, $_SESSION['user_id']));
    exit;
}

// Récupération utilisateur
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
} catch (Exception $e) {
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS
// ================================================================

$action = $_POST['action'] ?? $_GET['action'] ?? 'dashboard';
$message = '';
$message_type = '';

// Action Créer/Modifier Quota
if ($action === 'save_quota' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $quota_period = $_POST['quota_period'] ?? 'monthly';
        $tokens_limit = intval($_POST['tokens_limit'] ?? 10000);
        $cost_limit = floatval($_POST['cost_limit'] ?? 10.00);
        $requests_limit = intval($_POST['requests_limit'] ?? 100);
        $ai_provider = $_POST['ai_provider'] ?? null;
        
        if ($tokens_limit < 100 || $tokens_limit > 1000000) {
            throw new Exception("Limite tokens entre 100 et 1,000,000");
        }
        
        $dates = calculateQuotaPeriod($quota_period);
        
        $stmt = $pdo->prepare("
            SELECT id FROM user_ai_quotas 
            WHERE user_id = ? AND quota_period = ? AND ai_provider = ?
            AND quota_start_date = ?
        ");
        $stmt->execute([$user['id'], $quota_period, $ai_provider, $dates['start']]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            $stmt = $pdo->prepare("
                UPDATE user_ai_quotas SET 
                    tokens_limit = ?, cost_limit = ?, requests_limit = ?,
                    quota_end_date = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$tokens_limit, $cost_limit, $requests_limit, $dates['end'], $existing['id']]);
            $message = "✅ Quota mis à jour avec succès !";
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO user_ai_quotas 
                (user_id, quota_period, quota_start_date, quota_end_date, 
                 tokens_limit, cost_limit, requests_limit, ai_provider)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $user['id'], $quota_period, $dates['start'], $dates['end'],
                $tokens_limit, $cost_limit, $requests_limit, $ai_provider
            ]);
            $message = "✅ Nouveau quota créé avec succès !";
        }
        
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur quota : " . $e->getMessage();
        $message_type = 'error';
    }
}

// Action Reset Quota
if ($action === 'reset_quota' && isset($_GET['id'])) {
    try {
        $quota_id = intval($_GET['id']);
        
        $stmt = $pdo->prepare("
            UPDATE user_ai_quotas SET 
                tokens_used = 0, cost_used = 0, requests_used = 0,
                alert_sent = 0, updated_at = NOW()
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$quota_id, $user['id']]);
        
        $message = "✅ Quota remis à zéro !";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur reset : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 💰 ACTION TEST DÉBIT TOKENS (NOUVEAU)
if ($action === 'test_debit' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $tokens_to_debit = intval($_POST['tokens_amount'] ?? 100);
        
        if ($tokens_to_debit < 1 || $tokens_to_debit > 10000) {
            throw new Exception("Montant entre 1 et 10,000 tokens");
        }
        
        $debit_result = debitUserTokens($user['id'], $tokens_to_debit);
        
        if ($debit_result['success']) {
            $message = "✅ Débit test réussi : {$tokens_to_debit} tokens débités !";
            $message_type = 'success';
        } else {
            $message = "❌ Échec débit test : " . $debit_result['error'];
            $message_type = 'error';
        }
        
    } catch (Exception $e) {
        $message = "❌ Erreur test débit : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DES DONNÉES
// ================================================================

$global_stats = getGlobalTokenStats($user['id']);
$current_quotas = getCurrentQuotas($user['id']);
$provider_usage = getProviderUsage($user['id']);
$recent_activity = getRecentActivity($user['id']);
$usage_trends = getUsageTrends($user['id']);
$active_alerts = getActiveAlerts($user['id']);

// NOUVELLES DONNÉES PACKAGES
$user_packages = getUserPackages($user['id']);
$user_balance = getUserTokenBalance($user['id']);

// ================================================================
// 🔧 FONCTIONS MÉTIER
// ================================================================

function getRealtimeTokenData($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as requests_today,
                SUM(tokens_total) as tokens_today,
                SUM(cost_estimate) as cost_today,
                AVG(processing_time_ms) as avg_time_today,
                COUNT(CASE WHEN success = 0 THEN 1 END) as failed_today
            FROM ai_usage_logs 
            WHERE user_id = ? AND DATE(created_at) = CURDATE()
        ");
        $stmt->execute([$user_id]);
        $today_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stmt = $pdo->prepare("
            SELECT 
                quota_period,
                ROUND((tokens_used / tokens_limit) * 100, 1) as usage_percentage,
                tokens_remaining,
                CASE 
                    WHEN tokens_used >= tokens_limit THEN 'EXCEEDED'
                    WHEN (tokens_used / tokens_limit) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as status
            FROM user_ai_quotas 
            WHERE user_id = ? AND is_active = 1 
            AND CURDATE() BETWEEN quota_start_date AND quota_end_date
            ORDER BY quota_period DESC
            LIMIT 3
        ");
        $stmt->execute([$user_id]);
        $quotas_status = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $stmt = $pdo->prepare("
            SELECT created_at, request_type, tokens_total, cost_estimate, success
            FROM ai_usage_logs 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 5
        ");
        $stmt->execute([$user_id]);
        $last_activity = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return [
            'success' => true,
            'timestamp' => date('Y-m-d H:i:s'),
            'today_stats' => $today_stats,
            'quotas_status' => $quotas_status,
            'last_activity' => $last_activity
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// ================================================================
// 💰 NOUVELLES FONCTIONS - DÉBIT TOKENS UTILISATEUR
// ================================================================

function debitUserTokens($user_id, $tokens_used) {
    global $pdo;
    
    try {
        // Vérifier le solde avant débit
        $stmt = $pdo->prepare("
            SELECT id, tokens_remaining 
            FROM user_token_packages 
            WHERE user_id = ? AND status = 'active' 
            AND tokens_remaining >= ?
            ORDER BY purchase_date ASC
            LIMIT 1
        ");
        $stmt->execute([$user_id, $tokens_used]);
        $package = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$package) {
            throw new Exception("Solde tokens insuffisant");
        }
        
        // Débiter les tokens
        $stmt = $pdo->prepare("
            UPDATE user_token_packages 
            SET tokens_used = tokens_used + ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $success = $stmt->execute([$tokens_used, $package['id']]);
        
        if (!$success) {
            throw new Exception("Erreur lors du débit");
        }
        
        // Log du débit
        error_log("Débit tokens réussi - User: $user_id, Tokens: $tokens_used");
        
        return [
            'success' => true,
            'tokens_debited' => $tokens_used,
            'remaining_tokens' => $package['tokens_remaining'] - $tokens_used
        ];
        
    } catch (Exception $e) {
        error_log("Erreur débit tokens - User: $user_id, Tokens: $tokens_used, Erreur: " . $e->getMessage());
        
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function getUserTokenBalance($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                SUM(tokens_purchased) as total_purchased,
                SUM(tokens_used) as total_used,
                SUM(tokens_remaining) as current_balance
            FROM user_token_packages 
            WHERE user_id = ? AND status = 'active'
        ");
        $stmt->execute([$user_id]);
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Erreur récupération solde - User: $user_id, Erreur: " . $e->getMessage());
        return ['current_balance' => 0, 'total_purchased' => 0, 'total_used' => 0];
    }
}

function getUserPackages($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                utp.*,
                ROUND((utp.tokens_used / utp.tokens_purchased) * 100, 1) as usage_percentage,
                CASE 
                    WHEN utp.tokens_used >= utp.tokens_purchased THEN 'COMPLETED'
                    WHEN (utp.tokens_used / utp.tokens_purchased) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as package_status
            FROM user_token_packages utp
            WHERE utp.user_id = ? AND utp.status = 'active'
            ORDER BY utp.purchase_date DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserPackages: " . $e->getMessage());
        return [];
    }
}

function getGlobalTokenStats($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_requests,
                SUM(tokens_total) as total_tokens,
                SUM(cost_estimate) as total_cost,
                AVG(processing_time_ms) as avg_processing_time,
                COUNT(CASE WHEN success = 0 THEN 1 END) as failed_requests,
                MIN(created_at) as first_usage,
                MAX(created_at) as last_usage,
                COUNT(DISTINCT ai_profile_id) as profiles_used,
                COUNT(DISTINCT DATE(created_at)) as active_days
            FROM ai_usage_logs 
            WHERE user_id = ?
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getCurrentQuotas($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT uaq.*,
                   ROUND((uaq.tokens_used / uaq.tokens_limit) * 100, 1) as usage_percentage,
                   ROUND((uaq.cost_used / uaq.cost_limit) * 100, 1) as cost_percentage,
                   CASE 
                       WHEN uaq.tokens_used >= uaq.tokens_limit THEN 'EXCEEDED'
                       WHEN (uaq.tokens_used / uaq.tokens_limit) >= (uaq.alert_at_percentage / 100) THEN 'WARNING'
                       ELSE 'OK'
                   END as quota_status,
                   DATEDIFF(uaq.quota_end_date, CURDATE()) as days_remaining,
                   apc.provider_name
            FROM user_ai_quotas uaq
            LEFT JOIN ai_providers_catalog apc ON uaq.ai_provider = apc.provider_key
            WHERE uaq.user_id = ? AND uaq.is_active = 1 
            AND CURDATE() BETWEEN uaq.quota_start_date AND uaq.quota_end_date
            ORDER BY uaq.quota_period, uaq.ai_provider
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getProviderUsage($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                uap.ai_provider,
                apc.provider_name,
                COUNT(aul.id) as requests_count,
                SUM(aul.tokens_total) as total_tokens,
                SUM(aul.cost_estimate) as total_cost,
                AVG(aul.processing_time_ms) as avg_time,
                COUNT(CASE WHEN aul.success = 0 THEN 1 END) as failed_count,
                MAX(aul.created_at) as last_used
            FROM user_ai_profiles uap
            LEFT JOIN ai_usage_logs aul ON uap.id = aul.ai_profile_id
            LEFT JOIN ai_providers_catalog apc ON uap.ai_provider = apc.provider_key
            WHERE uap.user_id = ? AND uap.is_active = 1
            GROUP BY uap.ai_provider
            ORDER BY total_tokens DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getRecentActivity($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT aul.*, uap.display_name as profile_name, apc.provider_name
            FROM ai_usage_logs aul
            JOIN user_ai_profiles uap ON aul.ai_profile_id = uap.id
            LEFT JOIN ai_providers_catalog apc ON uap.ai_provider = apc.provider_key
            WHERE aul.user_id = ?
            ORDER BY aul.created_at DESC
            LIMIT 20
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getUsageTrends($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as requests,
                SUM(tokens_total) as tokens,
                SUM(cost_estimate) as cost
            FROM ai_usage_logs 
            WHERE user_id = ? AND created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getActiveAlerts($user_id) {
    global $pdo;
    
    $alerts = [];
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM user_ai_quotas 
            WHERE user_id = ? AND is_active = 1 
            AND CURDATE() BETWEEN quota_start_date AND quota_end_date
            AND (tokens_used / tokens_limit) >= (alert_at_percentage / 100)
        ");
        $stmt->execute([$user_id]);
        $quota_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($quota_alerts as $quota) {
            $percentage = round(($quota['tokens_used'] / $quota['tokens_limit']) * 100, 1);
            $alerts[] = [
                'type' => 'quota',
                'level' => $percentage >= 100 ? 'danger' : 'warning',
                'title' => "Quota {$quota['quota_period']} à {$percentage}%",
                'message' => "Limite: " . number_format($quota['tokens_limit']) . " tokens",
                'created_at' => $quota['updated_at']
            ];
        }
        
        // Alerte packages tokens faibles
        $stmt = $pdo->prepare("
            SELECT * FROM user_token_packages 
            WHERE user_id = ? AND status = 'active'
            AND (tokens_used / tokens_purchased) >= 0.9
        ");
        $stmt->execute([$user_id]);
        $package_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($package_alerts as $package) {
            $percentage = round(($package['tokens_used'] / $package['tokens_purchased']) * 100, 1);
            $alerts[] = [
                'type' => 'package',
                'level' => $percentage >= 100 ? 'danger' : 'warning',
                'title' => "Package {$package['package_name']} à {$percentage}%",
                'message' => "Reste: " . number_format($package['tokens_remaining']) . " tokens",
                'created_at' => $package['updated_at']
            ];
        }
        
        return $alerts;
        
    } catch (Exception $e) {
        return [];
    }
}

function calculateQuotaPeriod($period) {
    $start = date('Y-m-d');
    
    switch ($period) {
        case 'daily':
            $end = date('Y-m-d');
            break;
        case 'weekly':
            $start = date('Y-m-d', strtotime('monday this week'));
            $end = date('Y-m-d', strtotime('sunday this week'));
            break;
        case 'monthly':
            $start = date('Y-m-01');
            $end = date('Y-m-t');
            break;
        case 'yearly':
            $start = date('Y-01-01');
            $end = date('Y-12-31');
            break;
        default:
            $end = date('Y-m-t');
    }
    
    return ['start' => $start, 'end' => $end];
}

 

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📊 Dashboard Tokens + Guides - SMG Studio v2.0 DÉBIT AUTO</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #4ecdc4, #44a08d);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .nav-ia-dashboard {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-dashboard a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-dashboard a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .nav-ia-dashboard a.active {
            background: #ff6b6b;
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .live-indicator {
            display: flex;
            align-items: center;
            gap: 8px;
            background: #28a745;
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        .live-dot {
            width: 8px;
            height: 8px;
            background: white;
            border-radius: 50%;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }

        .tabs-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            margin-bottom: 20px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }

        .tabs-nav {
            display: flex;
            background: #f8f9fa;
            border-bottom: 2px solid #e9ecef;
        }

        .tab-button {
            flex: 1;
            padding: 15px 20px;
            background: transparent;
            border: none;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .tab-button.active {
            background: #4ecdc4;
            color: white;
        }

        .tab-button:hover:not(.active) {
            background: #e9ecef;
        }

        .tab-content {
            display: none;
            padding: 25px;
        }

        .tab-content.active {
            display: block;
        }

        .packages-section {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
        }

        .package-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .package-card {
            background: rgba(255, 255, 255, 0.2);
            padding: 20px;
            border-radius: 12px;
            text-align: center;
        }

        .package-progress {
            width: 100%;
            height: 8px;
            background: rgba(255, 255, 255, 0.3);
            border-radius: 4px;
            overflow: hidden;
            margin: 10px 0;
        }

        .package-progress-bar {
            height: 100%;
            background: white;
            transition: width 0.3s ease;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .stat-card.realtime {
            border: 2px solid #28a745;
        }

        .stat-card.realtime::before {
            content: "LIVE";
            position: absolute;
            top: -8px;
            right: 10px;
            background: #28a745;
            color: white;
            padding: 2px 8px;
            border-radius: 10px;
            font-size: 10px;
            font-weight: 600;
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #4ecdc4;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .test-section {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
            color: white;
            border-radius: 15px;
            padding: 25px;
            margin: 20px 0;
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #56CCF2, #2F80ED);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 10px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-warning {
            background: #fff3cd;
            border-color: #ffeaa7;
            color: #856404;
        }

        .alert-danger {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .nav-ia-dashboard {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .tabs-nav {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation IA -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">📊</div>
                <div>
                    <h1>Dashboard Tokens IA + Débit Auto</h1>
                    <p>Suivi temps réel + Débit automatique tokens</p>
                </div>
                <div class="live-indicator">
                    <div class="live-dot"></div>
                    <span>LIVE + AUTO DEBIT</span>
                </div>
            </div>
            <div style="display: flex; align-items: center; gap: 15px;">
                <div class="refresh-indicator">
                    <span>⏱️</span>
                    <span id="lastRefresh">Actualisation...</span>
                </div>
                
                <div class="nav-ia-dashboard">
                    <a href="dashboard.php">
                        ← Dashboard Principal
                    </a>
                    <a href="ai_content_generator.php">
                        🎨 Générateur
                    </a>
                    <a href="ai_dashboard.php" class="active">
                        📊 Dashboard IA
                    </a>
                    <a href="ai_profiles.php">
                        ⚙️ Profils
                    </a>
                </div>
                
                <button onclick="openQuotaModal()" class="btn btn-success">📊 Gérer Quotas</button>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type === 'success' ? 'success' : 'error' ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Section Packages Tokens -->
        <?php if (!empty($user_packages)): ?>
            <div class="packages-section">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h2>💰 Mes Packages Tokens</h2>
                    <div style="text-align: right;">
                        <div style="font-size: 28px; font-weight: 700;">
                            <?= number_format($user_balance['current_balance'] ?? 0) ?>
                        </div>
                        <div style="font-size: 14px; opacity: 0.9;">tokens disponibles</div>
                    </div>
                </div>
                
                <div class="package-grid">
                    <?php foreach ($user_packages as $package): ?>
                        <div class="package-card">
                            <div style="font-size: 20px; margin-bottom: 8px;">
                                <?= htmlspecialchars($package['package_name']) ?>
                            </div>
                            <div style="font-size: 14px; opacity: 0.9; margin-bottom: 10px;">
                                <?= number_format($package['tokens_used']) ?> / <?= number_format($package['tokens_purchased']) ?>
                            </div>
                            <div class="package-progress">
                                <div class="package-progress-bar" style="width: <?= min(100, $package['usage_percentage']) ?>%;"></div>
                            </div>
                            <div style="font-size: 12px; margin-top: 8px;">
                                <?= $package['usage_percentage'] ?>% utilisé
                            </div>
                            <div style="font-size: 16px; font-weight: 700; margin-top: 8px;">
                                <?= number_format($package['tokens_remaining']) ?> restants
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Section Test Débit -->
        <div class="test-section">
            <h3 style="margin-bottom: 20px;">🧪 Test Débit Automatique Tokens</h3>
            <p style="margin-bottom: 20px; opacity: 0.9;">
                Testez le système de débit automatique avec un montant personnalisé.
            </p>
            
            <form method="POST" style="display: flex; gap: 15px; align-items: end;">
                <input type="hidden" name="action" value="test_debit">
                
                <div>
                    <label style="display: block; margin-bottom: 8px; font-weight: 600;">💰 Tokens à débiter</label>
                    <input type="number" name="tokens_amount" value="100" min="1" max="10000" 
                           style="padding: 10px 15px; border: none; border-radius: 8px; font-size: 16px; width: 150px;">
                </div>
                
                <button type="submit" style="background: rgba(255,255,255,0.2); color: white; padding: 10px 20px; border: none; border-radius: 8px; cursor: pointer; font-weight: 600;">
                    🚀 Tester Débit
                </button>
            </form>
            
            <div style="font-size: 12px; margin-top: 15px; opacity: 0.8;">
                ⚠️ Ce test débite vraiment vos tokens ! Utilisez des montants faibles pour tester.
            </div>
        </div>

        <!-- Onglets Dashboard -->
        <div class="tabs-container">
            <div class="tabs-nav">
                <button class="tab-button active" onclick="switchTab('dashboard')">
                    📊 Dashboard Tokens
                </button>
                <button class="tab-button" onclick="switchTab('guide-simple')">
                    🎯 Guide Tokens Simple
                </button>
                <button class="tab-button" onclick="switchTab('calculateur')">
                    💰 Calculateur Coûts
                </button>
                <button class="tab-button" onclick="switchTab('question-reponse')">
                    🧮 Question + Réponse
                </button>
            </div>

            <!-- TAB 1: Dashboard tokens -->
            <div id="tab-dashboard" class="tab-content active">
                
                <!-- Alertes actives -->
                <?php if (!empty($active_alerts)): ?>
                    <div class="alerts-container">
                        <h3>🚨 Alertes Actives</h3>
                        <?php foreach ($active_alerts as $alert): ?>
                            <div class="alert alert-<?= $alert['level'] ?>">
                                <strong><?= htmlspecialchars($alert['title']) ?></strong>
                                <span><?= htmlspecialchars($alert['message']) ?></span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <!-- Stats temps réel -->
                <div class="stats-grid" id="realtimeStats">
                    <div class="stat-card realtime">
                        <span class="stat-icon">🔥</span>
                        <div class="stat-value" id="todayRequests"><?= number_format($global_stats['total_requests'] ?? 0) ?></div>
                        <div class="stat-label">Requêtes totales</div>
                    </div>
                    <div class="stat-card realtime">
                        <span class="stat-icon">  </span>
                        <div class="stat-value" id="todayTokens"><?= number_format($global_stats['total_tokens'] ?? 0) ?></div>
                        <div class="stat-label">Tokens utilisés</div>
                    </div>
                    <div class="stat-card realtime">
                        <span class="stat-icon">💰</span>
                        <div class="stat-value" id="todayCost"><?= number_format($global_stats['total_cost'] ?? 0, 2) ?>€</div>
                        <div class="stat-label">Coût total</div>
                    </div>
                    <div class="stat-card realtime">
                        <span class="stat-icon">⚡</span>
                        <div class="stat-value" id="avgTime"><?= round($global_stats['avg_processing_time'] ?? 0) ?>ms</div>
                        <div class="stat-label">Temps moyen</div>
                    </div>
                    <div class="stat-card">
                        <span class="stat-icon">❌</span>
                        <div class="stat-value"><?= $global_stats['failed_requests'] ?? 0 ?></div>
                        <div class="stat-label">Échecs</div>
                    </div>
                    <div class="stat-card">
                        <span class="stat-icon">🤖</span>
                        <div class="stat-value"><?= $global_stats['profiles_used'] ?? 0 ?></div>
                        <div class="stat-label">Profils utilisés</div>
                    </div>
                </div>
                
            </div>

            <!-- TAB 2: Guide Simple -->
            <div id="tab-guide-simple" class="tab-content">
                <div style="background: #f8f9fa; border-radius: 15px; padding: 25px; margin-bottom: 20px;">
                    <h2 style="color: #4ecdc4; margin-bottom: 20px; text-align: center;">🎯 QU'EST-CE QU'UN TOKEN ?</h2>
                    
                    <div style="text-align: center; margin-bottom: 30px;">
                        <div style="font-size: 48px; margin-bottom: 15px;">🔧</div>
                        <h3 style="color: #4ecdc4; margin-bottom: 10px;">Version Ultra-Simple</h3>
                        <p style="font-size: 18px; color: #666;">
                            Un <strong>token</strong> = un <strong>petit bout de mot</strong> que l'IA doit traiter
                        </p>
                    </div>

                    <div style="background: #fff3cd; border: 2px solid #ffc107; border-radius: 15px; padding: 25px; margin-bottom: 30px;">
                        <h3 style="color: #856404; text-align: center; margin-bottom: 15px;">🍽️ Analogie Restaurant</h3>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; text-align: center;">
                            <div>
                                <div style="font-size: 32px;">🍽️</div>
                                <strong>Restaurant</strong>
                                <br><small>Service IA (ChatGPT)</small>
                            </div>
                            <div>
                                <div style="font-size: 32px;">📏</div>
                                <strong>Portion</strong>
                                <br><small>Token</small>
                            </div>
                            <div>
                                <div style="font-size: 32px;">🧾</div>
                                <strong>Addition</strong>
                                <br><small>Coût basé sur portions</small>
                            </div>
                        </div>
                        <p style="text-align: center; margin-top: 15px; font-weight: 600; color: #856404;">
                            Plus vous commandez de portions, plus ça coûte !
                        </p>
                    </div>

                    <div style="background: #d4edda; border: 2px solid #28a745; border-radius: 15px; padding: 25px; text-align: center;">
                        <h3 style="color: #155724; margin-bottom: 15px;">💡 FORMULE MAGIQUE</h3>
                        <div style="font-size: 20px; font-weight: 700; color: #155724;">
                            Nombre de mots × 1.3 = Tokens approximatifs
                        </div>
                    </div>
                </div>
            </div>

            <!-- TAB 3: Calculateur -->
            <div id="tab-calculateur" class="tab-content">
                <div style="background: linear-gradient(135deg, #ff6b6b, #ff8e53); color: white; border-radius: 15px; padding: 25px;">
                    <h2 style="text-align: center; margin-bottom: 20px;">💰 Calculateur Coûts en Temps Réel</h2>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div>
                            <h3 style="margin-bottom: 20px;">📊 Votre Usage Mensuel</h3>
                            
                            <div style="margin-bottom: 20px;">
                                <label style="display: block; margin-bottom: 8px; font-weight: 600;">📱 Posts Instagram/Facebook</label>
                                <input type="number" id="calc-posts" value="30" min="0" style="width: 100%; padding: 12px 15px; border: none; border-radius: 8px;">
                            </div>
                            
                            <div style="margin-bottom: 20px;">
                                <label style="display: block; margin-bottom: 8px; font-weight: 600;">📄 Articles de blog</label>
                                <input type="number" id="calc-articles" value="4" min="0" style="width: 100%; padding: 12px 15px; border: none; border-radius: 8px;">
                            </div>
                            
                            <div style="margin-bottom: 20px;">
                                <label style="display: block; margin-bottom: 8px; font-weight: 600;">💬 Réponses clients</label>
                                <input type="number" id="calc-emails" value="50" min="0" style="width: 100%; padding: 12px 15px; border: none; border-radius: 8px;">
                            </div>
                        </div>

                        <div>
                            <h3 style="margin-bottom: 20px;">🎯 Coût Total Mensuel</h3>
                            
                            <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 10px; margin-bottom: 10px; display: flex; justify-content: space-between; align-items: center;">
                                <span>📊 Total Tokens</span>
                                <span id="calc-totalTokens">0</span>
                            </div>
                            
                            <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 10px; margin-bottom: 10px; display: flex; justify-content: space-between; align-items: center;">
                                <span>💰 Coût GPT-3.5</span>
                                <span id="calc-costGPT35">0€</span>
                            </div>
                            
                            <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 10px; margin-bottom: 10px; display: flex; justify-content: space-between; align-items: center;">
                                <span>💰 Coût GPT-4</span>
                                <span id="calc-costGPT4">0€</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- TAB 4: Question + Réponse -->
            <div id="tab-question-reponse" class="tab-content">
                <div style="background: linear-gradient(135deg, #ff6b6b, #ffa500); color: white; padding: 25px; border-radius: 15px; text-align: center; margin-bottom: 30px; font-size: 18px; font-weight: 700;">
                    🚨 ATTENTION : Vous payez la QUESTION + la RÉPONSE ! 🚨
                </div>

                <div style="background: #f8f9fa; border-radius: 15px; padding: 25px;">
                    <h2 style="color: #4ecdc4; margin-bottom: 20px; text-align: center;">🧮 FACTURATION : QUESTION + RÉPONSE = TOTAL</h2>
                    
                    <div style="background: #fff3cd; border: 2px solid #ffc107; border-radius: 15px; padding: 25px;">
                        <h3 style="color: #856404; margin-bottom: 20px;">📋 Exemple Concret</h3>
                        
                        <div style="background: linear-gradient(135deg, #ff6b6b, #ff8e53); color: white; padding: 20px; border-radius: 15px; text-align: center; margin-top: 50px;">
                            <h3>💸 TOTAL FACTURÉ</h3>
                            <div style="display: flex; justify-content: center; align-items: center; gap: 20px; font-size: 24px; font-weight: 700; margin: 20px 0;">
                                <span>15 tokens</span>
                                <span>+</span>
                                <span>45 tokens</span>
                                <span>=</span>
                                <span style="background: rgba(255,255,255,0.2); padding: 10px; border-radius: 10px;">60 TOKENS</span>
                            </div>
                            <p>Coût GPT-3.5 : <strong>0.00012€</strong></p>
                            <p>Coût GPT-4 : <strong>0.0036€</strong></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modal Quotas -->
        <div id="quotaModal" class="modal" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.5); backdrop-filter: blur(5px);">
            <div style="background: white; margin: 5% auto; padding: 30px; border-radius: 16px; width: 90%; max-width: 500px; box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3>⚙️ Gérer les Quotas</h3>
                    <button onclick="closeQuotaModal()" style="background: none; border: none; font-size: 24px; cursor: pointer;">×</button>
                </div>
                
                <form method="POST">
                    <input type="hidden" name="action" value="save_quota">
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 8px; font-weight: 600;">📅 Période</label>
                        <select name="quota_period" style="width: 100%; padding: 12px 15px; border: 2px solid #e9ecef; border-radius: 8px;" required>
                            <option value="daily">📆 Quotidien</option>
                            <option value="weekly">📋 Hebdomadaire</option>
                            <option value="monthly" selected>📊 Mensuel</option>
                            <option value="yearly">📈 Annuel</option>
                        </select>
                    </div>
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 8px; font-weight: 600;">🎯 Limite Tokens</label>
                        <input type="number" name="tokens_limit" style="width: 100%; padding: 12px 15px; border: 2px solid #e9ecef; border-radius: 8px;" 
                               value="10000" min="100" max="1000000" required>
                    </div>
                    
                    <div style="margin-bottom: 20px;">
                        <label style="display: block; margin-bottom: 8px; font-weight: 600;">💰 Limite Coût (€)</label>
                        <input type="number" name="cost_limit" style="width: 100%; padding: 12px 15px; border: 2px solid #e9ecef; border-radius: 8px;" 
                               value="10.00" min="0.01" max="1000" step="0.01" required>
                    </div>
                    
                    <div style="display: flex; gap: 15px; margin-top: 25px;">
                        <button type="submit" class="btn btn-primary">✅ Enregistrer</button>
                        <button type="button" onclick="closeQuotaModal()" class="btn btn-secondary">❌ Annuler</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Gestion onglets
        function switchTab(tabName) {
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active');
            });
            
            document.getElementById('tab-' + tabName).classList.add('active');
            event.target.classList.add('active');
        }

        // Configuration calculateur
        const TOKEN_ESTIMATES = {
            post: 30,
            article: 650,
            email: 25,
            rewrite: 100
        };

        const COSTS = {
            gpt35: 0.002,
            gpt4: 0.06
        };

        function updateCalculator() {
            const posts = parseInt(document.getElementById('calc-posts').value) || 0;
            const articles = parseInt(document.getElementById('calc-articles').value) || 0;
            const emails = parseInt(document.getElementById('calc-emails').value) || 0;

            const totalTokens = (
                posts * TOKEN_ESTIMATES.post +
                articles * TOKEN_ESTIMATES.article +
                emails * TOKEN_ESTIMATES.email
            );

            const costGPT35 = (totalTokens / 1000) * COSTS.gpt35;
            const costGPT4 = (totalTokens / 1000) * COSTS.gpt4;

            document.getElementById('calc-totalTokens').textContent = totalTokens.toLocaleString();
            document.getElementById('calc-costGPT35').textContent = costGPT35.toFixed(4) + '€';
            document.getElementById('calc-costGPT4').textContent = costGPT4.toFixed(4) + '€';
        }

        // Event listeners calculateur
        document.getElementById('calc-posts').addEventListener('input', updateCalculator);
        document.getElementById('calc-articles').addEventListener('input', updateCalculator);
        document.getElementById('calc-emails').addEventListener('input', updateCalculator);

        // Gestion modal
        function openQuotaModal() {
            document.getElementById('quotaModal').style.display = 'block';
        }

        function closeQuotaModal() {
            document.getElementById('quotaModal').style.display = 'none';
        }

        window.onclick = function(event) {
            const modal = document.getElementById('quotaModal');
            if (event.target === modal) {
                closeQuotaModal();
            }
        }

        // Système temps réel
        let refreshInterval;

        function startRealtimeUpdates() {
            refreshInterval = setInterval(updateRealtimeData, 10000);
            updateRealtimeData();
        }

        function updateRealtimeData() {
            fetch('?api=realtime')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        if (data.today_stats) {
                            updateStatCard('todayRequests', data.today_stats.requests_today || 0);
                            updateStatCard('todayTokens', data.today_stats.tokens_today || 0);
                            updateStatCard('todayCost', (data.today_stats.cost_today || 0).toFixed(2) + '€');
                            updateStatCard('avgTime', Math.round(data.today_stats.avg_time_today || 0) + 'ms');
                        }
                        
                        document.getElementById('lastRefresh').textContent = 
                            'Mis à jour : ' + new Date().toLocaleTimeString();
                    }
                })
                .catch(error => {
                    console.error('Erreur mise à jour temps réel:', error);
                    document.getElementById('lastRefresh').textContent = 
                        'Erreur connexion - ' + new Date().toLocaleTimeString();
                });
        }

        function updateStatCard(elementId, value) {
            const element = document.getElementById(elementId);
            if (element) {
                const currentValue = element.textContent;
                const newValue = typeof value === 'number' ? 
                    (value > 999 ? value.toLocaleString() : value.toString()) : value;
                
                if (currentValue !== newValue) {
                    element.style.transform = 'scale(1.1)';
                    element.style.color = '#28a745';
                    element.textContent = newValue;
                    
                    setTimeout(() => {
                        element.style.transform = 'scale(1)';
                        element.style.color = '#4ecdc4';
                    }, 300);
                }
            }
        }

        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            startRealtimeUpdates();
            updateCalculator();
            
            document.addEventListener('visibilitychange', function() {
                if (document.hidden) {
                    if (refreshInterval) {
                        clearInterval(refreshInterval);
                    }
                } else {
                    startRealtimeUpdates();
                }
            });
        });

        window.addEventListener('beforeunload', function() {
            if (refreshInterval) {
                clearInterval(refreshInterval);
            }
        });

        console.log('📊 SMG Studio Token Dashboard + DÉBIT AUTO Ready!');
        console.log('✅ Real-time Updates Every 10s');
        console.log('💰 Live Cost Tracking + Auto Debit');
        console.log('🧪 Test Debit System Active!');
    </script>
</body>
</html>