<?php
/**
 * Flying Book Creator - Configuration centralisée
 * /document/creator/config.php
 */

// Configuration des chemins
define('FLYING_BOOK_ROOT', __DIR__);
define('FLYING_BOOK_PROJECTS_DIR', FLYING_BOOK_ROOT . '/../projects/');
define('FLYING_BOOK_TEMP_DIR', FLYING_BOOK_ROOT . '/../temp/');
define('FLYING_BOOK_TEMPLATES_DIR', FLYING_BOOK_ROOT . '/templates/');
define('FLYING_BOOK_UPLOADS_DIR', FLYING_BOOK_ROOT . '/../uploads/');

// Configuration des URLs
define('FLYING_BOOK_BASE_URL', '/document/creator/');
define('FLYING_BOOK_PROJECTS_URL', '/document/projects/');
define('FLYING_BOOK_SHARE_URL', '/document/share/');

// Configuration des limites par défaut
const FLYING_BOOK_DEFAULT_LIMITS = [
    'free' => [
        'max_projects' => 3,
        'max_slides' => 10,
        'max_storage_mb' => 100,
        'max_shares' => 5,
        'max_templates' => 3,
        'watermark' => true,
        'analytics_basic' => true,
        'export_formats' => ['zip'],
        'support_level' => 'community'
    ],
    'pro' => [
        'max_projects' => 25,
        'max_slides' => 100,
        'max_storage_mb' => 1000,
        'max_shares' => 50,
        'max_templates' => 25,
        'watermark' => false,
        'analytics_advanced' => true,
        'export_formats' => ['zip', 'html', 'pdf'],
        'support_level' => 'email',
        'custom_domain' => true
    ],
    'business' => [
        'max_projects' => 100,
        'max_slides' => 500,
        'max_storage_mb' => 5000,
        'max_shares' => 200,
        'max_templates' => 100,
        'watermark' => false,
        'analytics_advanced' => true,
        'analytics_custom' => true,
        'export_formats' => ['zip', 'html', 'pdf', 'scorm'],
        'support_level' => 'priority',
        'custom_domain' => true,
        'white_label' => true,
        'api_access' => true
    ],
    'enterprise' => [
        'max_projects' => -1, // Illimité
        'max_slides' => -1,
        'max_storage_mb' => -1,
        'max_shares' => -1,
        'max_templates' => -1,
        'watermark' => false,
        'analytics_advanced' => true,
        'analytics_custom' => true,
        'export_formats' => ['zip', 'html', 'pdf', 'scorm', 'xapi'],
        'support_level' => 'dedicated',
        'custom_domain' => true,
        'white_label' => true,
        'api_access' => true,
        'sso' => true,
        'dedicated_server' => true
    ]
];

// Configuration des fonctionnalités par plan
const FLYING_BOOK_FEATURES = [
    'free' => [
        'projects' => true,
        'slides' => true,
        'templates_basic' => true,
        'export_zip' => true,
        'share_links' => true,
        'analytics_basic' => true,
        'community_support' => true
    ],
    'pro' => [
        'projects' => true,
        'slides' => true,
        'templates_basic' => true,
        'templates_premium' => true,
        'export_zip' => true,
        'export_html' => true,
        'share_links' => true,
        'share_password' => true,
        'analytics_basic' => true,
        'analytics_advanced' => true,
        'custom_branding' => true,
        'email_support' => true,
        'powerpoint_import' => true
    ],
    'business' => [
        'projects' => true,
        'slides' => true,
        'templates_basic' => true,
        'templates_premium' => true,
        'templates_custom' => true,
        'export_zip' => true,
        'export_html' => true,
        'export_pdf' => true,
        'export_scorm' => true,
        'share_links' => true,
        'share_password' => true,
        'share_tracking' => true,
        'analytics_basic' => true,
        'analytics_advanced' => true,
        'analytics_custom' => true,
        'custom_branding' => true,
        'white_label' => true,
        'priority_support' => true,
        'powerpoint_import' => true,
        'api_access' => true,
        'team_collaboration' => true
    ],
    'enterprise' => [
        'projects' => true,
        'slides' => true,
        'templates_basic' => true,
        'templates_premium' => true,
        'templates_custom' => true,
        'export_zip' => true,
        'export_html' => true,
        'export_pdf' => true,
        'export_scorm' => true,
        'export_xapi' => true,
        'share_links' => true,
        'share_password' => true,
        'share_tracking' => true,
        'share_analytics' => true,
        'analytics_basic' => true,
        'analytics_advanced' => true,
        'analytics_custom' => true,
        'custom_branding' => true,
        'white_label' => true,
        'dedicated_support' => true,
        'powerpoint_import' => true,
        'api_access' => true,
        'team_collaboration' => true,
        'sso_integration' => true,
        'dedicated_infrastructure' => true,
        'custom_integrations' => true
    ]
];

// Configuration des types de fichiers autorisés
const FLYING_BOOK_ALLOWED_FILE_TYPES = [
    'images' => [
        'extensions' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
        'mime_types' => ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml'],
        'max_size' => 10 * 1024 * 1024 // 10MB
    ],
    'videos' => [
        'extensions' => ['mp4', 'webm', 'mov', 'avi'],
        'mime_types' => ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo'],
        'max_size' => 100 * 1024 * 1024 // 100MB
    ],
    'audio' => [
        'extensions' => ['mp3', 'wav', 'ogg', 'm4a'],
        'mime_types' => ['audio/mpeg', 'audio/wav', 'audio/ogg', 'audio/mp4'],
        'max_size' => 20 * 1024 * 1024 // 20MB
    ],
    'documents' => [
        'extensions' => ['pdf', 'doc', 'docx', 'ppt', 'pptx'],
        'mime_types' => [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation'
        ],
        'max_size' => 50 * 1024 * 1024 // 50MB
    ]
];

// Configuration des thèmes disponibles
const FLYING_BOOK_THEMES = [
    'moderne' => [
        'name' => 'Moderne',
        'description' => 'Design contemporain avec gradients et animations',
        'preview' => 'themes/moderne/preview.jpg',
        'css_file' => 'themes/moderne/style.css',
        'is_premium' => false
    ],
    'classique' => [
        'name' => 'Classique',
        'description' => 'Style traditionnel et élégant',
        'preview' => 'themes/classique/preview.jpg',
        'css_file' => 'themes/classique/style.css',
        'is_premium' => false
    ],
    'corporate' => [
        'name' => 'Corporate',
        'description' => 'Professionnel pour présentations d\'entreprise',
        'preview' => 'themes/corporate/preview.jpg',
        'css_file' => 'themes/corporate/style.css',
        'is_premium' => false
    ],
    'minimal' => [
        'name' => 'Minimal',
        'description' => 'Design épuré et minimaliste',
        'preview' => 'themes/minimal/preview.jpg',
        'css_file' => 'themes/minimal/style.css',
        'is_premium' => false
    ],
    'premium' => [
        'name' => 'Premium',
        'description' => 'Design haut de gamme avec effets avancés',
        'preview' => 'themes/premium/preview.jpg',
        'css_file' => 'themes/premium/style.css',
        'is_premium' => true
    ]
];

// Configuration des animations PowerPoint supportées
const FLYING_BOOK_POWERPOINT_ANIMATIONS = [
    'entrance' => [
        'fadeIn' => ['fade-in', '0.5s'],
        'slideInLeft' => ['slide-in-left', '0.5s'],
        'slideInRight' => ['slide-in-right', '0.5s'],
        'slideInUp' => ['slide-in-up', '0.5s'],
        'slideInDown' => ['slide-in-down', '0.5s'],
        'bounceIn' => ['bounce-in', '0.6s'],
        'zoomIn' => ['zoom-in', '0.5s'],
        'rotateIn' => ['rotate-in', '0.6s'],
        'flipInX' => ['flip-in-x', '0.6s'],
        'flipInY' => ['flip-in-y', '0.6s']
    ],
    'exit' => [
        'fadeOut' => ['fade-out', '0.5s'],
        'slideOutLeft' => ['slide-out-left', '0.5s'],
        'slideOutRight' => ['slide-out-right', '0.5s'],
        'slideOutUp' => ['slide-out-up', '0.5s'],
        'slideOutDown' => ['slide-out-down', '0.5s'],
        'bounceOut' => ['bounce-out', '0.6s'],
        'zoomOut' => ['zoom-out', '0.5s'],
        'rotateOut' => ['rotate-out', '0.6s']
    ],
    'emphasis' => [
        'pulse' => ['pulse', '1s'],
        'bounce' => ['bounce', '1s'],
        'shake' => ['shake', '0.8s'],
        'flash' => ['flash', '1s'],
        'rubberBand' => ['rubber-band', '1s'],
        'swing' => ['swing', '1s']
    ]
];

// Configuration des rate limits
const FLYING_BOOK_RATE_LIMITS = [
    'save_project' => ['max' => 60, 'window' => 3600], // 60 saves par heure
    'upload_file' => ['max' => 30, 'window' => 1800],  // 30 uploads par 30min
    'save_slide' => ['max' => 120, 'window' => 3600],  // 120 slides par heure
    'generate_share_link' => ['max' => 10, 'window' => 3600], // 10 liens par heure
    'duplicate_project' => ['max' => 5, 'window' => 3600],     // 5 duplications par heure
    'load_templates' => ['max' => 100, 'window' => 3600],      // 100 chargements par heure
    'export_zip' => ['max' => 10, 'window' => 3600]           // 10 exports par heure
];

// Configuration des erreurs
const FLYING_BOOK_ERROR_MESSAGES = [
    'CSRF_INVALID' => 'Token de sécurité invalide. Veuillez recharger la page.',
    'SESSION_EXPIRED' => 'Votre session a expiré. Veuillez vous reconnecter.',
    'RATE_LIMIT_EXCEEDED' => 'Trop de requêtes. Veuillez patienter quelques instants.',
    'FILE_TOO_LARGE' => 'Fichier trop volumineux pour votre plan.',
    'STORAGE_LIMIT_EXCEEDED' => 'Limite de stockage atteinte. Upgradez votre plan.',
    'PROJECT_LIMIT_EXCEEDED' => 'Limite de projets atteinte. Upgradez votre plan.',
    'SLIDES_LIMIT_EXCEEDED' => 'Limite de slides atteinte. Upgradez votre plan.',
    'INVALID_FILE_TYPE' => 'Type de fichier non autorisé.',
    'PROJECT_NOT_FOUND' => 'Projet non trouvé ou accès refusé.',
    'INSUFFICIENT_PERMISSIONS' => 'Permissions insuffisantes.',
    'POWERPOINT_PROCESSING_FAILED' => 'Erreur lors du traitement du fichier PowerPoint.',
    'EXPORT_FAILED' => 'Erreur lors de la génération de l\'export.'
];

// Configuration des templates par défaut
const FLYING_BOOK_DEFAULT_TEMPLATES = [
    [
        'name' => 'Présentation Corporate',
        'category' => 'business',
        'description' => 'Template professionnel pour présentations d\'entreprise',
        'is_premium' => false,
        'config' => [
            'style' => 'corporate',
            'transitions' => '3d',
            'navigation' => 'all',
            'slides' => [
                ['type' => 'title', 'title' => 'Titre de votre présentation'],
                ['type' => 'content', 'title' => 'À propos'],
                ['type' => 'content', 'title' => 'Nos services'],
                ['type' => 'content', 'title' => 'Contact']
            ]
        ]
    ],
    [
        'name' => 'Portfolio Créatif',
        'category' => 'portfolio',
        'description' => 'Parfait pour présenter vos créations',
        'is_premium' => false,
        'config' => [
            'style' => 'moderne',
            'transitions' => '3d',
            'navigation' => 'thumbnails',
            'slides' => [
                ['type' => 'title', 'title' => 'Mon Portfolio'],
                ['type' => 'gallery', 'title' => 'Mes créations'],
                ['type' => 'content', 'title' => 'À propos de moi'],
                ['type' => 'contact', 'title' => 'Me contacter']
            ]
        ]
    ]
];

/**
 * Classe utilitaire pour la configuration Flying Book
 */
class FlyingBookConfig {
    
    public static function getLimits($planType) {
        return FLYING_BOOK_DEFAULT_LIMITS[$planType] ?? FLYING_BOOK_DEFAULT_LIMITS['free'];
    }
    
    public static function getFeatures($planType) {
        return FLYING_BOOK_FEATURES[$planType] ?? FLYING_BOOK_FEATURES['free'];
    }
    
    public static function hasFeature($planType, $feature) {
        $features = self::getFeatures($planType);
        return isset($features[$feature]) && $features[$feature] === true;
    }
    
    public static function isFileTypeAllowed($filename, $planType = 'free') {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        foreach (FLYING_BOOK_ALLOWED_FILE_TYPES as $category => $config) {
            if (in_array($extension, $config['extensions'])) {
                return true;
            }
        }
        
        return false;
    }
    
    public static function getMaxFileSize($fileType, $planType = 'free') {
        foreach (FLYING_BOOK_ALLOWED_FILE_TYPES as $category => $config) {
            if (in_array($fileType, $config['mime_types'])) {
                return $config['max_size'];
            }
        }
        
        return 1024 * 1024; // 1MB par défaut
    }
    
    public static function getTheme($themeName) {
        return FLYING_BOOK_THEMES[$themeName] ?? FLYING_BOOK_THEMES['moderne'];
    }
    
    public static function getAvailableThemes($planType = 'free') {
        $themes = [];
        $isPremium = $planType !== 'free';
        
        foreach (FLYING_BOOK_THEMES as $key => $theme) {
            if (!$theme['is_premium'] || $isPremium) {
                $themes[$key] = $theme;
            }
        }
        
        return $themes;
    }
    
    public static function getRateLimit($action) {
        return FLYING_BOOK_RATE_LIMITS[$action] ?? ['max' => 100, 'window' => 3600];
    }
    
    public static function getErrorMessage($errorCode) {
        return FLYING_BOOK_ERROR_MESSAGES[$errorCode] ?? 'Une erreur est survenue.';
    }
    
    public static function createProjectStructure($projectFolder) {
        $basePath = FLYING_BOOK_PROJECTS_DIR . $projectFolder;
        
        $folders = [
            $basePath,
            $basePath . '/images',
            $basePath . '/videos',
            $basePath . '/audio',
            $basePath . '/thumbnails',
            $basePath . '/exports'
        ];
        
        foreach ($folders as $folder) {
            if (!file_exists($folder)) {
                mkdir($folder, 0755, true);
            }
        }
        
        // Créer un fichier .htaccess pour sécuriser
        $htaccessContent = "# Flying Book Creator - Protection\n";
        $htaccessContent .= "Options -Indexes\n";
        $htaccessContent .= "DirectoryIndex index.html\n";
        
        file_put_contents($basePath . '/.htaccess', $htaccessContent);
        
        return $basePath;
    }
    
    public static function validateProjectData($data, $planType = 'free') {
        $errors = [];
        
        // Validation nom du projet
        if (empty($data['project_name'])) {
            $errors[] = 'Le nom du projet est obligatoire';
        } elseif (strlen($data['project_name']) > 255) {
            $errors[] = 'Le nom du projet est trop long (max 255 caractères)';
        } elseif (!preg_match('/^[a-zA-Z0-9\s\-_]+$/', $data['project_name'])) {
            $errors[] = 'Le nom du projet contient des caractères non autorisés';
        }
        
        // Validation titre
        if (empty($data['title'])) {
            $errors[] = 'Le titre est obligatoire';
        } elseif (strlen($data['title']) > 255) {
            $errors[] = 'Le titre est trop long (max 255 caractères)';
        }
        
        // Validation thème
        if (!empty($data['style_theme'])) {
            $availableThemes = self::getAvailableThemes($planType);
            if (!isset($availableThemes[$data['style_theme']])) {
                $errors[] = 'Thème non disponible pour votre plan';
            }
        }
        
        return $errors;
    }
}

// Initialisation des dossiers si ils n'existent pas
if (!file_exists(FLYING_BOOK_PROJECTS_DIR)) {
    mkdir(FLYING_BOOK_PROJECTS_DIR, 0755, true);
}

if (!file_exists(FLYING_BOOK_TEMP_DIR)) {
    mkdir(FLYING_BOOK_TEMP_DIR, 0755, true);
}

if (!file_exists(FLYING_BOOK_UPLOADS_DIR)) {
    mkdir(FLYING_BOOK_UPLOADS_DIR, 0755, true);
}

?>