<?php
/**
 * auth_check.php - Système d'authentification Radio SMG Studio
 * 
 * Utilise le fichier .env de la racine https://laloee-cours.fr/s/.env
 * À placer dans : https://laloee-cours.fr/config/auth_check.php
 * 
 * @author Radio SMG Studio v2.0 + AI Social Media Generator
 * @version Audiovizu Edition - FIXED
 */

// Démarrage de session sécurisée
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', 1);
    ini_set('session.use_strict_mode', 1);
    session_start();
}

/**
 * Charge le fichier .env depuis la racine
 */
function loadEnvFile() {
    $envPath = $_SERVER['DOCUMENT_ROOT'] . '/s/.env';
    
    if (!file_exists($envPath)) {
        error_log("Fichier .env introuvable : $envPath");
        die("Configuration manquante");
    }
    
    $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    foreach ($lines as $line) {
        // Ignorer les commentaires
        if (strpos(trim($line), '#') === 0) {
            continue;
        }
        
        // Parser les variables KEY=VALUE
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);
            
            // Nettoyer les guillemets
            $value = trim($value, '"\'');
            
            // Définir la variable d'environnement
            if (!array_key_exists($key, $_ENV)) {
                $_ENV[$key] = $value;
                putenv("$key=$value");
            }
        }
    }
}

// Charger les variables d'environnement
loadEnvFile();

/**
 * Configuration de la base de données depuis .env
 */
class DatabaseConfig {
    private static $pdo = null;
    
    public static function getConnection() {
        if (self::$pdo === null) {
            try {
                $host = $_ENV['DB_HOST'] ?? 'localhost';
                $dbname = $_ENV['DB_NAME'] ?? 'laloedata';
                $username = $_ENV['DB_USER'] ?? 'laloedata';
                $password = $_ENV['DB_PASS'] ?? 'PiL256Mx78!';
                $port = $_ENV['DB_PORT'] ?? '3306';
                $charset = $_ENV['DB_CHARSET'] ?? 'utf8mb4';
                
                $dsn = "mysql:host=$host;port=$port;dbname=$dbname;charset=$charset";
                
                self::$pdo = new PDO($dsn, $username, $password, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES $charset"
                ]);
                
            } catch (PDOException $e) {
                error_log("Erreur DB: " . $e->getMessage());
                
                // Mode production : message générique
                if (($_ENV['APP_ENV'] ?? 'production') === 'production') {
                    die("Service temporairement indisponible");
                } else {
                    die("Erreur DB: " . $e->getMessage());
                }
            }
        }
        
        return self::$pdo;
    }
}

/**
 * Classe de gestion de l'authentification
 */
class Auth {
    private static $pdo;
    
    public static function init() {
        self::$pdo = DatabaseConfig::getConnection();
    }
    
    /**
     * Vérifier si l'utilisateur est connecté
     */
    public static function check() {
        return isset($_SESSION['logged_in']) && 
               $_SESSION['logged_in'] === true && 
               isset($_SESSION['user_id']) && 
               is_numeric($_SESSION['user_id']);
    }
    
    /**
     * Rediriger vers login si pas connecté
     */
    public static function requireAuth($redirectUrl = '/login.php') {
        if (!self::check()) {
            // Sauvegarder l'URL demandée pour redirection après login
            $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
            
            header("Location: $redirectUrl");
            exit;
        }
        
        // Vérifier la validité de la session
        self::validateSession();
    }
    
    /**
     * Valider la session active
     */
    private static function validateSession() {
        // Timeout de session
        $sessionTimeout = (int)($_ENV['SESSION_TIMEOUT'] ?? 7200); // 2 heures par défaut
        
        if (isset($_SESSION['last_activity'])) {
            if (time() - $_SESSION['last_activity'] > $sessionTimeout) {
                self::logout();
                header('Location: /login.php?timeout=1');
                exit;
            }
        }
        
        $_SESSION['last_activity'] = time();
        
        // Vérifier que l'utilisateur existe toujours et est actif
        if (isset($_SESSION['user_id'])) {
            $stmt = self::$pdo->prepare("
                SELECT id, status 
                FROM users 
                WHERE id = ? AND status = 'active'
            ");
            $stmt->execute([$_SESSION['user_id']]);
            
            if (!$stmt->fetch()) {
                self::logout();
                header('Location: /login.php?inactive=1');
                exit;
            }
        }
    }
    
    /**
     * Tentative de connexion
     */
    public static function attempt($email, $password) {
        self::init();
        
        // Limitation des tentatives
        $maxAttempts = (int)($_ENV['MAX_LOGIN_ATTEMPTS'] ?? 5);
        $rateLimitWindow = (int)($_ENV['RATE_LIMIT_WINDOW'] ?? 900); // 15 minutes
        
        // Vérifier les tentatives récentes
        $stmt = self::$pdo->prepare("
            SELECT COUNT(*) as attempts 
            FROM login_attempts 
            WHERE ip_address = ? 
            AND attempted_at > DATE_SUB(NOW(), INTERVAL ? SECOND)
        ");
        $stmt->execute([$_SERVER['REMOTE_ADDR'], $rateLimitWindow]);
        $attempts = $stmt->fetchColumn();
        
        if ($attempts >= $maxAttempts) {
            self::logAttempt($email, false, 'Rate limit exceeded');
            return ['success' => false, 'error' => 'Trop de tentatives. Réessayez dans 15 minutes.'];
        }
        
        // Rechercher l'utilisateur
        $stmt = self::$pdo->prepare("
            SELECT id, name, email, password, role, status, last_login
            FROM users 
            WHERE email = ?
        ");
        $stmt->execute([trim($email)]);
        $user = $stmt->fetch();
        
        if (!$user) {
            self::logAttempt($email, false, 'User not found');
            return ['success' => false, 'error' => 'Email ou mot de passe incorrect'];
        }
        
        if ($user['status'] !== 'active') {
            self::logAttempt($email, false, 'User inactive: ' . $user['status']);
            return ['success' => false, 'error' => 'Compte désactivé'];
        }
        
        // Vérifier le mot de passe
        if (!password_verify($password, $user['password'])) {
            self::logAttempt($email, false, 'Invalid password');
            return ['success' => false, 'error' => 'Email ou mot de passe incorrect'];
        }
        
        // Connexion réussie
        self::createSession($user);
        self::logAttempt($email, true, 'Login successful');
        self::logConnection($user['id'], 'login');
        
        // Mettre à jour last_login
        $stmt = self::$pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$user['id']]);
        
        return ['success' => true, 'user' => $user];
    }
    
    /**
     * Créer la session utilisateur
     */
    private static function createSession($user) {
        // Régénérer l'ID de session pour sécurité
        session_regenerate_id(true);
        
        $_SESSION['logged_in'] = true;
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();
        
        // Token CSRF
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    
    /**
     * Déconnexion
     */
    public static function logout() {
        if (isset($_SESSION['user_id'])) {
            self::init();
            self::logConnection($_SESSION['user_id'], 'logout');
        }
        
        // Détruire la session
        session_destroy();
        session_start();
        session_regenerate_id(true);
    }
    
    /**
     * Obtenir l'utilisateur connecté
     */
    public static function user() {
        if (!self::check()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'name' => $_SESSION['user_name'],
            'email' => $_SESSION['user_email'],
            'role' => $_SESSION['user_role']
        ];
    }
    
    /**
     * Vérifier le rôle utilisateur
     */
    public static function hasRole($role) {
        $user = self::user();
        return $user && $user['role'] === $role;
    }
    
    /**
     * Vérifier si admin
     */
    public static function isAdmin() {
        return self::hasRole('admin');
    }
    
    /**
     * Log des tentatives de connexion
     */
    private static function logAttempt($email, $success, $reason = null) {
        try {
            $stmt = self::$pdo->prepare("
                INSERT INTO login_attempts 
                (email, ip_address, user_agent, success, failure_reason, attempted_at) 
                VALUES (?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $email,
                $_SERVER['REMOTE_ADDR'],
                $_SERVER['HTTP_USER_AGENT'] ?? '',
                $success ? 1 : 0,
                $reason
            ]);
        } catch (Exception $e) {
            error_log("Erreur log tentative: " . $e->getMessage());
        }
    }
    
    /**
     * Log des connexions/déconnexions
     */
    private static function logConnection($userId, $action) {
        try {
            $stmt = self::$pdo->prepare("
                INSERT INTO log_connexions 
                (user_id, ip, user_agent, action, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $userId,
                $_SERVER['REMOTE_ADDR'],
                $_SERVER['HTTP_USER_AGENT'] ?? '',
                $action
            ]);
        } catch (Exception $e) {
            error_log("Erreur log connexion: " . $e->getMessage());
        }
    }
    
    /**
     * Générer un token CSRF
     */
    public static function csrfToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
    
    /**
     * Vérifier le token CSRF
     */
    public static function verifyCsrfToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

// Initialiser la connexion DB
Auth::init();

/**
 * Fonctions helper pour compatibilité - VERSION SIMPLE
 */

// Vérifier l'authentification (fonction simple)
function checkAuth() {
    if (!isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
        header('Location: /login.php');
        exit;
    }
}

// Obtenir l'utilisateur connecté
function getUser() {
    return Auth::user();
}

// Vérifier si connecté
function isLoggedIn() {
    return Auth::check();
}

// Vérifier si admin
function isAdmin() {
    return Auth::isAdmin();
}

// Obtenir les infos de configuration
function getConfig($key, $default = null) {
    return $_ENV[$key] ?? $default;
}

/**
 * Protection CSRF pour les formulaires
 */
function csrfField() {
    $token = Auth::csrfToken();
    return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token) . '">';
}

// Constantes pour compatibilité (si besoin)
if (!defined('DB_HOST')) {
    define('DB_HOST', $_ENV['DB_HOST'] ?? 'localhost');
    define('DB_NAME', $_ENV['DB_NAME'] ?? 'laloedata');
    define('DB_USER', $_ENV['DB_USER'] ?? 'laloedata');
    define('DB_PASS', $_ENV['DB_PASS'] ?? 'PiL256Mx78!');
}

?>