<?php
/**
 * ================================================================
 * 🗄️ CRÉATION TABLES SMG - VERSION SÉCURISÉE MINIMALE
 * ================================================================
 * 
 * 🎯 OBJECTIF :
 * - Créer tables post_networks et post_medias
 * - Sans vérification session (mode setup)
 * - Script d'urgence pour déblocage
 * 
 * 💡 BY PAPY 77 - SETUP RAPIDE !
 * ================================================================
 */

require_once 'config/db.php';

echo "🚀 SMG Studio - Création Tables d'Urgence\n";
echo "==========================================\n\n";

$success = 0;
$errors = 0;

// ================================================================
// 🗄️ TABLE POST_NETWORKS
// ================================================================

echo "📋 Création table post_networks...\n";

try {
    $sql_post_networks = "
        CREATE TABLE IF NOT EXISTS post_networks (
            id INT PRIMARY KEY AUTO_INCREMENT,
            post_id INT NOT NULL,
            network VARCHAR(50) NOT NULL,
            is_active BOOLEAN DEFAULT 1,
            format_settings JSON NULL,
            export_status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
            export_path VARCHAR(500) NULL,
            published_at TIMESTAMP NULL,
            platform_post_id VARCHAR(100) NULL,
            engagement_data JSON NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            INDEX idx_post_id (post_id),
            INDEX idx_network (network),
            INDEX idx_status (export_status),
            INDEX idx_active (is_active),
            
            UNIQUE KEY unique_post_network (post_id, network)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    
    $pdo->exec($sql_post_networks);
    echo "✅ Table post_networks créée\n";
    $success++;
    
} catch (Exception $e) {
    echo "❌ Erreur post_networks : " . $e->getMessage() . "\n";
    $errors++;
}

// ================================================================
// 🗄️ TABLE POST_MEDIAS
// ================================================================

echo "\n📋 Création table post_medias...\n";

try {
    $sql_post_medias = "
        CREATE TABLE IF NOT EXISTS post_medias (
            id INT PRIMARY KEY AUTO_INCREMENT,
            post_id INT NOT NULL,
            media_path VARCHAR(500) NOT NULL,
            media_type ENUM('image', 'video') NOT NULL,
            media_source ENUM('upload', 'workflow', 'external') DEFAULT 'upload',
            display_order INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            
            INDEX idx_post_order (post_id, display_order)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    
    $pdo->exec($sql_post_medias);
    echo "✅ Table post_medias créée\n";
    $success++;
    
} catch (Exception $e) {
    echo "❌ Erreur post_medias : " . $e->getMessage() . "\n";
    $errors++;
}

// ================================================================
// 🗄️ TABLE SOCIAL_PLATFORMS
// ================================================================

echo "\n📋 Création table social_platforms...\n";

try {
    $sql_social_platforms = "
        CREATE TABLE IF NOT EXISTS social_platforms (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(50) UNIQUE NOT NULL,
            display_name VARCHAR(100) NOT NULL,
            icon VARCHAR(10) DEFAULT '📱',
            color VARCHAR(7) DEFAULT '#667eea',
            character_limit INT DEFAULT 2000,
            supports_images BOOLEAN DEFAULT 1,
            supports_videos BOOLEAN DEFAULT 1,
            supports_scheduling BOOLEAN DEFAULT 1,
            is_active BOOLEAN DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    
    $pdo->exec($sql_social_platforms);
    echo "✅ Table social_platforms créée\n";
    $success++;
    
} catch (Exception $e) {
    echo "❌ Erreur social_platforms : " . $e->getMessage() . "\n";
    $errors++;
}

// ================================================================
// 📊 INSERTION RÉSEAUX PAR DÉFAUT
// ================================================================

echo "\n📋 Insertion réseaux sociaux par défaut...\n";

try {
    $networks = [
        ['instagram', 'Instagram', '📸', '#E4405F', 2200, 1, 1, 1],
        ['facebook', 'Facebook', '📘', '#1877F2', 63206, 1, 1, 1],
        ['twitter', 'Twitter', '🐦', '#1DA1F2', 280, 1, 1, 1],
        ['linkedin', 'LinkedIn', '💼', '#0077B5', 3000, 1, 1, 1],
        ['tiktok', 'TikTok', '🎵', '#FF0050', 300, 1, 1, 1],
        ['youtube', 'YouTube', '📺', '#FF0000', 5000, 1, 1, 1]
    ];
    
    $stmt = $pdo->prepare("
        INSERT INTO social_platforms 
        (name, display_name, icon, color, character_limit, supports_images, supports_videos, supports_scheduling) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE 
        display_name = VALUES(display_name),
        icon = VALUES(icon),
        color = VALUES(color)
    ");
    
    $inserted = 0;
    foreach ($networks as $network) {
        $stmt->execute($network);
        $inserted++;
    }
    
    echo "✅ {$inserted} réseaux sociaux configurés\n";
    $success++;
    
} catch (Exception $e) {
    echo "❌ Erreur insertion réseaux : " . $e->getMessage() . "\n";
    $errors++;
}

// ================================================================
// 🔧 MISE À JOUR TABLE POSTS
// ================================================================

echo "\n📋 Mise à jour table posts...\n";

$columns_to_add = [
    "multi_network BOOLEAN DEFAULT 0 COMMENT 'Post pour plusieurs réseaux'",
    "primary_network VARCHAR(50) NULL COMMENT 'Réseau principal'",
    "max_photos INT DEFAULT 5 COMMENT 'Limite photos par post'",
    "max_videos INT DEFAULT 1 COMMENT 'Limite vidéos par post'",
    "link_url VARCHAR(500) NULL COMMENT 'Lien externe'"
];

foreach ($columns_to_add as $column) {
    try {
        $pdo->exec("ALTER TABLE posts ADD COLUMN $column");
        $column_name = explode(' ', $column)[0];
        echo "✅ Colonne ajoutée : {$column_name}\n";
        $success++;
    } catch (Exception $e) {
        if (strpos($e->getMessage(), 'Duplicate column') !== false) {
            $column_name = explode(' ', $column)[0];
            echo "ℹ️  Colonne existante : {$column_name}\n";
        } else {
            echo "⚠️  Erreur colonne : " . $e->getMessage() . "\n";
            $errors++;
        }
    }
}

// ================================================================
// 📊 RÉSUMÉ FINAL
// ================================================================

echo "\n📊 RÉSUMÉ DE L'INSTALLATION\n";
echo "============================\n";
echo "✅ Succès : {$success}\n";
echo "❌ Erreurs : {$errors}\n";

if ($errors === 0) {
    echo "\n🎉 INSTALLATION RÉUSSIE !\n";
    echo "Votre système SMG multi-réseaux est prêt !\n";
} else {
    echo "\n⚠️  Installation partielle - Vérifiez les erreurs ci-dessus\n";
}

// Vérification finale
try {
    $tables = ['posts', 'post_networks', 'post_medias', 'social_platforms'];
    
    echo "\n🔍 Vérification tables :\n";
    foreach ($tables as $table) {
        $stmt = $pdo->query("SHOW TABLES LIKE '{$table}'");
        if ($stmt->rowCount() > 0) {
            echo "   ✅ {$table}\n";
        } else {
            echo "   ❌ {$table} MANQUANTE\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Erreur vérification : " . $e->getMessage() . "\n";
}

echo "\n🚀 PRÊT POUR posts.php !\n";

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>🗄️ Setup SMG Tables</title>
    <style>
        body {
            font-family: 'Consolas', monospace;
            background: #1a1a1a;
            color: #00ff00;
            padding: 20px;
            line-height: 1.4;
        }
        
        pre {
            background: #000;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #00ff00;
            margin: 10px 0;
            white-space: pre-wrap;
        }
        
        .success { color: #00ff00; }
        .error { color: #ff4444; }
        .warning { color: #ffaa00; }
        
        .summary {
            background: #2a2a2a;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            text-align: center;
        }
        
        .btn {
            display: inline-block;
            padding: 10px 20px;
            margin: 10px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <h1>🗄️ SMG Studio - Setup Tables Terminé</h1>
    
    <div class="summary">
        <h3>✅ Installation Base de Données</h3>
        <p>Tables multi-réseaux créées avec succès</p>
        <p><strong>Succès:</strong> <?= $success ?> | <strong>Erreurs:</strong> <?= $errors ?></p>
        
        <?php if ($errors === 0): ?>
            <div class="success">
                <h4>🎉 SYSTÈME OPÉRATIONNEL !</h4>
                <a href="posts.php" class="btn">📝 Tester Posts Manager</a>
            </div>
        <?php else: ?>
            <div class="warning">
                <h4>⚠️ Installation partielle</h4>
                <p>Vérifiez les erreurs et recommencez si nécessaire</p>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        console.log('🗄️ SMG Setup terminé');
        console.log('Tables créées - Système prêt');
    </script>
</body>
</html>