<?php
/**
 * ================================================================
 * 🎬 EXPORT VIDEO SCRIPT - CLIPMAKER SMG STUDIO
 * ================================================================
 * 
 * 🎯 SCRIPT D'EXPORT EN ARRIÈRE-PLAN :
 * ✅ Traitement FFmpeg optimisé Ubuntu 24
 * ✅ Gestion des transitions
 * ✅ Sous-titres intégrés
 * ✅ Qualité adaptative
 * ✅ Progress tracking
 * ✅ Logs détaillés
 * 
 * 💡 BY PAPY 77 - VPS POWER !
 * 
 * Usage: php export_video.php <export_id>
 * ================================================================
 */

// Configuration pour script CLI
ini_set('memory_limit', '1G');
ini_set('max_execution_time', 0);
set_time_limit(0);

// Vérifier si lancé depuis CLI
if (php_sapi_name() !== 'cli') {
    die('Ce script doit être exécuté depuis la ligne de commande');
}

// Vérifier les arguments
if ($argc < 2) {
    die("Usage: php export_video.php <export_id>\n");
}

$export_id = intval($argv[1]);

// Inclure les fonctions
require_once dirname(__FILE__) . '/../auth_functions.php';

/**
 * ================================================================
 * 🎬 CLASSE EXPORT VIDEO
 * ================================================================
 */
class VideoExporter {
    private $pdo;
    private $export_id;
    private $export_data;
    private $project_data;
    private $temp_dir;
    private $output_dir;
    private $ffmpeg_path;
    
    public function __construct($export_id) {
        $this->export_id = $export_id;
        $this->pdo = getDbConnection();
        $this->temp_dir = '/tmp/clipmaker_export_' . $export_id;
        $this->output_dir = dirname(__FILE__) . '/../exports/';
        $this->ffmpeg_path = '/usr/bin/ffmpeg'; // Path Ubuntu 24
        
        // Créer les dossiers
        if (!is_dir($this->temp_dir)) {
            mkdir($this->temp_dir, 0755, true);
        }
        if (!is_dir($this->output_dir)) {
            mkdir($this->output_dir, 0755, true);
        }
    }
    
    /**
     * Lancer l'export
     */
    public function export() {
        try {
            $this->log("🎬 Début export vidéo #" . $this->export_id);
            
            // Charger les données
            $this->loadExportData();
            
            // Mettre à jour le statut
            $this->updateStatus('processing', 0);
            
            // Analyser le projet
            $this->analyzeProject();
            
            // Préparer les médias
            $this->prepareMedia();
            
            // Générer la vidéo
            $this->generateVideo();
            
            // Finaliser
            $this->finalize();
            
            $this->log("✅ Export terminé avec succès");
            $this->updateStatus('completed', 100);
            
        } catch (Exception $e) {
            $this->log("❌ Erreur export: " . $e->getMessage());
            $this->updateStatus('failed', 0, $e->getMessage());
        } finally {
            // Nettoyer les fichiers temporaires
            $this->cleanup();
        }
    }
    
    /**
     * Charger les données d'export
     */
    private function loadExportData() {
        $stmt = $this->pdo->prepare("
            SELECT e.*, p.timeline_data, p.resolution_width, p.resolution_height, p.fps
            FROM clipmaker_exports e
            JOIN clipmaker_projects p ON p.id = e.project_id
            WHERE e.id = ?
        ");
        
        $stmt->execute([$this->export_id]);
        $this->export_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$this->export_data) {
            throw new Exception("Export non trouvé");
        }
        
        $this->project_data = json_decode($this->export_data['timeline_data'], true);
        $this->log("📊 Données projet chargées");
    }
    
    /**
     * Analyser le projet
     */
    private function analyzeProject() {
        if (!isset($this->project_data['clips']) || empty($this->project_data['clips'])) {
            throw new Exception("Aucun clip dans le projet");
        }
        
        $this->log("🔍 Analyse: " . count($this->project_data['clips']) . " clips trouvés");
        
        // Vérifier que tous les médias existent
        foreach ($this->project_data['clips'] as $clip) {
            if (!file_exists($clip['filePath'])) {
                throw new Exception("Média manquant: " . $clip['filePath']);
            }
        }
        
        $this->updateStatus('processing', 20);
    }
    
    /**
     * Préparer les médias
     */
    private function prepareMedia() {
        $this->log("🎥 Préparation des médias...");
        
        $resolution = $this->export_data['resolution_width'] . 'x' . $this->export_data['resolution_height'];
        $fps = $this->export_data['fps'];
        
        $prepared_clips = [];
        
        foreach ($this->project_data['clips'] as $index => $clip) {
            $input_file = $clip['filePath'];
            $output_file = $this->temp_dir . '/clip_' . $index . '.mp4';
            
            // Commande FFmpeg pour normaliser le clip
            $command = [
                $this->ffmpeg_path,
                '-i', escapeshellarg($input_file),
                '-ss', $clip['inPoint'],
                '-t', $clip['duration'],
                '-vf', "scale=$resolution:flags=lanczos",
                '-r', $fps,
                '-c:v', 'libx264',
                '-preset', 'medium',
                '-crf', '23',
                '-c:a', 'aac',
                '-b:a', '128k',
                '-y',
                escapeshellarg($output_file)
            ];
            
            $cmd = implode(' ', $command);
            $this->log("🔧 Normalisation clip: " . basename($input_file));
            
            exec($cmd . ' 2>&1', $output, $return_code);
            
            if ($return_code !== 0) {
                throw new Exception("Erreur normalisation clip: " . implode("\n", $output));
            }
            
            $prepared_clips[] = $output_file;
            
            // Progression
            $progress = 20 + (($index + 1) / count($this->project_data['clips'])) * 40;
            $this->updateStatus('processing', $progress);
        }
        
        $this->prepared_clips = $prepared_clips;
        $this->log("✅ " . count($prepared_clips) . " clips préparés");
    }
    
    /**
     * Générer la vidéo finale
     */
    private function generateVideo() {
        $this->log("🎬 Génération vidéo finale...");
        
        // Nom du fichier final
        $output_filename = $this->export_data['export_name'] . '.' . $this->export_data['format'];
        $output_path = $this->output_dir . $output_filename;
        
        if (count($this->prepared_clips) === 1) {
            // Un seul clip - copie simple
            $this->singleClipExport($output_path);
        } else {
            // Plusieurs clips - concaténation
            $this->multiClipExport($output_path);
        }
        
        // Vérifier le fichier final
        if (!file_exists($output_path)) {
            throw new Exception("Fichier final non généré");
        }
        
        $file_size = filesize($output_path);
        $this->log("📊 Fichier final: " . number_format($file_size/1024/1024, 2) . " MB");
        
        // Mettre à jour la base
        $stmt = $this->pdo->prepare("
            UPDATE clipmaker_exports 
            SET file_path = ?, file_size = ?, completed_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$output_path, $file_size, $this->export_id]);
        
        $this->updateStatus('processing', 95);
    }
    
    /**
     * Export clip unique
     */
    private function singleClipExport($output_path) {
        $input_file = $this->prepared_clips[0];
        $quality_settings = $this->getQualitySettings();
        
        $command = [
            $this->ffmpeg_path,
            '-i', escapeshellarg($input_file),
            '-c:v', 'libx264',
            '-preset', $quality_settings['preset'],
            '-crf', $quality_settings['crf'],
            '-c:a', 'aac',
            '-b:a', '128k',
            '-movflags', '+faststart',
            '-y',
            escapeshellarg($output_path)
        ];
        
        $this->runFFmpegCommand($command);
    }
    
    /**
     * Export multi-clips avec transitions
     */
    private function multiClipExport($output_path) {
        // Créer le fichier de concaténation
        $concat_file = $this->temp_dir . '/concat.txt';
        $concat_content = '';
        
        foreach ($this->prepared_clips as $clip) {
            $concat_content .= "file '" . $clip . "'\n";
        }
        
        file_put_contents($concat_file, $concat_content);
        
        $quality_settings = $this->getQualitySettings();
        
        // Commande de concaténation
        $command = [
            $this->ffmpeg_path,
            '-f', 'concat',
            '-safe', '0',
            '-i', escapeshellarg($concat_file),
            '-c:v', 'libx264',
            '-preset', $quality_settings['preset'],
            '-crf', $quality_settings['crf'],
            '-c:a', 'aac',
            '-b:a', '128k',
            '-movflags', '+faststart',
            '-y',
            escapeshellarg($output_path)
        ];
        
        $this->runFFmpegCommand($command);
    }
    
    /**
     * Obtenir les paramètres de qualité
     */
    private function getQualitySettings() {
        $quality = $this->export_data['quality'];
        
        switch ($quality) {
            case 'high':
                return ['preset' => 'medium', 'crf' => '18'];
            case 'low':
                return ['preset' => 'faster', 'crf' => '28'];
            default: // medium
                return ['preset' => 'fast', 'crf' => '23'];
        }
    }
    
    /**
     * Exécuter une commande FFmpeg
     */
    private function runFFmpegCommand($command) {
        $cmd = implode(' ', $command);
        $this->log("⚙️ FFmpeg: " . $cmd);
        
        // Sauvegarder la commande
        $stmt = $this->pdo->prepare("
            UPDATE clipmaker_exports 
            SET ffmpeg_command = ?
            WHERE id = ?
        ");
        $stmt->execute([$cmd, $this->export_id]);
        
        // Exécuter avec capture de sortie
        $output = [];
        $return_code = 0;
        
        exec($cmd . ' 2>&1', $output, $return_code);
        
        if ($return_code !== 0) {
            $error = implode("\n", $output);
            throw new Exception("Erreur FFmpeg: " . $error);
        }
        
        $this->log("✅ FFmpeg terminé avec succès");
    }
    
    /**
     * Finaliser l'export
     */
    private function finalize() {
        $this->log("🏁 Finalisation...");
        
        // Générer la miniature
        $this->generateThumbnail();
        
        // Nettoyer les métadonnées
        $this->cleanupMetadata();
        
        $this->log("✅ Export finalisé");
    }
    
    /**
     * Générer une miniature
     */
    private function generateThumbnail() {
        $output_path = $this->export_data['file_path'];
        $thumbnail_path = $this->output_dir . 'thumb_' . basename($output_path, '.mp4') . '.jpg';
        
        $command = [
            $this->ffmpeg_path,
            '-i', escapeshellarg($output_path),
            '-ss', '00:00:01',
            '-vframes', '1',
            '-vf', 'scale=320:240',
            '-y',
            escapeshellarg($thumbnail_path)
        ];
        
        try {
            $this->runFFmpegCommand($command);
            $this->log("📸 Miniature générée");
        } catch (Exception $e) {
            $this->log("⚠️ Erreur miniature: " . $e->getMessage());
        }
    }
    
    /**
     * Nettoyer les métadonnées
     */
    private function cleanupMetadata() {
        // Supprimer les métadonnées sensibles
        $output_path = $this->export_data['file_path'];
        $temp_path = $output_path . '.tmp';
        
        $command = [
            $this->ffmpeg_path,
            '-i', escapeshellarg($output_path),
            '-map_metadata', '-1',
            '-c', 'copy',
            '-y',
            escapeshellarg($temp_path)
        ];
        
        try {
            $this->runFFmpegCommand($command);
            rename($temp_path, $output_path);
            $this->log("🧹 Métadonnées nettoyées");
        } catch (Exception $e) {
            $this->log("⚠️ Erreur nettoyage: " . $e->getMessage());
        }
    }
    
    /**
     * Nettoyer les fichiers temporaires
     */
    private function cleanup() {
        if (is_dir($this->temp_dir)) {
            $this->log("🧹 Nettoyage fichiers temporaires");
            $this->deleteDirectory($this->temp_dir);
        }
    }
    
    /**
     * Supprimer un dossier récursivement
     */
    private function deleteDirectory($dir) {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->deleteDirectory($path) : unlink($path);
        }
        
        return rmdir($dir);
    }
    
    /**
     * Mettre à jour le statut
     */
    private function updateStatus($status, $progress, $error = null) {
        $stmt = $this->pdo->prepare("
            UPDATE clipmaker_exports 
            SET status = ?, progress = ?, error_message = ?, updated_at = NOW()
            WHERE id = ?
        ");
        
        $stmt->execute([$status, $progress, $error, $this->export_id]);
        
        $this->log("📊 Statut: $status ($progress%)");
    }
    
    /**
     * Logger un message
     */
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[$timestamp] $message\n";
        
        echo $log_message;
        
        // Sauvegarder dans la base
        $stmt = $this->pdo->prepare("
            UPDATE clipmaker_exports 
            SET processing_log = CONCAT(COALESCE(processing_log, ''), ?)
            WHERE id = ?
        ");
        
        $stmt->execute([$log_message, $this->export_id]);
    }
}

/**
 * ================================================================
 * 🚀 EXÉCUTION PRINCIPALE
 * ================================================================
 */

try {
    // Vérifier FFmpeg
    if (!file_exists('/usr/bin/ffmpeg')) {
        throw new Exception("FFmpeg non trouvé. Installez avec: sudo apt install ffmpeg");
    }
    
    // Lancer l'export
    $exporter = new VideoExporter($export_id);
    $exporter->export();
    
    echo "✅ Export terminé avec succès!\n";
    
} catch (Exception $e) {
    echo "❌ Erreur: " . $e->getMessage() . "\n";
    exit(1);
}

?>
