<?php
/*
 * SMG Studio v2.0 - Page de Réinitialisation de Mot de Passe CORRIGÉE
 * Path: /var/www/laloee-cours.fr/social-media-generator/reset_password.php
 */

session_start();
require_once 'config/db.php'; // ✅ Bon chemin vers la config DB

$token = $_GET['token'] ?? '';
$message = '';
$error = '';
$valid_token = false;
$user_data = null;

// Vérification du token
if (!empty($token)) {
    try {
        $stmt = $pdo->prepare("
            SELECT id, name, email 
            FROM users 
            WHERE reset_token = ? 
            AND reset_token_expires > NOW() 
            AND status = 'active'
        ");
        $stmt->execute([$token]);
        $user_data = $stmt->fetch(PDO::FETCH_ASSOC); // ✅ Mode fetch explicite
        
        if ($user_data) {
            $valid_token = true;
        } else {
            $error = "❌ Lien de réinitialisation invalide ou expiré";
        }
        
    } catch (Exception $e) {
        error_log("Erreur vérification token: " . $e->getMessage());
        $error = "❌ Erreur de vérification du token";
    }
} else {
    $error = "❌ Token de réinitialisation manquant";
}

// Traitement du nouveau mot de passe
if ($_POST && $valid_token) {
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validation des données
    if (empty($new_password) || empty($confirm_password)) {
        $error = "❌ Tous les champs sont requis";
    } elseif (strlen($new_password) < 8) {
        $error = "❌ Le mot de passe doit contenir au moins 8 caractères";
    } elseif ($new_password !== $confirm_password) {
        $error = "❌ Les mots de passe ne correspondent pas";
    } elseif (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/', $new_password)) {
        $error = "❌ Le mot de passe doit contenir au moins une majuscule, une minuscule et un chiffre";
    } else {
        try {
            // Commencer une transaction
            $pdo->beginTransaction();
            
            // Hacher le nouveau mot de passe
            $hashed_password = password_hash($new_password, PASSWORD_ARGON2ID);
            
            // Mettre à jour en base et supprimer le token
            $stmt = $pdo->prepare("
                UPDATE users 
                SET password = ?, 
                    reset_token = NULL, 
                    reset_token_expires = NULL,
                    updated_at = NOW()
                WHERE id = ?
            ");
            $result = $stmt->execute([$hashed_password, $user_data['id']]);
            
            if (!$result) {
                throw new Exception("Échec de la mise à jour du mot de passe");
            }
            
            // Log de sécurité pour les tentatives de connexion
            $stmt = $pdo->prepare("
                INSERT INTO login_attempts (email, ip_address, user_agent, success, failure_reason, attempted_at)
                VALUES (?, ?, ?, 1, 'password_reset_success', NOW())
            ");
            $stmt->execute([
                $user_data['email'],
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
            
            // Confirmer la transaction
            $pdo->commit();
            
            // Log de succès
            error_log("Mot de passe réinitialisé avec succès pour : " . $user_data['email']);
            
            // Redirection vers login avec message de succès
            header('Location: login.php?reset_success=1');
            exit;
            
        } catch (Exception $e) {
            // Annuler la transaction en cas d'erreur
            $pdo->rollBack();
            error_log("Erreur reset password pour {$user_data['email']}: " . $e->getMessage());
            $error = "❌ Erreur lors de la mise à jour du mot de passe. Veuillez réessayer.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔄 SMG - Nouveau Mot de Passe</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #dc3545 0%, #6f42c1 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .reset-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 40px;
            width: 100%;
            max-width: 450px;
            text-align: center;
        }

        .logo {
            font-size: 48px;
            margin-bottom: 10px;
        }

        .title {
            color: #333;
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 8px;
        }

        .subtitle {
            color: #666;
            font-size: 16px;
            margin-bottom: 30px;
        }

        .user-info {
            background: #e7f3ff;
            color: #004085;
            padding: 15px;
            border-radius: 12px;
            margin-bottom: 25px;
            border: 1px solid #b6d7ff;
        }

        .form-group {
            margin-bottom: 20px;
            text-align: left;
        }

        .form-label {
            display: block;
            color: #555;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .form-input {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: #f8f9fa;
        }

        .form-input:focus {
            outline: none;
            border-color: #dc3545;
            background: white;
            box-shadow: 0 0 0 3px rgba(220, 53, 69, 0.1);
        }

        .password-strength {
            margin-top: 8px;
            font-size: 12px;
        }

        .strength-weak { color: #dc3545; }
        .strength-medium { color: #ffc107; }
        .strength-strong { color: #28a745; }

        .password-requirements {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            font-size: 13px;
            color: #6c757d;
            text-align: left;
            margin: 15px 0;
        }

        .password-requirements ul {
            margin: 8px 0 0 20px;
        }

        .btn {
            width: 100%;
            padding: 15px 20px;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
            text-decoration: none;
            display: inline-block;
            text-align: center;
        }

        .btn-reset {
            background: linear-gradient(135deg, #dc3545, #6f42c1);
            color: white;
        }

        .btn-back {
            background: #6c757d;
            color: white;
            font-size: 14px;
            padding: 10px 20px;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
        }

        .btn:active {
            transform: translateY(0);
        }

        .btn:disabled {
            background: #6c757d;
            cursor: not-allowed;
            transform: none;
            opacity: 0.6;
        }

        .error-message {
            background: #fee;
            color: #c33;
            padding: 12px;
            border-radius: 8px;
            margin: 15px 0;
            font-size: 14px;
            border: 1px solid #fcc;
        }

        .success-message {
            background: #efe;
            color: #363;
            padding: 12px;
            border-radius: 8px;
            margin: 15px 0;
            font-size: 14px;
            border: 1px solid #cfc;
        }

        .footer {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #eee;
            color: #888;
            font-size: 12px;
        }

        .expired-token {
            text-align: center;
            padding: 40px 20px;
        }

        .expired-token .icon {
            font-size: 64px;
            margin-bottom: 20px;
        }

        .security-info {
            background: #fff3cd;
            color: #856404;
            padding: 12px;
            border-radius: 8px;
            margin: 15px 0;
            font-size: 13px;
            border: 1px solid #ffeaa7;
        }

        /* Responsive */
        @media (max-width: 480px) {
            .reset-container {
                padding: 30px 20px;
            }
        }
    </style>
</head>
<body>
    <div class="reset-container">
        <div class="logo">🔄</div>
        <h1 class="title">Nouveau Mot de Passe</h1>
        <p class="subtitle">SMG Studio v2.0 - Audiovizu Edition</p>

        <?php if ($valid_token): ?>
            <!-- Formulaire de réinitialisation -->
            <div class="user-info">
                👤 <strong><?= htmlspecialchars($user_data['name']) ?></strong><br>
                📧 <?= htmlspecialchars($user_data['email']) ?>
            </div>

            <div class="security-info">
                🔒 <strong>Sécurité :</strong> Ce lien expire dans 1 heure et ne peut être utilisé qu'une seule fois.
            </div>

            <?php if ($error): ?>
                <div class="error-message"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>

            <form method="POST" id="resetForm">
                <div class="form-group">
                    <label class="form-label" for="new_password">🔐 Nouveau Mot de Passe</label>
                    <input 
                        type="password" 
                        id="new_password" 
                        name="new_password" 
                        class="form-input" 
                        placeholder="Votre nouveau mot de passe"
                        required
                        minlength="8"
                        autocomplete="new-password"
                        oninput="checkPasswordStrength()"
                    >
                    <div id="passwordStrength" class="password-strength"></div>
                </div>

                <div class="form-group">
                    <label class="form-label" for="confirm_password">🔒 Confirmer le Mot de Passe</label>
                    <input 
                        type="password" 
                        id="confirm_password" 
                        name="confirm_password" 
                        class="form-input" 
                        placeholder="Confirmez votre mot de passe"
                        required
                        autocomplete="new-password"
                        oninput="checkPasswordMatch()"
                    >
                    <div id="passwordMatch" class="password-strength"></div>
                </div>

                <div class="password-requirements">
                    <strong>📋 Exigences du mot de passe :</strong>
                    <ul>
                        <li>Au moins 8 caractères</li>
                        <li>Une lettre majuscule (A-Z)</li>
                        <li>Une lettre minuscule (a-z)</li>
                        <li>Au moins un chiffre (0-9)</li>
                        <li>Caractères spéciaux recommandés</li>
                    </ul>
                </div>

                <button type="submit" class="btn btn-reset" id="submitBtn" disabled>
                    🔄 Définir le Nouveau Mot de Passe
                </button>
            </form>

            <div style="margin-top: 20px;">
                <a href="login.php" class="btn btn-back">
                    ← Retour à la Connexion
                </a>
            </div>

        <?php else: ?>
            <!-- Token invalide ou expiré -->
            <div class="expired-token">
                <div class="icon">⏰</div>
                <h2>Lien Expiré ou Invalide</h2>
                
                <?php if ($error): ?>
                    <div class="error-message"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <p style="margin: 20px 0; color: #6c757d;">
                    Ce lien de réinitialisation n'est plus valide, a expiré ou a déjà été utilisé.
                </p>
                
                <a href="login.php" class="btn btn-back">
                    ← Retour à la Connexion
                </a>
                
                <p style="margin-top: 20px; font-size: 14px; color: #6c757d;">
                    Vous pouvez demander un nouveau lien de réinitialisation sur la page de connexion.
                </p>
            </div>
        <?php endif; ?>

        <div class="footer">
            <p>🔒 Réinitialisation sécurisée</p>
            <p>SMG Studio v2.0 - Audiovizu Edition by Papy 77</p>
        </div>
    </div>

    <?php if ($valid_token): ?>
    <script>
        function checkPasswordStrength() {
            const password = document.getElementById('new_password').value;
            const strengthDiv = document.getElementById('passwordStrength');
            
            let strength = 0;
            let message = '';
            
            // Critères de force
            if (password.length >= 8) strength++;
            if (password.match(/[a-z]/)) strength++;
            if (password.match(/[A-Z]/)) strength++;
            if (password.match(/[0-9]/)) strength++;
            if (password.match(/[^a-zA-Z0-9]/)) strength++;
            
            switch (strength) {
                case 0:
                case 1:
                case 2:
                    message = '🔴 Faible';
                    strengthDiv.className = 'password-strength strength-weak';
                    break;
                case 3:
                case 4:
                    message = '🟡 Moyen';
                    strengthDiv.className = 'password-strength strength-medium';
                    break;
                case 5:
                    message = '🟢 Fort';
                    strengthDiv.className = 'password-strength strength-strong';
                    break;
            }
            
            strengthDiv.textContent = message;
            checkFormValidity();
        }
        
        function checkPasswordMatch() {
            const password = document.getElementById('new_password').value;
            const confirm = document.getElementById('confirm_password').value;
            const matchDiv = document.getElementById('passwordMatch');
            
            if (confirm.length === 0) {
                matchDiv.textContent = '';
                matchDiv.className = 'password-strength';
            } else if (password === confirm) {
                matchDiv.textContent = '✅ Mots de passe identiques';
                matchDiv.className = 'password-strength strength-strong';
            } else {
                matchDiv.textContent = '❌ Mots de passe différents';
                matchDiv.className = 'password-strength strength-weak';
            }
            
            checkFormValidity();
        }
        
        function checkFormValidity() {
            const password = document.getElementById('new_password').value;
            const confirm = document.getElementById('confirm_password').value;
            const submitBtn = document.getElementById('submitBtn');
            
            // Vérification des critères de sécurité
            const isValid = password.length >= 8 && 
                           password === confirm && 
                           password.match(/[a-z]/) && 
                           password.match(/[A-Z]/) && 
                           password.match(/[0-9]/);
            
            submitBtn.disabled = !isValid;
            
            if (isValid) {
                submitBtn.style.opacity = '1';
                submitBtn.style.cursor = 'pointer';
            } else {
                submitBtn.style.opacity = '0.6';
                submitBtn.style.cursor = 'not-allowed';
            }
        }
        
        // Sécurité : empêcher le cache du mot de passe
        window.addEventListener('beforeunload', function() {
            document.getElementById('new_password').value = '';
            document.getElementById('confirm_password').value = '';
        });
        
        // Focus automatique
        document.getElementById('new_password').focus();
        
        console.log('🔄 SMG Password Reset Ready - CORRIGÉ');
    </script>
    <?php endif; ?>
</body>
</html>