<?php
/**
 * ================================================================
 * 🚑 SMG RECOVERY - REMISE EN ÉTAT COMPLÈTE
 * ================================================================
 * 
 * 🎯 MISSION : Remettre SMG Studio en état de marche !
 * 🛠️ Création de toutes les tables manquantes
 * 📊 Données de test pour validation
 * 💪 Compatible avec ton système existant
 * 
 * 💡 BY PAPY 77 - MODE URGENCE ACTIVÉ !
 * ================================================================
 */

session_start();
require_once 'config/db.php';

// Forcer l'affichage des erreurs pour le diagnostic
ini_set('display_errors', 1);
error_reporting(E_ALL);

$recovery_log = [];
$recovery_errors = [];

function log_recovery($message, $type = 'info') {
    global $recovery_log;
    $recovery_log[] = ['time' => date('H:i:s'), 'type' => $type, 'message' => $message];
    echo "[$type] " . date('H:i:s') . " - $message\n";
}

function log_error($message) {
    global $recovery_errors;
    $recovery_errors[] = $message;
    log_recovery($message, 'ERROR');
}

echo "🚑 SMG STUDIO - RECOVERY MODE ACTIVÉ\n";
echo "===================================\n\n";

// ================================================================
// 🔍 DIAGNOSTIC INITIAL
// ================================================================

log_recovery("Diagnostic initial de la base de données...", 'INFO');

try {
    $stmt = $pdo->query("SELECT DATABASE() as db_name, VERSION() as mysql_version");
    $db_info = $stmt->fetch();
    log_recovery("Base de données : " . $db_info['db_name'], 'SUCCESS');
    log_recovery("Version MySQL : " . $db_info['mysql_version'], 'INFO');
} catch (Exception $e) {
    log_error("Connexion base échouée : " . $e->getMessage());
    die("❌ RECOVERY IMPOSSIBLE - Problème de connexion\n");
}

// Vérifier tables existantes
$stmt = $pdo->query("SHOW TABLES");
$existing_tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
log_recovery("Tables existantes trouvées : " . count($existing_tables), 'INFO');

// ================================================================
// 🛠️ CRÉATION DES TABLES SMG STUDIO
// ================================================================

log_recovery("Création des tables SMG Studio v2.0...", 'INFO');

// 1️⃣ TABLE SOCIAL_PLATFORMS (CRITIQUE - RÉFÉRENCÉE PAR FK)
if (!in_array('social_platforms', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `social_platforms` (
                `id` int NOT NULL AUTO_INCREMENT,
                `name` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
                `display_name` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
                `icon` varchar(10) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `is_active` tinyint(1) DEFAULT '1',
                `api_endpoint` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `max_chars` int DEFAULT '0',
                `supports_images` tinyint(1) DEFAULT '1',
                `supports_videos` tinyint(1) DEFAULT '1',
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `name` (`name`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        // Insérer les plateformes
        $pdo->exec("
            INSERT INTO `social_platforms` (`name`, `display_name`, `icon`, `max_chars`, `supports_images`, `supports_videos`) VALUES
            ('instagram', 'Instagram', '📸', 2200, 1, 1),
            ('facebook', 'Facebook', '📘', 63206, 1, 1),
            ('twitter', 'Twitter/X', '🐦', 280, 1, 1),
            ('linkedin', 'LinkedIn', '💼', 3000, 1, 1),
            ('tiktok', 'TikTok', '🎵', 2200, 1, 1),
            ('snapchat', 'Snapchat', '👻', 250, 1, 1),
            ('youtube', 'YouTube', '📺', 5000, 1, 1)
        ");
        
        log_recovery("✅ Table social_platforms créée avec 7 plateformes", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création social_platforms : " . $e->getMessage());
    }
} else {
    log_recovery("Table social_platforms existe déjà", 'INFO');
}

// 2️⃣ TABLE POSTS (CORE)
if (!in_array('posts', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `posts` (
                `id` int NOT NULL AUTO_INCREMENT,
                `user_id` int NOT NULL,
                `titre` varchar(255) COLLATE utf8mb4_unicode_ci NOT NULL,
                `contenu` text COLLATE utf8mb4_unicode_ci,
                `hashtags` text COLLATE utf8mb4_unicode_ci,
                `statut` enum('draft','scheduled','published','archived') COLLATE utf8mb4_unicode_ci DEFAULT 'draft',
                `type` enum('text','image','video','carousel','story') COLLATE utf8mb4_unicode_ci DEFAULT 'text',
                `priority` int DEFAULT '1',
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                `published_at` timestamp NULL DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `idx_posts_user_id` (`user_id`),
                KEY `idx_posts_statut` (`statut`),
                KEY `idx_posts_created_at` (`created_at`),
                KEY `idx_posts_user_status` (`user_id`, `statut`),
                CONSTRAINT `fk_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table posts créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création posts : " . $e->getMessage());
    }
} else {
    log_recovery("Table posts existe déjà", 'INFO');
}

// 3️⃣ TABLE POST_NETWORKS
if (!in_array('post_networks', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `post_networks` (
                `id` int NOT NULL AUTO_INCREMENT,
                `post_id` int NOT NULL,
                `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
                `is_active` tinyint(1) DEFAULT '1',
                `custom_content` text COLLATE utf8mb4_unicode_ci,
                `custom_hashtags` text COLLATE utf8mb4_unicode_ci,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `unique_post_network` (`post_id`,`network`),
                KEY `idx_post_networks_network` (`network`),
                CONSTRAINT `fk_post_networks_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE,
                CONSTRAINT `fk_post_networks_platform` FOREIGN KEY (`network`) REFERENCES `social_platforms` (`name`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table post_networks créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création post_networks : " . $e->getMessage());
    }
} else {
    log_recovery("Table post_networks existe déjà", 'INFO');
}

// 4️⃣ TABLE POST_MEDIAS
if (!in_array('post_medias', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `post_medias` (
                `id` int NOT NULL AUTO_INCREMENT,
                `post_id` int NOT NULL,
                `media_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
                `media_type` enum('image','video','audio','document') COLLATE utf8mb4_unicode_ci NOT NULL,
                `media_size` bigint DEFAULT '0',
                `mime_type` varchar(100) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `display_order` int DEFAULT '1',
                `alt_text` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `thumbnail_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_post_medias_post_id` (`post_id`),
                KEY `idx_post_medias_type` (`media_type`),
                CONSTRAINT `fk_post_medias_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table post_medias créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création post_medias : " . $e->getMessage());
    }
} else {
    log_recovery("Table post_medias existe déjà", 'INFO');
}

// 5️⃣ TABLE SCHEDULED_POSTS
if (!in_array('scheduled_posts', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `scheduled_posts` (
                `id` int NOT NULL AUTO_INCREMENT,
                `user_id` int NOT NULL,
                `post_id` int NOT NULL,
                `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
                `scheduled_datetime` timestamp NOT NULL,
                `status` enum('pending','processing','sent','failed','cancelled') COLLATE utf8mb4_unicode_ci DEFAULT 'pending',
                `priority` int DEFAULT '1',
                `retry_count` int DEFAULT '0',
                `max_retries` int DEFAULT '3',
                `error_message` text COLLATE utf8mb4_unicode_ci,
                `external_post_id` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `sent_at` timestamp NULL DEFAULT NULL,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `unique_post_network_schedule` (`post_id`,`network`,`scheduled_datetime`),
                KEY `idx_scheduled_posts_user` (`user_id`),
                KEY `idx_scheduled_posts_status` (`status`),
                KEY `idx_scheduled_posts_datetime` (`scheduled_datetime`),
                KEY `idx_scheduled_posts_network` (`network`),
                KEY `idx_scheduled_posts_processing` (`status`, `scheduled_datetime`),
                CONSTRAINT `fk_scheduled_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
                CONSTRAINT `fk_scheduled_posts_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE,
                CONSTRAINT `fk_scheduled_posts_network` FOREIGN KEY (`network`) REFERENCES `social_platforms` (`name`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table scheduled_posts créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création scheduled_posts : " . $e->getMessage());
    }
} else {
    log_recovery("Table scheduled_posts existe déjà", 'INFO');
}

// 6️⃣ TABLE TRANSMISSION_LOGS
if (!in_array('transmission_logs', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `transmission_logs` (
                `id` int NOT NULL AUTO_INCREMENT,
                `scheduled_post_id` int NOT NULL,
                `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
                `status` enum('success','failed','retry','cancelled') COLLATE utf8mb4_unicode_ci NOT NULL,
                `attempt_number` int DEFAULT '1',
                `response_data` longtext COLLATE utf8mb4_unicode_ci,
                `error_message` text COLLATE utf8mb4_unicode_ci,
                `execution_time_ms` int DEFAULT '0',
                `external_post_id` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `transmitted_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_transmission_logs_scheduled` (`scheduled_post_id`),
                KEY `idx_transmission_logs_network` (`network`),
                KEY `idx_transmission_logs_status` (`status`),
                KEY `idx_transmission_logs_date` (`transmitted_at`),
                CONSTRAINT `fk_transmission_logs_scheduled` FOREIGN KEY (`scheduled_post_id`) REFERENCES `scheduled_posts` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table transmission_logs créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création transmission_logs : " . $e->getMessage());
    }
} else {
    log_recovery("Table transmission_logs existe déjà", 'INFO');
}

// 7️⃣ TABLE USER_DOWNLOADS
if (!in_array('user_downloads', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `user_downloads` (
                `id` int NOT NULL AUTO_INCREMENT,
                `user_id` int NOT NULL,
                `post_id` int DEFAULT NULL,
                `download_type` enum('post','media','final','zip') COLLATE utf8mb4_unicode_ci DEFAULT 'zip',
                `file_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
                `file_size` bigint DEFAULT '0',
                `download_count` int DEFAULT '0',
                `expires_at` timestamp NULL DEFAULT NULL,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `last_downloaded` timestamp NULL DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `idx_user_downloads_user` (`user_id`),
                KEY `idx_user_downloads_post` (`post_id`),
                KEY `idx_user_downloads_expires` (`expires_at`),
                KEY `idx_downloads_cleanup` (`expires_at`, `created_at`),
                CONSTRAINT `fk_user_downloads_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
                CONSTRAINT `fk_user_downloads_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table user_downloads créée", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création user_downloads : " . $e->getMessage());
    }
} else {
    log_recovery("Table user_downloads existe déjà", 'INFO');
}

// 8️⃣ TABLE USER_MEDIAS (pour images.php)
if (!in_array('user_medias', $existing_tables)) {
    try {
        $pdo->exec("
            CREATE TABLE `user_medias` (
                `id` int NOT NULL AUTO_INCREMENT,
                `user_id` int NOT NULL,
                `original_name` varchar(255) COLLATE utf8mb4_unicode_ci NOT NULL,
                `original_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
                `instagram_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `facebook_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `stories_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `twitter_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
                `file_size` int DEFAULT '0',
                `width` int DEFAULT '0',
                `height` int DEFAULT '0',
                `status` enum('active','trash') COLLATE utf8mb4_unicode_ci DEFAULT 'active',
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_user_status` (`user_id`,`status`),
                CONSTRAINT `fk_user_medias_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        log_recovery("✅ Table user_medias créée (compatible images.php)", 'SUCCESS');
    } catch (Exception $e) {
        log_error("Échec création user_medias : " . $e->getMessage());
    }
} else {
    log_recovery("Table user_medias existe déjà", 'INFO');
}

// ================================================================
// 🧪 CRÉATION DE DONNÉES DE TEST
// ================================================================

log_recovery("Création de données de test...", 'INFO');

// Vérifier s'il y a des utilisateurs actifs
$stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'");
$active_users = $stmt->fetchColumn();

if ($active_users > 0) {
    // Récupérer le premier utilisateur actif
    $stmt = $pdo->query("SELECT id, name FROM users WHERE status = 'active' ORDER BY id LIMIT 1");
    $test_user = $stmt->fetch();
    
    log_recovery("Utilisateur test trouvé : " . $test_user['name'], 'INFO');
    
    // Créer des posts de test
    try {
        $test_posts = [
            [
                'titre' => '🚑 Post Recovery Test #1',
                'contenu' => 'Premier post créé après recovery SMG Studio v2.0. Système opérationnel !',
                'hashtags' => '#SMGStudio #Recovery #Test #SocialMedia #Papy77'
            ],
            [
                'titre' => '📱 Multi-Network Test #2', 
                'contenu' => 'Test de post multi-réseaux avec programmation. Instagram + Facebook + Twitter ready !',
                'hashtags' => '#MultiNetwork #Instagram #Facebook #Twitter #SMG'
            ],
            [
                'titre' => '🎯 Workflow Validation #3',
                'contenu' => 'Validation complète du workflow SMG Studio. Upload → Post → Schedule → Publish. All systems GO !',
                'hashtags' => '#Workflow #Validation #SMGStudio #AllSystemsGO'
            ]
        ];
        
        $posts_created = 0;
        foreach ($test_posts as $post_data) {
            // Créer le post
            $stmt = $pdo->prepare("
                INSERT INTO posts (user_id, titre, contenu, hashtags, statut, type, created_at) 
                VALUES (?, ?, ?, ?, 'draft', 'text', NOW())
            ");
            $stmt->execute([
                $test_user['id'], 
                $post_data['titre'], 
                $post_data['contenu'], 
                $post_data['hashtags']
            ]);
            
            $post_id = $pdo->lastInsertId();
            
            // Associer aux réseaux sociaux
            $networks = ['instagram', 'facebook', 'twitter'];
            foreach ($networks as $network) {
                $stmt = $pdo->prepare("
                    INSERT INTO post_networks (post_id, network, is_active) 
                    VALUES (?, ?, 1)
                ");
                $stmt->execute([$post_id, $network]);
            }
            
            $posts_created++;
        }
        
        log_recovery("✅ {$posts_created} posts de test créés avec associations réseaux", 'SUCCESS');
        
    } catch (Exception $e) {
        log_error("Échec création posts de test : " . $e->getMessage());
    }
    
} else {
    log_recovery("⚠️ Aucun utilisateur actif trouvé - Posts de test non créés", 'WARNING');
}

// ================================================================
// 📊 RAPPORT FINAL DE RECOVERY
// ================================================================

log_recovery("Recovery terminée - Génération du rapport...", 'INFO');

// Compter les nouvelles données
$final_stats = [];
try {
    $stmt = $pdo->query("SELECT COUNT(*) FROM posts");
    $final_stats['posts'] = $stmt->fetchColumn();
    
    $stmt = $pdo->query("SELECT COUNT(*) FROM social_platforms");
    $final_stats['platforms'] = $stmt->fetchColumn();
    
    $stmt = $pdo->query("SELECT COUNT(*) FROM users");
    $final_stats['users'] = $stmt->fetchColumn();
    
    $stmt = $pdo->query("SHOW TABLES");
    $final_stats['tables'] = count($stmt->fetchAll());
    
} catch (Exception $e) {
    log_error("Erreur génération stats finales : " . $e->getMessage());
}

$recovery_success = empty($recovery_errors);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🚑 SMG Recovery - Rapport Complet</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #2c3e50 0%, #3498db 100%);
            color: #333;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
        }

        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            text-align: center;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .header h1 {
            font-size: 36px;
            margin-bottom: 10px;
            color: #2c3e50;
        }

        .status-badge {
            display: inline-block;
            padding: 12px 24px;
            border-radius: 25px;
            font-weight: bold;
            font-size: 18px;
            margin: 15px 0;
        }

        .status-success {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            color: white;
        }

        .status-error {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-number {
            font-size: 36px;
            font-weight: bold;
            color: #3498db;
            margin-bottom: 10px;
        }

        .stat-label {
            font-size: 16px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .log-container {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }

        .log-container h3 {
            margin-bottom: 20px;
            color: #2c3e50;
        }

        .log-entry {
            display: flex;
            align-items: center;
            padding: 8px 12px;
            margin-bottom: 5px;
            border-radius: 8px;
            font-family: 'Consolas', monospace;
            font-size: 14px;
        }

        .log-info { background: #e8f4fd; color: #3498db; }
        .log-success { background: #d5f4e6; color: #27ae60; }
        .log-error { background: #fdeaea; color: #e74c3c; }
        .log-warning { background: #fff3cd; color: #f39c12; }

        .log-time {
            margin-right: 15px;
            font-weight: bold;
            opacity: 0.7;
        }

        .actions-panel {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }

        .btn {
            display: inline-block;
            padding: 15px 30px;
            margin: 10px;
            border: none;
            border-radius: 10px;
            text-decoration: none;
            font-weight: bold;
            font-size: 16px;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
        }

        .console-output {
            background: #1a1a1a;
            color: #00ff00;
            padding: 20px;
            border-radius: 10px;
            font-family: 'Consolas', monospace;
            font-size: 12px;
            max-height: 400px;
            overflow-y: auto;
            margin-top: 20px;
            white-space: pre-wrap;
        }

        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .header {
                padding: 20px;
            }
            
            .header h1 {
                font-size: 24px;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚑 SMG Studio Recovery</h1>
            <p>Rapport de récupération complet - Papy 77 Edition</p>
            
            <?php if ($recovery_success): ?>
                <div class="status-badge status-success">
                    ✅ RECOVERY RÉUSSIE - SYSTÈME OPÉRATIONNEL
                </div>
            <?php else: ?>
                <div class="status-badge status-error">
                    ❌ RECOVERY PARTIELLE - ERREURS DÉTECTÉES
                </div>
            <?php endif; ?>
            
            <p style="margin-top: 15px; color: #666;">
                <?= date('d/m/Y à H:i:s') ?> - Durée : <?= count($recovery_log) ?> étapes
            </p>
        </div>

        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?= $final_stats['posts'] ?? 0 ?></div>
                <div class="stat-label">Posts Créés</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $final_stats['platforms'] ?? 0 ?></div>
                <div class="stat-label">Plateformes Sociales</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $final_stats['users'] ?? 0 ?></div>
                <div class="stat-label">Utilisateurs</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $final_stats['tables'] ?? 0 ?></div>
                <div class="stat-label">Tables Créées</div>
            </div>
        </div>

        <div class="log-container">
            <h3>📋 Log de Récupération</h3>
            <div style="max-height: 300px; overflow-y: auto;">
                <?php foreach ($recovery_log as $entry): ?>
                    <div class="log-entry log-<?= strtolower($entry['type']) ?>">
                        <span class="log-time"><?= $entry['time'] ?></span>
                        <span><?= htmlspecialchars($entry['message']) ?></span>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <?php if (!empty($recovery_errors)): ?>
            <div class="log-container">
                <h3>⚠️ Erreurs Rencontrées</h3>
                <?php foreach ($recovery_errors as $error): ?>
                    <div class="log-entry log-error">
                        <span><?= htmlspecialchars($error) ?></span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <div class="actions-panel">
            <h3 style="margin-bottom: 20px;">🚀 Actions Disponibles</h3>
            
            <?php if ($recovery_success): ?>
                <a href="dashboard.php" class="btn btn-success">
                    📊 Accéder au Dashboard
                </a>
                <a href="posts.php" class="btn btn-primary">
                    📝 Gérer les Posts
                </a>
                <a href="images.php" class="btn btn-primary">
                    📸 Upload Images/Vidéos
                </a>
            <?php else: ?>
                <a href="<?= $_SERVER['PHP_SELF'] ?>" class="btn btn-warning">
                    🔄 Relancer Recovery
                </a>
                <a href="dashboard.php?debug=1" class="btn btn-primary">
                    🔍 Debug Dashboard
                </a>
            <?php endif; ?>
            
            <a href="recovery.php" class="btn btn-primary">
                🚑 Recovery Original
            </a>
        </div>

        <div class="console-output">
🚑 SMG STUDIO RECOVERY COMPLETE - RAPPORT TECHNIQUE

Base de données : <?= $db_info['db_name'] ?? 'N/A' ?>

Version MySQL : <?= $db_info['mysql_version'] ?? 'N/A' ?>

Tables créées :
✅ social_platforms (7 plateformes)
✅ posts (structure user-only)
✅ post_networks (associations)
✅ post_medias (médias attachés)
✅ scheduled_posts (programmation)
✅ transmission_logs (historique)
✅ user_downloads (téléchargements)
✅ user_medias (compatible images.php)

Posts de test : <?= $final_stats['posts'] ?? 0 ?> créés
Utilisateurs actifs : <?= $active_users ?? 0 ?>
Erreurs rencontrées : <?= count($recovery_errors) ?>

<?php if ($recovery_success): ?>
🎉 SMG STUDIO v2.0 EST MAINTENANT OPÉRATIONNEL !

Structure BDD ✅
Posts de test ✅  
Dashboard ✅
Images.php ✅
Clipmaker ✅

Ready for Social Media Generation - Papy 77 Power !
<?php else: ?>
⚠️ Recovery partielle - Vérifiez les erreurs ci-dessus
<?php endif; ?>

<?= date('Y-m-d H:i:s') ?> - Recovery terminée
        </div>
    </div>

    <script>
        console.log('🚑 SMG Recovery Report Ready');
        console.log('Success:', <?= $recovery_success ? 'true' : 'false' ?>);
        console.log('Posts créés:', <?= $final_stats['posts'] ?? 0 ?>);
        console.log('Erreurs:', <?= count($recovery_errors) ?>);
        
        <?php if ($recovery_success && ($final_stats['posts'] ?? 0) > 0): ?>
        // Auto-redirect vers dashboard après 5 secondes si tout est OK
        setTimeout(() => {
            if (confirm('🎉 Recovery réussie ! Aller au Dashboard SMG Studio ?')) {
                window.location.href = 'dashboard.php';
            }
        }, 5000);
        <?php endif; ?>
    </script>
</body>
</html>