<?php
/**
 * ================================================================
 * 📝 POSTS.PHP COMPLET - MULTI-RÉSEAUX + WORKFLOW + CARROUSEL
 * ================================================================
 * 
 * 🎯 FUSION COMPLÈTE :
 * - Multi-réseaux avec switches (nouveau)
 * - Workflow vidéos user (old_posts.php)
 * - Carrousel photos avancé (old_posts.php)
 * - Intégration images.php (images.php)
 * - Interface responsive premium
 * 
 * 💡 BY PAPY 77 - FUSION DES FORCES !
 * ================================================================
 */

session_start();
require_once 'config/db.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur user posts: " . $e->getMessage());
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🎬 GESTION WORKFLOW VIDÉOS USER (FROM OLD_POSTS.PHP)
// ================================================================

function getUserWorkflowPath($user_id) {
    return "SMG_STUDIO_WORKFLOW/user_" . str_pad($user_id, 3, '0', STR_PAD_LEFT) . "/";
}

function getUserVideos($user_id) {
    $workflow_path = getUserWorkflowPath($user_id);
    $videos = [];
    
    // Vidéos actives
    $active_path = $workflow_path . "active_clip/03_social_ready/";
    if (is_dir($active_path)) {
        $files = glob($active_path . "*.{mp4,webm,mov,avi}", GLOB_BRACE);
        foreach ($files as $file) {
            $videos[] = [
                'type' => 'active',
                'path' => $file,
                'name' => basename($file),
                'size' => filesize($file),
                'date' => filemtime($file),
                'project' => getCurrentProject($user_id)
            ];
        }
    }
    
    // Vidéos archivées
    $archive_path = $workflow_path . "archived_clips/";
    if (is_dir($archive_path)) {
        $projects = glob($archive_path . "*", GLOB_ONLYDIR);
        foreach ($projects as $project_dir) {
            $social_files = glob($project_dir . "/03_social_ready/*.{mp4,webm,mov,avi}", GLOB_BRACE);
            foreach ($social_files as $file) {
                $videos[] = [
                    'type' => 'archived',
                    'path' => $file,
                    'name' => basename($file),
                    'size' => filesize($file),
                    'date' => filemtime($file),
                    'project' => basename($project_dir)
                ];
            }
        }
    }
    
    // Trier par date
    usort($videos, function($a, $b) {
        return $b['date'] - $a['date'];
    });
    
    return $videos;
}

function getCurrentProject($user_id) {
    $project_file = getUserWorkflowPath($user_id) . "active_clip/current_project.txt";
    return file_exists($project_file) ? trim(file_get_contents($project_file)) : "Projet sans nom";
}

// ================================================================
// 📸 INTÉGRATION GESTIONNAIRE MÉDIAS (FROM IMAGES.PHP)
// ================================================================

function getUserMediaPath($user_id) {
    return "SMG_STUDIO_WORKFLOW/user_" . str_pad($user_id, 3, '0', STR_PAD_LEFT) . "/photos/";
}

function getUserConvertedMedias($user_id) {
    global $pdo;
    
    try {
        // Créer table si nécessaire (from images.php)
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS user_medias (
                id INT PRIMARY KEY AUTO_INCREMENT,
                user_id INT NOT NULL,
                original_name VARCHAR(255) NOT NULL,
                original_path VARCHAR(500) NOT NULL,
                instagram_path VARCHAR(500) NULL,
                facebook_path VARCHAR(500) NULL,
                stories_path VARCHAR(500) NULL,
                twitter_path VARCHAR(500) NULL,
                file_size INT DEFAULT 0,
                width INT DEFAULT 0,
                height INT DEFAULT 0,
                status ENUM('active', 'trash') DEFAULT 'active',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_user_status (user_id, status)
            )
        ");
        
        $stmt = $pdo->prepare("
            SELECT * FROM user_medias 
            WHERE user_id = ? AND status = 'active' 
            ORDER BY created_at DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur chargement médias: " . $e->getMessage());
        return [];
    }
}

// ================================================================
// 🔧 FONCTIONS UTILITAIRES MULTI-RÉSEAUX (ENHANCED)
// ================================================================

function getNetworkFormatSettings($network) {
    $settings = [
        'instagram' => [
            'format' => 'square',
            'width' => 1080,
            'height' => 1080,
            'quality' => 85,
            'optimization' => 'mobile',
            'hashtag_limit' => 30,
            'character_limit' => 2200
        ],
        'facebook' => [
            'format' => 'landscape',
            'width' => 1200,
            'height' => 630,
            'quality' => 85,
            'optimization' => 'web',
            'hashtag_limit' => 20,
            'character_limit' => 63206
        ],
        'twitter' => [
            'format' => 'banner',
            'width' => 1024,
            'height' => 512,
            'quality' => 85,
            'optimization' => 'fast',
            'hashtag_limit' => 10,
            'character_limit' => 280
        ],
        'linkedin' => [
            'format' => 'professional',
            'width' => 1200,
            'height' => 627,
            'quality' => 90,
            'optimization' => 'quality',
            'hashtag_limit' => 5,
            'character_limit' => 3000
        ],
        'tiktok' => [
            'format' => 'vertical',
            'width' => 1080,
            'height' => 1920,
            'quality' => 85,
            'optimization' => 'mobile',
            'hashtag_limit' => 15,
            'character_limit' => 300
        ],
        'youtube' => [
            'format' => 'thumbnail',
            'width' => 1280,
            'height' => 720,
            'quality' => 90,
            'optimization' => 'quality',
            'hashtag_limit' => 15,
            'character_limit' => 5000
        ],
        'snapchat' => [
            'format' => 'vertical',
            'width' => 1080,
            'height' => 1920,
            'quality' => 85,
            'optimization' => 'mobile',
            'hashtag_limit' => 0,
            'character_limit' => 250
        ]
    ];
    
    return $settings[$network] ?? $settings['instagram'];
}

function getPostNetworks($post_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT pn.*, sp.display_name, sp.icon, sp.color 
            FROM post_networks pn 
            LEFT JOIN social_platforms sp ON pn.network = sp.name 
            WHERE pn.post_id = ? 
            ORDER BY pn.created_at ASC
        ");
        $stmt->execute([$post_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getPostNetworks: " . $e->getMessage());
        return [];
    }
}

function handleVideoUpload($file) {
    $upload_dir = 'uploads/posts/videos/';
    
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $allowed_types = ['video/mp4', 'video/webm', 'video/mov', 'video/avi', 'video/quicktime'];
    
    if (!in_array($file['type'], $allowed_types)) {
        return ['success' => false, 'error' => 'Type vidéo non autorisé'];
    }
    
    if ($file['size'] > 100 * 1024 * 1024) { // 100MB max
        return ['success' => false, 'error' => 'Vidéo trop volumineuse (max 100MB)'];
    }
    
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'video_' . time() . '_' . uniqid() . '.' . $extension;
    $filepath = $upload_dir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return ['success' => true, 'path' => $filepath];
    } else {
        return ['success' => false, 'error' => 'Erreur upload vidéo'];
    }
}

function handlePhotoUpload($file) {
    $upload_dir = 'uploads/posts/photos/';
    
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/gif'];
    
    if (!in_array($file['type'], $allowed_types)) {
        return ['success' => false, 'error' => 'Type image non autorisé'];
    }
    
    if ($file['size'] > 10 * 1024 * 1024) {
        return ['success' => false, 'error' => 'Image trop volumineuse (max 10MB)'];
    }
    
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'photo_' . time() . '_' . uniqid() . '.' . $extension;
    $filepath = $upload_dir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return ['success' => true, 'path' => $filepath];
    } else {
        return ['success' => false, 'error' => 'Erreur upload photo'];
    }
}

// ================================================================
// 🎨 FONCTIONS UTILITAIRES AFFICHAGE
// ================================================================
function formatDate($date) {
    return date('d/m/Y H:i', strtotime($date));
}

function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, 2) . ' ' . $units[$pow];
}

function getStatusBadge($status) {
    $badges = [
        'draft' => '<span class="badge badge-secondary">📝 Brouillon</span>',
        'scheduled' => '<span class="badge badge-warning">⏰ Programmé</span>',
        'published' => '<span class="badge badge-success">✅ Publié</span>',
        'failed' => '<span class="badge badge-danger">❌ Échec</span>',
        'cancelled' => '<span class="badge badge-dark">🚫 Annulé</span>'
    ];
    return $badges[$status] ?? $status;
}

function getNetworkIcon($network) {
    $icons = [
        'instagram' => '📸',
        'facebook' => '📘',
        'twitter' => '🐦',
        'linkedin' => '💼',
        'tiktok' => '🎵',
        'snapchat' => '👻',
        'youtube' => '📺'
    ];
    return $icons[$network] ?? '📱';
}

// ================================================================
// 🗄️ CRÉATION TABLES SI NÉCESSAIRES
// ================================================================
try {
    // Table posts
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS posts (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT NOT NULL,
            client_id INT NULL,
            titre VARCHAR(200) NOT NULL,
            contenu TEXT NOT NULL,
            hashtags TEXT NULL,
            link_url VARCHAR(500) NULL,
            statut ENUM('draft', 'scheduled', 'published', 'failed', 'cancelled') DEFAULT 'draft',
            multi_network BOOLEAN DEFAULT 0,
            primary_network VARCHAR(50) NULL,
            max_photos INT DEFAULT 5,
            max_videos INT DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )
    ");
    
    // Table post_networks
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS post_networks (
            id INT PRIMARY KEY AUTO_INCREMENT,
            post_id INT NOT NULL,
            network VARCHAR(50) NOT NULL,
            is_active BOOLEAN DEFAULT 1,
            format_settings JSON NULL,
            export_status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
            export_path VARCHAR(500) NULL,
            published_at TIMESTAMP NULL,
            platform_post_id VARCHAR(100) NULL,
            engagement_data JSON NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_post_network (post_id, network)
        )
    ");
    
    // Table post_medias
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS post_medias (
            id INT PRIMARY KEY AUTO_INCREMENT,
            post_id INT NOT NULL,
            media_path VARCHAR(500) NOT NULL,
            media_type ENUM('image', 'video') NOT NULL,
            media_source ENUM('upload', 'workflow', 'external') DEFAULT 'upload',
            display_order INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    // Table clients
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS clients (
            id INT PRIMARY KEY AUTO_INCREMENT,
            nom VARCHAR(100) NOT NULL,
            email VARCHAR(150) NULL,
            status ENUM('active', 'inactive') DEFAULT 'active',
            created_by INT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
} catch (Exception $e) {
    error_log("Erreur création tables: " . $e->getMessage());
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS
// ================================================================

$action = $_GET['action'] ?? 'list';
$post_id = $_GET['id'] ?? null;
$message = '';
$message_type = '';

// Récupération médias sélectionnés depuis images.php
$selected_media_ids = [];
if (isset($_GET['media_ids'])) {
    $selected_media_ids = explode(',', $_GET['media_ids']);
}

// Traitement des formulaires
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    if (isset($_POST['create_post'])) {
        try {
            $titre = trim($_POST['titre']);
            $contenu = trim($_POST['contenu']);
            $hashtags = trim($_POST['hashtags']);
            $networks = $_POST['networks'] ?? [];
            $client_id = $_POST['client_id'] ?: null;
            $link_url = trim($_POST['link_url']) ?: null;
            
            if (empty($titre) || empty($contenu)) {
                throw new Exception("Titre et contenu sont requis");
            }
            
            if (empty($networks)) {
                throw new Exception("Sélectionnez au moins un réseau social");
            }
            
            $primary_network = $networks[0];
            
            // Gestion médias avec limites
            $medias = [];
            $video_count = 0;
            $photo_count = 0;
            
            // Vidéo workflow sélectionnée (max 1)
            if (!empty($_POST['workflow_video'])) {
                $video_path = $_POST['workflow_video'];
                if (file_exists($video_path)) {
                    $video_count++;
                    if ($video_count > 1) {
                        throw new Exception("Maximum 1 vidéo par post");
                    }
                    
                    $medias[] = [
                        'path' => $video_path,
                        'type' => 'video',
                        'source' => 'workflow',
                        'order' => 0
                    ];
                }
            }
            
            // Vidéo uploadée directement (max 1)
            if (!empty($_FILES['upload_video']) && $_FILES['upload_video']['error'] === UPLOAD_ERR_OK) {
                $video_count++;
                if ($video_count > 1) {
                    throw new Exception("Maximum 1 vidéo par post");
                }
                
                $upload_result = handleVideoUpload($_FILES['upload_video']);
                
                if ($upload_result['success']) {
                    $medias[] = [
                        'path' => $upload_result['path'],
                        'type' => 'video',
                        'source' => 'upload',
                        'order' => -1
                    ];
                }
            }
            
            // Photos uploadées (max 5)
            if (isset($_FILES['photos']) && !empty($_FILES['photos']['name'][0])) {
                for ($i = 0; $i < count($_FILES['photos']['name']); $i++) {
                    if ($_FILES['photos']['error'][$i] === UPLOAD_ERR_OK) {
                        $photo_count++;
                        if ($photo_count > 5) {
                            throw new Exception("Maximum 5 photos par post");
                        }
                        
                        $upload_result = handlePhotoUpload([
                            'name' => $_FILES['photos']['name'][$i],
                            'type' => $_FILES['photos']['type'][$i],
                            'tmp_name' => $_FILES['photos']['tmp_name'][$i],
                            'size' => $_FILES['photos']['size'][$i]
                        ]);
                        
                        if ($upload_result['success']) {
                            $medias[] = [
                                'path' => $upload_result['path'],
                                'type' => 'image',
                                'source' => 'upload',
                                'order' => $i + 1
                            ];
                        }
                    }
                }
            }
            
            // Photos sélectionnées depuis images.php
            if (!empty($_POST['selected_medias'])) {
                $selected_media_ids = explode(',', $_POST['selected_medias']);
                $converted_medias = getUserConvertedMedias($user['id']);
                
                foreach ($converted_medias as $media) {
                    if (in_array($media['id'], $selected_media_ids)) {
                        $photo_count++;
                        if ($photo_count > 5) {
                            throw new Exception("Maximum 5 photos par post (médias sélectionnés inclus)");
                        }
                        
                        $medias[] = [
                            'path' => $media['instagram_path'] ?: $media['original_path'],
                            'type' => 'image',
                            'source' => 'converted',
                            'order' => 100 + $photo_count
                        ];
                    }
                }
            }
            
            // Démarrer transaction
            $pdo->beginTransaction();
            
            // Insertion post principal
            $stmt = $pdo->prepare("
                INSERT INTO posts (client_id, user_id, titre, contenu, hashtags, link_url, 
                                 multi_network, primary_network, max_photos, max_videos, statut, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 5, 1, 'draft', NOW())
            ");
            
            $stmt->execute([
                $client_id, 
                $user['id'], 
                $titre, 
                $contenu, 
                $hashtags, 
                $link_url,
                count($networks) > 1 ? 1 : 0,
                $primary_network
            ]);
            
            $post_id = $pdo->lastInsertId();
            
            // Insertion réseaux sociaux
            $stmt = $pdo->prepare("
                INSERT INTO post_networks (post_id, network, is_active, format_settings, created_at) 
                VALUES (?, ?, 1, ?, NOW())
            ");
            
            foreach ($networks as $network) {
                $format_settings = getNetworkFormatSettings($network);
                $stmt->execute([
                    $post_id, 
                    $network, 
                    json_encode($format_settings)
                ]);
            }
            
            // Insertion médias
            if (!empty($medias)) {
                $stmt = $pdo->prepare("
                    INSERT INTO post_medias (post_id, media_path, media_type, media_source, display_order) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                
                foreach ($medias as $media) {
                    $stmt->execute([
                        $post_id, 
                        $media['path'], 
                        $media['type'], 
                        $media['source'], 
                        $media['order']
                    ]);
                }
            }
            
            // Valider transaction
            $pdo->commit();
            
            $network_count = count($networks);
            $media_count = count($medias);
            $message = "✅ Post créé pour {$network_count} réseau(x) avec {$media_count} média(s) !";
            $message_type = 'success';
            $action = 'list';
            
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $message = "❌ Erreur : " . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['update_post'])) {
        // ✏️ MODIFICATION POST (similar logic)
        try {
            $titre = trim($_POST['titre']);
            $contenu = trim($_POST['contenu']);
            $hashtags = trim($_POST['hashtags']);
            $networks = $_POST['networks'] ?? [];
            $client_id = $_POST['client_id'] ?: null;
            $statut = $_POST['statut'];
            $link_url = trim($_POST['link_url']) ?: null;
            $post_id = $_POST['post_id'];
            
            if (empty($titre) || empty($contenu)) {
                throw new Exception("Titre et contenu sont requis");
            }
            
            if (empty($networks)) {
                throw new Exception("Sélectionnez au moins un réseau social");
            }
            
            $stmt = $pdo->prepare("SELECT id FROM posts WHERE id = ? AND user_id = ?");
            $stmt->execute([$post_id, $user['id']]);
            if (!$stmt->fetch()) {
                throw new Exception("Post non trouvé ou accès refusé");
            }
            
            $pdo->beginTransaction();
            
            $primary_network = $networks[0];
            $stmt = $pdo->prepare("
                UPDATE posts 
                SET titre = ?, contenu = ?, hashtags = ?, client_id = ?, statut = ?, link_url = ?, 
                    multi_network = ?, primary_network = ?, updated_at = NOW()
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([
                $titre, $contenu, $hashtags, $client_id, $statut, $link_url,
                count($networks) > 1 ? 1 : 0, $primary_network, $post_id, $user['id']
            ]);
            
            // Gestion réseaux - Suppression des anciens
            $stmt = $pdo->prepare("DELETE FROM post_networks WHERE post_id = ?");
            $stmt->execute([$post_id]);
            
            // Insertion nouveaux réseaux
            $stmt = $pdo->prepare("
                INSERT INTO post_networks (post_id, network, is_active, format_settings, created_at) 
                VALUES (?, ?, 1, ?, NOW())
            ");
            
            foreach ($networks as $network) {
                $format_settings = getNetworkFormatSettings($network);
                $stmt->execute([$post_id, $network, json_encode($format_settings)]);
            }
            
            $pdo->commit();
            
            $message = "✅ Post modifié pour " . count($networks) . " réseau(x) !";
            $message_type = 'success';
            $action = 'list';
            
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $message = "❌ Erreur : " . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Suppression via GET
if ($action === 'delete' && $post_id) {
    try {
        $stmt = $pdo->prepare("DELETE FROM post_medias WHERE post_id = ?");
        $stmt->execute([$post_id]);
        
        $stmt = $pdo->prepare("DELETE FROM post_networks WHERE post_id = ?");
        $stmt->execute([$post_id]);
        
        $stmt = $pdo->prepare("DELETE FROM posts WHERE id = ? AND user_id = ?");
        $stmt->execute([$post_id, $user['id']]);
        
        $message = "✅ Post et médias supprimés !";
        $message_type = 'success';
        $action = 'list';
    } catch (Exception $e) {
        $message = "❌ Erreur suppression : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DES DONNÉES
// ================================================================

// Liste des clients (simulée pour demo)
$clients = [
    ['id' => 1, 'nom' => 'Client Demo 1'],
    ['id' => 2, 'nom' => 'Client Demo 2'],
];

// Vidéos workflow user
$user_videos = getUserVideos($user['id']);

// Médias convertis depuis images.php
$converted_medias = getUserConvertedMedias($user['id']);

// Liste des posts
$posts = [];
if ($action === 'list') {
    try {
        $stmt = $pdo->prepare("
            SELECT p.*, 
                   COUNT(pm.id) as media_count,
                   GROUP_CONCAT(pn.network) as networks
            FROM posts p 
            LEFT JOIN post_medias pm ON p.id = pm.post_id
            LEFT JOIN post_networks pn ON p.id = pn.post_id
            WHERE p.user_id = ? 
            GROUP BY p.id
            ORDER BY p.created_at DESC
        ");
        $stmt->execute([$user['id']]);
        $posts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Charger détails réseaux pour chaque post
        foreach ($posts as &$post) {
            $stmt = $pdo->prepare("
                SELECT * FROM post_medias 
                WHERE post_id = ? 
                ORDER BY display_order ASC 
                LIMIT 1
            ");
            $stmt->execute([$post['id']]);
            $post['preview_media'] = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $post['network_details'] = getPostNetworks($post['id']);
            $post['active_networks'] = count(array_filter($post['network_details'], function($n) { return $n['is_active']; }));
        }
    } catch (Exception $e) {
        error_log("Erreur chargement posts: " . $e->getMessage());
    }
}

// Post à éditer
$current_post = null;
$current_medias = [];
$current_networks = [];
if ($action === 'edit' && $post_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM posts WHERE id = ? AND user_id = ?");
        $stmt->execute([$post_id, $user['id']]);
        $current_post = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($current_post) {
            $stmt = $pdo->prepare("
                SELECT * FROM post_medias 
                WHERE post_id = ? 
                ORDER BY display_order ASC
            ");
            $stmt->execute([$post_id]);
            $current_medias = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $current_networks = getPostNetworks($post_id);
            
        } else {
            $message = "❌ Post non trouvé";
            $message_type = 'error';
            $action = 'list';
        }
    } catch (Exception $e) {
        $message = "❌ Erreur chargement post : " . $e->getMessage();
        $message_type = 'error';
        $action = 'list';
    }
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📝 Posts Manager Complet - SMG Studio</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .header-title h1 {
            font-size: 28px;
            color: #333;
            margin-bottom: 5px;
        }

        .header-title p {
            color: #666;
            font-size: 14px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .user-badge {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 10px 20px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 14px;
        }

        .quick-nav {
            display: flex;
            gap: 10px;
        }

        .quick-nav a {
            padding: 8px 16px;
            background: rgba(255, 255, 255, 0.9);
            color: #667eea;
            text-decoration: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
        }

        .quick-nav a:hover {
            background: white;
            transform: translateY(-1px);
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 22px;
            font-weight: 600;
            color: #333;
        }

        .btn {
            padding: 12px 20px;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 14px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }

        .form-input, .form-select, .form-textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 14px;
            transition: all 0.3s ease;
            background: white;
        }

        .form-input:focus, .form-select:focus, .form-textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .form-textarea {
            min-height: 120px;
            resize: vertical;
        }

        /* ================================================================
         * 🎛️ INTERRUPTEURS RÉSEAUX SOCIAUX (FROM NEW)
         * ================================================================ */
        .networks-switches {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 15px;
            margin: 15px 0;
        }

        .network-switch {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 15px;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .network-switch:hover {
            border-color: #667eea;
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.1);
        }

        .network-switch.active {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.05);
        }

        .network-switch input[type="checkbox"] {
            display: none;
        }

        .switch-label {
            display: flex;
            align-items: center;
            gap: 12px;
            cursor: pointer;
            user-select: none;
            position: relative;
        }

        .switch-slider {
            width: 50px;
            height: 26px;
            background: #ccc;
            border-radius: 13px;
            position: relative;
            transition: all 0.3s ease;
            flex-shrink: 0;
        }

        .switch-slider::before {
            content: '';
            position: absolute;
            top: 2px;
            left: 2px;
            width: 22px;
            height: 22px;
            background: white;
            border-radius: 50%;
            transition: all 0.3s ease;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }

        .network-switch input:checked + .switch-label .switch-slider {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
        }

        .network-switch input:checked + .switch-label .switch-slider::before {
            transform: translateX(24px);
        }

        .switch-icon {
            font-size: 24px;
            flex-shrink: 0;
        }

        .switch-info {
            flex: 1;
            min-width: 0;
        }

        .switch-text {
            font-weight: 600;
            font-size: 16px;
            color: #333;
            display: block;
            margin-bottom: 2px;
        }

        .switch-format {
            font-size: 12px;
            color: #666;
            display: block;
        }

        /* ================================================================
         * 🎬 WORKFLOW VIDÉOS (FROM OLD_POSTS.PHP)
         * ================================================================ */
        .workflow-videos {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }

        .videos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }

        .video-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            border: 2px solid transparent;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .video-card:hover {
            border-color: #667eea;
            transform: translateY(-2px);
        }

        .video-card.selected {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.1);
        }

        .video-preview {
            width: 100%;
            height: 100px;
            background: #000;
            border-radius: 8px;
            margin-bottom: 10px;
            position: relative;
            overflow: hidden;
        }

        .video-preview video {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .video-info {
            font-size: 12px;
        }

        .video-name {
            font-weight: 600;
            margin-bottom: 5px;
            color: #333;
        }

        .video-meta {
            color: #666;
            display: flex;
            justify-content: space-between;
        }

        /* ================================================================
         * 📸 CARROUSEL PHOTOS (FROM OLD_POSTS.PHP)
         * ================================================================ */
        .photos-upload {
            border: 3px dashed #e9ecef;
            border-radius: 12px;
            padding: 30px;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
            margin-bottom: 20px;
        }

        .photos-upload:hover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.05);
        }

        .photos-upload.dragover {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.1);
        }

        .photos-preview {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .photo-item {
            position: relative;
            border-radius: 10px;
            overflow: hidden;
            aspect-ratio: 1;
        }

        .photo-item img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .photo-remove {
            position: absolute;
            top: 5px;
            right: 5px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            border: none;
            border-radius: 50%;
            width: 25px;
            height: 25px;
            cursor: pointer;
            font-size: 12px;
        }

        /* ================================================================
         * 🖼️ SÉLECTEUR MÉDIAS CONVERTIS (FROM IMAGES.PHP)
         * ================================================================ */
        .converted-medias {
            background: #e7f3ff;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }

        .medias-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            gap: 10px;
            margin-top: 15px;
        }

        .media-card {
            background: white;
            border-radius: 8px;
            padding: 8px;
            border: 2px solid transparent;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .media-card:hover {
            border-color: #667eea;
        }

        .media-card.selected {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.1);
        }

        .media-preview {
            width: 100%;
            height: 80px;
            border-radius: 6px;
            overflow: hidden;
            margin-bottom: 8px;
        }

        .media-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .media-formats {
            display: flex;
            gap: 2px;
            font-size: 10px;
        }

        .format-badge {
            background: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 1px 4px;
            border-radius: 3px;
            font-weight: 600;
        }

        /* ================================================================
         * 📋 TABLEAU POSTS AVEC RÉSEAUX
         * ================================================================ */
        .posts-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .posts-table th,
        .posts-table td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }

        .posts-table th {
            background: rgba(102, 126, 234, 0.1);
            font-weight: 600;
            color: #333;
        }

        .posts-table tr:hover {
            background: rgba(102, 126, 234, 0.05);
        }

        .networks-display {
            display: flex;
            flex-wrap: wrap;
            gap: 4px;
            align-items: center;
        }

        .network-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            position: relative;
            border: 1px solid transparent;
        }

        .network-badge.active {
            background: rgba(78, 205, 196, 0.1);
            color: #4ecdc4;
            border-color: rgba(78, 205, 196, 0.3);
        }

        .network-badge.inactive {
            background: rgba(108, 117, 125, 0.1);
            color: #6c757d;
            border-color: rgba(108, 117, 125, 0.3);
        }

        .badge {
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
            white-space: nowrap;
        }

        .badge-success { background: #d4edda; color: #155724; }
        .badge-warning { background: #fff3cd; color: #856404; }
        .badge-danger { background: #f8d7da; color: #721c24; }
        .badge-secondary { background: #e2e3e5; color: #383d41; }
        .badge-dark { background: #d1ecf1; color: #0c5460; }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .empty-state {
            text-align: center;
            padding: 60px;
            color: #666;
        }

        .media-limits {
            background: #fff3cd;
            color: #856404;
            padding: 12px 15px;
            border-radius: 8px;
            font-size: 13px;
            margin: 15px 0;
            border: 1px solid #ffeaa7;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .networks-switches {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header principal -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">🚀</div>
                <div class="header-title">
                    <h1>Posts Manager Complet</h1>
                    <p>Multi-réseaux + Workflow vidéo + Carrousel photos + Gestionnaire médias</p>
                </div>
            </div>
            <div class="header-right">
                <div class="user-badge">
                    <?= htmlspecialchars($user['name']) ?> - <?= ucfirst($user['role']) ?>
                </div>
                <div class="quick-nav">
                    <a href="dashboard.php">🏠 Dashboard</a>
                    <a href="images.php">📸 Gestionnaire Photos</a>
                    <a href="clients.php">👥 Clients</a>
                    <a href="logout.php">🚪 Déconnexion</a>
                </div>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- ================================================================
         * 📋 LISTE DES POSTS MULTI-RÉSEAUX
         * ================================================================ -->
        <?php if ($action === 'list'): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">📋 Vos Posts Multi-Réseaux (<?= count($posts) ?>)</h2>
                    <div style="display: flex; gap: 10px;">
                        <a href="images.php?select_mode=1&post_action=new" class="btn btn-secondary">
                            📸 Choisir Photos
                        </a>
                        <a href="?action=new" class="btn btn-primary">
                            ➕ Nouveau Post
                        </a>
                    </div>
                </div>
                
                <?php if (empty($posts)): ?>
                    <div class="empty-state">
                        <div style="font-size: 64px; margin-bottom: 20px;">🚀</div>
                        <h3>Aucun post créé</h3>
                        <p style="margin: 15px 0;">Créez votre premier post multi-réseaux avec vidéos et photos !</p>
                        <div style="display: flex; gap: 15px; justify-content: center; margin-top: 20px;">
                            <a href="?action=new" class="btn btn-primary">➕ Créer mon premier post</a>
                            <a href="images.php" class="btn btn-secondary">📸 Gestionnaire Photos</a>
                        </div>
                    </div>
                <?php else: ?>
                    <div style="margin-bottom: 20px; padding: 15px; background: #e7f3ff; border-radius: 8px; font-size: 14px; color: #004085;">
                        🎯 <strong>Fonctionnalités actives :</strong> Multi-réseaux, Workflow vidéo, Carrousel photos, Gestionnaire médias FFmpeg
                    </div>
                    
                    <table class="posts-table">
                        <thead>
                            <tr>
                                <th>📱 Médias</th>
                                <th>📝 Post</th>
                                <th>🌐 Réseaux</th>
                                <th>👤 Client</th>
                                <th>📊 Statut</th>
                                <th>🔗 Lien</th>
                                <th>📅 Date</th>
                                <th>⚙️ Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($posts as $post): ?>
                                <tr>
                                    <td>
                                        <div style="display: flex; gap: 5px; align-items: center;">
                                            <?php if ($post['preview_media']): ?>
                                                <?php if ($post['preview_media']['media_type'] === 'video'): ?>
                                                    <video style="width: 40px; height: 40px; object-fit: cover; border-radius: 6px;" controls>
                                                        <source src="<?= $post['preview_media']['media_path'] ?>">
                                                    </video>
                                                <?php else: ?>
                                                    <img src="<?= $post['preview_media']['media_path'] ?>" style="width: 40px; height: 40px; object-fit: cover; border-radius: 6px;">
                                                <?php endif; ?>
                                            <?php endif; ?>
                                            
                                            <?php if ($post['media_count'] > 1): ?>
                                                <span style="background: #667eea; color: white; padding: 2px 8px; border-radius: 12px; font-size: 12px; font-weight: 600;">
                                                    +<?= $post['media_count'] - 1 ?>
                                                </span>
                                            <?php elseif ($post['media_count'] == 0): ?>
                                                <div style="width: 40px; height: 40px; display: flex; align-items: center; justify-content: center; background: #f8f9fa; color: #6c757d; border-radius: 6px;">📄</div>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <strong><?= htmlspecialchars($post['titre']) ?></strong><br>
                                        <small style="color: #666;">
                                            <?= substr(htmlspecialchars($post['contenu']), 0, 100) . (strlen($post['contenu']) > 100 ? '...' : '') ?>
                                        </small>
                                    </td>
                                    <td>
                                        <div class="networks-display">
                                            <?php if (!empty($post['network_details'])): ?>
                                                <?php foreach ($post['network_details'] as $network): ?>
                                                    <span class="network-badge <?= $network['is_active'] ? 'active' : 'inactive' ?>" 
                                                          title="<?= htmlspecialchars($network['display_name'] ?? ucfirst($network['network'])) ?>">
                                                        <?= getNetworkIcon($network['network']) ?>
                                                        <?php if (!$network['is_active']): ?>
                                                            <span style="opacity: 0.5;">⏸️</span>
                                                        <?php endif; ?>
                                                    </span>
                                                <?php endforeach; ?>
                                                <div style="font-size: 10px; color: #666; margin-left: 8px;">
                                                    <?= $post['active_networks'] ?>/<?= count($post['network_details']) ?> actifs
                                                </div>
                                            <?php else: ?>
                                                <span class="network-badge inactive">📱 Aucun réseau</span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <?= htmlspecialchars($post['client_nom'] ?? 'Personnel') ?>
                                    </td>
                                    <td>
                                        <?= getStatusBadge($post['statut']) ?>
                                    </td>
                                    <td>
                                        <?php if ($post['link_url']): ?>
                                            <span style="background: rgba(78, 205, 196, 0.1); color: #4ecdc4; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: 600;">🔗 Lien</span>
                                        <?php else: ?>
                                            <span style="color: #ccc;">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?= formatDate($post['created_at']) ?>
                                    </td>
                                    <td>
                                        <div style="display: flex; gap: 8px;">
                                            <a href="?action=edit&id=<?= $post['id'] ?>" class="btn btn-warning" style="padding: 6px 12px; font-size: 12px;">
                                                ✏️ Éditer
                                            </a>
                                            <a href="?action=delete&id=<?= $post['id'] ?>" 
                                               onclick="return confirm('Supprimer ce post multi-réseaux ?')" 
                                               class="btn btn-danger" style="padding: 6px 12px; font-size: 12px;">
                                                🗑️ Supprimer
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

        <!-- ================================================================
         * ➕ FORMULAIRE NOUVEAU POST COMPLET
         * ================================================================ -->
        <?php elseif ($action === 'new'): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">➕ Créer un post multi-réseaux</h2>
                    <a href="posts.php" class="btn btn-secondary">← Retour</a>
                </div>
                
                <form method="POST" enctype="multipart/form-data" id="postForm">
                    <!-- Informations de base -->
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">📝 Titre du post *</label>
                            <input type="text" name="titre" class="form-input" required 
                                   placeholder="Titre accrocheur de votre post">
                        </div>

                        <div class="form-group">
                            <label class="form-label">👤 Client</label>
                            <select name="client_id" class="form-select">
                                <option value="">Personnel</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>">
                                        <?= htmlspecialchars($client['nom']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">📄 Contenu *</label>
                        <textarea name="contenu" class="form-textarea" required 
                                  placeholder="Rédigez votre contenu ici..."></textarea>
                    </div>

                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">#️⃣ Hashtags</label>
                            <input type="text" name="hashtags" class="form-input" 
                                   placeholder="#social #media #marketing">
                        </div>

                        <div class="form-group">
                            <label class="form-label">🔗 Lien externe</label>
                            <input type="url" name="link_url" class="form-input" 
                                   placeholder="https://monsite.com/page">
                        </div>
                    </div>

                    <!-- SWITCHES RÉSEAUX SOCIAUX -->
                    <div class="form-group">
                        <label class="form-label">🌐 Réseaux sociaux *</label>
                        <div class="networks-switches">
                            <div class="network-switch">
                                <input type="checkbox" id="net_instagram" name="networks[]" value="instagram">
                                <label for="net_instagram" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📸</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Instagram</span>
                                        <span class="switch-format">Carré 1080x1080</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_facebook" name="networks[]" value="facebook">
                                <label for="net_facebook" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📘</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Facebook</span>
                                        <span class="switch-format">Paysage 1200x630</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_twitter" name="networks[]" value="twitter">
                                <label for="net_twitter" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">🐦</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Twitter</span>
                                        <span class="switch-format">Bannière 1024x512</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_linkedin" name="networks[]" value="linkedin">
                                <label for="net_linkedin" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">💼</span>
                                    <div class="switch-info">
                                        <span class="switch-text">LinkedIn</span>
                                        <span class="switch-format">Professionnel 1200x627</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_tiktok" name="networks[]" value="tiktok">
                                <label for="net_tiktok" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">🎵</span>
                                    <div class="switch-info">
                                        <span class="switch-text">TikTok</span>
                                        <span class="switch-format">Vertical 1080x1920</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_snapchat" name="networks[]" value="snapchat">
                                <label for="net_snapchat" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">👻</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Snapchat</span>
                                        <span class="switch-format">Vertical 1080x1920</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="net_youtube" name="networks[]" value="youtube">
                                <label for="net_youtube" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📺</span>
                                    <div class="switch-info">
                                        <span class="switch-text">YouTube</span>
                                        <span class="switch-format">Thumbnail 1280x720</span>
                                    </div>
                                </label>
                            </div>
                        </div>
                        
                        <div style="background: #e7f3ff; color: #004085; padding: 12px 15px; border-radius: 8px; font-size: 13px; margin-top: 15px;">
                            💡 Sélectionnez un ou plusieurs réseaux - Les médias seront optimisés pour chaque format
                        </div>
                    </div>

                    <!-- LIMITES MÉDIAS -->
                    <div class="media-limits">
                        📊 <strong>Limites par post :</strong>
                        <span style="display: inline-flex; align-items: center; gap: 8px; background: rgba(255, 255, 255, 0.8); padding: 4px 12px; border-radius: 12px; font-weight: 600; margin: 0 8px;">
                            🎬 Vidéos : <span id="videoCount">0</span>/1
                        </span>
                        <span style="display: inline-flex; align-items: center; gap: 8px; background: rgba(255, 255, 255, 0.8); padding: 4px 12px; border-radius: 12px; font-weight: 600; margin: 0 8px;">
                            📸 Photos : <span id="photoCount">0</span>/5
                        </span>
                    </div>

                    <!-- WORKFLOW VIDÉOS -->
                    <?php if (!empty($user_videos)): ?>
                        <div class="workflow-videos">
                            <h4>🎬 Sélectionner une vidéo de votre workflow</h4>
                            <p style="color: #666; font-size: 14px; margin-bottom: 15px;">
                                Cliquez sur une vidéo pour l'inclure dans votre post (max 1 vidéo)
                            </p>
                            
                            <div class="videos-grid">
                                <?php foreach ($user_videos as $video): ?>
                                    <div class="video-card" onclick="selectVideo('<?= htmlspecialchars($video['path']) ?>', this)">
                                        <div class="video-preview">
                                            <video>
                                                <source src="<?= $video['path'] ?>">
                                            </video>
                                            <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); color: white; font-size: 24px;">
                                                ▶️
                                            </div>
                                        </div>
                                        <div class="video-info">
                                            <div class="video-name"><?= htmlspecialchars($video['name']) ?></div>
                                            <div class="video-meta">
                                                <span><?= $video['type'] === 'active' ? '🟢 Actif' : '📦 Archivé' ?></span>
                                                <span><?= formatFileSize($video['size']) ?></span>
                                            </div>
                                            <div class="video-meta">
                                                <span style="font-size: 11px;"><?= htmlspecialchars($video['project']) ?></span>
                                                <span style="font-size: 11px;"><?= formatDate(date('Y-m-d H:i:s', $video['date'])) ?></span>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <input type="hidden" name="workflow_video" id="selectedVideo" value="">
                        </div>
                    <?php endif; ?>

                    <!-- PHOTOS CONVERTIES DEPUIS IMAGES.PHP -->
                    <?php if (!empty($converted_medias)): ?>
                        <div class="converted-medias">
                            <h4>📸 Photos converties depuis le gestionnaire</h4>
                            <p style="color: #666; font-size: 14px; margin-bottom: 15px;">
                                Sélectionnez les photos déjà converties par FFmpeg (max 5 total)
                            </p>
                            
                            <div class="medias-grid">
                                <?php foreach (array_slice($converted_medias, 0, 10) as $media): ?>
                                    <div class="media-card" onclick="toggleConvertedMedia(<?= $media['id'] ?>, this)">
                                        <div class="media-preview">
                                            <?php if ($media['instagram_path'] && file_exists($media['instagram_path'])): ?>
                                                <img src="<?= $media['instagram_path'] ?>" alt="<?= htmlspecialchars($media['original_name']) ?>">
                                            <?php else: ?>
                                                <div style="display: flex; align-items: center; justify-content: center; height: 100%; background: #f8f9fa; color: #999; font-size: 12px;">
                                                    Conversion...
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="media-formats">
                                            <?php if ($media['instagram_path']): ?><span class="format-badge">📸</span><?php endif; ?>
                                            <?php if ($media['facebook_path']): ?><span class="format-badge">📘</span><?php endif; ?>
                                            <?php if ($media['stories_path']): ?><span class="format-badge">📺</span><?php endif; ?>
                                            <?php if ($media['twitter_path']): ?><span class="format-badge">🐦</span><?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <input type="hidden" name="selected_medias" id="selectedConvertedMedias" value="">
                            
                            <div style="text-align: center; margin-top: 15px;">
                                <a href="images.php?select_mode=1&post_action=new" class="btn btn-secondary" style="padding: 8px 16px; font-size: 14px;">
                                    🖼️ Gestionnaire Photos Complet
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- UPLOAD INTELLIGENT - PHOTOS OU VIDÉO -->
                    <div class="form-group">
                        <label class="form-label">📷 Médias Upload</label>
                        
                        <!-- Sélecteur type média -->
                        <div style="display: flex; gap: 15px; margin-bottom: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px;">
                            <label style="display: flex; align-items: center; gap: 8px; cursor: pointer; font-weight: 600;">
                                <input type="radio" name="media_type" value="photos" checked onchange="switchMediaType('photos')">
                                📸 Photos (carrousel max 5)
                            </label>
                            <label style="display: flex; align-items: center; gap: 8px; cursor: pointer; font-weight: 600;">
                                <input type="radio" name="media_type" value="video" onchange="switchMediaType('video')">
                                🎬 Vidéo MP4 (une seule)
                            </label>
                        </div>
                        
                        <!-- Zone upload photos -->
                        <div id="photosUploadSection">
                            <div class="photos-upload" id="photosUpload">
                                <input type="file" name="photos[]" multiple accept="image/*" style="display: none;" id="photosInput">
                                <div style="font-size: 48px; margin-bottom: 15px;">📷</div>
                                <h4>Ajoutez vos photos</h4>
                                <p style="color: #666; margin: 10px 0;">
                                    Cliquez ou glissez vos images ici (max 5 photos)
                                </p>
                                <p style="font-size: 12px; color: #999;">
                                    JPG, PNG, WebP - Max 10MB par photo
                                </p>
                            </div>
                            <div class="photos-preview" id="photosPreview"></div>
                        </div>
                        
                        <!-- Zone upload vidéo -->
                        <div id="videoUploadSection" style="display: none;">
                            <div class="photos-upload" id="videoUpload">
                                <input type="file" name="upload_video" accept="video/mp4,video/webm,video/mov,video/avi" style="display: none;" id="videoInput">
                                <div style="font-size: 48px; margin-bottom: 15px;">🎬</div>
                                <h4>Ajoutez votre vidéo</h4>
                                <p style="color: #666; margin: 10px 0;">
                                    Cliquez ou glissez votre vidéo ici (1 seule vidéo)
                                </p>
                                <p style="font-size: 12px; color: #999;">
                                    MP4, WebM, MOV, AVI - Max 100MB
                                </p>
                            </div>
                            <div class="photos-preview" id="videoPreview"></div>
                        </div>
                        
                        <div style="background: #e7f3ff; color: #004085; padding: 12px 15px; border-radius: 8px; font-size: 13px; margin-top: 15px;">
                            💡 <strong>Choix intelligent :</strong> Soit plusieurs photos en carrousel, soit une seule vidéo. Vous pouvez aussi utiliser les vidéos du workflow ci-dessus.
                        </div>
                    </div>

                    <!-- Médias sélectionnés depuis images.php -->
                    <?php if (!empty($selected_media_ids)): ?>
                        <div style="background: #d4edda; color: #155724; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                            <strong>✅ Photos pré-sélectionnées :</strong> <?= count($selected_media_ids) ?> média(s) depuis le gestionnaire
                            <input type="hidden" name="selected_medias" value="<?= implode(',', $selected_media_ids) ?>">
                        </div>
                    <?php endif; ?>

                    <div style="display: flex; gap: 15px; margin-top: 30px;">
                        <button type="submit" name="create_post" class="btn btn-primary">
                            ✅ Créer le post multi-réseaux
                        </button>
                        <a href="posts.php" class="btn btn-secondary">
                            ❌ Annuler
                        </a>
                    </div>
                </form>
            </div>

        <!-- ================================================================
         * ✏️ FORMULAIRE MODIFICATION POST COMPLET
         * ================================================================ -->
        <?php elseif ($action === 'edit' && $current_post): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">✏️ Modifier le post multi-réseaux</h2>
                    <a href="posts.php" class="btn btn-secondary">← Retour</a>
                </div>
                
                <div style="background: #fff3cd; color: #856404; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                    <strong>📝 Post en cours d'édition :</strong> <?= htmlspecialchars($current_post['titre']) ?>
                </div>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="post_id" value="<?= $current_post['id'] ?>">
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">📝 Titre du post *</label>
                            <input type="text" name="titre" class="form-input" required 
                                   value="<?= htmlspecialchars($current_post['titre']) ?>">
                        </div>

                        <div class="form-group">
                            <label class="form-label">📊 Statut</label>
                            <select name="statut" class="form-select">
                                <option value="draft" <?= $current_post['statut'] === 'draft' ? 'selected' : '' ?>>📝 Brouillon</option>
                                <option value="scheduled" <?= $current_post['statut'] === 'scheduled' ? 'selected' : '' ?>>⏰ Programmé</option>
                                <option value="published" <?= $current_post['statut'] === 'published' ? 'selected' : '' ?>>✅ Publié</option>
                                <option value="cancelled" <?= $current_post['statut'] === 'cancelled' ? 'selected' : '' ?>>🚫 Annulé</option>
                            </select>
                        </div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">📄 Contenu *</label>
                        <textarea name="contenu" class="form-textarea" required><?= htmlspecialchars($current_post['contenu']) ?></textarea>
                    </div>

                    <div class="form-grid">
                        <div class="form-group">
                            <label class="form-label">#️⃣ Hashtags</label>
                            <input type="text" name="hashtags" class="form-input" 
                                   value="<?= htmlspecialchars($current_post['hashtags']) ?>">
                        </div>

                        <div class="form-group">
                            <label class="form-label">🔗 Lien externe</label>
                            <input type="url" name="link_url" class="form-input" 
                                   value="<?= htmlspecialchars($current_post['link_url'] ?? '') ?>">
                        </div>

                        <div class="form-group">
                            <label class="form-label">👤 Client</label>
                            <select name="client_id" class="form-select">
                                <option value="">Personnel</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>" 
                                            <?= $current_post['client_id'] == $client['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($client['nom']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <!-- MODIFICATION RÉSEAUX SOCIAUX -->
                    <div class="form-group">
                        <label class="form-label">🌐 Réseaux sociaux *</label>
                        <div class="networks-switches">
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_instagram" name="networks[]" value="instagram"
                                       <?= in_array('instagram', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_instagram" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📸</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Instagram</span>
                                        <span class="switch-format">Carré 1080x1080</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_facebook" name="networks[]" value="facebook"
                                       <?= in_array('facebook', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_facebook" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📘</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Facebook</span>
                                        <span class="switch-format">Paysage 1200x630</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_twitter" name="networks[]" value="twitter"
                                       <?= in_array('twitter', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_twitter" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">🐦</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Twitter</span>
                                        <span class="switch-format">Bannière 1024x512</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_linkedin" name="networks[]" value="linkedin"
                                       <?= in_array('linkedin', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_linkedin" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">💼</span>
                                    <div class="switch-info">
                                        <span class="switch-text">LinkedIn</span>
                                        <span class="switch-format">Professionnel 1200x627</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_tiktok" name="networks[]" value="tiktok"
                                       <?= in_array('tiktok', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_tiktok" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">🎵</span>
                                    <div class="switch-info">
                                        <span class="switch-text">TikTok</span>
                                        <span class="switch-format">Vertical 1080x1920</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_snapchat" name="networks[]" value="snapchat"
                                       <?= in_array('snapchat', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_snapchat" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">👻</span>
                                    <div class="switch-info">
                                        <span class="switch-text">Snapchat</span>
                                        <span class="switch-format">Vertical 1080x1920</span>
                                    </div>
                                </label>
                            </div>
                            
                            <div class="network-switch">
                                <input type="checkbox" id="edit_net_youtube" name="networks[]" value="youtube"
                                       <?= in_array('youtube', array_column($current_networks, 'network')) ? 'checked' : '' ?>>
                                <label for="edit_net_youtube" class="switch-label">
                                    <span class="switch-slider"></span>
                                    <span class="switch-icon">📺</span>
                                    <div class="switch-info">
                                        <span class="switch-text">YouTube</span>
                                        <span class="switch-format">Thumbnail 1280x720</span>
                                    </div>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Affichage réseaux actuels -->
                        <?php if (!empty($current_networks)): ?>
                            <div style="margin-top: 15px;">
                                <h6 style="margin-bottom: 10px; color: #666;">📊 Réseaux actuels :</h6>
                                <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                                    <?php foreach ($current_networks as $network): ?>
                                        <div style="display: flex; flex-direction: column; gap: 4px; padding: 8px 12px; background: #f8f9fa; border-radius: 8px; border: 1px solid #e9ecef;">
                                            <span class="network-badge <?= $network['is_active'] ? 'active' : 'inactive' ?>">
                                                <?= getNetworkIcon($network['network']) ?>
                                                <?= htmlspecialchars($network['display_name'] ?? ucfirst($network['network'])) ?>
                                            </span>
                                            <span style="font-size: 10px; color: #666; text-transform: uppercase; letter-spacing: 0.5px;">
                                                Status: <?= $network['export_status'] ?>
                                            </span>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Médias existants -->
                    <?php if (!empty($current_medias)): ?>
                        <div class="form-group">
                            <label class="form-label">📱 Médias actuels</label>
                            <div class="photos-preview">
                                <?php foreach ($current_medias as $media): ?>
                                    <div class="photo-item">
                                        <?php if ($media['media_type'] === 'video'): ?>
                                            <video controls style="width: 100%; height: 100%; object-fit: cover;">
                                                <source src="<?= $media['media_path'] ?>">
                                            </video>
                                        <?php else: ?>
                                            <img src="<?= $media['media_path'] ?>" alt="Media">
                                        <?php endif; ?>
                                        
                                        <a href="?action=delete_media&media_id=<?= $media['id'] ?>&post_id=<?= $current_post['id'] ?>" 
                                           onclick="return confirm('Supprimer ce média ?')" 
                                           class="photo-remove">×</a>
                                        
                                        <div style="position: absolute; bottom: 5px; left: 5px; background: rgba(0,0,0,0.7); color: white; padding: 2px 6px; border-radius: 4px; font-size: 10px;">
                                            <?= ucfirst($media['media_source']) ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>

                    <div style="display: flex; gap: 15px; margin-top: 30px;">
                        <button type="submit" name="update_post" class="btn btn-success">
                            ✅ Sauvegarder les modifications
                        </button>
                        <a href="posts.php" class="btn btn-secondary">
                            ❌ Annuler
                        </a>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

    <script>
        /**
         * ================================================================
         * 🎯 JAVASCRIPT COMPLET - FUSION DES FONCTIONNALITÉS
         * ================================================================
         */

        // Variables globales
        let selectedPhotos = [];
        let selectedVideo = null;
        let selectedConvertedMedias = [];
        let selectedUploadVideo = null;
        const MAX_PHOTOS = 5;
        const MAX_VIDEOS = 1;

        // ================================================================
        // 🎛️ GESTION INTERRUPTEURS RÉSEAUX
        // ================================================================
        document.addEventListener('DOMContentLoaded', function() {
            // Activation des interrupteurs
            document.querySelectorAll('.network-switch input[type="checkbox"]').forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    const switchElement = this.closest('.network-switch');
                    
                    if (this.checked) {
                        switchElement.classList.add('active');
                    } else {
                        switchElement.classList.remove('active');
                    }
                    
                    updateNetworkSelection();
                });
            });
            
            // Initialisation édition avec réseaux pré-sélectionnés
            document.querySelectorAll('.network-switch input[type="checkbox"]:checked').forEach(checkbox => {
                const switchElement = checkbox.closest('.network-switch');
                switchElement.classList.add('active');
            });
            
            updateNetworkSelection();
        });

        function updateNetworkSelection() {
            const selectedNetworks = document.querySelectorAll('.network-switch input[type="checkbox"]:checked');
            const submitBtn = document.querySelector('button[name="create_post"], button[name="update_post"]');
            
            if (selectedNetworks.length === 0) {
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = '⚠️ Sélectionnez au moins un réseau';
                }
            } else {
                if (submitBtn) {
                    submitBtn.disabled = false;
                    const isUpdate = submitBtn.name === 'update_post';
                    submitBtn.innerHTML = isUpdate ? 
                        `✅ Sauvegarder pour ${selectedNetworks.length} réseau${selectedNetworks.length > 1 ? 'x' : ''}` :
                        `✅ Créer pour ${selectedNetworks.length} réseau${selectedNetworks.length > 1 ? 'x' : ''}`;
                }
            }
        }

        // ================================================================
        // 🎬 SÉLECTEUR TYPE MÉDIA
        // ================================================================
        function switchMediaType(type) {
            const photosSection = document.getElementById('photosUploadSection');
            const videoSection = document.getElementById('videoUploadSection');
            
            if (type === 'photos') {
                photosSection.style.display = 'block';
                videoSection.style.display = 'none';
                // Reset vidéo upload
                selectedUploadVideo = null;
                document.getElementById('videoPreview').innerHTML = '';
                const videoInput = document.getElementById('videoInput');
                if (videoInput) videoInput.value = '';
            } else if (type === 'video') {
                photosSection.style.display = 'none';
                videoSection.style.display = 'block';
                // Reset photos upload
                selectedPhotos = [];
                document.getElementById('photosPreview').innerHTML = '';
                const photosInput = document.getElementById('photosInput');
                if (photosInput) photosInput.value = '';
                resetPhotosUpload();
            }
            
            updateMediaCounters();
        }

        // ================================================================
        // 🎬 SÉLECTION VIDÉO WORKFLOW
        // ================================================================
        function selectVideo(videoPath, cardElement) {
            // Désélectionner toutes les autres cartes
            document.querySelectorAll('.video-card').forEach(card => {
                card.classList.remove('selected');
            });
            
            // Vérifier limite vidéo
            if (selectedVideo && selectedVideo !== videoPath) {
                selectedVideo = videoPath;
            } else if (!selectedVideo) {
                selectedVideo = videoPath;
            } else {
                selectedVideo = null;
                videoPath = '';
            }
            
            // Sélectionner la carte cliquée si vidéo sélectionnée
            if (selectedVideo) {
                cardElement.classList.add('selected');
            }
            
            // Mettre à jour le champ caché
            document.getElementById('selectedVideo').value = selectedVideo || '';
            
            // Mettre à jour compteurs
            updateMediaCounters();
            
            console.log('Vidéo sélectionnée:', selectedVideo || 'Aucune');
        }

        // ================================================================
        // 📸 SÉLECTION MÉDIAS CONVERTIS
        // ================================================================
        function toggleConvertedMedia(mediaId, cardElement) {
            const index = selectedConvertedMedias.indexOf(mediaId);
            
            if (index > -1) {
                // Désélectionner
                selectedConvertedMedias.splice(index, 1);
                cardElement.classList.remove('selected');
            } else {
                // Vérifier limite photos
                if (selectedConvertedMedias.length + selectedPhotos.length >= MAX_PHOTOS) {
                    alert(`⚠️ Maximum ${MAX_PHOTOS} photos autorisées par post`);
                    return;
                }
                
                // Sélectionner
                selectedConvertedMedias.push(mediaId);
                cardElement.classList.add('selected');
            }
            
            // Mettre à jour le champ caché
            document.getElementById('selectedConvertedMedias').value = selectedConvertedMedias.join(',');
            
            updateMediaCounters();
        }

        // ================================================================
        // 🎬 UPLOAD VIDÉO MP4
        // ================================================================
        document.addEventListener('DOMContentLoaded', function() {
            const videoUploadZone = document.getElementById('videoUpload');
            const videoInput = document.getElementById('videoInput');
            const videoPreview = document.getElementById('videoPreview');
            
            if (videoUploadZone && videoInput) {
                // Clic sur la zone d'upload
                videoUploadZone.addEventListener('click', () => {
                    videoInput.click();
                });

                // Changement de fichier vidéo
                videoInput.addEventListener('change', handleVideoSelect);

                // Drag & Drop pour vidéo
                ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                    videoUploadZone.addEventListener(eventName, preventDefaults, false);
                });

                ['dragenter', 'dragover'].forEach(eventName => {
                    videoUploadZone.addEventListener(eventName, () => {
                        videoUploadZone.classList.add('dragover');
                    }, false);
                });

                ['dragleave', 'drop'].forEach(eventName => {
                    videoUploadZone.addEventListener(eventName, () => {
                        videoUploadZone.classList.remove('dragover');
                    }, false);
                });

                videoUploadZone.addEventListener('drop', function(e) {
                    const files = e.dataTransfer.files;
                    if (files.length > 0) {
                        videoInput.files = files;
                        handleVideoSelect({ target: { files: files } });
                    }
                });
            }
        });

        function handleVideoSelect(e) {
            const file = e.target.files[0];
            if (!file) return;
            
            // Vérifier que c'est une vidéo
            if (!file.type.startsWith('video/')) {
                alert('⚠️ Sélectionnez un fichier vidéo (MP4, WebM, MOV, AVI)');
                return;
            }
            
            // Vérifier taille (100MB max)
            if (file.size > 100 * 1024 * 1024) {
                alert('⚠️ Vidéo trop volumineuse (max 100MB)');
                return;
            }
            
            selectedUploadVideo = file;
            
            // Créer preview vidéo
            const videoPreview = document.getElementById('videoPreview');
            const videoUploadZone = document.getElementById('videoUpload');
            
            const reader = new FileReader();
            reader.onload = function(e) {
                videoPreview.innerHTML = `
                    <div class="photo-item" style="aspect-ratio: 16/9;">
                        <video controls style="width: 100%; height: 100%; object-fit: cover; border-radius: 10px;">
                            <source src="${e.target.result}" type="${file.type}">
                        </video>
                        <button type="button" class="photo-remove" onclick="removeUploadVideo()">×</button>
                        <div style="position: absolute; bottom: 5px; left: 5px; background: rgba(0,0,0,0.7); color: white; padding: 2px 6px; border-radius: 4px; font-size: 10px;">
                            Upload • ${formatFileSize(file.size)}
                        </div>
                    </div>
                `;
                
                // Mettre à jour la zone upload
                videoUploadZone.innerHTML = `
                    <div style="font-size: 48px; margin-bottom: 15px;">✅</div>
                    <h4>Vidéo sélectionnée</h4>
                    <p style="color: #666; margin: 10px 0;">
                        ${file.name} (${formatFileSize(file.size)})
                    </p>
                    <p style="font-size: 12px; color: #999;">
                        Cliquez pour changer de vidéo
                    </p>
                `;
            };
            reader.readAsDataURL(file);
            
            updateMediaCounters();
        }

        function removeUploadVideo() {
            selectedUploadVideo = null;
            document.getElementById('videoPreview').innerHTML = '';
            document.getElementById('videoInput').value = '';
            
            // Reset zone upload
            const videoUploadZone = document.getElementById('videoUpload');
            videoUploadZone.innerHTML = `
                <div style="font-size: 48px; margin-bottom: 15px;">🎬</div>
                <h4>Ajoutez votre vidéo</h4>
                <p style="color: #666; margin: 10px 0;">
                    Cliquez ou glissez votre vidéo ici (1 seule vidéo)
                </p>
                <p style="font-size: 12px; color: #999;">
                    MP4, WebM, MOV, AVI - Max 100MB
                </p>
            `;
            
            updateMediaCounters();
        }

        function formatFileSize(bytes) {
            const units = ['B', 'KB', 'MB', 'GB'];
            let size = bytes;
            let unitIndex = 0;
            
            while (size >= 1024 && unitIndex < units.length - 1) {
                size /= 1024;
                unitIndex++;
            }
            
            return Math.round(size * 10) / 10 + ' ' + units[unitIndex];
        }

        // ================================================================
        // 📷 GESTION PHOTOS CARROUSEL (ENHANCED)
        // ================================================================
        document.addEventListener('DOMContentLoaded', function() {
            const uploadZone = document.getElementById('photosUpload');
            const fileInput = document.getElementById('photosInput');
            const previewZone = document.getElementById('photosPreview');
            
            if (!uploadZone || !fileInput) return;

            // Clic sur la zone d'upload
            uploadZone.addEventListener('click', () => {
                fileInput.click();
            });

            // Changement de fichiers
            fileInput.addEventListener('change', handleFileSelect);

            // Drag & Drop
            ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                uploadZone.addEventListener(eventName, preventDefaults, false);
            });

            ['dragenter', 'dragover'].forEach(eventName => {
                uploadZone.addEventListener(eventName, () => {
                    uploadZone.classList.add('dragover');
                }, false);
            });

            ['dragleave', 'drop'].forEach(eventName => {
                uploadZone.addEventListener(eventName, () => {
                    uploadZone.classList.remove('dragover');
                }, false);
            });

            uploadZone.addEventListener('drop', function(e) {
                const files = e.dataTransfer.files;
                handleFiles(files);
            });
        });

        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        function handleFileSelect(e) {
            handleFiles(e.target.files);
        }

        function handleFiles(files) {
            const previewZone = document.getElementById('photosPreview');
            
            Array.from(files).forEach((file, index) => {
                if (file.type.startsWith('image/')) {
                    // Vérifier limite photos
                    if (selectedPhotos.length + selectedConvertedMedias.length >= MAX_PHOTOS) {
                        alert(`⚠️ Maximum ${MAX_PHOTOS} photos autorisées par post`);
                        return;
                    }
                    
                    selectedPhotos.push(file);
                    
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        const photoDiv = document.createElement('div');
                        photoDiv.className = 'photo-item';
                        photoDiv.innerHTML = `
                            <img src="${e.target.result}" alt="Photo ${selectedPhotos.length}">
                            <button type="button" class="photo-remove" onclick="removePhoto(${selectedPhotos.length - 1}, this)">×</button>
                        `;
                        previewZone.appendChild(photoDiv);
                    };
                    reader.readAsDataURL(file);
                }
            });

            updatePhotosStatus();
            updateMediaCounters();
        }

        function removePhoto(index, buttonElement) {
            selectedPhotos.splice(index, 1);
            buttonElement.parentElement.remove();
            updatePhotosStatus();
            updateMediaCounters();
            
            // Recréer le FileList pour le formulaire
            updateFileInput();
        }

        function updatePhotosStatus() {
            const uploadZone = document.getElementById('photosUpload');
            const count = selectedPhotos.length;
            
            if (count > 0) {
                uploadZone.innerHTML = `
                    <div style="font-size: 48px; margin-bottom: 15px;">✅</div>
                    <h4>${count} photo${count > 1 ? 's' : ''} sélectionnée${count > 1 ? 's' : ''}</h4>
                    <p style="color: #666;">Cliquez pour en ajouter d'autres</p>
                `;
            }
        }

        function resetPhotosUpload() {
            const uploadZone = document.getElementById('photosUpload');
            uploadZone.innerHTML = `
                <div style="font-size: 48px; margin-bottom: 15px;">📷</div>
                <h4>Ajoutez vos photos</h4>
                <p style="color: #666; margin: 10px 0;">
                    Cliquez ou glissez vos images ici (max 5 photos)
                </p>
                <p style="font-size: 12px; color: #999;">
                    JPG, PNG, WebP - Max 10MB par photo
                </p>
            `;
        }

        function updateFileInput() {
            const fileInput = document.getElementById('photosInput');
            if (!fileInput) return;
            
            // Créer un nouveau DataTransfer pour reconstruire la FileList
            const dt = new DataTransfer();
            selectedPhotos.forEach(file => dt.items.add(file));
            fileInput.files = dt.files;
        }

        // ================================================================
        // 📊 COMPTEURS MÉDIAS AVEC LIMITES
        // ================================================================
        function updateMediaCounters() {
            const videoCount = document.getElementById('videoCount');
            const photoCount = document.getElementById('photoCount');
            
            // Compter vidéos : workflow OU upload
            let totalVideos = 0;
            if (selectedVideo) totalVideos++;
            if (selectedUploadVideo) totalVideos++;
            
            // Compter photos : upload + converties
            let totalPhotos = selectedPhotos.length + selectedConvertedMedias.length;
            
            if (videoCount) videoCount.textContent = totalVideos;
            if (photoCount) photoCount.textContent = totalPhotos;
            
            // Validation limites
            const mediaTypeRadios = document.querySelectorAll('input[name="media_type"]');
            const selectedType = Array.from(mediaTypeRadios).find(r => r.checked)?.value;
            
            // Désactiver options si limites atteintes
            if (selectedType === 'video' && totalVideos >= MAX_VIDEOS) {
                // Limite vidéo atteinte
                console.log('Limite vidéo atteinte');
            }
            
            if (selectedType === 'photos' && totalPhotos >= MAX_PHOTOS) {
                // Limite photos atteinte
                const photosInput = document.getElementById('photosInput');
                if (photosInput) photosInput.disabled = true;
            } else {
                const photosInput = document.getElementById('photosInput');
                if (photosInput) photosInput.disabled = false;
            }
        }

        // ================================================================
        // ✨ AMÉLIORATIONS UX
        // ================================================================

        // Auto-resize textarea
        document.querySelectorAll('textarea').forEach(textarea => {
            textarea.addEventListener('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
        });

        // Compteur de caractères avec limites réseaux
        document.querySelectorAll('textarea').forEach(textarea => {
            const counter = document.createElement('div');
            counter.style.cssText = 'text-align: right; font-size: 12px; color: #666; margin-top: 5px;';
            textarea.parentNode.appendChild(counter);
            
            function updateCounter() {
                const length = textarea.value.length;
                const selectedNetworks = Array.from(document.querySelectorAll('.network-switch input[type="checkbox"]:checked'))
                    .map(cb => cb.value);
                
                const limits = {
                    'instagram': 2200,
                    'twitter': 280,
                    'facebook': 63206,
                    'linkedin': 3000,
                    'tiktok': 300,
                    'snapchat': 250,
                    'youtube': 5000
                };
                
                let minLimit = Math.min(...selectedNetworks.map(n => limits[n] || 2000));
                
                counter.innerHTML = `${length} caractères`;
                
                if (selectedNetworks.length > 0) {
                    counter.innerHTML += ` / ${minLimit} (limite ${selectedNetworks.join(', ')})`;
                }
                
                // Couleur selon limite
                if (length > minLimit) {
                    counter.style.color = '#dc3545';
                } else if (length > minLimit * 0.8) {
                    counter.style.color = '#ffc107';
                } else {
                    counter.style.color = '#666';
                }
            }
            
            textarea.addEventListener('input', updateCounter);
            document.querySelectorAll('.network-switch input[type="checkbox"]').forEach(cb => {
                cb.addEventListener('change', updateCounter);
            });
            updateCounter();
        });

        // Initialisation compteurs médias
        updateMediaCounters();

        console.log('🚀 Posts Manager COMPLET Ready - Fusion + Upload Intelligent par Papy 77 !');
        console.log('✅ Multi-réseaux avec switches (7 réseaux)');
        console.log('🎬 Workflow vidéos intégré');
        console.log('📸 Carrousel photos drag & drop');
        console.log('🎥 Upload vidéo MP4 direct');
        console.log('🎯 Sélection intelligente photos OU vidéo');
        console.log('🖼️ Gestionnaire médias FFmpeg');
        console.log('🔗 Liens externes + compteurs caractères');
        console.log('📱 Interface responsive premium');
    </script>
</body>
</html>