<?php
/**
 * CLIPMAKER PRO - WORKFLOW BACKEND
 * Gestion complète du workflow vidéo 4 étapes
 * Compatible avec votre structure DB existante
 */

session_start();
require_once 'auth_functions.php';

// Configuration basée sur votre .env
$config = [
    'db' => [
        'host' => $_ENV['DB_HOST'] ?? 'localhost',
        'name' => $_ENV['DB_NAME'] ?? 'laloedata',
        'user' => $_ENV['DB_USER'] ?? 'laloedata',
        'pass' => $_ENV['DB_PASS'] ?? 'PiL256Mx78!',
        'charset' => $_ENV['DB_CHARSET'] ?? 'utf8mb4'
    ],
    'paths' => [
        'uploads' => '/uploads/clipmaker/',
        'workflow' => '/uploads/workflow/',
        'exports' => '/uploads/clipmaker/exports/',
        'temp' => '/tmp/clipmaker/'
    ],
    'ffmpeg' => [
        'path' => 'ffmpeg',
        'threads' => (int)($_ENV['CLIPMAKER_FFMPEG_THREADS'] ?? 4),
        'presets' => [
            'draft' => '-crf 28 -preset ultrafast',
            'preview' => '-crf 23 -preset fast', 
            'broadcast' => '-crf 18 -preset medium',
            'web' => '-crf 23 -preset fast -movflags +faststart',
            'mobile' => '-crf 26 -preset fast -movflags +faststart'
        ]
    ],
    'social_formats' => [
        'youtube' => [
            'resolution' => '1920x1080',
            'fps' => 30,
            'bitrate' => '5M',
            'audio_bitrate' => '192k'
        ],
        'instagram_post' => [
            'resolution' => '1080x1080',
            'fps' => 30,
            'bitrate' => '3M',
            'audio_bitrate' => '128k'
        ],
        'instagram_story' => [
            'resolution' => '1080x1920',
            'fps' => 30,
            'bitrate' => '2M',
            'audio_bitrate' => '128k'
        ],
        'tiktok' => [
            'resolution' => '1080x1920',
            'fps' => 60,
            'bitrate' => '3M',
            'audio_bitrate' => '128k'
        ],
        'facebook' => [
            'resolution' => '1920x1080',
            'fps' => 30,
            'bitrate' => '4M',
            'audio_bitrate' => '192k'
        ],
        'linkedin' => [
            'resolution' => '1920x1080',
            'fps' => 30,
            'bitrate' => '3M',
            'audio_bitrate' => '192k'
        ],
        'twitter' => [
            'resolution' => '1280x720',
            'fps' => 30,
            'bitrate' => '2M',
            'audio_bitrate' => '128k'
        ]
    ]
];

/**
 * Classe principale du Workflow Clipmaker
 */
class ClipmakerWorkflow {
    private $db;
    private $config;
    private $user_id;
    
    public function __construct($config) {
        $this->config = $config;
        $this->connectDB();
        $this->user_id = $_SESSION['user_id'] ?? 0;
    }
    
    private function connectDB() {
        try {
            $dsn = "mysql:host={$this->config['db']['host']};dbname={$this->config['db']['name']};charset={$this->config['db']['charset']}";
            $this->db = new PDO($dsn, $this->config['db']['user'], $this->config['db']['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } catch (PDOException $e) {
            throw new Exception("Erreur de connexion DB: " . $e->getMessage());
        }
    }
    
    /**
     * ÉTAPE 1: MONTAGE VIDÉO INTERACTIF
     */
    public function uploadMedia($file, $project_id = null) {
        try {
            $allowed_types = ['video/mp4', 'video/mov', 'video/avi', 'audio/mp3', 'audio/wav', 'image/jpeg', 'image/png'];
            
            if (!in_array($file['type'], $allowed_types)) {
                throw new Exception("Type de fichier non supporté: " . $file['type']);
            }
            
            // Génération nom unique
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $stored_filename = 'media_' . time() . '_' . uniqid() . '.' . $extension;
            $file_path = $this->config['paths']['uploads'] . 'media/' . $stored_filename;
            
            // Upload physique
            if (!move_uploaded_file($file['tmp_name'], $_SERVER['DOCUMENT_ROOT'] . $file_path)) {
                throw new Exception("Erreur lors de l'upload du fichier");
            }
            
            // Analyse métadonnées avec FFprobe
            $metadata = $this->analyzeMedia($_SERVER['DOCUMENT_ROOT'] . $file_path);
            
            // Insertion en DB
            $stmt = $this->db->prepare("
                INSERT INTO clipmaker_media 
                (user_id, project_id, original_filename, stored_filename, file_path, file_type, 
                 mime_type, file_size, width, height, duration, metadata, conversion_status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'completed', NOW())
            ");
            
            $file_type = explode('/', $file['type'])[0]; // video, audio, image
            
            $stmt->execute([
                $this->user_id,
                $project_id,
                $file['name'],
                $stored_filename,
                $file_path,
                $file_type,
                $file['type'],
                $file['size'],
                $metadata['width'] ?? null,
                $metadata['height'] ?? null,
                $metadata['duration'] ?? null,
                json_encode($metadata)
            ]);
            
            $media_id = $this->db->lastInsertId();
            
            // Génération thumbnail pour vidéos
            if ($file_type === 'video') {
                $this->generateThumbnail($file_path, $media_id);
            }
            
            return [
                'success' => true,
                'media_id' => $media_id,
                'filename' => $stored_filename,
                'metadata' => $metadata
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    private function analyzeMedia($file_path) {
        $cmd = "ffprobe -v quiet -print_format json -show_format -show_streams " . escapeshellarg($file_path);
        $output = shell_exec($cmd);
        
        if (!$output) {
            return ['error' => 'Impossible d\'analyser le fichier'];
        }
        
        return json_decode($output, true);
    }
    
    private function generateThumbnail($video_path, $media_id) {
        $thumbnail_path = $this->config['paths']['uploads'] . 'thumbnails/thumb_' . $media_id . '.jpg';
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $video_path) . 
               " -ss 00:00:01 -vframes 1 -y " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $thumbnail_path);
        
        shell_exec($cmd);
        
        // Mise à jour du chemin thumbnail en DB
        $stmt = $this->db->prepare("UPDATE clipmaker_media SET thumbnail_path = ? WHERE id = ?");
        $stmt->execute([$thumbnail_path, $media_id]);
    }
    
    public function saveTimeline($project_id, $timeline_data) {
        try {
            $stmt = $this->db->prepare("
                UPDATE clipmaker_projects 
                SET timeline_data = ?, updated_at = NOW() 
                WHERE id = ? AND user_id = ?
            ");
            
            $stmt->execute([
                json_encode($timeline_data),
                $project_id,
                $this->user_id
            ]);
            
            return ['success' => true];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * ÉTAPE 2: EXPORT VERS AUDIO POST-PRODUCTION
     */
    public function exportToAudioPostProd($project_id, $export_name) {
        try {
            // Récupération du projet
            $project = $this->getProject($project_id);
            if (!$project) {
                throw new Exception("Projet non trouvé");
            }
            
            // Création de l'export workflow
            $export_id = $this->createWorkflowExport($project_id, 'clipmaker', $export_name, [
                'format' => 'mp4',
                'quality_preset' => 'broadcast',
                'resolution_width' => 1920,
                'resolution_height' => 1080,
                'fps' => 30
            ]);
            
            // Lancement du rendu en arrière-plan
            $this->startBackgroundRender($export_id, $project['timeline_data']);
            
            return [
                'success' => true,
                'export_id' => $export_id,
                'message' => 'Export vers Audio Post-Production démarré'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    private function createWorkflowExport($project_id, $stage, $export_name, $settings) {
        $stmt = $this->db->prepare("
            INSERT INTO workflow_exports 
            (user_id, project_id, stage, export_name, original_filename, file_path, 
             format, resolution_width, resolution_height, fps, quality_preset, 
             status, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'processing', NOW())
        ");
        
        $filename = $export_name . '_' . $stage . '_' . date('Y-m-d_H-i-s') . '.' . $settings['format'];
        $file_path = $this->config['paths']['workflow'] . $stage . '/' . $filename;
        
        $stmt->execute([
            $this->user_id,
            $project_id,
            $stage,
            $export_name,
            $filename,
            $file_path,
            $settings['format'],
            $settings['resolution_width'],
            $settings['resolution_height'],
            $settings['fps'],
            $settings['quality_preset']
        ]);
        
        return $this->db->lastInsertId();
    }
    
    private function startBackgroundRender($export_id, $timeline_data) {
        // Mise à jour du statut
        $stmt = $this->db->prepare("
            UPDATE workflow_exports 
            SET status = 'processing', processing_started_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$export_id]);
        
        // Lancement du script de rendu en arrière-plan
        $cmd = "php " . __DIR__ . "/scripts/render_video.php $export_id > /dev/null 2>&1 &";
        shell_exec($cmd);
    }
    
    /**
     * ÉTAPE 3: TRAITEMENT AUDIO POST-PRODUCTION
     */
    public function processAudioKB11($export_id) {
        try {
            $export = $this->getWorkflowExport($export_id);
            if (!$export || $export['stage'] !== 'clipmaker') {
                throw new Exception("Export non valide pour traitement audio");
            }
            
            // Extraction piste audio
            $audio_path = $this->extractAudioTrack($export['file_path']);
            
            // Application du profil KB11 (personnalisé selon vos besoins)
            $processed_audio = $this->applyKB11Profile($audio_path);
            
            // Création nouvel export audio post-prod
            $new_export_id = $this->createWorkflowExport(
                $export['project_id'], 
                'audio_postprod', 
                $export['export_name'] . '_audio_processed',
                [
                    'format' => 'mp4',
                    'quality_preset' => 'broadcast',
                    'resolution_width' => $export['resolution_width'],
                    'resolution_height' => $export['resolution_height'],
                    'fps' => $export['fps']
                ]
            );
            
            // Remixage vidéo + audio traité
            $this->remixVideoAudio($export['file_path'], $processed_audio, $new_export_id);
            
            return [
                'success' => true,
                'export_id' => $new_export_id,
                'message' => 'Traitement audio KB11 terminé'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    private function extractAudioTrack($video_path) {
        $audio_path = str_replace('.mp4', '_audio.wav', $video_path);
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $video_path) . 
               " -vn -acodec pcm_s16le -ar 48000 -ac 2 -y " . 
               escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $audio_path);
        
        shell_exec($cmd);
        return $audio_path;
    }
    
    private function applyKB11Profile($audio_path) {
        $processed_path = str_replace('.wav', '_kb11.wav', $audio_path);
        
        // Profil KB11 personnalisé - ajustez selon vos besoins
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $audio_path) . 
               " -af \"" .
               "highpass=f=80," .           // Filtre passe-haut 80Hz
               "lowpass=f=15000," .         // Filtre passe-bas 15kHz  
               "compand=0.3,1:6:-70,-60,-20," . // Compression dynamique
               "volume=1.2," .              // Gain +1.2dB
               "agate=threshold=0.003:ratio=2:attack=3:release=100" . // Gate de bruit
               "\" -y " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $processed_path);
        
        shell_exec($cmd);
        return $processed_path;
    }
    
    private function remixVideoAudio($video_path, $audio_path, $export_id) {
        $export = $this->getWorkflowExport($export_id);
        $output_path = $_SERVER['DOCUMENT_ROOT'] . $export['file_path'];
        
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $video_path) . 
               " -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $audio_path) . 
               " -c:v copy -c:a aac -b:a 192k -map 0:v:0 -map 1:a:0 -y " . 
               escapeshellarg($output_path);
        
        $result = shell_exec($cmd . " 2>&1");
        
        // Mise à jour statut export
        $status = file_exists($output_path) ? 'completed' : 'failed';
        $stmt = $this->db->prepare("
            UPDATE workflow_exports 
            SET status = ?, completed_at = NOW(), processing_log = ? 
            WHERE id = ?
        ");
        $stmt->execute([$status, $result, $export_id]);
    }
    
    /**
     * ÉTAPE 4: MIX FINAL + MUSIQUE
     */
    public function addMusicAndMaster($export_id, $music_file = null) {
        try {
            $export = $this->getWorkflowExport($export_id);
            
            // Si musique fournie, l'ajouter
            if ($music_file) {
                $mixed_path = $this->mixWithMusic($export['file_path'], $music_file);
            } else {
                $mixed_path = $export['file_path'];
            }
            
            // Mastering final
            $mastered_path = $this->applyMastering($mixed_path);
            
            // Créer export "emission_final"
            $final_export_id = $this->createWorkflowExport(
                $export['project_id'],
                'emission_final',
                $export['export_name'] . '_final',
                [
                    'format' => 'mp4',
                    'quality_preset' => 'broadcast',
                    'resolution_width' => $export['resolution_width'],
                    'resolution_height' => $export['resolution_height'],
                    'fps' => $export['fps']
                ]
            );
            
            // Copier le fichier masterisé
            copy($_SERVER['DOCUMENT_ROOT'] . $mastered_path, 
                 $_SERVER['DOCUMENT_ROOT'] . $this->getWorkflowExport($final_export_id)['file_path']);
            
            // Marquer comme terminé
            $stmt = $this->db->prepare("
                UPDATE workflow_exports 
                SET status = 'completed', completed_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$final_export_id]);
            
            return [
                'success' => true,
                'export_id' => $final_export_id,
                'message' => 'Mix final et mastering terminés'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    private function mixWithMusic($video_path, $music_path) {
        $output_path = str_replace('.mp4', '_with_music.mp4', $video_path);
        
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $video_path) . 
               " -i " . escapeshellarg($music_path) . 
               " -filter_complex \"" .
               "[0:a][1:a]amix=inputs=2:duration=first:dropout_transition=2[a]" .
               "\" -map 0:v -map \"[a]\" -c:v copy -c:a aac -b:a 192k -y " .
               escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $output_path);
        
        shell_exec($cmd);
        return $output_path;
    }
    
    private function applyMastering($video_path) {
        $output_path = str_replace('.mp4', '_mastered.mp4', $video_path);
        
        // Mastering audio avec compression, EQ et limiteur
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $video_path) . 
               " -af \"" .
               "compand=0.1,0.3:6:-70/-70,-60/-20,-20/-10,-5/-5,0/-3," . // Compresseur
               "equalizer=f=100:width_type=o:width=2:g=-3," .           // EQ graves
               "equalizer=f=3000:width_type=o:width=2:g=2," .           // EQ médiums  
               "equalizer=f=10000:width_type=o:width=2:g=1," .          // EQ aigus
               "alimiter=level_in=1:level_out=0.95:limit=0.95" .        // Limiteur
               "\" -c:v copy -c:a aac -b:a 192k -y " .
               escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $output_path);
        
        shell_exec($cmd);
        return $output_path;
    }
    
    /**
     * ÉTAPE 5: MULTI-EXPORT RÉSEAUX SOCIAUX
     */
    public function exportAllSocialFormats($export_id) {
        try {
            $export = $this->getWorkflowExport($export_id);
            $results = [];
            
            foreach ($this->config['social_formats'] as $platform => $format) {
                $social_export_id = $this->exportSocialFormat($export_id, $platform, $format);
                $results[$platform] = $social_export_id;
            }
            
            return [
                'success' => true,
                'exports' => $results,
                'message' => 'Tous les formats sociaux sont en cours de génération'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    public function exportSocialFormat($source_export_id, $platform, $format) {
        $source_export = $this->getWorkflowExport($source_export_id);
        
        // Créer nouvel export social
        $social_export_id = $this->createWorkflowExport(
            $source_export['project_id'],
            'social_ready',
            $source_export['export_name'] . '_' . $platform,
            [
                'format' => 'mp4',
                'quality_preset' => 'web',
                'resolution_width' => (int)explode('x', $format['resolution'])[0],
                'resolution_height' => (int)explode('x', $format['resolution'])[1],
                'fps' => $format['fps']
            ]
        );
        
        // Lancer conversion
        $this->convertToSocialFormat($source_export['file_path'], $social_export_id, $format);
        
        return $social_export_id;
    }
    
    private function convertToSocialFormat($source_path, $export_id, $format) {
        $export = $this->getWorkflowExport($export_id);
        $output_path = $_SERVER['DOCUMENT_ROOT'] . $export['file_path'];
        
        $resolution = $format['resolution'];
        $fps = $format['fps'];
        $bitrate = $format['bitrate'];
        $audio_bitrate = $format['audio_bitrate'];
        
        $cmd = "ffmpeg -i " . escapeshellarg($_SERVER['DOCUMENT_ROOT'] . $source_path) . 
               " -vf \"scale=$resolution:force_original_aspect_ratio=decrease,pad=$resolution:(ow-iw)/2:(oh-ih)/2\" " .
               " -r $fps -c:v libx264 -b:v $bitrate -c:a aac -b:a $audio_bitrate " .
               " -movflags +faststart -y " . escapeshellarg($output_path);
        
        $result = shell_exec($cmd . " 2>&1");
        
        // Mise à jour statut
        $status = file_exists($output_path) ? 'completed' : 'failed';
        $stmt = $this->db->prepare("
            UPDATE workflow_exports 
            SET status = ?, completed_at = NOW(), processing_log = ? 
            WHERE id = ?
        ");
        $stmt->execute([$status, $result, $export_id]);
    }
    
    /**
     * MÉTHODES UTILITAIRES
     */
    private function getProject($project_id) {
        $stmt = $this->db->prepare("
            SELECT * FROM clipmaker_projects 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$project_id, $this->user_id]);
        return $stmt->fetch();
    }
    
    private function getWorkflowExport($export_id) {
        $stmt = $this->db->prepare("
            SELECT * FROM workflow_exports 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$export_id, $this->user_id]);
        return $stmt->fetch();
    }
    
    public function getProjectProgress($project_id) {
        $stmt = $this->db->prepare("
            SELECT stage, COUNT(*) as count, 
                   SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed
            FROM workflow_exports 
            WHERE project_id = ? AND user_id = ?
            GROUP BY stage
            ORDER BY 
                CASE stage 
                    WHEN 'clipmaker' THEN 1
                    WHEN 'audio_postprod' THEN 2  
                    WHEN 'emission_final' THEN 3
                    WHEN 'social_ready' THEN 4
                END
        ");
        $stmt->execute([$project_id, $this->user_id]);
        return $stmt->fetchAll();
    }
    
    public function getUserProjects() {
        $stmt = $this->db->prepare("
            SELECT p.*, 
                   (SELECT COUNT(*) FROM workflow_exports WHERE project_id = p.id) as total_exports,
                   (SELECT COUNT(*) FROM workflow_exports WHERE project_id = p.id AND status = 'completed') as completed_exports
            FROM clipmaker_projects p 
            WHERE p.user_id = ? 
            ORDER BY p.updated_at DESC
        ");
        $stmt->execute([$this->user_id]);
        return $stmt->fetchAll();
    }
}

/**
 * API REST ENDPOINT
 */
header('Content-Type: application/json');

// Vérification authentification
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Non authentifié']);
    exit;
}

try {
    $workflow = new ClipmakerWorkflow($config);
    
    $action = $_POST['action'] ?? $_GET['action'] ?? '';
    
    switch ($action) {
        case 'upload_media':
            if (!isset($_FILES['media'])) {
                throw new Exception('Aucun fichier fourni');
            }
            $result = $workflow->uploadMedia($_FILES['media'], $_POST['project_id'] ?? null);
            break;
            
        case 'save_timeline':
            $timeline_data = json_decode($_POST['timeline_data'], true);
            $result = $workflow->saveTimeline($_POST['project_id'], $timeline_data);
            break;
            
        case 'export_to_audio':
            $result = $workflow->exportToAudioPostProd($_POST['project_id'], $_POST['export_name']);
            break;
            
        case 'process_audio_kb11':
            $result = $workflow->processAudioKB11($_POST['export_id']);
            break;
            
        case 'add_music_master':
            $music_file = $_FILES['music']['tmp_name'] ?? null;
            $result = $workflow->addMusicAndMaster($_POST['export_id'], $music_file);
            break;
            
        case 'export_all_social':
            $result = $workflow->exportAllSocialFormats($_POST['export_id']);
            break;
            
        case 'export_social_format':
            $format = $config['social_formats'][$_POST['platform']];
            $result = $workflow->exportSocialFormat($_POST['export_id'], $_POST['platform'], $format);
            break;
            
        case 'get_project_progress':
            $result = ['success' => true, 'data' => $workflow->getProjectProgress($_GET['project_id'])];
            break;
            
        case 'get_user_projects':
            $result = ['success' => true, 'data' => $workflow->getUserProjects()];
            break;
            
        default:
            throw new Exception('Action non reconnue: ' . $action);
    }
    
    echo json_encode($result);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>