<?php
/**
 * ================================================================
 * 📁 UPLOAD DE MÉDIAS - CLIPMAKER VIDEO ENGINE
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * ✅ Upload de vidéos/images/audio
 * ✅ Validation des fichiers
 * ✅ Sauvegarde en base
 * ✅ Interface drag & drop
 * 
 * 💡 BY PAPY 77 - UPLOAD QUI MARCHE !
 * ================================================================
 */

require_once '../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// Traitement de l'upload
if ($_POST && isset($_FILES['media_file'])) {
    $response = handleMediaUpload($_FILES['media_file'], $user['id'], $pdo);
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}

function handleMediaUpload($file, $userId, $pdo) {
    try {
        // Vérifications de base
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'error' => 'Erreur lors de l\'upload'];
        }
        
        // Taille max : 500MB
        $maxSize = 500 * 1024 * 1024;
        if ($file['size'] > $maxSize) {
            return ['success' => false, 'error' => 'Fichier trop volumineux (max 500MB)'];
        }
        
        // Types autorisés
        $allowedTypes = [
            'video/mp4', 'video/webm', 'video/mov', 'video/avi', 'video/mkv',
            'image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/gif',
            'audio/mp3', 'audio/wav', 'audio/ogg', 'audio/m4a'
        ];
        
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            return ['success' => false, 'error' => 'Type de fichier non supporté'];
        }
        
        // Déterminer le type de fichier
        $fileType = 'document';
        if (strpos($mimeType, 'video/') === 0) $fileType = 'video';
        elseif (strpos($mimeType, 'image/') === 0) $fileType = 'image';
        elseif (strpos($mimeType, 'audio/') === 0) $fileType = 'audio';
        
        // Créer le dossier s'il n'existe pas
        $uploadDir = '/var/www/laloee-cours.fr/social-media-generator/uploads/clipmaker/media/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // Nom de fichier sécurisé
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $storedFilename = 'clipmaker_' . uniqid() . '_' . time() . '.' . $extension;
        $filePath = $uploadDir . $storedFilename;
        $webPath = '/social-media-generator/uploads/clipmaker/media/' . $storedFilename;
        
        // Déplacer le fichier
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            return ['success' => false, 'error' => 'Erreur lors de la sauvegarde'];
        }
        
        // Obtenir les métadonnées
        $metadata = getFileMetadata($filePath, $fileType);
        
        // Générer une miniature pour les vidéos/images
        $thumbnailPath = generateThumbnail($filePath, $fileType, $uploadDir);
        
        // Sauvegarder en base de données
        $stmt = $pdo->prepare("
            INSERT INTO clipmaker_media 
            (user_id, original_filename, stored_filename, file_path, file_type, mime_type, 
             file_size, width, height, duration, thumbnail_path, metadata, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $stmt->execute([
            $userId,
            $file['name'],
            $storedFilename,
            $webPath,
            $fileType,
            $mimeType,
            $file['size'],
            $metadata['width'] ?? null,
            $metadata['height'] ?? null,
            $metadata['duration'] ?? null,
            $thumbnailPath,
            json_encode($metadata)
        ]);
        
        $mediaId = $pdo->lastInsertId();
        
        return [
            'success' => true,
            'media_id' => $mediaId,
            'file_path' => $webPath,
            'file_type' => $fileType,
            'thumbnail' => $thumbnailPath,
            'metadata' => $metadata
        ];
        
    } catch (Exception $e) {
        error_log("Erreur upload média: " . $e->getMessage());
        return ['success' => false, 'error' => 'Erreur serveur: ' . $e->getMessage()];
    }
}

function getFileMetadata($filePath, $fileType) {
    $metadata = [];
    
    try {
        switch ($fileType) {
            case 'image':
                $imageInfo = getimagesize($filePath);
                if ($imageInfo) {
                    $metadata['width'] = $imageInfo[0];
                    $metadata['height'] = $imageInfo[1];
                    $metadata['mime_type'] = $imageInfo['mime'];
                }
                break;
                
            case 'video':
                // Utiliser ffprobe si disponible
                if (command_exists('ffprobe')) {
                    $cmd = "ffprobe -v quiet -print_format json -show_format -show_streams " . escapeshellarg($filePath);
                    $output = shell_exec($cmd);
                    $data = json_decode($output, true);
                    
                    if ($data && isset($data['streams'])) {
                        foreach ($data['streams'] as $stream) {
                            if ($stream['codec_type'] === 'video') {
                                $metadata['width'] = $stream['width'] ?? null;
                                $metadata['height'] = $stream['height'] ?? null;
                                $metadata['duration'] = floatval($stream['duration'] ?? 0);
                                break;
                            }
                        }
                    }
                }
                break;
                
            case 'audio':
                // Estimation basique pour l'audio
                $metadata['estimated'] = true;
                $metadata['duration'] = filesize($filePath) / 128000; // Estimation très approximative
                break;
        }
    } catch (Exception $e) {
        error_log("Erreur métadonnées: " . $e->getMessage());
    }
    
    return $metadata;
}

function generateThumbnail($filePath, $fileType, $uploadDir) {
    try {
        $thumbnailDir = str_replace('/media/', '/thumbnails/', $uploadDir);
        if (!is_dir($thumbnailDir)) {
            mkdir($thumbnailDir, 0755, true);
        }
        
        $filename = pathinfo($filePath, PATHINFO_FILENAME);
        $thumbnailPath = $thumbnailDir . $filename . '_thumb.jpg';
        $webThumbnailPath = str_replace('/var/www/laloee-cours.fr', '', $thumbnailPath);
        
        switch ($fileType) {
            case 'image':
                // Redimensionner l'image
                $image = null;
                $imageInfo = getimagesize($filePath);
                
                switch ($imageInfo['mime']) {
                    case 'image/jpeg':
                        $image = imagecreatefromjpeg($filePath);
                        break;
                    case 'image/png':
                        $image = imagecreatefrompng($filePath);
                        break;
                    case 'image/webp':
                        $image = imagecreatefromwebp($filePath);
                        break;
                }
                
                if ($image) {
                    $originalWidth = imagesx($image);
                    $originalHeight = imagesy($image);
                    
                    $thumbWidth = 200;
                    $thumbHeight = ($originalHeight / $originalWidth) * $thumbWidth;
                    
                    $thumbnail = imagecreatetruecolor($thumbWidth, $thumbHeight);
                    imagecopyresampled($thumbnail, $image, 0, 0, 0, 0, $thumbWidth, $thumbHeight, $originalWidth, $originalHeight);
                    
                    imagejpeg($thumbnail, $thumbnailPath, 80);
                    
                    imagedestroy($image);
                    imagedestroy($thumbnail);
                    
                    return $webThumbnailPath;
                }
                break;
                
            case 'video':
                // Extraire une frame avec ffmpeg si disponible
                if (command_exists('ffmpeg')) {
                    $cmd = "ffmpeg -i " . escapeshellarg($filePath) . " -vf thumbnail -frames:v 1 " . escapeshellarg($thumbnailPath) . " 2>/dev/null";
                    exec($cmd);
                    
                    if (file_exists($thumbnailPath)) {
                        return $webThumbnailPath;
                    }
                }
                break;
        }
    } catch (Exception $e) {
        error_log("Erreur génération miniature: " . $e->getMessage());
    }
    
    return null;
}

function command_exists($command) {
    $whereIsCommand = (PHP_OS == 'WINNT') ? 'where' : 'which';
    $process = proc_open(
        "$whereIsCommand $command",
        array(
            0 => array("pipe", "r"), //STDIN
            1 => array("pipe", "w"), //STDOUT
            2 => array("pipe", "w"), //STDERR
        ),
        $pipes
    );
    if ($process !== false) {
        $stdout = stream_get_contents($pipes[1]);
        $stderr = stream_get_contents($pipes[2]);
        fclose($pipes[1]);
        fclose($pipes[2]);
        fclose($pipes[0]);
        $retval = proc_close($process);
        return $retval === 0;
    }
    return false;
}

?>

<!-- Interface d'upload à intégrer dans clipmaker-video-engine.php -->
<script>
// ================================================================
// 📁 SYSTÈME D'UPLOAD FONCTIONNEL
// ================================================================

function initializeUpload() {
    const uploadZone = document.querySelector('.upload-zone');
    if (!uploadZone) return;
    
    // Remplacer le contenu de la zone d'upload
    uploadZone.innerHTML = `
        <div id="uploadContent">
            <div style="font-size: 32px; margin-bottom: 10px;">📁</div>
            <div>Glissez vos médias ici</div>
            <div style="font-size: 12px; color: #a0aec0; margin-top: 5px;">
                ou cliquez pour parcourir
            </div>
            <input type="file" id="fileInput" style="display: none;" 
                   accept="video/*,image/*,audio/*" multiple>
        </div>
        <div id="uploadProgress" style="display: none;">
            <div style="margin-bottom: 10px;">📤 Upload en cours...</div>
            <div style="background: rgba(255,255,255,0.1); height: 6px; border-radius: 3px;">
                <div id="progressBar" style="background: #667eea; height: 100%; width: 0%; border-radius: 3px; transition: width 0.3s;"></div>
            </div>
            <div id="progressText" style="font-size: 12px; margin-top: 5px;">0%</div>
        </div>
    `;
    
    const fileInput = document.getElementById('fileInput');
    
    // Clic pour ouvrir le sélecteur
    uploadZone.addEventListener('click', () => {
        fileInput.click();
    });
    
    // Drag & Drop
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#667eea';
        uploadZone.style.background = 'rgba(102, 126, 234, 0.1)';
    });
    
    uploadZone.addEventListener('dragleave', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#4a5568';
        uploadZone.style.background = 'rgba(255, 255, 255, 0.02)';
    });
    
    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.style.borderColor = '#4a5568';
        uploadZone.style.background = 'rgba(255, 255, 255, 0.02)';
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            handleFileUpload(files[0]);
        }
    });
    
    // Sélection de fichier
    fileInput.addEventListener('change', (e) => {
        if (e.target.files.length > 0) {
            handleFileUpload(e.target.files[0]);
        }
    });
}

function handleFileUpload(file) {
    // Vérifications côté client
    const maxSize = 500 * 1024 * 1024; // 500MB
    if (file.size > maxSize) {
        showNotification('❌ Fichier trop volumineux (max 500MB)', 'error');
        return;
    }
    
    const allowedTypes = [
        'video/mp4', 'video/webm', 'video/mov', 'video/avi', 'video/mkv',
        'image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/gif',
        'audio/mp3', 'audio/wav', 'audio/ogg', 'audio/m4a'
    ];
    
    if (!allowedTypes.includes(file.type)) {
        showNotification('❌ Type de fichier non supporté', 'error');
        return;
    }
    
    // Préparer l'upload
    const formData = new FormData();
    formData.append('media_file', file);
    
    // Afficher la progression
    showUploadProgress();
    
    // Upload via AJAX
    const xhr = new XMLHttpRequest();
    
    xhr.upload.addEventListener('progress', (e) => {
        if (e.lengthComputable) {
            const percentComplete = (e.loaded / e.total) * 100;
            updateUploadProgress(percentComplete);
        }
    });
    
    xhr.addEventListener('load', () => {
        hideUploadProgress();
        
        if (xhr.status === 200) {
            try {
                const response = JSON.parse(xhr.responseText);
                if (response.success) {
                    showNotification('✅ Fichier uploadé avec succès !', 'success');
                    
                    // Recharger la liste des médias
                    setTimeout(() => {
                        window.location.reload();
                    }, 1000);
                } else {
                    showNotification('❌ ' + response.error, 'error');
                }
            } catch (e) {
                showNotification('❌ Erreur lors du traitement', 'error');
            }
        } else {
            showNotification('❌ Erreur serveur', 'error');
        }
    });
    
    xhr.addEventListener('error', () => {
        hideUploadProgress();
        showNotification('❌ Erreur lors de l\'upload', 'error');
    });
    
    xhr.open('POST', window.location.href);
    xhr.send(formData);
}

function showUploadProgress() {
    document.getElementById('uploadContent').style.display = 'none';
    document.getElementById('uploadProgress').style.display = 'block';
}

function hideUploadProgress() {
    document.getElementById('uploadContent').style.display = 'block';
    document.getElementById('uploadProgress').style.display = 'none';
}

function updateUploadProgress(percent) {
    document.getElementById('progressBar').style.width = percent + '%';
    document.getElementById('progressText').textContent = Math.round(percent) + '%';
}

// Initialiser à la fin du chargement
document.addEventListener('DOMContentLoaded', function() {
    // Attendre un peu que le DOM soit complètement prêt
    setTimeout(() => {
        initializeUpload();
        console.log('📁 Système d\'upload initialisé !');
    }, 500);
});
</script>