<?php
/*
 * SMG Studio v2.0 - Workflow Manager API Extension
 * Gestion DESTRUCTIVE des étapes - ONE CLIP par utilisateur
 * Path: /var/www/laloee-cours.fr/social-media-generator/api/workflow_manager_api.php
 */

require_once 'workflow.php';

class SMGWorkflowDestructive extends SMGWorkflowAPI {
    private $stagesPaths;
    private $currentClipPath;
    
    public function __construct() {
        parent::__construct();
        
        // Structure dossiers par étapes
        $this->stagesPaths = [
            1 => 'active_clip/01_clipmaker/',
            2 => 'active_clip/02_audio_postprod/',
            3 => 'active_clip/03_emission_final/',
            4 => 'active_clip/04_social_ready/'
        ];
        
        $this->currentClipPath = $this->workspace . 'active_clip/';
        $this->initializeClipStructure();
    }
    
    private function initializeClipStructure() {
        // Créer structure active_clip si nécessaire
        if (!is_dir($this->currentClipPath)) {
            mkdir($this->currentClipPath, 0755, true);
            
            foreach ($this->stagesPaths as $stageDir) {
                mkdir($this->workspace . $stageDir, 0755, true);
            }
        }
    }
    
    public function handleWorkflowRequest() {
        $action = $_GET['action'] ?? $_POST['action'] ?? '';
        
        switch ($action) {
            case 'get_workflow_state':
                $this->getWorkflowState();
                break;
                
            case 'start_new_clip':
                $this->startNewClip();
                break;
                
            case 'validate_stage':
                $this->validateStage();
                break;
                
            case 'rollback_stage':
                $this->rollbackStage();
                break;
                
            case 'auto_detect_files':
                $this->autoDetectFiles();
                break;
                
            case 'jump_to_production':
                $this->jumpToProduction();
                break;
                
            case 'archive_clip':
                $this->archiveClip();
                break;
                
            case 'get_stage_files':
                $this->getStageFiles();
                break;
                
            default:
                // Déléguer aux actions parent
                parent::handleRequest();
        }
    }
    
    private function getWorkflowState() {
        try {
            $state = [
                'hasActiveClip' => $this->hasActiveClip(),
                'currentStage' => $this->getCurrentStage(),
                'clipInfo' => $this->getClipInfo(),
                'stagesStatus' => $this->getStagesStatus(),
                'canProceed' => $this->canProceedToNextStage(),
                'blockedReason' => $this->getBlockedReason()
            ];
            
            $this->sendSuccess(['workflow' => $state]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur état workflow: ' . $e->getMessage(), 500);
        }
    }
    
    private function hasActiveClip() {
        // Vérifier si un projet est en cours
        $projectFile = $this->workspace . 'current_project.json';
        return file_exists($projectFile);
    }
    
    private function getCurrentStage() {
        if (!$this->hasActiveClip()) {
            return 0; // Pas de clip actif
        }
        
        // Déterminer étape actuelle basée sur fichiers présents
        for ($stage = 4; $stage >= 1; $stage--) {
            if ($this->hasFilesInStage($stage)) {
                return $stage;
            }
        }
        
        return 1; // Par défaut, commencer au clipmaker
    }
    
    private function getClipInfo() {
        if (!$this->hasActiveClip()) {
            return null;
        }
        
        $projectFile = $this->workspace . 'current_project.json';
        $project = json_decode(file_get_contents($projectFile), true);
        
        return [
            'name' => $project['name'] ?? 'Clip Sans Nom',
            'startedAt' => $project['startedAt'] ?? date('Y-m-d H:i:s'),
            'duration' => $this->calculateClipDuration(),
            'totalSize' => $this->calculateTotalSize(),
            'lastActivity' => $project['lastActivity'] ?? date('Y-m-d H:i:s')
        ];
    }
    
    private function getStagesStatus() {
        $stages = [];
        
        for ($i = 1; $i <= 4; $i++) {
            $stages[$i] = [
                'hasFiles' => $this->hasFilesInStage($i),
                'filesCount' => $this->getFilesCountInStage($i),
                'totalSize' => $this->getStageTotalSize($i),
                'lastModified' => $this->getStageLastModified($i),
                'isBlocked' => $this->isStageBlocked($i),
                'canProcess' => $this->canProcessStage($i)
            ];
        }
        
        return $stages;
    }
    
    private function hasFilesInStage($stage) {
        $stagePath = $this->workspace . $this->stagesPaths[$stage];
        if (!is_dir($stagePath)) return false;
        
        $files = glob($stagePath . '*.*');
        return count($files) > 0;
    }
    
    private function getFilesCountInStage($stage) {
        $stagePath = $this->workspace . $this->stagesPaths[$stage];
        if (!is_dir($stagePath)) return 0;
        
        $files = glob($stagePath . '*.*');
        return count($files);
    }
    
    private function getStageTotalSize($stage) {
        $stagePath = $this->workspace . $this->stagesPaths[$stage];
        if (!is_dir($stagePath)) return 0;
        
        $files = glob($stagePath . '*.*');
        $totalSize = 0;
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $totalSize += filesize($file);
            }
        }
        
        return $totalSize;
    }
    
    private function getStageLastModified($stage) {
        $stagePath = $this->workspace . $this->stagesPaths[$stage];
        if (!is_dir($stagePath)) return null;
        
        $files = glob($stagePath . '*.*');
        $lastModified = 0;
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $lastModified = max($lastModified, filemtime($file));
            }
        }
        
        return $lastModified > 0 ? date('Y-m-d H:i:s', $lastModified) : null;
    }
    
    private function isStageBlocked($stage) {
        if ($stage == 1) return false; // Clipmaker jamais bloqué
        
        // Étape bloquée si la précédente n'a pas de fichiers
        return !$this->hasFilesInStage($stage - 1);
    }
    
    private function canProcessStage($stage) {
        return !$this->isStageBlocked($stage) && $this->hasFilesInStage($stage);
    }
    
    private function canProceedToNextStage() {
        $currentStage = $this->getCurrentStage();
        if ($currentStage >= 4) return false;
        
        return $this->hasFilesInStage($currentStage);
    }
    
    private function getBlockedReason() {
        $currentStage = $this->getCurrentStage();
        
        if ($currentStage == 0) {
            return "Aucun clip en cours - Commencez par uploader un rush";
        }
        
        if (!$this->hasFilesInStage($currentStage)) {
            return "Étape $currentStage vide - Ajoutez des fichiers pour continuer";
        }
        
        return null;
    }
    
    private function startNewClip() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            $clipName = $input['clipName'] ?? 'Mon Nouveau Clip';
            
            // DESTRUCTIF : Nettoyer tout clip existant
            $this->cleanActiveClip();
            
            // Créer nouveau projet
            $project = [
                'name' => SMGSecurityManager::sanitizeInput($clipName),
                'startedAt' => date('Y-m-d H:i:s'),
                'currentStage' => 1,
                'lastActivity' => date('Y-m-d H:i:s'),
                'version' => '2.0'
            ];
            
            file_put_contents($this->workspace . 'current_project.json', json_encode($project, JSON_PRETTY_PRINT));
            
            // Log création
            $this->logWorkflowAction('clip_started', $clipName);
            
            $this->sendSuccess([
                'clipCreated' => true,
                'clipName' => $clipName,
                'redirectTo' => 'modules/clipmaker_workflow_interface.html'
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur création clip: ' . $e->getMessage(), 500);
        }
    }
    
    private function validateStage() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            $stage = intval($input['stage'] ?? 0);
            
            if ($stage < 1 || $stage > 4) {
                $this->sendError('Étape invalide', 400);
                return;
            }
            
            if (!$this->hasFilesInStage($stage)) {
                $this->sendError("Aucun fichier dans l'étape $stage", 400);
                return;
            }
            
            // Valider et passer à l'étape suivante
            $nextStage = $stage + 1;
            
            if ($nextStage <= 4) {
                // Préparer étape suivante
                $this->prepareNextStage($nextStage);
                $message = "Étape $stage validée - Passage à l'étape $nextStage";
            } else {
                // Fin du workflow
                $message = "Clip terminé - Prêt pour archivage";
            }
            
            // Log validation
            $this->logWorkflowAction("stage_{$stage}_validated", "Progression vers étape $nextStage");
            
            $this->sendSuccess([
                'stageValidated' => $stage,
                'nextStage' => $nextStage <= 4 ? $nextStage : null,
                'message' => $message,
                'workflowComplete' => $nextStage > 4
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur validation: ' . $e->getMessage(), 500);
        }
    }
    
    private function rollbackStage() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            $fromStage = intval($input['fromStage'] ?? 0);
            $toStage = intval($input['toStage'] ?? 0);
            
            if ($toStage >= $fromStage) {
                $this->sendError('Rollback invalide', 400);
                return;
            }
            
            // DESTRUCTIF : Supprimer toutes les étapes > toStage
            $deletedStages = [];
            for ($stage = $fromStage; $stage > $toStage; $stage--) {
                if ($this->clearStage($stage)) {
                    $deletedStages[] = $stage;
                }
            }
            
            // Log rollback
            $this->logWorkflowAction("rollback_to_stage_$toStage", 
                "Supprimé étapes: " . implode(', ', $deletedStages));
            
            $this->sendSuccess([
                'rolledBack' => true,
                'currentStage' => $toStage,
                'deletedStages' => $deletedStages,
                'message' => "Retour à l'étape $toStage - Modifications suivantes perdues"
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur rollback: ' . $e->getMessage(), 500);
        }
    }
    
    private function clearStage($stage) {
        $stagePath = $this->workspace . $this->stagesPaths[$stage];
        if (!is_dir($stagePath)) return false;
        
        $files = glob($stagePath . '*.*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
        
        return true;
    }
    
    private function prepareNextStage($nextStage) {
        // Créer dossier étape suivante si nécessaire
        $nextStagePath = $this->workspace . $this->stagesPaths[$nextStage];
        if (!is_dir($nextStagePath)) {
            mkdir($nextStagePath, 0755, true);
        }
        
        // Logique spécifique par étape
        switch ($nextStage) {
            case 2: // Audio PostProd
                // Copier vidéo de clipmaker vers audio
                $this->copyStageFiles(1, 2, ['mp4', 'mov', 'avi']);
                break;
                
            case 3: // Émission Finale
                // Copier audio final vers émission
                $this->copyStageFiles(2, 3, ['mp4', 'wav', 'mp3']);
                break;
                
            case 4: // Social Ready
                // Copier émission finale vers social
                $this->copyStageFiles(3, 4, ['mp4', 'mov']);
                break;
        }
    }
    
    private function copyStageFiles($fromStage, $toStage, $extensions = []) {
        $fromPath = $this->workspace . $this->stagesPaths[$fromStage];
        $toPath = $this->workspace . $this->stagesPaths[$toStage];
        
        if (!is_dir($fromPath) || !is_dir($toPath)) return;
        
        $files = glob($fromPath . '*.*');
        foreach ($files as $file) {
            if (is_file($file)) {
                $fileExt = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                if (empty($extensions) || in_array($fileExt, $extensions)) {
                    $filename = basename($file);
                    copy($file, $toPath . $filename);
                }
            }
        }
    }
    
    private function autoDetectFiles() {
        try {
            $detected = [];
            
            // Scanner tous les dossiers possibles pour détecter fichiers
            $scanDirs = [
                'uploads/clipmaker/exports/',
                'uploads/audio/',
                'uploads/final/',
                'user_workspaces/' . $this->userId . '/temp/'
            ];
            
            foreach ($scanDirs as $dir) {
                $fullPath = '/var/www/laloee-cours.fr/social-media-generator/' . $dir;
                if (is_dir($fullPath)) {
                    $files = glob($fullPath . '*.*');
                    foreach ($files as $file) {
                        if (is_file($file)) {
                            $detected[] = [
                                'file' => basename($file),
                                'path' => $file,
                                'size' => filesize($file),
                                'modified' => date('Y-m-d H:i:s', filemtime($file)),
                                'suggestedStage' => $this->suggestStageForFile($file)
                            ];
                        }
                    }
                }
            }
            
            $this->sendSuccess([
                'filesDetected' => count($detected),
                'files' => $detected,
                'message' => count($detected) > 0 ? 
                    "Détecté " . count($detected) . " fichier(s) - Assignation automatique possible" :
                    "Aucun fichier détecté dans les dossiers de scan"
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur auto-détection: ' . $e->getMessage(), 500);
        }
    }
    
    private function suggestStageForFile($filePath) {
        $filename = strtolower(basename($filePath));
        
        // Logique de suggestion basée sur le nom de fichier
        if (strpos($filename, 'clip') !== false || strpos($filename, 'raw') !== false) {
            return 1; // Clipmaker
        }
        if (strpos($filename, 'audio') !== false || strpos($filename, 'mix') !== false) {
            return 2; // Audio PostProd
        }
        if (strpos($filename, 'final') !== false || strpos($filename, 'master') !== false) {
            return 3; // Émission Finale
        }
        if (strpos($filename, 'social') !== false || strpos($filename, 'export') !== false) {
            return 4; // Social Ready
        }
        
        return 1; // Par défaut, clipmaker
    }
    
    private function jumpToProduction() {
        try {
            // Simuler completion de toutes les étapes intermédiaires
            $this->logWorkflowAction('jump_to_production', 'Saut direct vers Social Ready');
            
            // Créer fichiers simulés pour déblocage
            for ($stage = 1; $stage <= 3; $stage++) {
                $stagePath = $this->workspace . $this->stagesPaths[$stage];
                if (!is_dir($stagePath)) {
                    mkdir($stagePath, 0755, true);
                }
                
                // Créer fichier marqueur
                file_put_contents($stagePath . 'auto_completed.txt', 
                    "Étape complétée automatiquement le " . date('Y-m-d H:i:s'));
            }
            
            // Activer étape 4
            $stage4Path = $this->workspace . $this->stagesPaths[4];
            if (!is_dir($stage4Path)) {
                mkdir($stage4Path, 0755, true);
            }
            
            $this->sendSuccess([
                'jumpCompleted' => true,
                'currentStage' => 4,
                'message' => 'Saut vers production réussi - Social Ready activé'
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur saut production: ' . $e->getMessage(), 500);
        }
    }
    
    private function archiveClip() {
        try {
            if (!$this->hasActiveClip()) {
                $this->sendError('Aucun clip à archiver', 400);
                return;
            }
            
            $clipInfo = $this->getClipInfo();
            $archiveName = $this->userId . '_' . date('Y-m-d_H-i-s') . '_' . 
                           preg_replace('/[^a-zA-Z0-9]/', '_', $clipInfo['name']);
            
            // Créer archive
            $archivePath = $this->workspace . 'archives/' . $archiveName . '/';
            if (!is_dir($archivePath)) {
                mkdir($archivePath, 0755, true);
            }
            
            // Copier tous les fichiers des étapes
            for ($stage = 1; $stage <= 4; $stage++) {
                $stagePath = $this->workspace . $this->stagesPaths[$stage];
                $archiveStageDir = $archivePath . 'stage_' . $stage . '/';
                
                if (is_dir($stagePath)) {
                    mkdir($archiveStageDir, 0755, true);
                    $this->copyDirectory($stagePath, $archiveStageDir);
                }
            }
            
            // Sauvegarder métadonnées
            $metadata = [
                'clipName' => $clipInfo['name'],
                'archiveDate' => date('Y-m-d H:i:s'),
                'originalDuration' => $clipInfo['duration'],
                'totalSize' => $clipInfo['totalSize'],
                'user' => $this->userId,
                'stages' => $this->getStagesStatus()
            ];
            
            file_put_contents($archivePath . 'metadata.json', json_encode($metadata, JSON_PRETTY_PRINT));
            
            // DESTRUCTIF : Nettoyer clip actif
            $this->cleanActiveClip();
            
            // Supprimer projet actuel
            $projectFile = $this->workspace . 'current_project.json';
            if (file_exists($projectFile)) {
                unlink($projectFile);
            }
            
            // Log archivage
            $this->logWorkflowAction('clip_archived', $archiveName);
            
            $this->sendSuccess([
                'archived' => true,
                'archiveName' => $archiveName,
                'archivePath' => $archivePath,
                'message' => 'Clip archivé avec succès - Workflow réinitialisé',
                'redirectTo' => 'posts.php'
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur archivage: ' . $e->getMessage(), 500);
        }
    }
    
    private function cleanActiveClip() {
        // Supprimer récursivement le dossier active_clip
        if (is_dir($this->currentClipPath)) {
            $this->deleteDirectory($this->currentClipPath);
        }
        
        // Recréer structure
        $this->initializeClipStructure();
    }
    
    private function deleteDirectory($dir) {
        if (!is_dir($dir)) return;
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $filePath = $dir . '/' . $file;
            if (is_dir($filePath)) {
                $this->deleteDirectory($filePath);
            } else {
                unlink($filePath);
            }
        }
        
        rmdir($dir);
    }
    
    private function copyDirectory($src, $dst) {
        if (!is_dir($src)) return;
        
        if (!is_dir($dst)) {
            mkdir($dst, 0755, true);
        }
        
        $files = array_diff(scandir($src), ['.', '..']);
        foreach ($files as $file) {
            $srcFile = $src . '/' . $file;
            $dstFile = $dst . '/' . $file;
            
            if (is_dir($srcFile)) {
                $this->copyDirectory($srcFile, $dstFile);
            } else {
                copy($srcFile, $dstFile);
            }
        }
    }
    
    private function getStageFiles() {
        try {
            $stage = intval($_GET['stage'] ?? 1);
            if ($stage < 1 || $stage > 4) {
                $this->sendError('Étape invalide', 400);
                return;
            }
            
            $stagePath = $this->workspace . $this->stagesPaths[$stage];
            $files = [];
            
            if (is_dir($stagePath)) {
                $fileList = glob($stagePath . '*.*');
                foreach ($fileList as $file) {
                    if (is_file($file)) {
                        $files[] = [
                            'name' => basename($file),
                            'size' => filesize($file),
                            'modified' => date('Y-m-d H:i:s', filemtime($file)),
                            'extension' => strtolower(pathinfo($file, PATHINFO_EXTENSION))
                        ];
                    }
                }
            }
            
            $this->sendSuccess([
                'stage' => $stage,
                'files' => $files,
                'totalFiles' => count($files),
                'totalSize' => array_sum(array_column($files, 'size'))
            ]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur récupération fichiers: ' . $e->getMessage(), 500);
        }
    }
    
    private function calculateClipDuration() {
        // Calculer durée basée sur fichiers vidéo présents
        // TODO: Intégrer avec FFprobe pour durée réelle
        return '0min';
    }
    
    private function calculateTotalSize() {
        $totalSize = 0;
        for ($stage = 1; $stage <= 4; $stage++) {
            $totalSize += $this->getStageTotalSize($stage);
        }
        return $totalSize;
    }
    
    private function logWorkflowAction($action, $details) {
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'user' => $this->userId,
            'action' => $action,
            'details' => $details,
            'stage' => 'workflow_management',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ];
        
        $logFile = $this->workspace . 'logs/' . date('Y-m-d') . '.log';
        file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
    }
}

// Point d'entrée spécialisé workflow
try {
    $workflowAPI = new SMGWorkflowDestructive();
    $workflowAPI->handleWorkflowRequest();
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erreur workflow serveur']);
    
    SMGSecurityManager::logSecurityEvent('workflow_error', null, $e->getMessage());
}
?>