<?php
/**
 * ================================================================
 * 🔍 DIAGNOSTIC TABLES SMG - POURQUOI LE DASHBOARD NE VOIT RIEN ?
 * ================================================================
 * 
 * 🎯 Vérifier si les tables existent vraiment
 * 🔧 Diagnostiquer le problème dashboard vs BDD
 * 💡 Solution rapide pour Papy 77
 * ================================================================
 */

require_once 'config/db.php';

echo "<h1>🔍 Diagnostic SMG Studio - Tables BDD</h1>";

// ================================================================
// 🗄️ VÉRIFICATION DIRECTE DES TABLES
// ================================================================

echo "<h2>📋 Étape 1 : Vérification directe des tables</h2>";

try {
    // Voir TOUTES les tables de la base
    $stmt = $pdo->query("SHOW TABLES");
    $all_tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    echo "<div style='background: #e7f3ff; padding: 15px; border-radius: 8px; margin: 10px 0;'>";
    echo "<h3>📊 Tables trouvées dans la base laloedata :</h3>";
    echo "<p><strong>Total :</strong> " . count($all_tables) . " tables</p>";
    echo "<ul>";
    foreach ($all_tables as $table) {
        echo "<li><strong>$table</strong></li>";
    }
    echo "</ul>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 8px;'>";
    echo "❌ Erreur SHOW TABLES : " . $e->getMessage();
    echo "</div>";
}

// ================================================================
// 🎯 TEST DES TABLES SMG SPÉCIFIQUES
// ================================================================

echo "<h2>🎯 Étape 2 : Test tables SMG spécifiques</h2>";

$smg_tables = [
    'users', 'posts', 'social_platforms', 'post_networks', 
    'post_medias', 'scheduled_posts', 'transmission_logs', 'user_downloads'
];

foreach ($smg_tables as $table) {
    echo "<div style='background: #f8f9fa; padding: 10px; margin: 5px 0; border-radius: 5px;'>";
    echo "<strong>Table : $table</strong><br>";
    
    try {
        // Test 1 : SHOW TABLES LIKE
        $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
        $stmt->execute([$table]);
        $exists_show = $stmt->rowCount() > 0;
        
        // Test 2 : SELECT COUNT
        $count = 0;
        if ($exists_show) {
            $stmt = $pdo->query("SELECT COUNT(*) FROM `$table`");
            $count = $stmt->fetchColumn();
        }
        
        // Test 3 : DESCRIBE
        $structure = '';
        if ($exists_show) {
            try {
                $stmt = $pdo->query("DESCRIBE `$table`");
                $columns = $stmt->fetchAll();
                $structure = count($columns) . " colonnes";
            } catch (Exception $e) {
                $structure = "Erreur DESCRIBE";
            }
        }
        
        if ($exists_show) {
            echo "✅ Existe | $count enregistrements | $structure";
        } else {
            echo "❌ N'existe pas";
        }
        
    } catch (Exception $e) {
        echo "❌ Erreur : " . $e->getMessage();
    }
    
    echo "</div>";
}

// ================================================================
// 🔧 TEST DE LA FONCTION table_exists DU DASHBOARD
// ================================================================

echo "<h2>🔧 Étape 3 : Test fonction table_exists du dashboard</h2>";

function table_exists_test($pdo, $table_name) {
    try {
        $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
        $stmt->execute([$table_name]);
        return $stmt->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}

echo "<div style='background: #fff3cd; padding: 15px; border-radius: 8px; margin: 10px 0;'>";
echo "<h3>🧪 Test de la fonction table_exists (comme dans dashboard.php) :</h3>";

foreach ($smg_tables as $table) {
    $exists = table_exists_test($pdo, $table);
    $status = $exists ? "✅ TRUE" : "❌ FALSE";
    echo "<p><strong>$table :</strong> $status</p>";
}
echo "</div>";

// ================================================================
// 🔍 DIAGNOSTIC AVANCÉ
// ================================================================

echo "<h2>🔍 Étape 4 : Diagnostic avancé</h2>";

try {
    // Vérifier la base de données active
    $stmt = $pdo->query("SELECT DATABASE() as current_db");
    $current_db = $stmt->fetchColumn();
    
    // Vérifier l'utilisateur connecté
    $stmt = $pdo->query("SELECT USER() as current_user");
    $current_user = $stmt->fetchColumn();
    
    // Vérifier les droits sur la base
    $stmt = $pdo->query("SHOW GRANTS");
    $grants = $stmt->fetchAll();
    
    echo "<div style='background: #d1ecf1; padding: 15px; border-radius: 8px; margin: 10px 0;'>";
    echo "<h3>📊 Informations de connexion :</h3>";
    echo "<p><strong>Base de données active :</strong> $current_db</p>";
    echo "<p><strong>Utilisateur connecté :</strong> $current_user</p>";
    echo "<p><strong>Droits utilisateur :</strong></p>";
    echo "<ul style='font-size: 12px;'>";
    foreach ($grants as $grant) {
        echo "<li>" . htmlspecialchars(array_values($grant)[0]) . "</li>";
    }
    echo "</ul>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; padding: 15px; border-radius: 8px;'>";
    echo "❌ Erreur diagnostic avancé : " . $e->getMessage();
    echo "</div>";
}

// ================================================================
// 🎯 SOLUTION RAPIDE SI TABLES MANQUANTES
// ================================================================

$missing_count = 0;
foreach ($smg_tables as $table) {
    if (!table_exists_test($pdo, $table)) {
        $missing_count++;
    }
}

if ($missing_count > 0) {
    echo "<div style='background: #f093fb; padding: 20px; border-radius: 8px; margin: 20px 0; color: white;'>";
    echo "<h3>🚨 $missing_count table(s) manquante(s) détectée(s) !</h3>";
    echo "<p>Le script de reset n'a pas créé toutes les tables ou elles ont été supprimées.</p>";
    echo "<p><strong>Solutions :</strong></p>";
    echo "<ol>";
    echo "<li><a href='?action=create_missing' style='color: white; font-weight: bold;'>🔧 Créer les tables manquantes automatiquement</a></li>";
    echo "<li>Relancer le script de reset complet</li>";
    echo "<li>Vérifier les droits de l'utilisateur laloedata</li>";
    echo "</ol>";
    echo "</div>";
}

// ================================================================
// 🔧 ACTION CRÉATION TABLES MANQUANTES
// ================================================================

if (isset($_GET['action']) && $_GET['action'] === 'create_missing') {
    echo "<h2>🔧 Création automatique des tables manquantes</h2>";
    
    // Créer seulement les tables qui manquent
    $creation_script = "
    SET FOREIGN_KEY_CHECKS = 0;
    
    CREATE TABLE IF NOT EXISTS `social_platforms` (
      `id` int NOT NULL AUTO_INCREMENT,
      `name` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
      `display_name` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
      `icon` varchar(10) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
      `is_active` tinyint(1) DEFAULT '1',
      `max_chars` int DEFAULT '0',
      `supports_images` tinyint(1) DEFAULT '1',
      `supports_videos` tinyint(1) DEFAULT '1',
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `name` (`name`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    INSERT IGNORE INTO `social_platforms` (`name`, `display_name`, `icon`, `max_chars`) VALUES
    ('instagram', 'Instagram', '📸', 2200),
    ('facebook', 'Facebook', '📘', 63206),
    ('twitter', 'Twitter/X', '🐦', 280),
    ('linkedin', 'LinkedIn', '💼', 3000);
    
    CREATE TABLE IF NOT EXISTS `posts` (
      `id` int NOT NULL AUTO_INCREMENT,
      `user_id` int NOT NULL,
      `titre` varchar(255) COLLATE utf8mb4_unicode_ci NOT NULL,
      `contenu` text COLLATE utf8mb4_unicode_ci,
      `hashtags` text COLLATE utf8mb4_unicode_ci,
      `statut` enum('draft','scheduled','published','archived') COLLATE utf8mb4_unicode_ci DEFAULT 'draft',
      `type` enum('text','image','video','carousel') COLLATE utf8mb4_unicode_ci DEFAULT 'text',
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      KEY `idx_posts_user_id` (`user_id`),
      CONSTRAINT `fk_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    CREATE TABLE IF NOT EXISTS `post_networks` (
      `id` int NOT NULL AUTO_INCREMENT,
      `post_id` int NOT NULL,
      `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
      `is_active` tinyint(1) DEFAULT '1',
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      UNIQUE KEY `unique_post_network` (`post_id`,`network`),
      CONSTRAINT `fk_post_networks_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    CREATE TABLE IF NOT EXISTS `post_medias` (
      `id` int NOT NULL AUTO_INCREMENT,
      `post_id` int NOT NULL,
      `media_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
      `media_type` enum('image','video') COLLATE utf8mb4_unicode_ci NOT NULL,
      `display_order` int DEFAULT '0',
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      CONSTRAINT `fk_post_medias_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    CREATE TABLE IF NOT EXISTS `scheduled_posts` (
      `id` int NOT NULL AUTO_INCREMENT,
      `user_id` int NOT NULL,
      `post_id` int NOT NULL,
      `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
      `scheduled_datetime` timestamp NOT NULL,
      `status` enum('pending','processing','sent','failed') COLLATE utf8mb4_unicode_ci DEFAULT 'pending',
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      CONSTRAINT `fk_scheduled_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
      CONSTRAINT `fk_scheduled_posts_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    CREATE TABLE IF NOT EXISTS `transmission_logs` (
      `id` int NOT NULL AUTO_INCREMENT,
      `scheduled_post_id` int NOT NULL,
      `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
      `status` enum('success','failed','retry') COLLATE utf8mb4_unicode_ci NOT NULL,
      `transmitted_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      CONSTRAINT `fk_transmission_logs_scheduled` FOREIGN KEY (`scheduled_post_id`) REFERENCES `scheduled_posts` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    CREATE TABLE IF NOT EXISTS `user_downloads` (
      `id` int NOT NULL AUTO_INCREMENT,
      `user_id` int NOT NULL,
      `post_id` int DEFAULT NULL,
      `download_type` enum('post','media','zip') COLLATE utf8mb4_unicode_ci DEFAULT 'zip',
      `file_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
      `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`id`),
      CONSTRAINT `fk_user_downloads_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    
    SET FOREIGN_KEY_CHECKS = 1;
    ";
    
    try {
        $pdo->exec($creation_script);
        echo "<div style='background: #d4edda; padding: 15px; border-radius: 8px;'>";
        echo "✅ Tables créées avec succès ! Rechargez le dashboard.";
        echo "</div>";
    } catch (Exception $e) {
        echo "<div style='background: #f8d7da; padding: 15px; border-radius: 8px;'>";
        echo "❌ Erreur création : " . $e->getMessage();
        echo "</div>";
    }
}

// ================================================================
// 🔄 LIENS DE NAVIGATION
// ================================================================

echo "<div style='background: #e9ecef; padding: 20px; border-radius: 8px; margin: 20px 0; text-align: center;'>";
echo "<h3>🔄 Actions Rapides</h3>";
echo "<a href='dashboard.php' style='display: inline-block; padding: 10px 20px; margin: 5px; background: #007bff; color: white; text-decoration: none; border-radius: 5px;'>📊 Tester Dashboard</a>";
echo "<a href='?refresh=1' style='display: inline-block; padding: 10px 20px; margin: 5px; background: #28a745; color: white; text-decoration: none; border-radius: 5px;'>🔄 Recharger Diagnostic</a>";
echo "<a href='?action=create_missing' style='display: inline-block; padding: 10px 20px; margin: 5px; background: #ffc107; color: black; text-decoration: none; border-radius: 5px;'>🔧 Créer Tables</a>";
echo "</div>";

echo "<p style='text-align: center; color: #666; margin-top: 30px;'>";
echo "🔍 Diagnostic SMG Studio - " . date('d/m/Y H:i:s') . " - by Papy 77";
echo "</p>";

?>