<?php
/**
 * ================================================================
 * 🔐 AUTH FUNCTIONS - SYSTÈME D'AUTHENTIFICATION SMG STUDIO
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * ✅ Connexion utilisateur sécurisée
 * ✅ Sessions et tokens
 * ✅ Vérification des permissions
 * ✅ Logs de connexion
 * ✅ Protection CSRF
 * ✅ Intégration base de données
 * 
 * 💡 BY PAPY 77 - SÉCURITÉ VPS UBUNTU 24
 * ================================================================
 */

// Configuration de session sécurisée
if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) ? 1 : 0);
    ini_set('session.cookie_samesite', 'Strict');
    ini_set('session.gc_maxlifetime', 86400); // 24 heures
    
    session_start();
}

/**
 * ================================================================
 * 🔗 CONNEXION BASE DE DONNÉES
 * ================================================================
 */
function getDbConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $host = 'localhost';
            $dbname = 'laloedata';
            $username = 'laloedata';
            $password = 'PiL256Mx78!';
            
            $pdo = new PDO(
                "mysql:host=$host;dbname=$dbname;charset=utf8mb4",
                $username,
                $password,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
                ]
            );
            
            // Test de connexion
            $pdo->query("SELECT 1");
            
        } catch (PDOException $e) {
            error_log("Erreur connexion BDD: " . $e->getMessage());
            
            // Message d'erreur plus explicite
            if (strpos($e->getMessage(), 'Access denied') !== false) {
                die("Erreur : Accès refusé à la base de données. Vérifiez les identifiants.");
            } elseif (strpos($e->getMessage(), 'Unknown database') !== false) {
                die("Erreur : Base de données '$dbname' non trouvée.");
            } else {
                die("Erreur de connexion à la base de données : " . $e->getMessage());
            }
        }
    }
    
    return $pdo;
}

/**
 * ================================================================
 * 🔐 AUTHENTIFICATION
 * ================================================================
 */

/**
 * Vérifier si l'utilisateur est connecté
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && 
           isset($_SESSION['user_email']) && 
           !empty($_SESSION['user_id']);
}

/**
 * Forcer la connexion (redirection si non connecté)
 */
function requireLogin() {
    if (!isLoggedIn()) {
        // Sauvegarder l'URL demandée
        $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
        
        // Déterminer le chemin vers login.php
        $login_path = '/social-media-generator/login.php';
        
        // Si on est déjà dans le dossier, ajuster le chemin
        if (strpos($_SERVER['REQUEST_URI'], '/social-media-generator/') !== false) {
            $login_path = 'login.php';
        }
        
        header("Location: $login_path");
        exit;
    }
    
    // Vérifier la validité de la session
    if (!validateSession()) {
        logout();
        header("Location: login.php?error=session_expired");
        exit;
    }
}

/**
 * Valider la session utilisateur
 */
function validateSession() {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    try {
        $pdo = getDbConnection();
        
        // Vérifier que l'utilisateur existe toujours
        $stmt = $pdo->prepare("
            SELECT id, email, status, last_login 
            FROM users 
            WHERE id = ? AND status = 'active'
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return false;
        }
        
        // Vérifier l'expiration de session (24h)
        if (isset($_SESSION['last_activity']) && 
            time() - $_SESSION['last_activity'] > 86400) {
            return false;
        }
        
        // Mettre à jour l'activité
        $_SESSION['last_activity'] = time();
        
        return true;
        
    } catch (Exception $e) {
        error_log("Erreur validation session: " . $e->getMessage());
        return false;
    }
}

/**
 * Connexion utilisateur
 */
function login($email, $password, $remember = false) {
    try {
        $pdo = getDbConnection();
        
        // Récupérer l'utilisateur
        $stmt = $pdo->prepare("
            SELECT id, name, email, password, role, status, last_login
            FROM users 
            WHERE email = ? AND status = 'active'
        ");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($password, $user['password'])) {
            logLoginAttempt($email, false, 'Invalid credentials');
            return false;
        }
        
        // Créer la session
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['last_activity'] = time();
        
        // Générer un token de session
        $session_token = bin2hex(random_bytes(32));
        $_SESSION['session_token'] = $session_token;
        
        // Mettre à jour la dernière connexion
        $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$user['id']]);
        
        // Log de connexion
        logLoginAttempt($email, true);
        logUserConnection($user['id'], 'login');
        
        // Remember me
        if ($remember) {
            $remember_token = bin2hex(random_bytes(32));
            $expires = time() + (30 * 24 * 60 * 60); // 30 jours
            
            setcookie('remember_token', $remember_token, $expires, '/', '', true, true);
            
            $stmt = $pdo->prepare("
                UPDATE users 
                SET remember_token = ?, remember_expires = FROM_UNIXTIME(?)
                WHERE id = ?
            ");
            $stmt->execute([$remember_token, $expires, $user['id']]);
        }
        
        return true;
        
    } catch (Exception $e) {
        error_log("Erreur login: " . $e->getMessage());
        return false;
    }
}

/**
 * Déconnexion
 */
function logout() {
    if (isset($_SESSION['user_id'])) {
        logUserConnection($_SESSION['user_id'], 'logout');
    }
    
    // Supprimer le cookie remember me
    if (isset($_COOKIE['remember_token'])) {
        setcookie('remember_token', '', time() - 3600, '/', '', true, true);
    }
    
    // Nettoyer la session
    $_SESSION = [];
    session_destroy();
    
    // Redémarrer une nouvelle session
    session_start();
}

/**
 * Obtenir l'utilisateur connecté
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $pdo = getDbConnection();
        
        $stmt = $pdo->prepare("
            SELECT id, name, email, role, status, avatar, timezone, language, last_login
            FROM users 
            WHERE id = ?
        ");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetch();
        
    } catch (Exception $e) {
        error_log("Erreur getCurrentUser: " . $e->getMessage());
        return null;
    }
}

/**
 * ================================================================
 * 🛡️ PERMISSIONS
 * ================================================================
 */

/**
 * Vérifier si l'utilisateur a un rôle spécifique
 */
function hasRole($role) {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === $role;
}

/**
 * Vérifier si l'utilisateur est admin
 */
function isAdmin() {
    return hasRole('admin');
}

/**
 * Vérifier si l'utilisateur est manager
 */
function isManager() {
    return hasRole('manager') || hasRole('admin');
}

/**
 * Forcer un rôle spécifique
 */
function requireRole($role) {
    requireLogin();
    
    if (!hasRole($role)) {
        header('HTTP/1.1 403 Forbidden');
        die("Accès refusé - Rôle requis: $role");
    }
}

/**
 * Forcer les droits admin
 */
function requireAdmin() {
    requireLogin();
    
    if (!isAdmin()) {
        header('HTTP/1.1 403 Forbidden');
        die("Accès refusé - Droits administrateur requis");
    }
}

/**
 * ================================================================
 * 📊 LOGS ET AUDIT
 * ================================================================
 */

/**
 * Logger les tentatives de connexion
 */
function logLoginAttempt($email, $success, $reason = null) {
    try {
        $pdo = getDbConnection();
        
        // Vérifier si la table existe
        $stmt = $pdo->query("SHOW TABLES LIKE 'login_attempts'");
        if ($stmt->rowCount() == 0) {
            return; // Table n'existe pas, skip
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO login_attempts 
            (email, ip_address, user_agent, success, failure_reason, attempted_at)
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $email,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            $success ? 1 : 0,
            $reason
        ]);
        
    } catch (Exception $e) {
        error_log("Erreur log login: " . $e->getMessage());
    }
}

/**
 * Logger les connexions utilisateur
 */
function logUserConnection($userId, $action) {
    try {
        $pdo = getDbConnection();
        
        // Vérifier si la table existe
        $stmt = $pdo->query("SHOW TABLES LIKE 'log_connexions'");
        if ($stmt->rowCount() == 0) {
            return; // Table n'existe pas, skip
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO log_connexions 
            (user_id, ip, user_agent, action, ip_address, created_at)
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $userId,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            $action,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);
        
    } catch (Exception $e) {
        error_log("Erreur log connexion: " . $e->getMessage());
    }
}

/**
 * Logger une activité utilisateur
 */
function logActivity($userId, $action, $entityType = null, $entityId = null, $description = null) {
    try {
        $pdo = getDbConnection();
        
        // Vérifier si la table existe
        $stmt = $pdo->query("SHOW TABLES LIKE 'activity_logs'");
        if ($stmt->rowCount() == 0) {
            return; // Table n'existe pas, skip
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, entity_type, entity_id, description, ip_address, user_agent, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $userId,
            $action,
            $entityType,
            $entityId,
            $description,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
        
    } catch (Exception $e) {
        error_log("Erreur log activity: " . $e->getMessage());
    }
}

/**
 * ================================================================
 * 🔒 SÉCURITÉ
 * ================================================================
 */

/**
 * Générer un token CSRF
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Vérifier le token CSRF
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && 
           hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Vérifier le rate limiting
 */
function checkRateLimit($key, $maxAttempts = 5, $timeWindow = 300) {
    if (!isset($_SESSION['rate_limits'])) {
        $_SESSION['rate_limits'] = [];
    }
    
    $now = time();
    $windowStart = $now - $timeWindow;
    
    // Nettoyer les anciennes tentatives
    if (isset($_SESSION['rate_limits'][$key])) {
        $_SESSION['rate_limits'][$key] = array_filter(
            $_SESSION['rate_limits'][$key],
            function($timestamp) use ($windowStart) {
                return $timestamp > $windowStart;
            }
        );
    } else {
        $_SESSION['rate_limits'][$key] = [];
    }
    
    // Vérifier la limite
    if (count($_SESSION['rate_limits'][$key]) >= $maxAttempts) {
        return false;
    }
    
    // Enregistrer cette tentative
    $_SESSION['rate_limits'][$key][] = $now;
    
    return true;
}

/**
 * Nettoyer et valider l'input utilisateur
 */
function cleanInput($input) {
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

/**
 * Valider un email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Hasher un mot de passe
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * ================================================================
 * 🚀 UTILITAIRES
 * ================================================================
 */

/**
 * Rediriger en toute sécurité
 */
function safeRedirect($url) {
    // Valider que l'URL est locale
    $parsed = parse_url($url);
    if (isset($parsed['host']) && $parsed['host'] !== $_SERVER['HTTP_HOST']) {
        $url = '/social-media-generator/dashboard.php';
    }
    
    header("Location: $url");
    exit;
}

/**
 * Obtenir l'IP réelle du client
 */
function getRealIpAddr() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'];
    }
}

/**
 * Vérifier si la connexion est sécurisée
 */
function isSecureConnection() {
    return isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
}

/**
 * Générer un UUID v4
 */
function generateUUID() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

/**
 * ================================================================
 * 🔄 AUTO-LOGIN VIA REMEMBER TOKEN
 * ================================================================
 */

/**
 * Vérifier le token remember me
 */
function checkRememberToken() {
    if (isLoggedIn()) {
        return true;
    }
    
    if (!isset($_COOKIE['remember_token'])) {
        return false;
    }
    
    try {
        $pdo = getDbConnection();
        
        $stmt = $pdo->prepare("
            SELECT id, name, email, role 
            FROM users 
            WHERE remember_token = ? 
            AND remember_expires > NOW() 
            AND status = 'active'
        ");
        $stmt->execute([$_COOKIE['remember_token']]);
        $user = $stmt->fetch();
        
        if ($user) {
            // Reconnecter automatiquement
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['name'];
            $_SESSION['user_role'] = $user['role'];
            $_SESSION['last_activity'] = time();
            
            logUserConnection($user['id'], 'auto_login');
            
            return true;
        }
        
    } catch (Exception $e) {
        error_log("Erreur remember token: " . $e->getMessage());
    }
    
    return false;
}

/**
 * ================================================================
 * 🎯 HELPERS SPÉCIFIQUES SMG STUDIO
 * ================================================================
 */

/**
 * Vérifier l'existence d'une table
 */
function tableExists($pdo, $tableName) {
    try {
        $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
        $stmt->execute([$tableName]);
        return $stmt->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Obtenir les stats utilisateur pour le dashboard
 */
function getUserStats($userId) {
    try {
        $pdo = getDbConnection();
        
        $stats = [
            'total_posts' => 0,
            'total_clients' => 0,
            'total_projects' => 0,
            'total_exports' => 0,
            'social_accounts' => 0
        ];
        
        // Posts
        if (tableExists($pdo, 'posts')) {
            try {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM posts WHERE user_id = ?");
                $stmt->execute([$userId]);
                $stats['total_posts'] = $stmt->fetch()['count'] ?? 0;
            } catch (Exception $e) {
                error_log("Erreur stats posts: " . $e->getMessage());
            }
        }
        
        // Clients
        if (tableExists($pdo, 'clients')) {
            try {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM clients WHERE created_by = ?");
                $stmt->execute([$userId]);
                $stats['total_clients'] = $stmt->fetch()['count'] ?? 0;
            } catch (Exception $e) {
                error_log("Erreur stats clients: " . $e->getMessage());
            }
        }
        
        // Projets Clipmaker
        if (tableExists($pdo, 'clipmaker_projects')) {
            try {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM clipmaker_projects WHERE user_id = ?");
                $stmt->execute([$userId]);
                $stats['total_projects'] = $stmt->fetch()['count'] ?? 0;
            } catch (Exception $e) {
                error_log("Erreur stats projects: " . $e->getMessage());
            }
        }
        
        // Exports Clipmaker
        if (tableExists($pdo, 'clipmaker_exports')) {
            try {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM clipmaker_exports WHERE user_id = ?");
                $stmt->execute([$userId]);
                $stats['total_exports'] = $stmt->fetch()['count'] ?? 0;
            } catch (Exception $e) {
                error_log("Erreur stats exports: " . $e->getMessage());
            }
        }
        
        // Comptes sociaux
        if (tableExists($pdo, 'social_accounts')) {
            try {
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM social_accounts WHERE user_id = ? AND is_active = 1");
                $stmt->execute([$userId]);
                $stats['social_accounts'] = $stmt->fetch()['count'] ?? 0;
            } catch (Exception $e) {
                error_log("Erreur stats social: " . $e->getMessage());
            }
        }
        
        return $stats;
        
    } catch (Exception $e) {
        error_log("Erreur getUserStats: " . $e->getMessage());
        return [
            'total_posts' => 0,
            'total_clients' => 0,
            'total_projects' => 0,
            'total_exports' => 0,
            'social_accounts' => 0
        ];
    }
}

/**
 * Vérifier l'accès à un projet Clipmaker
 */
function canAccessProject($projectId) {
    if (!isLoggedIn()) {
        return false;
    }
    
    try {
        $pdo = getDbConnection();
        
        if (!tableExists($pdo, 'clipmaker_projects')) {
            return false;
        }
        
        $stmt = $pdo->prepare("
            SELECT COUNT(*) 
            FROM clipmaker_projects 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$projectId, $_SESSION['user_id']]);
        
        return $stmt->fetchColumn() > 0;
        
    } catch (Exception $e) {
        error_log("Erreur vérification projet: " . $e->getMessage());
        return false;
    }
}

/**
 * Vérifier l'accès à un client
 */
function canAccessClient($clientId) {
    if (!isLoggedIn()) {
        return false;
    }
    
    try {
        $pdo = getDbConnection();
        
        if (!tableExists($pdo, 'clients')) {
            return false;
        }
        
        $stmt = $pdo->prepare("
            SELECT COUNT(*) 
            FROM clients 
            WHERE id = ? AND created_by = ?
        ");
        $stmt->execute([$clientId, $_SESSION['user_id']]);
        
        return $stmt->fetchColumn() > 0;
        
    } catch (Exception $e) {
        error_log("Erreur vérification client: " . $e->getMessage());
        return false;
    }
}

// Vérifier automatiquement le remember token
if (!isLoggedIn()) {
    checkRememberToken();
}

// ================================================================
// 🔒 PROTECTION AUTOMATIQUE
// ================================================================

// Protection contre les attaques par force brute
$clientIP = getRealIpAddr();
if (!checkRateLimit("global_$clientIP", 100, 3600)) {
    header('HTTP/1.1 429 Too Many Requests');
    die('Trop de requêtes. Veuillez patienter.');
}

// Headers de sécurité
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

if (isSecureConnection()) {
    header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
}

// ================================================================
// 🎯 INITIALISATION
// ================================================================

// Log du démarrage (dev uniquement)
if (defined('DEBUG') && DEBUG) {
    error_log("🔐 Auth system initialized for IP: " . $clientIP);
}

?>