<?php
/*
 * SMG Studio v2.0 - Social Media Generator - Audiovizu Edition
 * API Workflow - Gestion logs utilisateur sécurisée
 * Path: /var/www/laloee-cours.fr/social-media-generator/api/workflow.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Configuration sécurisée
class SMGSecurityManager {
    private static $allowedUsers = ['user_001', 'user_002', 'user_003']; // En production: DB
    private static $baseWorkspace = '/var/www/laloee-cours.fr/social-media-generator/user_workspaces/';
    
    public static function validateUser($userId) {
        // Validation format user ID
        if (!preg_match('/^user_\d{3}$/', $userId)) {
            return false;
        }
        
        // En production, vérifier en base de données
        return in_array($userId, self::$allowedUsers);
    }
    
    public static function getUserWorkspace($userId) {
        if (!self::validateUser($userId)) {
            throw new Exception('User non autorisé');
        }
        
        $workspace = self::$baseWorkspace . $userId . '/';
        if (!is_dir($workspace)) {
            mkdir($workspace, 0755, true);
            mkdir($workspace . 'logs/', 0755, true);
            mkdir($workspace . 'projects/', 0755, true);
            mkdir($workspace . 'archives/', 0755, true);
        }
        
        return $workspace;
    }
    
    public static function sanitizeInput($input) {
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }
    
    public static function logSecurityEvent($event, $userId = null, $details = '') {
        $logFile = self::$baseWorkspace . 'security.log';
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event' => $event,
            'user' => $userId,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'details' => $details
        ];
        
        file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
    }
}

// Classe principale API
class SMGWorkflowAPI {
    private $userId;
    private $workspace;
    
    public function __construct() {
        $this->userId = $this->getUserId();
        
        if (!$this->userId) {
            $this->sendError('User ID requis', 400);
        }
        
        try {
            $this->workspace = SMGSecurityManager::getUserWorkspace($this->userId);
        } catch (Exception $e) {
            SMGSecurityManager::logSecurityEvent('unauthorized_access', $this->userId, $e->getMessage());
            $this->sendError('Accès non autorisé', 403);
        }
    }
    
    private function getUserId() {
        if (isset($_GET['user'])) {
            return SMGSecurityManager::sanitizeInput($_GET['user']);
        }
        
        if (isset($_POST['user'])) {
            return SMGSecurityManager::sanitizeInput($_POST['user']);
        }
        
        // Pour requêtes JSON
        $input = json_decode(file_get_contents('php://input'), true);
        if (isset($input['user'])) {
            return SMGSecurityManager::sanitizeInput($input['user']);
        }
        
        return null;
    }
    
    public function handleRequest() {
        $action = $_GET['action'] ?? $_POST['action'] ?? '';
        
        switch ($action) {
            case 'stats':
                $this->getStats();
                break;
                
            case 'log':
                $this->logAction();
                break;
                
            case 'load_project':
                $this->loadProject();
                break;
                
            case 'save_project':
                $this->saveProject();
                break;
                
            case 'get_logs':
                $this->getLogs();
                break;
                
            default:
                $this->sendError('Action non reconnue', 400);
        }
    }
    
    private function getStats() {
        try {
            // Statistiques isolées par utilisateur
            $statsFile = $this->workspace . 'stats.json';
            
            if (file_exists($statsFile)) {
                $stats = json_decode(file_get_contents($statsFile), true);
            } else {
                // Stats par défaut
                $stats = [
                    'totalProjects' => 0,
                    'activeWorkflows' => 0,
                    'completedToday' => 0,
                    'avgTime' => '0h'
                ];
            }
            
            // Calculer stats en temps réel
            $this->updateRealTimeStats($stats);
            
            $this->sendSuccess($stats);
            
        } catch (Exception $e) {
            $this->sendError('Erreur récupération stats: ' . $e->getMessage(), 500);
        }
    }
    
    private function updateRealTimeStats(&$stats) {
        // Compter projets dans archives
        $archiveDir = $this->workspace . 'archives/';
        if (is_dir($archiveDir)) {
            $stats['totalProjects'] = count(glob($archiveDir . '*.json'));
        }
        
        // Compter logs d'aujourd'hui
        $logsDir = $this->workspace . 'logs/';
        $todayFile = $logsDir . date('Y-m-d') . '.log';
        if (file_exists($todayFile)) {
            $lines = file($todayFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $completed = 0;
            
            foreach ($lines as $line) {
                $logEntry = json_decode($line, true);
                if ($logEntry && isset($logEntry['action']) && 
                    strpos($logEntry['action'], 'complete') !== false) {
                    $completed++;
                }
            }
            
            $stats['completedToday'] = $completed;
        }
        
        // Workflows actifs (stages en cours)
        $stats['activeWorkflows'] = 1; // Projet actuel
        
        // Sauvegarder stats mises à jour
        file_put_contents($this->workspace . 'stats.json', json_encode($stats, JSON_PRETTY_PRINT));
    }
    
    private function logAction() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['action'])) {
                $this->sendError('Données log invalides', 400);
            }
            
            $logEntry = [
                'timestamp' => date('Y-m-d H:i:s'),
                'user' => $this->userId,
                'action' => SMGSecurityManager::sanitizeInput($input['action']),
                'details' => SMGSecurityManager::sanitizeInput($input['details'] ?? ''),
                'stage' => SMGSecurityManager::sanitizeInput($input['stage'] ?? ''),
                'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'session_id' => session_id()
            ];
            
            // Log dans fichier journalier isolé
            $logFile = $this->workspace . 'logs/' . date('Y-m-d') . '.log';
            file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
            
            // Log global sécurisé (sans détails sensibles)
            SMGSecurityManager::logSecurityEvent('user_action', $this->userId, $input['action']);
            
            $this->sendSuccess(['logged' => true]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur logging: ' . $e->getMessage(), 500);
        }
    }
    
    private function loadProject() {
        try {
            $projectFile = $this->workspace . 'current_project.json';
            
            if (file_exists($projectFile)) {
                $project = json_decode(file_get_contents($projectFile), true);
                $this->sendSuccess(['success' => true, 'project' => $project]);
            } else {
                $this->sendSuccess(['success' => false, 'message' => 'Aucun projet actuel']);
            }
            
        } catch (Exception $e) {
            $this->sendError('Erreur chargement projet: ' . $e->getMessage(), 500);
        }
    }
    
    private function saveProject() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['project'])) {
                $this->sendError('Données projet invalides', 400);
            }
            
            // Validation et nettoyage des données projet
            $project = $this->sanitizeProject($input['project']);
            
            $projectFile = $this->workspace . 'current_project.json';
            file_put_contents($projectFile, json_encode($project, JSON_PRETTY_PRINT));
            
            // Log de sauvegarde
            $this->logProjectAction('project_saved', $project['name'] ?? 'Unnamed');
            
            $this->sendSuccess(['saved' => true]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur sauvegarde projet: ' . $e->getMessage(), 500);
        }
    }
    
    private function sanitizeProject($project) {
        $sanitized = [];
        
        // Champs autorisés
        $allowedFields = ['name', 'currentStage', 'progress', 'duration', 'startedAt', 'files', 'metadata'];
        
        foreach ($allowedFields as $field) {
            if (isset($project[$field])) {
                if (is_string($project[$field])) {
                    $sanitized[$field] = SMGSecurityManager::sanitizeInput($project[$field]);
                } else {
                    $sanitized[$field] = $project[$field];
                }
            }
        }
        
        // Validation stage
        if (isset($sanitized['currentStage'])) {
            $sanitized['currentStage'] = max(1, min(4, intval($sanitized['currentStage'])));
        }
        
        // Validation progress
        if (isset($sanitized['progress'])) {
            $sanitized['progress'] = max(0, min(100, intval($sanitized['progress'])));
        }
        
        return $sanitized;
    }
    
    private function getLogs() {
        try {
            $date = $_GET['date'] ?? date('Y-m-d');
            $limit = min(100, intval($_GET['limit'] ?? 50));
            
            // Validation date
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
                $this->sendError('Format date invalide', 400);
            }
            
            $logFile = $this->workspace . 'logs/' . $date . '.log';
            $logs = [];
            
            if (file_exists($logFile)) {
                $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                $lines = array_slice(array_reverse($lines), 0, $limit); // Plus récents en premier
                
                foreach ($lines as $line) {
                    $logEntry = json_decode($line, true);
                    if ($logEntry) {
                        // Nettoyer données sensibles
                        unset($logEntry['ip'], $logEntry['session_id']);
                        $logs[] = $logEntry;
                    }
                }
            }
            
            $this->sendSuccess(['logs' => $logs, 'total' => count($logs)]);
            
        } catch (Exception $e) {
            $this->sendError('Erreur récupération logs: ' . $e->getMessage(), 500);
        }
    }
    
    private function logProjectAction($action, $projectName) {
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'user' => $this->userId,
            'action' => $action,
            'details' => $projectName,
            'stage' => 'project_management',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ];
        
        $logFile = $this->workspace . 'logs/' . date('Y-m-d') . '.log';
        file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
    }
    
    private function sendSuccess($data) {
        echo json_encode(array_merge(['status' => 'success'], $data));
        exit;
    }
    
    private function sendError($message, $code = 400) {
        http_response_code($code);
        echo json_encode(['status' => 'error', 'message' => $message]);
        exit;
    }
}

// Point d'entrée principal
try {
    $api = new SMGWorkflowAPI();
    $api->handleRequest();
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erreur serveur']);
    
    // Log erreur
    SMGSecurityManager::logSecurityEvent('api_error', null, $e->getMessage());
}