<?php
/**
 * ================================================================
 * 🎨 SMG STUDIO v2.0 - GÉNÉRATEUR CONTENU IA - CORRIGÉ COMPLET
 * ================================================================
 * 
 * 🔧 CORRECTIONS MAJEURES :
 * ✅ Navigation retour dashboard CORRIGÉE
 * ✅ Chargement profils utilisateur connecté CORRIGÉ
 * ✅ Génération OpenAI CORRIGÉE avec vraies clés API
 * ✅ Validation et sécurité CORRIGÉES
 * ✅ Gestion erreurs et debug AMÉLIORÉE
 * ✅ Interface utilisateur OPTIMISÉE
 * 
 * 💡 BY PAPY 77 - GÉNÉRATEUR IA FONCTIONNEL !
 * ================================================================
 */

session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur avec debug
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        error_log("❌ ERREUR: Aucun user actif trouvé pour ID " . $_SESSION['user_id']);
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
    
    error_log("✅ DEBUG Générateur: User connecté: " . $user['name'] . " (ID: " . $user['id'] . ")");
} catch (Exception $e) {
    error_log("❌ ERREUR DB user: " . $e->getMessage());
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS - CORRIGÉ
// ================================================================

$action = $_POST['action'] ?? $_GET['action'] ?? 'generator';
$message = '';
$message_type = '';

// Variables pour résultats de génération
$generated_content = null;
$generation_stats = null;

// Action Génération IA CORRIGÉE
if ($action === 'generate_content' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $ai_profile_id = intval($_POST['ai_profile_id'] ?? 0);
        $prompt = trim($_POST['prompt'] ?? '');
        $platform = $_POST['platform'] ?? 'instagram';
        $content_type = $_POST['content_type'] ?? 'post';
        $custom_temperature = floatval($_POST['temperature'] ?? 0.7);
        $custom_max_tokens = intval($_POST['max_tokens'] ?? 1000);
        
        error_log("🎨 DEBUG Génération: User {$user['id']}, Profile ID: $ai_profile_id, Prompt: " . substr($prompt, 0, 50) . "...");
        
        // Validation des entrées
        if (empty($prompt)) {
            throw new Exception("Le prompt est requis");
        }
        
        if ($ai_profile_id === 0) {
            throw new Exception("Veuillez sélectionner un profil IA");
        }
        
        if (strlen($prompt) < 10) {
            throw new Exception("Le prompt doit contenir au moins 10 caractères");
        }
        
        if ($custom_max_tokens < 50 || $custom_max_tokens > 4000) {
            throw new Exception("Tokens entre 50 et 4000");
        }
        
        // Générer le contenu avec la fonction corrigée
        $generation_result = generateAIContentFixed(
            $ai_profile_id, 
            $prompt, 
            $platform, 
            $content_type,
            $custom_temperature,
            $custom_max_tokens,
            $user['id']
        );
        
        if ($generation_result['success']) {
            $message = "✅ Contenu généré avec succès !";
            $message_type = 'success';
            $generated_content = $generation_result['content'];
            $generation_stats = $generation_result['stats'];
            
            error_log("✅ DEBUG: Génération réussie, tokens: " . ($generation_stats['tokens_used'] ?? 0));
        } else {
            throw new Exception($generation_result['error']);
        }
        
    } catch (Exception $e) {
        error_log("❌ ERREUR génération: " . $e->getMessage());
        $message = "❌ Erreur génération : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DES DONNÉES - CORRIGÉ
// ================================================================

// 🔧 CORRECTION: Chargement profils avec debug amélioré
$user_profiles = getUserAIProfilesForGenerator($user['id']);
error_log("✅ DEBUG Générateur: Profils chargés pour user {$user['id']}: " . count($user_profiles));

// Templates améliorés
$prompt_templates = [
    [
        'id' => 1,
        'name' => 'Post Instagram Engageant',
        'category' => 'instagram',
        'template' => 'Crée un post Instagram captivant sur "{sujet}" avec un ton motivant, des emojis pertinents et des hashtags populaires. Maximum 2200 caractères.'
    ],
    [
        'id' => 2,
        'name' => 'Post LinkedIn Professionnel',
        'category' => 'linkedin', 
        'template' => 'Rédige un post LinkedIn professionnel sur "{sujet}" avec des conseils pratiques, une structure claire et un appel à l\'action. Ton expert et bienveillant.'
    ],
    [
        'id' => 3,
        'name' => 'Post Facebook Convivial',
        'category' => 'facebook',
        'template' => 'Écris un post Facebook chaleureux sur "{sujet}" qui génère de l\'engagement et des commentaires. Ton amical et accessible.'
    ],
    [
        'id' => 4,
        'name' => 'Tweet Percutant',
        'category' => 'twitter',
        'template' => 'Compose un tweet percutant sur "{sujet}" en 280 caractères maximum. Accrocheur et viral.'
    ],
    [
        'id' => 5,
        'name' => 'Story Instagram',
        'category' => 'instagram',
        'template' => 'Crée un texte pour story Instagram sur "{sujet}" avec des emojis et un call-to-action. Court et impactant.'
    ],
    [
        'id' => 6,
        'name' => 'Post TikTok Viral',
        'category' => 'tiktok',
        'template' => 'Rédige une description TikTok accrocheuse sur "{sujet}" avec hashtags tendance et appel à l\'action pour likes et partages.'
    ]
];

// Quota avec gestion d'erreurs
$quota_status = getQuotaStatusSafe($user['id']);

// ================================================================
// 🔧 FONCTIONS MÉTIER CORRIGÉES
// ================================================================

function getUserAIProfilesForGenerator($user_id) {
    global $pdo;
    
    try {
        // 🔧 REQUÊTE CORRIGÉE avec jointures sécurisées
        $stmt = $pdo->prepare("
            SELECT uap.id,
                   uap.ai_provider,
                   uap.display_name,
                   uap.model_preference,
                   uap.max_tokens_per_request,
                   uap.temperature,
                   uap.system_prompt,
                   uap.is_default,
                   uap.is_active,
                   ak.encrypted_key,
                   ak.key_name,
                   apc.provider_name,
                   COALESCE(COUNT(aul.id), 0) as usage_count,
                   COALESCE(SUM(aul.tokens_total), 0) as total_tokens
            FROM user_ai_profiles uap
            LEFT JOIN api_keys ak ON uap.api_key_id = ak.id AND ak.user_id = uap.user_id
            LEFT JOIN ai_providers_catalog apc ON uap.ai_provider = apc.provider_key
            LEFT JOIN ai_usage_logs aul ON uap.id = aul.ai_profile_id
            WHERE uap.user_id = ? AND uap.is_active = 1
            GROUP BY uap.id
            ORDER BY uap.is_default DESC, uap.display_name ASC
        ");
        $stmt->execute([$user_id]);
        $profiles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Debug des profils
        foreach ($profiles as $profile) {
            $api_status = $profile['encrypted_key'] ? 'Clé OK' : 'Pas de clé';
            error_log("   📋 Profil {$profile['id']}: {$profile['display_name']} ({$profile['ai_provider']}) - $api_status");
        }
        
        return is_array($profiles) ? $profiles : [];
    } catch (Exception $e) {
        error_log("❌ ERREUR getUserAIProfilesForGenerator: " . $e->getMessage());
        return [];
    }
}

function generateAIContentFixed($ai_profile_id, $prompt, $platform, $content_type, $temperature, $max_tokens, $user_id) {
    global $pdo;
    
    try {
        error_log("🔧 DEBUG generateAIContentFixed: Profile ID: $ai_profile_id, User ID: $user_id");
        
        // 🔧 CORRECTION: Récupération profil avec validation stricte
        $stmt = $pdo->prepare("
            SELECT uap.*, ak.encrypted_key 
            FROM user_ai_profiles uap
            LEFT JOIN api_keys ak ON uap.api_key_id = ak.id
            WHERE uap.id = ? AND uap.user_id = ? AND uap.is_active = 1
        ");
        $stmt->execute([$ai_profile_id, $user_id]);
        $profile = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Validation profil
        if (!$profile || !is_array($profile)) {
            error_log("❌ DEBUG: Profil non trouvé - ID: $ai_profile_id, User: $user_id");
            return ['success' => false, 'error' => 'Profil IA non trouvé ou inactif'];
        }
        
        if (!isset($profile['ai_provider']) || empty($profile['ai_provider'])) {
            error_log("❌ DEBUG: ai_provider manquant dans le profil");
            return ['success' => false, 'error' => 'Provider IA non défini'];
        }
        
        error_log("✅ DEBUG: Profil trouvé: " . $profile['ai_provider'] . " - " . $profile['display_name']);
        
        $start_time = microtime(true);
        
        // Construire le prompt enrichi
        $enriched_prompt = buildEnrichedPrompt($prompt, $platform, $content_type, $profile);
        
        // Générer selon le provider
        switch ($profile['ai_provider']) {
            case 'openai':
                $result = generateWithOpenAIFixed($profile, $enriched_prompt, $temperature, $max_tokens);
                break;
            case 'smg_free':
                $result = generateWithSMGFreeFixed($enriched_prompt, $max_tokens, $platform);
                break;
            case 'claude':
                $result = generateWithClaudeFixed($profile, $enriched_prompt, $temperature, $max_tokens);
                break;
            default:
                error_log("❌ DEBUG: Provider non supporté: " . $profile['ai_provider']);
                return ['success' => false, 'error' => 'Provider non supporté: ' . $profile['ai_provider']];
        }
        
        $processing_time = (microtime(true) - $start_time) * 1000;
        
        if ($result['success']) {
            // Logger l'utilisation
            logAIUsageFixed($user_id, $ai_profile_id, $prompt, $result, $processing_time, $platform);
            
            return [
                'success' => true,
                'content' => $result['content'],
                'stats' => [
                    'tokens_used' => $result['tokens_total'] ?? 0,
                    'cost' => $result['cost'] ?? 0,
                    'time_ms' => round($processing_time),
                    'model' => $result['model'] ?? $profile['ai_provider'],
                    'platform' => $platform
                ]
            ];
        } else {
            return $result;
        }
        
    } catch (Exception $e) {
        error_log("❌ EXCEPTION dans generateAIContentFixed: " . $e->getMessage());
        return ['success' => false, 'error' => 'Erreur: ' . $e->getMessage()];
    }
}

function buildEnrichedPrompt($base_prompt, $platform, $content_type, $profile) {
    // Prompt système par défaut ou personnalisé
    $system_context = $profile['system_prompt'] ?? 
        "Tu es un expert en création de contenu pour les réseaux sociaux. Tu crées du contenu engageant, authentique et optimisé pour chaque plateforme.";
    
    // Contraintes par plateforme
    $platform_constraints = [
        'instagram' => 'Format Instagram: 2200 caractères max, emojis, hashtags pertinents',
        'facebook' => 'Format Facebook: ton convivial, encourage l\'interaction',
        'linkedin' => 'Format LinkedIn: professionnel, valeur ajoutée, expertise',
        'twitter' => 'Format Twitter: 280 caractères max, hashtags, accrocheur',
        'tiktok' => 'Format TikTok: jeune, dynamique, hashtags tendance',
        'youtube' => 'Format YouTube: description captivante, SEO-friendly'
    ];
    
    $constraint = $platform_constraints[$platform] ?? 'Format adapté aux réseaux sociaux';
    
    return $system_context . "\n\n" . 
           "Contraintes: " . $constraint . "\n\n" .
           "Demande: " . $base_prompt;
}

function generateWithOpenAIFixed($profile, $prompt, $temperature, $max_tokens) {
    // 🔧 VALIDATION: Vérifier clé API
    if (!isset($profile['encrypted_key']) || empty($profile['encrypted_key'])) {
        return ['success' => false, 'error' => 'Clé API OpenAI manquante. Configurez votre profil IA.'];
    }
    
    $api_key = base64_decode($profile['encrypted_key']);
    
    // Validation format clé OpenAI
    if (!preg_match('/^sk-/', $api_key)) {
        return ['success' => false, 'error' => 'Format de clé OpenAI invalide (doit commencer par sk-)'];
    }
    
    $model = $profile['model_preference'] ?? 'gpt-3.5-turbo';
    
    error_log("🤖 DEBUG OpenAI: Modèle: $model, Tokens: $max_tokens, Temp: $temperature");
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.openai.com/v1/chat/completions",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 45,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer " . $api_key,
            "Content-Type: application/json",
            "User-Agent: SMG-Studio/2.0"
        ],
        CURLOPT_POSTFIELDS => json_encode([
            'model' => $model,
            'messages' => [
                [
                    'role' => 'system', 
                    'content' => 'Tu es un expert en création de contenu pour les réseaux sociaux. Réponds en français uniquement.'
                ],
                [
                    'role' => 'user', 
                    'content' => $prompt
                ]
            ],
            'max_tokens' => $max_tokens,
            'temperature' => $temperature,
            'top_p' => 0.9,
            'frequency_penalty' => 0.1,
            'presence_penalty' => 0.1
        ])
    ]);
    
    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($curl);
    curl_close($curl);
    
    // Gestion erreurs CURL
    if ($curl_error) {
        error_log("❌ CURL Error: " . $curl_error);
        return ['success' => false, 'error' => 'Erreur de connexion à OpenAI: ' . $curl_error];
    }
    
    error_log("🌐 DEBUG OpenAI Response: HTTP $http_code");
    
    if ($http_code === 200) {
        $data = json_decode($response, true);
        
        // Validation structure réponse
        if ($data && isset($data['choices'][0]['message']['content'])) {
            $content = trim($data['choices'][0]['message']['content']);
            $tokens_used = $data['usage']['total_tokens'] ?? 0;
            
            error_log("✅ DEBUG OpenAI: Succès - Tokens: $tokens_used, Contenu: " . substr($content, 0, 100));
            
            return [
                'success' => true,
                'content' => $content,
                'tokens_total' => $tokens_used,
                'cost' => calculateOpenAICost($tokens_used, $model),
                'model' => $model
            ];
        } else {
            error_log("❌ DEBUG: Réponse OpenAI invalide: " . substr($response, 0, 200));
            return ['success' => false, 'error' => 'Réponse OpenAI invalide'];
        }
    } else {
        $error_data = json_decode($response, true);
        $error_message = $error_data['error']['message'] ?? 'Erreur API OpenAI';
        
        error_log("❌ DEBUG OpenAI Error: $error_message (Code: $http_code)");
        
        return [
            'success' => false, 
            'error' => $error_message . " (Code: $http_code)"
        ];
    }
}

function generateWithSMGFreeFixed($prompt, $max_tokens, $platform) {
    // 🔧 Réponses améliorées selon la plateforme
    $platform_responses = [
        'instagram' => [
            "🎨 Contenu Instagram créé par SMG Studio !\n\n✨ Voici un post optimisé pour votre audience :\n\n🚀 " . 
            substr($prompt, 0, 100) . "...\n\nInnovation et créativité au service de votre communication digitale ! " .
            "Chaque jour est une nouvelle opportunité de briller sur Instagram.\n\n" .
            "#SMGStudio #Instagram #CreativeContent #SocialMedia #Innovation",
            
            "📸 SMG Studio - Votre créateur de contenu !\n\n💡 Inspiré par votre demande :\n" .
            substr($prompt, 0, 120) . "...\n\n🌟 3 conseils pour Instagram :\n" .
            "1️⃣ Soyez authentique dans vos posts\n" .
            "2️⃣ Utilisez des visuels de qualité\n" .
            "3️⃣ Engagez avec votre communauté\n\n" .
            "#InstagramTips #SMG #ContentCreator"
        ],
        'linkedin' => [
            "💼 Contenu LinkedIn professionnel généré par SMG Studio\n\n" .
            "Dans le monde professionnel d'aujourd'hui, " . substr($prompt, 0, 100) . "...\n\n" .
            "🎯 Points clés à retenir :\n" .
            "• Innovation constante\n• Excellence opérationnelle\n• Vision stratégique\n\n" .
            "Qu'en pensez-vous ? Partagez votre expérience en commentaires.\n\n" .
            "#LinkedIn #Professionnel #SMGStudio #Innovation"
        ],
        'facebook' => [
            "👋 Hey tout le monde !\n\n📱 SMG Studio a créé ce contenu spécialement pour vous :\n\n" .
            substr($prompt, 0, 150) . "...\n\n" .
            "🤝 N'hésitez pas à partager vos avis et expériences dans les commentaires ! " .
            "Votre communauté Facebook compte beaucoup pour nous.\n\n" .
            "👍 Likez si ça vous plaît !\n\n#Facebook #Community #SMGStudio"
        ]
    ];
    
    $responses = $platform_responses[$platform] ?? $platform_responses['instagram'];
    $selected_response = $responses[array_rand($responses)];
    
    return [
        'success' => true,
        'content' => $selected_response,
        'tokens_total' => 150,
        'cost' => 0,
        'model' => 'smg-free-v2'
    ];
}

function generateWithClaudeFixed($profile, $prompt, $temperature, $max_tokens) {
    // Simulation Claude (à implémenter avec vraie API)
    return [
        'success' => true,
        'content' => "Contenu généré par Claude (Anthropic) - Simulation\n\n" . 
                    "Basé sur votre demande: " . substr($prompt, 0, 100) . "...\n\n" .
                    "Claude génèrerait ici du contenu de haute qualité, nuancé et adapté à votre plateforme.",
        'tokens_total' => $max_tokens,
        'cost' => $max_tokens * 0.003 / 1000,
        'model' => 'claude-3-sonnet'
    ];
}

 

function logAIUsageFixed($user_id, $ai_profile_id, $prompt, $result, $processing_time, $platform) {
    global $pdo;
    
    try {
        // Vérifier si la table existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'ai_usage_logs'");
        if ($table_check->rowCount() == 0) {
            error_log("⚠️ Table ai_usage_logs n'existe pas");
            return;
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO ai_usage_logs 
            (user_id, ai_profile_id, request_type, prompt_text, response_text, 
             tokens_total, cost_estimate, processing_time_ms, success, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $user_id,
            $ai_profile_id,
            'content_generation',
            substr($prompt, 0, 1000), // Limite pour éviter les textes trop longs
            substr($result['content'] ?? '', 0, 2000),
            $result['tokens_total'] ?? 0,
            $result['cost'] ?? 0,
            round($processing_time),
            1
        ]);
        
        error_log("✅ DEBUG: Usage loggé pour user $user_id - Tokens: " . ($result['tokens_total'] ?? 0));
    } catch (Exception $e) {
        error_log("❌ ERREUR logging usage: " . $e->getMessage());
    }
}

function getQuotaStatusSafe($user_id) {
    global $pdo;
    
    try {
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_ai_quotas'");
        if ($table_check->rowCount() == 0) {
            return [
                'tokens_limit' => 10000,
                'tokens_used' => 0,
                'usage_percentage' => 0,
                'quota_status' => 'OK',
                'days_remaining' => 30,
                'cost_limit' => 10.00,
                'cost_used' => 0.00,
                'requests_used' => 0
            ];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                tokens_limit,
                COALESCE(tokens_used, 0) as tokens_used,
                ROUND((COALESCE(tokens_used, 0) / GREATEST(tokens_limit, 1)) * 100, 1) as usage_percentage,
                cost_limit,
                COALESCE(cost_used, 0) as cost_used,
                COALESCE(requests_used, 0) as requests_used,
                DATEDIFF(quota_end_date, CURDATE()) as days_remaining,
                CASE 
                    WHEN COALESCE(tokens_used, 0) >= tokens_limit THEN 'EXCEEDED'
                    WHEN (COALESCE(tokens_used, 0) / GREATEST(tokens_limit, 1)) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as quota_status
            FROM user_ai_quotas 
            WHERE user_id = ? AND is_active = 1 
            AND CURDATE() BETWEEN quota_start_date AND quota_end_date
            ORDER BY quota_period DESC
            LIMIT 1
        ");
        $stmt->execute([$user_id]);
        $quota = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $quota ?: [
            'tokens_limit' => 10000,
            'tokens_used' => 0,
            'usage_percentage' => 0,
            'quota_status' => 'OK',
            'days_remaining' => 30,
            'cost_limit' => 10.00,
            'cost_used' => 0.00,
            'requests_used' => 0
        ];
        
    } catch (Exception $e) {
        error_log("❌ ERREUR getQuotaStatusSafe: " . $e->getMessage());
        return [
            'tokens_limit' => 10000,
            'tokens_used' => 0,
            'usage_percentage' => 0,
            'quota_status' => 'OK',
            'days_remaining' => 30,
            'cost_limit' => 10.00,
            'cost_used' => 0.00,
            'requests_used' => 0
        ];
    }
}


// ================================================================
// 💰 NOUVELLE FONCTION - DÉBIT TOKENS UTILISATEUR
// ================================================================
function debitUserTokens($user_id, $tokens_used) {
    global $pdo;
    
    try {
        // Vérifier le solde avant débit
        $stmt = $pdo->prepare("
            SELECT id, tokens_remaining 
            FROM user_token_packages 
            WHERE user_id = ? AND status = 'active' 
            AND tokens_remaining >= ?
            ORDER BY purchase_date ASC
            LIMIT 1
        ");
        $stmt->execute([$user_id, $tokens_used]);
        $package = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$package) {
            throw new Exception("Solde tokens insuffisant");
        }
        
        // Débiter les tokens
        $stmt = $pdo->prepare("
            UPDATE user_token_packages 
            SET tokens_used = tokens_used + ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $success = $stmt->execute([$tokens_used, $package['id']]);
        
        if (!$success) {
            throw new Exception("Erreur lors du débit");
        }
        
        // Log du débit
        error_log("Débit tokens réussi - User: $user_id, Tokens: $tokens_used");
        
        return [
            'success' => true,
            'tokens_debited' => $tokens_used,
            'remaining_tokens' => $package['tokens_remaining'] - $tokens_used
        ];
        
    } catch (Exception $e) {
        error_log("Erreur débit tokens - User: $user_id, Tokens: $tokens_used, Erreur: " . $e->getMessage());
        
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

// ================================================================
// 💰 FONCTION HELPER - VÉRIFIER SOLDE UTILISATEUR
// ================================================================
function getUserTokenBalance($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                SUM(tokens_purchased) as total_purchased,
                SUM(tokens_used) as total_used,
                SUM(tokens_remaining) as current_balance
            FROM user_token_packages 
            WHERE user_id = ? AND status = 'active'
        ");
        $stmt->execute([$user_id]);
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Erreur récupération solde - User: $user_id, Erreur: " . $e->getMessage());
        return ['current_balance' => 0];
    }
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🎨 Générateur IA - SMG Studio v2.0 FIXED</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #4ecdc4, #44a08d);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        /* 🔧 NAVIGATION IA GÉNÉRATEUR CORRIGÉE */
        .nav-ia-generator {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-generator a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-generator a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .nav-ia-generator a.active {
            background: #ff6b6b;
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        /* Debug utilisateur */
        .user-debug {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            padding: 10px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
        }
        
        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }
        
        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .quota-info {
            background: linear-gradient(135deg, rgba(78, 205, 196, 0.1), rgba(68, 160, 141, 0.1));
            border: 1px solid rgba(78, 205, 196, 0.3);
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
        }

        .templates-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }

        .template-card {
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .template-card:hover {
            border-color: #4ecdc4;
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .template-name {
            font-weight: 600;
            color: #333;
            margin-bottom: 5px;
        }

        .template-preview {
            font-size: 12px;
            color: #666;
            font-style: italic;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        .form-select, .form-input, .form-textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.3s ease;
        }

        .form-select:focus, .form-input:focus, .form-textarea:focus {
            outline: none;
            border-color: #4ecdc4;
            box-shadow: 0 0 0 3px rgba(78, 205, 196, 0.1);
        }

        .form-textarea {
            resize: vertical;
            min-height: 120px;
        }

        .form-row {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 15px;
        }
        
        .btn {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-secondary {
            background: #6c757d;
        }

        .btn-success {
            background: linear-gradient(135deg, #56CCF2, #2F80ED);
        }
        
        .result-area {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            margin-top: 20px;
            display: none;
        }
        
        .result-content {
            background: white;
            padding: 15px;
            border-radius: 6px;
            border: 2px solid #4ecdc4;
            white-space: pre-wrap;
            line-height: 1.6;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }

        .no-profiles-message {
            text-align: center;
            padding: 60px;
            background: linear-gradient(135deg, rgba(255, 107, 107, 0.1), rgba(255, 142, 83, 0.1));
            border: 2px dashed rgba(255, 107, 107, 0.3);
            border-radius: 15px;
            color: #666;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            .nav-ia-generator {
                flex-wrap: wrap;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation IA Corrigée -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">🎨</div>
                <div>
                    <h1>Générateur IA - FIXED</h1>
                    <p>Créateur de contenu intelligent avec navigation corrigée</p>
                </div>
            </div>
            <div style="display: flex; align-items: center; gap: 15px;">
                
                <!-- 🔧 Debug utilisateur connecté -->
                <div class="user-debug">
                    👤 <?= htmlspecialchars($user['name']) ?> (ID: <?= $user['id'] ?>)
                </div>
                
                <!-- 🔧 NAVIGATION IA GÉNÉRATEUR CORRIGÉE -->
                <div class="nav-ia-generator">
                    <a href="dashboard.php">
                        ← Dashboard Principal
                    </a>
                    <a href="ai_content_generator.php" class="active">
                        🎨 IA Générateur
                    </a>
                    <a href="ai_dashboard.php">
                        📊 Dashboard IA
                    </a>
                    <a href="ai_profiles.php">
                        ⚙️ Profils IA
                    </a>
                </div>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type === 'success' ? 'success' : 'error' ?>">
                <?= $message ?>
            </div>
        <?php endif; ?>

        <!-- Debug Section si aucun profil -->
        <?php if (empty($user_profiles)): ?>
            <div class="no-profiles-message">
                <div style="font-size: 64px; margin-bottom: 20px;">🤖</div>
                <h3>Aucun profil IA configuré</h3>
                <p style="margin: 15px 0;">Vous devez d'abord créer un profil IA pour utiliser le générateur.</p>
                <p><strong>Utilisateur :</strong> <?= htmlspecialchars($user['name']) ?> (ID: <?= $user['id'] ?>)</p>
                <div style="margin-top: 20px;">
                    <a href="ai_profiles.php" class="btn">
                        ⚙️ Créer un profil IA
                    </a>
                </div>
            </div>
        <?php else: ?>

            <!-- Information quotas -->
            <div class="quota-info">
                <h4>📊 Status de vos quotas</h4>
                <p>Tokens utilisés : <strong><?= number_format($quota_status['tokens_used']) ?></strong> / <?= number_format($quota_status['tokens_limit']) ?> (<?= $quota_status['usage_percentage'] ?>%)</p>
                <p>Coût estimé : <strong><?= number_format($quota_status['cost_used'], 4) ?>€</strong> / <?= number_format($quota_status['cost_limit'], 2) ?>€</p>
                <p>Statut : <strong><?= $quota_status['quota_status'] ?></strong> • <?= $quota_status['days_remaining'] ?> jours restants</p>
            </div>

            <!-- Templates rapides -->
            <div class="content-card">
                <h3>⚡ Templates Rapides</h3>
                <div class="templates-grid">
                    <?php foreach ($prompt_templates as $template): ?>
                        <div class="template-card" onclick="useTemplate('<?= htmlspecialchars($template['template']) ?>', '<?= $template['category'] ?>')">
                            <div class="template-name"><?= htmlspecialchars($template['name']) ?></div>
                            <div class="template-preview">
                                <?= substr(htmlspecialchars($template['template']), 0, 80) ?>...
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Formulaire de génération -->
            <div class="content-card">
                <h3>🎨 Générateur de Contenu IA</h3>
                
                <form method="POST">
                    <input type="hidden" name="action" value="generate_content">
                    
                    <div class="form-group">
                        <label class="form-label">🤖 Profil IA (<?= count($user_profiles) ?> disponible(s))</label>
                        <select name="ai_profile_id" class="form-select" required>
                            <option value="">Sélectionner un profil</option>
                            <?php foreach ($user_profiles as $profile): ?>
                                <option value="<?= $profile['id'] ?>">
                                    <?= getProviderIcon($profile['ai_provider']) ?> 
                                    <?= htmlspecialchars($profile['display_name']) ?>
                                    (<?= ucfirst($profile['ai_provider']) ?>)
                                    <?php if ($profile['ai_provider'] !== 'smg_free' && empty($profile['encrypted_key'])): ?>
                                        - ⚠️ Clé manquante
                                    <?php elseif ($profile['ai_provider'] !== 'smg_free'): ?>
                                        - ✅ Clé OK
                                    <?php endif; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">💬 Votre demande de contenu</label>
                        <textarea name="prompt" id="promptText" class="form-textarea" 
                                  placeholder="Exemple: Crée un post LinkedIn professionnel sur l'importance de la formation continue en marketing digital, avec des conseils pratiques et un appel à l'action engageant." 
                                  required></textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">📱 Plateforme</label>
                            <select name="platform" class="form-select">
                                <option value="instagram">📸 Instagram</option>
                                <option value="linkedin">💼 LinkedIn</option>
                                <option value="facebook">📘 Facebook</option>
                                <option value="twitter">🐦 Twitter</option>
                                <option value="tiktok">   TikTok</option>
                                <option value="youtube">📺 YouTube</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">🌡️ Créativité (0-2)</label>
                            <input type="number" name="temperature" class="form-input" 
                                   value="0.7" min="0" max="2" step="0.1">
                            <small style="color: #666; font-size: 11px;">0 = Précis, 2 = Créatif</small>
                        </div>
                        <div class="form-group">
                            <label class="form-label">🎯 Tokens max</label>
                            <input type="number" name="max_tokens" class="form-input" 
                                   value="500" min="50" max="4000">
                        </div>
                    </div>
                    
                    <button type="submit" class="btn">
                        ✨ Générer le Contenu
                    </button>
                </form>
                
                <!-- Zone de résultat -->
                <?php if (isset($generated_content)): ?>
                    <div class="result-area" style="display: block;">
                        <h3>✅ Contenu généré :</h3>
                        <div class="result-content"><?= htmlspecialchars($generated_content) ?></div>
                        
                        <?php if (isset($generation_stats)): ?>
                            <div style="margin-top: 15px; font-size: 12px; color: #666; display: flex; justify-content: space-between; flex-wrap: wrap;">
                                <span>📊 Tokens: <?= $generation_stats['tokens_used'] ?></span>
                                <span>💰 Coût: <?= number_format($generation_stats['cost'], 4) ?>€</span>
                                <span>⏱️ Temps: <?= $generation_stats['time_ms'] ?>ms</span>
                                <span>🤖 Modèle: <?= $generation_stats['model'] ?></span>
                                <span>📱 Plateforme: <?= ucfirst($generation_stats['platform']) ?></span>
                            </div>
                        <?php endif; ?>

                        <!-- Actions sur le contenu généré -->
                        <div style="margin-top: 15px; display: flex; gap: 10px;">
                            <button onclick="copyToClipboard()" class="btn btn-secondary">
                                📋 Copier
                            </button>
                            <button onclick="regenerateContent()" class="btn btn-success">
                                🔄 Regénérer
                            </button>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

        <?php endif; ?>
    </div>

    <script>
        // ================================================================
        // 🔧 JAVASCRIPT GÉNÉRATEUR IA CORRIGÉ
        // ================================================================

        function useTemplate(template, platform) {
            document.getElementById('promptText').value = template;
            const platformSelect = document.querySelector('select[name="platform"]');
            if (platformSelect) {
                platformSelect.value = platform;
            }
            
            // Scroll vers le formulaire
            document.getElementById('promptText').scrollIntoView({ behavior: 'smooth' });
            document.getElementById('promptText').focus();
        }

        function copyToClipboard() {
            const resultContent = document.querySelector('.result-content');
            if (resultContent) {
                const text = resultContent.textContent;
                navigator.clipboard.writeText(text).then(() => {
                    // Feedback visuel
                    const button = event.target;
                    const originalText = button.innerHTML;
                    button.innerHTML = '✅ Copié !';
                    button.style.background = '#28a745';
                    
                    setTimeout(() => {
                        button.innerHTML = originalText;
                        button.style.background = '';
                    }, 2000);
                }).catch(err => {
                    console.error('Erreur copie:', err);
                    alert('Erreur lors de la copie');
                });
            }
        }

        function regenerateContent() {
            // Soumettre à nouveau le formulaire
            const form = document.querySelector('form');
            if (form) {
                form.submit();
            }
        }

        // Raccourcis clavier
        document.addEventListener('keydown', function(e) {
            // Échap : Retour dashboard principal
            if (e.key === 'Escape') {
                window.location.href = 'dashboard.php';
            }
            
            // Ctrl+Enter : Générer le contenu
            if (e.ctrlKey && e.key === 'Enter') {
                const form = document.querySelector('form');
                if (form) {
                    form.submit();
                }
            }
        });

        // Auto-save du prompt dans le localStorage (si disponible)
        const promptText = document.getElementById('promptText');
        if (promptText) {
            // Charger le prompt sauvegardé
            const savedPrompt = localStorage.getItem('smg_last_prompt');
            if (savedPrompt && !promptText.value) {
                promptText.value = savedPrompt;
            }
            
            // Sauvegarder automatiquement
            promptText.addEventListener('input', function() {
                localStorage.setItem('smg_last_prompt', this.value);
            });
        }

        console.log('🎨 SMG Studio IA Generator FIXED!');
        console.log('✅ Navigation retour dashboard CORRIGÉE');
        console.log('✅ Chargement profils utilisateur CORRIGÉ');
        console.log('✅ Génération OpenAI avec vraies clés CORRIGÉE');
        console.log('✅ Validation et debug AMÉLIORÉS');
        console.log('📊 Profils disponibles: <?= count($user_profiles) ?>');
    </script>
</body>
</html>