 <?php
/**
 * Flying Book Analytics - Tracking Endpoint
 * API pour collecter les données analytics des présentations
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Gérer les requêtes OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Seules les requêtes POST sont acceptées
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => $e->getMessage()]);
    error_log("Analytics tracking error: " . $e->getMessage());
}

// Gérer une vue de page (première visite)
function handlePageView($pdo, $data, $visitor_ip, $user_agent, $country, $utm_data) {
    $device_info = $data['device_info'] ?? [];
    
    // Vérifier si cette session existe déjà
    $stmt = $pdo->prepare("
        SELECT id FROM flying_book_analytics 
        WHERE project_id = ? AND session_id = ?
    ");
    $stmt->execute([$data['project_id'], $data['session_id']]);
    
    if ($stmt->fetch()) {
        // Session existe déjà, mettre à jour
        updateExistingSession($pdo, $data);
        return;
    }
    
    // Nouvelle session - insérer
    $stmt = $pdo->prepare("
        INSERT INTO flying_book_analytics (
            project_id, visitor_ip, user_agent, session_id, page_views, time_spent,
            slides_viewed, max_slide_reached, completion_rate, interactions,
            referrer, utm_source, utm_medium, utm_campaign, device_type, browser, os,
            screen_resolution, country, city, first_visit, last_visit
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
    ");
    
    $stmt->execute([
        $data['project_id'],
        $visitor_ip,
        $user_agent,
        $data['session_id'],
        1, // page_views
        $data['time_spent'] ?? 0,
        json_encode($data['slides_viewed'] ?? []),
        $data['max_slide_reached'] ?? 1,
        $data['completion_rate'] ?? 0,
        json_encode($data['interactions'] ?? []),
        $device_info['referrer'] ?? null,
        $utm_data['utm_source'] ?? null,
        $utm_data['utm_medium'] ?? null,
        $utm_data['utm_campaign'] ?? null,
        $device_info['device_type'] ?? 'desktop',
        $device_info['browser'] ?? null,
        $device_info['os'] ?? null,
        $device_info['screen_resolution'] ?? null,
        $country,
        null, // city - à implémenter avec une API géo
        null, // bounce_rate - calculé plus tard
        null  // conversion_event
    ]);
}

// Mettre à jour une session existante
function updateExistingSession($pdo, $data) {
    $stmt = $pdo->prepare("
        UPDATE flying_book_analytics SET
            page_views = page_views + 1,
            time_spent = ?,
            slides_viewed = ?,
            max_slide_reached = GREATEST(max_slide_reached, ?),
            completion_rate = ?,
            interactions = ?,
            last_visit = NOW()
        WHERE project_id = ? AND session_id = ?
    ");
    
    $stmt->execute([
        $data['time_spent'] ?? 0,
        json_encode($data['slides_viewed'] ?? []),
        $data['max_slide_reached'] ?? 1,
        $data['completion_rate'] ?? 0,
        json_encode($data['interactions'] ?? []),
        $data['project_id'],
        $data['session_id']
    ]);
}

// Gérer un heartbeat (mise à jour périodique)
function handleHeartbeat($pdo, $data) {
    $stmt = $pdo->prepare("
        UPDATE flying_book_analytics SET
            time_spent = ?,
            slides_viewed = ?,
            max_slide_reached = GREATEST(max_slide_reached, ?),
            completion_rate = ?,
            interactions = ?,
            last_visit = NOW()
        WHERE project_id = ? AND session_id = ?
    ");
    
    $stmt->execute([
        $data['time_spent'] ?? 0,
        json_encode($data['slides_viewed'] ?? []),
        $data['max_slide_reached'] ?? 1,
        $data['completion_rate'] ?? 0,
        json_encode($data['interactions'] ?? []),
        $data['project_id'],
        $data['session_id']
    ]);
}

// Gérer la fermeture de page
function handlePageLeave($pdo, $data) {
    $stmt = $pdo->prepare("
        UPDATE flying_book_analytics SET
            time_spent = ?,
            slides_viewed = ?,
            max_slide_reached = GREATEST(max_slide_reached, ?),
            completion_rate = ?,
            interactions = ?,
            bounce_rate = CASE WHEN ? = 1 THEN 1 ELSE 0 END,
            last_visit = NOW()
        WHERE project_id = ? AND session_id = ?
    ");
    
    $bounce = ($data['max_slide_reached'] ?? 1) <= 1 ? 1 : 0;
    
    $stmt->execute([
        $data['time_spent'] ?? 0,
        json_encode($data['slides_viewed'] ?? []),
        $data['max_slide_reached'] ?? 1,
        $data['completion_rate'] ?? 0,
        json_encode($data['interactions'] ?? []),
        $bounce,
        $data['project_id'],
        $data['session_id']
    ]);
}

// Gérer les événements génériques
function handleGenericEvent($pdo, $data) {
    // Pour les événements comme slide_change, click, etc.
    // On met juste à jour les données de la session
    handleHeartbeat($pdo, $data);
}

// Détecter le pays basé sur l'IP (implémentation basique)
function detectCountry($ip) {
    // Implémentation basique - en production, utiliser une API comme MaxMind GeoIP
    if (empty($ip) || $ip === '127.0.0.1' || strpos($ip, '192.168.') === 0) {
        return null;
    }
    
    // Quelques IP françaises connues pour les tests
    $french_ranges = [
        '2.0.0.0' => '2.255.255.255',
        '5.0.0.0' => '5.255.255.255',
        '37.0.0.0' => '37.255.255.255'
    ];
    
    foreach ($french_ranges as $start => $end) {
        if (ip2long($ip) >= ip2long($start) && ip2long($ip) <= ip2long($end)) {
            return 'FR';
        }
    }
    
    // API gratuite pour la géolocalisation (limitée)
    try {
        $context = stream_context_create([
            'http' => [
                'timeout' => 2,
                'user_agent' => 'FlyingBook Analytics/1.0'
            ]
        ]);
        
        $response = @file_get_contents("http://ip-api.com/json/{$ip}?fields=country,countryCode", false, $context);
        if ($response) {
            $geo_data = json_decode($response, true);
            if ($geo_data && isset($geo_data['countryCode'])) {
                return $geo_data['countryCode'];
            }
        }
    } catch (Exception $e) {
        // Ignorer les erreurs de géolocalisation
    }
    
    return null;
}

// Extraire les paramètres UTM de l'URL
function extractUTMParams($url) {
    $utm_params = [
        'utm_source' => null,
        'utm_medium' => null,
        'utm_campaign' => null,
        'utm_term' => null,
        'utm_content' => null
    ];
    
    if (empty($url)) {
        return $utm_params;
    }
    
    $parsed_url = parse_url($url);
    if (!isset($parsed_url['query'])) {
        return $utm_params;
    }
    
    parse_str($parsed_url['query'], $query_params);
    
    foreach ($utm_params as $param => $value) {
        if (isset($query_params[$param])) {
            $utm_params[$param] = $query_params[$param];
        }
    }
    
    return $utm_params;
}

// Nettoyer les anciennes données (à appeler périodiquement)
function cleanupOldAnalytics($pdo, $days = 90) {
    $stmt = $pdo->prepare("
        DELETE FROM flying_book_analytics 
        WHERE first_visit < DATE_SUB(NOW(), INTERVAL ? DAY)
    ");
    $stmt->execute([$days]);
    
    return $stmt->rowCount();
}

// Calculer des métriques avancées (à appeler périodiquement)
function calculateAdvancedMetrics($pdo) {
    // Mettre à jour les taux de rebond
    $pdo->exec("
        UPDATE flying_book_analytics 
        SET bounce_rate = CASE 
            WHEN max_slide_reached <= 1 AND time_spent < 30 THEN 1 
            ELSE 0 
        END
        WHERE bounce_rate IS NULL
    ");
    
    // Identifier les conversions (utilisateurs qui ont terminé la présentation)
    $pdo->exec("
        UPDATE flying_book_analytics 
        SET conversion_event = 'completed'
        WHERE completion_rate >= 95 AND conversion_event IS NULL
    ");
}

?>' => 'Method not allowed']);
    exit;
}

require_once __DIR__ . '/../env/.env.php';

try {
    // Lire les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Invalid JSON data');
    }
    
    // Valider les données requises
    $required_fields = ['project_id', 'session_id', 'event_type'];
    foreach ($required_fields as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            throw new Exception("Missing required field: $field");
        }
    }
    
    $project_id = (int)$data['project_id'];
    $session_id = $data['session_id'];
    $event_type = $data['event_type'];
    
    // Vérifier que le projet existe
    $pdo = getDatabaseConnection();
    $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE id = ? AND status = 'published'");
    $stmt->execute([$project_id]);
    
    if (!$stmt->fetch()) {
        throw new Exception('Project not found or not published');
    }
    
    // Extraire les informations de l'appareil
    $device_info = $data['device_info'] ?? [];
    $visitor_ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    // Détecter le pays (implémentation basique)
    $country = detectCountry($visitor_ip);
    
    // Extraire les paramètres UTM de l'URL
    $utm_data = extractUTMParams($device_info['url'] ?? '');
    
    // Gérer différents types d'événements
    switch ($event_type) {
        case 'page_view':
            handlePageView($pdo, $data, $visitor_ip, $user_agent, $country, $utm_data);
            break;
            
        case 'heartbeat':
            handleHeartbeat($pdo, $data);
            break;
            
        case 'page_leave':
            handlePageLeave($pdo, $data);
            break;
            
        default:
            // Événements génériques (slide_change, click, etc.)
            handleGenericEvent($pdo, $data);
            break;
    }
    
    echo json_encode(['success' => true]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['error