<?php
session_start();
require_once './env/.env.php';
require_once 'includes/subscription_manager.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];

try {
    $pdo = getDatabaseConnection();
    $subscriptionManager = new FlyingBookSubscriptionManager($pdo);
    $userPlan = $subscriptionManager->getUserPlan($user_id);
    
    // ========== GESTION DES ACTIONS ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'use_template':
                $template_id = (int)$_POST['template_id'];
                $project_name = trim($_POST['project_name']);
                
                if (empty($project_name)) {
                    echo json_encode(['success' => false, 'message' => 'Nom du projet requis']);
                    exit;
                }
                
                // Vérifier les limites utilisateur
                $limitCheck = $subscriptionManager->checkUserLimit($user_id, 'max_projects', 1);
                if (!$limitCheck['allowed']) {
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Limite de projets atteinte. Upgradez votre plan.',
                        'upgrade_needed' => true
                    ]);
                    exit;
                }
                
                $result = useTemplate($pdo, $template_id, $user_id, $project_name);
                echo json_encode($result);
                exit;
                
            case 'search_templates':
                $search = trim($_POST['search'] ?? '');
                $category = trim($_POST['category'] ?? '');
                $sort = $_POST['sort'] ?? 'popular';
                
                $templates = searchTemplates($pdo, $search, $category, $sort);
                echo json_encode(['success' => true, 'templates' => $templates]);
                exit;
                
            case 'get_template_details':
                $template_id = (int)$_POST['template_id'];
                $details = getTemplateDetails($pdo, $template_id);
                echo json_encode(['success' => true, 'template' => $details]);
                exit;
                
            case 'rate_template':
                $template_id = (int)$_POST['template_id'];
                $rating = (int)$_POST['rating'];
                
                if ($rating < 1 || $rating > 5) {
                    echo json_encode(['success' => false, 'message' => 'Note invalide']);
                    exit;
                }
                
                $result = rateTemplate($pdo, $template_id, $user_id, $rating);
                echo json_encode($result);
                exit;
                
            case 'favorite_template':
                $template_id = (int)$_POST['template_id'];
                $result = toggleFavoriteTemplate($pdo, $template_id, $user_id);
                echo json_encode($result);
                exit;
        }
    }
    
    // Récupérer les templates par défaut (populaires)
    $templates = getTemplates($pdo);
    $categories = getTemplateCategories($pdo);
    $featured_templates = getFeaturedTemplates($pdo);
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement: " . $e->getMessage();
    logError("Templates error", ['error' => $e->getMessage(), 'user_id' => $user_id]);
}

// ========== FONCTIONS TEMPLATES ==========

function getTemplates($pdo, $search = '', $category = '', $sort = 'popular', $limit = 20) {
    try {
        $whereConditions = ["t.is_public = 1"];
        $params = [];
        
        if (!empty($search)) {
            $whereConditions[] = "(t.template_name LIKE ? OR t.template_description LIKE ? OR t.category LIKE ?)";
            $searchTerm = "%{$search}%";
            $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
        }
        
        if (!empty($category)) {
            $whereConditions[] = "t.category = ?";
            $params[] = $category;
        }
        
        $whereClause = implode(' AND ', $whereConditions);
        
        $orderBy = match($sort) {
            'recent' => 'ORDER BY t.created_at DESC',
            'rating' => 'ORDER BY t.rating DESC, t.rating_count DESC',
            'name' => 'ORDER BY t.template_name ASC',
            'popular' => 'ORDER BY t.usage_count DESC, t.rating DESC',
            default => 'ORDER BY t.is_featured DESC, t.usage_count DESC'
        };
        
        $stmt = $pdo->prepare("
            SELECT t.*, u.name as creator_name,
                   (SELECT COUNT(*) FROM template_favorites tf WHERE tf.template_id = t.id) as favorite_count
            FROM flying_book_templates t
            LEFT JOIN users u ON t.user_id = u.id
            WHERE {$whereClause}
            {$orderBy}
            LIMIT {$limit}
        ");
        
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        logError("Get templates error", ['error' => $e->getMessage()]);
        return [];
    }
}

function getFeaturedTemplates($pdo, $limit = 4) {
    try {
        $stmt = $pdo->prepare("
            SELECT t.*, u.name as creator_name
            FROM flying_book_templates t
            LEFT JOIN users u ON t.user_id = u.id
            WHERE t.is_public = 1 AND t.is_featured = 1
            ORDER BY t.rating DESC, t.usage_count DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getTemplateCategories($pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT category, COUNT(*) as count
            FROM flying_book_templates
            WHERE is_public = 1
            GROUP BY category
            ORDER BY count DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function useTemplate($pdo, $template_id, $user_id, $project_name) {
    try {
        $pdo->beginTransaction();
        
        // Récupérer le template
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_templates 
            WHERE id = ? AND (is_public = 1 OR user_id = ?)
        ");
        $stmt->execute([$template_id, $user_id]);
        $template = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$template) {
            throw new Exception("Template non trouvé");
        }
        
        // Décoder la configuration du template
        $templateConfig = json_decode($template['template_config'], true);
        if (!$templateConfig) {
            throw new Exception("Configuration template invalide");
        }
        
        // Générer un slug unique pour le nouveau projet
        $base_slug = generateSlug($project_name);
        $project_slug = $base_slug;
        $counter = 1;
        
        while (true) {
            $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE project_slug = ? AND user_id = ?");
            $stmt->execute([$project_slug, $user_id]);
            if (!$stmt->fetch()) break;
            
            $project_slug = $base_slug . '-' . $counter;
            $counter++;
        }
        
        // Créer le nouveau projet basé sur le template
        $project_folder = 'project_' . uniqid();
        
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_projects 
            (user_id, project_name, project_slug, title, subtitle, description, 
             style_theme, transition_type, navigation_type, project_folder, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
        ");
        
        $stmt->execute([
            $user_id,
            $project_name,
            $project_slug,
            $templateConfig['title'] ?? $project_name,
            $templateConfig['subtitle'] ?? '',
            $templateConfig['description'] ?? $template['template_description'],
            $templateConfig['theme'] ?? 'moderne',
            $templateConfig['transition'] ?? '3d',
            $templateConfig['navigation'] ?? 'all',
            $project_folder
        ]);
        
        $new_project_id = $pdo->lastInsertId();
        
        // Créer les slides basées sur le template
        if (isset($templateConfig['slides']) && is_array($templateConfig['slides'])) {
            foreach ($templateConfig['slides'] as $index => $slideConfig) {
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_slides 
                    (project_id, slide_number, slide_title, media_type, media_file, text_overlay, display_order)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $new_project_id,
                    $index + 1,
                    $slideConfig['title'] ?? "Slide " . ($index + 1),
                    $slideConfig['type'] ?? 'text_image',
                    $slideConfig['media'] ?? 'placeholder.jpg',
                    $slideConfig['content'] ?? '',
                    $index
                ]);
            }
        }
        
        // Incrémenter le compteur d'utilisation du template
        $stmt = $pdo->prepare("
            UPDATE flying_book_templates 
            SET usage_count = usage_count + 1, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$template_id]);
        
        $pdo->commit();
        
        return [
            'success' => true,
            'message' => 'Projet créé à partir du template',
            'project_id' => $new_project_id,
            'redirect' => "flying_book_slides.php?id={$new_project_id}&step=2"
        ];
        
    } catch (Exception $e) {
        $pdo->rollback();
        logError("Use template error", ['error' => $e->getMessage(), 'template_id' => $template_id]);
        return [
            'success' => false,
            'message' => 'Erreur lors de la création: ' . $e->getMessage()
        ];
    }
}

function searchTemplates($pdo, $search, $category, $sort) {
    return getTemplates($pdo, $search, $category, $sort, 50);
}

function getTemplateDetails($pdo, $template_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT t.*, u.name as creator_name,
                   (SELECT COUNT(*) FROM template_favorites tf WHERE tf.template_id = t.id) as favorite_count,
                   (SELECT AVG(rating) FROM template_ratings tr WHERE tr.template_id = t.id) as avg_rating,
                   (SELECT COUNT(*) FROM template_ratings tr WHERE tr.template_id = t.id) as rating_count
            FROM flying_book_templates t
            LEFT JOIN users u ON t.user_id = u.id
            WHERE t.id = ? AND (t.is_public = 1 OR t.user_id = ?)
        ");
        $stmt->execute([$template_id, $_SESSION['user_id']]);
        $template = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($template) {
            // Décoder la config pour preview
            $template['config_decoded'] = json_decode($template['template_config'], true);
        }
        
        return $template;
    } catch (Exception $e) {
        return null;
    }
}

function rateTemplate($pdo, $template_id, $user_id, $rating) {
    try {
        // Vérifier si l'utilisateur a déjà noté ce template
        $stmt = $pdo->prepare("
            SELECT id FROM template_ratings 
            WHERE template_id = ? AND user_id = ?
        ");
        $stmt->execute([$template_id, $user_id]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // Mettre à jour la note existante
            $stmt = $pdo->prepare("
                UPDATE template_ratings 
                SET rating = ?, updated_at = NOW()
                WHERE template_id = ? AND user_id = ?
            ");
            $stmt->execute([$rating, $template_id, $user_id]);
        } else {
            // Créer une nouvelle note
            $stmt = $pdo->prepare("
                INSERT INTO template_ratings (template_id, user_id, rating, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([$template_id, $user_id, $rating]);
        }
        
        // Mettre à jour les stats du template
        updateTemplateRatingStats($pdo, $template_id);
        
        return ['success' => true, 'message' => 'Note enregistrée'];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erreur lors de la notation'];
    }
}

function toggleFavoriteTemplate($pdo, $template_id, $user_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT id FROM template_favorites 
            WHERE template_id = ? AND user_id = ?
        ");
        $stmt->execute([$template_id, $user_id]);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // Supprimer des favoris
            $stmt = $pdo->prepare("
                DELETE FROM template_favorites 
                WHERE template_id = ? AND user_id = ?
            ");
            $stmt->execute([$template_id, $user_id]);
            $action = 'removed';
        } else {
            // Ajouter aux favoris
            $stmt = $pdo->prepare("
                INSERT INTO template_favorites (template_id, user_id, created_at)
                VALUES (?, ?, NOW())
            ");
            $stmt->execute([$template_id, $user_id]);
            $action = 'added';
        }
        
        return [
            'success' => true, 
            'action' => $action,
            'message' => $action === 'added' ? 'Ajouté aux favoris' : 'Retiré des favoris'
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erreur lors de la mise à jour'];
    }
}

function updateTemplateRatingStats($pdo, $template_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT AVG(rating) as avg_rating, COUNT(*) as rating_count
            FROM template_ratings
            WHERE template_id = ?
        ");
        $stmt->execute([$template_id]);
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stmt = $pdo->prepare("
            UPDATE flying_book_templates 
            SET rating = ?, rating_count = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            round($stats['avg_rating'], 2),
            $stats['rating_count'],
            $template_id
        ]);
    } catch (Exception $e) {
        logError("Update rating stats error", ['error' => $e->getMessage(), 'template_id' => $template_id]);
    }
}

function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}

// Créer les tables manquantes si nécessaire
function createTemplatesTables($pdo) {
    try {
        // Table des notes de templates
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS template_ratings (
                id INT AUTO_INCREMENT PRIMARY KEY,
                template_id INT NOT NULL,
                user_id INT NOT NULL,
                rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_user_rating (template_id, user_id),
                FOREIGN KEY (template_id) REFERENCES flying_book_templates(id) ON DELETE CASCADE,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        // Table des favoris de templates
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS template_favorites (
                id INT AUTO_INCREMENT PRIMARY KEY,
                template_id INT NOT NULL,
                user_id INT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY unique_user_favorite (template_id, user_id),
                FOREIGN KEY (template_id) REFERENCES flying_book_templates(id) ON DELETE CASCADE,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    } catch (Exception $e) {
        logError("Create templates tables error", ['error' => $e->getMessage()]);
    }
}

// Créer les tables
createTemplatesTables($pdo);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Templates - Marketplace</title>
    <link rel="stylesheet" href="assets/css/united-fixed.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Styles spécifiques au marketplace */
        .marketplace-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 4rem 2rem;
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .marketplace-header h1 {
            font-size: 3rem;
            margin-bottom: 1rem;
            font-weight: 800;
        }
        
        .marketplace-header p {
            font-size: 1.3rem;
            opacity: 0.9;
            max-width: 600px;
            margin: 0 auto;
        }
        
        .search-section {
            background: white;
            padding: 2rem;
            border-radius: 16px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
        }
        
        .search-controls {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }
        
        .search-input {
            position: relative;
        }
        
        .search-input input {
            width: 100%;
            padding: 1rem 1rem 1rem 3rem;
            border: 2px solid var(--gray-200);
            border-radius: 12px;
            font-size: 1rem;
            transition: all 0.3s ease;
        }
        
        .search-input input:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            outline: none;
        }
        
        .search-input i {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray-400);
            font-size: 1.2rem;
        }
        
        .featured-section {
            margin-bottom: 3rem;
        }
        
        .featured-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 2rem;
            margin-top: 2rem;
        }
        
        .template-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .template-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
        }
        
        .template-preview {
            height: 200px;
            background: linear-gradient(135deg, var(--gray-100), var(--gray-200));
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            overflow: hidden;
        }
        
        .template-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .template-preview i {
            font-size: 4rem;
            color: var(--gray-400);
        }
        
        .template-badges {
            position: absolute;
            top: 1rem;
            left: 1rem;
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .template-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .badge-featured {
            background: linear-gradient(135deg, #fbbf24, #f59e0b);
            color: white;
        }
        
        .badge-new {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
        }
        
        .badge-premium {
            background: linear-gradient(135deg, #8b5cf6, #7c3aed);
            color: white;
        }
        
        .badge-free {
            background: linear-gradient(135deg, #06b6d4, #0891b2);
            color: white;
        }
        
        .template-actions {
            position: absolute;
            top: 1rem;
            right: 1rem;
            display: flex;
            gap: 0.5rem;
        }
        
        .template-action-btn {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.9);
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
        }
        
        .template-action-btn:hover {
            background: white;
            transform: scale(1.1);
        }
        
        .template-action-btn.favorited {
            background: var(--error-color);
            color: white;
        }
        
        .template-info {
            padding: 1.5rem;
        }
        
        .template-title {
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--gray-800);
            margin-bottom: 0.5rem;
        }
        
        .template-description {
            color: var(--gray-600);
            font-size: 0.9rem;
            line-height: 1.4;
            margin-bottom: 1rem;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .template-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            font-size: 0.8rem;
            color: var(--gray-500);
        }
        
        .template-stats {
            display: flex;
            gap: 1rem;
        }
        
        .template-rating {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .template-rating .stars {
            color: #fbbf24;
        }
        
        .template-category {
            background: var(--gray-100);
            color: var(--gray-700);
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .template-footer {
            display: flex;
            gap: 0.5rem;
        }
        
        .template-btn {
            flex: 1;
            padding: 0.75rem 1rem;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .template-btn.primary {
            background: var(--primary-color);
            color: white;
        }
        
        .template-btn.primary:hover {
            background: var(--primary-hover);
            transform: translateY(-1px);
        }
        
        .template-btn.secondary {
            background: var(--gray-100);
            color: var(--gray-700);
        }
        
        .template-btn.secondary:hover {
            background: var(--gray-200);
        }
        
        .plan-warning {
            background: rgba(245, 158, 11, 0.1);
            border: 1px solid rgba(245, 158, 11, 0.3);
            border-radius: 12px;
            padding: 1rem;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .plan-warning i {
            color: var(--warning-color);
            font-size: 1.5rem;
        }
        
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--gray-600);
        }
        
        .empty-state i {
            font-size: 4rem;
            color: var(--gray-300);
            margin-bottom: 2rem;
        }
        
        .loading {
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 4rem;
        }
        
        .loading-spinner {
            width: 40px;
            height: 40px;
            border: 4px solid var(--gray-200);
            border-top: 4px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @media (max-width: 768px) {
            .marketplace-header h1 {
                font-size: 2rem;
            }
            
            .search-controls {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .featured-grid {
                grid-template-columns: 1fr;
            }
            
            .template-stats {
                flex-direction: column;
                gap: 0.5rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link active">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <a href="template-editor.php" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Créer Template
                </a>
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($user_name, 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($user_name) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <!-- Header Marketplace -->
        <div class="marketplace-header">
            <h1><i class="fas fa-store"></i> Templates Marketplace</h1>
            <p>Découvrez des templates professionnels créés par la communauté Flying Book</p>
        </div>

        <div class="container">
            <!-- Plan info -->
            <div class="plan-warning">
                <i class="fas fa-info-circle"></i>
                <div>
                    <strong>Plan <?= ucfirst($userPlan['plan_type']) ?>:</strong>
                    Vous pouvez créer <?= $userPlan['limits']['max_projects'] === -1 ? 'un nombre illimité' : $userPlan['limits']['max_projects'] ?> de projets.
                    <?php if ($userPlan['plan_type'] === 'free'): ?>
                        <a href="upgrade.php" style="color: var(--primary-color); font-weight: 600;">Upgrader pour plus de fonctionnalités</a>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Recherche -->
            <div class="search-section">
                <div class="search-controls">
                    <div class="search-input">
                        <i class="fas fa-search"></i>
                        <input type="text" id="searchInput" placeholder="Rechercher des templates..." />
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Catégorie</label>
                        <select id="categoryFilter" class="form-control">
                            <option value="">Toutes catégories</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?= htmlspecialchars($category['category']) ?>">
                                    <?= ucfirst($category['category']) ?> (<?= $category['count'] ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Trier par</label>
                        <select id="sortFilter" class="form-control">
                            <option value="popular">Populaire</option>
                            <option value="recent">Récent</option>
                            <option value="rating">Mieux noté</option>
                            <option value="name">Nom A-Z</option>
                        </select>
                    </div>
                    
                    <button class="btn btn-primary" onclick="searchTemplates()">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
            </div>
            
            <!-- Templates Mis en Avant -->
            <?php if (!empty($featured_templates)): ?>
            <div class="featured-section">
                <h2><i class="fas fa-star"></i> Templates Mis en Avant</h2>
                <div class="featured-grid">
                    <?php foreach ($featured_templates as $template): ?>
                        <?= renderTemplateCard($template, true) ?>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Tous les Templates -->
            <div class="templates-section">
                <h2><i class="fas fa-layer-group"></i> Tous les Templates</h2>
                <div class="templates-grid" id="templatesGrid">
                    <?php if (empty($templates)): ?>
                        <div class="empty-state">
                            <i class="fas fa-layer-group"></i>
                            <h3>Aucun template disponible</h3>
                            <p>Soyez le premier à créer un template pour la communauté !</p>
                            <a href="template-editor.php" class="btn btn-primary">
                                <i class="fas fa-plus"></i> Créer le premier template
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="featured-grid">
                            <?php foreach ($templates as $template): ?>
                                <?= renderTemplateCard($template) ?>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <!-- Modal de détails template -->
    <div id="templateModal" class="modal">
        <div class="modal-content" style="max-width: 700px;">
            <div class="modal-header">
                <h3 id="modalTitle">Détails du Template</h3>
                <button class="modal-close" onclick="closeTemplateModal()">&times;</button>
            </div>
            <div id="modalContent">
                <!-- Contenu dynamique -->
            </div>
        </div>
    </div>

    <script>
        let currentTemplates = <?= json_encode($templates) ?>;
        let userPlan = <?= json_encode($userPlan) ?>;
        
        // ========== RECHERCHE ET FILTRES ==========
        async function searchTemplates() {
            const search = document.getElementById('searchInput').value;
            const category = document.getElementById('categoryFilter').value;
            const sort = document.getElementById('sortFilter').value;
            
            showLoading();
            
            try {
                const response = await fetch('templates.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=search_templates&search=${encodeURIComponent(search)}&category=${encodeURIComponent(category)}&sort=${sort}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    displayTemplates(result.templates);
                } else {
                    showNotification('Erreur lors de la recherche', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        function displayTemplates(templates) {
            const grid = document.getElementById('templatesGrid');
            
            if (templates.length === 0) {
                grid.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-search"></i>
                        <h3>Aucun template trouvé</h3>
                        <p>Essayez avec d'autres critères de recherche</p>
                    </div>
                `;
                return;
            }
            
            const templatesHTML = templates.map(template => 
                renderTemplateCardJS(template)
            ).join('');
            
            grid.innerHTML = `<div class="featured-grid">${templatesHTML}</div>`;
        }
        
        function renderTemplateCardJS(template) {
            const badges = [];
            if (template.is_featured) badges.push('<span class="template-badge badge-featured">Featured</span>');
            if (template.is_premium) badges.push('<span class="template-badge badge-premium">Premium</span>');
            else badges.push('<span class="template-badge badge-free">Gratuit</span>');
            
            const stars = generateStars(template.rating || 0);
            
            return `
                <div class="template-card">
                    <div class="template-preview">
                        <i class="fas fa-presentation"></i>
                        <div class="template-badges">
                            ${badges.join('')}
                        </div>
                        <div class="template-actions">
                            <button class="template-action-btn" onclick="toggleFavorite(${template.id})" title="Favoris">
                                <i class="fas fa-heart"></i>
                            </button>
                            <button class="template-action-btn" onclick="showTemplateDetails(${template.id})" title="Détails">
                                <i class="fas fa-info"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="template-info">
                        <h3 class="template-title">${template.template_name}</h3>
                        <p class="template-description">${template.template_description || 'Aucune description'}</p>
                        
                        <div class="template-meta">
                            <div class="template-stats">
                                <div class="template-rating">
                                    <span class="stars">${stars}</span>
                                    <span>${template.rating || 0}/5</span>
                                </div>
                                <div>
                                    <i class="fas fa-download"></i> ${template.usage_count || 0}
                                </div>
                            </div>
                            <div class="template-category">${template.category}</div>
                        </div>
                        
                        <div class="template-footer">
                            <button class="template-btn primary" onclick="useTemplate(${template.id}, '${template.template_name}')">
                                <i class="fas fa-rocket"></i> Utiliser
                            </button>
                            <button class="template-btn secondary" onclick="showTemplateDetails(${template.id})">
                                <i class="fas fa-eye"></i> Voir
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // ========== ACTIONS TEMPLATES ==========
        async function useTemplate(templateId, templateName) {
            const projectName = prompt(`Nom du nouveau projet basé sur "${templateName}":`, `Mon ${templateName}`);
            if (!projectName) return;
            
            try {
                const response = await fetch('templates.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=use_template&template_id=${templateId}&project_name=${encodeURIComponent(projectName)}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    if (result.redirect) {
                        setTimeout(() => {
                            window.location.href = result.redirect;
                        }, 1500);
                    }
                } else {
                    if (result.upgrade_needed) {
                        showUpgradeModal();
                    } else {
                        showNotification(result.message, 'error');
                    }
                }
            } catch (error) {
                showNotification('Erreur lors de la création du projet', 'error');
            }
        }
        
        async function showTemplateDetails(templateId) {
            try {
                const response = await fetch('templates.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=get_template_details&template_id=${templateId}`
                });
                
                const result = await response.json();
                
                if (result.success && result.template) {
                    displayTemplateModal(result.template);
                } else {
                    showNotification('Template non trouvé', 'error');
                }
            } catch (error) {
                showNotification('Erreur lors du chargement', 'error');
            }
        }
        
        function displayTemplateModal(template) {
            document.getElementById('modalTitle').textContent = template.template_name;
            
            const slidesPreview = template.config_decoded && template.config_decoded.slides ? 
                template.config_decoded.slides.map((slide, index) => `
                    <div style="padding: 0.5rem; background: var(--gray-50); border-radius: 6px; margin-bottom: 0.5rem;">
                        <strong>Slide ${index + 1}:</strong> ${slide.title || 'Sans titre'}
                        <div style="font-size: 0.8rem; color: var(--gray-600);">${slide.type} - ${slide.content ? slide.content.substring(0, 100) + '...' : 'Pas de contenu'}</div>
                    </div>
                `).join('') : '<p>Aperçu non disponible</p>';
            
            document.getElementById('modalContent').innerHTML = `
                <div style="margin-bottom: 2rem;">
                    <p><strong>Description:</strong> ${template.template_description || 'Aucune description'}</p>
                    <p><strong>Catégorie:</strong> ${template.category}</p>
                    <p><strong>Créé par:</strong> ${template.creator_name || 'Anonyme'}</p>
                    <p><strong>Utilisations:</strong> ${template.usage_count || 0}</p>
                    <p><strong>Note:</strong> ${generateStars(template.rating || 0)} ${template.rating || 0}/5 (${template.rating_count || 0} votes)</p>
                </div>
                
                <div style="margin-bottom: 2rem;">
                    <h4>Aperçu des slides:</h4>
                    <div style="max-height: 200px; overflow-y: auto; border: 1px solid var(--gray-200); border-radius: 8px; padding: 1rem;">
                        ${slidesPreview}
                    </div>
                </div>
                
                <div style="display: flex; gap: 1rem;">
                    <button class="btn btn-primary" onclick="useTemplate(${template.id}, '${template.template_name}'); closeTemplateModal();">
                        <i class="fas fa-rocket"></i> Utiliser ce template
                    </button>
                    <button class="btn btn-secondary" onclick="closeTemplateModal()">
                        Fermer
                    </button>
                </div>
                
                <div style="margin-top: 2rem; padding-top: 1rem; border-top: 1px solid var(--gray-200);">
                    <h4>Noter ce template:</h4>
                    <div style="display: flex; gap: 0.5rem; margin-top: 0.5rem;">
                        ${[1,2,3,4,5].map(rating => `
                            <button onclick="rateTemplate(${template.id}, ${rating})" 
                                    style="background: none; border: none; font-size: 1.5rem; cursor: pointer; color: var(--gray-300);"
                                    onmouseover="highlightStars(this, ${rating})"
                                    onmouseout="resetStars(this)">
                                <i class="fas fa-star"></i>
                            </button>
                        `).join('')}
                    </div>
                </div>
            `;
            
            document.getElementById('templateModal').style.display = 'flex';
        }
        
        function closeTemplateModal() {
            document.getElementById('templateModal').style.display = 'none';
        }
        
        async function toggleFavorite(templateId) {
            try {
                const response = await fetch('templates.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=favorite_template&template_id=${templateId}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    // Mettre à jour l'icône
                    const btn = event.target.closest('.template-action-btn');
                    if (result.action === 'added') {
                        btn.classList.add('favorited');
                    } else {
                        btn.classList.remove('favorited');
                    }
                }
            } catch (error) {
                showNotification('Erreur lors de la mise à jour', 'error');
            }
        }
        
        async function rateTemplate(templateId, rating) {
            try {
                const response = await fetch('templates.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=rate_template&template_id=${templateId}&rating=${rating}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    // Refresh modal data
                    setTimeout(() => showTemplateDetails(templateId), 1000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la notation', 'error');
            }
        }
        
        // ========== UTILITAIRES ==========
        function generateStars(rating) {
            const fullStars = Math.floor(rating);
            const hasHalfStar = rating % 1 >= 0.5;
            let stars = '';
            
            for (let i = 0; i < fullStars; i++) {
                stars += '<i class="fas fa-star"></i>';
            }
            
            if (hasHalfStar) {
                stars += '<i class="fas fa-star-half-alt"></i>';
            }
            
            const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);
            for (let i = 0; i < emptyStars; i++) {
                stars += '<i class="far fa-star"></i>';
            }
            
            return stars;
        }
        
        function highlightStars(element, rating) {
            const stars = element.parentElement.querySelectorAll('button');
            stars.forEach((star, index) => {
                star.style.color = index < rating ? '#fbbf24' : 'var(--gray-300)';
            });
        }
        
        function resetStars(element) {
            const stars = element.parentElement.querySelectorAll('button');
            stars.forEach(star => {
                star.style.color = 'var(--gray-300)';
            });
        }
        
        function showLoading() {
            document.getElementById('templatesGrid').innerHTML = `
                <div class="loading">
                    <div class="loading-spinner"></div>
                </div>
            `;
        }
        
        function showUpgradeModal() {
            const modal = document.createElement('div');
            modal.className = 'modal';
            modal.style.display = 'flex';
            modal.innerHTML = `
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>Limite atteinte</h3>
                        <button class="modal-close" onclick="this.closest('.modal').remove()">&times;</button>
                    </div>
                    <div style="padding: 2rem; text-align: center;">
                        <i class="fas fa-rocket" style="font-size: 3rem; color: var(--primary-color); margin-bottom: 1rem;"></i>
                        <h3>Upgradez votre plan</h3>
                        <p>Vous avez atteint la limite de projets de votre plan gratuit.</p>
                        <div style="margin-top: 2rem;">
                            <a href="upgrade.php" class="btn btn-primary">
                                <i class="fas fa-crown"></i> Upgrader maintenant
                            </a>
                            <button class="btn btn-secondary" onclick="this.closest('.modal').remove()">
                                Plus tard
                            </button>
                        </div>
                    </div>
                </div>
            `;
            document.body.appendChild(modal);
        }
        
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `alert alert-${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                ${message}
            `;
            notification.style.cssText = `
                position: fixed; top: 2rem; right: 2rem; z-index: 3000; max-width: 400px;
                animation: slideIn 0.3s ease;
            `;
            
            document.body.appendChild(notification);
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease';
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            dropdown.style.display = dropdown.style.display === 'block' ? 'none' : 'block';
        }
        
        // ========== EVENT LISTENERS ==========
        document.addEventListener('DOMContentLoaded', function() {
            // Recherche en temps réel
            let searchTimeout;
            document.getElementById('searchInput').addEventListener('input', function() {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(searchTemplates, 500);
            });
            
            // Filtres
            document.getElementById('categoryFilter').addEventListener('change', searchTemplates);
            document.getElementById('sortFilter').addEventListener('change', searchTemplates);
            
            // Fermer modals en cliquant dehors
            document.addEventListener('click', function(e) {
                if (e.target.classList.contains('modal')) {
                    e.target.style.display = 'none';
                }
                
                if (!e.target.closest('.user-menu')) {
                    document.getElementById('userDropdown').style.display = 'none';
                }
            });
            
            showNotification('Marketplace Templates chargé ! 🏪', 'success');
        });
    </script>
</body>
</html>

<?php
// Fonction pour rendre une carte template en PHP
function renderTemplateCard($template, $featured = false) {
    $badges = [];
    if ($template['is_featured']) $badges[] = '<span class="template-badge badge-featured">Featured</span>';
    if ($template['is_premium']) $badges[] = '<span class="template-badge badge-premium">Premium</span>';
    else $badges[] = '<span class="template-badge badge-free">Gratuit</span>';
    
    $stars = '';
    $rating = $template['rating'] ?: 0;
    $fullStars = floor($rating);
    $hasHalfStar = ($rating % 1) >= 0.5;
    
    for ($i = 0; $i < $fullStars; $i++) {
        $stars .= '<i class="fas fa-star"></i>';
    }
    if ($hasHalfStar) $stars .= '<i class="fas fa-star-half-alt"></i>';
    $emptyStars = 5 - $fullStars - ($hasHalfStar ? 1 : 0);
    for ($i = 0; $i < $emptyStars; $i++) {
        $stars .= '<i class="far fa-star"></i>';
    }
    
    return '
        <div class="template-card">
            <div class="template-preview">
                <i class="fas fa-presentation"></i>
                <div class="template-badges">
                    ' . implode('', $badges) . '
                </div>
                <div class="template-actions">
                    <button class="template-action-btn" onclick="toggleFavorite(' . $template['id'] . ')" title="Favoris">
                        <i class="fas fa-heart"></i>
                    </button>
                    <button class="template-action-btn" onclick="showTemplateDetails(' . $template['id'] . ')" title="Détails">
                        <i class="fas fa-info"></i>
                    </button>
                </div>
            </div>
            
            <div class="template-info">
                <h3 class="template-title">' . htmlspecialchars($template['template_name']) . '</h3>
                <p class="template-description">' . htmlspecialchars($template['template_description'] ?: 'Aucune description') . '</p>
                
                <div class="template-meta">
                    <div class="template-stats">
                        <div class="template-rating">
                            <span class="stars">' . $stars . '</span>
                            <span>' . $rating . '/5</span>
                        </div>
                        <div>
                            <i class="fas fa-download"></i> ' . ($template['usage_count'] ?: 0) . '
                        </div>
                    </div>
                    <div class="template-category">' . htmlspecialchars($template['category']) . '</div>
                </div>
                
                <div class="template-footer">
                    <button class="template-btn primary" onclick="useTemplate(' . $template['id'] . ', \'' . htmlspecialchars($template['template_name']) . '\')">
                        <i class="fas fa-rocket"></i> Utiliser
                    </button>
                    <button class="template-btn secondary" onclick="showTemplateDetails(' . $template['id'] . ')">
                        <i class="fas fa-eye"></i> Voir
                    </button>
                </div>
            </div>
        </div>
    ';
}
?>