<?php
session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

$message = '';
$error = '';

// Configuration de base pour l'éditeur
$editor_config = [
    'themes' => [
        'moderne' => 'Moderne',
        'classique' => 'Classique', 
        'corporate' => 'Corporate',
        'minimal' => 'Minimal',
        'premium' => 'Premium'
    ],
    'transitions' => [
        'fade' => 'Fondu',
        'slide' => 'Glissement',
        'flip' => 'Retournement',
        '3d' => 'Effet 3D',
        'premium_3d' => 'Premium 3D'
    ],
    'categories' => [
        'business' => 'Business',
        'education' => 'Éducation',
        'portfolio' => 'Portfolio',
        'marketing' => 'Marketing',
        'sales' => 'Commercial'
    ]
];

try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['ajax_action']) {
            case 'save_template':
                $template_name = trim($_POST['template_name'] ?? '');
                $template_description = trim($_POST['template_description'] ?? '');
                $category = $_POST['category'] ?? 'custom';
                $theme = $_POST['theme'] ?? 'moderne';
                $transition = $_POST['transition'] ?? 'fade';
                $is_public = isset($_POST['is_public']) ? 1 : 0;
                
                if (empty($template_name)) {
                    echo json_encode(['success' => false, 'message' => 'Nom du template requis']);
                    exit;
                }
                
                // Configuration du template
                $template_config = [
                    'theme' => $theme,
                    'transition' => $transition,
                    'slides' => json_decode($_POST['slides_data'] ?? '[]', true),
                    'settings' => json_decode($_POST['template_settings'] ?? '{}', true)
                ];
                
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_templates 
                    (user_id, template_name, template_description, category, template_config, 
                     is_public, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, NOW())
                ");
                
                if ($stmt->execute([$user_id, $template_name, $template_description, $category, 
                                   json_encode($template_config), $is_public])) {
                    $template_id = $pdo->lastInsertId();
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Template sauvegardé avec succès',
                        'template_id' => $template_id
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la sauvegarde']);
                }
                exit;
                
            case 'preview_template':
                // Générer un aperçu du template
                $preview_data = [
                    'theme' => $_POST['theme'] ?? 'moderne',
                    'transition' => $_POST['transition'] ?? 'fade',
                    'slides_count' => count(json_decode($_POST['slides_data'] ?? '[]', true))
                ];
                
                echo json_encode([
                    'success' => true, 
                    'preview_url' => 'template-preview.php?data=' . base64_encode(json_encode($preview_data))
                ]);
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement de l'éditeur: " . $e->getMessage();
    error_log("Template editor error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Éditeur de Templates</title>
    
    <!-- CSS UNIFIÉ -->
    <link rel="stylesheet" href="assets/css/united.css">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- Styles spécifiques éditeur -->
    <style>
        .editor-container {
            display: grid;
            grid-template-columns: 300px 1fr 300px;
            gap: var(--space-lg);
            height: calc(100vh - 90px);
            margin-top: var(--space-lg);
        }
        
        .editor-sidebar {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-lg);
            box-shadow: var(--shadow-sm);
            overflow-y: auto;
            border: 1px solid var(--border-light);
        }
        
        .editor-main {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            display: flex;
            flex-direction: column;
            box-shadow: var(--shadow-md);
            border: 2px solid var(--border-color);
        }
        
        .editor-toolbar {
            padding: var(--space-md);
            border-bottom: 2px solid var(--border-light);
            background: var(--bg-secondary);
            border-radius: var(--radius-lg) var(--radius-lg) 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .editor-canvas {
            flex: 1;
            padding: var(--space-xl);
            background: linear-gradient(45deg, var(--bg-secondary) 25%, transparent 25%),
                        linear-gradient(-45deg, var(--bg-secondary) 25%, transparent 25%),
                        linear-gradient(45deg, transparent 75%, var(--bg-secondary) 75%),
                        linear-gradient(-45deg, transparent 75%, var(--bg-secondary) 75%);
            background-size: 20px 20px;
            background-position: 0 0, 0 10px, 10px -10px, -10px 0px;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
        }
        
        .canvas-slide {
            width: 800px;
            height: 600px;
            background: var(--bg-primary);
            border-radius: var(--radius-md);
            box-shadow: var(--shadow-xl);
            border: 3px solid var(--border-color);
            position: relative;
            overflow: hidden;
        }
        
        .slide-content {
            padding: var(--space-xl);
            height: 100%;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            text-align: center;
            color: var(--text-primary);
        }
        
        .sidebar-section {
            margin-bottom: var(--space-xl);
        }
        
        .sidebar-section h3 {
            color: var(--text-primary);
            margin-bottom: var(--space-md);
            font-size: var(--font-size-lg);
            display: flex;
            align-items: center;
            gap: var(--space-sm);
        }
        
        .tools-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--space-sm);
        }
        
        .tool-btn {
            padding: var(--space-md);
            background: var(--bg-secondary);
            border: 2px solid var(--border-light);
            border-radius: var(--radius-sm);
            cursor: pointer;
            transition: var(--transition-normal);
            text-align: center;
            color: var(--text-primary);
        }
        
        .tool-btn:hover {
            border-color: var(--border-color);
            background: var(--bg-tertiary);
            transform: translateY(-2px);
        }
        
        .tool-btn.active {
            border-color: var(--accent-color);
            background: var(--bg-tertiary);
        }
        
        .tool-btn i {
            display: block;
            font-size: var(--font-size-xl);
            margin-bottom: var(--space-xs);
            color: var(--accent-color);
        }
        
        .properties-panel {
            display: none;
        }
        
        .properties-panel.active {
            display: block;
        }
        
        .property-group {
            margin-bottom: var(--space-lg);
            padding: var(--space-md);
            background: var(--bg-secondary);
            border-radius: var(--radius-sm);
            border: 1px solid var(--border-light);
        }
        
        .property-group h4 {
            color: var(--text-primary);
            margin-bottom: var(--space-sm);
            font-size: var(--font-size-base);
        }
        
        .color-picker {
            display: grid;
            grid-template-columns: repeat(5, 1fr);
            gap: var(--space-xs);
            margin-top: var(--space-sm);
        }
        
        .color-swatch {
            width: 30px;
            height: 30px;
            border-radius: var(--radius-sm);
            cursor: pointer;
            border: 2px solid var(--border-light);
            transition: var(--transition-fast);
        }
        
        .color-swatch:hover {
            transform: scale(1.1);
            border-color: var(--border-color);
        }
        
        .slides-list {
            max-height: 300px;
            overflow-y: auto;
        }
        
        .slide-item {
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            padding: var(--space-sm);
            margin-bottom: var(--space-xs);
            background: var(--bg-secondary);
            border-radius: var(--radius-sm);
            cursor: pointer;
            transition: var(--transition-normal);
            border: 1px solid var(--border-light);
        }
        
        .slide-item:hover {
            background: var(--bg-tertiary);
            border-color: var(--border-color);
        }
        
        .slide-item.active {
            border-color: var(--accent-color);
            background: var(--bg-tertiary);
        }
        
        .slide-thumbnail {
            width: 40px;
            height: 30px;
            background: var(--bg-tertiary);
            border-radius: var(--radius-xs);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
        }
        
        .toolbar-group {
            display: flex;
            gap: var(--space-sm);
            align-items: center;
        }
        
        .zoom-controls {
            display: flex;
            align-items: center;
            gap: var(--space-xs);
            background: var(--bg-primary);
            padding: var(--space-xs);
            border-radius: var(--radius-sm);
            border: 1px solid var(--border-light);
        }
        
        .zoom-btn {
            width: 30px;
            height: 30px;
            border: none;
            background: transparent;
            cursor: pointer;
            border-radius: var(--radius-xs);
            color: var(--text-primary);
            transition: var(--transition-fast);
        }
        
        .zoom-btn:hover {
            background: var(--bg-secondary);
        }
        
        @media (max-width: 1200px) {
            .editor-container {
                grid-template-columns: 1fr;
                height: auto;
            }
            
            .editor-sidebar {
                order: 2;
            }
            
            .canvas-slide {
                width: 100%;
                max-width: 600px;
                height: 450px;
            }
        }
        
        @media (max-width: 768px) {
            .canvas-slide {
                width: 100%;
                height: 300px;
            }
            
            .tools-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <a href="dashboard.php" class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </a>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="template-editor.php" class="nav-link active">
                        <i class="fas fa-edit"></i> Éditeur
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <button class="btn btn-outline" onclick="previewTemplate()">
                    <i class="fas fa-eye"></i> Aperçu
                </button>
                <button class="btn btn-primary" onclick="saveTemplate()">
                    <i class="fas fa-save"></i> Sauvegarder
                </button>
                
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($user_name, 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($user_name) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Éditeur de templates -->
            <div class="editor-container">
                <!-- Sidebar gauche - Outils -->
                <div class="editor-sidebar">
                    <!-- Outils -->
                    <div class="sidebar-section">
                        <h3><i class="fas fa-toolbox"></i> Outils</h3>
                        <div class="tools-grid">
                            <div class="tool-btn active" data-tool="text" onclick="selectTool('text')">
                                <i class="fas fa-font"></i>
                                <span>Texte</span>
                            </div>
                            <div class="tool-btn" data-tool="image" onclick="selectTool('image')">
                                <i class="fas fa-image"></i>
                                <span>Image</span>
                            </div>
                            <div class="tool-btn" data-tool="shape" onclick="selectTool('shape')">
                                <i class="fas fa-shapes"></i>
                                <span>Forme</span>
                            </div>
                            <div class="tool-btn" data-tool="video" onclick="selectTool('video')">
                                <i class="fas fa-video"></i>
                                <span>Vidéo</span>
                            </div>
                        </div>
                    </div>

                    <!-- Slides -->
                    <div class="sidebar-section">
                        <h3><i class="fas fa-clone"></i> Slides</h3>
                        <div class="slides-list" id="slidesList">
                            <div class="slide-item active" data-slide="1">
                                <div class="slide-thumbnail">1</div>
                                <span>Slide 1</span>
                            </div>
                        </div>
                        <button class="btn btn-outline" style="width: 100%; margin-top: var(--space-md);" onclick="addSlide()">
                            <i class="fas fa-plus"></i> Ajouter slide
                        </button>
                    </div>
                </div>

                <!-- Zone principale - Canvas -->
                <div class="editor-main">
                    <div class="editor-toolbar">
                        <div class="toolbar-group">
                            <span style="color: var(--text-primary); font-weight: 600;">
                                <i class="fas fa-edit"></i> Slide 1
                            </span>
                        </div>
                        
                        <div class="toolbar-group">
                            <div class="zoom-controls">
                                <button class="zoom-btn" onclick="zoomOut()">
                                    <i class="fas fa-search-minus"></i>
                                </button>
                                <span style="color: var(--text-primary); font-size: var(--font-size-sm);">100%</span>
                                <button class="zoom-btn" onclick="zoomIn()">
                                    <i class="fas fa-search-plus"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="editor-canvas" id="editorCanvas">
                        <div class="canvas-slide" id="currentSlide">
                            <div class="slide-content">
                                <h1 style="color: var(--text-primary); margin-bottom: var(--space-lg);">
                                    Titre de votre slide
                                </h1>
                                <p style="color: var(--text-secondary); font-size: var(--font-size-lg);">
                                    Cliquez pour éditer le contenu
                                </p>
                                <button class="btn btn-primary" style="margin-top: var(--space-xl);">
                                    Action principale
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Sidebar droite - Propriétés -->
                <div class="editor-sidebar">
                    <!-- Propriétés générales -->
                    <div class="sidebar-section">
                        <h3><i class="fas fa-cog"></i> Template</h3>
                        
                        <div class="form-group">
                            <label class="form-label">Nom du template</label>
                            <input type="text" class="form-control" id="templateName" placeholder="Mon template">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Thème</label>
                            <select class="form-control" id="templateTheme">
                                <?php foreach ($editor_config['themes'] as $key => $name): ?>
                                    <option value="<?= $key ?>"><?= $name ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Transition</label>
                            <select class="form-control" id="templateTransition">
                                <?php foreach ($editor_config['transitions'] as $key => $name): ?>
                                    <option value="<?= $key ?>"><?= $name ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <!-- Propriétés de l'élément sélectionné -->
                    <div id="textProperties" class="properties-panel active">
                        <div class="sidebar-section">
                            <h3><i class="fas fa-font"></i> Texte</h3>
                            
                            <div class="property-group">
                                <h4>Police</h4>
                                <select class="form-control">
                                    <option>Inter</option>
                                    <option>Arial</option>
                                    <option>Helvetica</option>
                                    <option>Georgia</option>
                                </select>
                            </div>
                            
                            <div class="property-group">
                                <h4>Couleurs</h4>
                                <div class="color-picker">
                                    <div class="color-swatch" style="background: var(--text-primary);" onclick="selectColor('#111827')"></div>
                                    <div class="color-swatch" style="background: var(--accent-color);" onclick="selectColor('#c2410c')"></div>
                                    <div class="color-swatch" style="background: var(--primary-color);" onclick="selectColor('#4338ca')"></div>
                                    <div class="color-swatch" style="background: var(--success-color);" onclick="selectColor('#047857')"></div>
                                    <div class="color-swatch" style="background: var(--warning-color);" onclick="selectColor('#d97706')"></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div id="imageProperties" class="properties-panel">
                        <div class="sidebar-section">
                            <h3><i class="fas fa-image"></i> Image</h3>
                            
                            <div class="property-group">
                                <button class="btn btn-outline" style="width: 100%;">
                                    <i class="fas fa-upload"></i> Changer l'image
                                </button>
                            </div>
                            
                            <div class="property-group">
                                <h4>Taille</h4>
                                <div class="form-row">
                                    <input type="number" class="form-control" placeholder="Largeur">
                                    <input type="number" class="form-control" placeholder="Hauteur">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Modal de sauvegarde -->
    <div id="saveTemplateModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-save"></i> Sauvegarder le template</h3>
                <button class="modal-close" onclick="closeModal('saveTemplateModal')">&times;</button>
            </div>
            
            <div class="modal-body">
                <form onsubmit="submitTemplate(event)">
                    <div class="form-group">
                        <label class="form-label">Nom du template *</label>
                        <input type="text" name="template_name" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Description</label>
                        <textarea name="template_description" class="form-control" rows="3" 
                                  placeholder="Décrivez votre template..."></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Catégorie</label>
                        <select name="category" class="form-control">
                            <?php foreach ($editor_config['categories'] as $key => $name): ?>
                                <option value="<?= $key ?>"><?= $name ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label style="display: flex; align-items: center; gap: var(--space-sm); cursor: pointer;">
                            <input type="checkbox" name="is_public">
                            <span>Rendre ce template public</span>
                        </label>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" onclick="closeModal('saveTemplateModal')">
                            Annuler
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            Sauvegarder
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script>
        // ========== VARIABLES GLOBALES ==========
        let currentTool = 'text';
        let currentSlide = 1;
        let slides = [
            {
                id: 1,
                name: 'Slide 1',
                content: {},
                elements: []
            }
        ];
        let zoomLevel = 100;
        
        // ========== GESTION DES OUTILS ==========
        function selectTool(tool) {
            currentTool = tool;
            
            // Mise à jour interface
            document.querySelectorAll('.tool-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            document.querySelector(`[data-tool="${tool}"]`).classList.add('active');
            
            // Afficher le panneau de propriétés correspondant
            document.querySelectorAll('.properties-panel').forEach(panel => {
                panel.classList.remove('active');
            });
            
            const panelId = tool + 'Properties';
            const panel = document.getElementById(panelId);
            if (panel) {
                panel.classList.add('active');
            }
            
            showNotification(`Outil ${tool} sélectionné`, 'info');
        }
        
        // ========== GESTION DES SLIDES ==========
        function addSlide() {
            const newSlideId = slides.length + 1;
            slides.push({
                id: newSlideId,
                name: `Slide ${newSlideId}`,
                content: {},
                elements: []
            });
            
            updateSlidesList();
            selectSlide(newSlideId);
            showNotification('Nouvelle slide ajoutée', 'success');
        }
        
        function selectSlide(slideId) {
            currentSlide = slideId;
            
            // Mise à jour interface
            document.querySelectorAll('.slide-item').forEach(item => {
                item.classList.remove('active');
            });
            document.querySelector(`[data-slide="${slideId}"]`).classList.add('active');
            
            // Charger le contenu de la slide
            loadSlideContent(slideId);
        }
        
        function updateSlidesList() {
            const slidesList = document.getElementById('slidesList');
            slidesList.innerHTML = '';
            
            slides.forEach(slide => {
                const slideItem = document.createElement('div');
                slideItem.className = `slide-item ${slide.id === currentSlide ? 'active' : ''}`;
                slideItem.setAttribute('data-slide', slide.id);
                slideItem.onclick = () => selectSlide(slide.id);
                
                slideItem.innerHTML = `
                    <div class="slide-thumbnail">${slide.id}</div>
                    <span>${slide.name}</span>
                `;
                
                slidesList.appendChild(slideItem);
            });
        }
        
        function loadSlideContent(slideId) {
            // Simuler le chargement du contenu
            console.log(`Chargement slide ${slideId}`);
        }
        
        // ========== ZOOM ==========
        function zoomIn() {
            if (zoomLevel < 200) {
                zoomLevel += 25;
                updateZoom();
            }
        }
        
        function zoomOut() {
            if (zoomLevel > 25) {
                zoomLevel -= 25;
                updateZoom();
            }
        }
        
        function updateZoom() {
            const canvas = document.getElementById('currentSlide');
            canvas.style.transform = `scale(${zoomLevel / 100})`;
            document.querySelector('.zoom-controls span').textContent = `${zoomLevel}%`;
        }
        
        // ========== COULEURS ==========
        function selectColor(color) {
            console.log(`Couleur sélectionnée: ${color}`);
            showNotification(`Couleur ${color} appliquée`, 'success');
        }
        
        // ========== SAUVEGARDE ==========
        function saveTemplate() {
            document.getElementById('saveTemplateModal').style.display = 'block';
        }
        
        async function submitTemplate(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            
            // Ajouter les données du template
            formData.append('ajax_action', 'save_template');
            formData.append('theme', document.getElementById('templateTheme').value);
            formData.append('transition', document.getElementById('templateTransition').value);
            formData.append('slides_data', JSON.stringify(slides));
            formData.append('template_settings', JSON.stringify({
                zoom: zoomLevel,
                current_slide: currentSlide
            }));
            
            try {
                showNotification('Sauvegarde en cours...', 'info');
                
                const response = await fetch('template-editor.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    closeModal('saveTemplateModal');
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la sauvegarde', 'error');
            }
        }
        
        // ========== APERÇU ==========
        async function previewTemplate() {
            try {
                const response = await fetch('template-editor.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=preview_template&theme=${document.getElementById('templateTheme').value}&transition=${document.getElementById('templateTransition').value}&slides_data=${JSON.stringify(slides)}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    window.open(result.preview_url, '_blank');
                } else {
                    showNotification('Erreur lors de la génération de l\'aperçu', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        // ========== UTILITAIRES ==========
        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }
        
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les modals en cliquant à l'extérieur
        window.onclick = function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.style.display = 'none';
            }
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // ========== SYSTÈME DE NOTIFICATIONS ==========
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.75rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.2rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.8;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        // ========== INITIALISATION ==========
        document.addEventListener('DOMContentLoaded', function() {
            updateSlidesList();
            showNotification('Éditeur de templates chargé ! 🎨', 'success');
        });
        
        // Raccourcis clavier
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey || e.metaKey) {
                switch(e.key) {
                    case 's':
                        e.preventDefault();
                        saveTemplate();
                        break;
                    case 'z':
                        e.preventDefault();
                        // Undo
                        showNotification('Annulation (Ctrl+Z)', 'info');
                        break;
                    case 'y':
                        e.preventDefault();
                        // Redo
                        showNotification('Rétablir (Ctrl+Y)', 'info');
                        break;
                }
            }
        });
    </script>
</body>
</html>