<?php
/**
 * Gestionnaire d'Abonnements Flying Book Creator v2.0
 * Gestion des plans freemium, premium et enterprise
 */

class FlyingBookSubscriptionManager
{
    private $pdo;
    
    // Définition des limites par plan
    const FLYING_BOOK_LIMITS = [
        'free' => [
            'max_projects' => 3,
            'max_slides_per_project' => 10,
            'max_storage' => 100 * 1024 * 1024, // 100MB
            'max_file_size' => 10 * 1024 * 1024, // 10MB
            'premium_themes' => false,
            'premium_transitions' => false,
            'custom_domain' => false,
            'analytics_detailed' => false,
            'remove_watermark' => false,
            'priority_support' => false,
            'api_access' => false,
            'collaboration' => false,
            'export_formats' => ['html'],
            'max_views_per_month' => 1000
        ],
        'pro' => [
            'max_projects' => 25,
            'max_slides_per_project' => 50,
            'max_storage' => 5 * 1024 * 1024 * 1024, // 5GB
            'max_file_size' => 100 * 1024 * 1024, // 100MB
            'premium_themes' => true,
            'premium_transitions' => true,
            'custom_domain' => false,
            'analytics_detailed' => true,
            'remove_watermark' => true,
            'priority_support' => false,
            'api_access' => false,
            'collaboration' => true,
            'export_formats' => ['html', 'pdf', 'video'],
            'max_views_per_month' => 10000
        ],
        'business' => [
            'max_projects' => 100,
            'max_slides_per_project' => 200,
            'max_storage' => 50 * 1024 * 1024 * 1024, // 50GB
            'max_file_size' => 500 * 1024 * 1024, // 500MB
            'premium_themes' => true,
            'premium_transitions' => true,
            'custom_domain' => true,
            'analytics_detailed' => true,
            'remove_watermark' => true,
            'priority_support' => true,
            'api_access' => true,
            'collaboration' => true,
            'export_formats' => ['html', 'pdf', 'video', 'scorm'],
            'max_views_per_month' => 100000
        ],
        'enterprise' => [
            'max_projects' => -1, // Illimité
            'max_slides_per_project' => -1,
            'max_storage' => -1,
            'max_file_size' => 2 * 1024 * 1024 * 1024, // 2GB
            'premium_themes' => true,
            'premium_transitions' => true,
            'custom_domain' => true,
            'analytics_detailed' => true,
            'remove_watermark' => true,
            'priority_support' => true,
            'api_access' => true,
            'collaboration' => true,
            'export_formats' => ['html', 'pdf', 'video', 'scorm', 'pptx'],
            'max_views_per_month' => -1,
            'white_label' => true,
            'custom_branding' => true,
            'dedicated_support' => true
        ]
    ];
    
    const PLAN_PRICES = [
        'free' => ['monthly' => 0, 'yearly' => 0],
        'pro' => ['monthly' => 19.99, 'yearly' => 199.99],
        'business' => ['monthly' => 49.99, 'yearly' => 499.99],
        'enterprise' => ['monthly' => 149.99, 'yearly' => 1499.99]
    ];
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Obtenir le plan actuel d'un utilisateur
     */
    public function getUserPlan($userId) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT s.*, u.is_premium
                FROM flying_book_subscriptions s
                RIGHT JOIN users u ON s.user_id = u.id
                WHERE u.id = ? AND (s.status = 'active' OR s.status IS NULL)
                ORDER BY s.created_at DESC
                LIMIT 1
            ");
            $stmt->execute([$userId]);
            $subscription = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$subscription || !$subscription['plan_type']) {
                return [
                    'plan_type' => 'free',
                    'plan_name' => 'Gratuit',
                    'status' => 'active',
                    'limits' => self::FLYING_BOOK_LIMITS['free'],
                    'features' => $this->getPlanFeatures('free'),
                    'expires_at' => null,
                    'is_trial' => false
                ];
            }
            
            $planType = $subscription['plan_type'];
            
            return [
                'plan_type' => $planType,
                'plan_name' => $subscription['plan_name'],
                'status' => $subscription['status'],
                'billing_cycle' => $subscription['billing_cycle'],
                'current_period_end' => $subscription['current_period_end'],
                'limits' => self::FLYING_BOOK_LIMITS[$planType],
                'features' => json_decode($subscription['features'] ?? '{}', true) ?: $this->getPlanFeatures($planType),
                'is_trial' => $subscription['trial_ends_at'] && strtotime($subscription['trial_ends_at']) > time(),
                'trial_ends_at' => $subscription['trial_ends_at']
            ];
            
        } catch (Exception $e) {
            // En cas d'erreur, retourner le plan gratuit
            return [
                'plan_type' => 'free',
                'plan_name' => 'Gratuit',
                'status' => 'active',
                'limits' => self::FLYING_BOOK_LIMITS['free'],
                'features' => $this->getPlanFeatures('free'),
                'expires_at' => null,
                'is_trial' => false
            ];
        }
    }
    
    /**
     * Vérifier si un utilisateur peut effectuer une action
     */
    public function checkUserLimit($userId, $limitType, $requestedAmount = 1) {
        $userPlan = $this->getUserPlan($userId);
        $limits = $userPlan['limits'];
        
        // Si la limite est illimitée (-1), autoriser
        if ($limits[$limitType] === -1) {
            return ['allowed' => true, 'remaining' => -1];
        }
        
        // Obtenir l'utilisation actuelle
        $currentUsage = $this->getCurrentUsage($userId, $limitType);
        
        $remaining = $limits[$limitType] - $currentUsage;
        $allowed = $remaining >= $requestedAmount;
        
        return [
            'allowed' => $allowed,
            'current' => $currentUsage,
            'limit' => $limits[$limitType],
            'remaining' => max(0, $remaining),
            'plan_type' => $userPlan['plan_type']
        ];
    }
    
    /**
     * Obtenir l'utilisation actuelle d'un utilisateur
     */
    private function getCurrentUsage($userId, $limitType) {
        try {
            switch ($limitType) {
                case 'max_projects':
                    $stmt = $this->pdo->prepare("
                        SELECT COUNT(*) FROM flying_book_projects 
                        WHERE user_id = ? AND status != 'deleted'
                    ");
                    $stmt->execute([$userId]);
                    return (int)$stmt->fetchColumn();
                    
                case 'max_storage':
                    $stmt = $this->pdo->prepare("
                        SELECT COALESCE(SUM(file_size_total), 0) FROM flying_book_projects 
                        WHERE user_id = ? AND status != 'deleted'
                    ");
                    $stmt->execute([$userId]);
                    return (int)$stmt->fetchColumn();
                    
                case 'max_views_per_month':
                    $stmt = $this->pdo->prepare("
                        SELECT COUNT(*) FROM flying_book_analytics a
                        JOIN flying_book_projects p ON a.project_id = p.id
                        WHERE p.user_id = ? 
                        AND a.first_visit >= DATE_SUB(NOW(), INTERVAL 1 MONTH)
                    ");
                    $stmt->execute([$userId]);
                    return (int)$stmt->fetchColumn();
                    
                default:
                    return 0;
            }
        } catch (Exception $e) {
            return 0;
        }
    }
    
    /**
     * Créer ou mettre à jour un abonnement
     */
    public function createSubscription($userId, $planType, $billingCycle = 'monthly', $stripeData = null) {
        try {
            $this->pdo->beginTransaction();
            
            // Annuler l'abonnement actuel s'il existe
            $this->cancelCurrentSubscription($userId);
            
            $planFeatures = $this->getPlanFeatures($planType);
            $planLimits = self::FLYING_BOOK_LIMITS[$planType];
            $planPrice = self::PLAN_PRICES[$planType][$billingCycle];
            
            $currentPeriodStart = date('Y-m-d H:i:s');
            $currentPeriodEnd = date('Y-m-d H:i:s', strtotime('+1 ' . ($billingCycle === 'yearly' ? 'year' : 'month')));
            
            $stmt = $this->pdo->prepare("
                INSERT INTO flying_book_subscriptions 
                (user_id, plan_type, plan_name, price_monthly, price_yearly, billing_cycle,
                 stripe_subscription_id, stripe_customer_id, status, current_period_start,
                 current_period_end, features, limits, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'active', ?, ?, ?, ?, NOW(), NOW())
            ");
            
            $stmt->execute([
                $userId,
                $planType,
                $this->getPlanName($planType),
                self::PLAN_PRICES[$planType]['monthly'],
                self::PLAN_PRICES[$planType]['yearly'],
                $billingCycle,
                $stripeData['subscription_id'] ?? null,
                $stripeData['customer_id'] ?? null,
                $currentPeriodStart,
                $currentPeriodEnd,
                json_encode($planFeatures),
                json_encode($planLimits)
            ]);
            
            $subscriptionId = $this->pdo->lastInsertId();
            
            // Mettre à jour le statut premium de l'utilisateur
            $isPremium = $planType !== 'free';
            $stmt = $this->pdo->prepare("
                UPDATE users SET is_premium = ?, updated_at = NOW() WHERE id = ?
            ");
            $stmt->execute([$isPremium ? 1 : 0, $userId]);
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'subscription_id' => $subscriptionId,
                'message' => 'Abonnement créé avec succès'
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollback();
            return [
                'success' => false,
                'error' => 'Erreur lors de la création de l\'abonnement: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Annuler l'abonnement actuel
     */
    private function cancelCurrentSubscription($userId) {
        $stmt = $this->pdo->prepare("
            UPDATE flying_book_subscriptions 
            SET status = 'cancelled', cancelled_at = NOW(), updated_at = NOW()
            WHERE user_id = ? AND status = 'active'
        ");
        $stmt->execute([$userId]);
    }
    
    /**
     * Démarrer un essai gratuit
     */
    public function startTrial($userId, $planType, $trialDays = 14) {
        try {
            $this->pdo->beginTransaction();
            
            // Vérifier si l'utilisateur a déjà eu un essai
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM flying_book_subscriptions 
                WHERE user_id = ? AND trial_ends_at IS NOT NULL
            ");
            $stmt->execute([$userId]);
            
            if ($stmt->fetchColumn() > 0) {
                return [
                    'success' => false,
                    'error' => 'Essai gratuit déjà utilisé'
                ];
            }
            
            $trialEndDate = date('Y-m-d H:i:s', strtotime("+{$trialDays} days"));
            $planFeatures = $this->getPlanFeatures($planType);
            $planLimits = self::FLYING_BOOK_LIMITS[$planType];
            
            $stmt = $this->pdo->prepare("
                INSERT INTO flying_book_subscriptions 
                (user_id, plan_type, plan_name, billing_cycle, status, trial_ends_at,
                 current_period_start, current_period_end, features, limits, created_at, updated_at)
                VALUES (?, ?, ?, 'monthly', 'trialing', ?, NOW(), ?, ?, ?, NOW(), NOW())
            ");
            
            $stmt->execute([
                $userId,
                $planType,
                $this->getPlanName($planType),
                $trialEndDate,
                $trialEndDate,
                json_encode($planFeatures),
                json_encode($planLimits)
            ]);
            
            // Mettre à jour le statut premium temporaire
            $stmt = $this->pdo->prepare("
                UPDATE users SET is_premium = 1, updated_at = NOW() WHERE id = ?
            ");
            $stmt->execute([$userId]);
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'trial_ends_at' => $trialEndDate,
                'message' => "Essai gratuit de {$trialDays} jours activé"
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollback();
            return [
                'success' => false,
                'error' => 'Erreur lors de l\'activation de l\'essai: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Obtenir les fonctionnalités d'un plan
     */
    private function getPlanFeatures($planType) {
        $allFeatures = [
            'free' => [
                'Jusqu\'à 3 projets',
                'Jusqu\'à 10 slides par projet',
                '100MB de stockage',
                'Thèmes de base',
                'Transitions standard',
                'Support communautaire'
            ],
            'pro' => [
                'Jusqu\'à 25 projets',
                'Jusqu\'à 50 slides par projet',
                '5GB de stockage',
                'Thèmes premium',
                'Transitions avancées',
                'Analytics détaillées',
                'Suppression du watermark',
                'Collaboration d\'équipe',
                'Export PDF et vidéo'
            ],
            'business' => [
                'Jusqu\'à 100 projets',
                'Jusqu\'à 200 slides par projet',
                '50GB de stockage',
                'Tous les thèmes et transitions',
                'Domaine personnalisé',
                'Analytics avancées',
                'Support prioritaire',
                'API d\'intégration',
                'Export SCORM',
                'Gestion d\'équipe avancée'
            ],
            'enterprise' => [
                'Projets illimités',
                'Slides illimitées',
                'Stockage illimité',
                'White-label complet',
                'Branding personnalisé',
                'Support dédié',
                'Intégrations sur mesure',
                'Conformité entreprise',
                'SLA garanti',
                'Formation personnalisée'
            ]
        ];
        
        return $allFeatures[$planType] ?? [];
    }
    
    /**
     * Obtenir le nom d'un plan
     */
    private function getPlanName($planType) {
        $names = [
            'free' => 'Gratuit',
            'pro' => 'Professionnel',
            'business' => 'Business',
            'enterprise' => 'Enterprise'
        ];
        
        return $names[$planType] ?? 'Inconnu';
    }
    
    /**
     * Obtenir tous les plans disponibles
     */
    public function getAvailablePlans() {
        $plans = [];
        
        foreach (self::FLYING_BOOK_LIMITS as $planType => $limits) {
            $plans[$planType] = [
                'type' => $planType,
                'name' => $this->getPlanName($planType),
                'limits' => $limits,
                'features' => $this->getPlanFeatures($planType),
                'prices' => self::PLAN_PRICES[$planType],
                'recommended' => $planType === 'pro',
                'popular' => $planType === 'business'
            ];
        }
        
        return $plans;
    }
    
    /**
     * Vérifier les abonnements expirés
     */
    public function checkExpiredSubscriptions() {
        try {
            // Marquer les essais expirés
            $stmt = $this->pdo->prepare("
                UPDATE flying_book_subscriptions 
                SET status = 'expired', updated_at = NOW()
                WHERE status = 'trialing' AND trial_ends_at < NOW()
            ");
            $stmt->execute();
            
            // Marquer les abonnements expirés
            $stmt = $this->pdo->prepare("
                UPDATE flying_book_subscriptions 
                SET status = 'expired', updated_at = NOW()
                WHERE status = 'active' AND current_period_end < NOW()
            ");
            $stmt->execute();
            
            // Rétrograder les utilisateurs vers le plan gratuit
            $stmt = $this->pdo->prepare("
                UPDATE users u
                LEFT JOIN flying_book_subscriptions s ON u.id = s.user_id AND s.status = 'active'
                SET u.is_premium = 0, u.updated_at = NOW()
                WHERE s.id IS NULL AND u.is_premium = 1
            ");
            $stmt->execute();
            
            return ['success' => true, 'message' => 'Vérification des abonnements terminée'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Obtenir les statistiques d'utilisation d'un utilisateur
     */
    public function getUserUsageStats($userId) {
        $userPlan = $this->getUserPlan($userId);
        $limits = $userPlan['limits'];
        
        $stats = [];
        
        foreach (['max_projects', 'max_storage', 'max_views_per_month'] as $limitType) {
            $usage = $this->getCurrentUsage($userId, $limitType);
            $limit = $limits[$limitType];
            
            $stats[$limitType] = [
                'current' => $usage,
                'limit' => $limit,
                'percentage' => $limit > 0 ? round(($usage / $limit) * 100, 1) : 0,
                'is_unlimited' => $limit === -1
            ];
        }
        
        return $stats;
    }
}

// Fonctions utilitaires globales
function getUserPlanLimits($userId) {
    global $pdo;
    static $subscriptionManager = null;
    
    if ($subscriptionManager === null) {
        $subscriptionManager = new FlyingBookSubscriptionManager($pdo);
    }
    
    $userPlan = $subscriptionManager->getUserPlan($userId);
    return $userPlan['limits'];
}

function checkUserLimits($userId, $limitType, $requestedAmount = 1) {
    global $pdo;
    static $subscriptionManager = null;
    
    if ($subscriptionManager === null) {
        $subscriptionManager = new FlyingBookSubscriptionManager($pdo);
    }
    
    $result = $subscriptionManager->checkUserLimit($userId, $limitType, $requestedAmount);
    return $result['allowed'];
}

// Utilisation dans les autres fichiers PHP
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        $subscriptionManager = new FlyingBookSubscriptionManager($pdo);
        
        switch ($_POST['action']) {
            case 'get_user_plan':
                $userId = $_SESSION['user_id'];
                $plan = $subscriptionManager->getUserPlan($userId);
                echo json_encode(['success' => true, 'plan' => $plan]);
                exit;
                
            case 'get_usage_stats':
                $userId = $_SESSION['user_id'];
                $stats = $subscriptionManager->getUserUsageStats($userId);
                echo json_encode(['success' => true, 'stats' => $stats]);
                exit;
                
            case 'start_trial':
                $userId = $_SESSION['user_id'];
                $planType = $_POST['plan_type'] ?? 'pro';
                $result = $subscriptionManager->startTrial($userId, $planType);
                echo json_encode($result);
                exit;
                
            case 'get_available_plans':
                $plans = $subscriptionManager->getAvailablePlans();
                echo json_encode(['success' => true, 'plans' => $plans]);
                exit;
                
            case 'upgrade_plan':
                $userId = $_SESSION['user_id'];
                $planType = $_POST['plan_type'];
                $billingCycle = $_POST['billing_cycle'] ?? 'monthly';
                
                // En production, intégrer Stripe ici
                $result = $subscriptionManager->createSubscription($userId, $planType, $billingCycle);
                echo json_encode($result);
                exit;
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}
?>