<?php
session_start();
require_once '../../env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

$project_id = intval($_GET['project_id'] ?? 0);
if (!$project_id) {
    header('Location: ../dashboard.php');
    exit;
}

try {
    $pdo = getDatabaseConnection();
    
    // Récupérer le projet
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$project_id, $_SESSION['user_id']]);
    $project = $stmt->fetch();
    
    if (!$project) {
        header('Location: ../dashboard.php');
        exit;
    }
    
    // Récupérer les slides
    $stmt = $pdo->prepare("
        SELECT s.*, m.original_filename, m.file_size, m.mime_type, m.thumbnail_path
        FROM flying_book_slides s
        LEFT JOIN flying_book_media m ON s.media_file = m.stored_filename AND m.project_id = s.project_id
        WHERE s.project_id = ? AND s.is_active = 1
        ORDER BY s.display_order ASC
    ");
    $stmt->execute([$project_id]);
    $slides = $stmt->fetchAll();
    
    // Récupérer les médias du projet
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_media 
        WHERE project_id = ? AND upload_status = 'ready'
        ORDER BY created_at DESC
    ");
    $stmt->execute([$project_id]);
    $media_library = $stmt->fetchAll();
    
} catch (Exception $e) {
    logError("Erreur lors du chargement du projet", ['project_id' => $project_id, 'error' => $e->getMessage()]);
    header('Location: ../dashboard.php');
    exit;
}

// Traitement des actions AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        switch ($_POST['action']) {
            case 'update_slide':
                $slide_id = intval($_POST['slide_id']);
                $slide_title = trim($_POST['slide_title'] ?? '');
                $media_type = $_POST['media_type'] ?? 'text_image';
                $media_file = $_POST['media_file'] ?? '';
                $text_overlay = $_POST['text_overlay'] ?? '';
                $slide_notes = $_POST['slide_notes'] ?? '';
                $auto_duration = intval($_POST['auto_duration'] ?? 0);
                
                $stmt = $pdo->prepare("
                    UPDATE flying_book_slides 
                    SET slide_title = ?, media_type = ?, media_file = ?, 
                        text_overlay = ?, slide_notes = ?, auto_duration = ?, updated_at = NOW()
                    WHERE id = ? AND project_id = ?
                ");
                $stmt->execute([
                    $slide_title, $media_type, $media_file, 
                    $text_overlay, $slide_notes, $auto_duration, 
                    $slide_id, $project_id
                ]);
                
                echo json_encode(['success' => true, 'message' => 'Slide mise à jour']);
                break;
                
            case 'reorder_slides':
                $slide_orders = json_decode($_POST['slide_orders'], true);
                
                foreach ($slide_orders as $slide_id => $order) {
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_slides 
                        SET display_order = ? 
                        WHERE id = ? AND project_id = ?
                    ");
                    $stmt->execute([$order, $slide_id, $project_id]);
                }
                
                echo json_encode(['success' => true, 'message' => 'Ordre des slides mis à jour']);
                break;
                
            case 'add_slide':
                $slide_number = intval($_POST['slide_number'] ?? 0);
                $position = $_POST['position'] ?? 'end'; // 'before', 'after', 'end'
                
                if ($position === 'end') {
                    $stmt = $pdo->prepare("SELECT MAX(display_order) as max_order FROM flying_book_slides WHERE project_id = ?");
                    $stmt->execute([$project_id]);
                    $max_order = $stmt->fetch()['max_order'] ?? 0;
                    $new_order = $max_order + 1;
                } else {
                    // Décaler les slides existantes
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_slides 
                        SET display_order = display_order + 1 
                        WHERE project_id = ? AND display_order >= ?
                    ");
                    $operator = $position === 'after' ? '>' : '>=';
                    $stmt->execute([$project_id, $slide_number]);
                    $new_order = $position === 'after' ? $slide_number + 1 : $slide_number;
                }
                
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_slides 
                    (project_id, slide_number, slide_title, media_type, media_file, display_order, created_at)
                    VALUES (?, ?, ?, 'text_image', 'placeholder.jpg', ?, NOW())
                ");
                $stmt->execute([$project_id, $new_order, "Nouvelle slide", $new_order]);
                
                $new_slide_id = $pdo->lastInsertId();
                
                // Mettre à jour le total des slides dans le projet
                $stmt = $pdo->prepare("
                    UPDATE flying_book_projects 
                    SET total_slides = (SELECT COUNT(*) FROM flying_book_slides WHERE project_id = ? AND is_active = 1)
                    WHERE id = ?
                ");
                $stmt->execute([$project_id, $project_id]);
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Nouvelle slide ajoutée',
                    'slide_id' => $new_slide_id
                ]);
                break;
                
            case 'delete_slide':
                $slide_id = intval($_POST['slide_id']);
                
                $stmt = $pdo->prepare("
                    UPDATE flying_book_slides 
                    SET is_active = 0 
                    WHERE id = ? AND project_id = ?
                ");
                $stmt->execute([$slide_id, $project_id]);
                
                // Réorganiser les ordres
                $stmt = $pdo->prepare("
                    SELECT id, display_order FROM flying_book_slides 
                    WHERE project_id = ? AND is_active = 1 
                    ORDER BY display_order ASC
                ");
                $stmt->execute([$project_id]);
                $remaining_slides = $stmt->fetchAll();
                
                foreach ($remaining_slides as $index => $slide) {
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_slides 
                        SET display_order = ? 
                        WHERE id = ?
                    ");
                    $stmt->execute([$index + 1, $slide['id']]);
                }
                
                // Mettre à jour le total
                $stmt = $pdo->prepare("
                    UPDATE flying_book_projects 
                    SET total_slides = (SELECT COUNT(*) FROM flying_book_slides WHERE project_id = ? AND is_active = 1)
                    WHERE id = ?
                ");
                $stmt->execute([$project_id, $project_id]);
                
                echo json_encode(['success' => true, 'message' => 'Slide supprimée']);
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        }
    } catch (Exception $e) {
        logError("Erreur lors de l'action sur les slides", ['action' => $_POST['action'], 'error' => $e->getMessage()]);
        echo json_encode(['success' => false, 'message' => 'Erreur lors de l\'opération']);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book - Édition des slides</title>
    <link rel="stylesheet" href="../assets/css/slides-editor.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/Sortable/1.15.0/sortable.min.css">
</head>
<body>
    <!-- Header de l'éditeur -->
    <header class="editor-header">
        <div class="header-container">
            <div class="header-left">
                <a href="../dashboard.php" class="back-btn">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <div class="project-info">
                    <h1><?= htmlspecialchars($project['project_name']) ?></h1>
                    <div class="project-meta">
                        <span class="slide-count"><?= count($slides) ?> slides</span>
                        <span class="theme-info"><?= ucfirst($project['style_theme']) ?></span>
                        <span class="status <?= $project['status'] ?>"><?= ucfirst($project['status']) ?></span>
                    </div>
                </div>
            </div>
            
            <div class="header-actions">
                <button class="btn btn-ghost" onclick="togglePreview()">
                    <i class="fas fa-eye"></i>
                    Aperçu
                </button>
                
                <button class="btn btn-outline" onclick="saveProject()">
                    <i class="fas fa-save"></i>
                    Sauvegarder
                </button>
                
                <div class="dropdown">
                    <button class="btn btn-primary dropdown-toggle" onclick="togglePublishMenu()">
                        <i class="fas fa-rocket"></i>
                        Publier
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    <div class="dropdown-menu" id="publishMenu">
                        <a href="#" onclick="publishProject('draft')">
                            <i class="fas fa-edit"></i>
                            Sauvegarder comme brouillon
                        </a>
                        <a href="#" onclick="publishProject('published')">
                            <i class="fas fa-globe"></i>
                            Publier en ligne
                        </a>
                        <div class="dropdown-divider"></div>
                        <a href="#" onclick="showExportOptions()">
                            <i class="fas fa-download"></i>
                            Exporter
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <div class="editor-layout">
        <!-- Sidebar des slides -->
        <aside class="slides-sidebar">
            <div class="sidebar-header">
                <h3>
                    <i class="fas fa-images"></i>
                    Slides
                </h3>
                <button class="btn btn-sm btn-primary" onclick="addSlide('end')">
                    <i class="fas fa-plus"></i>
                    Ajouter
                </button>
            </div>
            
            <div class="slides-list" id="slidesList">
                <?php foreach ($slides as $index => $slide): ?>
                <div class="slide-item <?= $index === 0 ? 'active' : '' ?>" 
                     data-slide-id="<?= $slide['id'] ?>"
                     data-order="<?= $slide['display_order'] ?>"
                     onclick="selectSlide(<?= $slide['id'] ?>)">
                    
                    <div class="slide-thumbnail">
                        <?php if ($slide['media_file'] && $slide['media_file'] !== 'placeholder.jpg'): ?>
                            <?php if (in_array($slide['media_type'], ['image', 'text_image'])): ?>
                                <img src="../document/projects/<?= $project['project_folder'] ?>/thumbnails/<?= $slide['media_file'] ?>" 
                                     alt="Slide <?= $slide['slide_number'] ?>"
                                     onerror="this.src='../assets/images/slide-placeholder.jpg'">
                            <?php elseif ($slide['media_type'] === 'video'): ?>
                                <div class="video-thumbnail">
                                    <i class="fas fa-play-circle"></i>
                                </div>
                            <?php else: ?>
                                <div class="media-thumbnail">
                                    <i class="fas fa-file"></i>
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="placeholder-thumbnail">
                                <i class="fas fa-image"></i>
                            </div>
                        <?php endif; ?>
                        
                        <div class="slide-number"><?= $slide['display_order'] ?></div>
                    </div>
                    
                    <div class="slide-info">
                        <h4><?= htmlspecialchars($slide['slide_title'] ?: "Slide {$slide['display_order']}") ?></h4>
                        <p class="slide-type"><?= ucfirst(str_replace('_', ' + ', $slide['media_type'])) ?></p>
                        <?php if ($slide['auto_duration']): ?>
                            <p class="slide-duration">
                                <i class="fas fa-clock"></i>
                                <?= $slide['auto_duration'] ?>s
                            </p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="slide-actions">
                        <button class="action-btn" onclick="event.stopPropagation(); duplicateSlide(<?= $slide['id'] ?>)" title="Dupliquer">
                            <i class="fas fa-copy"></i>
                        </button>
                        <button class="action-btn delete" onclick="event.stopPropagation(); deleteSlide(<?= $slide['id'] ?>)" title="Supprimer">
                            <i class="fas fa-trash"></i>
                        </button>
                        <div class="drag-handle" title="Glisser pour réorganiser">
                            <i class="fas fa-grip-vertical"></i>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="sidebar-footer">
                <button class="btn btn-outline btn-block" onclick="addSlide('end')">
                    <i class="fas fa-plus"></i>
                    Nouvelle slide
                </button>
            </div>
        </aside>

        <!-- Zone principale d'édition -->
        <main class="editor-main">
            <div class="editor-content" id="editorContent">
                <!-- Le contenu sera chargé dynamiquement -->
                <div class="empty-editor">
                    <i class="fas fa-mouse-pointer"></i>
                    <h3>Sélectionnez une slide pour commencer</h3>
                    <p>Cliquez sur une slide dans la barre latérale pour l'éditer</p>
                </div>
            </div>
        </main>

        <!-- Panneau de propriétés -->
        <aside class="properties-panel" id="propertiesPanel">
            <div class="panel-header">
                <h3>
                    <i class="fas fa-cog"></i>
                    Propriétés
                </h3>
                <button class="panel-toggle" onclick="togglePropertiesPanel()">
                    <i class="fas fa-chevron-right"></i>
                </button>
            </div>
            
            <div class="panel-content" id="panelContent">
                <!-- Le contenu sera chargé selon la slide sélectionnée -->
            </div>
        </aside>
    </div>

    <!-- Modal d'upload de fichiers -->
    <div class="modal" id="uploadModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>
                    <i class="fas fa-upload"></i>
                    Ajouter des médias
                </h3>
                <button class="modal-close" onclick="closeUploadModal()">&times;</button>
            </div>
            
            <div class="modal-body">
                <div class="upload-tabs">
                    <button class="tab-btn active" onclick="switchUploadTab('files')">
                        <i class="fas fa-file-upload"></i>
                        Fichiers
                    </button>
                    <button class="tab-btn" onclick="switchUploadTab('library')">
                        <i class="fas fa-folder"></i>
                        Bibliothèque
                    </button>
                    <button class="tab-btn" onclick="switchUploadTab('url')">
                        <i class="fas fa-link"></i>
                        URL
                    </button>
                </div>
                
                <div class="upload-content">
                    <!-- Onglet Upload -->
                    <div class="upload-tab active" id="uploadTabFiles">
                        <div class="upload-zone" id="uploadZone">
                            <div class="upload-icon">
                                <i class="fas fa-cloud-upload-alt"></i>
                            </div>
                            <h4>Glissez vos fichiers ici</h4>
                            <p>ou <button class="upload-btn" onclick="document.getElementById('fileInput').click()">choisissez des fichiers</button></p>
                            <small>Formats supportés: JPG, PNG, GIF, MP4, MP3, PDF (max <?= formatFileSize(FLYING_BOOK_MAX_FILE_SIZE) ?>)</small>
                            <input type="file" id="fileInput" multiple accept="<?= str_replace(',', ',.', FLYING_BOOK_ALLOWED_TYPES) ?>" style="display: none;">
                        </div>
                        
                        <div class="upload-progress" id="uploadProgress" style="display: none;">
                            <div class="progress-files" id="progressFiles"></div>
                        </div>
                    </div>
                    
                    <!-- Onglet Bibliothèque -->
                    <div class="upload-tab" id="uploadTabLibrary">
                        <div class="media-library">
                            <?php if (empty($media_library)): ?>
                                <div class="empty-library">
                                    <i class="fas fa-folder-open"></i>
                                    <p>Aucun média dans votre bibliothèque</p>
                                </div>
                            <?php else: ?>
                                <div class="media-grid">
                                    <?php foreach ($media_library as $media): ?>
                                        <div class="media-item" onclick="selectMedia('<?= $media['stored_filename'] ?>', '<?= $media['file_type'] ?>')">
                                            <?php if ($media['file_type'] === 'image'): ?>
                                                <img src="../document/projects/<?= $project['project_folder'] ?>/thumbnails/<?= $media['stored_filename'] ?>" 
                                                     alt="<?= htmlspecialchars($media['original_filename']) ?>">
                                            <?php else: ?>
                                                <div class="file-icon">
                                                    <i class="fas fa-file-<?= $media['file_type'] === 'video' ? 'video' : ($media['file_type'] === 'audio' ? 'audio' : 'alt') ?>"></i>
                                                </div>
                                            <?php endif; ?>
                                            <div class="media-info">
                                                <h5><?= htmlspecialchars($media['original_filename']) ?></h5>
                                                <p><?= formatFileSize($media['file_size']) ?></p>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Onglet URL -->
                    <div class="upload-tab" id="uploadTabUrl">
                        <div class="url-input">
                            <label for="mediaUrl">URL du média</label>
                            <input type="url" id="mediaUrl" placeholder="https://exemple.com/image.jpg">
                            <button class="btn btn-primary" onclick="addMediaFromUrl()">
                                <i class="fas fa-plus"></i>
                                Ajouter
                            </button>
                        </div>
                        
                        <div class="url-examples">
                            <h5>Exemples d'URLs supportées :</h5>
                            <ul>
                                <li>Images: JPG, PNG, GIF, WebP</li>
                                <li>Vidéos: MP4, WebM</li>
                                <li>Audio: MP3, WAV, OGG</li>
                                <li>YouTube, Vimeo (intégration automatique)</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal de prévisualisation -->
    <div class="modal fullscreen" id="previewModal">
        <div class="modal-content">
            <div class="preview-header">
                <h3><?= htmlspecialchars($project['title']) ?></h3>
                <div class="preview-controls">
                    <button class="btn btn-ghost" onclick="prevSlidePreview()">
                        <i class="fas fa-chevron-left"></i>
                    </button>
                    <span class="slide-indicator" id="slideIndicator">1 / <?= count($slides) ?></span>
                    <button class="btn btn-ghost" onclick="nextSlidePreview()">
                        <i class="fas fa-chevron-right"></i>
                    </button>
                </div>
                <button class="modal-close" onclick="closePreview()">&times;</button>
            </div>
            
            <div class="preview-content" id="previewContent">
                <!-- Le contenu de prévisualisation sera chargé ici -->
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/Sortable/1.15.0/Sortable.min.js"></script>
    <script>
        // Variables globales
        const PROJECT_ID = <?= $project_id ?>;
        const PROJECT_DATA = <?= json_encode($project) ?>;
        const SLIDES_DATA = <?= json_encode($slides) ?>;
        let currentSlideId = <?= !empty($slides) ? $slides[0]['id'] : 'null' ?>;
        let previewSlideIndex = 0;
    </script>
    <script src="../assets/js/slides-editor.js"></script>
</body>
</html>