<?php
session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

$message = '';
$error = '';

// Configuration des plans
$PLANS_CONFIG = [
    'free' => [
        'name' => 'Gratuit',
        'price_monthly' => 0,
        'price_yearly' => 0,
        'color' => '#6b7280',
        'features' => [
            '3 projets maximum',
            '500 MB de stockage',
            'Templates de base',
            'Export PDF',
            'Support email'
        ],
        'limits' => [
            'max_projects' => 3,
            'max_storage' => 500 * 1024 * 1024, // 500MB
            'max_slides_per_project' => 20,
            'templates_access' => 'basic',
            'priority_support' => false
        ]
    ],
    'pro' => [
        'name' => 'Pro',
        'price_monthly' => 9.99,
        'price_yearly' => 99.99,
        'color' => '#667eea',
        'badge' => 'Populaire',
        'features' => [
            '50 projets maximum',
            '10 GB de stockage',
            'Tous les templates',
            'Analytics avancées',
            'Domaine personnalisé',
            'Support prioritaire'
        ],
        'limits' => [
            'max_projects' => 50,
            'max_storage' => 10 * 1024 * 1024 * 1024, // 10GB
            'max_slides_per_project' => 100,
            'templates_access' => 'all',
            'priority_support' => true
        ]
    ],
    'business' => [
        'name' => 'Business',
        'price_monthly' => 29.99,
        'price_yearly' => 299.99,
        'color' => '#f59e0b',
        'features' => [
            'Projets illimités',
            '100 GB de stockage',
            'Templates premium',
            'API access',
            'White-label',
            'Support dédié'
        ],
        'limits' => [
            'max_projects' => -1, // illimité
            'max_storage' => 100 * 1024 * 1024 * 1024, // 100GB
            'max_slides_per_project' => -1, // illimité
            'templates_access' => 'premium',
            'priority_support' => true
        ]
    ]
];

// Fonctions utilitaires
function getDatabaseConnection() {
    try {
        return new PDO(
            "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
            DB_USER,
            DB_PASS,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } catch (PDOException $e) {
        throw new Exception("Erreur de connexion à la base de données");
    }
}

function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    $k = 1024;
    $sizes = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes) / log($k));
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

function getUserUsage($pdo, $user_id) {
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT p.id) as projects_count,
            COALESCE(SUM(m.file_size), 0) as storage_used,
            p.plan_type as current_plan
        FROM users u
        LEFT JOIN flying_book_projects p ON u.id = p.user_id AND p.status != 'deleted'
        LEFT JOIN flying_book_media m ON u.id = m.user_id AND m.upload_status = 'ready'
        WHERE u.id = ?
        GROUP BY u.id
    ");
    $stmt->execute([$user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$result) {
        return [
            'projects_count' => 0,
            'storage_used' => 0,
            'current_plan' => 'free'
        ];
    }
    
    return $result;
}

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['ajax_action']) {
            case 'simulate_upgrade':
                $plan = $_POST['plan'] ?? '';
                $billing_cycle = $_POST['billing_cycle'] ?? 'monthly';
                
                if (!isset($PLANS_CONFIG[$plan])) {
                    echo json_encode(['success' => false, 'message' => 'Plan invalide']);
                    exit;
                }
                
                $plan_config = $PLANS_CONFIG[$plan];
                $price = $billing_cycle === 'yearly' ? $plan_config['price_yearly'] : $plan_config['price_monthly'];
                
                // Simulation du changement de plan
                $stmt = $pdo->prepare("
                    UPDATE flying_book_projects 
                    SET plan_type = ? 
                    WHERE user_id = ?
                ");
                $stmt->execute([$plan, $user_id]);
                
                // Créer un faux abonnement pour la démo
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_subscriptions 
                    (user_id, plan_type, plan_name, price_monthly, price_yearly, billing_cycle, 
                     status, current_period_start, current_period_end, features, limits)
                    VALUES (?, ?, ?, ?, ?, ?, 'active', NOW(), DATE_ADD(NOW(), INTERVAL 1 MONTH), ?, ?)
                    ON DUPLICATE KEY UPDATE
                        plan_type = VALUES(plan_type),
                        plan_name = VALUES(plan_name),
                        billing_cycle = VALUES(billing_cycle),
                        status = 'active',
                        current_period_end = DATE_ADD(NOW(), INTERVAL 1 MONTH),
                        features = VALUES(features),
                        limits = VALUES(limits)
                ");
                
                $stmt->execute([
                    $user_id,
                    $plan,
                    $plan_config['name'],
                    $plan_config['price_monthly'],
                    $plan_config['price_yearly'],
                    $billing_cycle,
                    json_encode($plan_config['features']),
                    json_encode($plan_config['limits'])
                ]);
                
                echo json_encode([
                    'success' => true, 
                    'message' => "Abonnement {$plan_config['name']} activé (simulation)",
                    'plan' => $plan,
                    'price' => $price
                ]);
                exit;
                
            case 'calculate_usage':
                $usage = getUserUsage($pdo, $user_id);
                $current_plan = $usage['current_plan'] ?? 'free';
                $plan_limits = $PLANS_CONFIG[$current_plan]['limits'];
                
                $usage_percentage = [
                    'projects' => $plan_limits['max_projects'] > 0 ? 
                        min(100, ($usage['projects_count'] / $plan_limits['max_projects']) * 100) : 0,
                    'storage' => $plan_limits['max_storage'] > 0 ? 
                        min(100, ($usage['storage_used'] / $plan_limits['max_storage']) * 100) : 0
                ];
                
                echo json_encode([
                    'success' => true,
                    'usage' => $usage,
                    'limits' => $plan_limits,
                    'percentage' => $usage_percentage,
                    'formatted_storage' => formatFileSize($usage['storage_used'])
                ]);
                exit;
                
            case 'export_data':
                $export_type = $_POST['export_type'] ?? 'all';
                
                // Simuler l'export (en réalité, on génèrerait un fichier)
                $export_data = [
                    'user_info' => [],
                    'projects' => [],
                    'media' => [],
                    'analytics' => []
                ];
                
                if ($export_type === 'all' || $export_type === 'user') {
                    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $export_data['user_info'] = $stmt->fetch(PDO::FETCH_ASSOC);
                }
                
                if ($export_type === 'all' || $export_type === 'projects') {
                    $stmt = $pdo->prepare("SELECT * FROM flying_book_projects WHERE user_id = ?");
                    $stmt->execute([$user_id]);
                    $export_data['projects'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                }
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Export généré (simulation)',
                    'data_size' => count($export_data['projects']) . ' projets'
                ]);
                exit;
                
            case 'delete_account':
                $password = $_POST['password'] ?? '';
                
                if (empty($password)) {
                    echo json_encode(['success' => false, 'message' => 'Mot de passe requis']);
                    exit;
                }
                
                // Vérifier le mot de passe
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $stored_password = $stmt->fetchColumn();
                
                if (!password_verify($password, $stored_password)) {
                    echo json_encode(['success' => false, 'message' => 'Mot de passe incorrect']);
                    exit;
                }
                
                // Marquer le compte comme supprimé (soft delete)
                $stmt = $pdo->prepare("
                    UPDATE users 
                    SET status = 'deleted', deleted_at = NOW(), email = CONCAT(email, '_deleted_', UNIX_TIMESTAMP())
                    WHERE id = ?
                ");
                
                if ($stmt->execute([$user_id])) {
                    session_destroy();
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Compte supprimé avec succès',
                        'redirect' => 'login.php'
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la suppression']);
                }
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Informations utilisateur
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Abonnement actuel
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_subscriptions 
        WHERE user_id = ? AND status = 'active'
        ORDER BY created_at DESC LIMIT 1
    ");
    $stmt->execute([$user_id]);
    $subscription = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $current_plan = $subscription['plan_type'] ?? 'free';
    
    // Utilisation actuelle
    $usage = getUserUsage($pdo, $user_id);
    $plan_limits = $PLANS_CONFIG[$current_plan]['limits'];
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement des paramètres: " . $e->getMessage();
    error_log("Settings page error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Paramètres</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .settings-tabs {
            display: flex;
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-sm);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-sm);
            overflow-x: auto;
        }
        
        .settings-tab {
            flex: 1;
            min-width: 140px;
            padding: var(--space-md);
            background: none;
            border: none;
            border-radius: var(--radius-sm);
            cursor: pointer;
            transition: var(--transition-normal);
            font-weight: 600;
            color: var(--text-secondary);
            white-space: nowrap;
        }
        
        .settings-tab.active {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--text-white);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .plan-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: var(--space-xl);
            margin-bottom: var(--space-xl);
        }
        
        .plan-card {
            background: var(--bg-primary);
            border-radius: var(--radius-xl);
            padding: var(--space-xl);
            position: relative;
            transition: var(--transition-normal);
            border: 2px solid transparent;
            box-shadow: var(--shadow-sm);
        }
        
        .plan-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }
        
        .plan-card.current {
            border-color: var(--success-color);
            background: rgba(16, 185, 129, 0.05);
        }
        
        .plan-card.popular {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.05);
        }
        
        .plan-badge {
            position: absolute;
            top: -10px;
            right: var(--space-lg);
            background: var(--primary-color);
            color: var(--text-white);
            padding: var(--space-xs) var(--space-md);
            border-radius: var(--radius-full);
            font-size: var(--font-size-xs);
            font-weight: 700;
            text-transform: uppercase;
        }
        
        .plan-badge.current {
            background: var(--success-color);
        }
        
        .plan-header {
            text-align: center;
            margin-bottom: var(--space-xl);
        }
        
        .plan-name {
            font-size: var(--font-size-2xl);
            font-weight: 700;
            margin-bottom: var(--space-sm);
        }
        
        .plan-price {
            font-size: var(--font-size-3xl);
            font-weight: 900;
            color: var(--primary-color);
            margin-bottom: var(--space-xs);
        }
        
        .plan-price .currency {
            font-size: var(--font-size-xl);
        }
        
        .plan-price .period {
            font-size: var(--font-size-base);
            color: var(--text-secondary);
            font-weight: 400;
        }
        
        .plan-features {
            list-style: none;
            margin-bottom: var(--space-xl);
        }
        
        .plan-features li {
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            margin-bottom: var(--space-sm);
            font-size: var(--font-size-sm);
        }
        
        .plan-features i {
            color: var(--success-color);
            width: 16px;
        }
        
        .billing-toggle {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--space-md);
            margin-bottom: var(--space-xl);
            background: var(--bg-primary);
            padding: var(--space-lg);
            border-radius: var(--radius-md);
        }
        
        .billing-switch {
            position: relative;
            width: 60px;
            height: 30px;
            background: var(--border-color);
            border-radius: var(--radius-full);
            cursor: pointer;
            transition: var(--transition-normal);
        }
        
        .billing-switch.yearly {
            background: var(--primary-color);
        }
        
        .billing-switch::after {
            content: '';
            position: absolute;
            top: 3px;
            left: 3px;
            width: 24px;
            height: 24px;
            background: var(--text-white);
            border-radius: var(--radius-full);
            transition: var(--transition-normal);
        }
        
        .billing-switch.yearly::after {
            transform: translateX(30px);
        }
        
        .usage-section {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-xl);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-sm);
        }
        
        .usage-item {
            margin-bottom: var(--space-lg);
        }
        
        .usage-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-sm);
        }
        
        .usage-bar {
            height: 8px;
            background: var(--bg-tertiary);
            border-radius: var(--radius-xs);
            overflow: hidden;
        }
        
        .usage-progress {
            height: 100%;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            transition: var(--transition-normal);
        }
        
        .usage-progress.high {
            background: linear-gradient(135deg, var(--warning-color), var(--error-color));
        }
        
        .danger-zone {
            background: rgba(239, 68, 68, 0.05);
            border: 1px solid rgba(239, 68, 68, 0.2);
            border-radius: var(--radius-lg);
            padding: var(--space-xl);
            margin-top: var(--space-xl);
        }
        
        .danger-zone h3 {
            color: var(--error-color);
            margin-bottom: var(--space-lg);
        }
        
        .export-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-md);
            margin-bottom: var(--space-lg);
        }
        
        .export-option {
            background: var(--bg-secondary);
            border: 1px solid var(--border-color);
            border-radius: var(--radius-sm);
            padding: var(--space-md);
            text-align: center;
            cursor: pointer;
            transition: var(--transition-normal);
        }
        
        .export-option:hover {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.05);
        }
        
        .export-option.selected {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.1);
        }
        
        @media (max-width: 768px) {
            .plan-grid {
                grid-template-columns: 1fr;
            }
            
            .settings-tabs {
                overflow-x: auto;
            }
            
            .settings-tab {
                min-width: 120px;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <a href="dashboard.php" class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </a>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="media.php" class="nav-link">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>
                    <a href="analytics.html" class="nav-link">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?php if ($user['avatar']): ?>
                                <img src="<?= htmlspecialchars($user['avatar']) ?>" alt="Avatar">
                            <?php else: ?>
                                <?= strtoupper(substr($user['name'], 0, 1)) ?>
                            <?php endif; ?>
                        </div>
                        <span><?= htmlspecialchars($user['name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php" class="active"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- Page Header -->
            <div class="page-header">
                <div class="header-content">
                    <h1><i class="fas fa-cog"></i> Paramètres</h1>
                    <p>Gérez votre abonnement et vos préférences</p>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Onglets des paramètres -->
            <div class="settings-tabs">
                <button class="settings-tab active" onclick="showTab('subscription')">
                    <i class="fas fa-credit-card"></i> Abonnement
                </button>
                <button class="settings-tab" onclick="showTab('usage')">
                    <i class="fas fa-chart-pie"></i> Utilisation
                </button>
                <button class="settings-tab" onclick="showTab('billing')">
                    <i class="fas fa-receipt"></i> Facturation
                </button>
                <button class="settings-tab" onclick="showTab('data')">
                    <i class="fas fa-database"></i> Données
                </button>
                <button class="settings-tab" onclick="showTab('account')">
                    <i class="fas fa-user-cog"></i> Compte
                </button>
            </div>

            <!-- Contenu des onglets -->
            
            <!-- Onglet Abonnement -->
            <div id="subscription-tab" class="tab-content active">
                <!-- Toggle facturation -->
                <div class="billing-toggle">
                    <span id="monthlyLabel" class="active">Mensuel</span>
                    <div class="billing-switch" id="billingSwitch" onclick="toggleBilling()"></div>
                    <span id="yearlyLabel">Annuel <span style="color: var(--success-color); font-weight: 600;">(-17%)</span></span>
                </div>

                <!-- Grille des plans -->
                <div class="plan-grid">
                    <?php foreach ($PLANS_CONFIG as $plan_key => $plan): ?>
                        <div class="plan-card <?= $current_plan === $plan_key ? 'current' : '' ?> <?= isset($plan['badge']) ? 'popular' : '' ?>">
                            <?php if ($current_plan === $plan_key): ?>
                                <div class="plan-badge current">Actuel</div>
                            <?php elseif (isset($plan['badge'])): ?>
                                <div class="plan-badge"><?= $plan['badge'] ?></div>
                            <?php endif; ?>
                            
                            <div class="plan-header">
                                <h3 class="plan-name" style="color: <?= $plan['color'] ?>"><?= $plan['name'] ?></h3>
                                <div class="plan-price">
                                    <span class="currency">€</span>
                                    <span class="amount" data-monthly="<?= $plan['price_monthly'] ?>" data-yearly="<?= $plan['price_yearly'] ?>">
                                        <?= $plan['price_monthly'] ?>
                                    </span>
                                    <span class="period">/mois</span>
                                </div>
                            </div>
                            
                            <ul class="plan-features">
                                <?php foreach ($plan['features'] as $feature): ?>
                                    <li>
                                        <i class="fas fa-check"></i>
                                        <?= htmlspecialchars($feature) ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                            
                            <?php if ($current_plan === $plan_key): ?>
                                <button class="btn btn-outline" style="width: 100%;" disabled>
                                    <i class="fas fa-check"></i>
                                    Plan actuel
                                </button>
                            <?php else: ?>
                                <button class="btn btn-primary" style="width: 100%;" 
                                        onclick="upgradePlan('<?= $plan_key ?>')">
                                    <?= $plan_key === 'free' ? 'Rétrograder' : 'Choisir ce plan' ?>
                                </button>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Onglet Utilisation -->
            <div id="usage-tab" class="tab-content">
                <div class="usage-section">
                    <h3><i class="fas fa-chart-pie"></i> Utilisation actuelle</h3>
                    
                    <div class="usage-item">
                        <div class="usage-header">
                            <span><i class="fas fa-folder"></i> Projets</span>
                            <span>
                                <?= $usage['projects_count'] ?> / 
                                <?= $plan_limits['max_projects'] > 0 ? $plan_limits['max_projects'] : '∞' ?>
                            </span>
                        </div>
                        <div class="usage-bar">
                            <?php 
                            $projects_percentage = $plan_limits['max_projects'] > 0 ? 
                                min(100, ($usage['projects_count'] / $plan_limits['max_projects']) * 100) : 0;
                            ?>
                            <div class="usage-progress <?= $projects_percentage > 80 ? 'high' : '' ?>" 
                                 style="width: <?= $projects_percentage ?>%"></div>
                        </div>
                    </div>
                    
                    <div class="usage-item">
                        <div class="usage-header">
                            <span><i class="fas fa-hdd"></i> Stockage</span>
                            <span>
                                <?= formatFileSize($usage['storage_used']) ?> / 
                                <?= $plan_limits['max_storage'] > 0 ? formatFileSize($plan_limits['max_storage']) : '∞' ?>
                            </span>
                        </div>
                        <div class="usage-bar">
                            <?php 
                            $storage_percentage = $plan_limits['max_storage'] > 0 ? 
                                min(100, ($usage['storage_used'] / $plan_limits['max_storage']) * 100) : 0;
                            ?>
                            <div class="usage-progress <?= $storage_percentage > 80 ? 'high' : '' ?>" 
                                 style="width: <?= $storage_percentage ?>%"></div>
                        </div>
                    </div>
                    
                    <div style="margin-top: var(--space-xl); text-align: center;">
                        <button class="btn btn-outline" onclick="refreshUsage()">
                            <i class="fas fa-sync-alt"></i>
                            Actualiser l'utilisation
                        </button>
                    </div>
                </div>
            </div>

            <!-- Onglet Facturation -->
            <div id="billing-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-receipt"></i> Historique de facturation</h3>
                    
                    <?php if ($subscription): ?>
                        <div class="card">
                            <div class="card-body">
                                <h4>Abonnement <?= htmlspecialchars($subscription['plan_name']) ?></h4>
                                <p>Status: <span class="badge <?= $subscription['status'] === 'active' ? 'success' : 'warning' ?>">
                                    <?= ucfirst($subscription['status']) ?>
                                </span></p>
                                <p>Période: <?= date('d/m/Y', strtotime($subscription['current_period_start'])) ?> - 
                                   <?= date('d/m/Y', strtotime($subscription['current_period_end'])) ?></p>
                                <p>Prochain paiement: 
                                   <strong>€<?= $subscription['billing_cycle'] === 'yearly' ? $subscription['price_yearly'] : $subscription['price_monthly'] ?></strong>
                                </p>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-receipt"></i>
                            <h3>Aucun abonnement actif</h3>
                            <p>Vous utilisez le plan gratuit</p>
                        </div>
                    <?php endif; ?>
                    
                    <div style="margin-top: var(--space-xl);">
                        <button class="btn btn-primary" onclick="showNotification('Fonction de paiement simulée ! 💳', 'info')">
                            <i class="fas fa-credit-card"></i>
                            Mettre à jour le mode de paiement
                        </button>
                    </div>
                </div>
            </div>

            <!-- Onglet Données -->
            <div id="data-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-download"></i> Export de données</h3>
                    <p>Exportez vos données personnelles selon le RGPD</p>
                    
                    <div class="export-options">
                        <div class="export-option selected" data-type="all">
                            <i class="fas fa-globe"></i>
                            <h4>Toutes les données</h4>
                            <p>Export complet</p>
                        </div>
                        <div class="export-option" data-type="projects">
                            <i class="fas fa-folder"></i>
                            <h4>Projets uniquement</h4>
                            <p>Vos créations</p>
                        </div>
                        <div class="export-option" data-type="user">
                            <i class="fas fa-user"></i>
                            <h4>Profil utilisateur</h4>
                            <p>Informations personnelles</p>
                        </div>
                    </div>
                    
                    <button class="btn btn-primary" onclick="exportData()">
                        <i class="fas fa-download"></i>
                        Générer l'export
                    </button>
                </div>
                
                <div class="danger-zone">
                    <h3><i class="fas fa-exclamation-triangle"></i> Zone de danger</h3>
                    <p>Les actions suivantes sont irréversibles. Procédez avec prudence.</p>
                    
                    <button class="btn btn-error" onclick="showDeleteAccountModal()">
                        <i class="fas fa-trash-alt"></i>
                        Supprimer mon compte
                    </button>
                </div>
            </div>

            <!-- Onglet Compte -->
            <div id="account-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-user-cog"></i> Gestion du compte</h3>
                    
                    <div class="grid grid-cols-2">
                        <div>
                            <h4>Informations du compte</h4>
                            <p><strong>Email:</strong> <?= htmlspecialchars($user['email']) ?></p>
                            <p><strong>Membre depuis:</strong> <?= date('d/m/Y', strtotime($user['created_at'])) ?></p>
                            <p><strong>Dernière connexion:</strong> <?= $user['last_login'] ? date('d/m/Y H:i', strtotime($user['last_login'])) : 'Jamais' ?></p>
                            <p><strong>Statut:</strong> <span class="badge success"><?= ucfirst($user['status']) ?></span></p>
                        </div>
                        <div>
                            <h4>Actions rapides</h4>
                            <div style="display: flex; flex-direction: column; gap: var(--space-sm);">
                                <a href="profile.php" class="btn btn-outline">
                                    <i class="fas fa-user-edit"></i>
                                    Modifier le profil
                                </a>
                                <button class="btn btn-outline" onclick="logoutAllSessions()">
                                    <i class="fas fa-sign-out-alt"></i>
                                    Déconnecter toutes les sessions
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Modal de suppression de compte -->
    <div id="deleteAccountModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-exclamation-triangle"></i> Supprimer le compte</h3>
                <button class="modal-close" onclick="closeModal('deleteAccountModal')">&times;</button>
            </div>
            
            <div class="modal-body">
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-triangle"></i>
                    <div>
                        <strong>Attention !</strong> Cette action est irréversible.
                        <br>Toutes vos données seront définitivement supprimées.
                    </div>
                </div>
                
                <form onsubmit="deleteAccount(event)">
                    <div class="form-group">
                        <label class="form-label">Confirmez avec votre mot de passe</label>
                        <input type="password" name="password" class="form-control" 
                               placeholder="Saisissez votre mot de passe" required>
                    </div>
                    
                    <div class="form-group">
                        <label style="display: flex; align-items: center; gap: var(--space-sm); cursor: pointer;">
                            <input type="checkbox" required>
                            <span>Je comprends que cette action est irréversible</span>
                        </label>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" onclick="closeModal('deleteAccountModal')">
                            Annuler
                        </button>
                        <button type="submit" class="btn btn-error">
                            <i class="fas fa-trash-alt"></i>
                            Supprimer définitivement
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script>
        // ========== VARIABLES GLOBALES ==========
        let currentBillingCycle = 'monthly';
        let selectedExportType = 'all';
        
        // ========== GESTION DES ONGLETS ==========
        function showTab(tabName) {
            // Masquer tous les contenus
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Désactiver tous les onglets
            document.querySelectorAll('.settings-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Afficher le contenu sélectionné
            document.getElementById(tabName + '-tab').classList.add('active');
            
            // Activer l'onglet sélectionné
            event.target.classList.add('active');
        }
        
        // ========== GESTION FACTURATION ==========
        function toggleBilling() {
            const billingSwitch = document.getElementById('billingSwitch');
            const monthlyLabel = document.getElementById('monthlyLabel');
            const yearlyLabel = document.getElementById('yearlyLabel');
            const priceElements = document.querySelectorAll('.amount');
            const periodElements = document.querySelectorAll('.period');
            
            if (currentBillingCycle === 'monthly') {
                currentBillingCycle = 'yearly';
                billingSwitch.classList.add('yearly');
                monthlyLabel.classList.remove('active');
                yearlyLabel.classList.add('active');
                
                priceElements.forEach(el => {
                    el.textContent = el.dataset.yearly;
                });
                
                periodElements.forEach(el => {
                    el.textContent = '/an';
                });
            } else {
                currentBillingCycle = 'monthly';
                billingSwitch.classList.remove('yearly');
                monthlyLabel.classList.add('active');
                yearlyLabel.classList.remove('active');
                
                priceElements.forEach(el => {
                    el.textContent = el.dataset.monthly;
                });
                
                periodElements.forEach(el => {
                    el.textContent = '/mois';
                });
            }
        }
        
        // ========== UPGRADE PLAN ==========
        async function upgradePlan(plan) {
            if (!confirm(`Êtes-vous sûr de vouloir changer pour le plan ${plan.toUpperCase()} ?`)) {
                return;
            }
            
            showNotification('Traitement en cours...', 'info');
            
            try {
                const response = await fetch('settings.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=simulate_upgrade&plan=${plan}&billing_cycle=${currentBillingCycle}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    setTimeout(() => location.reload(), 2000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors du changement de plan', 'error');
            }
        }
        
        // ========== ACTUALISER UTILISATION ==========
        async function refreshUsage() {
            showNotification('Actualisation en cours...', 'info');
            
            try {
                const response = await fetch('settings.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'ajax_action=calculate_usage'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Mettre à jour les barres de progression
                    updateUsageBars(result.percentage);
                    showNotification('Utilisation mise à jour', 'success');
                } else {
                    showNotification('Erreur lors de l\'actualisation', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        function updateUsageBars(percentages) {
            const progressBars = document.querySelectorAll('.usage-progress');
            progressBars[0].style.width = `${percentages.projects}%`;
            progressBars[1].style.width = `${percentages.storage}%`;
            
            // Ajouter la classe 'high' si > 80%
            progressBars.forEach((bar, index) => {
                const percentage = Object.values(percentages)[index];
                if (percentage > 80) {
                    bar.classList.add('high');
                } else {
                    bar.classList.remove('high');
                }
            });
        }
        
        // ========== EXPORT DE DONNÉES ==========
        function selectExportOption(type) {
            document.querySelectorAll('.export-option').forEach(option => {
                option.classList.remove('selected');
            });
            
            event.target.closest('.export-option').classList.add('selected');
            selectedExportType = type;
        }
        
        // Ajouter les event listeners pour les options d'export
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.export-option').forEach(option => {
                option.addEventListener('click', function() {
                    selectExportOption(this.dataset.type);
                });
            });
        });
        
        async function exportData() {
            showNotification('Génération de l\'export...', 'info');
            
            try {
                const response = await fetch('settings.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=export_data&export_type=${selectedExportType}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(`${result.message} (${result.data_size})`, 'success');
                    // Ici, on pourrait déclencher le téléchargement du fichier
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de l\'export', 'error');
            }
        }
        
        // ========== SUPPRESSION DE COMPTE ==========
        function showDeleteAccountModal() {
            document.getElementById('deleteAccountModal').style.display = 'block';
        }
        
        async function deleteAccount(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            formData.append('ajax_action', 'delete_account');
            
            if (!confirm('DERNIÈRE CONFIRMATION\n\nÊtes-vous absolument certain de vouloir supprimer votre compte ?\n\nCette action est IRRÉVERSIBLE !')) {
                return;
            }
            
            try {
                showNotification('Suppression en cours...', 'warning');
                
                const response = await fetch('settings.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    setTimeout(() => {
                        if (result.redirect) {
                            window.location.href = result.redirect;
                        }
                    }, 2000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la suppression', 'error');
            }
        }
        
        // ========== ACTIONS DIVERSES ==========
        function logoutAllSessions() {
            if (confirm('Déconnecter toutes les sessions actives ?')) {
                showNotification('Toutes les sessions ont été fermées (simulation)', 'success');
                setTimeout(() => {
                    window.location.href = 'login.php';
                }, 2000);
            }
        }
        
        // ========== UTILITAIRES ==========
        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }
        
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les modals en cliquant à l'extérieur
        window.onclick = function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.style.display = 'none';
            }
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // ========== SYSTÈME DE NOTIFICATIONS ==========
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.75rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.2rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.8;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, type === 'error' ? 8000 : 5000);
        }
        
        // ========== STYLES DYNAMIQUES ==========
        const style = document.createElement('style');
        style.textContent = `
            .badge {
                padding: 0.25rem 0.5rem;
                border-radius: 0.25rem;
                font-size: 0.75rem;
                font-weight: 600;
                text-transform: uppercase;
            }
            
            .badge.success {
                background: rgba(16, 185, 129, 0.2);
                color: #059669;
            }
            
            .badge.warning {
                background: rgba(245, 158, 11, 0.2);
                color: #d97706;
            }
            
            .badge.error {
                background: rgba(239, 68, 68, 0.2);
                color: #dc2626;
            }
            
            #monthlyLabel.active,
            #yearlyLabel.active {
                font-weight: 700;
                color: var(--primary-color);
            }
        `;
        document.head.appendChild(style);
        
        // Message de bienvenue
        document.addEventListener('DOMContentLoaded', function() {
            showNotification('Paramètres chargés ! ⚙️', 'success');
        });
    </script>
</body>
</html>