<?php
/**
 * Flying Book Creator v2.0 - Moteur de Rendu HTML Intelligent
 * Génère les Flying Books en HTML5 avec toutes les fonctionnalités avancées
 */

require_once './env/.env.php';

class FlyingBookRenderEngine {
    
    private $pdo;
    private $project;
    private $slides;
    private $config;
    
    public function __construct() {
        $this->pdo = getDatabaseConnection();
    }
    
    /**
     * Générer un Flying Book complet
     */
    public function generateFlyingBook($projectId, $options = []) {
        try {
            // Récupérer le projet
            $stmt = $this->pdo->prepare("
                SELECT p.*, pc.advanced_config 
                FROM flying_book_projects p
                LEFT JOIN flying_book_project_configs pc ON p.id = pc.project_id
                WHERE p.id = ? AND p.status != 'deleted'
            ");
            $stmt->execute([$projectId]);
            $this->project = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$this->project) {
                throw new Exception("Projet non trouvé");
            }
            
            // Récupérer les slides
            $stmt = $this->pdo->prepare("
                SELECT * FROM flying_book_slides 
                WHERE project_id = ? AND is_active = 1 
                ORDER BY display_order ASC, slide_number ASC
            ");
            $stmt->execute([$projectId]);
            $this->slides = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Configuration avancée
            $this->config = $this->project['advanced_config'] ? 
                json_decode($this->project['advanced_config'], true) : [];
            
            // Générer le HTML
            $html = $this->buildHTML($options);
            
            // Sauvegarder le fichier
            $outputPath = $this->saveGeneratedBook($html, $options);
            
            // Enregistrer l'analytics de génération
            $this->logGeneration($projectId, $outputPath);
            
            return [
                'success' => true,
                'path' => $outputPath,
                'url' => $this->getPublicURL($outputPath),
                'stats' => $this->getGenerationStats()
            ];
            
        } catch (Exception $e) {
            logError("Erreur génération Flying Book", [
                'project_id' => $projectId,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Construire le HTML complet
     */
    private function buildHTML($options = []) {
        $theme = $this->project['style_theme'] ?? 'moderne';
        $transition = $this->project['transition_type'] ?? '3d';
        $navigation = $this->project['navigation_type'] ?? 'all';
        
        $html = '<!DOCTYPE html>
<html lang="fr" data-theme="' . $theme . '">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($this->project['title'] ?: $this->project['project_name']) . '</title>
    
    <!-- Métadonnées SEO -->
    <meta name="description" content="' . htmlspecialchars($this->project['description']) . '">
    <meta name="author" content="Flying Book Creator">
    <meta property="og:title" content="' . htmlspecialchars($this->project['title']) . '">
    <meta property="og:description" content="' . htmlspecialchars($this->project['description']) . '">
    <meta property="og:type" content="presentation">
    
    <!-- Styles intégrés -->
    <style>
        ' . $this->getThemeCSS($theme) . '
        ' . $this->getTransitionCSS($transition) . '
        ' . $this->getNavigationCSS($navigation) . '
        ' . $this->getResponsiveCSS() . '
    </style>
    
    <!-- Analytics si activé -->
    ' . ($this->project['analytics_enabled'] ? $this->getAnalyticsScript() : '') . '
</head>
<body class="flying-book-theme-' . $theme . '">
    
    <!-- Watermark si activé -->
    ' . ($this->project['watermark_enabled'] ? $this->getWatermark() : '') . '
    
    <!-- Loading Screen -->
    <div id="loadingScreen" class="loading-screen">
        <div class="loading-spinner"></div>
        <p>Chargement de votre Flying Book...</p>
    </div>
    
    <!-- Navigation principale -->
    ' . $this->buildNavigation($navigation) . '
    
    <!-- Container principal -->
    <div id="flyingBookContainer" class="flying-book-container" data-transition="' . $transition . '">
        ' . $this->buildSlides() . '
    </div>
    
    <!-- Contrôles de présentation -->
    ' . $this->buildControls() . '
    
    <!-- Mini-map si activée -->
    ' . ($this->config['show_minimap'] ?? false ? $this->buildMiniMap() : '') . '
    
    <!-- Progress bar -->
    <div class="progress-bar">
        <div class="progress-fill" id="progressFill"></div>
    </div>
    
    <!-- Scripts JavaScript -->
    <script>
        ' . $this->getFlyingBookJS() . '
        ' . $this->getInteractionJS() . '
        ' . ($this->project['analytics_enabled'] ? $this->getAnalyticsJS() : '') . '
    </script>
    
</body>
</html>';

        return $html;
    }
    
    /**
     * Construire les slides
     */
    private function buildSlides() {
        $slidesHTML = '';
        $slideCount = count($this->slides);
        
        foreach ($this->slides as $index => $slide) {
            $isActive = $index === 0 ? 'active' : '';
            $slideId = 'slide-' . $slide['id'];
            
            $slidesHTML .= '<div class="slide ' . $isActive . '" id="' . $slideId . '" data-slide="' . ($index + 1) . '">';
            
            // Contenu selon le type de slide
            switch ($slide['media_type']) {
                case 'image':
                    $slidesHTML .= $this->buildImageSlide($slide);
                    break;
                case 'video':
                    $slidesHTML .= $this->buildVideoSlide($slide);
                    break;
                case 'audio_image':
                    $slidesHTML .= $this->buildAudioImageSlide($slide);
                    break;
                case 'text_image':
                    $slidesHTML .= $this->buildTextImageSlide($slide);
                    break;
                case 'interactive':
                    $slidesHTML .= $this->buildInteractiveSlide($slide);
                    break;
                default:
                    $slidesHTML .= $this->buildDefaultSlide($slide);
            }
            
            // Text overlay si présent
            if (!empty($slide['text_overlay'])) {
                $slidesHTML .= '<div class="slide-overlay">' . $slide['text_overlay'] . '</div>';
            }
            
            // Éléments interactifs si présents
            if (!empty($slide['interactive_elements'])) {
                $slidesHTML .= $this->buildInteractiveElements($slide['interactive_elements']);
            }
            
            $slidesHTML .= '</div>';
        }
        
        return $slidesHTML;
    }
    
    /**
     * Slide image
     */
    private function buildImageSlide($slide) {
        return '<div class="slide-content slide-image">
            <img src="' . $this->getMediaURL($slide['media_file']) . '" 
                 alt="' . htmlspecialchars($slide['slide_title']) . '"
                 class="slide-media"
                 loading="lazy" />
        </div>';
    }
    
    /**
     * Slide vidéo avec contrôles avancés
     */
    private function buildVideoSlide($slide) {
        $videoSettings = json_decode($slide['video_settings'] ?? '{}', true);
        $autoplay = $videoSettings['autoplay'] ?? false;
        $muted = $videoSettings['muted'] ?? true;
        $loop = $videoSettings['loop'] ?? false;
        
        return '<div class="slide-content slide-video">
            <video class="slide-media" 
                   ' . ($autoplay ? 'autoplay' : '') . '
                   ' . ($muted ? 'muted' : '') . '
                   ' . ($loop ? 'loop' : '') . '
                   controls
                   preload="metadata">
                <source src="' . $this->getMediaURL($slide['media_file']) . '" type="video/mp4">
                Votre navigateur ne supporte pas la vidéo.
            </video>
        </div>';
    }
    
    /**
     * Slide audio + image
     */
    private function buildAudioImageSlide($slide) {
        return '<div class="slide-content slide-audio-image">
            <img src="' . $this->getMediaURL($slide['media_file']) . '" 
                 alt="' . htmlspecialchars($slide['slide_title']) . '"
                 class="slide-media" />
            ' . (!empty($slide['background_audio']) ? 
                '<audio class="slide-audio" controls autoplay loop>
                    <source src="' . $this->getMediaURL($slide['background_audio']) . '" type="audio/mpeg">
                </audio>' : '') . '
        </div>';
    }
    
    /**
     * Slide texte + image
     */
    private function buildTextImageSlide($slide) {
        return '<div class="slide-content slide-text-image">
            <div class="slide-image-container">
                <img src="' . $this->getMediaURL($slide['media_file']) . '" 
                     alt="' . htmlspecialchars($slide['slide_title']) . '"
                     class="slide-media" />
            </div>
            <div class="slide-text-container">
                <h2>' . htmlspecialchars($slide['slide_title']) . '</h2>
                <div class="slide-text-content">' . $slide['text_overlay'] . '</div>
            </div>
        </div>';
    }
    
    /**
     * Slide interactive avec éléments avancés
     */
    private function buildInteractiveSlide($slide) {
        $elements = json_decode($slide['interactive_elements'] ?? '[]', true);
        
        $html = '<div class="slide-content slide-interactive">';
        $html .= '<img src="' . $this->getMediaURL($slide['media_file']) . '" class="slide-background" />';
        
        foreach ($elements as $element) {
            switch ($element['type']) {
                case 'hotspot':
                    $html .= $this->buildHotspot($element);
                    break;
                case 'button':
                    $html .= $this->buildInteractiveButton($element);
                    break;
                case 'form':
                    $html .= $this->buildInteractiveForm($element);
                    break;
                case 'quiz':
                    $html .= $this->buildQuizElement($element);
                    break;
            }
        }
        
        $html .= '</div>';
        return $html;
    }
    
    /**
     * Slide par défaut
     */
    private function buildDefaultSlide($slide) {
        return '<div class="slide-content slide-default">
            <div class="slide-placeholder">
                <i class="fas fa-image"></i>
                <p>Contenu de la slide</p>
            </div>
        </div>';
    }
    
    /**
     * Construire hotspot interactif
     */
    private function buildHotspot($element) {
        return '<div class="hotspot" 
                     style="left: ' . ($element['x'] ?? 50) . '%; top: ' . ($element['y'] ?? 50) . '%;"
                     data-content="' . htmlspecialchars($element['content'] ?? '') . '">
            <div class="hotspot-pulse"></div>
            <div class="hotspot-tooltip">' . ($element['tooltip'] ?? 'Cliquez pour plus d\'infos') . '</div>
        </div>';
    }
    
    /**
     * Construire bouton interactif
     */
    private function buildInteractiveButton($element) {
        return '<button class="interactive-btn" 
                        style="left: ' . ($element['x'] ?? 50) . '%; top: ' . ($element['y'] ?? 50) . '%;"
                        onclick="' . ($element['action'] ?? '') . '">
            ' . ($element['text'] ?? 'Cliquez ici') . '
        </button>';
    }
    
    /**
     * Navigation selon le type
     */
    private function buildNavigation($type) {
        $html = '<nav class="flying-book-nav nav-' . $type . '">';
        
        if (in_array($type, ['buttons', 'all'])) {
            $html .= '<button class="nav-btn nav-prev" onclick="FlyingBook.previousSlide()">
                <i class="fas fa-chevron-left"></i>
            </button>
            <button class="nav-btn nav-next" onclick="FlyingBook.nextSlide()">
                <i class="fas fa-chevron-right"></i>
            </button>';
        }
        
        if (in_array($type, ['thumbnails', 'all'])) {
            $html .= '<div class="nav-thumbnails">';
            foreach ($this->slides as $index => $slide) {
                $html .= '<div class="nav-thumb" onclick="FlyingBook.goToSlide(' . ($index + 1) . ')">
                    <img src="' . $this->getMediaURL($slide['media_thumbnail'] ?: $slide['media_file']) . '" />
                    <span>' . ($index + 1) . '</span>
                </div>';
            }
            $html .= '</div>';
        }
        
        $html .= '</nav>';
        return $html;
    }
    
    /**
     * Contrôles de présentation
     */
    private function buildControls() {
        return '<div class="presentation-controls">
            <button class="control-btn" onclick="FlyingBook.toggleFullscreen()" title="Plein écran">
                <i class="fas fa-expand"></i>
            </button>
            <button class="control-btn" onclick="FlyingBook.toggleAutoplay()" title="Lecture auto">
                <i class="fas fa-play"></i>
            </button>
            <button class="control-btn" onclick="FlyingBook.showHelp()" title="Aide">
                <i class="fas fa-question"></i>
            </button>
        </div>';
    }
    
    /**
     * Mini-map de navigation
     */
    private function buildMiniMap() {
        $html = '<div class="minimap">
            <div class="minimap-slides">';
        
        foreach ($this->slides as $index => $slide) {
            $html .= '<div class="minimap-slide" data-slide="' . ($index + 1) . '" onclick="FlyingBook.goToSlide(' . ($index + 1) . ')"></div>';
        }
        
        $html .= '</div>
        </div>';
        
        return $html;
    }
    
    /**
     * CSS du thème
     */
    private function getThemeCSS($theme) {
        $themes = [
            'moderne' => '
                :root {
                    --primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    --secondary: #f093fb;
                    --accent: #c471ed;
                    --text: #2d3748;
                    --bg: #f7fafc;
                }
                .flying-book-container {
                    background: var(--bg);
                    font-family: "Segoe UI", system-ui, sans-serif;
                }
                .slide {
                    background: white;
                    border-radius: 20px;
                    box-shadow: 0 20px 60px rgba(0,0,0,0.1);
                    backdrop-filter: blur(10px);
                }
            ',
            'classique' => '
                :root {
                    --primary: #2c3e50;
                    --secondary: #3498db;
                    --accent: #e74c3c;
                    --text: #2c3e50;
                    --bg: #ecf0f1;
                }
                .flying-book-container {
                    background: var(--bg);
                    font-family: Georgia, serif;
                }
                .slide {
                    background: white;
                    border: 2px solid var(--primary);
                    border-radius: 10px;
                }
            ',
            'corporate' => '
                :root {
                    --primary: #34495e;
                    --secondary: #2c3e50;
                    --accent: #3498db;
                    --text: #2c3e50;
                    --bg: #bdc3c7;
                }
                .flying-book-container {
                    background: linear-gradient(135deg, #bdc3c7, #2c3e50);
                    font-family: "Arial", sans-serif;
                }
                .slide {
                    background: white;
                    border-radius: 5px;
                    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
                }
            '
        ];
        
        $baseCSS = '
            * { margin: 0; padding: 0; box-sizing: border-box; }
            
            .flying-book-container {
                width: 100vw;
                height: 100vh;
                overflow: hidden;
                position: relative;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .slide {
                position: absolute;
                width: 90vw;
                height: 90vh;
                max-width: 1200px;
                max-height: 800px;
                opacity: 0;
                transform: scale(0.8);
                transition: all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 2rem;
            }
            
            .slide.active {
                opacity: 1;
                transform: scale(1);
                z-index: 10;
            }
            
            .slide-media {
                max-width: 100%;
                max-height: 100%;
                object-fit: contain;
                border-radius: 10px;
            }
            
            .slide-content {
                width: 100%;
                height: 100%;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .slide-text-image {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 2rem;
                align-items: center;
            }
            
            .slide-text-container h2 {
                font-size: 2.5rem;
                margin-bottom: 1rem;
                color: var(--text);
            }
            
            .slide-text-content {
                font-size: 1.2rem;
                line-height: 1.6;
                color: var(--text);
            }
            
            .nav-btn {
                position: fixed;
                top: 50%;
                transform: translateY(-50%);
                z-index: 1000;
                background: rgba(255,255,255,0.9);
                border: none;
                width: 60px;
                height: 60px;
                border-radius: 50%;
                cursor: pointer;
                font-size: 1.5rem;
                transition: all 0.3s ease;
                box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            }
            
            .nav-btn:hover {
                background: white;
                transform: translateY(-50%) scale(1.1);
            }
            
            .nav-prev { left: 2rem; }
            .nav-next { right: 2rem; }
            
            .nav-thumbnails {
                position: fixed;
                bottom: 2rem;
                left: 50%;
                transform: translateX(-50%);
                display: flex;
                gap: 0.5rem;
                z-index: 1000;
            }
            
            .nav-thumb {
                width: 60px;
                height: 40px;
                background: white;
                border-radius: 8px;
                overflow: hidden;
                cursor: pointer;
                position: relative;
                box-shadow: 0 2px 8px rgba(0,0,0,0.1);
                transition: all 0.3s ease;
            }
            
            .nav-thumb:hover {
                transform: scale(1.1);
            }
            
            .nav-thumb img {
                width: 100%;
                height: 100%;
                object-fit: cover;
            }
            
            .nav-thumb span {
                position: absolute;
                bottom: 2px;
                right: 4px;
                background: rgba(0,0,0,0.7);
                color: white;
                font-size: 0.7rem;
                padding: 2px 4px;
                border-radius: 3px;
            }
            
            .progress-bar {
                position: fixed;
                bottom: 0;
                left: 0;
                width: 100%;
                height: 4px;
                background: rgba(255,255,255,0.3);
                z-index: 1000;
            }
            
            .progress-fill {
                height: 100%;
                background: var(--primary);
                width: 0%;
                transition: width 0.3s ease;
            }
            
            .loading-screen {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: var(--primary);
                display: flex;
                flex-direction: column;
                align-items: center;
                justify-content: center;
                z-index: 9999;
                color: white;
            }
            
            .loading-spinner {
                width: 50px;
                height: 50px;
                border: 3px solid rgba(255,255,255,0.3);
                border-top: 3px solid white;
                border-radius: 50%;
                animation: spin 1s linear infinite;
                margin-bottom: 1rem;
            }
            
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            
            .watermark {
                position: fixed;
                bottom: 1rem;
                right: 1rem;
                background: rgba(0,0,0,0.7);
                color: white;
                padding: 0.5rem 1rem;
                border-radius: 20px;
                font-size: 0.8rem;
                z-index: 1001;
            }
            
            .presentation-controls {
                position: fixed;
                top: 2rem;
                right: 2rem;
                display: flex;
                gap: 0.5rem;
                z-index: 1000;
            }
            
            .control-btn {
                width: 40px;
                height: 40px;
                background: rgba(255,255,255,0.9);
                border: none;
                border-radius: 50%;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                transition: all 0.3s ease;
                box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            }
            
            .control-btn:hover {
                background: white;
                transform: scale(1.1);
            }
            
            .hotspot {
                position: absolute;
                width: 20px;
                height: 20px;
                cursor: pointer;
                z-index: 100;
            }
            
            .hotspot-pulse {
                width: 100%;
                height: 100%;
                background: rgba(255,255,255,0.8);
                border: 2px solid var(--primary);
                border-radius: 50%;
                animation: pulse 2s infinite;
            }
            
            @keyframes pulse {
                0% { transform: scale(1); opacity: 1; }
                100% { transform: scale(2); opacity: 0; }
            }
            
            .hotspot-tooltip {
                position: absolute;
                bottom: 30px;
                left: 50%;
                transform: translateX(-50%);
                background: rgba(0,0,0,0.8);
                color: white;
                padding: 0.5rem 1rem;
                border-radius: 4px;
                font-size: 0.8rem;
                white-space: nowrap;
                opacity: 0;
                transition: opacity 0.3s ease;
            }
            
            .hotspot:hover .hotspot-tooltip {
                opacity: 1;
            }
            
            .interactive-btn {
                position: absolute;
                background: var(--primary);
                color: white;
                border: none;
                padding: 0.75rem 1.5rem;
                border-radius: 25px;
                cursor: pointer;
                font-weight: 600;
                transition: all 0.3s ease;
                box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            }
            
            .interactive-btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 20px rgba(0,0,0,0.3);
            }
            
            .minimap {
                position: fixed;
                top: 2rem;
                left: 2rem;
                background: rgba(255,255,255,0.9);
                border-radius: 8px;
                padding: 1rem;
                z-index: 1000;
            }
            
            .minimap-slides {
                display: flex;
                gap: 2px;
            }
            
            .minimap-slide {
                width: 8px;
                height: 6px;
                background: rgba(0,0,0,0.3);
                cursor: pointer;
                transition: all 0.3s ease;
            }
            
            .minimap-slide.active {
                background: var(--primary);
            }
            
            .minimap-slide:hover {
                transform: scaleY(2);
            }
            
            /* Responsive */
            @media (max-width: 768px) {
                .slide {
                    width: 95vw;
                    height: 95vh;
                    padding: 1rem;
                }
                
                .slide-text-image {
                    grid-template-columns: 1fr;
                    gap: 1rem;
                }
                
                .slide-text-container h2 {
                    font-size: 1.8rem;
                }
                
                .nav-btn {
                    width: 50px;
                    height: 50px;
                    font-size: 1.2rem;
                }
                
                .nav-prev { left: 1rem; }
                .nav-next { right: 1rem; }
                
                .nav-thumbnails {
                    bottom: 1rem;
                }
                
                .presentation-controls {
                    top: 1rem;
                    right: 1rem;
                }
            }
        ';
        
        return $baseCSS . ($themes[$theme] ?? $themes['moderne']);
    }
    
    /**
     * CSS des transitions
     */
    private function getTransitionCSS($transition) {
        $transitions = [
            'fade' => '
                .slide { transition: opacity 0.8s ease; }
            ',
            'slide' => '
                .slide { 
                    transition: transform 0.8s ease; 
                    transform: translateX(100%);
                }
                .slide.active { transform: translateX(0); }
            ',
            '3d' => '
                .flying-book-container { perspective: 1000px; }
                .slide { 
                    transition: all 0.8s ease;
                    transform: rotateY(90deg) scale(0.8);
                }
                .slide.active { 
                    transform: rotateY(0deg) scale(1);
                }
            '
        ];
        
        return $transitions[$transition] ?? $transitions['3d'];
    }
    
    /**
     * CSS de navigation
     */
    private function getNavigationCSS($navigation) {
        if ($navigation === 'keyboard') {
            return '.nav-btn, .nav-thumbnails { display: none; }';
        }
        
        return '';
    }
    
    /**
     * CSS responsive
     */
    private function getResponsiveCSS() {
        return '
            @media (max-width: 480px) {
                .slide-text-container h2 {
                    font-size: 1.5rem;
                }
                
                .slide-text-content {
                    font-size: 1rem;
                }
                
                .nav-thumbnails {
                    flex-wrap: wrap;
                    max-width: 90vw;
                }
                
                .hotspot-tooltip {
                    font-size: 0.7rem;
                }
            }
        ';
    }
    
    /**
     * JavaScript principal
     */
    private function getFlyingBookJS() {
        return '
        class FlyingBookEngine {
            constructor() {
                this.currentSlide = 1;
                this.totalSlides = ' . count($this->slides) . ';
                this.isTransitioning = false;
                this.isAutoplay = false;
                this.autoplayInterval = null;
                this.analytics = {
                    startTime: Date.now(),
                    slideViews: {},
                    interactions: []
                };
                
                this.init();
            }
            
            init() {
                // Masquer le loading
                setTimeout(() => {
                    document.getElementById("loadingScreen").style.display = "none";
                }, 1000);
                
                // Événements clavier
                document.addEventListener("keydown", (e) => {
                    switch(e.key) {
                        case "ArrowRight":
                        case " ":
                            e.preventDefault();
                            this.nextSlide();
                            break;
                        case "ArrowLeft":
                            e.preventDefault();
                            this.previousSlide();
                            break;
                        case "Home":
                            e.preventDefault();
                            this.goToSlide(1);
                            break;
                        case "End":
                            e.preventDefault();
                            this.goToSlide(this.totalSlides);
                            break;
                        case "Escape":
                            if (document.fullscreenElement) {
                                document.exitFullscreen();
                            }
                            break;
                    }
                });
                
                // Événements tactiles
                this.initTouchEvents();
                
                // Mise à jour progress bar
                this.updateProgress();
                
                // Analytics - première vue
                this.trackSlideView(1);
            }
            
            nextSlide() {
                if (this.currentSlide < this.totalSlides && !this.isTransitioning) {
                    this.goToSlide(this.currentSlide + 1);
                }
            }
            
            previousSlide() {
                if (this.currentSlide > 1 && !this.isTransitioning) {
                    this.goToSlide(this.currentSlide - 1);
                }
            }
            
            goToSlide(slideNumber) {
                if (slideNumber === this.currentSlide || this.isTransitioning) return;
                
                this.isTransitioning = true;
                
                // Désactiver slide actuelle
                const currentSlideEl = document.querySelector(".slide.active");
                if (currentSlideEl) {
                    currentSlideEl.classList.remove("active");
                }
                
                // Activer nouvelle slide
                const newSlideEl = document.querySelector(`[data-slide="${slideNumber}"]`);
                if (newSlideEl) {
                    newSlideEl.classList.add("active");
                }
                
                // Mettre à jour minimap
                document.querySelectorAll(".minimap-slide").forEach((slide, index) => {
                    slide.classList.toggle("active", index + 1 === slideNumber);
                });
                
                this.currentSlide = slideNumber;
                this.updateProgress();
                this.trackSlideView(slideNumber);
                
                // Réactiver transitions après animation
                setTimeout(() => {
                    this.isTransitioning = false;
                }, 800);
            }
            
            updateProgress() {
                const progress = (this.currentSlide / this.totalSlides) * 100;
                document.getElementById("progressFill").style.width = progress + "%";
            }
            
            toggleFullscreen() {
                if (!document.fullscreenElement) {
                    document.documentElement.requestFullscreen();
                } else {
                    document.exitFullscreen();
                }
            }
            
            toggleAutoplay() {
                this.isAutoplay = !this.isAutoplay;
                
                if (this.isAutoplay) {
                    this.autoplayInterval = setInterval(() => {
                        if (this.currentSlide < this.totalSlides) {
                            this.nextSlide();
                        } else {
                            this.goToSlide(1);
                        }
                    }, 5000);
                } else {
                    clearInterval(this.autoplayInterval);
                }
                
                // Changer icône
                const playBtn = document.querySelector(".presentation-controls .control-btn");
                const icon = playBtn.querySelector("i");
                icon.className = this.isAutoplay ? "fas fa-pause" : "fas fa-play";
            }
            
            showHelp() {
                alert("Contrôles:\\n\\n" +
                      "← → : Navigation\\n" +
                      "Espace : Slide suivante\\n" +
                      "Home/End : Première/Dernière slide\\n" +
                      "Échap : Quitter plein écran\\n\\n" +
                      "Vous pouvez aussi utiliser les boutons et miniatures.");
            }
            
            trackSlideView(slideNumber) {
                const now = Date.now();
                if (!this.analytics.slideViews[slideNumber]) {
                    this.analytics.slideViews[slideNumber] = {
                        firstView: now,
                        totalTime: 0,
                        viewCount: 0
                    };
                }
                
                this.analytics.slideViews[slideNumber].viewCount++;
                this.analytics.slideViews[slideNumber].lastView = now;
                
                // Envoyer analytics si activé
                if (typeof this.sendAnalytics === "function") {
                    this.sendAnalytics("slide_view", {
                        slide: slideNumber,
                        timestamp: now
                    });
                }
            }
            
            initTouchEvents() {
                let startX = 0;
                let startY = 0;
                
                document.addEventListener("touchstart", (e) => {
                    startX = e.touches[0].clientX;
                    startY = e.touches[0].clientY;
                });
                
                document.addEventListener("touchend", (e) => {
                    const endX = e.changedTouches[0].clientX;
                    const endY = e.changedTouches[0].clientY;
                    
                    const diffX = startX - endX;
                    const diffY = startY - endY;
                    
                    // Swipe horizontal dominant
                    if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > 50) {
                        if (diffX > 0) {
                            this.nextSlide(); // Swipe left = slide suivante
                        } else {
                            this.previousSlide(); // Swipe right = slide précédente
                        }
                    }
                });
            }
        }
        
        // Initialiser le Flying Book
        const FlyingBook = new FlyingBookEngine();
        
        // API publique
        window.FlyingBook = FlyingBook;
        ';
    }
    
    /**
     * JavaScript d'interaction
     */
    private function getInteractionJS() {
        return '
        // Interactions avec les hotspots
        document.addEventListener("click", function(e) {
            if (e.target.closest(".hotspot")) {
                const hotspot = e.target.closest(".hotspot");
                const content = hotspot.dataset.content;
                
                if (content) {
                    // Afficher contenu dans modal ou popup
                    showHotspotContent(content);
                }
                
                // Analytics
                if (typeof FlyingBook.sendAnalytics === "function") {
                    FlyingBook.sendAnalytics("hotspot_click", {
                        slide: FlyingBook.currentSlide,
                        hotspot: content
                    });
                }
            }
        });
        
        function showHotspotContent(content) {
            const modal = document.createElement("div");
            modal.className = "hotspot-modal";
            modal.innerHTML = `
                <div class="hotspot-modal-content">
                    <button class="hotspot-modal-close" onclick="this.parentElement.parentElement.remove()">×</button>
                    <div class="hotspot-modal-body">${content}</div>
                </div>
            `;
            
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.8);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 2000;
            `;
            
            document.body.appendChild(modal);
        }
        ';
    }
    
    /**
     * Obtenir l'URL publique du média
     */
    private function getMediaURL($mediaFile) {
        if (empty($mediaFile)) return '';
        
        // Si c'est déjà une URL complète
        if (strpos($mediaFile, 'http') === 0) {
            return $mediaFile;
        }
        
        // Construire l'URL relative
        return BASE_URL . '/document/projects/' . $this->project['project_folder'] . '/' . $mediaFile;
    }
    
    /**
     * Sauvegarder le Flying Book généré
     */
    private function saveGeneratedBook($html, $options) {
        $projectFolder = FLYING_BOOK_PROJECTS_PATH . $this->project['project_folder'];
        
        if (!is_dir($projectFolder)) {
            mkdir($projectFolder, 0755, true);
        }
        
        $filename = $options['filename'] ?? 'index.html';
        $outputPath = $projectFolder . '/' . $filename;
        
        if (file_put_contents($outputPath, $html) === false) {
            throw new Exception("Impossible de sauvegarder le fichier");
        }
        
        return $outputPath;
    }
    
    /**
     * Obtenir l'URL publique du Flying Book
     */
    private function getPublicURL($filePath) {
        $relativePath = str_replace($_SERVER['DOCUMENT_ROOT'], '', $filePath);
        return BASE_URL . $relativePath;
    }
    
    /**
     * Analytics script si activé
     */
    private function getAnalyticsScript() {
        return '<script>
            (function() {
                window.flyingBookAnalytics = {
                    projectId: ' . $this->project['id'] . ',
                    sessionId: "' . session_id() . '",
                    visitorId: "' . (session_id() . '_' . time()) . '"
                };
            })();
        </script>';
    }
    
    /**
     * JavaScript Analytics
     */
    private function getAnalyticsJS() {
        return '
        FlyingBook.sendAnalytics = function(event, data) {
            fetch("' . BASE_URL . '/analytics_collector.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({
                    project_id: window.flyingBookAnalytics.projectId,
                    session_id: window.flyingBookAnalytics.sessionId,
                    visitor_id: window.flyingBookAnalytics.visitorId,
                    event: event,
                    data: data,
                    timestamp: Date.now(),
                    url: window.location.href
                })
            }).catch(console.error);
        };
        
        // Tracking automatique
        window.addEventListener("beforeunload", () => {
            FlyingBook.sendAnalytics("session_end", {
                totalTime: Date.now() - FlyingBook.analytics.startTime,
                slideViews: FlyingBook.analytics.slideViews,
                maxSlideReached: FlyingBook.currentSlide
            });
        });
        ';
    }
    
    /**
     * Watermark si activé
     */
    private function getWatermark() {
        $text = $this->project['watermark_text'] ?: 'Created with Flying Book';
        return '<div class="watermark">' . htmlspecialchars($text) . '</div>';
    }
    
    /**
     * Logger la génération
     */
    private function logGeneration($projectId, $outputPath) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO flying_book_activity_logs 
                (user_id, action, entity_type, entity_id, description, metadata) 
                VALUES (?, 'generate_book', 'project', ?, 'Flying Book généré', ?)
            ");
            
            $metadata = json_encode([
                'output_path' => $outputPath,
                'theme' => $this->project['style_theme'],
                'transition' => $this->project['transition_type'],
                'slides_count' => count($this->slides),
                'generation_time' => date('Y-m-d H:i:s')
            ]);
            
            $stmt->execute([
                $this->project['user_id'],
                $projectId,
                $metadata
            ]);
        } catch (Exception $e) {
            logError("Erreur logging génération", ['error' => $e->getMessage()]);
        }
    }
    
    /**
     * Statistiques de génération
     */
    private function getGenerationStats() {
        return [
            'slides_generated' => count($this->slides),
            'theme_used' => $this->project['style_theme'],
            'transition_type' => $this->project['transition_type'],
            'analytics_enabled' => $this->project['analytics_enabled'],
            'watermark_enabled' => $this->project['watermark_enabled']
        ];
    }
}

// ========== API ENDPOINTS ==========

// Vérification de session
session_start();
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Non authentifié']);
    exit;
}

// Traitement des requêtes
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'generate_book':
            $projectId = intval($_POST['project_id'] ?? 0);
            $options = json_decode($_POST['options'] ?? '{}', true);
            
            if (!$projectId) {
                echo json_encode(['success' => false, 'error' => 'ID projet manquant']);
                exit;
            }
            
            $engine = new FlyingBookRenderEngine();
            $result = $engine->generateFlyingBook($projectId, $options);
            
            echo json_encode($result);
            break;
            
        default:
            echo json_encode(['success' => false, 'error' => 'Action inconnue']);
    }
}

// Affichage de l'interface si GET
elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && !empty($_GET['preview'])) {
    $projectId = intval($_GET['preview']);
    
    if ($projectId) {
        $engine = new FlyingBookRenderEngine();
        $result = $engine->generateFlyingBook($projectId, ['preview' => true]);
        
        if ($result['success']) {
            // Rediriger vers le Flying Book généré
            header('Location: ' . $result['url']);
            exit;
        } else {
            echo "Erreur: " . $result['error'];
        }
    }
}
?>