<?php
/**
 * Projects.php - Flying Book Creator v2.0
 * Gestion complète des projets avec suppression, duplication, etc.
 */

session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'delete_project':
                $project_id = (int)$_POST['project_id'];
                
                if (!$project_id) {
                    echo json_encode(['success' => false, 'message' => 'ID du projet manquant']);
                    exit;
                }
                
                try {
                    $pdo->beginTransaction();
                    
                    // Vérifier que le projet appartient à l'utilisateur (ou admin)
                    $stmt = $pdo->prepare("
                        SELECT * FROM flying_book_projects 
                        WHERE id = ? AND (user_id = ? OR ? = 1)
                    ");
                    $stmt->execute([$project_id, $user_id, $is_admin ? 1 : 0]);
                    $project = $stmt->fetch();
                    
                    if (!$project) {
                        echo json_encode(['success' => false, 'message' => 'Projet non trouvé ou accès refusé']);
                        exit;
                    }
                    
                    // Supprimer les fichiers physiques
                    $project_folder = $project['project_folder'];
                    if ($project_folder) {
                        $folder_path = __DIR__ . '/uploads/projects/' . $project_folder;
                        if (file_exists($folder_path)) {
                            deleteDirectory($folder_path);
                        }
                    }
                    
                    // Supprimer les enregistrements en cascade
                    $tables = [
                        'flying_book_slides',
                        'flying_book_media', 
                        'flying_book_analytics',
                        'flying_book_leads',
                        'flying_book_shares',
                        'flying_book_project_configs',
                        'flying_book_project_tags',
                        'flying_book_project_versions'
                    ];
                    
                    foreach ($tables as $table) {
                        $stmt = $pdo->prepare("DELETE FROM {$table} WHERE project_id = ?");
                        $stmt->execute([$project_id]);
                    }
                    
                    // Marquer le projet comme supprimé plutôt que de le supprimer définitivement
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET status = 'deleted', updated_at = NOW() 
                        WHERE id = ?
                    ");
                    $stmt->execute([$project_id]);
                    
                    $pdo->commit();
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Projet supprimé avec succès'
                    ]);
                    
                } catch (Exception $e) {
                    $pdo->rollback();
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'duplicate_project':
                $project_id = (int)$_POST['project_id'];
                $new_name = trim($_POST['new_name'] ?? '');
                
                if (!$project_id) {
                    echo json_encode(['success' => false, 'message' => 'ID du projet manquant']);
                    exit;
                }
                
                if (empty($new_name)) {
                    echo json_encode(['success' => false, 'message' => 'Nom du nouveau projet requis']);
                    exit;
                }
                
                try {
                    $result = duplicateProject($pdo, $project_id, $user_id, $new_name);
                    echo json_encode($result);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'toggle_publication':
                $project_id = (int)$_POST['project_id'];
                $is_published = (bool)$_POST['is_published'];
                
                try {
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET is_published = ?, 
                            published_at = ?,
                            status = ?,
                            updated_at = NOW()
                        WHERE id = ? AND user_id = ?
                    ");
                    
                    $published_at = $is_published ? date('Y-m-d H:i:s') : null;
                    $status = $is_published ? 'published' : 'draft';
                    
                    $stmt->execute([$is_published ? 1 : 0, $published_at, $status, $project_id, $user_id]);
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => $is_published ? 'Projet publié' : 'Projet dépublié'
                    ]);
                    
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'archive_project':
                $project_id = (int)$_POST['project_id'];
                
                try {
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET status = 'archived', updated_at = NOW()
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([$project_id, $user_id]);
                    
                    echo json_encode(['success' => true, 'message' => 'Projet archivé']);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'restore_project':
                $project_id = (int)$_POST['project_id'];
                
                try {
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_projects 
                        SET status = 'draft', updated_at = NOW()
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([$project_id, $user_id]);
                    
                    echo json_encode(['success' => true, 'message' => 'Projet restauré']);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'bulk_action':
                $action = $_POST['bulk_action'] ?? '';
                $project_ids = json_decode($_POST['project_ids'] ?? '[]', true);
                
                if (empty($project_ids) || !is_array($project_ids)) {
                    echo json_encode(['success' => false, 'message' => 'Aucun projet sélectionné']);
                    exit;
                }
                
                try {
                    $count = 0;
                    foreach ($project_ids as $project_id) {
                        $project_id = (int)$project_id;
                        
                        switch ($action) {
                            case 'delete':
                                $stmt = $pdo->prepare("
                                    UPDATE flying_book_projects 
                                    SET status = 'deleted', updated_at = NOW()
                                    WHERE id = ? AND user_id = ?
                                ");
                                $stmt->execute([$project_id, $user_id]);
                                break;
                                
                            case 'archive':
                                $stmt = $pdo->prepare("
                                    UPDATE flying_book_projects 
                                    SET status = 'archived', updated_at = NOW()
                                    WHERE id = ? AND user_id = ?
                                ");
                                $stmt->execute([$project_id, $user_id]);
                                break;
                                
                            case 'publish':
                                $stmt = $pdo->prepare("
                                    UPDATE flying_book_projects 
                                    SET is_published = 1, status = 'published', 
                                        published_at = NOW(), updated_at = NOW()
                                    WHERE id = ? AND user_id = ?
                                ");
                                $stmt->execute([$project_id, $user_id]);
                                break;
                                
                            case 'unpublish':
                                $stmt = $pdo->prepare("
                                    UPDATE flying_book_projects 
                                    SET is_published = 0, status = 'draft', updated_at = NOW()
                                    WHERE id = ? AND user_id = ?
                                ");
                                $stmt->execute([$project_id, $user_id]);
                                break;
                        }
                        
                        if ($stmt->rowCount() > 0) {
                            $count++;
                        }
                    }
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => "{$count} projet(s) traité(s) avec succès"
                    ]);
                    
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
        }
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Filtres
    $status_filter = $_GET['status'] ?? 'all';
    $search = trim($_GET['search'] ?? '');
    $sort = $_GET['sort'] ?? 'updated_at';
    $order = $_GET['order'] ?? 'DESC';
    $page = max(1, (int)($_GET['page'] ?? 1));
    $per_page = 12;
    $offset = ($page - 1) * $per_page;
    
    // Construction de la requête
    $where_conditions = ["p.user_id = ?"];
    $params = [$user_id];
    
    if ($status_filter !== 'all') {
        if ($status_filter === 'published') {
            $where_conditions[] = "p.is_published = 1";
        } elseif ($status_filter === 'draft') {
            $where_conditions[] = "p.status = 'draft'";
        } elseif ($status_filter === 'archived') {
            $where_conditions[] = "p.status = 'archived'";
        }
    } else {
        $where_conditions[] = "p.status != 'deleted'";
    }
    
    if (!empty($search)) {
        $where_conditions[] = "(p.project_name LIKE ? OR p.title LIKE ? OR p.description LIKE ?)";
        $search_param = "%{$search}%";
        $params = array_merge($params, [$search_param, $search_param, $search_param]);
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Requête principale avec statistiques
    $stmt = $pdo->prepare("
        SELECT p.*, 
               COUNT(DISTINCT s.id) as slide_count,
               COUNT(DISTINCT a.id) as view_count,
               COALESCE(SUM(m.file_size), 0) as storage_used,
               MAX(a.last_visit) as last_viewed
        FROM flying_book_projects p
        LEFT JOIN flying_book_slides s ON p.id = s.project_id AND s.is_active = 1
        LEFT JOIN flying_book_analytics a ON p.id = a.project_id
        LEFT JOIN flying_book_media m ON p.id = m.project_id AND m.upload_status = 'ready'
        WHERE {$where_clause}
        GROUP BY p.id
        ORDER BY p.{$sort} {$order}
        LIMIT {$per_page} OFFSET {$offset}
    ");
    $stmt->execute($params);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Compter le total pour la pagination
    $count_stmt = $pdo->prepare("
        SELECT COUNT(DISTINCT p.id) as total
        FROM flying_book_projects p
        WHERE {$where_clause}
    ");
    $count_stmt->execute($params);
    $total_projects = $count_stmt->fetch()['total'];
    $total_pages = ceil($total_projects / $per_page);
    
    // Statistiques globales
    $stats_stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN is_published = 1 THEN 1 ELSE 0 END) as published,
            SUM(CASE WHEN status = 'draft' THEN 1 ELSE 0 END) as draft,
            SUM(CASE WHEN status = 'archived' THEN 1 ELSE 0 END) as archived,
            COALESCE(SUM(file_size_total), 0) as total_storage
        FROM flying_book_projects 
        WHERE user_id = ? AND status != 'deleted'
    ");
    $stats_stmt->execute([$user_id]);
    $stats = $stats_stmt->fetch();
    
} catch (Exception $e) {
    $error = "Erreur: " . $e->getMessage();
}

// Fonctions utilitaires
function deleteDirectory($dir) {
    if (!is_dir($dir)) return false;
    
    $files = array_diff(scandir($dir), ['.', '..']);
    foreach ($files as $file) {
        $path = $dir . DIRECTORY_SEPARATOR . $file;
        is_dir($path) ? deleteDirectory($path) : unlink($path);
    }
    return rmdir($dir);
}

function duplicateProject($pdo, $original_id, $user_id, $new_name) {
    try {
        $pdo->beginTransaction();
        
        // Récupérer le projet original
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_projects 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$original_id, $user_id]);
        $original = $stmt->fetch();
        
        if (!$original) {
            throw new Exception('Projet original non trouvé');
        }
        
        // Créer le nouveau projet
        $new_slug = generateSlug($new_name);
        $new_folder = 'project_' . uniqid();
        
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_projects 
            (user_id, project_name, project_slug, title, subtitle, description,
             style_theme, transition_type, navigation_type, project_folder,
             plan_type, status, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', NOW(), NOW())
        ");
        
        $stmt->execute([
            $user_id, $new_name, $new_slug, $new_name,
            $original['subtitle'], $original['description'],
            $original['style_theme'], $original['transition_type'], 
            $original['navigation_type'], $new_folder,
            $original['plan_type']
        ]);
        
        $new_project_id = $pdo->lastInsertId();
        
        // Dupliquer les slides
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_slides 
            WHERE project_id = ? ORDER BY display_order
        ");
        $stmt->execute([$original_id]);
        $slides = $stmt->fetchAll();
        
        foreach ($slides as $slide) {
            $stmt = $pdo->prepare("
                INSERT INTO flying_book_slides 
                (project_id, slide_number, slide_title, media_type, media_file,
                 background_audio, auto_duration, text_overlay, display_order, is_active)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $new_project_id, $slide['slide_number'], $slide['slide_title'],
                $slide['media_type'], $slide['media_file'], $slide['background_audio'],
                $slide['auto_duration'], $slide['text_overlay'], 
                $slide['display_order'], $slide['is_active']
            ]);
        }
        
        // Dupliquer la configuration avancée si elle existe
        $stmt = $pdo->prepare("
            SELECT advanced_config FROM flying_book_project_configs 
            WHERE project_id = ?
        ");
        $stmt->execute([$original_id]);
        $config = $stmt->fetch();
        
        if ($config) {
            $stmt = $pdo->prepare("
                INSERT INTO flying_book_project_configs 
                (project_id, advanced_config, created_at, updated_at)
                VALUES (?, ?, NOW(), NOW())
            ");
            $stmt->execute([$new_project_id, $config['advanced_config']]);
        }
        
        $pdo->commit();
        
        return [
            'success' => true,
            'message' => 'Projet dupliqué avec succès',
            'new_project_id' => $new_project_id
        ];
        
    } catch (Exception $e) {
        $pdo->rollback();
        throw $e;
    }
}

function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}

// SUPPRIMÉ: function formatFileSize() - Utilise celle de .env.php

function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'À l\'instant';
    if ($time < 3600) return floor($time/60) . ' min';
    if ($time < 86400) return floor($time/3600) . ' h';
    if ($time < 2592000) return floor($time/86400) . ' j';
    return date('d/m/Y', strtotime($datetime));
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mes Projets - Flying Book Creator v2.0</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Styles améliorés pour la gestion des projets */
        .projects-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            gap: 1rem;
        }
        
        .projects-stats {
            display: flex;
            gap: 2rem;
            margin-bottom: 2rem;
            flex-wrap: wrap;
        }
        
        .stat-quick {
            background: white;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            text-align: center;
            min-width: 120px;
        }
        
        .stat-quick .number {
            font-size: 1.5rem;
            font-weight: 700;
            color: #667eea;
            display: block;
        }
        
        .stat-quick .label {
            font-size: 0.8rem;
            color: #64748b;
            text-transform: uppercase;
            margin-top: 0.25rem;
        }
        
        .projects-filters {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
        }
        
        .filters-row {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .filter-group label {
            font-size: 0.8rem;
            font-weight: 600;
            color: #64748b;
            text-transform: uppercase;
        }
        
        .projects-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .project-card {
            background: white;
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .project-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 12px 35px rgba(0, 0, 0, 0.15);
        }
        
        .project-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #667eea, #764ba2);
        }
        
        .project-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .project-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1e293b;
            margin: 0 0 0.25rem 0;
            line-height: 1.3;
        }
        
        .project-meta {
            font-size: 0.8rem;
            color: #64748b;
            margin-bottom: 1rem;
        }
        
        .project-status {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.75rem;
            border-radius: 15px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .status-draft {
            background: rgba(107, 114, 128, 0.1);
            color: #374151;
        }
        
        .status-published {
            background: rgba(16, 185, 129, 0.1);
            color: #047857;
        }
        
        .status-archived {
            background: rgba(245, 158, 11, 0.1);
            color: #92400e;
        }
        
        .project-stats {
            display: flex;
            gap: 1rem;
            margin: 1rem 0;
            font-size: 0.8rem;
            color: #64748b;
        }
        
        .project-stat {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .project-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
            flex-wrap: wrap;
        }
        
        .btn-action {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 8px;
            font-size: 0.8rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
        }
        
        .btn-edit {
            background: #667eea;
            color: white;
        }
        
        .btn-duplicate {
            background: #f59e0b;
            color: white;
        }
        
        .btn-delete {
            background: #ef4444;
            color: white;
        }
        
        .btn-archive {
            background: #6b7280;
            color: white;
        }
        
        .btn-publish {
            background: #10b981;
            color: white;
        }
        
        .btn-action:hover {
            transform: translateY(-1px);
            opacity: 0.9;
        }
        
        .bulk-actions {
            background: white;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            margin-bottom: 1rem;
            display: none;
        }
        
        .bulk-actions.active {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .project-checkbox {
            position: absolute;
            top: 1rem;
            left: 1rem;
            width: 20px;
            height: 20px;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            gap: 0.5rem;
            margin-top: 2rem;
        }
        
        .pagination a, .pagination span {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            text-decoration: none;
            color: #64748b;
            border: 1px solid #e2e8f0;
        }
        
        .pagination .current {
            background: #667eea;
            color: white;
            border-color: #667eea;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .projects-grid {
                grid-template-columns: 1fr;
            }
            
            .filters-row {
                flex-direction: column;
                align-items: stretch;
            }
            
            .projects-stats {
                flex-direction: column;
                gap: 1rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link active">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="media.php" class="nav-link">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($_SESSION['user_name'], 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($_SESSION['user_name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- En-tête des projets -->
            <div class="projects-header">
                <div>
                    <h1>Mes Projets Flying Book</h1>
                    <p>Gérez et organisez tous vos projets</p>
                </div>
                <div style="display: flex; gap: 1rem;">
                    <button class="btn btn-outline" onclick="toggleBulkMode()">
                        <i class="fas fa-check-square"></i> Sélection multiple
                    </button>
                    <a href="create-project.php" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Nouveau Projet
                    </a>
                </div>
            </div>

            <!-- Statistiques rapides -->
            <div class="projects-stats">
                <div class="stat-quick">
                    <span class="number"><?= $stats['total'] ?></span>
                    <span class="label">Total</span>
                </div>
                <div class="stat-quick">
                    <span class="number"><?= $stats['published'] ?></span>
                    <span class="label">Publiés</span>
                </div>
                <div class="stat-quick">
                    <span class="number"><?= $stats['draft'] ?></span>
                    <span class="label">Brouillons</span>
                </div>
                <div class="stat-quick">
                    <span class="number"><?= $stats['archived'] ?></span>
                    <span class="label">Archivés</span>
                </div>
                <div class="stat-quick">
                    <span class="number"><?= formatFileSize($stats['total_storage']) ?></span>
                    <span class="label">Stockage</span>
                </div>
            </div>

            <!-- Filtres -->
            <div class="projects-filters">
                <form method="GET" class="filters-row">
                    <div class="filter-group">
                        <label>Recherche</label>
                        <input type="text" name="search" class="form-control" 
                               placeholder="Nom, titre, description..." 
                               value="<?= htmlspecialchars($search) ?>" style="width: 250px;">
                    </div>
                    
                    <div class="filter-group">
                        <label>Statut</label>
                        <select name="status" class="form-control">
                            <option value="all" <?= $status_filter === 'all' ? 'selected' : '' ?>>Tous</option>
                            <option value="draft" <?= $status_filter === 'draft' ? 'selected' : '' ?>>Brouillons</option>
                            <option value="published" <?= $status_filter === 'published' ? 'selected' : '' ?>>Publiés</option>
                            <option value="archived" <?= $status_filter === 'archived' ? 'selected' : '' ?>>Archivés</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>Tri</label>
                        <select name="sort" class="form-control">
                            <option value="updated_at" <?= $sort === 'updated_at' ? 'selected' : '' ?>>Dernière modification</option>
                            <option value="created_at" <?= $sort === 'created_at' ? 'selected' : '' ?>>Date de création</option>
                            <option value="project_name" <?= $sort === 'project_name' ? 'selected' : '' ?>>Nom</option>
                            <option value="view_count" <?= $sort === 'view_count' ? 'selected' : '' ?>>Vues</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>Ordre</label>
                        <select name="order" class="form-control">
                            <option value="DESC" <?= $order === 'DESC' ? 'selected' : '' ?>>Décroissant</option>
                            <option value="ASC" <?= $order === 'ASC' ? 'selected' : '' ?>>Croissant</option>
                        </select>
                    </div>
                    
                    <div class="filter-group" style="align-self: flex-end;">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search"></i> Filtrer
                        </button>
                    </div>
                </form>
            </div>

            <!-- Actions en lot -->
            <div class="bulk-actions" id="bulkActions">
                <span id="selectedCount">0 projet(s) sélectionné(s)</span>
                <select id="bulkActionSelect" class="form-control" style="width: auto;">
                    <option value="">Choisir une action...</option>
                    <option value="publish">Publier</option>
                    <option value="unpublish">Dépublier</option>
                    <option value="archive">Archiver</option>
                    <option value="delete">Supprimer</option>
                </select>
                <button class="btn btn-primary" onclick="executeBulkAction()">Exécuter</button>
                <button class="btn btn-outline" onclick="toggleBulkMode()">Annuler</button>
            </div>

            <!-- Grille des projets -->
            <?php if (empty($projects)): ?>
                <div style="text-align: center; padding: 4rem 2rem; color: #64748b;">
                    <i class="fas fa-folder-open" style="font-size: 4rem; margin-bottom: 1rem;"></i>
                    <h3>Aucun projet trouvé</h3>
                    <p>Créez votre premier Flying Book ou modifiez vos filtres</p>
                    <a href="create-project.php" class="btn btn-primary" style="margin-top: 1rem;">
                        <i class="fas fa-plus"></i> Créer un projet
                    </a>
                </div>
            <?php else: ?>
                <div class="projects-grid">
                    <?php foreach ($projects as $project): ?>
                        <div class="project-card" data-project-id="<?= $project['id'] ?>">
                            <input type="checkbox" class="project-checkbox" style="display: none;" 
                                   value="<?= $project['id'] ?>" onchange="updateBulkSelection()">
                            
                            <div class="project-header">
                                <div>
                                    <h3 class="project-title"><?= htmlspecialchars($project['title'] ?: $project['project_name']) ?></h3>
                                    <div class="project-meta">
                                        Créé le <?= date('d/m/Y', strtotime($project['created_at'])) ?> • 
                                        Modifié <?= timeAgo($project['updated_at']) ?>
                                    </div>
                                </div>
                                <span class="project-status status-<?= $project['status'] ?>">
                                    <?php if ($project['is_published']): ?>
                                        <i class="fas fa-globe"></i> Publié
                                    <?php else: ?>
                                        <i class="fas fa-edit"></i> <?= ucfirst($project['status']) ?>
                                    <?php endif; ?>
                                </span>
                            </div>

                            <?php if ($project['description']): ?>
                                <p style="color: #64748b; font-size: 0.9rem; margin-bottom: 1rem; line-height: 1.4;">
                                    <?= htmlspecialchars(substr($project['description'], 0, 120)) ?>
                                    <?= strlen($project['description']) > 120 ? '...' : '' ?>
                                </p>
                            <?php endif; ?>

                            <div class="project-stats">
                                <div class="project-stat">
                                    <i class="fas fa-images"></i>
                                    <?= $project['slide_count'] ?> slides
                                </div>
                                <div class="project-stat">
                                    <i class="fas fa-eye"></i>
                                    <?= $project['view_count'] ?> vues
                                </div>
                                <div class="project-stat">
                                    <i class="fas fa-hdd"></i>
                                    <?= formatFileSize($project['storage_used']) ?>
                                </div>
                            </div>

                            <div class="project-actions">
                                <a href="flying_book_slides.php?id=<?= $project['id'] ?>" class="btn-action btn-edit">
                                    <i class="fas fa-edit"></i> Éditer
                                </a>
                                
                                <?php if ($project['is_published']): ?>
                                    <button class="btn-action btn-archive" onclick="togglePublication(<?= $project['id'] ?>, false)">
                                        <i class="fas fa-eye-slash"></i> Dépublier
                                    </button>
                                <?php else: ?>
                                    <button class="btn-action btn-publish" onclick="togglePublication(<?= $project['id'] ?>, true)">
                                        <i class="fas fa-globe"></i> Publier
                                    </button>
                                <?php endif; ?>
                                
                                <button class="btn-action btn-duplicate" onclick="duplicateProject(<?= $project['id'] ?>, '<?= htmlspecialchars($project['project_name']) ?>')">
                                    <i class="fas fa-copy"></i> Dupliquer
                                </button>
                                
                                <button class="btn-action btn-delete" onclick="deleteProject(<?= $project['id'] ?>, '<?= htmlspecialchars($project['project_name']) ?>')">
                                    <i class="fas fa-trash"></i> Supprimer
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php
                        $query_params = $_GET;
                        
                        // Page précédente
                        if ($page > 1):
                            $query_params['page'] = $page - 1;
                            $prev_url = '?' . http_build_query($query_params);
                        ?>
                            <a href="<?= $prev_url ?>">&laquo; Précédent</a>
                        <?php endif; ?>
                        
                        <?php
                        // Pages numériques
                        $start = max(1, $page - 2);
                        $end = min($total_pages, $page + 2);
                        
                        for ($i = $start; $i <= $end; $i++):
                            $query_params['page'] = $i;
                            $page_url = '?' . http_build_query($query_params);
                        ?>
                            <?php if ($i == $page): ?>
                                <span class="current"><?= $i ?></span>
                            <?php else: ?>
                                <a href="<?= $page_url ?>"><?= $i ?></a>
                            <?php endif; ?>
                        <?php endfor; ?>
                        
                        <?php
                        // Page suivante
                        if ($page < $total_pages):
                            $query_params['page'] = $page + 1;
                            $next_url = '?' . http_build_query($query_params);
                        ?>
                            <a href="<?= $next_url ?>">Suivant &raquo;</a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </main>

    <script>
        let bulkMode = false;
        let selectedProjects = new Set();

        // Gestion du mode sélection multiple
        function toggleBulkMode() {
            bulkMode = !bulkMode;
            const checkboxes = document.querySelectorAll('.project-checkbox');
            const bulkActions = document.getElementById('bulkActions');
            
            checkboxes.forEach(cb => {
                cb.style.display = bulkMode ? 'block' : 'none';
                cb.checked = false;
            });
            
            if (bulkMode) {
                bulkActions.classList.add('active');
            } else {
                bulkActions.classList.remove('active');
                selectedProjects.clear();
                updateBulkSelection();
            }
        }

        // Mise à jour de la sélection
        function updateBulkSelection() {
            const checkboxes = document.querySelectorAll('.project-checkbox:checked');
            selectedProjects.clear();
            
            checkboxes.forEach(cb => {
                selectedProjects.add(parseInt(cb.value));
            });
            
            document.getElementById('selectedCount').textContent = 
                `${selectedProjects.size} projet(s) sélectionné(s)`;
        }

        // Exécution d'action en lot
        async function executeBulkAction() {
            const action = document.getElementById('bulkActionSelect').value;
            
            if (!action || selectedProjects.size === 0) {
                showNotification('warning', 'Attention', 'Sélectionnez une action et des projets');
                return;
            }
            
            const actionNames = {
                'publish': 'publier',
                'unpublish': 'dépublier', 
                'archive': 'archiver',
                'delete': 'supprimer'
            };
            
            if (!confirm(`Êtes-vous sûr de vouloir ${actionNames[action]} ${selectedProjects.size} projet(s) ?`)) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'bulk_action');
                formData.append('bulk_action', action);
                formData.append('project_ids', JSON.stringify([...selectedProjects]));
                
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('success', 'Succès', result.message);
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification('error', 'Erreur', result.message);
                }
                
            } catch (error) {
                showNotification('error', 'Erreur', 'Erreur de connexion');
            }
        }

        // Supprimer un projet
        async function deleteProject(projectId, projectName) {
            if (!confirm(`Êtes-vous sûr de vouloir supprimer le projet "${projectName}" ?\n\nCette action est irréversible.`)) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete_project');
                formData.append('project_id', projectId);
                
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('success', 'Supprimé', result.message);
                    
                    // Supprimer visuellement la carte
                    const projectCard = document.querySelector(`[data-project-id="${projectId}"]`);
                    if (projectCard) {
                        projectCard.style.transform = 'scale(0)';
                        projectCard.style.opacity = '0';
                        setTimeout(() => projectCard.remove(), 300);
                    }
                } else {
                    showNotification('error', 'Erreur', result.message);
                }
                
            } catch (error) {
                showNotification('error', 'Erreur', 'Erreur de connexion');
            }
        }

        // Dupliquer un projet
        async function duplicateProject(projectId, projectName) {
            const newName = prompt(`Nom du nouveau projet (copie de "${projectName}"):`, `${projectName} - Copie`);
            
            if (!newName || newName.trim() === '') {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'duplicate_project');
                formData.append('project_id', projectId);
                formData.append('new_name', newName.trim());
                
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('success', 'Dupliqué', result.message);
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification('error', 'Erreur', result.message);
                }
                
            } catch (error) {
                showNotification('error', 'Erreur', 'Erreur de connexion');
            }
        }

        // Basculer la publication
        async function togglePublication(projectId, isPublished) {
            try {
                const formData = new FormData();
                formData.append('action', 'toggle_publication');
                formData.append('project_id', projectId);
                formData.append('is_published', isPublished ? '1' : '0');
                
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('success', 'Mis à jour', result.message);
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showNotification('error', 'Erreur', result.message);
                }
                
            } catch (error) {
                showNotification('error', 'Erreur', 'Erreur de connexion');
            }
        }

        // Système de notifications
        function showNotification(type, title, message) {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle', 
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 1rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.5rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${title}</div>
                        <div style="font-size: 0.9rem; opacity: 0.9;">${message}</div>
                    </div>
                </div>
            `;
            
            notification.style.cssText = `
                position: fixed;
                top: 2rem;
                right: 2rem;
                z-index: 3000;
                max-width: 400px;
                background: white;
                border-radius: 12px;
                padding: 1rem 1.5rem;
                box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
                border-left: 4px solid var(--${type === 'success' ? 'success' : type === 'error' ? 'error' : type === 'warning' ? 'warning' : 'info'}-color);
                transform: translateX(500px);
                transition: transform 0.3s ease;
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => {
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.style.transform = 'translateX(500px)';
                setTimeout(() => notification.remove(), 300);
            }, type === 'error' ? 8000 : 5000);
        }

        // Menu utilisateur
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }

        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
    </script>
</body>
</html>