<?php
session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

$message = '';
$error = '';

// Fonction pour créer les tables manquantes
function createMissingTables($pdo) {
    try {
        // Table user_settings
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS `flying_book_user_settings` (
              `id` int NOT NULL AUTO_INCREMENT,
              `user_id` int NOT NULL,
              `setting_key` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
              `setting_value` text COLLATE utf8mb4_unicode_ci,
              `setting_type` enum('string','int','bool','json') COLLATE utf8mb4_unicode_ci DEFAULT 'string',
              `is_encrypted` tinyint(1) DEFAULT '0',
              `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
              `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
              PRIMARY KEY (`id`),
              UNIQUE KEY `user_setting_unique` (`user_id`, `setting_key`),
              KEY `idx_user_settings` (`user_id`),
              CONSTRAINT `fk_user_settings_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Table activity_logs
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS `flying_book_activity_logs` (
              `id` int NOT NULL AUTO_INCREMENT,
              `user_id` int NOT NULL,
              `action` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
              `entity_type` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
              `entity_id` int DEFAULT NULL,
              `description` text COLLATE utf8mb4_unicode_ci,
              `ip_address` varchar(45) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
              `user_agent` text COLLATE utf8mb4_unicode_ci,
              `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
              PRIMARY KEY (`id`),
              KEY `idx_user_activity` (`user_id`, `created_at` DESC),
              KEY `idx_action` (`action`),
              KEY `idx_entity` (`entity_type`, `entity_id`),
              CONSTRAINT `fk_activity_logs_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        return true;
    } catch (Exception $e) {
        error_log("Failed to create missing tables: " . $e->getMessage());
        return false;
    }
}

// Fonctions utilitaires
function getDatabaseConnection() {
    try {
        return new PDO(
            "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
            DB_USER,
            DB_PASS,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } catch (PDOException $e) {
        throw new Exception("Erreur de connexion à la base de données");
    }
}

function logActivity($pdo, $user_id, $action, $entity_type, $entity_id, $description) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_activity_logs 
            (user_id, action, entity_type, entity_id, description, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user_id, $action, $entity_type, $entity_id, $description,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    } catch (Exception $e) {
        error_log("Failed to log activity: " . $e->getMessage());
    }
}

function getUserSetting($pdo, $user_id, $key, $default = null) {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM flying_book_user_settings WHERE user_id = ? AND setting_key = ?");
        $stmt->execute([$user_id, $key]);
        $result = $stmt->fetchColumn();
        return $result !== false ? $result : $default;
    } catch (Exception $e) {
        return $default;
    }
}

function setUserSetting($pdo, $user_id, $key, $value, $type = 'string') {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_user_settings (user_id, setting_key, setting_value, setting_type)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
                setting_value = VALUES(setting_value),
                setting_type = VALUES(setting_type),
                updated_at = CURRENT_TIMESTAMP
        ");
        return $stmt->execute([$user_id, $key, $value, $type]);
    } catch (Exception $e) {
        error_log("Failed to set user setting: " . $e->getMessage());
        return false;
    }
}

try {
    $pdo = getDatabaseConnection();
    
    // Créer les tables manquantes si nécessaire
    createMissingTables($pdo);
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['ajax_action']) {
            case 'update_profile':
                $name = trim($_POST['name'] ?? '');
                $email = trim($_POST['email'] ?? '');
                $first_name = trim($_POST['first_name'] ?? '');
                $last_name = trim($_POST['last_name'] ?? '');
                $phone = trim($_POST['phone'] ?? '');
                $timezone = $_POST['timezone'] ?? 'Europe/Paris';
                $language = $_POST['language'] ?? 'fr';
                
                if (empty($name) || empty($email)) {
                    echo json_encode(['success' => false, 'message' => 'Nom et email obligatoires']);
                    exit;
                }
                
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    echo json_encode(['success' => false, 'message' => 'Email invalide']);
                    exit;
                }
                
                // Vérifier si l'email est déjà utilisé par un autre utilisateur
                $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
                $stmt->execute([$email, $user_id]);
                if ($stmt->fetch()) {
                    echo json_encode(['success' => false, 'message' => 'Email déjà utilisé']);
                    exit;
                }
                
                $stmt = $pdo->prepare("
                    UPDATE users 
                    SET name = ?, email = ?, first_name = ?, last_name = ?, phone = ?, timezone = ?, language = ?
                    WHERE id = ?
                ");
                
                if ($stmt->execute([$name, $email, $first_name, $last_name, $phone, $timezone, $language, $user_id])) {
                    $_SESSION['user_name'] = $name;
                    logActivity($pdo, $user_id, 'update_profile', 'user', $user_id, 'Profil utilisateur mis à jour');
                    echo json_encode(['success' => true, 'message' => 'Profil mis à jour avec succès']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la mise à jour']);
                }
                exit;
                
            case 'change_password':
                $current_password = $_POST['current_password'] ?? '';
                $new_password = $_POST['new_password'] ?? '';
                $confirm_password = $_POST['confirm_password'] ?? '';
                
                if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                    echo json_encode(['success' => false, 'message' => 'Tous les champs sont obligatoires']);
                    exit;
                }
                
                if ($new_password !== $confirm_password) {
                    echo json_encode(['success' => false, 'message' => 'Les mots de passe ne correspondent pas']);
                    exit;
                }
                
                if (strlen($new_password) < 8) {
                    echo json_encode(['success' => false, 'message' => 'Le mot de passe doit contenir au moins 8 caractères']);
                    exit;
                }
                
                // Vérifier le mot de passe actuel
                $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $stored_password = $stmt->fetchColumn();
                
                if (!password_verify($current_password, $stored_password)) {
                    echo json_encode(['success' => false, 'message' => 'Mot de passe actuel incorrect']);
                    exit;
                }
                
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                
                if ($stmt->execute([$hashed_password, $user_id])) {
                    logActivity($pdo, $user_id, 'change_password', 'user', $user_id, 'Mot de passe modifié');
                    echo json_encode(['success' => true, 'message' => 'Mot de passe modifié avec succès']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la modification']);
                }
                exit;
                
            case 'update_preferences':
                $theme = $_POST['theme'] ?? 'moderne';
                $dashboard_layout = $_POST['dashboard_layout'] ?? 'grid';
                $notifications_enabled = isset($_POST['notifications_enabled']) ? '1' : '0';
                $auto_save = isset($_POST['auto_save']) ? '1' : '0';
                
                $settings = [
                    'theme' => $theme,
                    'dashboard_layout' => $dashboard_layout,
                    'notifications_enabled' => $notifications_enabled,
                    'auto_save' => $auto_save
                ];
                
                $success = true;
                foreach ($settings as $key => $value) {
                    if (!setUserSetting($pdo, $user_id, $key, $value, $key === 'notifications_enabled' || $key === 'auto_save' ? 'bool' : 'string')) {
                        $success = false;
                        break;
                    }
                }
                
                if ($success) {
                    logActivity($pdo, $user_id, 'update_preferences', 'user', $user_id, 'Préférences mises à jour');
                    echo json_encode(['success' => true, 'message' => 'Préférences sauvegardées']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la sauvegarde']);
                }
                exit;
                
            case 'upload_avatar':
                if (!isset($_FILES['avatar']) || $_FILES['avatar']['error'] !== UPLOAD_ERR_OK) {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de l\'upload']);
                    exit;
                }
                
                $file = $_FILES['avatar'];
                $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                $max_size = 5 * 1024 * 1024; // 5MB
                
                if (!in_array($file['type'], $allowed_types)) {
                    echo json_encode(['success' => false, 'message' => 'Type de fichier non autorisé']);
                    exit;
                }
                
                if ($file['size'] > $max_size) {
                    echo json_encode(['success' => false, 'message' => 'Fichier trop volumineux (max 5MB)']);
                    exit;
                }
                
                $upload_dir = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/avatars/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = 'avatar_' . $user_id . '_' . time() . '.' . $file_ext;
                $file_path = $upload_dir . $filename;
                $relative_path = '/flying-book/uploads/avatars/' . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $file_path)) {
                    // Supprimer l'ancien avatar s'il existe
                    $stmt = $pdo->prepare("SELECT avatar FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $old_avatar = $stmt->fetchColumn();
                    
                    if ($old_avatar && file_exists($_SERVER['DOCUMENT_ROOT'] . $old_avatar)) {
                        unlink($_SERVER['DOCUMENT_ROOT'] . $old_avatar);
                    }
                    
                    $stmt = $pdo->prepare("UPDATE users SET avatar = ? WHERE id = ?");
                    if ($stmt->execute([$relative_path, $user_id])) {
                        logActivity($pdo, $user_id, 'upload_avatar', 'user', $user_id, 'Avatar mis à jour');
                        echo json_encode(['success' => true, 'message' => 'Avatar mis à jour', 'avatar_path' => $relative_path]);
                    } else {
                        echo json_encode(['success' => false, 'message' => 'Erreur lors de la sauvegarde']);
                    }
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de l\'upload']);
                }
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Informations utilisateur
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        header('Location: login.php');
        exit;
    }
    
    // Préférences utilisateur
    $preferences = [
        'theme' => getUserSetting($pdo, $user_id, 'theme', 'moderne'),
        'dashboard_layout' => getUserSetting($pdo, $user_id, 'dashboard_layout', 'grid'),
        'notifications_enabled' => getUserSetting($pdo, $user_id, 'notifications_enabled', '1'),
        'auto_save' => getUserSetting($pdo, $user_id, 'auto_save', '1')
    ];
    
    // Statistiques utilisateur
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT p.id) as total_projects,
            COUNT(DISTINCT m.id) as total_media,
            COALESCE(SUM(m.file_size), 0) as storage_used,
            COUNT(DISTINCT a.id) as total_views
        FROM users u
        LEFT JOIN flying_book_projects p ON u.id = p.user_id AND p.status != 'deleted'
        LEFT JOIN flying_book_media m ON u.id = m.user_id AND m.upload_status = 'ready'
        LEFT JOIN flying_book_analytics a ON p.id = a.project_id
        WHERE u.id = ?
    ");
    $stmt->execute([$user_id]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Activité récente
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_activity_logs 
        WHERE user_id = ?
        ORDER BY created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$user_id]);
    $recent_activity = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement du profil: " . $e->getMessage();
    error_log("Profile page error: " . $e->getMessage());
}

function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    $k = 1024;
    $sizes = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes) / log($k));
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'À l\'instant';
    if ($time < 3600) return floor($time/60) . ' min';
    if ($time < 86400) return floor($time/3600) . ' h';
    if ($time < 2592000) return floor($time/86400) . ' j';
    return date('d/m/Y', strtotime($datetime));
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Mon Profil</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: var(--radius-xl);
            padding: var(--space-xl);
            color: var(--text-white);
            margin-bottom: var(--space-xl);
            display: flex;
            align-items: center;
            gap: var(--space-xl);
        }
        
        .profile-avatar-section {
            position: relative;
        }
        
        .profile-avatar {
            width: 120px;
            height: 120px;
            border-radius: var(--radius-full);
            background: rgba(255, 255, 255, 0.2);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: var(--font-size-4xl);
            color: var(--text-white);
            font-weight: 700;
            border: 4px solid rgba(255, 255, 255, 0.2);
            position: relative;
            overflow: hidden;
        }
        
        .profile-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .avatar-upload-btn {
            position: absolute;
            bottom: 0;
            right: 0;
            width: 35px;
            height: 35px;
            border-radius: var(--radius-full);
            background: var(--accent-color);
            border: 3px solid var(--text-white);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--text-white);
            transition: var(--transition-normal);
        }
        
        .avatar-upload-btn:hover {
            transform: scale(1.1);
        }
        
        .profile-info h1 {
            margin-bottom: var(--space-sm);
            font-size: var(--font-size-4xl);
        }
        
        .profile-info p {
            opacity: 0.9;
            font-size: var(--font-size-lg);
        }
        
        .profile-stats {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: var(--space-lg);
            margin-top: var(--space-xl);
        }
        
        .profile-stat {
            text-align: center;
            background: rgba(255, 255, 255, 0.1);
            padding: var(--space-md);
            border-radius: var(--radius-md);
        }
        
        .profile-stat h3 {
            font-size: var(--font-size-2xl);
            margin-bottom: var(--space-xs);
        }
        
        .profile-stat p {
            font-size: var(--font-size-sm);
            opacity: 0.8;
        }
        
        .profile-tabs {
            display: flex;
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-sm);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-sm);
        }
        
        .profile-tab {
            flex: 1;
            padding: var(--space-md);
            background: none;
            border: none;
            border-radius: var(--radius-sm);
            cursor: pointer;
            transition: var(--transition-normal);
            font-weight: 600;
            color: var(--text-secondary);
        }
        
        .profile-tab.active {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--text-white);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .form-section {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-xl);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-sm);
        }
        
        .form-section h3 {
            margin-bottom: var(--space-lg);
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: var(--space-sm);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--space-lg);
        }
        
        .activity-item {
            display: flex;
            align-items: center;
            gap: var(--space-md);
            padding: var(--space-md);
            border-radius: var(--radius-sm);
            margin-bottom: var(--space-sm);
            background: var(--bg-secondary);
        }
        
        .activity-icon {
            width: 40px;
            height: 40px;
            border-radius: var(--radius-full);
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--text-white);
            font-size: var(--font-size-sm);
        }
        
        .activity-info h4 {
            margin: 0 0 var(--space-xs) 0;
            font-size: var(--font-size-base);
            color: var(--text-primary);
        }
        
        .activity-info p {
            margin: 0;
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }
        
        .password-strength {
            margin-top: var(--space-sm);
            height: 4px;
            background: var(--bg-tertiary);
            border-radius: var(--radius-xs);
            overflow: hidden;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            transition: var(--transition-normal);
        }
        
        .password-strength.weak .password-strength-bar {
            width: 25%;
            background: var(--error-color);
        }
        
        .password-strength.fair .password-strength-bar {
            width: 50%;
            background: var(--warning-color);
        }
        
        .password-strength.good .password-strength-bar {
            width: 75%;
            background: var(--info-color);
        }
        
        .password-strength.strong .password-strength-bar {
            width: 100%;
            background: var(--success-color);
        }
        
        @media (max-width: 768px) {
            .profile-header {
                flex-direction: column;
                text-align: center;
            }
            
            .profile-stats {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <a href="dashboard.php" class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </a>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="media.php" class="nav-link">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>
                    <a href="analytics.html" class="nav-link">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?php if ($user['avatar']): ?>
                                <img src="<?= htmlspecialchars($user['avatar']) ?>" alt="Avatar">
                            <?php else: ?>
                                <?= strtoupper(substr($user['name'], 0, 1)) ?>
                            <?php endif; ?>
                        </div>
                        <span><?= htmlspecialchars($user['name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php" class="active"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- En-tête du profil -->
            <div class="profile-header">
                <div class="profile-avatar-section">
                    <div class="profile-avatar" id="profileAvatar">
                        <?php if ($user['avatar']): ?>
                            <img src="<?= htmlspecialchars($user['avatar']) ?>" alt="Avatar">
                        <?php else: ?>
                            <?= strtoupper(substr($user['name'], 0, 2)) ?>
                        <?php endif; ?>
                    </div>
                    <div class="avatar-upload-btn" onclick="document.getElementById('avatarUpload').click()">
                        <i class="fas fa-camera"></i>
                    </div>
                    <input type="file" id="avatarUpload" accept="image/*" style="display: none;" onchange="uploadAvatar()">
                </div>
                
                <div class="profile-info">
                    <h1><?= htmlspecialchars($user['name']) ?></h1>
                    <p><?= htmlspecialchars($user['email']) ?></p>
                    <p><i class="fas fa-shield-alt"></i> <?= ucfirst($user['role']) ?></p>
                    
                    <div class="profile-stats">
                        <div class="profile-stat">
                            <h3><?= $stats['total_projects'] ?></h3>
                            <p>Projets</p>
                        </div>
                        <div class="profile-stat">
                            <h3><?= $stats['total_media'] ?></h3>
                            <p>Médias</p>
                        </div>
                        <div class="profile-stat">
                            <h3><?= formatFileSize($stats['storage_used']) ?></h3>
                            <p>Stockage</p>
                        </div>
                        <div class="profile-stat">
                            <h3><?= $stats['total_views'] ?></h3>
                            <p>Vues</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Onglets du profil -->
            <div class="profile-tabs">
                <button class="profile-tab active" onclick="showTab('info')">
                    <i class="fas fa-user"></i> Informations
                </button>
                <button class="profile-tab" onclick="showTab('password')">
                    <i class="fas fa-lock"></i> Sécurité
                </button>
                <button class="profile-tab" onclick="showTab('preferences')">
                    <i class="fas fa-cog"></i> Préférences
                </button>
                <button class="profile-tab" onclick="showTab('activity')">
                    <i class="fas fa-history"></i> Activité
                </button>
            </div>

            <!-- Contenu des onglets -->
            
            <!-- Onglet Informations -->
            <div id="info-tab" class="tab-content active">
                <div class="form-section">
                    <h3><i class="fas fa-user"></i> Informations personnelles</h3>
                    <form onsubmit="updateProfile(event)">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Nom d'affichage *</label>
                                <input type="text" name="name" class="form-control" 
                                       value="<?= htmlspecialchars($user['name']) ?>" required>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Email *</label>
                                <input type="email" name="email" class="form-control" 
                                       value="<?= htmlspecialchars($user['email']) ?>" required>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Prénom</label>
                                <input type="text" name="first_name" class="form-control" 
                                       value="<?= htmlspecialchars($user['first_name'] ?? '') ?>">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Nom de famille</label>
                                <input type="text" name="last_name" class="form-control" 
                                       value="<?= htmlspecialchars($user['last_name'] ?? '') ?>">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Téléphone</label>
                                <input type="tel" name="phone" class="form-control" 
                                       value="<?= htmlspecialchars($user['phone'] ?? '') ?>">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Fuseau horaire</label>
                                <select name="timezone" class="form-control">
                                    <option value="Europe/Paris" <?= $user['timezone'] === 'Europe/Paris' ? 'selected' : '' ?>>Europe/Paris</option>
                                    <option value="Europe/London" <?= $user['timezone'] === 'Europe/London' ? 'selected' : '' ?>>Europe/London</option>
                                    <option value="America/New_York" <?= $user['timezone'] === 'America/New_York' ? 'selected' : '' ?>>America/New_York</option>
                                    <option value="America/Los_Angeles" <?= $user['timezone'] === 'America/Los_Angeles' ? 'selected' : '' ?>>America/Los_Angeles</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Langue</label>
                            <select name="language" class="form-control">
                                <option value="fr" <?= $user['language'] === 'fr' ? 'selected' : '' ?>>Français</option>
                                <option value="en" <?= $user['language'] === 'en' ? 'selected' : '' ?>>English</option>
                                <option value="es" <?= $user['language'] === 'es' ? 'selected' : '' ?>>Español</option>
                            </select>
                        </div>
                        
                        <div style="display: flex; justify-content: flex-end; gap: var(--space-md); margin-top: var(--space-xl);">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i>
                                Sauvegarder
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Onglet Sécurité -->
            <div id="password-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-lock"></i> Modifier le mot de passe</h3>
                    <form onsubmit="changePassword(event)">
                        <div class="form-group">
                            <label class="form-label">Mot de passe actuel *</label>
                            <input type="password" name="current_password" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Nouveau mot de passe *</label>
                            <input type="password" name="new_password" class="form-control" 
                                   id="newPassword" oninput="checkPasswordStrength()" required>
                            <div class="password-strength" id="passwordStrength">
                                <div class="password-strength-bar"></div>
                            </div>
                            <small class="text-muted">Minimum 8 caractères, avec majuscules, minuscules et chiffres</small>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Confirmer le nouveau mot de passe *</label>
                            <input type="password" name="confirm_password" class="form-control" required>
                        </div>
                        
                        <div style="display: flex; justify-content: flex-end; gap: var(--space-md); margin-top: var(--space-xl);">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-lock"></i>
                                Modifier le mot de passe
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Onglet Préférences -->
            <div id="preferences-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-cog"></i> Préférences d'interface</h3>
                    <form onsubmit="updatePreferences(event)">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Thème</label>
                                <select name="theme" class="form-control">
                                    <option value="moderne" <?= $preferences['theme'] === 'moderne' ? 'selected' : '' ?>>Moderne</option>
                                    <option value="classique" <?= $preferences['theme'] === 'classique' ? 'selected' : '' ?>>Classique</option>
                                    <option value="minimal" <?= $preferences['theme'] === 'minimal' ? 'selected' : '' ?>>Minimal</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Layout Dashboard</label>
                                <select name="dashboard_layout" class="form-control">
                                    <option value="grid" <?= $preferences['dashboard_layout'] === 'grid' ? 'selected' : '' ?>>Grille</option>
                                    <option value="list" <?= $preferences['dashboard_layout'] === 'list' ? 'selected' : '' ?>>Liste</option>
                                    <option value="compact" <?= $preferences['dashboard_layout'] === 'compact' ? 'selected' : '' ?>>Compact</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label style="display: flex; align-items: center; gap: var(--space-sm); cursor: pointer;">
                                <input type="checkbox" name="notifications_enabled" 
                                       <?= $preferences['notifications_enabled'] === '1' ? 'checked' : '' ?>>
                                <span>Activer les notifications</span>
                            </label>
                        </div>
                        
                        <div class="form-group">
                            <label style="display: flex; align-items: center; gap: var(--space-sm); cursor: pointer;">
                                <input type="checkbox" name="auto_save" 
                                       <?= $preferences['auto_save'] === '1' ? 'checked' : '' ?>>
                                <span>Sauvegarde automatique</span>
                            </label>
                        </div>
                        
                        <div style="display: flex; justify-content: flex-end; gap: var(--space-md); margin-top: var(--space-xl);">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i>
                                Sauvegarder les préférences
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Onglet Activité -->
            <div id="activity-tab" class="tab-content">
                <div class="form-section">
                    <h3><i class="fas fa-history"></i> Activité récente</h3>
                    
                    <?php if (empty($recent_activity)): ?>
                        <div class="empty-state">
                            <i class="fas fa-history"></i>
                            <h3>Aucune activité récente</h3>
                            <p>Votre activité apparaîtra ici</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($recent_activity as $activity): ?>
                            <div class="activity-item">
                                <div class="activity-icon">
                                    <?php
                                    $action_icons = [
                                        'login' => 'fa-sign-in-alt',
                                        'logout' => 'fa-sign-out-alt',
                                        'create_project' => 'fa-plus',
                                        'update_project' => 'fa-edit',
                                        'delete_project' => 'fa-trash',
                                        'upload_media' => 'fa-upload',
                                        'update_profile' => 'fa-user-edit',
                                        'change_password' => 'fa-lock'
                                    ];
                                    $icon = $action_icons[$activity['action']] ?? 'fa-info';
                                    ?>
                                    <i class="fas <?= $icon ?>"></i>
                                </div>
                                <div class="activity-info">
                                    <h4><?= htmlspecialchars($activity['description']) ?></h4>
                                    <p><?= timeAgo($activity['created_at']) ?> • IP: <?= htmlspecialchars($activity['ip_address'] ?? 'N/A') ?></p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <!-- Scripts -->
    <script>
        // ========== GESTION DES ONGLETS ==========
        function showTab(tabName) {
            // Masquer tous les contenus
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Désactiver tous les onglets
            document.querySelectorAll('.profile-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Afficher le contenu sélectionné
            document.getElementById(tabName + '-tab').classList.add('active');
            
            // Activer l'onglet sélectionné
            event.target.classList.add('active');
        }
        
        // ========== UPLOAD AVATAR ==========
        async function uploadAvatar() {
            const fileInput = document.getElementById('avatarUpload');
            const file = fileInput.files[0];
            
            if (!file) return;
            
            if (file.size > 5 * 1024 * 1024) {
                showNotification('Fichier trop volumineux (max 5MB)', 'error');
                return;
            }
            
            const formData = new FormData();
            formData.append('avatar', file);
            formData.append('ajax_action', 'upload_avatar');
            
            try {
                showNotification('Upload en cours...', 'info');
                
                const response = await fetch('profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Mettre à jour l'avatar dans l'interface
                    const avatarImg = `<img src="${result.avatar_path}" alt="Avatar">`;
                    document.getElementById('profileAvatar').innerHTML = avatarImg;
                    
                    // Mettre à jour l'avatar dans le header
                    const headerAvatar = document.querySelector('.header-right .avatar');
                    if (headerAvatar) {
                        headerAvatar.innerHTML = avatarImg;
                    }
                    
                    showNotification(result.message, 'success');
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de l\'upload', 'error');
            }
        }
        
        // ========== MISE À JOUR PROFIL ==========
        async function updateProfile(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            formData.append('ajax_action', 'update_profile');
            
            try {
                const response = await fetch('profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    // Recharger la page pour mettre à jour les informations
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la mise à jour', 'error');
            }
        }
        
        // ========== CHANGEMENT MOT DE PASSE ==========
        async function changePassword(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            formData.append('ajax_action', 'change_password');
            
            try {
                const response = await fetch('profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    event.target.reset();
                    document.getElementById('passwordStrength').className = 'password-strength';
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la modification', 'error');
            }
        }
        
        // ========== MISE À JOUR PRÉFÉRENCES ==========
        async function updatePreferences(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            formData.append('ajax_action', 'update_preferences');
            
            try {
                const response = await fetch('profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la sauvegarde', 'error');
            }
        }
        
        // ========== VÉRIFICATION FORCE MOT DE PASSE ==========
        function checkPasswordStrength() {
            const password = document.getElementById('newPassword').value;
            const strengthDiv = document.getElementById('passwordStrength');
            
            let strength = 0;
            
            if (password.length >= 8) strength++;
            if (/[a-z]/.test(password)) strength++;
            if (/[A-Z]/.test(password)) strength++;
            if (/[0-9]/.test(password)) strength++;
            if (/[^A-Za-z0-9]/.test(password)) strength++;
            
            const classes = ['', 'weak', 'fair', 'good', 'strong'];
            strengthDiv.className = `password-strength ${classes[strength]}`;
        }
        
        // ========== UTILITAIRES ==========
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // ========== SYSTÈME DE NOTIFICATIONS ==========
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.75rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.2rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.8;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, type === 'error' ? 8000 : 5000);
        }
        
        // Message de bienvenue
        document.addEventListener('DOMContentLoaded', function() {
            showNotification('Profil chargé ! ✨', 'success');
        });
    </script>
</body>
</html>