<?php
session_start();
require_once '../../env/.env.php';

header('Content-Type: application/json');

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Non authentifié']);
    exit;
}

// Vérification de la méthode
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Méthode non autorisée']);
    exit;
}

$project_id = intval($_POST['project_id'] ?? 0);

try {
    $pdo = getDatabaseConnection();
    
    // Vérifier que le projet appartient à l'utilisateur
    $stmt = $pdo->prepare("
        SELECT * FROM flying_book_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$project_id, $_SESSION['user_id']]);
    $project = $stmt->fetch();
    
    if (!$project) {
        throw new Exception("Projet non trouvé ou accès refusé");
    }
    
    // Vérifier les limites de l'utilisateur
    $limits = getUserPlanLimits($_SESSION['user_id']);
    
    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception("Erreur lors de l'upload du fichier");
    }
    
    $file = $_FILES['file'];
    $originalName = $file['name'];
    $tempPath = $file['tmp_name'];
    $fileSize = $file['size'];
    $mimeType = $file['type'];
    
    // Validation de la taille
    if ($fileSize > FLYING_BOOK_MAX_FILE_SIZE) {
        throw new Exception("Fichier trop volumineux. Maximum autorisé: " . formatFileSize(FLYING_BOOK_MAX_FILE_SIZE));
    }
    
    // Validation du type de fichier
    $allowedMimeTypes = [
        'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp',
        'video/mp4', 'video/webm', 'video/quicktime',
        'audio/mpeg', 'audio/mp3', 'audio/wav', 'audio/ogg'
    ];
    
    if (!in_array($mimeType, $allowedMimeTypes)) {
        throw new Exception("Type de fichier non autorisé: " . $mimeType);
    }
    
    // Vérifier les limites de stockage
    if (!checkUserLimits($_SESSION['user_id'], 'max_storage', $fileSize)) {
        throw new Exception("Limite de stockage dépassée pour votre plan");
    }
    
    // Déterminer le type de fichier
    $fileType = 'document';
    if (strpos($mimeType, 'image/') === 0) {
        $fileType = 'image';
    } elseif (strpos($mimeType, 'video/') === 0) {
        $fileType = 'video';
    } elseif (strpos($mimeType, 'audio/') === 0) {
        $fileType = 'audio';
    }
    
    // Générer un nom de fichier unique
    $extension = pathinfo($originalName, PATHINFO_EXTENSION);
    $storedFilename = uniqid() . '_' . time() . '.' . $extension;
    $storedFilename = sanitizeFileName($storedFilename);
    
    // Créer les répertoires du projet
    $projectPath = FLYING_BOOK_PROJECTS_PATH . $project['project_folder'];
    $mediaPath = $projectPath . '/media';
    $thumbnailPath = $projectPath . '/thumbnails';
    
    if (!is_dir($mediaPath)) {
        mkdir($mediaPath, 0755, true);
    }
    if (!is_dir($thumbnailPath)) {
        mkdir($thumbnailPath, 0755, true);
    }
    
    // Déplacer le fichier
    $targetPath = $mediaPath . '/' . $storedFilename;
    if (!move_uploaded_file($tempPath, $targetPath)) {
        throw new Exception("Erreur lors de la sauvegarde du fichier");
    }
    
    // Générer une miniature pour les images et vidéos
    $thumbnailGenerated = false;
    $thumbnailFile = null;
    
    if ($fileType === 'image') {
        $thumbnailFile = generateImageThumbnail($targetPath, $thumbnailPath, $storedFilename);
        $thumbnailGenerated = $thumbnailFile !== false;
    } elseif ($fileType === 'video') {
        $thumbnailFile = generateVideoThumbnail($targetPath, $thumbnailPath, $storedFilename);
        $thumbnailGenerated = $thumbnailFile !== false;
    }
    
    // Obtenir les dimensions pour les images
    $dimensions = null;
    $duration = null;
    
    if ($fileType === 'image') {
        $imageInfo = getimagesize($targetPath);
        if ($imageInfo) {
            $dimensions = $imageInfo[0] . 'x' . $imageInfo[1];
        }
    } elseif ($fileType === 'video') {
        // Utiliser ffprobe si disponible pour obtenir la durée
        $duration = getVideoDuration($targetPath);
    } elseif ($fileType === 'audio') {
        // Utiliser getID3 ou autre pour obtenir la durée
        $duration = getAudioDuration($targetPath);
    }
    
    // Insérer dans la base de données
    $stmt = $pdo->prepare("
        INSERT INTO flying_book_media 
        (project_id, user_id, original_filename, stored_filename, file_path, 
         file_type, mime_type, file_size, dimensions, duration, thumbnail_path, 
         upload_status, created_at) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ready', NOW())
    ");
    
    $stmt->execute([
        $project_id,
        $_SESSION['user_id'],
        $originalName,
        $storedFilename,
        $targetPath,
        $fileType,
        $mimeType,
        $fileSize,
        $dimensions,
        $duration,
        $thumbnailFile
    ]);
    
    $mediaId = $pdo->lastInsertId();
    
    // Mettre à jour la taille totale du projet
    $stmt = $pdo->prepare("
        UPDATE flying_book_projects 
        SET file_size_total = (
            SELECT SUM(file_size) 
            FROM flying_book_media 
            WHERE project_id = ? AND upload_status = 'ready'
        ), updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$project_id, $project_id]);
    
    // Réponse de succès
    echo json_encode([
        'success' => true,
        'message' => 'Fichier uploadé avec succès',
        'media' => [
            'id' => $mediaId,
            'original_filename' => $originalName,
            'stored_filename' => $storedFilename,
            'file_type' => $fileType,
            'file_size' => $fileSize,
            'dimensions' => $dimensions,
            'duration' => $duration,
            'thumbnail_path' => $thumbnailFile
        ]
    ]);
    
} catch (Exception $e) {
    logError("Erreur lors de l'upload", [
        'user_id' => $_SESSION['user_id'],
        'project_id' => $project_id,
        'error' => $e->getMessage()
    ]);
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

/**
 * Générer une miniature pour une image
 */
function generateImageThumbnail($sourcePath, $thumbnailDir, $filename) {
    try {
        $thumbnailPath = $thumbnailDir . '/thumb_' . $filename;
        $maxWidth = 300;
        $maxHeight = 200;
        
        // Obtenir les informations de l'image
        $imageInfo = getimagesize($sourcePath);
        if (!$imageInfo) {
            return false;
        }
        
        list($width, $height, $type) = $imageInfo;
        
        // Créer l'image source
        switch ($type) {
            case IMAGETYPE_JPEG:
                $source = imagecreatefromjpeg($sourcePath);
                break;
            case IMAGETYPE_PNG:
                $source = imagecreatefrompng($sourcePath);
                break;
            case IMAGETYPE_GIF:
                $source = imagecreatefromgif($sourcePath);
                break;
            case IMAGETYPE_WEBP:
                $source = imagecreatefromwebp($sourcePath);
                break;
            default:
                return false;
        }
        
        // Calculer les nouvelles dimensions
        $ratio = min($maxWidth / $width, $maxHeight / $height);
        $newWidth = intval($width * $ratio);
        $newHeight = intval($height * $ratio);
        
        // Créer la miniature
        $thumbnail = imagecreatetruecolor($newWidth, $newHeight);
        
        // Préserver la transparence pour PNG et GIF
        if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_GIF) {
            imagealphablending($thumbnail, false);
            imagesavealpha($thumbnail, true);
            $transparent = imagecolorallocatealpha($thumbnail, 255, 255, 255, 127);
            imagefilledrectangle($thumbnail, 0, 0, $newWidth, $newHeight, $transparent);
        }
        
        // Redimensionner
        imagecopyresampled($thumbnail, $source, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
        
        // Sauvegarder
        $success = false;
        switch ($type) {
            case IMAGETYPE_JPEG:
                $success = imagejpeg($thumbnail, $thumbnailPath, 85);
                break;
            case IMAGETYPE_PNG:
                $success = imagepng($thumbnail, $thumbnailPath, 8);
                break;
            case IMAGETYPE_GIF:
                $success = imagegif($thumbnail, $thumbnailPath);
                break;
            case IMAGETYPE_WEBP:
                $success = imagewebp($thumbnail, $thumbnailPath, 85);
                break;
        }
        
        // Nettoyer
        imagedestroy($source);
        imagedestroy($thumbnail);
        
        return $success ? 'thumb_' . $filename : false;
        
    } catch (Exception $e) {
        logError("Erreur génération miniature image", ['error' => $e->getMessage()]);
        return false;
    }
}

/**
 * Générer une miniature pour une vidéo
 */
function generateVideoThumbnail($sourcePath, $thumbnailDir, $filename) {
    try {
        $thumbnailPath = $thumbnailDir . '/thumb_' . pathinfo($filename, PATHINFO_FILENAME) . '.jpg';
        
        // Vérifier si ffmpeg est disponible
        $ffmpegPath = trim(shell_exec('which ffmpeg 2>/dev/null'));
        if (empty($ffmpegPath)) {
            // Essayer des chemins communs
            $commonPaths = ['/usr/bin/ffmpeg', '/usr/local/bin/ffmpeg', '/opt/homebrew/bin/ffmpeg'];
            foreach ($commonPaths as $path) {
                if (file_exists($path)) {
                    $ffmpegPath = $path;
                    break;
                }
            }
        }
        
        if (empty($ffmpegPath) || !file_exists($ffmpegPath)) {
            // Créer une miniature par défaut
            return createDefaultVideoThumbnail($thumbnailPath);
        }
        
        // Générer la miniature avec ffmpeg
        $command = sprintf(
            '%s -i %s -ss 00:00:01 -vframes 1 -vf "scale=300:200:force_original_aspect_ratio=decrease,pad=300:200:(ow-iw)/2:(oh-ih)/2" %s 2>/dev/null',
            escapeshellarg($ffmpegPath),
            escapeshellarg($sourcePath),
            escapeshellarg($thumbnailPath)
        );
        
        exec($command, $output, $returnCode);
        
        if ($returnCode === 0 && file_exists($thumbnailPath)) {
            return 'thumb_' . pathinfo($filename, PATHINFO_FILENAME) . '.jpg';
        } else {
            return createDefaultVideoThumbnail($thumbnailPath);
        }
        
    } catch (Exception $e) {
        logError("Erreur génération miniature vidéo", ['error' => $e->getMessage()]);
        return createDefaultVideoThumbnail($thumbnailPath);
    }
}

/**
 * Créer une miniature par défaut pour les vidéos
 */
function createDefaultVideoThumbnail($thumbnailPath) {
    try {
        $width = 300;
        $height = 200;
        
        $image = imagecreatetruecolor($width, $height);
        $bgColor = imagecolorallocate($image, 64, 75, 96); // Couleur sombre
        $textColor = imagecolorallocate($image, 255, 255, 255);
        
        imagefill($image, 0, 0, $bgColor);
        
        // Ajouter l'icône de lecture (triangle)
        $points = [
            $width/2 - 20, $height/2 - 15,
            $width/2 - 20, $height/2 + 15,
            $width/2 + 15, $height/2
        ];
        imagefilledpolygon($image, $points, 3, $textColor);
        
        // Sauvegarder
        $success = imagejpeg($image, $thumbnailPath, 85);
        imagedestroy($image);
        
        return $success ? basename($thumbnailPath) : false;
        
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Obtenir la durée d'une vidéo
 */
function getVideoDuration($filePath) {
    try {
        $ffprobePath = trim(shell_exec('which ffprobe 2>/dev/null'));
        if (empty($ffprobePath)) {
            $commonPaths = ['/usr/bin/ffprobe', '/usr/local/bin/ffprobe', '/opt/homebrew/bin/ffprobe'];
            foreach ($commonPaths as $path) {
                if (file_exists($path)) {
                    $ffprobePath = $path;
                    break;
                }
            }
        }
        
        if (empty($ffprobePath) || !file_exists($ffprobePath)) {
            return null;
        }
        
        $command = sprintf(
            '%s -v quiet -show_entries format=duration -of csv="p=0" %s 2>/dev/null',
            escapeshellarg($ffprobePath),
            escapeshellarg($filePath)
        );
        
        $duration = trim(shell_exec($command));
        
        return is_numeric($duration) ? floatval($duration) : null;
        
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Obtenir la durée d'un fichier audio
 */
function getAudioDuration($filePath) {
    // Pour l'instant, retourner null
    // Implémentation future avec getID3 ou ffprobe
    return null;
}
?>