<?php
session_start();
require_once './env/.env.php';

// Vérification de la connexion
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

$message = '';
$error = '';

// ========== DIAGNOSTIC ET RÉPARATION AUTOMATIQUE ==========
function createUploadDirectories() {
    $base_dir = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/';
    $dirs_to_create = [
        $base_dir,
        $base_dir . 'images/',
        $base_dir . 'videos/',
        $base_dir . 'audio/',
        $base_dir . 'documents/',
        $base_dir . 'thumbnails/',
        $base_dir . 'temp/'
    ];
    
    $created = [];
    $errors = [];
    
    foreach ($dirs_to_create as $dir) {
        if (!is_dir($dir)) {
            if (@mkdir($dir, 0755, true)) {
                $created[] = $dir;
                // Créer un fichier .htaccess pour la sécurité
                if (!file_exists($dir . '.htaccess')) {
                    file_put_contents($dir . '.htaccess', "Options -Indexes\nDeny from all\n<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|mp4|webm|mov|mp3|wav|ogg|pdf)$\">\n    Allow from all\n</FilesMatch>");
                }
            } else {
                $errors[] = $dir;
            }
        }
    }
    
    return ['created' => $created, 'errors' => $errors];
}

function checkUploadConfiguration() {
    return [
        'php_version' => PHP_VERSION,
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'upload_max_filesize_bytes' => return_bytes(ini_get('upload_max_filesize')),
        'post_max_size' => ini_get('post_max_size'),
        'post_max_size_bytes' => return_bytes(ini_get('post_max_size')),
        'max_execution_time' => ini_get('max_execution_time'),
        'memory_limit' => ini_get('memory_limit'),
        'file_uploads' => ini_get('file_uploads') ? 'ON' : 'OFF',
        'upload_tmp_dir' => ini_get('upload_tmp_dir') ?: sys_get_temp_dir(),
        'upload_tmp_dir_writable' => is_writable(ini_get('upload_tmp_dir') ?: sys_get_temp_dir()),
        'extensions' => [
            'gd' => extension_loaded('gd'),
            'imagick' => extension_loaded('imagick'),
            'fileinfo' => extension_loaded('fileinfo'),
        ]
    ];
}

function return_bytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = (float) $val;
    switch($last) {
        case 'g': $val *= 1024;
        case 'm': $val *= 1024;
        case 'k': $val *= 1024;
    }
    return $val;
}

// Configuration upload AMÉLIORÉE
$upload_dir = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/';
$allowed_types = [
    'image' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp'],
    'video' => ['mp4', 'webm', 'mov', 'avi', 'mkv'],
    'audio' => ['mp3', 'wav', 'ogg', 'aac', 'm4a'],
    'document' => ['pdf', 'doc', 'docx', 'ppt', 'pptx', 'txt']
];
$max_file_size = 200 * 1024 * 1024; // 200MB

// Créer automatiquement les répertoires
$dir_status = createUploadDirectories();

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['ajax_action']) {
            case 'diagnostic_system':
                $config = checkUploadConfiguration();
                $dir_check = createUploadDirectories();
                
                echo json_encode([
                    'success' => true,
                    'config' => $config,
                    'directories' => $dir_check,
                    'upload_dir' => $upload_dir,
                    'dir_exists' => is_dir($upload_dir),
                    'dir_writable' => is_writable($upload_dir)
                ]);
                exit;
                
            case 'upload_file':
                // ========== UPLOAD AMÉLIORÉ ==========
                if (!isset($_FILES['file'])) {
                    echo json_encode(['success' => false, 'message' => 'Aucun fichier sélectionné']);
                    exit;
                }
                
                $file = $_FILES['file'];
                $original_name = $file['name'];
                $file_size = $file['size'];
                $file_ext = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
                $temp_name = $file['tmp_name'];
                
                // Diagnostic détaillé des erreurs
                $upload_errors = [
                    UPLOAD_ERR_OK => 'Aucune erreur',
                    UPLOAD_ERR_INI_SIZE => 'Fichier trop volumineux (limite PHP)',
                    UPLOAD_ERR_FORM_SIZE => 'Fichier trop volumineux (limite formulaire)',
                    UPLOAD_ERR_PARTIAL => 'Upload partiel',
                    UPLOAD_ERR_NO_FILE => 'Aucun fichier',
                    UPLOAD_ERR_NO_TMP_DIR => 'Dossier temporaire manquant',
                    UPLOAD_ERR_CANT_WRITE => 'Écriture impossible',
                    UPLOAD_ERR_EXTENSION => 'Extension bloquée'
                ];
                
                if ($file['error'] !== UPLOAD_ERR_OK) {
                    $error_msg = $upload_errors[$file['error']] ?? 'Erreur inconnue';
                    echo json_encode([
                        'success' => false, 
                        'message' => "Erreur upload: {$error_msg}",
                        'debug' => [
                            'error_code' => $file['error'],
                            'php_config' => checkUploadConfiguration()
                        ]
                    ]);
                    exit;
                }
                
                // Déterminer le type de fichier
                $file_type = null;
                foreach ($allowed_types as $type => $extensions) {
                    if (in_array($file_ext, $extensions)) {
                        $file_type = $type;
                        break;
                    }
                }
                
                if (!$file_type) {
                    echo json_encode([
                        'success' => false, 
                        'message' => "Type de fichier non autorisé: {$file_ext}",
                        'allowed' => $allowed_types
                    ]);
                    exit;
                }
                
                if ($file_size > $max_file_size) {
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Fichier trop volumineux (max ' . formatFileSize($max_file_size) . ')'
                    ]);
                    exit;
                }
                
                // Vérification de sécurité supplémentaire
                if (!is_uploaded_file($temp_name)) {
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Fichier suspect détecté'
                    ]);
                    exit;
                }
                
                // Générer un nom unique et sécurisé
                $unique_name = uniqid() . '_' . time() . '.' . $file_ext;
                $folder = $file_type . 's/';
                $file_path = $upload_dir . $folder . $unique_name;
                $relative_path = '/flying-book/uploads/' . $folder . $unique_name;
                
                // Créer le dossier si nécessaire
                $target_dir = $upload_dir . $folder;
                if (!is_dir($target_dir)) {
                    if (!@mkdir($target_dir, 0755, true)) {
                        echo json_encode([
                            'success' => false, 
                            'message' => "Impossible de créer le dossier: {$target_dir}",
                            'debug' => [
                                'permissions' => substr(sprintf('%o', fileperms(dirname($target_dir))), -4),
                                'owner' => function_exists('posix_getpwuid') ? posix_getpwuid(fileowner(dirname($target_dir)))['name'] : 'unknown'
                            ]
                        ]);
                        exit;
                    }
                }
                
                // Déplacer le fichier avec vérification
                if (!move_uploaded_file($temp_name, $file_path)) {
                    $error_info = error_get_last();
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Erreur lors de la sauvegarde',
                        'debug' => [
                            'target_path' => $file_path,
                            'temp_path' => $temp_name,
                            'target_dir_writable' => is_writable($target_dir),
                            'target_dir_exists' => is_dir($target_dir),
                            'last_error' => $error_info
                        ]
                    ]);
                    exit;
                }
                
                // Vérifier que le fichier a bien été créé
                if (!file_exists($file_path)) {
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Le fichier n\'a pas été sauvegardé correctement'
                    ]);
                    exit;
                }
                
                // Générer une miniature si possible
                $thumbnail_path = null;
                if ($file_type === 'image') {
                    $thumbnail_path = generateImageThumbnail($file_path, $unique_name);
                }
                
                // Obtenir les métadonnées
                $dimensions = null;
                $duration = null;
                
                if ($file_type === 'image') {
                    $image_info = @getimagesize($file_path);
                    if ($image_info) {
                        $dimensions = $image_info[0] . 'x' . $image_info[1];
                    }
                }
                
                // Sauvegarder en base
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_media 
                    (user_id, original_filename, stored_filename, file_path, file_type, 
                     mime_type, file_size, dimensions, duration, thumbnail_path, upload_status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ready')
                ");
                
                $stmt->execute([
                    $user_id,
                    $original_name,
                    $unique_name,
                    $relative_path,
                    $file_type,
                    $file['type'],
                    $file_size,
                    $dimensions,
                    $duration,
                    $thumbnail_path
                ]);
                
                $media_id = $pdo->lastInsertId();
                
                // Log de l'activité
                logActivity($pdo, $user_id, 'upload_media', 'media', $media_id, "Fichier uploadé: {$original_name}");
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Fichier uploadé avec succès',
                    'media' => [
                        'id' => $media_id,
                        'name' => $original_name,
                        'type' => $file_type,
                        'size' => $file_size,
                        'path' => $relative_path,
                        'thumbnail' => $thumbnail_path
                    ]
                ]);
                exit;
                
            case 'delete_media':
                $media_id = (int)$_POST['media_id'];
                
                $stmt = $pdo->prepare("
                    SELECT * FROM flying_book_media 
                    WHERE id = ? AND (user_id = ? OR ? = 1)
                ");
                $stmt->execute([$media_id, $user_id, $is_admin]);
                $media = $stmt->fetch();
                
                if (!$media) {
                    echo json_encode(['success' => false, 'message' => 'Média non trouvé']);
                    exit;
                }
                
                // Supprimer les fichiers physiques
                $full_path = $_SERVER['DOCUMENT_ROOT'] . $media['file_path'];
                if (file_exists($full_path)) {
                    @unlink($full_path);
                }
                
                if ($media['thumbnail_path']) {
                    $thumb_path = $_SERVER['DOCUMENT_ROOT'] . $media['thumbnail_path'];
                    if (file_exists($thumb_path)) {
                        @unlink($thumb_path);
                    }
                }
                
                $stmt = $pdo->prepare("DELETE FROM flying_book_media WHERE id = ?");
                $stmt->execute([$media_id]);
                
                logActivity($pdo, $user_id, 'delete_media', 'media', $media_id, "Média supprimé: {$media['original_filename']}");
                
                echo json_encode(['success' => true, 'message' => 'Média supprimé']);
                exit;
                
            case 'update_meta':
                $media_id = (int)$_POST['media_id'];
                $alt_text = trim($_POST['alt_text'] ?? '');
                $title = trim($_POST['title'] ?? '');
                
                $stmt = $pdo->prepare("
                    UPDATE flying_book_media 
                    SET alt_text = ?, original_filename = ?, updated_at = NOW() 
                    WHERE id = ? AND (user_id = ? OR ? = 1)
                ");
                $stmt->execute([$alt_text, $title ?: null, $media_id, $user_id, $is_admin]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Métadonnées mises à jour']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la mise à jour']);
                }
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Paramètres de filtrage
    $type_filter = $_GET['type'] ?? '';
    $search = trim($_GET['search'] ?? '');
    $sort = $_GET['sort'] ?? 'newest';
    $view = $_GET['view'] ?? 'grid';
    
    // Construction de la requête
    $where_conditions = [];
    $params = [];
    
    if (!$is_admin) {
        $where_conditions[] = "user_id = ?";
        $params[] = $user_id;
    }
    
    if ($type_filter) {
        $where_conditions[] = "file_type = ?";
        $params[] = $type_filter;
    }
    
    if ($search) {
        $where_conditions[] = "(original_filename LIKE ? OR alt_text LIKE ?)";
        $search_param = "%$search%";
        $params = array_merge($params, [$search_param, $search_param]);
    }
    
    $where_conditions[] = "upload_status = 'ready'";
    
    // Tri
    $order_clause = match($sort) {
        'newest' => "created_at DESC",
        'oldest' => "created_at ASC", 
        'name' => "original_filename ASC",
        'size' => "file_size DESC",
        default => "created_at DESC"
    };
    
    $where_clause = empty($where_conditions) ? "1=1" : implode(' AND ', $where_conditions);
    
    // Récupérer les médias
    $stmt = $pdo->prepare("
        SELECT m.*, u.name as uploader_name
        FROM flying_book_media m
        LEFT JOIN users u ON m.user_id = u.id
        WHERE $where_clause
        ORDER BY $order_clause
    ");
    $stmt->execute($params);
    $media_files = $stmt->fetchAll();
    
    // Statistiques
    $stats_where = $is_admin ? "upload_status = 'ready'" : "user_id = $user_id AND upload_status = 'ready'";
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN file_type = 'image' THEN 1 ELSE 0 END) as images,
            SUM(CASE WHEN file_type = 'video' THEN 1 ELSE 0 END) as videos,
            SUM(CASE WHEN file_type = 'audio' THEN 1 ELSE 0 END) as audios,
            SUM(CASE WHEN file_type = 'document' THEN 1 ELSE 0 END) as documents,
            SUM(file_size) as total_size
        FROM flying_book_media 
        WHERE $stats_where
    ");
    $stats = $stmt->fetch();
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement de la médiathèque: " . $e->getMessage();
    logError("Media page error: " . $e->getMessage());
}

// ========== FONCTIONS UTILITAIRES ==========

function generateImageThumbnail($source_path, $filename) {
    if (!extension_loaded('gd')) {
        return null;
    }
    
    try {
        $thumbnail_dir = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/thumbnails/';
        $thumbnail_name = 'thumb_' . pathinfo($filename, PATHINFO_FILENAME) . '.jpg';
        $thumbnail_path = $thumbnail_dir . $thumbnail_name;
        $relative_path = '/flying-book/uploads/thumbnails/' . $thumbnail_name;
        
        if (!is_dir($thumbnail_dir)) {
            mkdir($thumbnail_dir, 0755, true);
        }
        
        $image_info = @getimagesize($source_path);
        if (!$image_info) return null;
        
        $source_image = match($image_info['mime']) {
            'image/jpeg' => @imagecreatefromjpeg($source_path),
            'image/png' => @imagecreatefrompng($source_path),
            'image/gif' => @imagecreatefromgif($source_path),
            'image/webp' => @imagecreatefromwebp($source_path),
            default => null
        };
        
        if (!$source_image) return null;
        
        $original_width = imagesx($source_image);
        $original_height = imagesy($source_image);
        
        $max_size = 300;
        if ($original_width > $original_height) {
            $new_width = $max_size;
            $new_height = ($original_height * $max_size) / $original_width;
        } else {
            $new_height = $max_size;
            $new_width = ($original_width * $max_size) / $original_height;
        }
        
        $thumbnail = imagecreatetruecolor($new_width, $new_height);
        
        // Préserver la transparence pour PNG
        if ($image_info['mime'] === 'image/png') {
            imagealphablending($thumbnail, false);
            imagesavealpha($thumbnail, true);
            $transparent = imagecolorallocatealpha($thumbnail, 255, 255, 255, 127);
            imagefill($thumbnail, 0, 0, $transparent);
        }
        
        imagecopyresampled($thumbnail, $source_image, 0, 0, 0, 0, $new_width, $new_height, $original_width, $original_height);
        
        if (@imagejpeg($thumbnail, $thumbnail_path, 85)) {
            imagedestroy($source_image);
            imagedestroy($thumbnail);
            return $relative_path;
        }
        
        imagedestroy($source_image);
        imagedestroy($thumbnail);
        return null;
        
    } catch (Exception $e) {
        error_log("Thumbnail generation error: " . $e->getMessage());
        return null;
    }
}

function logActivity($pdo, $user_id, $action, $entity_type, $entity_id, $description) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_activity_logs 
            (user_id, action, entity_type, entity_id, description, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user_id, $action, $entity_type, $entity_id, $description,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    } catch (Exception $e) {
        error_log("Failed to log activity: " . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Médiathèque v2.0</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Styles existants + améliorations */
        .diagnostic-panel {
            background: linear-gradient(135deg, #fbbf24, #f59e0b);
            border-radius: 16px;
            padding: 2rem;
            color: white;
            margin-bottom: 2rem;
        }
        
        .diagnostic-panel h3 {
            margin-bottom: 1rem;
            color: white;
        }
        
        .diagnostic-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .diagnostic-item {
            background: rgba(255, 255, 255, 0.2);
            border-radius: 8px;
            padding: 1rem;
            text-align: center;
        }
        
        .diagnostic-item.good {
            background: rgba(16, 185, 129, 0.3);
        }
        
        .diagnostic-item.warning {
            background: rgba(245, 158, 11, 0.3);
        }
        
        .diagnostic-item.error {
            background: rgba(239, 68, 68, 0.3);
        }
        
        .upload-area-enhanced {
            border: 3px dashed #d1d5db;
            border-radius: 16px;
            padding: 4rem 2rem;
            text-align: center;
            margin-bottom: 2rem;
            transition: all 0.3s ease;
            background: white;
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }
        
        .upload-area-enhanced.dragover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.05);
            transform: scale(1.02);
        }
        
        .upload-area-enhanced input[type="file"] {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            opacity: 0;
            cursor: pointer;
        }
        
        .upload-icon-enhanced {
            font-size: 4rem;
            color: #667eea;
            margin-bottom: 1rem;
            transition: all 0.3s ease;
        }
        
        .upload-area-enhanced:hover .upload-icon-enhanced {
            transform: scale(1.1);
            color: #5a67d8;
        }
        
        .progress-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(102, 126, 234, 0.9);
            display: none;
            align-items: center;
            justify-content: center;
            flex-direction: column;
            color: white;
        }
        
        .progress-circle {
            width: 80px;
            height: 80px;
            border: 4px solid rgba(255, 255, 255, 0.3);
            border-top: 4px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-bottom: 1rem;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .media-grid-enhanced {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .media-card-enhanced {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .media-card-enhanced:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 50px rgba(0, 0, 0, 0.2);
        }
        
        .media-preview-enhanced {
            height: 200px;
            position: relative;
            overflow: hidden;
            background: linear-gradient(135deg, #f8fafc, #e2e8f0);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .media-preview-enhanced img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.3s ease;
        }
        
        .media-card-enhanced:hover .media-preview-enhanced img {
            transform: scale(1.05);
        }
        
        .media-type-badge-enhanced {
            position: absolute;
            top: 1rem;
            left: 1rem;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            backdrop-filter: blur(10px);
        }
        
        .type-image {
            background: rgba(16, 185, 129, 0.9);
            color: white;
        }
        
        .type-video {
            background: rgba(239, 68, 68, 0.9);
            color: white;
        }
        
        .type-audio {
            background: rgba(245, 158, 11, 0.9);
            color: white;
        }
        
        .type-document {
            background: rgba(107, 114, 128, 0.9);
            color: white;
        }
        
        .media-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .media-card-enhanced:hover .media-overlay {
            opacity: 1;
        }
        
        .media-actions-overlay {
            display: flex;
            gap: 0.5rem;
        }
        
        .action-btn-overlay {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            border: none;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            color: white;
        }
        
        .action-btn-overlay.view {
            background: rgba(102, 126, 234, 0.9);
        }
        
        .action-btn-overlay.download {
            background: rgba(16, 185, 129, 0.9);
        }
        
        .action-btn-overlay.edit {
            background: rgba(245, 158, 11, 0.9);
        }
        
        .action-btn-overlay.delete {
            background: rgba(239, 68, 68, 0.9);
        }
        
        .action-btn-overlay:hover {
            transform: scale(1.1);
        }
        
        .media-info-enhanced {
            padding: 1.5rem;
        }
        
        .media-name-enhanced {
            font-weight: 700;
            color: #1f2937;
            margin-bottom: 0.5rem;
            font-size: 1rem;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .media-meta-enhanced {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.85rem;
            color: #6b7280;
            margin-bottom: 1rem;
        }
        
        .quick-stats {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            margin-bottom: 2rem;
        }
        
        .stats-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
        }
        
        .stat-item {
            text-align: center;
            padding: 1rem;
            border-radius: 8px;
            background: #f8fafc;
        }
        
        .stat-number {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1f2937;
            margin-bottom: 0.25rem;
        }
        
        .stat-label {
            font-size: 0.8rem;
            color: #6b7280;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .filters-enhanced {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            margin-bottom: 2rem;
        }
        
        .filters-grid {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }
        
        @media (max-width: 1024px) {
            .filters-grid {
                grid-template-columns: 1fr;
            }
        }
        
        .modal-enhanced {
            display: none;
            position: fixed;
            z-index: 2000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(5px);
        }
        
        .modal-content-enhanced {
            background: white;
            margin: 3% auto;
            padding: 0;
            border-radius: 16px;
            width: 90%;
            max-width: 800px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
        }
        
        .modal-header-enhanced {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 1.5rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-body-enhanced {
            padding: 2rem;
            max-height: 70vh;
            overflow-y: auto;
        }
        
        .notification-enhanced {
            position: fixed;
            top: 2rem;
            right: 2rem;
            background: white;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            z-index: 3000;
            transform: translateX(400px);
            transition: transform 0.3s ease;
            border-left: 4px solid;
            min-width: 300px;
        }
        
        .notification-enhanced.show {
            transform: translateX(0);
        }
        
        .notification-enhanced.success {
            border-left-color: #10b981;
        }
        
        .notification-enhanced.error {
            border-left-color: #ef4444;
        }
        
        .notification-enhanced.warning {
            border-left-color: #f59e0b;
        }
        
        .notification-enhanced.info {
            border-left-color: #3b82f6;
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                    <a href="media.php" class="nav-link active">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>
                    <a href="analytics.html" class="nav-link">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($_SESSION['user_name'], 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($_SESSION['user_name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- Page Header -->
            <div class="page-header">
                <div class="header-content">
                    <h1><i class="fas fa-images"></i> Médiathèque v2.0</h1>
                    <p>Gérez vos médias avec diagnostic intégré</p>
                </div>
                <div class="header-actions">
                    <button class="btn btn-outline" onclick="runDiagnostic()">
                        <i class="fas fa-stethoscope"></i>
                        Diagnostic
                    </button>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Panneau de diagnostic -->
            <div class="diagnostic-panel" id="diagnosticPanel" style="display: none;">
                <h3><i class="fas fa-heartbeat"></i> Diagnostic Système</h3>
                <p>État de la configuration d'upload</p>
                <div class="diagnostic-grid" id="diagnosticGrid">
                    <!-- Contenu généré dynamiquement -->
                </div>
            </div>

            <!-- Zone d'upload améliorée -->
            <div class="upload-area-enhanced" id="uploadArea">
                <input type="file" id="fileInput" multiple accept="image/*,video/*,audio/*,.pdf,.doc,.docx,.ppt,.pptx">
                <div class="upload-icon-enhanced">
                    <i class="fas fa-cloud-upload-alt"></i>
                </div>
                <h3>Glissez vos fichiers ici ou cliquez pour sélectionner</h3>
                <p>Formats supportés : Images, Vidéos, Audio, Documents PDF/Office</p>
                <p>Taille max : <?= formatFileSize($max_file_size) ?></p>
                
                <div class="progress-overlay" id="progressOverlay">
                    <div class="progress-circle"></div>
                    <div id="progressText">Upload en cours...</div>
                </div>
            </div>

            <!-- Statistiques rapides -->
            <div class="quick-stats">
                <div class="stats-row">
                    <div class="stat-item">
                        <div class="stat-number"><?= $stats['total'] ?></div>
                        <div class="stat-label">Total fichiers</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?= $stats['images'] ?></div>
                        <div class="stat-label">Images</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?= $stats['videos'] ?></div>
                        <div class="stat-label">Vidéos</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?= $stats['audios'] ?></div>
                        <div class="stat-label">Audio</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?= formatFileSize($stats['total_size'] ?? 0) ?></div>
                        <div class="stat-label">Stockage</div>
                    </div>
                </div>
            </div>

            <!-- Filtres -->
            <div class="filters-enhanced">
                <form method="GET" id="filtersForm">
                    <div class="filters-grid">
                        <div class="filter-group">
                            <label class="filter-label">Rechercher</label>
                            <input type="text" name="search" class="filter-input" 
                                   placeholder="Nom de fichier..." 
                                   value="<?= htmlspecialchars($search) ?>">
                        </div>
                        
                        <div class="filter-group">
                            <label class="filter-label">Type</label>
                            <select name="type" class="filter-input">
                                <option value="">Tous les types</option>
                                <option value="image" <?= $type_filter === 'image' ? 'selected' : '' ?>>Images</option>
                                <option value="video" <?= $type_filter === 'video' ? 'selected' : '' ?>>Vidéos</option>
                                <option value="audio" <?= $type_filter === 'audio' ? 'selected' : '' ?>>Audio</option>
                                <option value="document" <?= $type_filter === 'document' ? 'selected' : '' ?>>Documents</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label class="filter-label">Trier par</label>
                            <select name="sort" class="filter-input">
                                <option value="newest" <?= $sort === 'newest' ? 'selected' : '' ?>>Plus récents</option>
                                <option value="oldest" <?= $sort === 'oldest' ? 'selected' : '' ?>>Plus anciens</option>
                                <option value="name" <?= $sort === 'name' ? 'selected' : '' ?>>Nom A-Z</option>
                                <option value="size" <?= $sort === 'size' ? 'selected' : '' ?>>Taille</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <label class="filter-label">Vue</label>
                            <select name="view" class="filter-input" onchange="toggleView(this.value)">
                                <option value="grid" <?= $view === 'grid' ? 'selected' : '' ?>>Grille</option>
                                <option value="list" <?= $view === 'list' ? 'selected' : '' ?>>Liste</option>
                            </select>
                        </div>
                        
                        <div class="filter-group">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Liste des médias -->
            <?php if (empty($media_files)): ?>
                <div class="empty-state">
                    <i class="fas fa-images"></i>
                    <h3>Aucun média trouvé</h3>
                    <p>
                        <?php if ($search || $type_filter): ?>
                            Essayez de modifier vos critères de recherche.
                        <?php else: ?>
                            Commencez par uploader vos premiers fichiers !
                        <?php endif; ?>
                    </p>
                </div>
            <?php else: ?>
                <div class="media-grid-enhanced" id="mediaGrid">
                    <?php foreach ($media_files as $media): ?>
                        <div class="media-card-enhanced" data-media-id="<?= $media['id'] ?>">
                            <div class="media-preview-enhanced">
                                <?php if ($media['thumbnail_path']): ?>
                                    <img src="<?= htmlspecialchars($media['thumbnail_path']) ?>" 
                                         alt="<?= htmlspecialchars($media['original_filename']) ?>">
                                <?php else: ?>
                                    <div class="media-icon" style="font-size: 4rem; color: #9ca3af;">
                                        <?php
                                        $icon_map = [
                                            'image' => 'fa-image',
                                            'video' => 'fa-video',
                                            'audio' => 'fa-music',
                                            'document' => 'fa-file-pdf'
                                        ];
                                        $icon = $icon_map[$media['file_type']] ?? 'fa-file';
                                        ?>
                                        <i class="fas <?= $icon ?>"></i>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="media-type-badge-enhanced type-<?= $media['file_type'] ?>">
                                    <?= ucfirst($media['file_type']) ?>
                                </div>
                                
                                <div class="media-overlay">
                                    <div class="media-actions-overlay">
                                        <button class="action-btn-overlay view" onclick="viewMedia(<?= $media['id'] ?>)" title="Voir">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button class="action-btn-overlay download" onclick="downloadMedia('<?= htmlspecialchars($media['file_path']) ?>')" title="Télécharger">
                                            <i class="fas fa-download"></i>
                                        </button>
                                        <button class="action-btn-overlay edit" onclick="editMedia(<?= $media['id'] ?>)" title="Éditer">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <?php if ($media['user_id'] == $user_id || $is_admin): ?>
                                            <button class="action-btn-overlay delete" onclick="deleteMedia(<?= $media['id'] ?>, '<?= htmlspecialchars($media['original_filename']) ?>')" title="Supprimer">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="media-info-enhanced">
                                <div class="media-name-enhanced" title="<?= htmlspecialchars($media['original_filename']) ?>">
                                    <?= htmlspecialchars($media['original_filename']) ?>
                                </div>
                                
                                <div class="media-meta-enhanced">
                                    <span class="media-size">
                                        <?= formatFileSize($media['file_size']) ?>
                                    </span>
                                    <span class="media-date">
                                        <?= date('d/m/Y', strtotime($media['created_at'])) ?>
                                    </span>
                                </div>
                                
                                <?php if ($media['dimensions']): ?>
                                    <div style="font-size: 0.8rem; color: #9ca3af; text-align: center;">
                                        <?= htmlspecialchars($media['dimensions']) ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Modal pour éditer un média -->
    <div id="editMediaModal" class="modal-enhanced">
        <div class="modal-content-enhanced">
            <div class="modal-header-enhanced">
                <h3><i class="fas fa-edit"></i> Modifier le média</h3>
                <button class="modal-close" onclick="closeModal('editMediaModal')" style="background: none; border: none; color: white; font-size: 1.5rem; cursor: pointer;">&times;</button>
            </div>
            
            <div class="modal-body-enhanced">
                <form onsubmit="updateMedia(event)">
                    <input type="hidden" id="editMediaId" name="media_id">
                    
                    <div class="form-group">
                        <label class="form-label">Nom du fichier</label>
                        <input type="text" id="editTitle" name="title" class="form-control" 
                               placeholder="Nom d'affichage">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Texte alternatif (Alt)</label>
                        <input type="text" id="editAltText" name="alt_text" class="form-control" 
                               placeholder="Description du média pour l'accessibilité">
                    </div>
                    
                    <div style="display: flex; gap: 1rem; justify-content: flex-end; margin-top: 2rem;">
                        <button type="button" class="btn btn-secondary" onclick="closeModal('editMediaModal')">
                            Annuler
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            Sauvegarder
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script>
        // ========== VARIABLES GLOBALES ==========
        let currentView = '<?= $view ?>';
        let uploadInProgress = false;
        
        // ========== INITIALISATION ==========
        document.addEventListener('DOMContentLoaded', function() {
            initializeUpload();
            showNotification('Médiathèque v2.0 chargée avec diagnostic intégré ! 📁', 'success');
        });
        
        // ========== GESTION UPLOAD ==========
        function initializeUpload() {
            const uploadArea = document.getElementById('uploadArea');
            const fileInput = document.getElementById('fileInput');
            
            uploadArea.addEventListener('click', () => {
                if (!uploadInProgress) fileInput.click();
            });
            
            uploadArea.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadArea.classList.add('dragover');
            });
            
            uploadArea.addEventListener('dragleave', () => {
                uploadArea.classList.remove('dragover');
            });
            
            uploadArea.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadArea.classList.remove('dragover');
                if (!uploadInProgress) handleFiles(e.dataTransfer.files);
            });
            
            fileInput.addEventListener('change', (e) => {
                if (!uploadInProgress) handleFiles(e.target.files);
            });
        }
        
        async function handleFiles(files) {
            if (files.length === 0) return;
            
            uploadInProgress = true;
            showProgress(true);
            
            let successCount = 0;
            let errorCount = 0;
            
            for (let i = 0; i < files.length; i++) {
                updateProgress(`Upload ${i + 1}/${files.length}: ${files[i].name}`);
                
                try {
                    await uploadFile(files[i]);
                    successCount++;
                } catch (error) {
                    errorCount++;
                    console.error('Upload error:', error);
                }
            }
            
            showProgress(false);
            uploadInProgress = false;
            
            if (successCount > 0) {
                showNotification(`${successCount} fichier(s) uploadé(s) avec succès`, 'success');
                setTimeout(() => location.reload(), 1500);
            }
            
            if (errorCount > 0) {
                showNotification(`${errorCount} erreur(s) d'upload`, 'warning');
            }
        }
        
        async function uploadFile(file) {
            const formData = new FormData();
            formData.append('file', file);
            formData.append('ajax_action', 'upload_file');
            
            const response = await fetch('media.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (!result.success) {
                // Afficher les détails de l'erreur en mode debug
                if (result.debug) {
                    console.error('Upload debug info:', result.debug);
                    showNotification(`Erreur: ${result.message}`, 'error');
                    
                    // Suggestions automatiques de réparation
                    if (result.message.includes('Permission denied') || result.message.includes('Écriture impossible')) {
                        showNotification('💡 Suggestion: Vérifiez les permissions des dossiers d\'upload', 'info');
                        setTimeout(() => runDiagnostic(), 2000);
                    }
                } else {
                    showNotification(result.message, 'error');
                }
                throw new Error(result.message);
            }
            
            return result;
        }
        
        function showProgress(show, text = 'Upload en cours...') {
            const overlay = document.getElementById('progressOverlay');
            const progressText = document.getElementById('progressText');
            
            overlay.style.display = show ? 'flex' : 'none';
            if (text) progressText.textContent = text;
        }
        
        function updateProgress(text) {
            const progressText = document.getElementById('progressText');
            progressText.textContent = text;
        }
        
        // ========== DIAGNOSTIC SYSTÈME ==========
        async function runDiagnostic() {
            showNotification('Diagnostic en cours...', 'info');
            
            try {
                const response = await fetch('media.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'ajax_action=diagnostic_system'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    displayDiagnostic(result);
                    showNotification('Diagnostic terminé', 'success');
                } else {
                    showNotification('Erreur lors du diagnostic', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        function displayDiagnostic(data) {
            const panel = document.getElementById('diagnosticPanel');
            const grid = document.getElementById('diagnosticGrid');
            
            let html = '';
            const config = data.config;
            
            // Tests de configuration
            const tests = [
                {
                    name: 'PHP Version',
                    value: config.php_version,
                    status: parseFloat(config.php_version) >= 7.4 ? 'good' : 'warning'
                },
                {
                    name: 'File Uploads',
                    value: config.file_uploads,
                    status: config.file_uploads === 'ON' ? 'good' : 'error'
                },
                {
                    name: 'Upload Max',
                    value: config.upload_max_filesize,
                    status: config.upload_max_filesize_bytes >= (50 * 1024 * 1024) ? 'good' : 'warning'
                },
                {
                    name: 'POST Max',
                    value: config.post_max_size,
                    status: config.post_max_size_bytes >= (50 * 1024 * 1024) ? 'good' : 'warning'
                },
                {
                    name: 'Upload Dir',
                    value: data.dir_exists ? 'OK' : 'Manquant',
                    status: data.dir_exists ? 'good' : 'error'
                },
                {
                    name: 'Permissions',
                    value: data.dir_writable ? 'OK' : 'Lecture seule',
                    status: data.dir_writable ? 'good' : 'error'
                },
                {
                    name: 'GD Extension',
                    value: config.extensions.gd ? 'OK' : 'Manquante',
                    status: config.extensions.gd ? 'good' : 'warning'
                },
                {
                    name: 'Temp Dir',
                    value: config.upload_tmp_dir_writable ? 'OK' : 'Problème',
                    status: config.upload_tmp_dir_writable ? 'good' : 'error'
                }
            ];
            
            tests.forEach(test => {
                html += `
                    <div class="diagnostic-item ${test.status}">
                        <strong>${test.name}</strong><br>
                        <span>${test.value}</span>
                    </div>
                `;
            });
            
            // Suggestions de réparation
            const hasErrors = tests.some(t => t.status === 'error');
            if (hasErrors) {
                html += `
                    <div class="diagnostic-item error" style="grid-column: 1 / -1;">
                        <strong>🔧 Actions de réparation :</strong><br>
                        <small>
                            ${!data.dir_exists ? '• Créer le dossier uploads/<br>' : ''}
                            ${!data.dir_writable ? '• chmod 755 ' + data.upload_dir + '<br>' : ''}
                            ${config.file_uploads !== 'ON' ? '• Activer file_uploads dans php.ini<br>' : ''}
                            ${!config.upload_tmp_dir_writable ? '• Vérifier les permissions du dossier temp<br>' : ''}
                        </small>
                    </div>
                `;
            }
            
            grid.innerHTML = html;
            panel.style.display = 'block';
        }
        
        // ========== ACTIONS MÉDIAS ==========
        async function deleteMedia(mediaId, filename) {
            if (!confirm(`Supprimer "${filename}" ?\n\nCette action est irréversible.`)) {
                return;
            }
            
            try {
                const response = await fetch('media.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=delete_media&media_id=${mediaId}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    document.querySelector(`[data-media-id="${mediaId}"]`).remove();
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la suppression', 'error');
            }
        }
        
        function editMedia(mediaId) {
            // Récupérer les données actuelles
            const card = document.querySelector(`[data-media-id="${mediaId}"]`);
            const filename = card.querySelector('.media-name-enhanced').textContent.trim();
            
            document.getElementById('editMediaId').value = mediaId;
            document.getElementById('editTitle').value = filename;
            document.getElementById('editAltText').value = '';
            
            document.getElementById('editMediaModal').style.display = 'block';
        }
        
        async function updateMedia(event) {
            event.preventDefault();
            const formData = new FormData(event.target);
            formData.append('ajax_action', 'update_meta');
            
            try {
                const response = await fetch('media.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    closeModal('editMediaModal');
                    // Optionnel: recharger la page pour voir les changements
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de la mise à jour', 'error');
            }
        }
        
        function viewMedia(mediaId) {
            const card = document.querySelector(`[data-media-id="${mediaId}"]`);
            const img = card.querySelector('img');
            if (img) {
                window.open(img.src.replace('/thumbnails/thumb_', '/images/'), '_blank');
            } else {
                showNotification('Aperçu non disponible', 'info');
            }
        }
        
        function downloadMedia(filePath) {
            const link = document.createElement('a');
            link.href = filePath;
            link.download = '';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
        
        // ========== UTILITAIRES ==========
        function toggleView(view) {
            currentView = view;
            // Ici vous pouvez ajouter la logique pour changer l'affichage
            showNotification(`Vue changée vers: ${view}`, 'info');
        }
        
        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }
        
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Auto-submit des filtres
        document.getElementById('filtersForm').addEventListener('change', function(e) {
            if (e.target.name !== 'view') {
                this.submit();
            }
        });
        
        // Fermer les modals en cliquant à l'extérieur
        window.onclick = function(event) {
            if (event.target.classList.contains('modal-enhanced')) {
                event.target.style.display = 'none';
            }
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // ========== SYSTÈME DE NOTIFICATIONS ==========
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification-enhanced').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification-enhanced ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.75rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.2rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.8;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, type === 'error' ? 8000 : 5000);
        }
    </script>
</body>
</html>