<?php
/**
 * Flying Book Creator v2.0 - Logic Builder Interface
 * Interface visuelle pour créer des flowcharts et connecter la logique aux slides
 */

session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

try {
    $pdo = getDatabaseConnection();
    
    // Récupérer les projets de l'utilisateur pour le sélecteur
    $stmt = $pdo->prepare("
        SELECT id, project_name, title 
        FROM flying_book_projects 
        WHERE user_id = ? AND status != 'deleted' 
        ORDER BY updated_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Si un projet est sélectionné, récupérer ses slides
    $selectedProject = null;
    $slides = [];
    if (!empty($_GET['project_id'])) {
        $projectId = intval($_GET['project_id']);
        
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_projects 
            WHERE id = ? AND user_id = ? AND status != 'deleted'
        ");
        $stmt->execute([$projectId, $_SESSION['user_id']]);
        $selectedProject = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($selectedProject) {
            $stmt = $pdo->prepare("
                SELECT * FROM flying_book_slides 
                WHERE project_id = ? AND is_active = 1 
                ORDER BY display_order ASC, slide_number ASC
            ");
            $stmt->execute([$projectId]);
            $slides = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
    }
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'save_logic_flow':
                $projectId = intval($_POST['project_id'] ?? 0);
                $logicData = $_POST['logic_data'] ?? '';
                
                if (!$projectId) {
                    echo json_encode(['success' => false, 'error' => 'ID projet manquant']);
                    exit;
                }
                
                try {
                    // Sauvegarder ou mettre à jour la configuration avancée
                    $stmt = $pdo->prepare("
                        INSERT INTO flying_book_project_configs (project_id, advanced_config) 
                        VALUES (?, ?) 
                        ON DUPLICATE KEY UPDATE 
                        advanced_config = VALUES(advanced_config),
                        updated_at = CURRENT_TIMESTAMP
                    ");
                    
                    $config = [
                        'logic_flow' => json_decode($logicData, true),
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $stmt->execute([$projectId, json_encode($config)]);
                    
                    echo json_encode(['success' => true, 'message' => 'Logic flow sauvegardé']);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
                }
                exit;
                
            case 'load_logic_flow':
                $projectId = intval($_POST['project_id'] ?? 0);
                
                if (!$projectId) {
                    echo json_encode(['success' => false, 'error' => 'ID projet manquant']);
                    exit;
                }
                
                try {
                    $stmt = $pdo->prepare("
                        SELECT advanced_config 
                        FROM flying_book_project_configs 
                        WHERE project_id = ?
                    ");
                    $stmt->execute([$projectId]);
                    $config = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    $logicFlow = [];
                    if ($config && $config['advanced_config']) {
                        $configData = json_decode($config['advanced_config'], true);
                        $logicFlow = $configData['logic_flow'] ?? [];
                    }
                    
                    echo json_encode(['success' => true, 'logic_flow' => $logicFlow]);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
                }
                exit;
                
            case 'generate_logic_preview':
                $projectId = intval($_POST['project_id'] ?? 0);
                $logicData = $_POST['logic_data'] ?? '';
                
                if (!$projectId) {
                    echo json_encode(['success' => false, 'error' => 'ID projet manquant']);
                    exit;
                }
                
                // Générer aperçu avec logique
                $previewHTML = generateLogicPreview($projectId, json_decode($logicData, true));
                
                echo json_encode([
                    'success' => true, 
                    'preview_html' => $previewHTML
                ]);
                exit;
        }
    }
    
} catch (PDOException $e) {
    $error = "Erreur de base de données : " . $e->getMessage();
}

/**
 * Générer aperçu avec logique appliquée
 */
function generateLogicPreview($projectId, $logicFlow) {
    // Ici on pourrait générer un aperçu HTML avec la logique appliquée
    return '<div class="logic-preview">Aperçu avec logique appliquée...</div>';
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Logic Builder v2.0</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* ========== LOGIC BUILDER STYLES ========== */
        .logic-builder-container {
            display: grid;
            grid-template-columns: 300px 1fr 350px;
            height: calc(100vh - 70px);
            gap: 1rem;
            padding: 1rem;
        }
        
        .logic-sidebar {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            overflow-y: auto;
        }
        
        .logic-canvas {
            background: #f8fafc;
            border-radius: 12px;
            position: relative;
            overflow: hidden;
            border: 2px dashed #e2e8f0;
            cursor: default;
        }
        
        .logic-properties {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            overflow-y: auto;
        }
        
        .project-selector {
            margin-bottom: 2rem;
            padding: 1rem;
            background: linear-gradient(135deg, #667eea, #764ba2);
            border-radius: 8px;
            color: white;
        }
        
        .project-selector select {
            width: 100%;
            padding: 0.5rem;
            border: none;
            border-radius: 4px;
            background: white;
        }
        
        .node-category {
            margin-bottom: 1.5rem;
        }
        
        .node-category h3 {
            color: var(--gray-700);
            font-size: 0.9rem;
            font-weight: 600;
            margin-bottom: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .node-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            background: var(--gray-50);
            border-radius: 8px;
            margin-bottom: 0.5rem;
            cursor: grab;
            transition: all 0.3s ease;
            border: 1px solid transparent;
        }
        
        .node-item:hover {
            background: var(--primary-color);
            color: white;
            transform: translateX(5px);
        }
        
        .node-item:active {
            cursor: grabbing;
        }
        
        .node-icon {
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            font-size: 0.9rem;
            color: white;
        }
        
        .flow-node {
            position: absolute;
            background: white;
            border-radius: 8px;
            padding: 1rem;
            min-width: 150px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            cursor: move;
            user-select: none;
            border: 2px solid transparent;
            transition: all 0.3s ease;
        }
        
        .flow-node:hover {
            border-color: var(--primary-color);
            transform: translateY(-2px);
        }
        
        .flow-node.active {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(240, 147, 251, 0.2);
        }
        
        .flow-node.slide-node {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }
        
        .flow-node.condition-node {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
        }
        
        .flow-node.action-node {
            background: linear-gradient(135deg, #4facfe, #00f2fe);
            color: white;
        }
        
        .flow-node.logic-node {
            background: linear-gradient(135deg, #f59e0b, #d97706);
            color: white;
        }
        
        .node-header {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0.5rem;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .node-content {
            font-size: 0.8rem;
            opacity: 0.9;
        }
        
        .connection-point {
            position: absolute;
            width: 12px;
            height: 12px;
            background: white;
            border: 2px solid var(--primary-color);
            border-radius: 50%;
            cursor: crosshair;
            transition: all 0.3s ease;
        }
        
        .connection-point:hover {
            transform: scale(1.3);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.2);
        }
        
        .connection-point.input {
            left: -6px;
            top: 50%;
            transform: translateY(-50%);
        }
        
        .connection-point.output {
            right: -6px;
            top: 50%;
            transform: translateY(-50%);
        }
        
        .connection-point.connecting {
            background: var(--accent-color);
            animation: pulse 1s infinite;
        }
        
        .canvas-grid {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            opacity: 0.1;
            background-image: 
                linear-gradient(rgba(0,0,0,0.1) 1px, transparent 1px),
                linear-gradient(90deg, rgba(0,0,0,0.1) 1px, transparent 1px);
            background-size: 20px 20px;
        }
        
        .logic-toolbar {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 1rem;
            padding: 1rem;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        }
        
        .toolbar-btn {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: var(--gray-100);
            border: none;
            border-radius: 6px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 0.9rem;
        }
        
        .toolbar-btn:hover {
            background: var(--primary-color);
            color: white;
        }
        
        .toolbar-btn.active {
            background: var(--primary-color);
            color: white;
        }
        
        .property-group {
            margin-bottom: 1.5rem;
        }
        
        .property-group h4 {
            color: var(--gray-700);
            font-size: 0.9rem;
            margin-bottom: 0.75rem;
            font-weight: 600;
        }
        
        .property-field {
            margin-bottom: 1rem;
        }
        
        .property-label {
            display: block;
            font-size: 0.8rem;
            font-weight: 500;
            color: var(--gray-600);
            margin-bottom: 0.25rem;
        }
        
        .property-input {
            width: 100%;
            padding: 0.5rem;
            border: 1px solid var(--gray-300);
            border-radius: 4px;
            font-size: 0.9rem;
        }
        
        .property-input:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 2px rgba(102, 126, 234, 0.1);
        }
        
        .mini-preview {
            background: var(--gray-50);
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }
        
        .preview-slide {
            aspect-ratio: 16/9;
            background: white;
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8rem;
            color: var(--gray-500);
            border: 1px solid var(--gray-200);
        }
        
        .flow-controls {
            position: absolute;
            top: 1rem;
            right: 1rem;
            display: flex;
            gap: 0.5rem;
            z-index: 100;
        }
        
        .control-btn {
            width: 40px;
            height: 40px;
            background: white;
            border: 1px solid var(--gray-300);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .control-btn:hover {
            background: var(--primary-color);
            color: white;
            transform: scale(1.1);
        }
        
        .connection-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 2000;
        }
        
        .modal-content {
            background: white;
            border-radius: 12px;
            padding: 2rem;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .condition-builder {
            background: var(--gray-50);
            border-radius: 8px;
            padding: 1rem;
            margin: 1rem 0;
        }
        
        .condition-row {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 0.5rem;
            align-items: center;
        }
        
        .condition-select {
            flex: 1;
            padding: 0.5rem;
            border: 1px solid var(--gray-300);
            border-radius: 4px;
        }
        
        .slides-list {
            max-height: 200px;
            overflow-y: auto;
            border: 1px solid var(--gray-200);
            border-radius: 4px;
            padding: 0.5rem;
        }
        
        .slide-option {
            padding: 0.5rem;
            cursor: pointer;
            border-radius: 4px;
            transition: background 0.3s ease;
        }
        
        .slide-option:hover {
            background: var(--gray-100);
        }
        
        .slide-option.selected {
            background: var(--primary-color);
            color: white;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .logic-builder-container {
                grid-template-columns: 250px 1fr 300px;
            }
        }
        
        @media (max-width: 768px) {
            .logic-builder-container {
                grid-template-columns: 1fr;
                grid-template-rows: auto 1fr auto;
                height: auto;
                min-height: calc(100vh - 70px);
            }
            
            .logic-sidebar,
            .logic-properties {
                max-height: 200px;
            }
        }
        
        /* Animations */
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .connection-line {
            stroke: var(--primary-color);
            stroke-width: 2;
            fill: none;
            marker-end: url(#arrowhead);
        }
        
        .connection-line.temporary {
            stroke-dasharray: 5,5;
            animation: dash 1s linear infinite;
        }
        
        @keyframes dash {
            to { stroke-dashoffset: -10; }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-sitemap"></i>
                    <span>Logic Builder v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="logic_builder.php" class="nav-link active">
                        <i class="fas fa-sitemap"></i> Logic Builder
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <button class="btn btn-primary" onclick="saveLogicFlow()">
                    <i class="fas fa-save"></i> Sauvegarder
                </button>
                <button class="btn btn-outline" onclick="previewLogic()">
                    <i class="fas fa-play"></i> Tester
                </button>
                <button class="btn btn-secondary" onclick="exportLogic()">
                    <i class="fas fa-download"></i> Exporter
                </button>
            </div>
        </div>
    </header>

    <main class="main-content">
        <!-- Toolbar -->
        <div class="logic-toolbar">
            <button class="toolbar-btn active" data-mode="select">
                <i class="fas fa-mouse-pointer"></i> Sélection
            </button>
            <button class="toolbar-btn" data-mode="connect">
                <i class="fas fa-link"></i> Connexion
            </button>
            <button class="toolbar-btn" data-mode="pan">
                <i class="fas fa-hand-paper"></i> Navigation
            </button>
            <div style="flex: 1;"></div>
            <button class="toolbar-btn" onclick="LogicBuilder.zoomIn()">
                <i class="fas fa-search-plus"></i>
            </button>
            <button class="toolbar-btn" onclick="LogicBuilder.zoomOut()">
                <i class="fas fa-search-minus"></i>
            </button>
            <button class="toolbar-btn" onclick="LogicBuilder.resetZoom()">
                <i class="fas fa-expand"></i>
            </button>
            <button class="toolbar-btn" onclick="LogicBuilder.clearCanvas()">
                <i class="fas fa-trash"></i> Effacer
            </button>
        </div>
        
        <div class="logic-builder-container">
            <!-- Sidebar des composants -->
            <div class="logic-sidebar">
                <!-- Sélecteur de projet -->
                <div class="project-selector">
                    <h3><i class="fas fa-project-diagram"></i> Projet</h3>
                    <select id="projectSelector" onchange="loadProject(this.value)">
                        <option value="">Sélectionner un projet...</option>
                        <?php foreach ($projects as $project): ?>
                            <option value="<?= $project['id'] ?>" <?= $selectedProject && $selectedProject['id'] == $project['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($project['title'] ?: $project['project_name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <h2><i class="fas fa-puzzle-piece"></i> Composants</h2>
                
                <div class="node-category">
                    <h3>📍 Points d'entrée</h3>
                    <div class="node-item" draggable="true" data-node-type="start">
                        <div class="node-icon" style="background: #10b981;"><i class="fas fa-play"></i></div>
                        <div>
                            <div style="font-weight: 600;">Démarrage</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Point d'entrée</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="user_action">
                        <div class="node-icon" style="background: #3b82f6;"><i class="fas fa-mouse-pointer"></i></div>
                        <div>
                            <div style="font-weight: 600;">Action Utilisateur</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Clic, hover, etc.</div>
                        </div>
                    </div>
                </div>
                
                <div class="node-category">
                    <h3>🎞️ Slides</h3>
                    <div class="node-item" draggable="true" data-node-type="slide">
                        <div class="node-icon" style="background: #667eea;"><i class="fas fa-presentation"></i></div>
                        <div>
                            <div style="font-weight: 600;">Afficher Slide</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Présenter contenu</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="slide_transition">
                        <div class="node-icon" style="background: #8b5cf6;"><i class="fas fa-exchange-alt"></i></div>
                        <div>
                            <div style="font-weight: 600;">Transition</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Passage animé</div>
                        </div>
                    </div>
                </div>
                
                <div class="node-category">
                    <h3>🧠 Logique</h3>
                    <div class="node-item" draggable="true" data-node-type="condition">
                        <div class="node-icon" style="background: #f59e0b;"><i class="fas fa-question"></i></div>
                        <div>
                            <div style="font-weight: 600;">Condition</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Si... alors...</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="variable">
                        <div class="node-icon" style="background: #06b6d4;"><i class="fas fa-code"></i></div>
                        <div>
                            <div style="font-weight: 600;">Variable</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Stocker données</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="timer">
                        <div class="node-icon" style="background: #84cc16;"><i class="fas fa-clock"></i></div>
                        <div>
                            <div style="font-weight: 600;">Timer</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Temporisation</div>
                        </div>
                    </div>
                </div>
                
                <div class="node-category">
                    <h3>⚡ Actions</h3>
                    <div class="node-item" draggable="true" data-node-type="show_popup">
                        <div class="node-icon" style="background: #ec4899;"><i class="fas fa-window-restore"></i></div>
                        <div>
                            <div style="font-weight: 600;">Popup</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Afficher modal</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="send_data">
                        <div class="node-icon" style="background: #14b8a6;"><i class="fas fa-paper-plane"></i></div>
                        <div>
                            <div style="font-weight: 600;">Envoyer Données</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Analytics, lead</div>
                        </div>
                    </div>
                    
                    <div class="node-item" draggable="true" data-node-type="redirect">
                        <div class="node-icon" style="background: #f97316;"><i class="fas fa-external-link-alt"></i></div>
                        <div>
                            <div style="font-weight: 600;">Redirection</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Vers URL externe</div>
                        </div>
                    </div>
                </div>
                
                <div class="node-category">
                    <h3>🎯 Points de sortie</h3>
                    <div class="node-item" draggable="true" data-node-type="end">
                        <div class="node-icon" style="background: #ef4444;"><i class="fas fa-stop"></i></div>
                        <div>
                            <div style="font-weight: 600;">Fin</div>
                            <div style="font-size: 0.7rem; opacity: 0.7;">Terminer flow</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Canvas principal -->
            <div class="logic-canvas" id="logicCanvas">
                <div class="canvas-grid"></div>
                
                <!-- Contrôles de zoom -->
                <div class="flow-controls">
                    <div class="control-btn" onclick="LogicBuilder.zoomIn()" title="Zoom +">
                        <i class="fas fa-plus"></i>
                    </div>
                    <div class="control-btn" onclick="LogicBuilder.zoomOut()" title="Zoom -">
                        <i class="fas fa-minus"></i>
                    </div>
                    <div class="control-btn" onclick="LogicBuilder.resetZoom()" title="Reset">
                        <i class="fas fa-expand"></i>
                    </div>
                    <div class="control-btn" onclick="LogicBuilder.autoLayout()" title="Auto Layout">
                        <i class="fas fa-magic"></i>
                    </div>
                </div>
                
                <!-- SVG pour les connexions -->
                <svg id="connectionSvg" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; pointer-events: none;">
                    <defs>
                        <marker id="arrowhead" markerWidth="10" markerHeight="7" 
                                refX="9" refY="3.5" orient="auto">
                            <polygon points="0 0, 10 3.5, 0 7" fill="#667eea" />
                        </marker>
                    </defs>
                </svg>
            </div>
            
            <!-- Panneau des propriétés -->
            <div class="logic-properties">
                <h2><i class="fas fa-cog"></i> Propriétés</h2>
                <div id="propertiesContent">
                    <div style="text-align: center; padding: 2rem; color: var(--gray-500);">
                        <i class="fas fa-mouse-pointer" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                        <p>Sélectionnez un nœud pour voir ses propriétés</p>
                    </div>
                </div>
                
                <!-- Aperçu mini -->
                <div class="mini-preview">
                    <h4>Aperçu Logic Flow</h4>
                    <div class="preview-slide" id="logicPreview">
                        <i class="fas fa-sitemap"></i>
                        Diagramme logique
                    </div>
                    <button class="btn btn-outline btn-sm" onclick="generatePreview()" style="width: 100%; margin-top: 0.5rem;">
                        <i class="fas fa-eye"></i> Générer aperçu
                    </button>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Modal de connexion -->
    <div class="connection-modal" id="connectionModal">
        <div class="modal-content">
            <h3>Configurer la connexion</h3>
            <div class="condition-builder">
                <div class="condition-row">
                    <label>Condition:</label>
                    <select class="condition-select" id="connectionCondition">
                        <option value="always">Toujours</option>
                        <option value="click">Si utilisateur clique</option>
                        <option value="time">Si temps écoulé</option>
                        <option value="variable">Si variable égale à</option>
                        <option value="slide_end">Si slide terminée</option>
                    </select>
                </div>
                <div class="condition-row" id="conditionalInputs" style="display: none;">
                    <input type="text" class="condition-select" id="conditionValue" placeholder="Valeur...">
                </div>
            </div>
            
            <div style="display: flex; gap: 1rem; justify-content: flex-end; margin-top: 1.5rem;">
                <button class="btn btn-secondary" onclick="closeConnectionModal()">Annuler</button>
                <button class="btn btn-primary" onclick="confirmConnection()">Confirmer</button>
            </div>
        </div>
    </div>

    <script>
        // ========== LOGIC BUILDER ENGINE ==========
        
        class LogicBuilderEngine {
            constructor() {
                this.canvas = document.getElementById('logicCanvas');
                this.svg = document.getElementById('connectionSvg');
                this.nodes = new Map();
                this.connections = [];
                this.selectedNode = null;
                this.mode = 'select';
                this.zoom = 1;
                this.pan = { x: 0, y: 0 };
                this.isDragging = false;
                this.dragStart = { x: 0, y: 0 };
                this.connectionStart = null;
                this.nodeCounter = 0;
                this.currentProject = <?= json_encode($selectedProject) ?>;
                this.availableSlides = <?= json_encode($slides) ?>;
                
                this.init();
            }
            
            init() {
                this.setupEventListeners();
                this.setupDragAndDrop();
                this.setupToolbar();
                
                // Créer un nœud de démarrage par défaut
                setTimeout(() => {
                    if (this.currentProject) {
                        this.createNode('start', { x: 100, y: 200 });
                        this.loadExistingFlow();
                    }
                }, 100);
            }
            
            setupEventListeners() {
                // Canvas events
                this.canvas.addEventListener('click', (e) => this.handleCanvasClick(e));
                this.canvas.addEventListener('mousemove', (e) => this.handleCanvasMouseMove(e));
                this.canvas.addEventListener('mousedown', (e) => this.handleCanvasMouseDown(e));
                this.canvas.addEventListener('mouseup', (e) => this.handleCanvasMouseUp(e));
                
                // Zoom avec molette
                this.canvas.addEventListener('wheel', (e) => {
                    e.preventDefault();
                    const delta = e.deltaY > 0 ? 0.9 : 1.1;
                    this.setZoom(this.zoom * delta);
                });
                
                // Raccourcis clavier
                document.addEventListener('keydown', (e) => {
                    if (e.key === 'Delete' && this.selectedNode) {
                        this.deleteNode(this.selectedNode);
                    }
                    if (e.key === 'Escape') {
                        this.deselectAll();
                    }
                    if (e.ctrlKey && e.key === 's') {
                        e.preventDefault();
                        saveLogicFlow();
                    }
                });
            }
            
            setupDragAndDrop() {
                // Drag depuis la sidebar
                document.querySelectorAll('.node-item[draggable="true"]').forEach(item => {
                    item.addEventListener('dragstart', (e) => {
                        e.dataTransfer.setData('text/plain', e.target.dataset.nodeType || e.target.closest('[data-node-type]').dataset.nodeType);
                    });
                });
                
                // Drop sur le canvas
                this.canvas.addEventListener('dragover', (e) => {
                    e.preventDefault();
                });
                
                this.canvas.addEventListener('drop', (e) => {
                    e.preventDefault();
                    const nodeType = e.dataTransfer.getData('text/plain');
                    const rect = this.canvas.getBoundingClientRect();
                    const x = (e.clientX - rect.left) / this.zoom - this.pan.x;
                    const y = (e.clientY - rect.top) / this.zoom - this.pan.y;
                    
                    this.createNode(nodeType, { x, y });
                });
            }
            
            setupToolbar() {
                document.querySelectorAll('.toolbar-btn[data-mode]').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        document.querySelector('.toolbar-btn.active').classList.remove('active');
                        btn.classList.add('active');
                        this.mode = btn.dataset.mode;
                        this.canvas.style.cursor = this.getCursorForMode(this.mode);
                    });
                });
            }
            
            getCursorForMode(mode) {
                switch (mode) {
                    case 'select': return 'default';
                    case 'connect': return 'crosshair';
                    case 'pan': return 'grab';
                    default: return 'default';
                }
            }
            
            createNode(type, position) {
                const nodeId = `node_${++this.nodeCounter}`;
                const nodeConfig = this.getNodeConfig(type);
                
                const nodeElement = document.createElement('div');
                nodeElement.className = `flow-node ${nodeConfig.class}`;
                nodeElement.style.left = position.x + 'px';
                nodeElement.style.top = position.y + 'px';
                nodeElement.setAttribute('data-node-id', nodeId);
                
                nodeElement.innerHTML = `
                    <div class="node-header">
                        <i class="${nodeConfig.icon}"></i>
                        <span>${nodeConfig.title}</span>
                    </div>
                    <div class="node-content">${nodeConfig.description}</div>
                    ${nodeConfig.hasInput ? '<div class="connection-point input" data-type="input"></div>' : ''}
                    ${nodeConfig.hasOutput ? '<div class="connection-point output" data-type="output"></div>' : ''}
                `;
                
                // Event listeners pour le nœud
                nodeElement.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.selectNode(nodeId);
                });
                
                nodeElement.addEventListener('mousedown', (e) => {
                    if (this.mode === 'select') {
                        this.startNodeDrag(nodeId, e);
                    }
                });
                
                // Connection points
                const connectionPoints = nodeElement.querySelectorAll('.connection-point');
                connectionPoints.forEach(point => {
                    point.addEventListener('click', (e) => {
                        e.stopPropagation();
                        this.handleConnectionPoint(nodeId, point.dataset.type);
                    });
                });
                
                this.canvas.appendChild(nodeElement);
                
                // Sauvegarder le nœud
                this.nodes.set(nodeId, {
                    id: nodeId,
                    type: type,
                    position: position,
                    element: nodeElement,
                    config: nodeConfig,
                    properties: { ...nodeConfig.defaultProperties }
                });
                
                // Sélectionner automatiquement
                this.selectNode(nodeId);
                
                return nodeId;
            }
            
            getNodeConfig(type) {
                const configs = {
                    start: {
                        title: 'Démarrage',
                        description: 'Point d\'entrée du flow',
                        icon: 'fas fa-play',
                        class: 'slide-node',
                        hasInput: false,
                        hasOutput: true,
                        defaultProperties: {
                            autoStart: true,
                            delay: 0
                        }
                    },
                    slide: {
                        title: 'Afficher Slide',
                        description: 'Présenter du contenu',
                        icon: 'fas fa-presentation',
                        class: 'slide-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            slideId: null,
                            autoAdvance: false,
                            duration: 5000
                        }
                    },
                    condition: {
                        title: 'Condition',
                        description: 'Branchement logique',
                        icon: 'fas fa-question',
                        class: 'condition-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            variable: '',
                            operator: 'equals',
                            value: '',
                            trueAction: null,
                            falseAction: null
                        }
                    },
                    user_action: {
                        title: 'Action Utilisateur',
                        description: 'Déclencheur interactif',
                        icon: 'fas fa-mouse-pointer',
                        class: 'action-node',
                        hasInput: false,
                        hasOutput: true,
                        defaultProperties: {
                            actionType: 'click',
                            element: 'button',
                            waitFor: true
                        }
                    },
                    timer: {
                        title: 'Timer',
                        description: 'Temporisation',
                        icon: 'fas fa-clock',
                        class: 'logic-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            duration: 3000,
                            pausable: true,
                            showProgress: false
                        }
                    },
                    variable: {
                        title: 'Variable',
                        description: 'Stocker données',
                        icon: 'fas fa-code',
                        class: 'logic-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            name: '',
                            operation: 'set',
                            value: '',
                            dataType: 'string'
                        }
                    },
                    show_popup: {
                        title: 'Popup',
                        description: 'Afficher modal',
                        icon: 'fas fa-window-restore',
                        class: 'action-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            title: 'Information',
                            content: '',
                            size: 'medium',
                            modal: true
                        }
                    },
                    send_data: {
                        title: 'Envoyer Données',
                        description: 'Analytics, lead capture',
                        icon: 'fas fa-paper-plane',
                        class: 'action-node',
                        hasInput: true,
                        hasOutput: true,
                        defaultProperties: {
                            endpoint: '',
                            method: 'POST',
                            data: '{}',
                            async: true
                        }
                    },
                    redirect: {
                        title: 'Redirection',
                        description: 'Vers URL externe',
                        icon: 'fas fa-external-link-alt',
                        class: 'action-node',
                        hasInput: true,
                        hasOutput: false,
                        defaultProperties: {
                            url: '',
                            target: '_blank',
                            delay: 0
                        }
                    },
                    end: {
                        title: 'Fin',
                        description: 'Terminer le flow',
                        icon: 'fas fa-stop',
                        class: 'condition-node',
                        hasInput: true,
                        hasOutput: false,
                        defaultProperties: {
                            message: '',
                            analytics: true
                        }
                    }
                };
                
                return configs[type] || configs.slide;
            }
            
            selectNode(nodeId) {
                // Désélectionner le précédent
                if (this.selectedNode) {
                    const prevNode = this.nodes.get(this.selectedNode);
                    if (prevNode) {
                        prevNode.element.classList.remove('active');
                    }
                }
                
                // Sélectionner le nouveau
                this.selectedNode = nodeId;
                const node = this.nodes.get(nodeId);
                if (node) {
                    node.element.classList.add('active');
                    this.showNodeProperties(node);
                }
            }
            
            showNodeProperties(node) {
                const propertiesContent = document.getElementById('propertiesContent');
                let html = `
                    <div class="property-group">
                        <h4><i class="${node.config.icon}"></i> ${node.config.title}</h4>
                        <p style="font-size: 0.8rem; color: var(--gray-600); margin-bottom: 1rem;">
                            ${node.config.description}
                        </p>
                    </div>
                `;
                
                // Propriétés spécifiques selon le type
                switch (node.type) {
                    case 'slide':
                        html += this.getSlideProperties(node);
                        break;
                    case 'condition':
                        html += this.getConditionProperties(node);
                        break;
                    case 'timer':
                        html += this.getTimerProperties(node);
                        break;
                    case 'variable':
                        html += this.getVariableProperties(node);
                        break;
                    case 'show_popup':
                        html += this.getPopupProperties(node);
                        break;
                    case 'send_data':
                        html += this.getDataProperties(node);
                        break;
                    case 'redirect':
                        html += this.getRedirectProperties(node);
                        break;
                    default:
                        html += this.getGenericProperties(node);
                }
                
                // Actions du nœud
                html += `
                    <div class="property-group">
                        <button class="btn btn-outline btn-sm" onclick="LogicBuilder.duplicateNode('${node.id}')" style="width: 100%; margin-bottom: 0.5rem;">
                            <i class="fas fa-copy"></i> Dupliquer
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="LogicBuilder.deleteNode('${node.id}')" style="width: 100%;">
                            <i class="fas fa-trash"></i> Supprimer
                        </button>
                    </div>
                `;
                
                propertiesContent.innerHTML = html;
            }
            
            getSlideProperties(node) {
                let html = `
                    <div class="property-group">
                        <h4>Configuration Slide</h4>
                        <div class="property-field">
                            <label class="property-label">Slide à afficher:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'slideId', this.value)">
                                <option value="">Sélectionner une slide...</option>
                `;
                
                this.availableSlides.forEach(slide => {
                    const selected = node.properties.slideId == slide.id ? 'selected' : '';
                    html += `<option value="${slide.id}" ${selected}>
                        Slide ${slide.slide_number}: ${slide.slide_title || 'Sans titre'}
                    </option>`;
                });
                
                html += `
                            </select>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Avancement automatique:</label>
                            <input type="checkbox" ${node.properties.autoAdvance ? 'checked' : ''} 
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'autoAdvance', this.checked)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Durée (ms):</label>
                            <input type="number" class="property-input" value="${node.properties.duration || 5000}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'duration', parseInt(this.value))">
                        </div>
                    </div>
                `;
                
                return html;
            }
            
            getConditionProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Condition</h4>
                        <div class="property-field">
                            <label class="property-label">Variable:</label>
                            <input type="text" class="property-input" value="${node.properties.variable || ''}"
                                   placeholder="nom_variable"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'variable', this.value)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Opérateur:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'operator', this.value)">
                                <option value="equals" ${node.properties.operator === 'equals' ? 'selected' : ''}>Égal à</option>
                                <option value="not_equals" ${node.properties.operator === 'not_equals' ? 'selected' : ''}>Différent de</option>
                                <option value="greater" ${node.properties.operator === 'greater' ? 'selected' : ''}>Plus grand que</option>
                                <option value="less" ${node.properties.operator === 'less' ? 'selected' : ''}>Plus petit que</option>
                                <option value="contains" ${node.properties.operator === 'contains' ? 'selected' : ''}>Contient</option>
                            </select>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Valeur:</label>
                            <input type="text" class="property-input" value="${node.properties.value || ''}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'value', this.value)">
                        </div>
                    </div>
                `;
            }
            
            getTimerProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Timer</h4>
                        <div class="property-field">
                            <label class="property-label">Durée (ms):</label>
                            <input type="number" class="property-input" value="${node.properties.duration || 3000}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'duration', parseInt(this.value))">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Pausable:</label>
                            <input type="checkbox" ${node.properties.pausable ? 'checked' : ''} 
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'pausable', this.checked)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Afficher progression:</label>
                            <input type="checkbox" ${node.properties.showProgress ? 'checked' : ''} 
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'showProgress', this.checked)">
                        </div>
                    </div>
                `;
            }
            
            getVariableProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Variable</h4>
                        <div class="property-field">
                            <label class="property-label">Nom de la variable:</label>
                            <input type="text" class="property-input" value="${node.properties.name || ''}"
                                   placeholder="ma_variable"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'name', this.value)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Opération:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'operation', this.value)">
                                <option value="set" ${node.properties.operation === 'set' ? 'selected' : ''}>Définir</option>
                                <option value="add" ${node.properties.operation === 'add' ? 'selected' : ''}>Ajouter</option>
                                <option value="subtract" ${node.properties.operation === 'subtract' ? 'selected' : ''}>Soustraire</option>
                                <option value="multiply" ${node.properties.operation === 'multiply' ? 'selected' : ''}>Multiplier</option>
                            </select>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Valeur:</label>
                            <input type="text" class="property-input" value="${node.properties.value || ''}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'value', this.value)">
                        </div>
                    </div>
                `;
            }
            
            getPopupProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Popup</h4>
                        <div class="property-field">
                            <label class="property-label">Titre:</label>
                            <input type="text" class="property-input" value="${node.properties.title || 'Information'}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'title', this.value)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Contenu:</label>
                            <textarea class="property-input" rows="3" 
                                      onchange="LogicBuilder.updateNodeProperty('${node.id}', 'content', this.value)">${node.properties.content || ''}</textarea>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Taille:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'size', this.value)">
                                <option value="small" ${node.properties.size === 'small' ? 'selected' : ''}>Petite</option>
                                <option value="medium" ${node.properties.size === 'medium' ? 'selected' : ''}>Moyenne</option>
                                <option value="large" ${node.properties.size === 'large' ? 'selected' : ''}>Grande</option>
                            </select>
                        </div>
                    </div>
                `;
            }
            
            getDataProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Envoi</h4>
                        <div class="property-field">
                            <label class="property-label">Endpoint:</label>
                            <input type="url" class="property-input" value="${node.properties.endpoint || ''}"
                                   placeholder="https://api.example.com/data"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'endpoint', this.value)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Méthode:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'method', this.value)">
                                <option value="POST" ${node.properties.method === 'POST' ? 'selected' : ''}>POST</option>
                                <option value="PUT" ${node.properties.method === 'PUT' ? 'selected' : ''}>PUT</option>
                                <option value="PATCH" ${node.properties.method === 'PATCH' ? 'selected' : ''}>PATCH</option>
                            </select>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Données JSON:</label>
                            <textarea class="property-input" rows="3" 
                                      onchange="LogicBuilder.updateNodeProperty('${node.id}', 'data', this.value)">${node.properties.data || '{}'}</textarea>
                        </div>
                    </div>
                `;
            }
            
            getRedirectProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Configuration Redirection</h4>
                        <div class="property-field">
                            <label class="property-label">URL de destination:</label>
                            <input type="url" class="property-input" value="${node.properties.url || ''}"
                                   placeholder="https://example.com"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'url', this.value)">
                        </div>
                        <div class="property-field">
                            <label class="property-label">Cible:</label>
                            <select class="property-input" onchange="LogicBuilder.updateNodeProperty('${node.id}', 'target', this.value)">
                                <option value="_self" ${node.properties.target === '_self' ? 'selected' : ''}>Même fenêtre</option>
                                <option value="_blank" ${node.properties.target === '_blank' ? 'selected' : ''}>Nouvelle fenêtre</option>
                            </select>
                        </div>
                        <div class="property-field">
                            <label class="property-label">Délai (ms):</label>
                            <input type="number" class="property-input" value="${node.properties.delay || 0}"
                                   onchange="LogicBuilder.updateNodeProperty('${node.id}', 'delay', parseInt(this.value))">
                        </div>
                    </div>
                `;
            }
            
            getGenericProperties(node) {
                return `
                    <div class="property-group">
                        <h4>Propriétés génériques</h4>
                        <p style="color: var(--gray-600); font-size: 0.8rem;">
                            Ce type de nœud n'a pas de propriétés spécifiques configurables.
                        </p>
                    </div>
                `;
            }
            
            updateNodeProperty(nodeId, property, value) {
                const node = this.nodes.get(nodeId);
                if (node) {
                    node.properties[property] = value;
                    
                    // Mettre à jour l'affichage si nécessaire
                    if (property === 'slideId' && this.availableSlides) {
                        const slide = this.availableSlides.find(s => s.id == value);
                        if (slide) {
                            const contentDiv = node.element.querySelector('.node-content');
                            contentDiv.textContent = `Slide ${slide.slide_number}: ${slide.slide_title || 'Sans titre'}`;
                        }
                    }
                }
            }
            
            handleConnectionPoint(nodeId, pointType) {
                if (this.mode !== 'connect') {
                    return;
                }
                
                if (!this.connectionStart) {
                    // Commencer une connexion
                    if (pointType === 'output') {
                        this.connectionStart = { nodeId, pointType };
                        const node = this.nodes.get(nodeId);
                        const point = node.element.querySelector('.connection-point.output');
                        point.classList.add('connecting');
                    }
                } else {
                    // Terminer une connexion
                    if (pointType === 'input' && this.connectionStart.nodeId !== nodeId) {
                        this.createConnection(this.connectionStart.nodeId, nodeId);
                    }
                    
                    // Réinitialiser
                    const startNode = this.nodes.get(this.connectionStart.nodeId);
                    const startPoint = startNode.element.querySelector('.connection-point.output');
                    startPoint.classList.remove('connecting');
                    this.connectionStart = null;
                }
            }
            
            createConnection(fromNodeId, toNodeId) {
                // Vérifier si la connexion existe déjà
                const existingConnection = this.connections.find(c => 
                    c.from === fromNodeId && c.to === toNodeId
                );
                
                if (existingConnection) {
                    return;
                }
                
                const connection = {
                    id: `connection_${Date.now()}`,
                    from: fromNodeId,
                    to: toNodeId,
                    condition: 'always',
                    properties: {}
                };
                
                this.connections.push(connection);
                this.drawConnection(connection);
            }
            
            drawConnection(connection) {
                const fromNode = this.nodes.get(connection.from);
                const toNode = this.nodes.get(connection.to);
                
                if (!fromNode || !toNode) return;
                
                const fromRect = fromNode.element.getBoundingClientRect();
                const toRect = toNode.element.getBoundingClientRect();
                const canvasRect = this.canvas.getBoundingClientRect();
                
                const x1 = fromRect.right - canvasRect.left;
                const y1 = fromRect.top + fromRect.height / 2 - canvasRect.top;
                const x2 = toRect.left - canvasRect.left;
                const y2 = toRect.top + toRect.height / 2 - canvasRect.top;
                
                // Créer une courbe de Bézier
                const cx1 = x1 + (x2 - x1) / 3;
                const cy1 = y1;
                const cx2 = x2 - (x2 - x1) / 3;
                const cy2 = y2;
                
                const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                path.setAttribute('d', `M ${x1} ${y1} C ${cx1} ${
path.setAttribute('d', `M ${x1} ${y1} C ${cx1} ${cy1} ${cx2} ${cy2} ${x2} ${y2}`);
                path.setAttribute('class', 'connection-line');
                path.setAttribute('data-connection-id', connection.id);
                
                // Ajouter événement de clic pour configurer la connexion
                path.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.showConnectionModal(connection);
                });
                
                this.svg.appendChild(path);
            }
            
            showConnectionModal(connection) {
                const modal = document.getElementById('connectionModal');
                modal.style.display = 'flex';
                
                // Remplir les données actuelles
                document.getElementById('connectionCondition').value = connection.condition || 'always';
                
                // Stocker la connexion en cours d'édition
                modal.dataset.connectionId = connection.id;
            }
            
            redrawAllConnections() {
                // Supprimer toutes les connexions SVG
                this.svg.querySelectorAll('.connection-line').forEach(line => line.remove());
                
                // Redessiner toutes les connexions
                this.connections.forEach(connection => {
                    this.drawConnection(connection);
                });
            }
            
            startNodeDrag(nodeId, event) {
                const node = this.nodes.get(nodeId);
                if (!node) return;
                
                this.isDragging = true;
                this.dragStart = {
                    x: event.clientX - node.position.x,
                    y: event.clientY - node.position.y
                };
                
                const handleMouseMove = (e) => {
                    if (!this.isDragging) return;
                    
                    node.position.x = e.clientX - this.dragStart.x;
                    node.position.y = e.clientY - this.dragStart.y;
                    
                    node.element.style.left = node.position.x + 'px';
                    node.element.style.top = node.position.y + 'px';
                    
                    // Redessiner les connexions
                    this.redrawAllConnections();
                };
                
                const handleMouseUp = () => {
                    this.isDragging = false;
                    document.removeEventListener('mousemove', handleMouseMove);
                    document.removeEventListener('mouseup', handleMouseUp);
                };
                
                document.addEventListener('mousemove', handleMouseMove);
                document.addEventListener('mouseup', handleMouseUp);
            }
            
            deleteNode(nodeId) {
                const node = this.nodes.get(nodeId);
                if (!node) return;
                
                // Supprimer les connexions liées
                this.connections = this.connections.filter(connection => 
                    connection.from !== nodeId && connection.to !== nodeId
                );
                
                // Supprimer l'élément DOM
                node.element.remove();
                
                // Supprimer du Map
                this.nodes.delete(nodeId);
                
                // Redessiner les connexions
                this.redrawAllConnections();
                
                // Désélectionner si c'était le nœud sélectionné
                if (this.selectedNode === nodeId) {
                    this.selectedNode = null;
                    document.getElementById('propertiesContent').innerHTML = `
                        <div style="text-align: center; padding: 2rem; color: var(--gray-500);">
                            <i class="fas fa-mouse-pointer" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                            <p>Sélectionnez un nœud pour voir ses propriétés</p>
                        </div>
                    `;
                }
            }
            
            duplicateNode(nodeId) {
                const originalNode = this.nodes.get(nodeId);
                if (!originalNode) return;
                
                const newPosition = {
                    x: originalNode.position.x + 50,
                    y: originalNode.position.y + 50
                };
                
                const newNodeId = this.createNode(originalNode.type, newPosition);
                const newNode = this.nodes.get(newNodeId);
                
                // Copier les propriétés
                newNode.properties = { ...originalNode.properties };
            }
            
            clearCanvas() {
                if (confirm('Êtes-vous sûr de vouloir effacer tout le canvas ?')) {
                    // Supprimer tous les nœuds
                    this.nodes.forEach((node, nodeId) => {
                        node.element.remove();
                    });
                    this.nodes.clear();
                    
                    // Supprimer toutes les connexions
                    this.connections = [];
                    this.svg.querySelectorAll('.connection-line').forEach(line => line.remove());
                    
                    // Réinitialiser la sélection
                    this.selectedNode = null;
                    this.nodeCounter = 0;
                    
                    // Vider les propriétés
                    document.getElementById('propertiesContent').innerHTML = `
                        <div style="text-align: center; padding: 2rem; color: var(--gray-500);">
                            <i class="fas fa-mouse-pointer" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                            <p>Canvas vide - Ajoutez des nœuds depuis la sidebar</p>
                        </div>
                    `;
                }
            }
            
            zoomIn() {
                this.setZoom(this.zoom * 1.2);
            }
            
            zoomOut() {
                this.setZoom(this.zoom / 1.2);
            }
            
            resetZoom() {
                this.setZoom(1);
            }
            
            setZoom(newZoom) {
                newZoom = Math.max(0.1, Math.min(3, newZoom));
                this.zoom = newZoom;
                
                // Appliquer le zoom au canvas
                this.canvas.style.transform = `scale(${this.zoom})`;
                this.canvas.style.transformOrigin = 'top left';
            }
            
            autoLayout() {
                // Algorithme simple d'auto-layout
                const nodes = Array.from(this.nodes.values());
                if (nodes.length === 0) return;
                
                const spacing = { x: 200, y: 150 };
                const startPos = { x: 100, y: 100 };
                
                // Trouver les nœuds de démarrage (sans input)
                const startNodes = nodes.filter(node => !node.config.hasInput);
                const processedNodes = new Set();
                
                let currentY = startPos.y;
                
                const layoutBranch = (node, x, y) => {
                    if (processedNodes.has(node.id)) return;
                    
                    processedNodes.add(node.id);
                    node.position = { x, y };
                    node.element.style.left = x + 'px';
                    node.element.style.top = y + 'px';
                    
                    // Trouver les nœuds connectés
                    const connectedConnections = this.connections.filter(c => c.from === node.id);
                    let nextY = y;
                    
                    connectedConnections.forEach(connection => {
                        const nextNode = this.nodes.get(connection.to);
                        if (nextNode && !processedNodes.has(nextNode.id)) {
                            layoutBranch(nextNode, x + spacing.x, nextY);
                            nextY += spacing.y;
                        }
                    });
                };
                
                // Layout depuis chaque nœud de démarrage
                startNodes.forEach((startNode, index) => {
                    layoutBranch(startNode, startPos.x, currentY);
                    currentY += spacing.y * 3;
                });
                
                // Layout des nœuds non connectés
                nodes.forEach((node, index) => {
                    if (!processedNodes.has(node.id)) {
                        node.position = { 
                            x: startPos.x + (index % 4) * spacing.x, 
                            y: currentY + Math.floor(index / 4) * spacing.y 
                        };
                        node.element.style.left = node.position.x + 'px';
                        node.element.style.top = node.position.y + 'px';
                    }
                });
                
                // Redessiner les connexions
                this.redrawAllConnections();
            }
            
            handleCanvasClick(e) {
                if (e.target === this.canvas || e.target.classList.contains('canvas-grid')) {
                    this.deselectAll();
                }
            }
            
            handleCanvasMouseMove(e) {
                // Gestion du pan si en mode pan
                if (this.mode === 'pan' && this.isDragging) {
                    // Logique de pan ici si nécessaire
                }
            }
            
            handleCanvasMouseDown(e) {
                if (this.mode === 'pan') {
                    this.isDragging = true;
                    this.dragStart = { x: e.clientX, y: e.clientY };
                }
            }
            
            handleCanvasMouseUp(e) {
                this.isDragging = false;
            }
            
            deselectAll() {
                if (this.selectedNode) {
                    const node = this.nodes.get(this.selectedNode);
                    if (node) {
                        node.element.classList.remove('active');
                    }
                    this.selectedNode = null;
                    
                    document.getElementById('propertiesContent').innerHTML = `
                        <div style="text-align: center; padding: 2rem; color: var(--gray-500);">
                            <i class="fas fa-mouse-pointer" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                            <p>Sélectionnez un nœud pour voir ses propriétés</p>
                        </div>
                    `;
                }
            }
            
            exportLogicFlow() {
                const flowData = {
                    nodes: Array.from(this.nodes).map(([id, node]) => ({
                        id: id,
                        type: node.type,
                        position: node.position,
                        properties: node.properties
                    })),
                    connections: this.connections,
                    metadata: {
                        projectId: this.currentProject?.id,
                        exportDate: new Date().toISOString(),
                        version: '2.0'
                    }
                };
                
                return flowData;
            }
            
            importLogicFlow(flowData) {
                if (!flowData || !flowData.nodes) return false;
                
                try {
                    // Effacer le canvas actuel
                    this.clearCanvas();
                    
                    // Importer les nœuds
                    flowData.nodes.forEach(nodeData => {
                        const nodeId = this.createNode(nodeData.type, nodeData.position);
                        const node = this.nodes.get(nodeId);
                        if (node) {
                            node.properties = { ...nodeData.properties };
                        }
                    });
                    
                    // Importer les connexions
                    if (flowData.connections) {
                        this.connections = [...flowData.connections];
                        this.redrawAllConnections();
                    }
                    
                    return true;
                } catch (error) {
                    console.error('Erreur import logic flow:', error);
                    return false;
                }
            }
            
            loadExistingFlow() {
                if (!this.currentProject) return;
                
                fetch('logic_builder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },  
                    body: `action=load_logic_flow&project_id=${this.currentProject.id}`
                })
                .then(response => response.json())
                .then(result => {
                    if (result.success && result.logic_flow) {
                        this.importLogicFlow(result.logic_flow);
                    }
                })
                .catch(console.error);
            }
        }
        
        // ========== FONCTIONS GLOBALES ==========
        
        // Initialiser le Logic Builder
        const LogicBuilder = new LogicBuilderEngine();
        
        // Charger un projet
        function loadProject(projectId) {
            if (projectId) {
                window.location.href = `logic_builder.php?project_id=${projectId}`;
            }
        }
        
        // Sauvegarder le flow logique
        async function saveLogicFlow() {
            if (!LogicBuilder.currentProject) {
                alert('Aucun projet sélectionné');
                return;
            }
            
            const flowData = LogicBuilder.exportLogicFlow();
            
            try {
                const response = await fetch('logic_builder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=save_logic_flow&project_id=${LogicBuilder.currentProject.id}&logic_data=${encodeURIComponent(JSON.stringify(flowData))}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Logic flow sauvegardé avec succès !', 'success');
                } else {
                    showNotification('Erreur lors de la sauvegarde: ' + result.error, 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        // Prévisualiser la logique
        async function previewLogic() {
            if (!LogicBuilder.currentProject) {
                alert('Aucun projet sélectionné');
                return;
            }
            
            const flowData = LogicBuilder.exportLogicFlow();
            
            try {
                const response = await fetch('logic_builder.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=generate_logic_preview&project_id=${LogicBuilder.currentProject.id}&logic_data=${encodeURIComponent(JSON.stringify(flowData))}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Ouvrir aperçu dans nouvelle fenêtre
                    const previewWindow = window.open('', '_blank', 'width=1200,height=800');
                    previewWindow.document.write(`
                        <!DOCTYPE html>
                        <html>
                        <head>
                            <title>Aperçu Logic Flow</title>
                            <style>
                                body { font-family: Arial, sans-serif; padding: 2rem; }
                                .preview-container { max-width: 1000px; margin: 0 auto; }
                                .logic-summary { background: #f8f9fa; padding: 1rem; border-radius: 8px; margin-bottom: 2rem; }
                            </style>
                        </head>
                        <body>
                            <div class="preview-container">
                                <h1>🧠 Aperçu Logic Flow</h1>
                                <div class="logic-summary">
                                    <h3>Résumé</h3>
                                    <p><strong>Nœuds:</strong> ${flowData.nodes.length}</p>
                                    <p><strong>Connexions:</strong> ${flowData.connections.length}</p>
                                    <p><strong>Projet:</strong> ${LogicBuilder.currentProject.title || LogicBuilder.currentProject.project_name}</p>
                                </div>
                                ${result.preview_html}
                            </div>
                        </body>
                        </html>
                    `);
                } else {
                    showNotification('Erreur génération aperçu: ' + result.error, 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        // Exporter la logique
        function exportLogic() {
            const flowData = LogicBuilder.exportLogicFlow();
            
            const blob = new Blob([JSON.stringify(flowData, null, 2)], { 
                type: 'application/json' 
            });
            
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `logic_flow_${LogicBuilder.currentProject?.id || 'export'}.json`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            
            showNotification('Logic flow exporté !', 'success');
        }
        
        // Générer aperçu dans le panneau
        function generatePreview() {
            const previewDiv = document.getElementById('logicPreview');
            const flowData = LogicBuilder.exportLogicFlow();
            
            if (flowData.nodes.length === 0) {
                previewDiv.innerHTML = '<i class="fas fa-sitemap"></i><br>Canvas vide';
                return;
            }
            
            // Générer un mini-diagramme ASCII
            let preview = `<div style="font-family: monospace; font-size: 0.7rem; line-height: 1.2;">`;
            
            flowData.nodes.forEach((node, index) => {
                const nodeType = node.type;
                const symbol = getNodeSymbol(nodeType);
                
                preview += `${symbol} ${nodeType}<br>`;
                
                // Afficher les connexions
                const nodeConnections = flowData.connections.filter(c => c.from === node.id);
                nodeConnections.forEach(conn => {
                    const targetNode = flowData.nodes.find(n => n.id === conn.to);
                    if (targetNode) {
                        preview += `  └→ ${targetNode.type}<br>`;
                    }
                });
            });
            
            preview += `</div>`;
            previewDiv.innerHTML = preview;
        }
        
        // Obtenir symbole pour type de nœud
        function getNodeSymbol(nodeType) {
            const symbols = {
                start: '▶️',
                slide: '📄',
                condition: '❓',
                timer: '⏱️',
                variable: '📊',
                user_action: '👆',
                show_popup: '💬',
                send_data: '📤',
                redirect: '🔗',
                end: '⏹️'
            };
            
            return symbols[nodeType] || '⚪';
        }
        
        // Fermer modal de connexion
        function closeConnectionModal() {
            document.getElementById('connectionModal').style.display = 'none';
        }
        
        // Confirmer configuration de connexion
        function confirmConnection() {
            const modal = document.getElementById('connectionModal');
            const connectionId = modal.dataset.connectionId;
            const condition = document.getElementById('connectionCondition').value;
            const conditionValue = document.getElementById('conditionValue').value;
            
            const connection = LogicBuilder.connections.find(c => c.id === connectionId);
            if (connection) {
                connection.condition = condition;
                connection.properties = {
                    value: conditionValue
                };
                
                // Redessiner la connexion avec nouvelle couleur/style selon condition
                LogicBuilder.redrawAllConnections();
            }
            
            closeConnectionModal();
        }
        
        // Gestion des inputs conditionnels dans le modal
        document.getElementById('connectionCondition').addEventListener('change', function() {
            const conditionalInputs = document.getElementById('conditionalInputs');
            const needsValue = ['time', 'variable'].includes(this.value);
            conditionalInputs.style.display = needsValue ? 'flex' : 'none';
        });
        
        // Notification système
        function showNotification(message, type = 'info') {
            // Supprimer notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.style.cssText = `
                position: fixed;
                top: 2rem;
                right: 2rem;
                background: white;
                padding: 1rem 1.5rem;
                border-radius: 8px;
                box-shadow: 0 8px 25px rgba(0,0,0,0.15);
                z-index: 2000;
                border-left: 4px solid ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#667eea'};
                transform: translateX(400px);
                transition: transform 0.3s ease;
            `;
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.5rem;">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                    ${message}
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => {
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.style.transform = 'translateX(400px)';
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        // Message de bienvenue
        document.addEventListener('DOMContentLoaded', function() {
            <?php if ($selectedProject): ?>
                showNotification(`Logic Builder chargé pour: ${LogicBuilder.currentProject.title || LogicBuilder.currentProject.project_name}`, 'success');
            <?php else: ?>
                showNotification('Sélectionnez un projet pour commencer !', 'info');
            <?php endif; ?>
        });
    </script>
</body>
</html>