<?php
/**
 * Flying Book Upload Manager v2.0
 * Gestionnaire d'upload de fichiers avec optimisation et sécurité
 */

// Include des fonctions centralisées
require_once './env/.env.php';

class FlyingBookUploadManager
{
    private $pdo;
    private $uploadBasePath;
    private $allowedTypes;
    private $maxFileSize;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->uploadBasePath = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/projects/';
        $this->maxFileSize = 500 * 1024 * 1024; // 500MB par défaut
        
        $this->allowedTypes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp',
            'video/mp4', 'video/webm', 'video/mov', 'video/avi',
            'audio/mp3', 'audio/wav', 'audio/ogg', 'audio/m4a'
        ];
        
        // Créer le dossier d'upload si nécessaire
        if (!is_dir($this->uploadBasePath)) {
            mkdir($this->uploadBasePath, 0755, true);
        }
    }
    
    /**
     * Upload un fichier pour un projet
     */
    public function uploadFile($projectId, $userId, $file) {
        try {
            // Validation du projet
            $project = $this->getProject($projectId, $userId);
            if (!$project) {
                return ['success' => false, 'error' => 'Projet non trouvé'];
            }
            
            // Validation du fichier
            $validation = $this->validateFile($file, $userId);
            if (!$validation['valid']) {
                return ['success' => false, 'error' => $validation['error']];
            }
            
            // Créer le dossier du projet
            $projectFolder = $project['project_folder'];
            $projectPath = $this->uploadBasePath . $projectFolder . '/';
            
            if (!is_dir($projectPath)) {
                mkdir($projectPath, 0755, true);
            }
            
            // Générer un nom de fichier unique
            $originalName = $file['name'];
            $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
            $storedName = uniqid() . '_' . time() . '.' . $extension;
            $fullPath = $projectPath . $storedName;
            
            // Déplacer le fichier
            if (!move_uploaded_file($file['tmp_name'], $fullPath)) {
                return ['success' => false, 'error' => 'Impossible de déplacer le fichier'];
            }
            
            // Obtenir les informations du fichier
            $fileInfo = $this->getFileInfo($fullPath, $file['type']);
            
            // Optimiser le fichier si nécessaire
            $optimization = $this->optimizeFile($fullPath, $file['type']);
            
            // Générer une thumbnail pour les images/vidéos
            $thumbnailPath = $this->generateThumbnail($fullPath, $file['type'], $projectPath);
            
            // Enregistrer en base de données
            $mediaId = $this->saveToDatabase([
                'project_id' => $projectId,
                'user_id' => $userId,
                'original_filename' => $originalName,
                'stored_filename' => $storedName,
                'file_path' => 'uploads/projects/' . $projectFolder . '/' . $storedName,
                'file_type' => $this->getFileType($file['type']),
                'mime_type' => $file['type'],
                'file_size' => $file['size'],
                'dimensions' => $fileInfo['dimensions'],
                'duration' => $fileInfo['duration'],
                'thumbnail_path' => $thumbnailPath ? basename($thumbnailPath) : null,
                'compression_applied' => $optimization['applied'],
                'optimization_log' => $optimization['log']
            ]);
            
            if (!$mediaId) {
                // Supprimer le fichier si la BDD a échoué
                unlink($fullPath);
                if ($thumbnailPath) unlink($thumbnailPath);
                return ['success' => false, 'error' => 'Erreur lors de l\'enregistrement'];
            }
            
            // Mettre à jour la taille totale du projet
            $this->updateProjectSize($projectId);
            
            return [
                'success' => true,
                'media_id' => $mediaId,
                'file_path' => 'uploads/projects/' . $projectFolder . '/' . $storedName,
                'thumbnail_path' => $thumbnailPath ? 'uploads/projects/' . $projectFolder . '/' . basename($thumbnailPath) : null,
                'message' => 'Fichier uploadé avec succès'
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => 'Erreur: ' . $e->getMessage()];
        }
    }
    
    /**
     * Valider un fichier
     */
    private function validateFile($file, $userId) {
        // Vérifier les erreurs d'upload
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['valid' => false, 'error' => 'Erreur lors de l\'upload: ' . $file['error']];
        }
        
        // Vérifier la taille avec la fonction centralisée formatFileSize()
        if ($file['size'] > $this->maxFileSize) {
            return ['valid' => false, 'error' => 'Fichier trop volumineux (max ' . formatFileSize($this->maxFileSize) . ')'];
        }
        
        // Vérifier le type MIME
        if (!in_array($file['type'], $this->allowedTypes)) {
            return ['valid' => false, 'error' => 'Type de fichier non autorisé'];
        }
        
        // Vérifier l'extension
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'mp4', 'webm', 'mov', 'avi', 'mp3', 'wav', 'ogg', 'm4a'];
        
        if (!in_array($extension, $allowedExtensions)) {
            return ['valid' => false, 'error' => 'Extension de fichier non autorisée'];
        }
        
        // Vérifier les limites de l'utilisateur avec la fonction centralisée
        $userLimits = getUserPlanLimits($userId);
        if ($file['size'] > $userLimits['max_file_size']) {
            return ['valid' => false, 'error' => 'Fichier dépasse la limite de votre plan'];
        }
        
        return ['valid' => true];
    }
    
    /**
     * Obtenir les informations d'un fichier
     */
    private function getFileInfo($filePath, $mimeType) {
        $info = ['dimensions' => null, 'duration' => null];
        
        if (strpos($mimeType, 'image/') === 0) {
            $imageInfo = getimagesize($filePath);
            if ($imageInfo) {
                $info['dimensions'] = $imageInfo[0] . 'x' . $imageInfo[1];
            }
        } elseif (strpos($mimeType, 'video/') === 0) {
            // Utiliser ffmpeg si disponible
            if (function_exists('exec')) {
                $output = [];
                exec("ffprobe -v quiet -print_format json -show_format -show_streams " . escapeshellarg($filePath), $output);
                $json = implode('', $output);
                $data = json_decode($json, true);
                
                if ($data && isset($data['streams'])) {
                    foreach ($data['streams'] as $stream) {
                        if ($stream['codec_type'] === 'video') {
                            $info['dimensions'] = $stream['width'] . 'x' . $stream['height'];
                            if (isset($data['format']['duration'])) {
                                $info['duration'] = (float)$data['format']['duration'];
                            }
                            break;
                        }
                    }
                }
            }
        }
        
        return $info;
    }
    
    /**
     * Optimiser un fichier
     */
    private function optimizeFile($filePath, $mimeType) {
        $result = ['applied' => false, 'log' => ''];
        
        if (strpos($mimeType, 'image/') === 0) {
            $result = $this->optimizeImage($filePath);
        }
        
        return $result;
    }
    
    /**
     * Optimiser une image
     */
    private function optimizeImage($filePath) {
        $originalSize = filesize($filePath);
        $applied = false;
        $log = '';
        
        try {
            $imageInfo = getimagesize($filePath);
            if (!$imageInfo) return ['applied' => false, 'log' => 'Image invalide'];
            
            $mimeType = $imageInfo['mime'];
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            
            // Redimensionner si trop grande
            $maxDimension = 1920;
            if ($width > $maxDimension || $height > $maxDimension) {
                $ratio = min($maxDimension / $width, $maxDimension / $height);
                $newWidth = (int)($width * $ratio);
                $newHeight = (int)($height * $ratio);
                
                $image = null;
                switch ($mimeType) {
                    case 'image/jpeg':
                        $image = imagecreatefromjpeg($filePath);
                        break;
                    case 'image/png':
                        $image = imagecreatefrompng($filePath);
                        break;
                    case 'image/gif':
                        $image = imagecreatefromgif($filePath);
                        break;
                    case 'image/webp':
                        if (function_exists('imagecreatefromwebp')) {
                            $image = imagecreatefromwebp($filePath);
                        }
                        break;
                }
                
                if ($image) {
                    $resized = imagecreatetruecolor($newWidth, $newHeight);
                    
                    // Préserver la transparence pour PNG
                    if ($mimeType === 'image/png') {
                        imagealphablending($resized, false);
                        imagesavealpha($resized, true);
                    }
                    
                    imagecopyresampled($resized, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
                    
                    // Sauvegarder l'image redimensionnée
                    switch ($mimeType) {
                        case 'image/jpeg':
                            imagejpeg($resized, $filePath, 85);
                            break;
                        case 'image/png':
                            imagepng($resized, $filePath, 6);
                            break;
                        case 'image/gif':
                            imagegif($resized, $filePath);
                            break;
                        case 'image/webp':
                            if (function_exists('imagewebp')) {
                                imagewebp($resized, $filePath, 85);
                            }
                            break;
                    }
                    
                    imagedestroy($image);
                    imagedestroy($resized);
                    
                    $applied = true;
                    $log .= "Redimensionné de {$width}x{$height} à {$newWidth}x{$newHeight}. ";
                }
            }
            
            $newSize = filesize($filePath);
            $savings = $originalSize - $newSize;
            
            if ($savings > 0) {
                $log .= "Taille réduite de " . formatFileSize($savings) . " (" . round(($savings / $originalSize) * 100, 1) . "%)";
            }
            
        } catch (Exception $e) {
            $log = 'Erreur optimisation: ' . $e->getMessage();
        }
        
        return ['applied' => $applied, 'log' => $log];
    }
    
    /**
     * Générer une thumbnail
     */
    private function generateThumbnail($filePath, $mimeType, $projectPath) {
        if (strpos($mimeType, 'image/') !== 0) {
            return null; // Pas de thumbnail pour les non-images
        }
        
        try {
            $imageInfo = getimagesize($filePath);
            if (!$imageInfo) return null;
            
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            $type = $imageInfo['mime'];
            
            // Créer la thumbnail (150x150)
            $thumbSize = 150;
            $thumbPath = $projectPath . 'thumb_' . basename($filePath);
            
            $image = null;
            switch ($type) {
                case 'image/jpeg':
                    $image = imagecreatefromjpeg($filePath);
                    break;
                case 'image/png':
                    $image = imagecreatefrompng($filePath);
                    break;
                case 'image/gif':
                    $image = imagecreatefromgif($filePath);
                    break;
                case 'image/webp':
                    if (function_exists('imagecreatefromwebp')) {
                        $image = imagecreatefromwebp($filePath);
                    }
                    break;
            }
            
            if (!$image) return null;
            
            // Calculer les dimensions de crop (carré centré)
            $minDimension = min($width, $height);
            $cropX = ($width - $minDimension) / 2;
            $cropY = ($height - $minDimension) / 2;
            
            $thumbnail = imagecreatetruecolor($thumbSize, $thumbSize);
            
            // Préserver la transparence pour PNG
            if ($type === 'image/png') {
                imagealphablending($thumbnail, false);
                imagesavealpha($thumbnail, true);
            }
            
            imagecopyresampled(
                $thumbnail, $image,
                0, 0, $cropX, $cropY,
                $thumbSize, $thumbSize, $minDimension, $minDimension
            );
            
            // Sauvegarder en JPEG pour les thumbnails
            imagejpeg($thumbnail, $thumbPath, 80);
            
            imagedestroy($image);
            imagedestroy($thumbnail);
            
            return $thumbPath;
            
        } catch (Exception $e) {
            return null;
        }
    }
    
    /**
     * Enregistrer en base de données
     */
    private function saveToDatabase($data) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO flying_book_media 
                (project_id, user_id, original_filename, stored_filename, file_path, 
                 file_type, mime_type, file_size, dimensions, duration, thumbnail_path,
                 compression_applied, optimization_log, upload_status, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ready', NOW(), NOW())
            ");
            
            $stmt->execute([
                $data['project_id'], $data['user_id'], $data['original_filename'],
                $data['stored_filename'], $data['file_path'], $data['file_type'],
                $data['mime_type'], $data['file_size'], $data['dimensions'],
                $data['duration'], $data['thumbnail_path'],
                $data['compression_applied'] ? 1 : 0, $data['optimization_log']
            ]);
            
            return $this->pdo->lastInsertId();
            
        } catch (Exception $e) {
            error_log("Erreur sauvegarde média: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtenir les médias d'un projet
     */
    public function getProjectMedia($projectId, $userId) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT m.*, p.project_folder
                FROM flying_book_media m
                JOIN flying_book_projects p ON m.project_id = p.id
                WHERE m.project_id = ? AND p.user_id = ? AND m.upload_status = 'ready'
                ORDER BY m.created_at DESC
            ");
            $stmt->execute([$projectId, $userId]);
            
            return [
                'success' => true,
                'media' => $stmt->fetchAll(PDO::FETCH_ASSOC)
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Supprimer un média
     */
    public function deleteMedia($mediaId, $userId) {
        try {
            // Récupérer les infos du média
            $stmt = $this->pdo->prepare("
                SELECT m.*, p.project_folder
                FROM flying_book_media m
                JOIN flying_book_projects p ON m.project_id = p.id
                WHERE m.id = ? AND p.user_id = ?
            ");
            $stmt->execute([$mediaId, $userId]);
            $media = $stmt->fetch();
            
            if (!$media) {
                return ['success' => false, 'error' => 'Média non trouvé'];
            }
            
            // Supprimer les fichiers physiques
            $filePath = $this->uploadBasePath . $media['project_folder'] . '/' . $media['stored_filename'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            if ($media['thumbnail_path']) {
                $thumbPath = $this->uploadBasePath . $media['project_folder'] . '/' . $media['thumbnail_path'];
                if (file_exists($thumbPath)) {
                    unlink($thumbPath);
                }
            }
            
            // Supprimer de la base
            $stmt = $this->pdo->prepare("DELETE FROM flying_book_media WHERE id = ?");
            $stmt->execute([$mediaId]);
            
            // Mettre à jour la taille du projet
            $this->updateProjectSize($media['project_id']);
            
            return ['success' => true, 'message' => 'Média supprimé'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Utilitaires
     */
    private function getProject($projectId, $userId) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM flying_book_projects 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$projectId, $userId]);
        return $stmt->fetch();
    }
    
    private function getFileType($mimeType) {
        if (strpos($mimeType, 'image/') === 0) return 'image';
        if (strpos($mimeType, 'video/') === 0) return 'video';
        if (strpos($mimeType, 'audio/') === 0) return 'audio';
        return 'document';
    }
    
    private function updateProjectSize($projectId) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE flying_book_projects 
                SET file_size_total = (
                    SELECT COALESCE(SUM(file_size), 0) 
                    FROM flying_book_media 
                    WHERE project_id = ? AND upload_status = 'ready'
                )
                WHERE id = ?
            ");
            $stmt->execute([$projectId, $projectId]);
        } catch (Exception $e) {
            error_log("Erreur mise à jour taille projet: " . $e->getMessage());
        }
    }
}

// Gestion des actions AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if (!isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'error' => 'Non authentifié']);
        exit;
    }
    
    try {
        $pdo = getDatabaseConnection();
        $uploadManager = new FlyingBookUploadManager($pdo);
        
        switch ($_POST['action']) {
            case 'upload_media':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                
                if (!isset($_FILES['file'])) {
                    echo json_encode(['success' => false, 'error' => 'Aucun fichier']);
                    exit;
                }
                
                $result = $uploadManager->uploadFile($projectId, $userId, $_FILES['file']);
                echo json_encode($result);
                exit;
                
            case 'get_project_media':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                
                $result = $uploadManager->getProjectMedia($projectId, $userId);
                echo json_encode($result);
                exit;
                
            case 'delete_media':
                $mediaId = (int)$_POST['media_id'];
                $userId = $_SESSION['user_id'];
                
                $result = $uploadManager->deleteMedia($mediaId, $userId);
                echo json_encode($result);
                exit;
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}
?>