<?php
/**
 * Gestionnaire de Configuration Flying Book Creator v2.0
 * Sauvegarde et récupération des paramètres visuels et de navigation
 */

class FlyingBookConfigManager
{
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Sauvegarder la configuration d'un projet
     */
    public function saveProjectConfig($projectId, $userId, $config) {
        try {
            $this->pdo->beginTransaction();
            
            // Validation de la configuration
            $validatedConfig = $this->validateConfig($config);
            
            // Mettre à jour la table principale
            $stmt = $this->pdo->prepare("
                UPDATE flying_book_projects 
                SET style_theme = ?, 
                    transition_type = ?, 
                    navigation_type = ?,
                    title = ?,
                    subtitle = ?,
                    description = ?,
                    updated_at = NOW()
                WHERE id = ? AND user_id = ?
            ");
            
            $stmt->execute([
                $validatedConfig['style_theme'],
                $validatedConfig['transition_type'],
                $validatedConfig['navigation_type'],
                $validatedConfig['title'] ?? '',
                $validatedConfig['subtitle'] ?? null,
                $validatedConfig['description'] ?? null,
                $projectId,
                $userId
            ]);
            
            // Sauvegarder les paramètres avancés en JSON
            $advancedConfig = [
                'custom_css' => $validatedConfig['custom_css'] ?? '',
                'background_color' => $validatedConfig['background_color'] ?? '#ffffff',
                'text_color' => $validatedConfig['text_color'] ?? '#333333',
                'accent_color' => $validatedConfig['accent_color'] ?? '#667eea',
                'font_family' => $validatedConfig['font_family'] ?? 'Inter',
                'animation_speed' => $validatedConfig['animation_speed'] ?? 'medium',
                'auto_play' => $validatedConfig['auto_play'] ?? false,
                'loop_presentation' => $validatedConfig['loop_presentation'] ?? false,
                'show_controls' => $validatedConfig['show_controls'] ?? true,
                'show_progress' => $validatedConfig['show_progress'] ?? true,
                'keyboard_navigation' => $validatedConfig['keyboard_navigation'] ?? true,
                'swipe_navigation' => $validatedConfig['swipe_navigation'] ?? true,
                'fullscreen_mode' => $validatedConfig['fullscreen_mode'] ?? false
            ];
            
            // Vérifier si un enregistrement de config existe déjà
            $stmt = $this->pdo->prepare("
                SELECT id FROM flying_book_project_configs 
                WHERE project_id = ?
            ");
            $stmt->execute([$projectId]);
            $existingConfig = $stmt->fetch();
            
            if ($existingConfig) {
                // Mettre à jour
                $stmt = $this->pdo->prepare("
                    UPDATE flying_book_project_configs 
                    SET advanced_config = ?, updated_at = NOW()
                    WHERE project_id = ?
                ");
                $stmt->execute([json_encode($advancedConfig), $projectId]);
            } else {
                // Créer nouveau
                $stmt = $this->pdo->prepare("
                    INSERT INTO flying_book_project_configs 
                    (project_id, advanced_config, created_at, updated_at)
                    VALUES (?, ?, NOW(), NOW())
                ");
                $stmt->execute([$projectId, json_encode($advancedConfig)]);
            }
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'message' => 'Configuration sauvegardée avec succès'
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollback();
            return [
                'success' => false,
                'error' => 'Erreur lors de la sauvegarde: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Récupérer la configuration d'un projet
     */
    public function getProjectConfig($projectId, $userId = null) {
        try {
            $sql = "
                SELECT p.*, pc.advanced_config 
                FROM flying_book_projects p
                LEFT JOIN flying_book_project_configs pc ON p.id = pc.project_id
                WHERE p.id = ?
            ";
            
            $params = [$projectId];
            
            if ($userId) {
                $sql .= " AND p.user_id = ?";
                $params[] = $userId;
            }
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $project = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$project) {
                return [
                    'success' => false,
                    'error' => 'Projet non trouvé'
                ];
            }
            
            // Fusionner la config de base avec la config avancée
            $baseConfig = [
                'project_id' => $project['id'],
                'project_name' => $project['project_name'],
                'title' => $project['title'],
                'subtitle' => $project['subtitle'],
                'description' => $project['description'],
                'style_theme' => $project['style_theme'],
                'transition_type' => $project['transition_type'],
                'navigation_type' => $project['navigation_type'],
                'is_published' => $project['is_published'],
                'is_public' => $project['is_public']
            ];
            
            $advancedConfig = [];
            if ($project['advanced_config']) {
                $advancedConfig = json_decode($project['advanced_config'], true) ?: [];
            }
            
            return [
                'success' => true,
                'config' => array_merge($baseConfig, $advancedConfig)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Erreur lors de la récupération: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Valider et nettoyer la configuration
     */
    private function validateConfig($config) {
        $allowedThemes = ['moderne', 'classique', 'corporate', 'minimal', 'premium'];
        $allowedTransitions = ['fade', 'slide', 'flip', '3d', 'premium_3d'];
        $allowedNavigation = ['buttons', 'thumbnails', 'keyboard', 'all'];
        $allowedFonts = ['Inter', 'Roboto', 'Open Sans', 'Lato', 'Montserrat'];
        $allowedSpeeds = ['slow', 'medium', 'fast'];
        
        return [
            'title' => $this->sanitizeConfigString($config['title'] ?? ''),
            'subtitle' => $this->sanitizeConfigString($config['subtitle'] ?? ''),
            'description' => $this->sanitizeConfigString($config['description'] ?? ''),
            'style_theme' => in_array($config['style_theme'] ?? '', $allowedThemes) ? $config['style_theme'] : 'moderne',
            'transition_type' => in_array($config['transition_type'] ?? '', $allowedTransitions) ? $config['transition_type'] : '3d',
            'navigation_type' => in_array($config['navigation_type'] ?? '', $allowedNavigation) ? $config['navigation_type'] : 'all',
            'custom_css' => $this->sanitizeConfigCSS($config['custom_css'] ?? ''),
            'background_color' => $this->sanitizeConfigColor($config['background_color'] ?? '#ffffff'),
            'text_color' => $this->sanitizeConfigColor($config['text_color'] ?? '#333333'),
            'accent_color' => $this->sanitizeConfigColor($config['accent_color'] ?? '#667eea'),
            'font_family' => in_array($config['font_family'] ?? '', $allowedFonts) ? $config['font_family'] : 'Inter',
            'animation_speed' => in_array($config['animation_speed'] ?? '', $allowedSpeeds) ? $config['animation_speed'] : 'medium',
            'auto_play' => (bool)($config['auto_play'] ?? false),
            'loop_presentation' => (bool)($config['loop_presentation'] ?? false),
            'show_controls' => (bool)($config['show_controls'] ?? true),
            'show_progress' => (bool)($config['show_progress'] ?? true),
            'keyboard_navigation' => (bool)($config['keyboard_navigation'] ?? true),
            'swipe_navigation' => (bool)($config['swipe_navigation'] ?? true),
            'fullscreen_mode' => (bool)($config['fullscreen_mode'] ?? false)
        ];
    }
    
    /**
     * Nettoyer une chaîne de caractères pour la config
     */
    private function sanitizeConfigString($input) {
        return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
    
    /**
     * Valider et nettoyer une couleur hexadécimale pour la config
     */
    private function sanitizeConfigColor($color) {
        if (preg_match('/^#[a-fA-F0-9]{6}$/', $color)) {
            return $color;
        }
        return '#ffffff'; // couleur par défaut
    }
    
    /**
     * Nettoyer le CSS personnalisé pour la config
     */
    private function sanitizeConfigCSS($css) {
        // Supprimer les éléments potentiellement dangereux
        $dangerous = ['javascript:', 'expression(', 'behavior:', 'binding:', '-moz-binding'];
        $css = str_ireplace($dangerous, '', $css);
        
        // Limiter la taille
        return substr(trim($css), 0, 10000);
    }
    
    /**
     * Obtenir les thèmes disponibles avec leurs paramètres
     */
    public function getAvailableThemes() {
        return [
            'moderne' => [
                'name' => 'Moderne',
                'description' => 'Design contemporain avec dégradés',
                'preview' => '/assets/previews/theme-moderne.jpg',
                'primary_color' => '#667eea',
                'secondary_color' => '#764ba2',
                'premium' => false
            ],
            'classique' => [
                'name' => 'Classique',
                'description' => 'Style traditionnel et élégant',
                'preview' => '/assets/previews/theme-classique.jpg',
                'primary_color' => '#2c3e50',
                'secondary_color' => '#34495e',
                'premium' => false
            ],
            'corporate' => [
                'name' => 'Corporate',
                'description' => 'Professionnel pour l\'entreprise',
                'preview' => '/assets/previews/theme-corporate.jpg',
                'primary_color' => '#1a365d',
                'secondary_color' => '#2d3748',
                'premium' => false
            ],
            'minimal' => [
                'name' => 'Minimal',
                'description' => 'Épuré et moderne',
                'preview' => '/assets/previews/theme-minimal.jpg',
                'primary_color' => '#4a5568',
                'secondary_color' => '#718096',
                'premium' => false
            ],
            'premium' => [
                'name' => 'Premium',
                'description' => 'Animations avancées et effets 3D',
                'preview' => '/assets/previews/theme-premium.jpg',
                'primary_color' => '#805ad5',
                'secondary_color' => '#9f7aea',
                'premium' => true
            ]
        ];
    }
    
    /**
     * Obtenir les transitions disponibles
     */
    public function getAvailableTransitions() {
        return [
            'fade' => [
                'name' => 'Fondu',
                'description' => 'Transition en douceur',
                'premium' => false
            ],
            'slide' => [
                'name' => 'Glissement',
                'description' => 'Glissement horizontal',
                'premium' => false
            ],
            'flip' => [
                'name' => 'Retournement',
                'description' => 'Effet de carte qui se retourne',
                'premium' => false
            ],
            '3d' => [
                'name' => '3D Standard',
                'description' => 'Effets 3D basiques',
                'premium' => false
            ],
            'premium_3d' => [
                'name' => '3D Premium',
                'description' => 'Effets 3D avancés avec parallaxe',
                'premium' => true
            ]
        ];
    }
    
    /**
     * Dupliquer la configuration d'un projet
     */
    public function duplicateConfig($sourceProjectId, $targetProjectId, $userId) {
        try {
            $configResult = $this->getProjectConfig($sourceProjectId, $userId);
            
            if (!$configResult['success']) {
                return $configResult;
            }
            
            $config = $configResult['config'];
            unset($config['project_id']); // Supprimer l'ID du projet source
            
            return $this->saveProjectConfig($targetProjectId, $userId, $config);
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Erreur lors de la duplication: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Exporter la configuration en JSON
     */
    public function exportConfig($projectId, $userId) {
        $configResult = $this->getProjectConfig($projectId, $userId);
        
        if (!$configResult['success']) {
            return $configResult;
        }
        
        $config = $configResult['config'];
        $exportData = [
            'flying_book_version' => '2.0',
            'export_date' => date('Y-m-d H:i:s'),
            'config' => $config
        ];
        
        return [
            'success' => true,
            'json' => json_encode($exportData, JSON_PRETTY_PRINT),
            'filename' => 'flying_book_config_' . date('Y-m-d_H-i-s') . '.json'
        ];
    }
    
    /**
     * Importer une configuration depuis JSON
     */
    public function importConfig($projectId, $userId, $jsonData) {
        try {
            $data = json_decode($jsonData, true);
            
            if (!$data || !isset($data['config'])) {
                return [
                    'success' => false,
                    'error' => 'Format JSON invalide'
                ];
            }
            
            $config = $data['config'];
            $config['project_id'] = $projectId; // S'assurer que l'ID est correct
            
            return $this->saveProjectConfig($projectId, $userId, $config);
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Erreur lors de l\'import: ' . $e->getMessage()
            ];
        }
    }
}

// Créer la table de configuration si elle n'existe pas
function createConfigTable($pdo) {
    $sql = "
        CREATE TABLE IF NOT EXISTS flying_book_project_configs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            project_id INT NOT NULL,
            advanced_config JSON NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (project_id) REFERENCES flying_book_projects(id) ON DELETE CASCADE,
            UNIQUE KEY unique_project_config (project_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    
    try {
        $pdo->exec($sql);
        return true;
    } catch (PDOException $e) {
        error_log("Erreur création table config: " . $e->getMessage());
        return false;
    }
}

// Utilisation dans flying_book_slides.php
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        $configManager = new FlyingBookConfigManager($pdo);
        
        switch ($_POST['action']) {
            case 'save_config':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                $config = json_decode($_POST['config'], true);
                
                if (!$config) {
                    echo json_encode(['success' => false, 'error' => 'Configuration invalide']);
                    exit;
                }
                
                $result = $configManager->saveProjectConfig($projectId, $userId, $config);
                echo json_encode($result);
                exit;
                
            case 'get_config':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                
                $result = $configManager->getProjectConfig($projectId, $userId);
                echo json_encode($result);
                exit;
                
            case 'get_themes':
                $themes = $configManager->getAvailableThemes();
                echo json_encode(['success' => true, 'themes' => $themes]);
                exit;
                
            case 'get_transitions':
                $transitions = $configManager->getAvailableTransitions();
                echo json_encode(['success' => true, 'transitions' => $transitions]);
                exit;
                
            case 'export_config':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                
                $result = $configManager->exportConfig($projectId, $userId);
                echo json_encode($result);
                exit;
                
            case 'import_config':
                $projectId = (int)$_POST['project_id'];
                $userId = $_SESSION['user_id'];
                $jsonData = $_POST['json_data'];
                
                $result = $configManager->importConfig($projectId, $userId, $jsonData);
                echo json_encode($result);
                exit;
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// Créer la table au premier lancement
if (isset($pdo)) {
    createConfigTable($pdo);
}
?>