<?php
session_start();

// ⚡ CORRECTION : Include .env.php AVANT tout le reste
require_once './env/.env.php';

// ⚡ CORRECTION : Initialiser PDO AVANT les autres includes
try {
    $pdo = getDatabaseConnection();
} catch (Exception $e) {
    die("Erreur de connexion à la base de données : " . $e->getMessage());
}

require_once 'includes/config_manager.php';
require_once 'includes/upload_manager.php';
require_once 'includes/subscription_manager.php';

// ⚡ CORRECTION : Créer les instances APRÈS avoir $pdo
$uploadManager = new FlyingBookUploadManager($pdo);
$configManager = new FlyingBookConfigManager($pdo);
$subscriptionManager = new FlyingBookSubscriptionManager($pdo);

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

// Vérifier les limites utilisateur
$userPlan = $subscriptionManager->getUserPlan($user_id);
$userLimits = $userPlan['limits'];

$message = '';
$error = '';
$project_id = null;
$project = null;
$slides = [];
$current_step = 1;

try {
    // ========== GESTION DES ACTIONS ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'create_project':
                $project_name = trim($_POST['project_name']);
                $title = trim($_POST['title'] ?? $project_name);
                $subtitle = trim($_POST['subtitle'] ?? '');
                $description = trim($_POST['description'] ?? '');
                
                if (empty($project_name)) {
                    echo json_encode(['success' => false, 'message' => 'Nom du projet requis']);
                    exit;
                }
                
                // Vérifier les limites avant création
                if (!checkUserLimits($user_id, 'max_projects', 1)) {
                    echo json_encode(['success' => false, 'message' => 'Limite de projets atteinte pour votre plan']);
                    exit;
                }
                
                // Générer un slug unique
                $base_slug = generateSlug($project_name);
                $project_slug = $base_slug;
                $counter = 1;
                
                // Vérifier unicité du slug
                while (true) {
                    $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE project_slug = ? AND user_id = ?");
                    $stmt->execute([$project_slug, $user_id]);
                    if (!$stmt->fetch()) break;
                    
                    $project_slug = $base_slug . '-' . $counter;
                    $counter++;
                }
                
                $project_folder = 'project_' . uniqid();
                
                $stmt = $pdo->prepare("
                    INSERT INTO flying_book_projects 
                    (user_id, project_name, project_slug, title, subtitle, description, 
                     style_theme, transition_type, navigation_type, project_folder, 
                     plan_type, status, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, 'moderne', '3d', 'all', ?, ?, 'draft', NOW(), NOW())
                ");
                
                if ($stmt->execute([$user_id, $project_name, $project_slug, $title, $subtitle, $description, $project_folder, $userPlan['plan_type']])) {
                    $new_project_id = $pdo->lastInsertId();
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Projet créé avec succès',
                        'project_id' => $new_project_id,
                        'redirect' => "flying_book_slides.php?id={$new_project_id}&step=2"
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la création']);
                }
                exit;
                
            case 'update_project':
                $project_id = (int)$_POST['project_id'];
                $title = trim($_POST['title']);
                $subtitle = trim($_POST['subtitle'] ?? '');
                $description = trim($_POST['description'] ?? '');
                $style_theme = $_POST['style_theme'];
                $transition_type = $_POST['transition_type'];
                $navigation_type = $_POST['navigation_type'];
                
                $stmt = $pdo->prepare("
                    UPDATE flying_book_projects 
                    SET title = ?, subtitle = ?, description = ?, style_theme = ?, 
                        transition_type = ?, navigation_type = ?, updated_at = NOW()
                    WHERE id = ? AND user_id = ?
                ");
                
                if ($stmt->execute([$title, $subtitle, $description, $style_theme, 
                                   $transition_type, $navigation_type, $project_id, $user_id])) {
                    echo json_encode(['success' => true, 'message' => 'Configuration mise à jour']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la mise à jour']);
                }
                exit;
                
            case 'save_slide':
                $project_id = (int)$_POST['project_id'];
                $slide_number = (int)$_POST['slide_number'];
                $slide_title = trim($_POST['slide_title']);
                $media_type = $_POST['media_type'];
                $text_overlay = $_POST['text_overlay'] ?? '';
                
                // Vérifier les limites de slides
                $currentSlideCount = 0;
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM flying_book_slides WHERE project_id = ?");
                $stmt->execute([$project_id]);
                $currentSlideCount = $stmt->fetchColumn();
                
                // Vérifier si la slide existe
                $stmt = $pdo->prepare("SELECT id FROM flying_book_slides WHERE project_id = ? AND slide_number = ?");
                $stmt->execute([$project_id, $slide_number]);
                $existing = $stmt->fetch();
                
                if (!$existing && $currentSlideCount >= $userLimits['max_slides_per_project'] && $userLimits['max_slides_per_project'] !== -1) {
                    echo json_encode(['success' => false, 'message' => 'Limite de slides atteinte pour votre plan']);
                    exit;
                }
                
                if ($existing) {
                    // Mettre à jour
                    $stmt = $pdo->prepare("
                        UPDATE flying_book_slides 
                        SET slide_title = ?, media_type = ?, text_overlay = ?, updated_at = NOW()
                        WHERE project_id = ? AND slide_number = ?
                    ");
                    $stmt->execute([$slide_title, $media_type, $text_overlay, $project_id, $slide_number]);
                } else {
                    // Créer
                    $stmt = $pdo->prepare("
                        INSERT INTO flying_book_slides 
                        (project_id, slide_number, slide_title, media_type, media_file, text_overlay, display_order, is_active, created_at, updated_at)
                        VALUES (?, ?, ?, ?, 'placeholder.jpg', ?, ?, 1, NOW(), NOW())
                    ");
                    $stmt->execute([$project_id, $slide_number, $slide_title, $media_type, $text_overlay, $slide_number - 1]);
                }
                
                echo json_encode(['success' => true, 'message' => 'Slide sauvegardée']);
                exit;
                
            case 'delete_slide':
                $project_id = (int)$_POST['project_id'];
                $slide_number = (int)$_POST['slide_number'];
                
                $stmt = $pdo->prepare("DELETE FROM flying_book_slides WHERE project_id = ? AND slide_number = ?");
                $stmt->execute([$project_id, $slide_number]);
                
                // Réorganiser les numéros de slides
                $stmt = $pdo->prepare("
                    UPDATE flying_book_slides 
                    SET slide_number = slide_number - 1, display_order = display_order - 1
                    WHERE project_id = ? AND slide_number > ?
                ");
                $stmt->execute([$project_id, $slide_number]);
                
                echo json_encode(['success' => true, 'message' => 'Slide supprimée']);
                exit;
                
            case 'upload_media':
                $project_id = (int)$_POST['project_id'];
                
                if (!isset($_FILES['file'])) {
                    echo json_encode(['success' => false, 'error' => 'Aucun fichier']);
                    exit;
                }
                
                $result = $uploadManager->uploadFile($project_id, $user_id, $_FILES['file']);
                echo json_encode($result);
                exit;
                
            case 'get_project_media':
                $project_id = (int)$_POST['project_id'];
                $result = $uploadManager->getProjectMedia($project_id, $user_id);
                echo json_encode($result);
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
    // ========== CHARGEMENT DES DONNÉES ==========
    
    // Récupérer ou créer le projet
    if (isset($_GET['id'])) {
        $project_id = (int)$_GET['id'];
        
        // ⚡ CORRECTION : Utiliser soft delete
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_projects 
            WHERE id = ? AND user_id = ? AND (deleted_at IS NULL OR is_deleted = FALSE)
        ");
        $stmt->execute([$project_id, $user_id]);
        $project = $stmt->fetch();
        
        if (!$project) {
            header('Location: projects.php');
            exit;
        }
        
        // Récupérer les slides
        $stmt = $pdo->prepare("
            SELECT * FROM flying_book_slides 
            WHERE project_id = ? 
            ORDER BY display_order, slide_number
        ");
        $stmt->execute([$project_id]);
        $slides = $stmt->fetchAll();
        
    } elseif (isset($_GET['action']) && $_GET['action'] === 'create') {
        // Mode création rapide depuis dashboard
        $project_name = $_GET['name'] ?? 'Nouveau Flying Book ' . date('d/m/Y');
        $current_step = 1;
    }
    
    // Déterminer l'étape actuelle
    $current_step = (int)($_GET['step'] ?? 1);
    if ($project && empty($slides)) {
        $current_step = max($current_step, 3); // Aller directement aux slides si pas de slides
    }
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement: " . $e->getMessage();
    error_log("Flying Book Slides error: " . $e->getMessage());
}

function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - <?= $project ? htmlspecialchars($project['title']) : 'Nouveau Projet' ?></title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Styles optimisés pour Flying Book Creator */
        .wizard-container {
            max-width: 1000px;
            margin: 2rem auto;
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
        }
        
        .wizard-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 0;
        }
        
        .wizard-steps {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 1.5rem 2rem;
        }
        
        .step {
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
            color: rgba(255, 255, 255, 0.7);
            min-width: 120px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .step.active {
            color: white;
        }
        
        .step.completed {
            color: #4ade80;
        }
        
        .step-number {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.5rem;
            font-weight: 700;
            font-size: 1.1rem;
        }
        
        .step.active .step-number {
            background: white;
            color: #667eea;
        }
        
        .step.completed .step-number {
            background: #4ade80;
            color: white;
        }
        
        .step-label {
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .wizard-content {
            padding: 3rem;
            min-height: 500px;
        }
        
        .step-content {
            display: none;
        }
        
        .step-content.active {
            display: block;
            animation: fadeIn 0.3s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .slides-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-top: 2rem;
        }
        
        .slide-card {
            background: white;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .slide-card:hover {
            border-color: #667eea;
            transform: translateY(-2px);
        }
        
        .slide-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }
        
        .slide-number {
            width: 30px;
            height: 30px;
            background: #667eea;
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
        }
        
        .wizard-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 2rem 3rem;
            background: #f8fafc;
            border-top: 1px solid #e2e8f0;
        }
        
        .add-slide-btn {
            border: 2px dashed #d1d5db;
            background: #f9fafb;
            color: #6b7280;
            min-height: 200px;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            border-radius: 12px;
        }
        
        .add-slide-btn:hover {
            border-color: #667eea;
            color: #667eea;
            background: rgba(102, 126, 234, 0.05);
        }
        
        .config-grid {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 2rem;
            margin: 2rem 0;
        }
        
        .upload-zone {
            border: 2px dashed #667eea;
            border-radius: 12px;
            padding: 2rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            margin: 1rem 0;
        }
        
        .upload-zone:hover {
            background: rgba(102, 126, 234, 0.05);
        }
        
        .plan-info {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
        }
        
        .plan-info h4 {
            margin: 0 0 0.5rem 0;
            color: white;
        }
        
        @media (max-width: 768px) {
            .wizard-steps {
                flex-wrap: wrap;
                gap: 1rem;
            }
            
            .step {
                min-width: 80px;
            }
            
            .wizard-content {
                padding: 2rem 1.5rem;
            }
            
            .config-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Header simplifié -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
            </div>
            <div class="header-right">
                <a href="projects.php" class="btn btn-outline">
                    <i class="fas fa-arrow-left"></i> Mes Projets
                </a>
            </div>
        </div>
    </header>

    <main class="main-content">
        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <div class="wizard-container">
            <!-- Header avec étapes -->
            <div class="wizard-header">
                <div class="wizard-steps">
                    <div class="step <?= $current_step >= 1 ? ($current_step == 1 ? 'active' : 'completed') : '' ?>" onclick="goToStep(1)">
                        <div class="step-number">1</div>
                        <div class="step-label">Informations</div>
                    </div>
                    
                    <div class="step <?= $current_step >= 2 ? ($current_step == 2 ? 'active' : 'completed') : '' ?>" onclick="goToStep(2)">
                        <div class="step-number">2</div>
                        <div class="step-label">Configuration</div>
                    </div>
                    
                    <div class="step <?= $current_step >= 3 ? ($current_step == 3 ? 'active' : 'completed') : '' ?>" onclick="goToStep(3)">
                        <div class="step-number">3</div>
                        <div class="step-label">Slides</div>
                    </div>
                    
                    <div class="step <?= $current_step >= 4 ? ($current_step == 4 ? 'active' : 'completed') : '' ?>" onclick="goToStep(4)">
                        <div class="step-number">4</div>
                        <div class="step-label">Aperçu</div>
                    </div>
                    
                    <div class="step <?= $current_step >= 5 ? ($current_step == 5 ? 'active' : 'completed') : '' ?>" onclick="goToStep(5)">
                        <div class="step-number">5</div>
                        <div class="step-label">Publication</div>
                    </div>
                </div>
            </div>

            <!-- Contenu des étapes -->
            <div class="wizard-content">
                <!-- Information sur le plan utilisateur -->
                <div class="plan-info">
                    <h4><i class="fas fa-crown"></i> Plan <?= htmlspecialchars($userPlan['plan_name']) ?></h4>
                    <div style="display: flex; gap: 2rem; font-size: 0.9rem;">
                        <span>Projets : <?= $userLimits['max_projects'] === -1 ? 'Illimité' : $userLimits['max_projects'] ?></span>
                        <span>Slides/projet : <?= $userLimits['max_slides_per_project'] === -1 ? 'Illimité' : $userLimits['max_slides_per_project'] ?></span>
                        <span>Stockage : <?= $userLimits['max_storage'] === -1 ? 'Illimité' : formatFileSize($userLimits['max_storage']) ?></span>
                    </div>
                </div>

                <!-- Étape 1: Informations -->
                <div class="step-content <?= $current_step == 1 ? 'active' : '' ?>" id="step1">
                    <div class="page-header" style="text-align: center; margin-bottom: 2rem;">
                        <h1><i class="fas fa-info-circle"></i> Informations du projet</h1>
                    </div>
                    
                    <form id="projectInfoForm">
                        <?php if ($project): ?>
                            <input type="hidden" name="project_id" value="<?= $project['id'] ?>">
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label class="form-label">Nom du projet *</label>
                            <input type="text" name="project_name" class="form-control" 
                                   value="<?= htmlspecialchars($project['project_name'] ?? $_GET['name'] ?? '') ?>" 
                                   placeholder="Mon Flying Book Amazing" required>
                            <small style="color: #6b7280;">Ce nom sera utilisé pour l'URL du Flying Book</small>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Titre principal *</label>
                            <input type="text" name="title" class="form-control" 
                                   value="<?= htmlspecialchars($project['title'] ?? '') ?>" 
                                   placeholder="Titre qui apparaîtra sur votre Flying Book">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Sous-titre</label>
                            <input type="text" name="subtitle" class="form-control" 
                                   value="<?= htmlspecialchars($project['subtitle'] ?? '') ?>" 
                                   placeholder="Sous-titre optionnel">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Description</label>
                            <textarea name="description" class="form-control" rows="4" 
                                      placeholder="Décrivez brièvement votre Flying Book..."><?= htmlspecialchars($project['description'] ?? '') ?></textarea>
                        </div>
                    </form>
                </div>

                <!-- Étape 2: Configuration -->
                <div class="step-content <?= $current_step == 2 ? 'active' : '' ?>" id="step2">
                    <div class="page-header" style="text-align: center; margin-bottom: 2rem;">
                        <h1><i class="fas fa-cog"></i> Configuration du Flying Book</h1>
                    </div>
                    
                    <form id="projectConfigForm">
                        <div class="config-grid">
                            <div class="form-group">
                                <label class="form-label">Style visuel</label>
                                <select name="style_theme" class="form-control">
                                    <option value="moderne" <?= ($project['style_theme'] ?? '') === 'moderne' ? 'selected' : '' ?>>Moderne</option>
                                    <option value="classique" <?= ($project['style_theme'] ?? '') === 'classique' ? 'selected' : '' ?>>Classique</option>
                                    <option value="corporate" <?= ($project['style_theme'] ?? '') === 'corporate' ? 'selected' : '' ?>>Corporate</option>
                                    <option value="minimal" <?= ($project['style_theme'] ?? '') === 'minimal' ? 'selected' : '' ?>>Minimal</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Type de transition</label>
                                <select name="transition_type" class="form-control">
                                    <option value="3d" <?= ($project['transition_type'] ?? '') === '3d' ? 'selected' : '' ?>>3D (Recommandé)</option>
                                    <option value="fade" <?= ($project['transition_type'] ?? '') === 'fade' ? 'selected' : '' ?>>Fondu</option>
                                    <option value="slide" <?= ($project['transition_type'] ?? '') === 'slide' ? 'selected' : '' ?>>Glissement</option>
                                    <option value="flip" <?= ($project['transition_type'] ?? '') === 'flip' ? 'selected' : '' ?>>Retournement</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Navigation</label>
                                <select name="navigation_type" class="form-control">
                                    <option value="all" <?= ($project['navigation_type'] ?? '') === 'all' ? 'selected' : '' ?>>Complète</option>
                                    <option value="buttons" <?= ($project['navigation_type'] ?? '') === 'buttons' ? 'selected' : '' ?>>Boutons uniquement</option>
                                    <option value="keyboard" <?= ($project['navigation_type'] ?? '') === 'keyboard' ? 'selected' : '' ?>>Clavier uniquement</option>
                                    <option value="thumbnails" <?= ($project['navigation_type'] ?? '') === 'thumbnails' ? 'selected' : '' ?>>Miniatures uniquement</option>
                                </select>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Étape 3: Création des slides -->
                <div class="step-content <?= $current_step == 3 ? 'active' : '' ?>" id="step3">
                    <div class="page-header" style="margin-bottom: 2rem;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <h1><i class="fas fa-images"></i> Création des slides</h1>
                            <div>
                                <span style="margin-right: 1rem;">Slides: <strong id="slidesCount"><?= count($slides) ?></strong></span>
                                <button type="button" class="btn btn-primary" onclick="addSlide()">
                                    <i class="fas fa-plus"></i> Ajouter une slide
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Zone d'upload -->
                    <?php if ($project_id): ?>
                    <div class="upload-zone" id="uploadZone" onclick="document.getElementById('fileInput').click()">
                        <div>
                            <i class="fas fa-cloud-upload-alt" style="font-size: 3rem; color: #667eea; margin-bottom: 1rem;"></i>
                            <div style="font-size: 1.2rem; font-weight: 600; margin-bottom: 0.5rem;">
                                Glissez vos fichiers ici ou cliquez pour parcourir
                            </div>
                            <div style="color: #64748b; font-size: 0.9rem;">
                                Images, vidéos, audio - Max <?= formatFileSize($userLimits['max_file_size'] ?? 50 * 1024 * 1024) ?> par fichier
                            </div>
                        </div>
                    </div>
                    <input type="file" id="fileInput" multiple accept="image/*,video/*,audio/*" style="display: none;">
                    
                    <!-- Galerie des médias -->
                    <div id="mediaGallery" style="margin-bottom: 2rem;">
                        <!-- Sera peuplé par JavaScript -->
                    </div>
                    <?php endif; ?>
                    
                    <div class="slides-grid" id="slidesContainer">
                        <?php if (empty($slides)): ?>
                            <!-- Slide par défaut -->
                            <div class="slide-card" data-slide="1">
                                <div class="slide-header">
                                    <div class="slide-number">1</div>
                                    <button type="button" class="btn btn-sm btn-secondary" onclick="removeSlide(1)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label">Type de média</label>
                                    <select name="media_type" class="form-control">
                                        <option value="image">Image</option>
                                        <option value="video">Vidéo</option>
                                        <option value="text_image">Texte + Image</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label">Titre (optionnel)</label>
                                    <input type="text" name="slide_title" class="form-control" 
                                           placeholder="Titre de la slide 1">
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label">Contenu/Texte</label>
                                    <textarea name="text_overlay" class="form-control" rows="3" 
                                              placeholder="Texte à afficher sur cette slide..."></textarea>
                                </div>
                            </div>
                        <?php else: ?>
                            <?php foreach ($slides as $slide): ?>
                                <div class="slide-card" data-slide="<?= $slide['slide_number'] ?>">
                                    <div class="slide-header">
                                        <div class="slide-number"><?= $slide['slide_number'] ?></div>
                                        <button type="button" class="btn btn-sm btn-secondary" onclick="removeSlide(<?= $slide['slide_number'] ?>)">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label">Type de média</label>
                                        <select name="media_type" class="form-control">
                                            <option value="image" <?= $slide['media_type'] === 'image' ? 'selected' : '' ?>>Image</option>
                                            <option value="video" <?= $slide['media_type'] === 'video' ? 'selected' : '' ?>>Vidéo</option>
                                            <option value="text_image" <?= $slide['media_type'] === 'text_image' ? 'selected' : '' ?>>Texte + Image</option>
                                        </select>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label">Titre (optionnel)</label>
                                        <input type="text" name="slide_title" class="form-control" 
                                               value="<?= htmlspecialchars($slide['slide_title']) ?>"
                                               placeholder="Titre de la slide <?= $slide['slide_number'] ?>">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="form-label">Contenu/Texte</label>
                                        <textarea name="text_overlay" class="form-control" rows="3" 
                                                  placeholder="Texte à afficher..."><?= htmlspecialchars($slide['text_overlay']) ?></textarea>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        
                        <!-- Bouton d'ajout -->
                        <div class="add-slide-btn" onclick="addSlide()">
                            <i class="fas fa-plus" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                            <h3>Ajouter une slide</h3>
                        </div>
                    </div>
                </div>

                <!-- Étape 4: Aperçu -->
                <div class="step-content <?= $current_step == 4 ? 'active' : '' ?>" id="step4">
                    <div class="page-header" style="text-align: center; margin-bottom: 2rem;">
                        <h1><i class="fas fa-eye"></i> Aperçu du Flying Book</h1>
                    </div>
                    
                    <div style="text-align: center; color: #6b7280; padding: 3rem;">
                        <i class="fas fa-rocket" style="font-size: 4rem; margin-bottom: 2rem; color: #667eea;"></i>
                        <h2>Aperçu en cours de développement</h2>
                        <p>L'aperçu temps réel sera disponible dans la prochaine version</p>
                        <p>Vous pouvez continuer vers la publication pour finaliser votre Flying Book</p>
                    </div>
                </div>

                <!-- Étape 5: Publication -->
                <div class="step-content <?= $current_step == 5 ? 'active' : '' ?>" id="step5">
                    <div class="page-header" style="text-align: center; margin-bottom: 2rem;">
                        <h1><i class="fas fa-cloud-upload-alt"></i> Publication</h1>
                    </div>
                    
                    <div style="text-align: center; color: #6b7280; padding: 3rem;">
                        <i class="fas fa-tools" style="font-size: 4rem; margin-bottom: 2rem; color: #f59e0b;"></i>
                        <h2>Publication en cours de développement</h2>
                        <p>La fonctionnalité de publication sera disponible dans la prochaine version</p>
                        <p><strong>Votre Flying Book a été sauvegardé</strong> et vous pouvez continuer à l'éditer</p>
                    </div>
                </div>
            </div>

            <!-- Actions du wizard -->
            <div class="wizard-actions">
                <button type="button" class="btn btn-secondary" onclick="previousStep()" id="prevBtn" 
                        style="<?= $current_step <= 1 ? 'visibility: hidden;' : '' ?>">
                    <i class="fas fa-arrow-left"></i> Précédent
                </button>
                
                <div>
                    <button type="button" class="btn btn-outline" onclick="saveAndExit()">
                        <i class="fas fa-save"></i> Sauvegarder et Quitter
                    </button>
                    
                    <button type="button" class="btn btn-primary" onclick="nextStep()" id="nextBtn">
                        <span id="nextBtnText">Suivant</span>
                        <i class="fas fa-arrow-right"></i>
                    </button>
                </div>
            </div>
        </div>
    </main>

    <!-- Scripts -->
    <script>
        let currentStep = <?= $current_step ?>;
        let projectId = <?= $project_id ?? 'null' ?>;
        let slideCount = <?= count($slides) ?: 1 ?>;
        
        // Gestion du wizard
        function goToStep(step) {
            if (step <= currentStep || step <= getCurrentMaxStep()) {
                currentStep = step;
                updateWizard();
                updateURL();
            }
        }
        
        function getCurrentMaxStep() {
            if (!projectId) return 1;
            return 5;
        }
        
        function nextStep() {
            const maxStep = 5;
            
            if (currentStep < maxStep) {
                if (currentStep === 1) {
                    saveProjectInfo().then(success => {
                        if (success) {
                            currentStep++;
                            updateWizard();
                            updateURL();
                        }
                    });
                } else if (currentStep === 2) {
                    saveProjectConfig().then(success => {
                        if (success) {
                            currentStep++;
                            updateWizard();
                            updateURL();
                        }
                    });
                } else if (currentStep === 3) {
                    saveAllSlides().then(success => {
                        if (success) {
                            currentStep++;
                            updateWizard();
                            updateURL();
                        }
                    });
                } else {
                    currentStep++;
                    updateWizard();
                    updateURL();
                }
            } else {
                showNotification('Publication en cours de développement', 'info');
            }
        }
        
        function previousStep() {
            if (currentStep > 1) {
                currentStep--;
                updateWizard();
                updateURL();
            }
        }
        
        function updateWizard() {
            // Mettre à jour les étapes
            document.querySelectorAll('.step').forEach((step, index) => {
                const stepNum = index + 1;
                step.className = 'step';
                
                if (stepNum < currentStep) {
                    step.classList.add('completed');
                } else if (stepNum === currentStep) {
                    step.classList.add('active');
                }
            });
            
            // Mettre à jour le contenu
            document.querySelectorAll('.step-content').forEach(content => {
                content.classList.remove('active');
            });
            document.getElementById(`step${currentStep}`).classList.add('active');
            
            // Mettre à jour les boutons
            document.getElementById('prevBtn').style.visibility = currentStep <= 1 ? 'hidden' : 'visible';
            
            const nextBtn = document.getElementById('nextBtnText');
            if (currentStep === 5) {
                nextBtn.textContent = 'Publier';
            } else {
                nextBtn.textContent = 'Suivant';
            }
        }
        
        function updateURL() {
            const url = new URL(window.location);
            url.searchParams.set('step', currentStep);
            if (projectId) {
                url.searchParams.set('id', projectId);
            }
            window.history.replaceState({}, '', url);
        }
        
        // Sauvegarde des données
        async function saveProjectInfo() {
            const form = document.getElementById('projectInfoForm');
            const formData = new FormData(form);
            
            if (projectId) {
                formData.append('action', 'update_project');
                formData.append('project_id', projectId);
            } else {
                formData.append('action', 'create_project');
            }
            
            try {
                const response = await fetch('flying_book_slides.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    if (result.project_id && !projectId) {
                        projectId = result.project_id;
                        // Initialiser l'upload maintenant qu'on a un project_id
                        initUploadSystem();
                    }
                    showNotification(result.message, 'success');
                    return true;
                } else {
                    showNotification(result.message, 'error');
                    return false;
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
                return false;
            }
        }
        
        async function saveProjectConfig() {
            if (!projectId) {
                showNotification('Veuillez d\'abord sauvegarder les informations du projet', 'warning');
                return false;
            }
            
            const form = document.getElementById('projectConfigForm');
            const formData = new FormData(form);
            formData.append('action', 'update_project');
            formData.append('project_id', projectId);
            
            // Récupérer les infos de l'étape 1
            const infoForm = document.getElementById('projectInfoForm');
            const infoData = new FormData(infoForm);
            for (let [key, value] of infoData.entries()) {
                if (key !== 'action' && key !== 'project_id') {
                    formData.append(key, value);
                }
            }
            
            try {
                const response = await fetch('flying_book_slides.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    return true;
                } else {
                    showNotification(result.message, 'error');
                    return false;
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
                return false;
            }
        }
        
        async function saveAllSlides() {
            if (!projectId) {
                showNotification('Projet non trouvé', 'error');
                return false;
            }
            
            const slides = document.querySelectorAll('.slide-card');
            let saved = 0;
            
            for (let slide of slides) {
                const slideNumber = slide.getAttribute('data-slide');
                if (!slideNumber) continue;
                
                const formData = new FormData();
                formData.append('action', 'save_slide');
                formData.append('project_id', projectId);
                formData.append('slide_number', slideNumber);
                formData.append('slide_title', slide.querySelector('input[name="slide_title"]').value);
                formData.append('media_type', slide.querySelector('select[name="media_type"]').value);
                formData.append('text_overlay', slide.querySelector('textarea[name="text_overlay"]').value);
                
                try {
                    const response = await fetch('flying_book_slides.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const result = await response.json();
                    if (result.success) {
                        saved++;
                    }
                } catch (error) {
                    console.error('Erreur sauvegarde slide:', error);
                }
            }
            
            if (saved > 0) {
                showNotification(`${saved} slide(s) sauvegardée(s)`, 'success');
                return true;
            } else {
                showNotification('Aucune slide sauvegardée', 'warning');
                return false;
            }
        }
        
        // Gestion des slides
        function addSlide() {
            slideCount++;
            updateSlidesCount();
            
            const container = document.getElementById('slidesContainer');
            const addBtn = container.querySelector('.add-slide-btn');
            
            const slideHTML = `
                <div class="slide-card" data-slide="${slideCount}">
                    <div class="slide-header">
                        <div class="slide-number">${slideCount}</div>
                        <button type="button" class="btn btn-sm btn-secondary" onclick="removeSlide(${slideCount})">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Type de média</label>
                        <select name="media_type" class="form-control">
                            <option value="image">Image</option>
                            <option value="video">Vidéo</option>
                            <option value="text_image">Texte + Image</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Titre (optionnel)</label>
                        <input type="text" name="slide_title" class="form-control" 
                               placeholder="Titre de la slide ${slideCount}">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Contenu/Texte</label>
                        <textarea name="text_overlay" class="form-control" rows="3" 
                                  placeholder="Texte à afficher sur cette slide..."></textarea>
                    </div>
                </div>
            `;
            
            addBtn.insertAdjacentHTML('beforebegin', slideHTML);
            showNotification(`Slide ${slideCount} ajoutée`, 'success');
        }
        
        function removeSlide(slideNumber) {
            if (slideCount <= 1) {
                showNotification('Vous devez avoir au moins une slide', 'warning');
                return;
            }
            
            if (confirm(`Supprimer la slide ${slideNumber} ?`)) {
                const slideCard = document.querySelector(`[data-slide="${slideNumber}"]`);
                slideCard.remove();
                
                slideCount--;
                updateSlidesCount();
                renumberSlides();
                showNotification(`Slide ${slideNumber} supprimée`, 'success');
            }
        }
        
        function renumberSlides() {
            const slides = document.querySelectorAll('.slide-card');
            slides.forEach((slide, index) => {
                const newNumber = index + 1;
                slide.setAttribute('data-slide', newNumber);
                slide.querySelector('.slide-number').textContent = newNumber;
                slide.querySelector('input[name="slide_title"]').placeholder = `Titre de la slide ${newNumber}`;
                slide.querySelector('button[onclick]').setAttribute('onclick', `removeSlide(${newNumber})`);
            });
        }
        
        function updateSlidesCount() {
            document.getElementById('slidesCount').textContent = slideCount;
        }
        
        function saveAndExit() {
            if (currentStep === 1) {
                saveProjectInfo().then(() => {
                    window.location.href = 'projects.php';
                });
            } else {
                window.location.href = 'projects.php';
            }
        }
        
        // Upload des médias
        function initUploadSystem() {
            if (!projectId) return;
            
            const uploadZone = document.getElementById('uploadZone');
            const fileInput = document.getElementById('fileInput');
            
            if (!uploadZone || !fileInput) return;
            
            // Drag & Drop
            uploadZone.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadZone.style.borderColor = '#764ba2';
                uploadZone.style.background = 'rgba(102, 126, 234, 0.1)';
            });
            
            uploadZone.addEventListener('dragleave', (e) => {
                e.preventDefault();
                uploadZone.style.borderColor = '#667eea';
                uploadZone.style.background = 'transparent';
            });
            
            uploadZone.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadZone.style.borderColor = '#667eea';
                uploadZone.style.background = 'transparent';
                handleFiles(e.dataTransfer.files);
            });
            
            fileInput.addEventListener('change', (e) => {
                handleFiles(e.target.files);
            });
            
            // Charger les médias existants
            loadProjectMedia();
        }
        
        function handleFiles(files) {
            for (let file of files) {
                uploadFile(file);
            }
        }
        
        async function uploadFile(file) {
            const formData = new FormData();
            formData.append('action', 'upload_media');
            formData.append('project_id', projectId);
            formData.append('file', file);
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Fichier uploadé avec succès', 'success');
                    loadProjectMedia();
                } else {
                    showNotification(result.error, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de l\'upload', 'error');
            }
        }
        
        async function loadProjectMedia() {
            if (!projectId) return;
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=get_project_media&project_id=${projectId}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    displayMediaGallery(result.media);
                }
            } catch (error) {
                console.error('Erreur chargement média:', error);
            }
        }
        
        function displayMediaGallery(mediaList) {
            const gallery = document.getElementById('mediaGallery');
            
            if (!mediaList || mediaList.length === 0) {
                gallery.innerHTML = '<p style="text-align: center; color: #64748b; padding: 1rem;">Aucun média uploadé pour ce projet</p>';
                return;
            }
            
            gallery.innerHTML = `
                <h4 style="margin-bottom: 1rem;"><i class="fas fa-images"></i> Médiathèque du projet (${mediaList.length} fichier(s))</h4>
                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 1rem;">
                    ${mediaList.map(media => `
                        <div style="background: white; border-radius: 8px; padding: 1rem; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                            <div style="height: 80px; background: #f8fafc; border-radius: 6px; display: flex; align-items: center; justify-content: center; margin-bottom: 0.5rem;">
                                ${media.file_type === 'image' ? 
                                    `<img src="${media.file_path}" style="max-width: 100%; max-height: 100%; object-fit: cover; border-radius: 4px;">` :
                                    `<i class="fas fa-${media.file_type === 'video' ? 'video' : media.file_type === 'audio' ? 'music' : 'file'}" style="font-size: 2rem; color: #667eea;"></i>`
                                }
                            </div>
                            <div style="font-size: 0.8rem; font-weight: 600; margin-bottom: 0.25rem; word-break: break-word;">${media.original_filename}</div>
                            <div style="font-size: 0.7rem; color: #64748b; margin-bottom: 0.5rem;">${formatFileSize(media.file_size)}</div>
                            <button onclick="useMedia('${media.file_path}', '${media.file_type}')" 
                                    style="background: #667eea; color: white; border: none; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.7rem; cursor: pointer; width: 100%;">
                                Utiliser
                            </button>
                        </div>
                    `).join('')}
                </div>
            `;
        }
        
        function useMedia(filePath, fileType) {
            showNotification('Fonctionnalité d\'utilisation des médias en cours de développement', 'info');
        }
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 B';
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Notifications
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `alert alert-${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : type === 'warning' ? 'exclamation-triangle' : 'info-circle'}"></i>
                ${message}
            `;
            notification.style.cssText = `
                position: fixed;
                top: 2rem;
                right: 2rem;
                z-index: 3000;
                max-width: 400px;
                animation: slideIn 0.3s ease;
                box-shadow: 0 8px 25px rgba(0,0,0,0.15);
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease';
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            updateWizard();
            
            if (projectId) {
                initUploadSystem();
            }
            
            showNotification('Flying Book Creator chargé ! 🎪', 'success');
        });
    </script>
</body>
</html>