<?php
session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

try {
    $pdo = getDatabaseConnection(); // UTILISE la fonction de .env.php
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'reset_all_projects':
                // Fonction de nettoyage/réinitialisation
                if ($_SESSION['user_role'] !== 'admin') {
                    echo json_encode(['success' => false, 'message' => 'Accès refusé']);
                    exit;
                }
                
                $user_filter = $_POST['user_id'] ?? 'all';
                
                try {
                    $pdo->beginTransaction();
                    
                    if ($user_filter === 'all') {
                        // Nettoyer tous les projets
                        $pdo->exec("DELETE FROM flying_book_slides WHERE project_id IN (SELECT id FROM flying_book_projects WHERE status = 'draft')");
                        $pdo->exec("DELETE FROM flying_book_analytics WHERE project_id IN (SELECT id FROM flying_book_projects WHERE status = 'draft')");
                        $pdo->exec("DELETE FROM flying_book_projects WHERE status = 'draft'");
                        $message = "Tous les projets brouillons ont été supprimés";
                    } else {
                        // Nettoyer pour un utilisateur spécifique
                        $stmt = $pdo->prepare("DELETE FROM flying_book_slides WHERE project_id IN (SELECT id FROM flying_book_projects WHERE user_id = ? AND status = 'draft')");
                        $stmt->execute([$user_filter]);
                        
                        $stmt = $pdo->prepare("DELETE FROM flying_book_analytics WHERE project_id IN (SELECT id FROM flying_book_projects WHERE user_id = ? AND status = 'draft')");
                        $stmt->execute([$user_filter]);
                        
                        $stmt = $pdo->prepare("DELETE FROM flying_book_projects WHERE user_id = ? AND status = 'draft'");
                        $stmt->execute([$user_filter]);
                        
                        $message = "Projets brouillons de l'utilisateur supprimés";
                    }
                    
                    $pdo->commit();
                    echo json_encode(['success' => true, 'message' => $message]);
                } catch (Exception $e) {
                    $pdo->rollback();
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'check_system_status':
                // Diagnostic du système
                $status = [
                    'php_version' => PHP_VERSION,
                    'upload_max_filesize' => ini_get('upload_max_filesize'),
                    'post_max_size' => ini_get('post_max_size'),
                    'max_execution_time' => ini_get('max_execution_time'),
                    'memory_limit' => ini_get('memory_limit'),
                    'uploads_dir_exists' => is_dir($_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/'),
                    'uploads_dir_writable' => is_writable($_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/'),
                    'temp_dir' => sys_get_temp_dir(),
                    'temp_dir_writable' => is_writable(sys_get_temp_dir())
                ];
                
                echo json_encode(['success' => true, 'status' => $status]);
                exit;
        }
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Récupérer les projets de l'utilisateur avec informations détaillées
    $stmt = $pdo->prepare("
        SELECT p.*, 
               COUNT(DISTINCT s.id) as total_slides_actual,
               COUNT(DISTINCT a.id) as total_views,
               MAX(a.last_visit) as last_viewed,
               COALESCE(SUM(m.file_size), 0) as storage_used
        FROM flying_book_projects p
        LEFT JOIN flying_book_slides s ON p.id = s.project_id AND s.is_active = 1
        LEFT JOIN flying_book_analytics a ON p.id = a.project_id
        LEFT JOIN flying_book_media m ON p.id = m.project_id AND m.upload_status = 'ready'
        WHERE p.user_id = ? AND p.status != 'deleted'
        GROUP BY p.id
        ORDER BY p.updated_at DESC
        LIMIT 6
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $recent_projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Statistiques globales améliorées
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_projects,
            SUM(CASE WHEN status = 'published' THEN 1 ELSE 0 END) as published_projects,
            SUM(CASE WHEN status = 'draft' THEN 1 ELSE 0 END) as draft_projects,
            SUM(CASE WHEN status = 'archived' THEN 1 ELSE 0 END) as archived_projects,
            COALESCE(SUM(file_size_total), 0) as total_storage,
            AVG(CASE WHEN status = 'published' THEN 1 ELSE 0 END) * 100 as publish_rate
        FROM flying_book_projects 
        WHERE user_id = ? AND status != 'deleted'
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Templates populaires
    $stmt = $pdo->prepare("
        SELECT t.*, u.name as creator_name
        FROM flying_book_templates t
        LEFT JOIN users u ON t.user_id = u.id
        WHERE t.is_public = 1 OR t.user_id = ?
        ORDER BY t.is_featured DESC, t.usage_count DESC, t.rating DESC
        LIMIT 4
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $popular_templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Activité récente
    $stmt = $pdo->prepare("
        SELECT 
            'project' as type,
            p.title as title,
            p.updated_at as date,
            p.status,
            p.id
        FROM flying_book_projects p
        WHERE p.user_id = ? AND p.status != 'deleted'
        UNION ALL
        SELECT 
            'template' as type,
            t.template_name as title,
            t.updated_at as date,
            'active' as status,
            t.id
        FROM flying_book_templates t
        WHERE t.user_id = ?
        ORDER BY date DESC
        LIMIT 5
    ");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $recent_activity = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error = "Erreur de base de données : " . $e->getMessage();
}

// ========== FONCTIONS SPÉCIFIQUES AU DASHBOARD ==========
function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'À l\'instant';
    if ($time < 3600) return floor($time/60) . ' min';
    if ($time < 86400) return floor($time/3600) . ' h';
    if ($time < 2592000) return floor($time/86400) . ' j';
    return date('d/m/Y', strtotime($datetime));
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Dashboard v2.0</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* ========== STYLES AMÉLIORÉS ========== */
        .dashboard-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 2rem;
            margin-bottom: 2rem;
        }
        
        @media (max-width: 1024px) {
            .dashboard-grid {
                grid-template-columns: 1fr;
            }
        }
        
        .quick-actions-enhanced {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 16px;
            padding: 2rem;
            color: white;
        }
        
        .quick-actions-enhanced h2 {
            margin-bottom: 1.5rem;
            color: white;
        }
        
        .actions-grid-enhanced {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }
        
        .action-card-enhanced {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border-radius: 12px;
            padding: 1.5rem;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .action-card-enhanced:hover {
            background: rgba(255, 255, 255, 0.2);
            transform: translateY(-2px);
        }
        
        .action-card-enhanced i {
            font-size: 2rem;
            margin-bottom: 0.5rem;
            display: block;
        }
        
        .recent-section {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);
        }
        
        .project-mini-card {
            display: flex;
            align-items: center;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 0.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .project-mini-card:hover {
            background: #f8fafc;
            transform: translateX(5px);
        }
        
        .project-mini-thumbnail {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #e2e8f0, #cbd5e1);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
            color: #64748b;
        }
        
        .project-mini-info h4 {
            margin: 0 0 0.25rem 0;
            font-size: 0.9rem;
            color: #1e293b;
        }
        
        .project-mini-info p {
            margin: 0;
            font-size: 0.8rem;
            color: #64748b;
        }
        
        .admin-controls {
            background: linear-gradient(135deg, #f59e0b, #d97706);
            border-radius: 16px;
            padding: 2rem;
            color: white;
            margin-bottom: 2rem;
        }
        
        .admin-controls h3 {
            margin-bottom: 1rem;
            color: white;
        }
        
        .admin-button {
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-right: 1rem;
            margin-bottom: 0.5rem;
            display: inline-block;
        }
        
        .admin-button:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateY(-1px);
        }
        
        .admin-button.danger {
            background: rgba(239, 68, 68, 0.3);
            border-color: rgba(239, 68, 68, 0.5);
        }
        
        .admin-button.danger:hover {
            background: rgba(239, 68, 68, 0.5);
        }
        
        .system-status {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            margin-top: 1rem;
            border: 1px solid #e2e8f0;
        }
        
        .status-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.5rem 0;
            border-bottom: 1px solid #f1f5f9;
        }
        
        .status-item:last-child {
            border-bottom: none;
        }
        
        .status-good {
            color: #059669;
        }
        
        .status-warning {
            color: #d97706;
        }
        
        .status-error {
            color: #dc2626;
        }
        
        .template-mini-card {
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 0.5rem;
            transition: all 0.3s ease;
        }
        
        .template-mini-card:hover {
            border-color: #667eea;
            transform: translateY(-1px);
        }
        
        .notification {
            position: fixed;
            top: 2rem;
            right: 2rem;
            background: white;
            padding: 1rem 1.5rem;
            border-radius: 8px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            z-index: 2000;
            transform: translateX(400px);
            transition: transform 0.3s ease;
        }
        
        .notification.show {
            transform: translateX(0);
        }
        
        .notification.success {
            border-left: 4px solid #10b981;
        }
        
        .notification.error {
            border-left: 4px solid #ef4444;
        }
        
        .notification.warning {
            border-left: 4px solid #f59e0b;
        }
    </style>
</head>
<body>
    <!-- Header Navigation (existant) -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                     <i class="fas fa-layer-group"></i> Templates
                    </a>
		    <a href="template-editor.php" class="btn btn-primary">
  	            <i class="fas fa-plus"></i> Créer un template
		    </a>
                    <a href="media.php" class="nav-link">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>

                    <a href="analytics.html" class="nav-link">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($_SESSION['user_name'], 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($_SESSION['user_name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- Page Header -->
            <div class="page-header">
                <div class="header-content">
                    <h1>Tableau de bord</h1>
                    <p>Bienvenue dans Flying Book Creator v2.0 - Connecté intelligemment !</p>
                </div>
                <div class="header-actions">
                    <button class="btn btn-outline" onclick="checkSystemHealth()">
                        <i class="fas fa-stethoscope"></i>
                        Diagnostic Système
                    </button>
                </div>
            </div>

            <!-- Contrôles Admin -->
            <?php if ($_SESSION['user_role'] === 'admin'): ?>
            <div class="admin-controls">
                <h3><i class="fas fa-tools"></i> Contrôles Administrateur</h3>
                <button class="admin-button" onclick="checkSystemHealth()">
                    <i class="fas fa-heartbeat"></i> État du Système
                </button>
                <button class="admin-button danger" onclick="confirmResetProjects()">
                    <i class="fas fa-broom"></i> Nettoyer Projets Brouillons
                </button>
                <button class="admin-button" onclick="window.open('phpinfo.php', '_blank')">
                    <i class="fas fa-info-circle"></i> Info PHP
                </button>
                
                <div class="system-status" id="systemStatus" style="display: none;">
                    <h4>État du Système</h4>
                    <div id="statusContent">Vérification en cours...</div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-book"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $stats['total_projects'] ?? 0 ?></h3>
                        <p>Projets créés</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-eye"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $stats['published_projects'] ?? 0 ?></h3>
                        <p>Projets publiés</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-edit"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $stats['draft_projects'] ?? 0 ?></h3>
                        <p>Brouillons</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-hdd"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= formatFileSize($stats['total_storage'] ?? 0) ?></h3>
                        <p>Stockage utilisé</p>
                    </div>
                </div>
            </div>

            <!-- Dashboard Grid Principal -->
            <div class="dashboard-grid">
                <!-- Actions Rapides Améliorées -->
                <div class="quick-actions-enhanced">
                    <h2><i class="fas fa-rocket"></i> Actions Rapides</h2>
                    <div class="actions-grid-enhanced">
                        <div class="action-card-enhanced" onclick="createNewProject()">
                            <i class="fas fa-plus-circle"></i>
                            <h3>Nouveau Projet</h3>
                            <p>Créer depuis zéro</p>
                        </div>
                        
                        <div class="action-card-enhanced" onclick="window.location.href='templates.php'">
                            <i class="fas fa-layer-group"></i>
                            <h3>Templates</h3>
                            <p>Utiliser un modèle</p>
                        </div>
                        
                        <div class="action-card-enhanced" onclick="window.location.href='media.php'">
                            <i class="fas fa-cloud-upload-alt"></i>
                            <h3>Médiathèque</h3>
                            <p>Gérer vos médias</p>
                        </div>
                        
                        <div class="action-card-enhanced" onclick="window.location.href='projects.php'">
                            <i class="fas fa-folder-open"></i>
                            <h3>Mes Projets</h3>
                            <p>Voir tous</p>
                        </div>
                    </div>
                </div>

                <!-- Projets Récents -->
                <div class="recent-section">
                    <h2><i class="fas fa-history"></i> Projets Récents</h2>
                    
                    <?php if (empty($recent_projects)): ?>
                        <div style="text-align: center; padding: 2rem; color: #64748b;">
                            <i class="fas fa-folder-open" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                            <p>Aucun projet encore</p>
                            <button class="btn btn-primary" onclick="createNewProject()">
                                Créer votre premier projet
                            </button>
                        </div>
                    <?php else: ?>
                        <?php foreach ($recent_projects as $project): ?>
                            <div class="project-mini-card" onclick="editProject(<?= $project['id'] ?>)">
                                <div class="project-mini-thumbnail">
                                    <i class="fas fa-presentation"></i>
                                </div>
                                <div class="project-mini-info">
                                    <h4><?= htmlspecialchars($project['title'] ?: $project['project_name']) ?></h4>
                                    <p>
                                        <?= $project['total_slides_actual'] ?> slides • 
                                        <?= ucfirst($project['status']) ?> • 
                                        <?= timeAgo($project['updated_at']) ?>
                                    </p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        
                        <div style="text-align: center; margin-top: 1rem;">
                            <a href="projects.php" class="btn btn-outline">
                                Voir tous les projets
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Templates Populaires -->
            <?php if (!empty($popular_templates)): ?>
            <div class="recent-section">
                <h2><i class="fas fa-fire"></i> Templates Populaires</h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1rem;">
                    <?php foreach ($popular_templates as $template): ?>
                        <div class="template-mini-card" onclick="useTemplate(<?= $template['id'] ?>)">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.5rem;">
                                <h4 style="margin: 0;"><?= htmlspecialchars($template['template_name']) ?></h4>
                                <?php if ($template['is_featured']): ?>
                                    <span style="background: #f59e0b; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.7rem;">
                                        <i class="fas fa-star"></i> Featured
                                    </span>
                                <?php endif; ?>
                            </div>
                            <p style="color: #64748b; font-size: 0.8rem; margin: 0;">
                                <?= htmlspecialchars($template['category']) ?> • 
                                <?= $template['usage_count'] ?> utilisations
                            </p>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <!-- Activité Récente -->
            <?php if (!empty($recent_activity)): ?>
            <div class="recent-section">
                <h2><i class="fas fa-bell"></i> Activité Récente</h2>
                <?php foreach ($recent_activity as $activity): ?>
                    <div class="project-mini-card">
                        <div class="project-mini-thumbnail">
                            <i class="fas fa-<?= $activity['type'] === 'project' ? 'presentation' : 'layer-group' ?>"></i>
                        </div>
                        <div class="project-mini-info">
                            <h4><?= htmlspecialchars($activity['title']) ?></h4>
                            <p>
                                <?= ucfirst($activity['type']) ?> • 
                                <?= ucfirst($activity['status']) ?> • 
                                <?= timeAgo($activity['date']) ?>
                            </p>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Scripts -->
    <script>
        // ========== FONCTIONS PRINCIPALES ==========
        
        // Créer un nouveau projet (LIEN INTELLIGENT vers flying_book_slides.php)
        function createNewProject() {
            const projectName = prompt("Nom de votre nouveau projet :", "Mon Flying Book " + new Date().toLocaleDateString());
            if (projectName) {
                window.location.href = `flying_book_slides.php?action=create&name=${encodeURIComponent(projectName)}`;
            }
        }
        
        // Éditer un projet existant
        function editProject(projectId) {
            window.location.href = `flying_book_slides.php?id=${projectId}`;
        }
        
        // Utiliser un template
        function useTemplate(templateId) {
            window.location.href = `templates.php?use=${templateId}`;
        }
        
        // ========== FONCTIONS ADMIN ==========
        
        // Diagnostic système
        async function checkSystemHealth() {
            showNotification('Vérification du système...', 'info');
            
            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=check_system_status'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    displaySystemStatus(result.status);
                    showNotification('Diagnostic terminé', 'success');
                } else {
                    showNotification('Erreur lors du diagnostic', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        // Afficher l'état du système
        function displaySystemStatus(status) {
            const statusDiv = document.getElementById('systemStatus');
            const contentDiv = document.getElementById('statusContent');
            
            let html = '';
            
            // Vérifications PHP
            html += `<div class="status-item">
                <span>Version PHP</span>
                <span class="${parseFloat(status.php_version) >= 7.4 ? 'status-good' : 'status-warning'}">${status.php_version}</span>
            </div>`;
            
            html += `<div class="status-item">
                <span>Upload Max</span>
                <span class="status-good">${status.upload_max_filesize}</span>
            </div>`;
            
            html += `<div class="status-item">
                <span>POST Max</span>
                <span class="status-good">${status.post_max_size}</span>
            </div>`;
            
            html += `<div class="status-item">
                <span>Limite Mémoire</span>
                <span class="status-good">${status.memory_limit}</span>
            </div>`;
            
            // Vérifications des dossiers
            html += `<div class="status-item">
                <span>Dossier uploads existe</span>
                <span class="${status.uploads_dir_exists ? 'status-good' : 'status-error'}">
                    <i class="fas fa-${status.uploads_dir_exists ? 'check' : 'times'}"></i>
                </span>
            </div>`;
            
            html += `<div class="status-item">
                <span>Dossier uploads inscriptible</span>
                <span class="${status.uploads_dir_writable ? 'status-good' : 'status-error'}">
                    <i class="fas fa-${status.uploads_dir_writable ? 'check' : 'times'}"></i>
                </span>
            </div>`;
            
            html += `<div class="status-item">
                <span>Dossier temp inscriptible</span>
                <span class="${status.temp_dir_writable ? 'status-good' : 'status-error'}">
                    <i class="fas fa-${status.temp_dir_writable ? 'check' : 'times'}"></i>
                </span>
            </div>`;
            
            // Conseils de réparation
            if (!status.uploads_dir_exists || !status.uploads_dir_writable) {
                html += `<div style="background: #fef2f2; border: 1px solid #fecaca; border-radius: 8px; padding: 1rem; margin-top: 1rem;">
                    <h5 style="color: #dc2626; margin: 0 0 0.5rem 0;">🚨 Problème détecté</h5>
                    <p style="margin: 0; font-size: 0.9rem;">
                        Créez le dossier et définissez les permissions :<br>
                        <code>mkdir -p ${document.location.origin}/flying-book/uploads/</code><br>
                        <code>chmod 755 ${document.location.origin}/flying-book/uploads/</code>
                    </p>
                </div>`;
            }
            
            contentDiv.innerHTML = html;
            statusDiv.style.display = 'block';
        }
        
        // Confirmer la réinitialisation des projets
        function confirmResetProjects() {
            if (confirm('⚠️ ATTENTION ⚠️\n\nCette action va supprimer TOUS les projets en mode "brouillon".\n\nÊtes-vous absolument sûr de vouloir continuer ?')) {
                if (confirm('Dernière confirmation !\n\nCette action est IRRÉVERSIBLE.\n\nVoulez-vous vraiment nettoyer tous les projets brouillons ?')) {
                    resetAllProjects();
                }
            }
        }
        
        // Réinitialiser tous les projets
        async function resetAllProjects() {
            showNotification('Nettoyage en cours...', 'warning');
            
            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=reset_all_projects&user_id=all'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    setTimeout(() => location.reload(), 2000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors du nettoyage', 'error');
            }
        }
        
        // ========== UTILITAIRES ==========
        
        // Système de notifications
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.5rem;">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : type === 'warning' ? 'exclamation-triangle' : 'info-circle'}"></i>
                    ${message}
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, 4000);
        }
        
        // Menu utilisateur
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // Message de bienvenue au chargement
        document.addEventListener('DOMContentLoaded', function() {
            showNotification('Bienvenue dans Flying Book Creator v2.0 ! 🎪', 'success');
        });
    </script>
</body>
</html>