<?php
session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_role = $_SESSION['user_role'];
$is_admin = ($user_role === 'admin');

$message = '';
$error = '';

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'quick_create':
                $project_name = trim($_POST['project_name']);
                $template_id = (int)($_POST['template_id'] ?? 0);
                
                if (empty($project_name)) {
                    echo json_encode(['success' => false, 'message' => 'Nom du projet requis']);
                    exit;
                }
                
                $pdo->beginTransaction();
                
                try {
                    // Générer un slug unique
                    $base_slug = generateSlug($project_name);
                    $project_slug = $base_slug;
                    $counter = 1;
                    
                    // Vérifier unicité du slug
                    while (true) {
                        $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE project_slug = ? AND user_id = ?");
                        $stmt->execute([$project_slug, $user_id]);
                        if (!$stmt->fetch()) break;
                        
                        $project_slug = $base_slug . '-' . $counter;
                        $counter++;
                    }
                    
                    // Créer le projet
                    $project_folder = 'project_' . uniqid();
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO flying_book_projects 
                        (user_id, project_name, project_slug, title, project_folder, status)
                        VALUES (?, ?, ?, ?, ?, 'draft')
                    ");
                    
                    $stmt->execute([
                        $user_id, $project_name, $project_slug, 
                        $project_name, $project_folder
                    ]);
                    
                    $new_project_id = $pdo->lastInsertId();
                    
                    // Si un template est sélectionné, l'appliquer
                    if ($template_id > 0) {
                        $stmt = $pdo->prepare("SELECT * FROM flying_book_templates WHERE id = ?");
                        $stmt->execute([$template_id]);
                        $template = $stmt->fetch();
                        
                        if ($template) {
                            $template_data = json_decode($template['template_config'], true);
                            
                            // Mettre à jour le projet avec les données du template
                            $stmt = $pdo->prepare("
                                UPDATE flying_book_projects 
                                SET title = ?, subtitle = ?, description = ?, 
                                    style_theme = ?, transition_type = ?, navigation_type = ?
                                WHERE id = ?
                            ");
                            $stmt->execute([
                                $template_data['title'] ?? $project_name,
                                $template_data['subtitle'] ?? null,
                                $template_data['description'] ?? null,
                                $template_data['theme'] ?? 'moderne',
                                $template_data['transition'] ?? '3d',
                                $template_data['navigation'] ?? 'all',
                                $new_project_id
                            ]);
                            
                            // Créer les slides du template
                            if (isset($template_data['slides']) && is_array($template_data['slides'])) {
                                foreach ($template_data['slides'] as $index => $slide) {
                                    $stmt = $pdo->prepare("
                                        INSERT INTO flying_book_slides 
                                        (project_id, slide_number, slide_title, media_type, text_overlay, display_order)
                                        VALUES (?, ?, ?, ?, ?, ?)
                                    ");
                                    $stmt->execute([
                                        $new_project_id,
                                        $index + 1,
                                        $slide['title'] ?? "Slide " . ($index + 1),
                                        $slide['type'] ?? 'text_image',
                                        $slide['content'] ?? '',
                                        $index
                                    ]);
                                }
                            }
                            
                            // Incrémenter l'usage du template
                            $stmt = $pdo->prepare("UPDATE flying_book_templates SET usage_count = usage_count + 1 WHERE id = ?");
                            $stmt->execute([$template_id]);
                        }
                    }
                    
                    $pdo->commit();
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Projet créé avec succès',
                        'project_id' => $new_project_id,
                        'redirect' => "flying_book_slides.php?id={$new_project_id}"
                    ]);
                    
                } catch (Exception $e) {
                    $pdo->rollback();
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'import_ppt':
                // TODO: Implémentation future de l'import PowerPoint
                echo json_encode(['success' => false, 'message' => 'Import PowerPoint bientôt disponible']);
                exit;
        }
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ==========
    
    // Templates disponibles (récupérer les vrais templates)
    $stmt = $pdo->prepare("
        SELECT t.*, u.name as creator_name
        FROM flying_book_templates t
        LEFT JOIN users u ON t.user_id = u.id
        WHERE t.is_public = 1 OR t.user_id = ?
        ORDER BY t.is_featured DESC, t.usage_count DESC
        LIMIT 12
    ");
    $stmt->execute([$user_id]);
    $templates = $stmt->fetchAll();
    
    // Statistiques utilisateur
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_projects,
            SUM(CASE WHEN status = 'published' THEN 1 ELSE 0 END) as published_projects,
            SUM(file_size_total) as total_storage
        FROM flying_book_projects 
        WHERE user_id = ? AND status != 'deleted'
    ");
    $stmt->execute([$user_id]);
    $user_stats = $stmt->fetch();
    
    // Projets récents pour suggestions de noms
    $stmt = $pdo->prepare("
        SELECT project_name, title 
        FROM flying_book_projects 
        WHERE user_id = ? AND status != 'deleted'
        ORDER BY created_at DESC 
        LIMIT 5
    ");
    $stmt->execute([$user_id]);
    $recent_projects = $stmt->fetchAll();
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement: " . $e->getMessage();
}

function generateSlug($name) {
    $slug = strtolower(trim($name));
    $slug = preg_replace('/[^a-z0-9-]/', '-', $slug);
    $slug = preg_replace('/-+/', '-', $slug);
    return trim($slug, '-');
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Nouveau Projet</title>
    <link rel="stylesheet" href="assets/css/united-fixed.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Styles identiques au fichier précédent */
        .creation-hero {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            border-radius: 20px;
            padding: 3rem 2rem;
            color: white;
            margin-bottom: 3rem;
            text-align: center;
            position: relative;
            overflow: hidden;
        }
        
        .creation-hero::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: repeating-linear-gradient(
                45deg,
                transparent,
                transparent 10px,
                rgba(255, 255, 255, 0.05) 10px,
                rgba(255, 255, 255, 0.05) 20px
            );
            animation: movePattern 20s linear infinite;
        }
        
        @keyframes movePattern {
            0% { transform: translate(-50%, -50%) rotate(0deg); }
            100% { transform: translate(-50%, -50%) rotate(360deg); }
        }
        
        .creation-hero > * {
            position: relative;
            z-index: 2;
        }
        
        .creation-methods {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 2rem;
            margin-bottom: 3rem;
        }
        
        .method-card {
            background: white;
            border-radius: 20px;
            padding: 2.5rem;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .method-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }
        
        .method-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
        }
        
        .method-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 2rem;
            margin-bottom: 1.5rem;
        }
        
        .quick-templates {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            gap: 1rem;
            margin: 1.5rem 0;
        }
        
        .template-quick {
            background: #f8fafc;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 0.85rem;
        }
        
        .template-quick:hover,
        .template-quick.selected {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.1);
            transform: translateY(-2px);
        }
        
        .suggestions {
            background: #fff7ed;
            border: 1px solid #fed7aa;
            border-radius: 12px;
            padding: 1rem;
            margin-bottom: 1rem;
        }
        
        .suggestion-chips {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
            margin-top: 0.5rem;
        }
        
        .suggestion-chip {
            background: white;
            border: 1px solid #d97706;
            color: #d97706;
            padding: 0.25rem 0.75rem;
            border-radius: 15px;
            font-size: 0.8rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .suggestion-chip:hover {
            background: #d97706;
            color: white;
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="templates.php" class="nav-link">
                        <i class="fas fa-layer-group"></i> Templates
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($_SESSION['user_name'], 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($_SESSION['user_name']) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- Hero Section -->
            <div class="creation-hero">
                <h1><i class="fas fa-magic"></i> Créer un Nouveau Flying Book</h1>
                <p>Donnez vie à vos idées avec Flying Book Creator v2.0</p>
                <div style="display: flex; gap: 1rem; justify-content: center; flex-wrap: wrap;">
                    <a href="dashboard.php" class="btn btn-outline" style="color: white; border-color: white;">
                        <i class="fas fa-arrow-left"></i> Retour Dashboard
                    </a>
                    <a href="templates.php" class="btn btn-outline" style="color: white; border-color: white;">
                        <i class="fas fa-layer-group"></i> Parcourir Templates
                    </a>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Méthodes de création -->
            <div class="creation-methods">
                <!-- Création rapide -->
                <div class="method-card">
                    <div class="method-icon">
                        <i class="fas fa-rocket"></i>
                    </div>
                    <h3>Création Rapide</h3>
                    <p>Créez un nouveau Flying Book en quelques secondes avec ou sans template</p>
                    
                    <div class="method-form">
                        <form id="quickCreateForm">
                            <div class="suggestions">
                                <strong><i class="fas fa-lightbulb"></i> Suggestions de noms :</strong>
                                <div class="suggestion-chips">
                                    <span class="suggestion-chip" onclick="useSuggestion('Flying Book <?= date('Y') ?>')">Flying Book <?= date('Y') ?></span>
                                    <span class="suggestion-chip" onclick="useSuggestion('Mon Portfolio')">Mon Portfolio</span>
                                    <span class="suggestion-chip" onclick="useSuggestion('Présentation <?= date('M') ?>')">Présentation <?= date('M') ?></span>
                                    <?php if (!empty($recent_projects)): ?>
                                        <span class="suggestion-chip" onclick="useSuggestion('<?= htmlspecialchars($recent_projects[0]['project_name']) ?> v2')">
                                            <?= htmlspecialchars($recent_projects[0]['project_name']) ?> v2
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Nom du projet *</label>
                                <input type="text" id="projectName" class="form-control" 
                                       placeholder="Ex: Ma Présentation Incroyable" required>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Template (optionnel)</label>
                                <div class="quick-templates">
                                    <div class="template-quick selected" data-template="0">
                                        <i class="fas fa-file-alt"></i>
                                        Vierge
                                    </div>
                                    <?php if (!empty($templates)): ?>
                                        <?php foreach (array_slice($templates, 0, 5) as $template): ?>
                                            <div class="template-quick" data-template="<?= $template['id'] ?>">
                                                <i class="fas fa-layer-group"></i>
                                                <?= htmlspecialchars(substr($template['template_name'], 0, 15)) ?>
                                                <?= strlen($template['template_name']) > 15 ? '...' : '' ?>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <div style="grid-column: 1 / -1; text-align: center; color: #6b7280; padding: 1rem;">
                                            <i class="fas fa-info-circle"></i>
                                            Aucun template disponible. Créez-en un depuis l'éditeur de templates !
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary" style="width: 100%;">
                                <i class="fas fa-magic"></i>
                                Créer le Flying Book
                            </button>
                        </form>
                    </div>
                    
                    <div class="progress-indicator" id="quickProgress" style="display: none; text-align: center; padding: 2rem;">
                        <div style="width: 60px; height: 60px; border: 4px solid #f3f4f6; border-top: 4px solid var(--primary-color); border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto 1rem;"></div>
                        <p>Création en cours...</p>
                    </div>
                </div>

                <!-- Import PowerPoint -->
                <div class="method-card">
                    <div class="method-icon">
                        <i class="fas fa-file-powerpoint"></i>
                    </div>
                    <h3>Import PowerPoint</h3>
                    <p>Importez votre présentation PowerPoint existante et convertissez-la en Flying Book</p>
                    
                    <div class="method-form">
                        <form id="importForm">
                            <div class="form-group">
                                <label class="form-label">Fichier PowerPoint</label>
                                <input type="file" class="form-control" accept=".ppt,.pptx" disabled>
                                <small style="color: #6b7280;">Formats supportés: .ppt, .pptx (bientôt disponible)</small>
                            </div>
                            
                            <button type="button" class="btn btn-outline" style="width: 100%;" onclick="showPowerPointInfo()">
                                <i class="fas fa-upload"></i>
                                Importer PowerPoint
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Assistant IA -->
                <div class="method-card">
                    <div class="method-icon">
                        <i class="fas fa-robot"></i>
                    </div>
                    <h3>Assistant IA</h3>
                    <p>Laissez l'intelligence artificielle vous aider à créer une présentation sur mesure</p>
                    
                    <div class="method-form">
                        <form id="aiForm">
                            <div class="form-group">
                                <label class="form-label">Décrivez votre présentation</label>
                                <textarea class="form-control" rows="3" 
                                          placeholder="Ex: Une présentation sur les avantages du télétravail pour une réunion d'équipe" disabled></textarea>
                            </div>
                            
                            <button type="button" class="btn btn-outline" style="width: 100%;" onclick="showAIInfo()">
                                <i class="fas fa-brain"></i>
                                Créer avec IA
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Templates populaires -->
            <?php if (!empty($templates)): ?>
            <div style="background: white; border-radius: 16px; padding: 2rem; box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);">
                <h2><i class="fas fa-fire"></i> Templates Populaires</h2>
                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 1rem; margin-top: 1rem;">
                    <?php foreach ($templates as $template): ?>
                        <div style="background: white; border: 1px solid #e2e8f0; border-radius: 8px; padding: 1rem; cursor: pointer; transition: all 0.3s ease;" 
                             onclick="useTemplate(<?= $template['id'] ?>)"
                             onmouseover="this.style.borderColor='#667eea'; this.style.transform='translateY(-1px)'"
                             onmouseout="this.style.borderColor='#e2e8f0'; this.style.transform='translateY(0)'">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.5rem;">
                                <h4 style="margin: 0;"><?= htmlspecialchars($template['template_name']) ?></h4>
                                <?php if ($template['is_featured']): ?>
                                    <span style="background: #f59e0b; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.7rem;">
                                        <i class="fas fa-star"></i> Featured
                                    </span>
                                <?php endif; ?>
                            </div>
                            <p style="color: #64748b; font-size: 0.8rem; margin: 0;">
                                <?= htmlspecialchars($template['category']) ?> • 
                                <?= $template['usage_count'] ?> utilisations
                            </p>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div style="text-align: center; margin-top: 2rem;">
                    <a href="templates.php" class="btn btn-outline">
                        <i class="fas fa-layer-group"></i>
                        Voir Tous les Templates
                    </a>
                </div>
            </div>
            <?php endif; ?>

            <!-- Statistiques utilisateur -->
            <div style="background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 12px; padding: 1.5rem; margin-top: 2rem;">
                <h3><i class="fas fa-chart-bar"></i> Vos Statistiques</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-top: 1rem;">
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; background: white; border-radius: 8px;">
                        <span>Flying Books créés</span>
                        <strong><?= $user_stats['total_projects'] ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; background: white; border-radius: 8px;">
                        <span>Flying Books publiés</span>
                        <strong><?= $user_stats['published_projects'] ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; background: white; border-radius: 8px;">
                        <span>Stockage utilisé</span>
                        <strong><?= formatFileSize($user_stats['total_storage']) ?></strong>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Scripts -->
    <script>
        // Variables globales
        let selectedTemplate = 0;
        
        // Initialisation
        document.addEventListener('DOMContentLoaded', function() {
            initializeTemplateSelection();
            initializeForm();
            showNotification('Interface de création Flying Book chargée ! 🚀', 'success');
        });
        
        // Gestion des templates
        function initializeTemplateSelection() {
            document.querySelectorAll('.template-quick').forEach(template => {
                template.addEventListener('click', function() {
                    // Désélectionner tous
                    document.querySelectorAll('.template-quick').forEach(t => t.classList.remove('selected'));
                    
                    // Sélectionner celui-ci
                    this.classList.add('selected');
                    selectedTemplate = parseInt(this.dataset.template) || 0;
                });
            });
        }
        
        // Gestion du formulaire
        function initializeForm() {
            document.getElementById('quickCreateForm').addEventListener('submit', async function(e) {
                e.preventDefault();
                
                const projectName = document.getElementById('projectName').value.trim();
                if (!projectName) {
                    showNotification('Veuillez saisir un nom de Flying Book', 'warning');
                    return;
                }
                
                showProgress(true);
                
                try {
                    const formData = new FormData();
                    formData.append('action', 'quick_create');
                    formData.append('project_name', projectName);
                    formData.append('template_id', selectedTemplate);
                    
                    const response = await fetch('create-project.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        showNotification(result.message, 'success');
                        
                        // Animation de succès
                        setTimeout(() => {
                            showNotification('Redirection vers l\'éditeur Flying Book...', 'info');
                            window.location.href = result.redirect;
                        }, 1500);
                    } else {
                        showNotification(result.message, 'error');
                        showProgress(false);
                    }
                } catch (error) {
                    showNotification('Erreur de connexion', 'error');
                    showProgress(false);
                    console.error('Error:', error);
                }
            });
        }
        
        // Utilitaires
        function useSuggestion(name) {
            document.getElementById('projectName').value = name;
            document.getElementById('projectName').focus();
        }
        
        function useTemplate(templateId) {
            const projectName = prompt('Nom de votre nouveau Flying Book:', 'Mon Flying Book depuis template');
            if (projectName) {
                // Créer directement avec le template
                selectedTemplate = templateId;
                document.getElementById('projectName').value = projectName;
                document.getElementById('quickCreateForm').dispatchEvent(new Event('submit'));
            }
        }
        
        function showPowerPointInfo() {
            showNotification('Import PowerPoint sera disponible dans la prochaine version ! 📊', 'info');
        }
        
        function showAIInfo() {
            showNotification('Assistant IA sera disponible dans la prochaine version ! 🤖', 'info');
        }
        
        function showProgress(show) {
            const form = document.querySelector('.method-form');
            const progress = document.getElementById('quickProgress');
            
            if (show) {
                form.style.display = 'none';
                progress.style.display = 'block';
            } else {
                form.style.display = 'block';
                progress.style.display = 'none';
            }
        }
        
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // Système de notifications
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 1rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.5rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.9;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            notification.style.cssText = `
                position: fixed;
                top: 2rem;
                right: 2rem;
                z-index: 3000;
                max-width: 400px;
                animation: slideIn 0.3s ease;
                background: white;
                border-radius: 12px;
                padding: 1rem 1.5rem;
                box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
                border-left: 4px solid var(--${type === 'success' ? 'success' : type === 'error' ? 'error' : type === 'warning' ? 'warning' : 'info'}-color);
            `;
            
            document.body.appendChild(notification);
            
            // Auto-suppression
            setTimeout(() => {
                notification.style.animation = 'slideOut 0.3s ease';
                setTimeout(() => notification.remove(), 300);
            }, type === 'success' ? 6000 : 5000);
        }
        
        // CSS pour les animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            @keyframes slideIn {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>