/**
 * Flying Book - Wizard JavaScript
 * Gestion de l'interface de création
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeWizard();
});

/**
 * Initialisation du wizard
 */
function initializeWizard() {
    // Initialiser les sélecteurs
    initializeThemeSelector();
    initializeTransitionSelector();
    initializeNavigationSelector();
    
    // Initialiser les compteurs
    initializeSlideCounter();
    
    // Validation des formulaires
    initializeFormValidation();
    
    // Auto-save
    initializeAutoSave();
    
    // Raccourcis clavier
    initializeKeyboardShortcuts();
    
    // Animations
    initializeAnimations();
}

/**
 * Initialiser le sélecteur de thème
 */
function initializeThemeSelector() {
    const themeOptions = document.querySelectorAll('.theme-option');
    
    themeOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Désélectionner tous les thèmes
            themeOptions.forEach(opt => opt.classList.remove('selected'));
            
            // Sélectionner le thème cliqué
            this.classList.add('selected');
            
            // Cocher le radio button
            const radio = this.querySelector('input[type="radio"]');
            if (radio) {
                radio.checked = true;
            }
            
            // Animation de sélection
            this.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.style.transform = 'scale(1)';
            }, 150);
            
            // Prévisualisation temps réel
            previewTheme(radio.value);
        });
    });
}

/**
 * Initialiser le sélecteur de transition
 */
function initializeTransitionSelector() {
    const transitionOptions = document.querySelectorAll('.transition-option');
    
    transitionOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Désélectionner toutes les transitions
            transitionOptions.forEach(opt => opt.classList.remove('selected'));
            
            // Sélectionner la transition cliquée
            this.classList.add('selected');
            
            // Cocher le radio button
            const radio = this.querySelector('input[type="radio"]');
            if (radio) {
                radio.checked = true;
            }
            
            // Animation de démonstration
            demonstrateTransition(this);
        });
        
        // Animation au survol
        option.addEventListener('mouseenter', function() {
            const demo = this.querySelector('.transition-demo');
            if (demo) {
                demo.style.transform = 'scale(1.05)';
            }
        });
        
        option.addEventListener('mouseleave', function() {
            const demo = this.querySelector('.transition-demo');
            if (demo) {
                demo.style.transform = 'scale(1)';
            }
        });
    });
}

/**
 * Initialiser le sélecteur de navigation
 */
function initializeNavigationSelector() {
    const navOptions = document.querySelectorAll('.nav-option');
    
    navOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Désélectionner toutes les options
            navOptions.forEach(opt => opt.classList.remove('selected'));
            
            // Sélectionner l'option cliquée
            this.classList.add('selected');
            
            // Cocher le radio button
            const radio = this.querySelector('input[type="radio"]');
            if (radio) {
                radio.checked = true;
            }
            
            // Animation de démonstration
            demonstrateNavigation(this);
        });
    });
}

/**
 * Initialiser le compteur de slides
 */
function initializeSlideCounter() {
    const slideInput = document.getElementById('total_slides');
    const minusBtn = document.querySelector('.counter-btn:first-child');
    const plusBtn = document.querySelector('.counter-btn:last-child');
    
    if (!slideInput || !minusBtn || !plusBtn) return;
    
    // Mise à jour des boutons selon la valeur
    function updateButtons() {
        const value = parseInt(slideInput.value);
        minusBtn.disabled = value <= 1;
        plusBtn.disabled = value >= 100;
        
        minusBtn.style.opacity = value <= 1 ? '0.5' : '1';
        plusBtn.style.opacity = value >= 100 ? '0.5' : '1';
        
        // Mettre à jour l'estimation de temps
        updateTimeEstimate(value);
    }
    
    updateButtons();
    slideInput.addEventListener('input', updateButtons);
}

/**
 * Changer le nombre de slides
 */
function changeSlideCount(delta) {
    const slideInput = document.getElementById('total_slides');
    if (!slideInput) return;
    
    const currentValue = parseInt(slideInput.value) || 5;
    const newValue = Math.max(1, Math.min(100, currentValue + delta));
    
    slideInput.value = newValue;
    
    // Animation du compteur
    slideInput.style.transform = 'scale(1.1)';
    slideInput.style.color = 'var(--primary-color)';
    
    setTimeout(() => {
        slideInput.style.transform = 'scale(1)';
        slideInput.style.color = '';
    }, 200);
    
    // Déclencher l'événement input pour mettre à jour les boutons
    slideInput.dispatchEvent(new Event('input'));
}

/**
 * Mettre à jour l'estimation de temps
 */
function updateTimeEstimate(slideCount) {
    const estimateElement = document.getElementById('timeEstimate');
    if (!estimateElement) return;
    
    const estimatedMinutes = Math.ceil(slideCount * 2); // 2 min par slide
    estimateElement.textContent = `Temps estimé: ${estimatedMinutes} minutes`;
}

/**
 * Prévisualiser un thème
 */
function previewTheme(themeName) {
    // Changer les couleurs CSS variables pour la prévisualisation
    const root = document.documentElement;
    
    switch (themeName) {
        case 'moderne':
            root.style.setProperty('--preview-primary', '#667eea');
            root.style.setProperty('--preview-secondary', '#764ba2');
            break;
        case 'classique':
            root.style.setProperty('--preview-primary', '#2c3e50');
            root.style.setProperty('--preview-secondary', '#3498db');
            break;
        case 'corporate':
            root.style.setProperty('--preview-primary', '#34495e');
            root.style.setProperty('--preview-secondary', '#2c3e50');
            break;
        case 'minimal':
            root.style.setProperty('--preview-primary', '#95a5a6');
            root.style.setProperty('--preview-secondary', '#bdc3c7');
            break;
        case 'premium':
            root.style.setProperty('--preview-primary', '#ff6b6b');
            root.style.setProperty('--preview-secondary', '#feca57');
            break;
    }
    
    // Animation de transition
    const wizardContent = document.querySelector('.wizard-content');
    if (wizardContent) {
        wizardContent.style.transition = 'all 0.3s ease';
        wizardContent.style.transform = 'scale(0.98)';
        
        setTimeout(() => {
            wizardContent.style.transform = 'scale(1)';
        }, 150);
    }
}

/**
 * Démontrer une transition
 */
function demonstrateTransition(option) {
    const demo = option.querySelector('.transition-demo');
    const slides = demo.querySelectorAll('.slide-demo');
    
    if (slides.length < 2) return;
    
    const transitionType = option.dataset.transition;
    
    // Reset
    slides.forEach(slide => {
        slide.style.transition = 'all 0.5s ease';
        slide.style.transform = '';
        slide.style.opacity = '1';
    });
    
    // Animation selon le type
    setTimeout(() => {
        switch (transitionType) {
            case 'fade':
                slides[0].style.opacity = '0';
                slides[1].style.opacity = '1';
                break;
            case 'slide':
                slides[0].style.transform = 'translateX(-100%)';
                slides[1].style.transform = 'translateX(0)';
                break;
            case 'flip':
                slides[0].style.transform = 'rotateY(-90deg)';
                slides[1].style.transform = 'rotateY(0deg)';
                break;
            case '3d':
                slides[0].style.transform = 'perspective(200px) rotateX(-90deg)';
                slides[1].style.transform = 'perspective(200px) rotateX(0deg)';
                break;
            case 'premium_3d':
                slides[0].style.transform = 'perspective(200px) rotateY(-180deg) rotateX(-45deg)';
                slides[1].style.transform = 'perspective(200px) rotateY(0deg) rotateX(0deg)';
                break;
        }
    }, 100);
    
    // Reset après la démo
    setTimeout(() => {
        slides.forEach(slide => {
            slide.style.transform = '';
            slide.style.opacity = '';
            slide.style.transition = '';
        });
    }, 2000);
}

/**
 * Démontrer la navigation
 */
function demonstrateNavigation(option) {
    const preview = option.querySelector('.nav-preview');
    const navType = option.dataset.nav;
    
    // Animation selon le type de navigation
    preview.style.transition = 'all 0.3s ease';
    
    switch (navType) {
        case 'buttons':
            const arrows = preview.querySelectorAll('i');
            arrows.forEach(arrow => {
                arrow.style.transform = 'scale(1.2)';
                arrow.style.color = 'var(--success-color)';
            });
            setTimeout(() => {
                arrows.forEach(arrow => {
                    arrow.style.transform = 'scale(1)';
                    arrow.style.color = '';
                });
            }, 500);
            break;
            
        case 'thumbnails':
            const thumbs = preview.querySelectorAll('.thumb');
            thumbs.forEach((thumb, index) => {
                setTimeout(() => {
                    thumb.style.transform = 'scale(1.3)';
                    thumb.style.background = 'var(--primary-color)';
                    setTimeout(() => {
                        thumb.style.transform = 'scale(1)';
                        if (!thumb.classList.contains('active')) {
                            thumb.style.background = '';
                        }
                    }, 200);
                }, index * 150);
            });
            break;
            
        case 'keyboard':
            const keyboardHint = preview.querySelector('.keyboard-hint');
            if (keyboardHint) {
                keyboardHint.style.transform = 'scale(1.3)';
                keyboardHint.style.color = 'var(--success-color)';
                setTimeout(() => {
                    keyboardHint.style.transform = 'scale(1)';
                    keyboardHint.style.color = '';
                }, 500);
            }
            break;
            
        case 'all':
            // Animer tous les éléments
            setTimeout(() => demonstrateNavigation({
                querySelector: () => preview,
                dataset: { nav: 'buttons' }
            }), 0);
            setTimeout(() => demonstrateNavigation({
                querySelector: () => preview,
                dataset: { nav: 'thumbnails' }
            }), 600);
            break;
    }
}

/**
 * Initialiser la validation des formulaires
 */
function initializeFormValidation() {
    const forms = document.querySelectorAll('.wizard-form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(event) {
            if (!validateForm(this)) {
                event.preventDefault();
                return false;
            }
            
            // Afficher le loading sur le bouton
            const submitBtn = this.querySelector('button[type="submit"]');
            if (submitBtn) {
                setButtonLoading(submitBtn, true);
            }
        });
        
        // Validation en temps réel
        const inputs = form.querySelectorAll('input[required], textarea[required]');
        inputs.forEach(input => {
            input.addEventListener('blur', () => validateInput(input));
            input.addEventListener('input', () => clearInputError(input));
        });
    });
}

/**
 * Valider un formulaire
 */
function validateForm(form) {
    let isValid = true;
    const requiredInputs = form.querySelectorAll('input[required], textarea[required]');
    
    requiredInputs.forEach(input => {
        if (!validateInput(input)) {
            isValid = false;
        }
    });
    
    // Validations spécifiques selon l'étape
    const step = form.querySelector('input[name="step"]')?.value;
    
    if (step === '1') {
        const projectName = form.querySelector('input[name="project_name"]');
        if (projectName && projectName.value.trim().length < 3) {
            showInputError(projectName, 'Le nom doit contenir au moins 3 caractères');
            isValid = false;
        }
    }
    
    if (step === '2') {
        const title = form.querySelector('input[name="title"]');
        if (title && title.value.trim().length < 3) {
            showInputError(title, 'Le titre doit contenir au moins 3 caractères');
            isValid = false;
        }
    }
    
    if (step === '3') {
        const totalSlides = form.querySelector('input[name="total_slides"]');
        const slideCount = parseInt(totalSlides?.value || 0);
        if (slideCount < 1 || slideCount > 100) {
            showToast('Le nombre de slides doit être entre 1 et 100', 'error');
            isValid = false;
        }
    }
    
    return isValid;
}

/**
 * Valider un input individuel
 */
function validateInput(input) {
    const value = input.value.trim();
    
    // Vérifier si requis
    if (input.required && !value) {
        showInputError(input, 'Ce champ est requis');
        return false;
    }
    
    // Validations spécifiques par type
    if (input.type === 'email' && value) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(value)) {
            showInputError(input, 'Adresse email invalide');
            return false;
        }
    }
    
    if (input.type === 'number' && value) {
        const num = parseInt(value);
        const min = parseInt(input.min);
        const max = parseInt(input.max);
        
        if (min && num < min) {
            showInputError(input, `La valeur doit être au moins ${min}`);
            return false;
        }
        
        if (max && num > max) {
            showInputError(input, `La valeur ne peut pas dépasser ${max}`);
            return false;
        }
    }
    
    return true;
}

/**
 * Afficher une erreur sur un input
 */
function showInputError(input, message) {
    const formGroup = input.closest('.form-group');
    if (!formGroup) return;
    
    // Supprimer l'ancienne erreur
    const existingError = formGroup.querySelector('.input-error');
    if (existingError) {
        existingError.remove();
    }
    
    // Ajouter la nouvelle erreur
    const errorDiv = document.createElement('div');
    errorDiv.className = 'input-error';
    errorDiv.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
    errorDiv.style.cssText = `
        color: var(--error-color);
        font-size: 0.75rem;
        margin-top: 0.5rem;
        display: flex;
        align-items: center;
        gap: 0.25rem;
        animation: slideIn 0.3s ease;
    `;
    
    formGroup.appendChild(errorDiv);
    
    // Ajouter la classe d'erreur à l'input
    input.style.borderColor = 'var(--error-color)';
    input.focus();
}

/**
 * Nettoyer l'erreur d'un input
 */
function clearInputError(input) {
    const formGroup = input.closest('.form-group');
    if (!formGroup) return;
    
    const errorDiv = formGroup.querySelector('.input-error');
    if (errorDiv) {
        errorDiv.remove();
    }
    
    input.style.borderColor = '';
}

/**
 * Initialiser l'auto-sauvegarde
 */
function initializeAutoSave() {
    const forms = document.querySelectorAll('.wizard-form');
    
    forms.forEach(form => {
        const inputs = form.querySelectorAll('input, textarea, select');
        
        inputs.forEach(input => {
            input.addEventListener('input', debounce(() => {
                saveFormData(form);
            }, 1000));
        });
    });
    
    // Charger les données sauvegardées
    loadFormData();
}

/**
 * Sauvegarder les données du formulaire
 */
function saveFormData(form) {
    const formData = new FormData(form);
    const data = Object.fromEntries(formData.entries());
    const step = data.step || 'unknown';
    
    try {
        localStorage.setItem(`flying_book_wizard_step_${step}`, JSON.stringify(data));
        
        // Indication visuelle de sauvegarde
        showToast('Brouillon sauvegardé', 'success', 2000);
    } catch (e) {
        console.warn('Impossible de sauvegarder le brouillon', e);
    }
}

/**
 * Charger les données sauvegardées
 */
function loadFormData() {
    const currentForm = document.querySelector('.wizard-form');
    if (!currentForm) return;
    
    const step = currentForm.querySelector('input[name="step"]')?.value;
    if (!step) return;
    
    try {
        const savedData = localStorage.getItem(`flying_book_wizard_step_${step}`);
        if (!savedData) return;
        
        const data = JSON.parse(savedData);
        
        // Remplir les champs
        Object.keys(data).forEach(key => {
            const input = currentForm.querySelector(`[name="${key}"]`);
            if (input && input.type !== 'hidden') {
                if (input.type === 'radio') {
                    if (input.value === data[key]) {
                        input.checked = true;
                        // Déclencher l'événement pour mettre à jour l'interface
                        input.dispatchEvent(new Event('change'));
                    }
                } else {
                    input.value = data[key];
                }
            }
        });
        
        showToast('Brouillon restauré', 'info', 2000);
    } catch (e) {
        console.warn('Impossible de charger le brouillon', e);
    }
}

/**
 * Initialiser les raccourcis clavier
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(event) {
        // Échapper pour revenir en arrière
        if (event.key === 'Escape') {
            const backBtn = document.querySelector('.btn-outline[href*="step"]');
            if (backBtn) {
                window.location.href = backBtn.href;
            } else {
                window.location.href = '../dashboard.php';
            }
        }
        
        // Ctrl/Cmd + Entrée pour continuer
        if ((event.ctrlKey || event.metaKey) && event.key === 'Enter') {
            event.preventDefault();
            const submitBtn = document.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.disabled) {
                submitBtn.click();
            }
        }
        
        // Flèches pour naviguer dans les sélecteurs
        if (event.key === 'ArrowLeft' || event.key === 'ArrowRight') {
            const focusedOption = document.querySelector('.theme-option:focus, .transition-option:focus, .nav-option:focus');
            if (focusedOption) {
                event.preventDefault();
                navigateOptions(focusedOption, event.key === 'ArrowRight' ? 1 : -1);
            }
        }
    });
}

/**
 * Naviguer entre les options avec les flèches
 */
function navigateOptions(currentOption, direction) {
    const container = currentOption.parentElement;
    const options = Array.from(container.children);
    const currentIndex = options.indexOf(currentOption);
    const newIndex = Math.max(0, Math.min(options.length - 1, currentIndex + direction));
    
    if (newIndex !== currentIndex) {
        options[newIndex].click();
        options[newIndex].focus();
    }
}

/**
 * Initialiser les animations
 */
function initializeAnimations() {
    // Animation d'entrée pour les étapes
    const wizardStep = document.querySelector('.wizard-step');
    if (wizardStep) {
        setTimeout(() => {
            wizardStep.classList.add('active');
        }, 100);
    }
    
    // Animations au scroll
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    });
    
    const animatedElements = document.querySelectorAll('.form-section, .theme-option, .transition-option, .nav-option');
    animatedElements.forEach((el, index) => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(20px)';
        el.style.transition = `opacity 0.5s ease ${index * 0.1}s, transform 0.5s ease ${index * 0.1}s`;
        observer.observe(el);
    });
}

/**
 * Gérer l'état de chargement d'un bouton
 */
function setButtonLoading(button, loading) {
    if (loading) {
        button.disabled = true;
        button.classList.add('loading');
        button.style.pointerEvents = 'none';
    } else {
        button.disabled = false;
        button.classList.remove('loading');
        button.style.pointerEvents = 'auto';
    }
}

/**
 * Afficher une notification toast
 */
function showToast(message, type = 'info', duration = 3000) {
    // Créer l'élément toast s'il n'existe pas
    let toastContainer = document.getElementById('toastContainer');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toastContainer';
        toastContainer.style.cssText = `
            position: fixed;
            top: 1rem;
            right: 1rem;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        `;
        document.body.appendChild(toastContainer);
    }
    
    // Créer le toast
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    const colors = {
        success: 'var(--success-color)',
        error: 'var(--error-color)',
        warning: 'var(--warning-color)',
        info: 'var(--info-color)'
    };
    
    toast.innerHTML = `
        <i class="fas ${icons[type]}"></i>
        <span>${message}</span>
        <button class="toast-close" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    toast.style.cssText = `
        background: white;
        border: 1px solid ${colors[type]};
        border-left: 4px solid ${colors[type]};
        border-radius: var(--radius-md);
        padding: 0.75rem 1rem;
        box-shadow: var(--shadow-lg);
        display: flex;
        align-items: center;
        gap: 0.75rem;
        min-width: 250px;
        max-width: 400px;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        font-size: 0.875rem;
    `;
    
    toast.querySelector('i:first-child').style.color = colors[type];
    
    const closeButton = toast.querySelector('.toast-close');
    closeButton.style.cssText = `
        background: none;
        border: none;
        color: var(--text-light);
        cursor: pointer;
        padding: 0.25rem;
        margin-left: auto;
        border-radius: var(--radius-sm);
    `;
    
    toastContainer.appendChild(toast);
    
    // Animation d'entrée
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 100);
    
    // Suppression automatique
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (toast.parentElement) {
                toast.remove();
            }
        }, 300);
    }, duration);
}

/**
 * Fonction utilitaire de debounce
 */
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

/**
 * Nettoyage au changement de page
 */
window.addEventListener('beforeunload', function() {
    // Nettoyer les données temporaires si le wizard est terminé
    const isCompleted = window.location.href.includes('slides/edit.php');
    if (isCompleted) {
        ['1', '2', '3'].forEach(step => {
            localStorage.removeItem(`flying_book_wizard_step_${step}`);
        });
    }
});

// Rendre les fonctions globales pour les onclick
window.changeSlideCount = changeSlideCount;