/**
 * Flying Book - Slides Editor JavaScript
 * Gestion de l'éditeur de slides avec drag & drop
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeSlidesEditor();
});

/**
 * Initialisation de l'éditeur de slides
 */
function initializeSlidesEditor() {
    // Initialiser le drag & drop pour les slides
    initializeSortableSlides();
    
    // Initialiser l'upload de fichiers
    initializeFileUpload();
    
    // Initialiser les onglets d'upload
    initializeUploadTabs();
    
    // Charger la première slide
    if (currentSlideId) {
        loadSlideEditor(currentSlideId);
    }
    
    // Initialiser les raccourcis clavier
    initializeKeyboardShortcuts();
    
    // Auto-save périodique
    initializeAutoSave();
    
    // Gestion des événements
    initializeEventListeners();
}

/**
 * Initialiser le système de tri des slides
 */
function initializeSortableSlides() {
    const slidesList = document.getElementById('slidesList');
    if (!slidesList) return;
    
    new Sortable(slidesList, {
        animation: 150,
        ghostClass: 'sortable-ghost',
        chosenClass: 'sortable-chosen',
        dragClass: 'sortable-drag',
        handle: '.drag-handle',
        onEnd: function(event) {
            // Mettre à jour l'ordre des slides
            updateSlidesOrder();
        }
    });
}

/**
 * Mettre à jour l'ordre des slides
 */
async function updateSlidesOrder() {
    const slideItems = document.querySelectorAll('.slide-item');
    const slideOrders = {};
    
    slideItems.forEach((item, index) => {
        const slideId = item.dataset.slideId;
        slideOrders[slideId] = index + 1;
        
        // Mettre à jour le numéro affiché
        const numberElement = item.querySelector('.slide-number');
        if (numberElement) {
            numberElement.textContent = index + 1;
        }
    });
    
    try {
        const response = await fetch('edit.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=reorder_slides&slide_orders=${encodeURIComponent(JSON.stringify(slideOrders))}`
        });
        
        const result = await response.json();
        if (result.success) {
            showToast('Ordre des slides mis à jour', 'success');
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur lors de la réorganisation:', error);
        showToast('Erreur lors de la réorganisation', 'error');
    }
}

/**
 * Sélectionner une slide
 */
function selectSlide(slideId) {
    // Mettre à jour l'état actif
    document.querySelectorAll('.slide-item').forEach(item => {
        item.classList.remove('active');
    });
    
    const selectedItem = document.querySelector(`[data-slide-id="${slideId}"]`);
    if (selectedItem) {
        selectedItem.classList.add('active');
    }
    
    // Charger l'éditeur de la slide
    currentSlideId = slideId;
    loadSlideEditor(slideId);
}

/**
 * Charger l'éditeur pour une slide spécifique
 */
function loadSlideEditor(slideId) {
    const slide = SLIDES_DATA.find(s => s.id == slideId);
    if (!slide) return;
    
    const editorContent = document.getElementById('editorContent');
    const panelContent = document.getElementById('panelContent');
    
    // Générer l'interface d'édition de la slide
    editorContent.innerHTML = `
        <div class="slide-editor">
            <div class="slide-preview" id="slidePreview">
                ${generateSlidePreview(slide)}
            </div>
            <div class="slide-controls">
                <div class="slide-toolbar">
                    <button class="btn btn-outline btn-sm" onclick="openUploadModal()">
                        <i class="fas fa-plus"></i>
                        Ajouter média
                    </button>
                    <button class="btn btn-outline btn-sm" onclick="resetSlide(${slideId})">
                        <i class="fas fa-undo"></i>
                        Réinitialiser
                    </button>
                    <button class="btn btn-primary btn-sm" onclick="saveSlide(${slideId})">
                        <i class="fas fa-save"></i>
                        Sauvegarder
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Générer le panneau de propriétés
    panelContent.innerHTML = generatePropertiesPanel(slide);
    
    // Initialiser les événements du panneau
    initializeSlideProperties(slideId);
}

/**
 * Générer l'aperçu de la slide
 */
function generateSlidePreview(slide) {
    if (!slide.media_file || slide.media_file === 'placeholder.jpg') {
        return `
            <div class="placeholder">
                <i class="fas fa-image"></i>
                <p>Cliquez pour ajouter un média</p>
            </div>
        `;
    }
    
    const mediaPath = `../document/projects/${PROJECT_DATA.project_folder}/media/${slide.media_file}`;
    
    switch (slide.media_type) {
        case 'image':
        case 'text_image':
            return `
                <img src="${mediaPath}" alt="${slide.slide_title}" onclick="openUploadModal()">
                ${slide.text_overlay ? `<div class="text-overlay">${slide.text_overlay}</div>` : ''}
            `;
            
        case 'video':
            return `
                <video src="${mediaPath}" controls onclick="openUploadModal()">
                    Votre navigateur ne supporte pas la lecture vidéo.
                </video>
                ${slide.text_overlay ? `<div class="text-overlay">${slide.text_overlay}</div>` : ''}
            `;
            
        case 'audio_image':
            return `
                <div class="audio-slide" onclick="openUploadModal()">
                    <img src="${mediaPath}" alt="${slide.slide_title}">
                    <div class="audio-controls">
                        <i class="fas fa-play-circle"></i>
                        <span>Audio intégré</span>
                    </div>
                </div>
                ${slide.text_overlay ? `<div class="text-overlay">${slide.text_overlay}</div>` : ''}
            `;
            
        default:
            return `
                <div class="placeholder" onclick="openUploadModal()">
                    <i class="fas fa-file"></i>
                    <p>Média non supporté</p>
                </div>
            `;
    }
}

/**
 * Générer le panneau de propriétés
 */
function generatePropertiesPanel(slide) {
    return `
        <div class="property-group">
            <h4><i class="fas fa-info-circle"></i> Informations</h4>
            
            <div class="form-group">
                <label for="slideTitle">Titre de la slide</label>
                <input type="text" id="slideTitle" value="${slide.slide_title || ''}" 
                       placeholder="Titre de la slide" onchange="updateSlideProperty('slide_title', this.value)">
            </div>
            
            <div class="form-group">
                <label for="slideNotes">Notes du présentateur</label>
                <textarea id="slideNotes" placeholder="Notes pour le présentateur..." 
                          onchange="updateSlideProperty('slide_notes', this.value)">${slide.slide_notes || ''}</textarea>
            </div>
        </div>
        
        <div class="property-group">
            <h4><i class="fas fa-photo-video"></i> Type de média</h4>
            
            <div class="media-type-selector">
                <div class="media-type-btn ${slide.media_type === 'image' ? 'active' : ''}" 
                     onclick="changeMediaType('image')">
                    <i class="fas fa-image"></i>
                    Image
                </div>
                <div class="media-type-btn ${slide.media_type === 'video' ? 'active' : ''}" 
                     onclick="changeMediaType('video')">
                    <i class="fas fa-video"></i>
                    Vidéo
                </div>
                <div class="media-type-btn ${slide.media_type === 'audio_image' ? 'active' : ''}" 
                     onclick="changeMediaType('audio_image')">
                    <i class="fas fa-music"></i>
                    Audio + Image
                </div>
                <div class="media-type-btn ${slide.media_type === 'text_image' ? 'active' : ''}" 
                     onclick="changeMediaType('text_image')">
                    <i class="fas fa-font"></i>
                    Texte + Image
                </div>
            </div>
        </div>
        
        <div class="property-group">
            <h4><i class="fas fa-text-width"></i> Texte superposé</h4>
            
            <div class="form-group">
                <label for="textOverlay">Texte à afficher</label>
                <textarea id="textOverlay" placeholder="Texte à superposer sur le média..." 
                          onchange="updateSlideProperty('text_overlay', this.value)">${slide.text_overlay || ''}</textarea>
            </div>
        </div>
        
        <div class="property-group">
            <h4><i class="fas fa-clock"></i> Timing</h4>
            
            <div class="form-group">
                <label for="autoDuration">Durée automatique (secondes)</label>
                <input type="number" id="autoDuration" min="0" max="300" 
                       value="${slide.auto_duration || 0}" 
                       onchange="updateSlideProperty('auto_duration', this.value)">
                <small>0 = navigation manuelle</small>
            </div>
        </div>
        
        <div class="property-group">
            <h4><i class="fas fa-cogs"></i> Actions</h4>
            
            <button class="btn btn-outline btn-sm btn-block" onclick="duplicateSlide(${slide.id})">
                <i class="fas fa-copy"></i>
                Dupliquer cette slide
            </button>
            
            <button class="btn btn-outline btn-sm btn-block" onclick="addSlide('after', ${slide.display_order})">
                <i class="fas fa-plus"></i>
                Ajouter slide après
            </button>
            
            <button class="btn btn-outline btn-sm btn-block text-danger" onclick="deleteSlide(${slide.id})">
                <i class="fas fa-trash"></i>
                Supprimer cette slide
            </button>
        </div>
    `;
}

/**
 * Initialiser les propriétés de la slide
 */
function initializeSlideProperties(slideId) {
    // Auto-save sur les changements
    const inputs = document.querySelectorAll('#panelContent input, #panelContent textarea');
    inputs.forEach(input => {
        input.addEventListener('input', debounce(() => {
            const property = input.id.replace(/([A-Z])/g, '_$1').toLowerCase();
            updateSlideProperty(property, input.value);
        }, 1000));
    });
}

/**
 * Mettre à jour une propriété de slide
 */
function updateSlideProperty(property, value) {
    if (!currentSlideId) return;
    
    // Mettre à jour localement
    const slide = SLIDES_DATA.find(s => s.id == currentSlideId);
    if (slide) {
        slide[property] = value;
    }
    
    // Marquer comme modifié pour l'auto-save
    markSlideAsModified(currentSlideId);
    
    // Mettre à jour l'aperçu si nécessaire
    if (['text_overlay', 'media_type'].includes(property)) {
        refreshSlidePreview();
    }
}

/**
 * Changer le type de média
 */
function changeMediaType(mediaType) {
    // Mettre à jour les boutons
    document.querySelectorAll('.media-type-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.closest('.media-type-btn').classList.add('active');
    
    // Mettre à jour la slide
    updateSlideProperty('media_type', mediaType);
}

/**
 * Ajouter une nouvelle slide
 */
async function addSlide(position = 'end', slideNumber = null) {
    try {
        const response = await fetch('edit.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=add_slide&position=${position}&slide_number=${slideNumber || ''}`
        });
        
        const result = await response.json();
        if (result.success) {
            showToast('Nouvelle slide ajoutée', 'success');
            // Recharger la page pour afficher la nouvelle slide
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur lors de l\'ajout:', error);
        showToast('Erreur lors de l\'ajout de la slide', 'error');
    }
}

/**
 * Dupliquer une slide
 */
async function duplicateSlide(slideId) {
    if (!confirm('Voulez-vous dupliquer cette slide ?')) return;
    
    try {
        const slide = SLIDES_DATA.find(s => s.id == slideId);
        if (!slide) throw new Error('Slide non trouvée');
        
        // Créer une copie de la slide
        await addSlide('after', slide.display_order);
        
        showToast('Slide dupliquée avec succès', 'success');
    } catch (error) {
        console.error('Erreur lors de la duplication:', error);
        showToast('Erreur lors de la duplication', 'error');
    }
}

/**
 * Supprimer une slide
 */
async function deleteSlide(slideId) {
    if (!confirm('Voulez-vous vraiment supprimer cette slide ?')) return;
    
    try {
        const response = await fetch('edit.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: `action=delete_slide&slide_id=${slideId}`
        });
        
        const result = await response.json();
        if (result.success) {
            showToast('Slide supprimée', 'success');
            
            // Supprimer de l'interface
            const slideItem = document.querySelector(`[data-slide-id="${slideId}"]`);
            if (slideItem) {
                slideItem.remove();
            }
            
            // Sélectionner une autre slide
            const remainingSlides = document.querySelectorAll('.slide-item');
            if (remainingSlides.length > 0) {
                const nextSlide = remainingSlides[0];
                selectSlide(nextSlide.dataset.slideId);
            } else {
                // Aucune slide restante
                document.getElementById('editorContent').innerHTML = `
                    <div class="empty-editor">
                        <i class="fas fa-images"></i>
                        <h3>Aucune slide</h3>
                        <p>Ajoutez votre première slide pour commencer</p>
                        <button class="btn btn-primary" onclick="addSlide('end')">
                            <i class="fas fa-plus"></i>
                            Ajouter une slide
                        </button>
                    </div>
                `;
            }
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur lors de la suppression:', error);
        showToast('Erreur lors de la suppression', 'error');
    }
}

/**
 * Sauvegarder une slide
 */
async function saveSlide(slideId) {
    const slide = SLIDES_DATA.find(s => s.id == slideId);
    if (!slide) return;
    
    try {
        const response = await fetch('edit.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'update_slide',
                slide_id: slideId,
                slide_title: slide.slide_title || '',
                media_type: slide.media_type || 'text_image',
                media_file: slide.media_file || '',
                text_overlay: slide.text_overlay || '',
                slide_notes: slide.slide_notes || '',
                auto_duration: slide.auto_duration || 0
            })
        });
        
        const result = await response.json();
        if (result.success) {
            showToast('Slide sauvegardée', 'success');
            unmarkSlideAsModified(slideId);
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur lors de la sauvegarde:', error);
        showToast('Erreur lors de la sauvegarde', 'error');
    }
}

/**
 * Initialiser l'upload de fichiers
 */
function initializeFileUpload() {
    const uploadZone = document.getElementById('uploadZone');
    const fileInput = document.getElementById('fileInput');
    
    if (!uploadZone || !fileInput) return;
    
    // Drag & Drop
    uploadZone.addEventListener('dragover', function(e) {
        e.preventDefault();
        this.classList.add('dragover');
    });
    
    uploadZone.addEventListener('dragleave', function(e) {
        e.preventDefault();
        this.classList.remove('dragover');
    });
    
    uploadZone.addEventListener('drop', function(e) {
        e.preventDefault();
        this.classList.remove('dragover');
        
        const files = Array.from(e.dataTransfer.files);
        handleFileUpload(files);
    });
    
    // Upload classique
    fileInput.addEventListener('change', function(e) {
        const files = Array.from(e.target.files);
        handleFileUpload(files);
        this.value = ''; // Reset input
    });
}

/**
 * Gérer l'upload de fichiers
 */
async function handleFileUpload(files) {
    const uploadProgress = document.getElementById('uploadProgress');
    const progressFiles = document.getElementById('progressFiles');
    
    uploadProgress.style.display = 'block';
    progressFiles.innerHTML = '';
    
    for (const file of files) {
        await uploadSingleFile(file, progressFiles);
    }
    
    // Fermer la modal après un délai
    setTimeout(() => {
        closeUploadModal();
        // Recharger la bibliothèque de médias
        loadMediaLibrary();
    }, 2000);
}

/**
 * Upload d'un fichier individuel
 */
async function uploadSingleFile(file, container) {
    // Validation
    if (!validateFile(file)) {
        showFileProgress(file, container, 'error', 'Fichier non valide');
        return;
    }
    
    const progressElement = showFileProgress(file, container, 'uploading', 'Upload...');
    
    try {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('project_id', PROJECT_ID);
        formData.append('action', 'upload_media');
        
        const response = await fetch('../api/upload-media.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showFileProgress(file, container, 'success', 'Terminé', progressElement);
            
            // Ajouter à la liste des médias
            if (result.media) {
                addMediaToLibrary(result.media);
            }
        } else {
            throw new Error(result.message || 'Erreur d\'upload');
        }
    } catch (error) {
        console.error('Erreur upload:', error);
        showFileProgress(file, container, 'error', error.message, progressElement);
    }
}

/**
 * Valider un fichier
 */
function validateFile(file) {
    const maxSize = 500 * 1024 * 1024; // 500MB
    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 
                         'video/mp4', 'video/webm', 'audio/mp3', 'audio/wav', 'audio/ogg'];
    
    if (file.size > maxSize) {
        showToast(`Fichier trop volumineux: ${file.name}`, 'error');
        return false;
    }
    
    if (!allowedTypes.includes(file.type)) {
        showToast(`Type de fichier non supporté: ${file.name}`, 'error');
        return false;
    }
    
    return true;
}

/**
 * Afficher la progression d'un fichier
 */
function showFileProgress(file, container, status, message, existingElement = null) {
    const element = existingElement || document.createElement('div');
    element.className = 'progress-file';
    
    const statusIcon = {
        uploading: '<i class="fas fa-spinner fa-spin"></i>',
        success: '<i class="fas fa-check"></i>',
        error: '<i class="fas fa-times"></i>'
    };
    
    element.innerHTML = `
        <div class="progress-file-info">
            <div class="progress-file-name">${file.name}</div>
            <div class="progress-file-size">${formatFileSize(file.size)}</div>
        </div>
        <div class="progress-bar">
            <div class="progress-fill" style="width: ${status === 'success' ? '100' : (status === 'error' ? '0' : '50')}%"></div>
        </div>
        <div class="progress-status">
            ${statusIcon[status]}
        </div>
    `;
    
    if (!existingElement) {
        container.appendChild(element);
    }
    
    return element;
}

/**
 * Initialiser les onglets d'upload
 */
function initializeUploadTabs() {
    const tabButtons = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.upload-tab');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', () => {
            const tabName = button.onclick.toString().match(/switchUploadTab\('(\w+)'\)/)?.[1];
            if (tabName) {
                switchUploadTab(tabName);
            }
        });
    });
}

/**
 * Changer d'onglet d'upload
 */
function switchUploadTab(tabName) {
    // Mettre à jour les boutons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Mettre à jour le contenu
    document.querySelectorAll('.upload-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    document.getElementById(`uploadTab${tabName.charAt(0).toUpperCase() + tabName.slice(1)}`).classList.add('active');
}

/**
 * Ouvrir la modal d'upload
 */
function openUploadModal() {
    const modal = document.getElementById('uploadModal');
    modal.classList.add('show');
    
    // Reset du formulaire
    const uploadProgress = document.getElementById('uploadProgress');
    if (uploadProgress) {
        uploadProgress.style.display = 'none';
    }
}

/**
 * Fermer la modal d'upload
 */
function closeUploadModal() {
    const modal = document.getElementById('uploadModal');
    modal.classList.remove('show');
}

/**
 * Sélectionner un média de la bibliothèque
 */
function selectMedia(filename, fileType) {
    if (!currentSlideId) {
        showToast('Aucune slide sélectionnée', 'warning');
        return;
    }
    
    // Mettre à jour la slide courante
    updateSlideProperty('media_file', filename);
    
    // Déterminer le type de média
    let mediaType = 'image';
    if (fileType === 'video') mediaType = 'video';
    else if (fileType === 'audio') mediaType = 'audio_image';
    
    updateSlideProperty('media_type', mediaType);
    
    // Rafraîchir l'aperçu
    refreshSlidePreview();
    
    // Fermer la modal
    closeUploadModal();
    
    showToast('Média ajouté à la slide', 'success');
}

/**
 * Ajouter média depuis URL
 */
async function addMediaFromUrl() {
    const urlInput = document.getElementById('mediaUrl');
    const url = urlInput.value.trim();
    
    if (!url) {
        showToast('Veuillez saisir une URL', 'warning');
        return;
    }
    
    try {
        const response = await fetch('../api/add-media-url.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                url: url,
                project_id: PROJECT_ID
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showToast('Média ajouté depuis URL', 'success');
            urlInput.value = '';
            closeUploadModal();
            loadMediaLibrary();
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur URL:', error);
        showToast('Erreur lors de l\'ajout depuis URL', 'error');
    }
}

/**
 * Rafraîchir l'aperçu de la slide
 */
function refreshSlidePreview() {
    if (!currentSlideId) return;
    
    const slide = SLIDES_DATA.find(s => s.id == currentSlideId);
    if (!slide) return;
    
    const preview = document.getElementById('slidePreview');
    if (preview) {
        preview.innerHTML = generateSlidePreview(slide);
    }
}

/**
 * Basculer la prévisualisation
 */
function togglePreview() {
    const modal = document.getElementById('previewModal');
    modal.classList.add('show');
    
    previewSlideIndex = 0;
    loadPreviewSlide(previewSlideIndex);
}

/**
 * Fermer la prévisualisation
 */
function closePreview() {
    const modal = document.getElementById('previewModal');
    modal.classList.remove('show');
}

/**
 * Slide précédente en prévisualisation
 */
function prevSlidePreview() {
    if (previewSlideIndex > 0) {
        previewSlideIndex--;
        loadPreviewSlide(previewSlideIndex);
    }
}

/**
 * Slide suivante en prévisualisation
 */
function nextSlidePreview() {
    if (previewSlideIndex < SLIDES_DATA.length - 1) {
        previewSlideIndex++;
        loadPreviewSlide(previewSlideIndex);
    }
}

/**
 * Charger une slide en prévisualisation
 */
function loadPreviewSlide(index) {
    const slide = SLIDES_DATA[index];
    if (!slide) return;
    
    const indicator = document.getElementById('slideIndicator');
    const content = document.getElementById('previewContent');
    
    indicator.textContent = `${index + 1} / ${SLIDES_DATA.length}`;
    
    content.innerHTML = `
        <div class="preview-slide">
            ${generateSlidePreview(slide)}
            ${slide.slide_title ? `<h3>${slide.slide_title}</h3>` : ''}
        </div>
    `;
}

/**
 * Basculer le menu de publication
 */
function togglePublishMenu() {
    const menu = document.getElementById('publishMenu');
    menu.classList.toggle('show');
    
    // Fermer le menu si on clique ailleurs
    document.addEventListener('click', function closeMenu(event) {
        if (!event.target.closest('.dropdown')) {
            menu.classList.remove('show');
            document.removeEventListener('click', closeMenu);
        }
    });
}

/**
 * Publier le projet
 */
async function publishProject(status) {
    try {
        const response = await fetch('../api/publish-project.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                project_id: PROJECT_ID,
                status: status
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            const message = status === 'published' ? 'Projet publié avec succès' : 'Brouillon sauvegardé';
            showToast(message, 'success');
            
            // Mettre à jour le statut dans l'interface
            const statusElement = document.querySelector('.project-meta .status');
            if (statusElement) {
                statusElement.textContent = status === 'published' ? 'Publié' : 'Brouillon';
                statusElement.className = `status ${status}`;
            }
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Erreur de publication:', error);
        showToast('Erreur lors de la publication', 'error');
    }
}

/**
 * Sauvegarder le projet
 */
async function saveProject() {
    const modifiedSlides = document.querySelectorAll('.slide-item.modified');
    
    for (const slideItem of modifiedSlides) {
        const slideId = slideItem.dataset.slideId;
        await saveSlide(slideId);
    }
    
    if (modifiedSlides.length === 0) {
        showToast('Aucune modification à sauvegarder', 'info');
    }
}

/**
 * Basculer le panneau de propriétés
 */
function togglePropertiesPanel() {
    const panel = document.getElementById('propertiesPanel');
    panel.classList.toggle('collapsed');
}

/**
 * Marquer une slide comme modifiée
 */
function markSlideAsModified(slideId) {
    const slideItem = document.querySelector(`[data-slide-id="${slideId}"]`);
    if (slideItem) {
        slideItem.classList.add('modified');
    }
}

/**
 * Démarquer une slide comme modifiée
 */
function unmarkSlideAsModified(slideId) {
    const slideItem = document.querySelector(`[data-slide-id="${slideId}"]`);
    if (slideItem) {
        slideItem.classList.remove('modified');
    }
}

/**
 * Initialiser l'auto-save
 */
function initializeAutoSave() {
    setInterval(() => {
        const modifiedSlides = document.querySelectorAll('.slide-item.modified');
        if (modifiedSlides.length > 0) {
            saveProject();
        }
    }, 30000); // Auto-save toutes les 30 secondes
}

/**
 * Initialiser les raccourcis clavier
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(event) {
        // Ctrl/Cmd + S : Sauvegarder
        if ((event.ctrlKey || event.metaKey) && event.key === 's') {
            event.preventDefault();
            saveProject();
        }
        
        // Flèches : Navigation entre slides
        if (event.key === 'ArrowLeft' || event.key === 'ArrowRight') {
            const slides = Array.from(document.querySelectorAll('.slide-item'));
            const currentIndex = slides.findIndex(slide => slide.classList.contains('active'));
            
            if (currentIndex !== -1) {
                const direction = event.key === 'ArrowRight' ? 1 : -1;
                const newIndex = Math.max(0, Math.min(slides.length - 1, currentIndex + direction));
                
                if (newIndex !== currentIndex) {
                    const newSlideId = slides[newIndex].dataset.slideId;
                    selectSlide(newSlideId);
                }
            }
        }
        
        // Échap : Fermer les modales
        if (event.key === 'Escape') {
            const modals = document.querySelectorAll('.modal.show');
            modals.forEach(modal => modal.classList.remove('show'));
        }
        
        // Suppr : Supprimer la slide courante
        if (event.key === 'Delete' && currentSlideId) {
            if (confirm('Supprimer la slide courante ?')) {
                deleteSlide(currentSlideId);
            }
        }
    });
}

/**
 * Initialiser les événements
 */
function initializeEventListeners() {
    // Fermer les dropdowns en cliquant ailleurs
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.dropdown')) {
            document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                menu.classList.remove('show');
            });
        }
    });
    
    // Empêcher la fermeture des modales en cliquant sur le contenu
    document.querySelectorAll('.modal-content').forEach(content => {
        content.addEventListener('click', function(event) {
            event.stopPropagation();
        });
    });
    
    // Fermer les modales en cliquant sur l'overlay
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', function(event) {
            if (event.target === this) {
                this.classList.remove('show');
            }
        });
    });
}

/**
 * Utilitaires
 */

function formatFileSize(bytes) {
    if (bytes >= 1073741824) {
        return (bytes / 1073741824).toFixed(2) + ' GB';
    } else if (bytes >= 1048576) {
        return (bytes / 1048576).toFixed(2) + ' MB';
    } else if (bytes >= 1024) {
        return (bytes / 1024).toFixed(2) + ' KB';
    } else {
        return bytes + ' B';
    }
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function showToast(message, type = 'info', duration = 3000) {
    // Créer l'élément toast s'il n'existe pas
    let toastContainer = document.getElementById('toastContainer');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toastContainer';
        toastContainer.style.cssText = `
            position: fixed;
            top: 1rem;
            right: 1rem;
            z-index: 10000;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        `;
        document.body.appendChild(toastContainer);
    }
    
    // Créer le toast
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    const colors = {
        success: 'var(--success-color)',
        error: 'var(--error-color)',
        warning: 'var(--warning-color)',
        info: 'var(--info-color)'
    };
    
    toast.innerHTML = `
        <i class="fas ${icons[type]}"></i>
        <span>${message}</span>
        <button class="toast-close" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    toast.style.cssText = `
        background: white;
        border: 1px solid ${colors[type]};
        border-left: 4px solid ${colors[type]};
        border-radius: var(--radius-md);
        padding: 0.75rem 1rem;
        box-shadow: var(--shadow-lg);
        display: flex;
        align-items: center;
        gap: 0.75rem;
        min-width: 250px;
        max-width: 400px;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        font-size: 0.875rem;
    `;
    
    toast.querySelector('i:first-child').style.color = colors[type];
    
    const closeButton = toast.querySelector('.toast-close');
    closeButton.style.cssText = `
        background: none;
        border: none;
        color: var(--text-light);
        cursor: pointer;
        padding: 0.25rem;
        margin-left: auto;
        border-radius: var(--radius-sm);
    `;
    
    toastContainer.appendChild(toast);
    
    // Animation d'entrée
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 100);
    
    // Suppression automatique
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (toast.parentElement) {
                toast.remove();
            }
        }, 300);
    }, duration);
}